/**
 * @file LuaInterpreter.cpp
 * @author kut2hi
 * @copyright (c) 2013 Robert Bosch Car Multimedia GmbH
 * @addtogroup ai_sw_update/common
 * @{
 */


//@todo
//- add error handling
//- support multiple objects of this class (check and remove static operations where required)
//- replace lua_pushnumber, lua_pushstring by wrapper functions, equal to 'getLuaStackVariable'
//- replace function descriptions taken over from Gen2 Download by doxygen descriptions


#include "LuaInterpreter.h"
#include "AppProxy.h"
#include <string>
#include <map>
#include "lua.hpp"

#include "base/imp/SystemCall.h"
#include "base/imp/gen2_FileOperations.h"
#include "base/imp/BootChainMgr.h"
#include "base/imp/MTD_IO.h"
#include "base/imp/UpdateLog.h"
#include "base/imp/DownloadPipe.h"
#include "adr3/dl_tclDABPlusMain.h"

#include "util/swu_globallog.h"
#include "util/swu_bootChainAccess.hpp"
#include <unistd.h>

#include <fcntl.h>        // for print2errmem() hack.


#include "base/imp/swupd_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SWUPDATE_CORE
  #include "trcGenProj/Header/LuaInterpreter.cpp.trc.h"
#endif


namespace ai_sw_update {
namespace common {

//FIXME: Bad solution, but ETG_TRACE_ERRMEM is unreadable
static void print2errmem(const char *Format, ...)
{
  int     fd;
  char    msg[1024];
  va_list arglst;

  fd=open("/dev/errmem", O_WRONLY);//O_RDWR);
  if(fd == -1)
  {
	 ETG_TRACE_USR2(("LuaInterpreter: errmem file descriptor open failure"));
	 return;
  }

  va_start(arglst, Format);
  vsnprintf(msg, sizeof(msg), Format, arglst);
  va_end(arglst);
  write(fd, msg, strlen(msg));
  close(fd);
}

const lua_Number LuaInterpreter::METHOD_SUCCESS = 1.0; // lua is configured to use double as type for numbers
const lua_Number LuaInterpreter::METHOD_FAILURE = 0.0; // lua is configured to use double as type for numbers
int       LuaInterpreter::iRegistered_functions = 0;

/**
 * Constructor.
 *
 * Initializes lua state.
 * Registers lua extension functionality. E.g. custom functions which can be called from within a lua script.
 */
#ifdef LUA_RESTART
void LuaInterpreter::LuaInitFramework()
#else
  LuaInterpreter::LuaInterpreter()
: _LState(luaL_newstate())
#endif
{
  //@todo check return value of luaL_newstate, see reference: http://www.lua.org/manual/5.2/manual.html#luaL_newstate
  ETG_TRACE_USR2(("LuaInterpreter: Ctor"));

#ifdef LUA_RESTART
   _LState = luaL_newstate();
#endif

  // Open all standard Lua libraries into the given state.
  luaL_openlibs(_LState);

  // Register lua debug hook function
  lua_sethook(_LState, luaHook, LUA_MASKCALL | LUA_MASKRET | LUA_MASKLINE, 0);


  ETG_TRACE_USR2(("LuaInterpreter: Register lua extensions"));

  // Functions names are matching the current lua scripts.
  // @todo: Check whether to rename them to more appropiate names.
  lua_register (_LState, "_writeAdr3", writeAdr3);
  lua_register (_LState, "fileAvailable", fileAvailable);
  lua_register (_LState, "_writeMagic", writeMagic);
  lua_register (_LState, "_writeMagicToActive", writeMagicToActive);
  lua_register (_LState, "_readMagic", readMagic);
  lua_register (_LState, "writeMagicBootChain", writeMagicBootChain);
  lua_register (_LState, "_writeMagicBootChain", writeMagicBootChainFR);
  lua_register (_LState, "printLog", printLog);
  lua_register (_LState, "printDisplay", printDisplay);
  lua_register (_LState, "waitTime", waitTime);
  lua_register (_LState, "_remoteExec", remoteExecFR);
  lua_register (_LState, "remoteExec", remoteExec);
  lua_register (_LState, "_writeRawNor", writeRawNor);
  lua_register (_LState, "enableDebug",  enableDebug);
  lua_register (_LState, "disableDebug", disableDebug);
  lua_register (_LState, "vinDelete",    vinDelete);
  lua_register (_LState, "setError",      setError);
  lua_register (_LState, "_setError",     setErrorFR);
  lua_register (_LState, "swu_setError",  swu_setError);
  lua_register (_LState, "_swu_setError", swu_setErrorFR);
  lua_register (_LState, "bgimageUpdate", bgimageUpdate);

  UpdateLog::vUpdateLogTrace("%s download_registered_functions is %d",
                     		 DL_LOG_SERVICE,
                     		 LuaInterpreter::iRegistered_functions);
}

#ifdef LUA_RESTART
LuaInterpreter::LuaInterpreter()
:_LState(NULL)
{
  ETG_TRACE_USR2(("LuaInterpreter: Ctor"));
}
#endif
/**
 * Destructor.
 */
LuaInterpreter::~LuaInterpreter()
{
  ETG_TRACE_USR2(("LuaInterpreter: Dtor"));
#ifndef LUA_RESTART
  lua_close(_LState);
#endif
} // @todo: check lint: Info 1740: pointer member 'ai_sw_update::common::LuaInterpreter::_LState' (line 61, file /home/kut2hi/samba/views/kut2hi_AI_PRJ_G3G_LINUX_14.0F35.vws/ai_sw_update/common/core/imp/LuaInterpreter.h) not directly freed or zeroed by destructor


const bool LuaInterpreter::execScript(const std::string & inFilePath)
{
  ETG_TRACE_USR2(("LuaInterpreter::execScript: inFilePath=%s", inFilePath.c_str()));

  tChar tmpBuffer [1024] = {0};
  bool execScriptResult = false;  // default assumption: execution of script fails

  // Load lua script from file or stdin when no file name is given.
  // http://www.lua.org/manual/5.2/manual.html#pdf-loadfile
  int luaLoadFileResult = luaL_loadfile (_LState, inFilePath.c_str()); // @todo: Check meaning of return code.

  lua_pushcfunction(_LState, traceback_handler);
  lua_insert(_LState, 1);

  if (LUA_OK == luaLoadFileResult)
  {
    ETG_TRACE_USR2(("LuaInterpreter::execScript: load lua script: success"));

    // no error during load of lua script, execute it
    ETG_TRACE_USR2(("LuaInterpreter::execScript: start execution of lua script"));
	  int luaPCallResult = lua_pcall(_LState, 0, LUA_MULTRET, -2);


    if (LUA_OK == luaPCallResult)
    {

  	  ETG_TRACE_COMP(("LuaInterpreter::execScript: lua script execution: success"));

	 // success case lua_pcall
	 execScriptResult = true;
    }
    else
    {
	  // error case lua_pcall
	  ETG_TRACE_FATAL(("LuaInterpreter::execScript: lua script execution: error (%10s), msg (%s)",
	 	luaThreadStatusToString(luaPCallResult).c_str(),
	  	lua_tostring(_LState, -1)));

	  //dump stack in donwloadPipe
    	  sprintf(tmpBuffer, "AbortDownload :: stackDump (%10s), msg (%s)", \
		luaThreadStatusToString(luaPCallResult).c_str(),
                lua_tostring(_LState, -1));
    	  printToDownloadPipe(tmpBuffer);


	  // Set software update error
	  iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_SCRIPT,
                        (int)SW_UPDATE_ERROR_TYPE_SCRIPT_EXECUTE,
                        luaThreadStatusToString(luaPCallResult).c_str(),
                        lua_tostring(_LState, -1));
    }
  }
  else
  {
  	  // error case luaL_loadfile
	  ETG_TRACE_FATAL(("LuaInterpreter::execScript: load lua script: error (%10s), msg (%s)",
        	luaThreadStatusToString(luaLoadFileResult).c_str(),
	        lua_tostring(_LState, -1)));

	  //dump stack in donwloadPipe
          sprintf(tmpBuffer, "AbortDownload :: stackDump (%10s), msg (%s)", \
                luaThreadStatusToString(luaLoadFileResult).c_str(),
                lua_tostring(_LState, -1));
          printToDownloadPipe(tmpBuffer);


          // Set software update error
          iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_SCRIPT,
                            (int)SW_UPDATE_ERROR_TYPE_SCRIPT_LOAD,
                            luaThreadStatusToString(luaLoadFileResult).c_str(),
                            lua_tostring(_LState, -1));
  }

#ifdef LUA_RESTART
  // after the execution of the script, close the lua state
  // new lua state will be opened before execscript from SWUCoreControl
  lua_close(_LState);
#endif
  return execScriptResult;
}


void LuaInterpreter::setGlobalVariable(const std::string & inName, const std::string & inValue)
{
  // push string value to stack
  lua_pushstring(_LState, inValue.c_str());

  // pop value from stack and set it as the new value of global 'inName'
  lua_setglobal(_LState, inName.c_str());

  // also add it to tabel SwuEnv
  addToSwuEnvTable(inName.c_str(), inValue.c_str());
}

void LuaInterpreter::addToSwuEnvTable(const std::string & inName, const std::string & inValue)
{
  luaSaveField(inName.c_str(), inValue.c_str());
}


const bool LuaInterpreter::getLuaStackValue(lua_State* inLState, const int inIdx, std::string& outValue)
{
  if (inIdx > lua_gettop (inLState))
  {
    // invalid index
    ETG_TRACE_ERR(("LuaInterpreter::getLuaStackValue(string): invalid index (%i)", inIdx));
    return false;
  }

  if (LUA_TSTRING == (lua_type (inLState, inIdx)))
  {
    outValue = lua_tostring (inLState, inIdx);
    ETG_TRACE_USR4(("LuaInterpreter::getLuaStackValue(string): index (%i), value (%s)", inIdx, outValue.c_str()));
    return true;
  }
  else
  {
    // invalid type
    ETG_TRACE_ERR(("LuaInterpreter::getLuaStackValue(string): index (%i), expected type string", inIdx));
  }

  return false;
}


const bool LuaInterpreter::getLuaStackValue(lua_State* inLState, const int inIdx, unsigned int* outValue)
{
  if (inIdx > lua_gettop (inLState))
  {
    // invalid index
    ETG_TRACE_ERR(("LuaInterpreter::getLuaStackValue(integer): invalid index (%i)", inIdx));
    return false;
  }

  if (LUA_TNUMBER == (lua_type (inLState, inIdx)))
  {
    *outValue = (unsigned int)lua_tonumber (inLState, inIdx);
    ETG_TRACE_USR4(("LuaInterpreter::getLuaStackValue(integer): index (%i), value (%i)", inIdx, *outValue));
    return true;
  }
  else
  {
    // invalid type
    ETG_TRACE_ERR(("LuaInterpreter::getLuaStackValue(integer): index (%i), expected type integer", inIdx));
  }

  return false;
}


const std::string& LuaInterpreter::luaThreadStatusToString(const int inThreadStatus)
{
  static const std::string ok("OK");
  static const std::string yield("YIELD");
  static const std::string errrun("ERRRUN");
  static const std::string errsyntax("ERRSYNTAX");
  static const std::string errmem("ERRMEM");
  static const std::string errgcmm("ERRGCMM");
  static const std::string errerr("ERRERR");
  static const std::string errfile("ERRFILE");
  static const std::string unknown("unknown");

  switch (inThreadStatus)
  {
    case LUA_OK:        return ok;        //break;
    case LUA_YIELD:     return yield;     //break;
    case LUA_ERRRUN:    return errrun;    //break;
    case LUA_ERRSYNTAX: return errsyntax; //break;
    case LUA_ERRMEM:    return errmem;    //break;
    case LUA_ERRGCMM:   return errgcmm;   //break;
    case LUA_ERRERR:    return errerr;    //break;
    case LUA_ERRFILE:   return errfile;   //break;
    default:
      ETG_TRACE_ERR(("LuaInterpreter::luaThreadStatusToString: unknown (%i)", inThreadStatus));
      return unknown;
      //break;
  }
}


const std::string& LuaInterpreter::luaHookEventToString(const int inHookEvent)
{
  static const std::string hookcall("CALL");
  static const std::string hookret("RET");
  static const std::string hooktailcall("TAILCALL");
  static const std::string hookline("LINE");
  static const std::string hookcount("COUNT");
  static const std::string unknown("unknown");

  switch (inHookEvent)
  {
    case LUA_HOOKCALL:      return hookcall;      //break;
    case LUA_HOOKRET:       return hookret;       //break;
    case LUA_HOOKTAILCALL:  return hooktailcall;  //break;
    case LUA_HOOKLINE:      return hookline;      //break;
    case LUA_HOOKCOUNT:     return hookcount;     //break;
    default:
      ETG_TRACE_ERR(("LuaInterpreter::luaHookEventToString: unknown (%i)", inHookEvent));
      return unknown;
      //break;
  }
}

/*************************************************************
*FUNCTION:    LuaInterpreter::bgimageUpdate
*DESCRIPTION  Background image details to be displayed on RecoveryHMI
*             The function is called with 3 parameters:
*             1 = Dir path of the image
*             2 = image name
*
*			 Image property will be updated
*
*PARAMETER    lua_State *lState
*             contains the parameter
*RETURNVALUE  gives back the count of return values
*             the method. In this case it is 1
*
*
*HISTORY 	  3-4-2015 RBEI/ECA1 Anusha G
*
*Initial Revision
**************************************************************/
int LuaInterpreter::bgimageUpdate (lua_State* ioLState)
{
  // lua function added for script compatibility to get background image details

  // get parameter imgDirPath and imgName from lua runtime
  std::string imgDirPath;
  std::string imgName;

  if (false == getLuaStackValue(ioLState, 1, imgDirPath))
  {
    imgDirPath = "";
  }

  if (false == getLuaStackValue(ioLState, 2, imgName))
  {
    imgName = "";
  }

  ETG_TRACE_USR4(("LuaInterpreter::bgimageUpdate imgDirPath : %s", imgDirPath.c_str()));
  ETG_TRACE_USR4(("LuaInterpreter::bgimageUpdate imgName : %s", imgName.c_str()));


  ai_sw_update::common::core::imp::AppProxy::bgImageDetails(imgDirPath,imgName);

  return 1;
}

/*************************************************************
*FUNCTION:    LuaInterpreter::writeIntoAdr3
*DESCRIPTION  Calls the ADR3 flash component
*             The function is called with 3 parameters:
*             1 = what to do (0/1/2=blocking_mode/non_blocking_mode/status)
*             2 = XLoader
*             3 = ADR3-File
*			  Calls in the backend one of the functions
*             dl_tclDABPlusMain::poGetInstance()->bRunSequence
*             dl_tclDABPlusMain::poGetInstance()->bRunADR3Thread
*             dl_tclDABPlusMain::u32GetStatusADR3
*             No-critical-method
*
*PARAMETER    lua_State *lState
*             contains the parameter
*RETURNVALUE  gives back the count of return values
*             the method. In this case it is 2
*
*
*HISTORY 	 19.09.13 CM-AI/PJ-CB38 Landsvogt
*
*Initial Revision
**************************************************************/
int LuaInterpreter::writeAdr3 (lua_State* ioLState)
{
	int returnData = 0;
// Removed the content because ADR3 is shifted to independent process (UpdateCtrl)
	lua_pushnumber (ioLState, (lua_Number) returnData); //@todo replace by wrapper
	lua_pushnumber (ioLState, METHOD_FAILURE); //@todo replace by wrapper
	
	ETG_TRACE_USR4(("LuaInterpreter::writeAdr3: %s","METHOD_FAILURE"));
  	UpdateLog::vUpdateLogTrace("writeAdr3 %s ", "METHOD_FAILURE");
	
  	abortDownload (ioLState,"writeAdr3 failed",SW_UPDATE_ERROR_TYPE_SCRIPT_EXECUTE);
	return 2;
}
/*************************************************************
*FUNCTION:    LuaInterpreter::fileAvailable
*DESCRIPTION  LuaInterpreter fileAvailable checks if a file is available
*PARAMETER    lua_State *lState contains the parameter
*RETURNVALUE  gives back the count of return values the method. In this case it is 1
*
*HISTORY   30.07.10 TMS CM-AI/PJ-GM54 Weimann
*
*Initial Revision
**************************************************************/
int LuaInterpreter::fileAvailable (lua_State* ioLState)
{
  bool isFileAvailable = false;

  std::string fileName;
  ai_sw_update::common::core::imp::AppProxy::dispatchText("","","fileAvailable");
  if (false == getLuaStackValue(ioLState, 1, fileName))
  {
    // no assert in case of non-critical Lua API
    ETG_TRACE_ERR(("LuaInterpreter::fileAvailable: error filename"));
  }

  isFileAvailable = ai_sw_update::common::fileAvailable(fileName.c_str());

  if (isFileAvailable)
  {
    ETG_TRACE_USR4(("LuaInterpreter::fileAvailable: file available"));
  }
  else
  {
    ETG_TRACE_USR4(("LuaInterpreter::fileAvailable: file unavailable"));
  }

  lua_pushnumber (ioLState, isFileAvailable ? METHOD_SUCCESS : METHOD_FAILURE);
  UpdateLog::vUpdateLogTrace("fileAvailable %s = %d", fileName.c_str(), isFileAvailable);

  return 1;
}


/*************************************************************
*FUNCTION:    dl_LuaHandler::writeMagic
*DESCRIPTION  write into Raw Nor area a flag
*       command to write into raw area.
*       Calls in the backend dl_tclFlashOperation
*       function
*
*PARAMETER    lua_State *lState
*       contains the parameter
*RETURNVALUE  gives back the count of return values
*       the method. In this case it is 1
*
*
*HISTORY   30.07.10 TMS CM-AI/PJ-GM54 Weimann
*
*Initial Revision
**************************************************************/
int LuaInterpreter::writeMagic (lua_State *ioLState)
{
  bool returnValue=false;
  bool paramAvail=true;

  ai_sw_update::common::core::imp::AppProxy::dispatchText("","","writeMagic");
  // get parameter adress from lua runtime
  unsigned int address = 0;
  if (false == getLuaStackValue(ioLState, 1, &address))
  {
    // missing parameter, fail the Lua script
    paramAvail=false;
  }

  // get parameter value from lua runtime
  unsigned int value = 0;
  if ((false == getLuaStackValue(ioLState, 2, &value)) && paramAvail)
  {
    // missing parameter, fail the Lua script
    paramAvail=false;
  }

  if(paramAvail)
  {
      // critical command execution, if returnValue is false, the lua script is aborted
      print2errmem("LuaInterpreter::writeMagic executed\n");
#ifndef VARIANT_S_FTR_ENABLE_G4G
      returnValue = (bool)ai_sw_update::common::bWriteMagicToRawFlash (static_cast< unsigned int >(address), static_cast< unsigned int >(value));
#else      
      returnValue = swu::BootChainAccess<swu::rcarBootChain>::writeMagic(static_cast<tU32>(address), static_cast<tU32>(value));
#endif
	  //returnValue = (bool)ai_sw_update::common::bWriteMagicToRawFlash (static_cast< unsigned int >(address), static_cast< unsigned int >(value));
	  ETG_TRACE_USR4(("LuaInterpreter::writeMagic: %s", returnValue ? "METHOD_SUCCESS" : "METHOD_FAILURE"));
	  UpdateLog::vUpdateLogTrace("writeMagic(0x%x, 0x%x) ", address, value);
  }
  else {
     ETG_TRACE_FATAL(("LuaInterpreter::writeMagic parameter reading Error"));
  }

  lua_pushnumber (ioLState, returnValue ? METHOD_SUCCESS : METHOD_FAILURE);

  // abort lua script in case of any of the failure
  if(returnValue == false)
  	abortDownload (ioLState,"bWriteMagicToRawFlash failed",SW_UPDATE_ERROR_TYPE_SCRIPT_EXECUTE);

  return 1;
}

int LuaInterpreter::writeMagicToActive (lua_State *ioLState) {

   bool returnValue=false;
   bool paramAvail=true;
   ai_sw_update::common::core::imp::AppProxy::dispatchText("","","writeMagicToActive");

   tU32 address = 0;
   if (false == getLuaStackValue(ioLState, 1, &address)){
      paramAvail=false;
   }

   tU32 value = 0;
   if ((false == getLuaStackValue(ioLState, 2, &value)) && paramAvail){      
      paramAvail=false;
   }

   if(paramAvail) {
      print2errmem("LuaInterpreter::writeMagicToActive executed\n");
      #ifdef VARIANT_S_FTR_ENABLE_G4G
      returnValue = swu::BootChainAccess<swu::rcarBootChain>::writeMagicToActiveSector(address, value);
      #else
      returnValue = false;
      #endif   
      ETG_TRACE_USR4(("LuaInterpreter::writeMagicToActive: %s", returnValue ? "METHOD_SUCCESS" : "METHOD_FAILURE"));
   } else {
      ETG_TRACE_FATAL(("LuaInterpreter::writeMagicToActive parameter reading Error"));
   }
   lua_pushnumber (ioLState, returnValue ? METHOD_SUCCESS : METHOD_FAILURE);

   if(returnValue == false) {
      abortDownload (ioLState,"writeMagicToActive failed",SW_UPDATE_ERROR_TYPE_SCRIPT_EXECUTE);
   }

  return 1;
}
/*************************************************************
*FUNCTION:    dl_LuaHandler::readMagic
*DESCRIPTION  read from Raw Nor area a magic
*       command to read from raw area.
*       Calls in the backend dl_tclFlashOperation
*       function
*
*PARAMETER    lua_State *lState
*       contains the parameter
*RETURNVALUE  gives back the count of return values
*       the method. In this case it is 1
*
*
*HISTORY   30.07.10 TMS CM-AI/PJ-GM54 Weimann
*
*Initial Revision
**************************************************************/
int LuaInterpreter::readMagic (lua_State *ioLState)
{
  // get parameter address from lua runtime
  unsigned int address = 0;
  bool returnValue = false;
  bool paramAvail = true;

  ai_sw_update::common::core::imp::AppProxy::dispatchText("","","readMagic");

  if (false == getLuaStackValue(ioLState, 1, &address))
  {
	paramAvail = false;
  }

  unsigned int value = 0; //< value storing the readout flag
  if(paramAvail)
  {
#ifndef VARIANT_S_FTR_ENABLE_G4G
    returnValue = (bool)bReadMagicFromRawFlash(static_cast< unsigned int >(address), value);
#else
    returnValue = swu::BootChainAccess<swu::rcarBootChain>::readMagic(static_cast<tU32>(address), value);
#endif
  }
  else
  {
	ETG_TRACE_FATAL(("LuaInterpreter::readMagic address reading error"));
	printf("LuaInterpreter::readMagic address reading error");
  }

  lua_pushnumber(ioLState, (lua_Number) value);
  lua_pushnumber(ioLState, returnValue ? METHOD_SUCCESS : METHOD_FAILURE);

  ETG_TRACE_USR4(("LuaInterpreter::readMagic: %s", returnValue ? "METHOD_SUCCESS" : "METHOD_FAILURE"));
  UpdateLog::vUpdateLogTrace("readMagic(0x%x, 0x%x) ", address, value);

  if(returnValue == false)
  	abortDownload (ioLState,"readMagic failed",SW_UPDATE_ERROR_TYPE_SCRIPT_EXECUTE);

  return 2;
}


/*************************************************************
*FUNCTION:    dl_LuaHandler::writeMagicBootChain
*DESCRIPTION  write into Raw Nor area a flag
*       command to write into raw area.
*       Calls in the backend dl_tclFlashOperation
*       function
*
*PARAMETER    lua_State *lState
*       contains the parameter
*RETURNVALUE  gives back the count of return values
*       the method. In this case it is 1
*
*
*HISTORY   30.07.10 TMS CM-AI/PJ-GM54 Weimann
*
*Initial Revision
**************************************************************/
int LuaInterpreter::writeMagicBootChain (lua_State *ioLState)
{
  // get parameter address from lua runtime
  unsigned int address = 0;
  bool paramAvail = true;
  bool returnValue = false;

  ai_sw_update::common::core::imp::AppProxy::dispatchText("","","writeMagicBootChain");
  if (false == getLuaStackValue(ioLState, 1, &address))
  {
	paramAvail = false;
  }

  // get parameter value from lua runtime
  unsigned int value = 0;

  if ((false == getLuaStackValue(ioLState, 2, &value)) && paramAvail)
  {
	paramAvail = false;
  }

  if(paramAvail)
  {
    print2errmem("LuaInterpreter::writeMagicBootChain executed\n");
#ifndef VARIANT_S_FTR_ENABLE_G4G
    returnValue = (bool)bWriteMagicBootChain(address, value);
#else
    returnValue = swu::BootChainAccess<swu::rcarBootChain>::writeMagic(address, value);
    returnValue = swu::BootChainAccess<swu::rcarBootChain>::switchBootSectors();
#endif
    ETG_TRACE_USR4(("LuaInterpreter::writeMagicBootChain: %s", returnValue ? "METHOD_SUCCESS" : "METHOD_FAILURE"));
    UpdateLog::vUpdateLogTrace("writeMagicBootChain(0x%x, 0x%x) ", address, value);
  }
  else
	  ETG_TRACE_ERR(("LuaInterpreter::writeMagicBootChain parameter read error"));

  lua_pushnumber(ioLState, returnValue ? METHOD_SUCCESS : METHOD_FAILURE);
  return 1;
}
/*************************************************************
*FUNCTION:    dl_LuaHandler::writeMagicBootChainFR
*DESCRIPTION  write into Raw Nor area a flag
*       command to write into raw area.
*       Calls in the backend dl_tclFlashOperation
*       function. Abort the lua script in case of failure.
*
*PARAMETER    lua_State *lState
*       contains the parameter
*RETURNVALUE  gives back the count of return values
*       the method. In this case it is 1
*
*
*HISTORY   20.08.14 TMS CM-AI/ECA2 Multani
*
*Initial Revision
**************************************************************/
int LuaInterpreter::writeMagicBootChainFR (lua_State *ioLState)
{
  // get parameter address from lua runtime
  unsigned int address = 0;
  bool paramAvail = true;
  bool returnValue = false;

  ai_sw_update::common::core::imp::AppProxy::dispatchText("","","_writeMagicBootChain");
  if (false == getLuaStackValue(ioLState, 1, &address))
  {
    paramAvail = false;
  }

  // get parameter value from lua runtime
  unsigned int value = 0;

  if ((false == getLuaStackValue(ioLState, 2, &value)) && paramAvail)
  {
    paramAvail = false;
  }

  if(paramAvail)
  {
    print2errmem("LuaInterpreter::writeMagicBootChain executed\n");
#ifndef VARIANT_S_FTR_ENABLE_G4G
    returnValue = (bool)bWriteMagicBootChain(address, value);
#else
    returnValue = swu::BootChainAccess<swu::rcarBootChain>::writeMagic(address, value);
    returnValue = swu::BootChainAccess<swu::rcarBootChain>::switchBootSectors();
#endif
    ETG_TRACE_USR4(("LuaInterpreter::writeMagicBootChain: %s", returnValue ? "METHOD_SUCCESS" : "METHOD_FAILURE"));
    UpdateLog::vUpdateLogTrace("writeMagicBootChain(0x%x, 0x%x) ", address, value);
  }
  else
    ETG_TRACE_FATAL(("LuaInterpreter::writeMagicBootChain parameter read error"));

  lua_pushnumber(ioLState, returnValue ? METHOD_SUCCESS : METHOD_FAILURE);

  // abort lua script in case of any of the failure
  if(returnValue == false)
  	abortDownload (ioLState,"writeMagicBootChainFR failed",SW_UPDATE_ERROR_TYPE_SCRIPT_EXECUTE);

  return 1;
}
/*************************************************************
*FUNCTION:    dl_LuaHandler::printLog
*DESCRIPTION  with printLog it is possible
*       to write into download log file
* SYNTAX      no parameter printLog(0, "bla bla")
*             n parameter  printLof (n, <the same as in printf>)
*
*PARAMETER    lua_State *lState - lua Stack
*RETURNVALUE  tS32
*       value is always 1
*
*HISTORY   30.07.10 TMS CM-AI/PJ-GM54 Weimann
*
*Initial Revision
**************************************************************/
int LuaInterpreter::printLog(lua_State *ioLState)
{
  // @todo: analyse Gen2 DL-Core implementation. Take over or make new concept.
  // lua function added for script compatibility
  ETG_TRACE_USR4(("LuaInterpreter::printLog ioLState=%p", ioLState));

  return 1;
}

/*************************************************************
*FUNCTION:    dl_LuaHandler::printTrace
*DESCRIPTION  with printTrace it is possible
*       to write into trace channel
* SYNTAX      no parameter printLog(0, "bla bla")
*             n parameter  printLof (n, <the same as in printf>)
*
*PARAMETER    lua_State *lState - lua Stack
*RETURNVALUE  tS32
*       value is always 1
*
*HISTORY   30.07.10 TMS CM-AI/PJ-GM54 Weimann
*
*Initial Revision
**************************************************************/
int LuaInterpreter::printDisplay(lua_State *ioLState)
{
  //@todo analyse printDisplayCommand implemetation from GM Gen2 DL-Core
  // lua function added for script compatibility

  // get parameter displayTest from lua runtime
  std::string displayText1;
  std::string displayText2;

  if (false == getLuaStackValue(ioLState, 1, displayText1))
  {
    displayText1 = "";
  }

  if (false == getLuaStackValue(ioLState, 2, displayText2))
  {
    displayText2 = "";
  }

  ai_sw_update::common::core::imp::AppProxy::dispatchText(displayText1,displayText2,"");

  ETG_TRACE_USR4(("LuaInterpreter::printDisplay: %s", displayText1.c_str()));
  ETG_TRACE_USR4(("LuaInterpreter::printDisplay: %s", displayText2.c_str()));

  return 1;
}


/*************************************************************
*FUNCTION:    dl_LuaHandler::waitTime
*DESCRIPTION  dl_LuaHandler waits for given mSeconds
*       internally with OSAL_s32ThreadWait operation
*
*PARAMETER    lua_State *lState
*       contains the parameter
*RETURNVALUE  gives back the count of return values
*       the method. In this case it is 1
*
*HISTORY   30.07.10 TMS CM-AI/PJ-GM54 Weimann
*
*Initial Revision
**************************************************************/
int LuaInterpreter::waitTime(lua_State *ioLState)
{
  // @todo: remove OSAL

  int waitingTime = 0;

  ai_sw_update::common::core::imp::AppProxy::dispatchText("","","waitTime");

  std::string waitTimeStringRepresentation;
  if (false == getLuaStackValue(ioLState, 1, waitTimeStringRepresentation))
  {
	// no assert, non-critical Lua API
	ETG_TRACE_ERR(("LuaInterpreter :: waitTime reading error : waitTimeStringRepresentation"));
  }


  if (OSAL_s32IsDigit (*(waitTimeStringRepresentation.c_str())))
  {
    waitingTime = OSAL_s32AsciiToS32 (waitTimeStringRepresentation.c_str());
  }

  ETG_TRACE_USR4(("LuaInterpreter::waitTime: wait %i ms", waitingTime));
  if (waitingTime > 0)
  {
    OSAL_s32ThreadWait ((OSAL_tMSecond) waitingTime);
  }

  return 1;
}
/*************************************************************
*FUNCTION:    dl_LuaHandler::remoteExecFR
*DESCRIPTION  Execute critically lua command
*
*PARAMETER    lua_State *lState
*       contains the parameter
*RETURNVALUE  gives back the count of return values
*       the method. In this case it is 1
*
*HISTORY   14.08.14 TMS CM-AI/ECA2 Multani
*
*Initial Revision
**************************************************************/

int LuaInterpreter::remoteExecFR(lua_State* ioLState)
{
   bool returnValue = false;
   std::string command;

   ai_sw_update::common::core::imp::AppProxy::dispatchText("","","_remoteExec");
   if (false == getLuaStackValue(ioLState, 1, command))
   {
	ETG_TRACE_FATAL(("LuaInterpreter :: remoteExec [critical], parameter missing "));
   }
   else
   {
      if (!SystemCall::hasError()) {
         returnValue = SystemCall::exec(command);
      } else {
         ETG_TRACE_USR4(("Will not execute script because of previous error: %s", command.c_str()));
      }
   }

   UpdateLog::vUpdateLogTrace("remoteExec %s ", command.c_str());
   if(!returnValue) {
      swu::LOG_FAILURE("command:%s", command.c_str());
   }

   // Simulate one error, if the file /tmp/errsim9.txt exists:
   if (bSwUpdateErrorSimulation())
   {
       ETG_TRACE_USR4(("LuaInterpreter::remoteExecFR ERROR_SIMULATION ACTIVE"));
       returnValue = false;
   }

   // Lua handling based on the execution of exec system call
   lua_pushnumber (ioLState, returnValue ? METHOD_SUCCESS : METHOD_FAILURE);
   // critical Lua command failure, abort the Lua script
   if(returnValue == false)
    	abortDownload (ioLState,command.c_str(),SW_UPDATE_ERROR_TYPE_SCRIPT_EXECUTE);

   return 1;
}

int LuaInterpreter::remoteExec(lua_State* ioLState)
{
   bool returnValue = false;
   std::string command;
   ai_sw_update::common::core::imp::AppProxy::dispatchText("","","_remoteExec");
   if (false == getLuaStackValue(ioLState, 1, command))
   {
	ETG_TRACE_ERR(("LuaInterpreter :: remoteExec [non-critical], parameter missing "));
   }
   else
   {
      if (!SystemCall::hasError()) {
         returnValue = SystemCall::exec(command, 0, false);
      } else {
         ETG_TRACE_USR4(("Will not execute script because of previous error: %s", command.c_str()));
      }
   }

   UpdateLog::vUpdateLogTrace("remoteExec %s ", command.c_str());
   if(!returnValue) {
      swu::LOG_FAILURE("command:%s", command.c_str());
   }

   // Simulate one error, if the file /tmp/errsim9.txt exists:
   if (bSwUpdateErrorSimulation())
   {
       ETG_TRACE_USR4(("LuaInterpreter::remoteExec ERROR_SIMULATION ACTIVE"));
       returnValue = false;
   }

   lua_pushnumber (ioLState, returnValue ? METHOD_SUCCESS : METHOD_FAILURE);

   // One return value is pushed on the stack:
   return 1;
}

/*************************************************************
*FUNCTION:    dl_LuaHandler::writeIntoRawNor
*DESCRIPTION  write file into Raw Nor area
*       command to write into raw area.
*       Calls in the backend dl_tclFlashOperation
*       function
*
*PARAMETER    lua_State *lState
*       contains the parameter
*RETURNVALUE  gives back the count of return values
*       the method. In this case it is 1
*
*
*HISTORY   30.07.10 TMS CM-AI/PJ-GM54 Weimann
*
*Initial Revision
**************************************************************/
int LuaInterpreter::writeRawNor(lua_State *ioLState)
{
  // get parameter 'address' from lua runtime
  unsigned int address = 0;
  bool returnValue=false;
  bool paramAvail=true;

  ai_sw_update::common::core::imp::AppProxy::dispatchText("","","writeRawNor");

  if (false == getLuaStackValue(ioLState, 1, &address))
  {
    ETG_TRACE_FATAL(("LuaInterpreter :: writeRawNor[critical], address read error"));
    paramAvail=false;
  }

  // get parameter 'filePath' from lua runtime
  std::string filePath;
  if ((false == getLuaStackValue(ioLState, 2, filePath)) && paramAvail)
  {
    ETG_TRACE_FATAL(("LuaInterpreter :: writeRawNor[critical], filepath read error"));
    paramAvail=false;
  }

  if(paramAvail)
  {
    print2errmem("LuaInterpreter::writeRawNor executed\n");
	  returnValue = (bool)bProgramDataToRawFlash(
	      filePath.c_str(),
	      (off_t)address,  //@todo: check whether conversion works and check also underlying implementation
	      TRUE); // Erase flash sectors before programming

	  ETG_TRACE_USR4(("LuaInterpreter::writeRawNor: %s", returnValue ? "METHOD_SUCCESS" : "METHOD_FAILURE"));
	  UpdateLog::vUpdateLogTrace("writeRawNor(0x%x,%s) ", address, filePath.c_str());
  }


  lua_pushnumber(ioLState, returnValue ? METHOD_SUCCESS : METHOD_FAILURE);
  if(returnValue == false)
  	abortDownload (ioLState,"writeRawNor",SW_UPDATE_ERROR_TYPE_SCRIPT_EXECUTE);

  return 1;
}

/*************************************************************
*FUNCTION:    LuaInterpreter::enableDebug
*DESCRIPTION  enables the debug interface
*       by opening Putty and TTFIS-Ethernet
*
*PARAMETER    lua_State *ioLState
*       contains the parameter
*RETURNVALUE  gives back the count of return values
*       the method. In this case it is 1
*
*
*HISTORY   2014-05-22 CM-AI/ECA2-Stautz
*
*Initial Revision
**************************************************************/
tS32 LuaInterpreter::enableDebug (lua_State *ioLState)
{
  
  tCString method = "LuaInterpreter::enableDebug";
  ai_sw_update::common::core::imp::AppProxy::dispatchText("","","enableDebug");

  OSAL_tMSecond measuringBegin = 0;
  OSAL_tMSecond measuringEnd   = 0;
  ETG_TRACE_USR4 (("%s was entered", method));

  if (!ioLState){
    NORMAL_M_ASSERT_ALWAYS ();
    return 1;
  }
  //start counting
  measuringBegin = OSAL_ClockGetElapsedTime ();
  //show lock string on display
  //printDisplay (lockInterfaceStr);

  // Call a script to open the debug interfaces
  SystemCall::execCommand("/var/opt/bosch/static/security/locking/unlock.sh");
  measuringEnd = OSAL_ClockGetElapsedTime() - measuringBegin;
  UpdateLog::vUpdateLogTrace("enableDebug, duration=%d", measuringEnd);

  lua_pushnumber (ioLState, METHOD_SUCCESS);

  ETG_TRACE_USR4 (("%s was left", method));
  return 1;
}

/*************************************************************
*FUNCTION:    LuaInterpreter::disableDebug
*DESCRIPTION  disables the debug interface
*       by closing Putty and TTFIS-Ethernet
*
*PARAMETER    lua_State *ioLState
*       contains the parameter
*RETURNVALUE  gives back the count of return values
*       the method. In this case it is 1
*
*
*HISTORY   2014-05-22 CM-AI/ECA2-Stautz
*
*Initial Revision
**************************************************************/
tS32 LuaInterpreter::disableDebug (lua_State *ioLState)
{  
  tCString method = "LuaInterpreter::disableDebug";  

  ai_sw_update::common::core::imp::AppProxy::dispatchText("","","disableDebug");
  OSAL_tMSecond measuringBegin = 0;
  OSAL_tMSecond measuringEnd   = 0;

  ETG_TRACE_USR4 (("%s was entered", method));
  if (!ioLState)
  {
    NORMAL_M_ASSERT_ALWAYS ();
    return 1;
  }
  
  //start counting
  measuringBegin = OSAL_ClockGetElapsedTime ();
  
  // Call a script to open the debug interfaces
  SystemCall::execCommand("/var/opt/bosch/static/security/locking/lock.sh");

  //measure the time
  measuringEnd = OSAL_ClockGetElapsedTime() - measuringBegin;
  UpdateLog::vUpdateLogTrace("disableDebug, duration=%d", measuringEnd);
  lua_pushnumber (ioLState, METHOD_SUCCESS);

  ETG_TRACE_USR4 (("%s was left", method));
  return 1;
}

/*************************************************************
*FUNCTION:    LuaInterpreter::vinDelete
*DESCRIPTION  Deletes the files for VehicleInformationNumber
*        to allow the installation of the target in a different car
*
*PARAMETER    lua_State *ioLState
*       contains the parameter
*RETURNVALUE  gives back the count of return values
*       the method. In this case it is 1
*
*
*HISTORY   2014-05-22 CM-AI/ECA2-Stautz
*
*Initial Revision
**************************************************************/
tS32 LuaInterpreter::vinDelete (lua_State *ioLState)
{
	tCString methodName   = "LuaInterpreter::vinDelete";
	OSAL_tMSecond measuringBegin = 0;
	OSAL_tMSecond measuringEnd = 0;	

	ETG_TRACE_USR4(("%s",methodName));
	ETG_TRACE_USR4(("was entered\n"));

    ai_sw_update::common::core::imp::AppProxy::dispatchText("","","vinDelete");
	/*pathological cases */
	if (!ioLState)
	{
		NORMAL_M_ASSERT_ALWAYS();
		return 1;
	}
	//start counting
	measuringBegin = OSAL_ClockGetElapsedTime ();
    //show unlock string on display
    //printDisplay (vinDeleteStr);

	// Delete the files which contain the VIN in the old directory:
    unlink("/var/opt/bosch/dynamic/ffs/lmm_pers/FC_SPM_CRY.dat");
    unlink("/var/opt/bosch/dynamic/ffs/lmm_pers/FC_SPM_CMM.dat");
    unlink("/var/opt/bosch/dynamic/ffs/lmm_pers/SPMVH.BIN");

	// Delete the files which contain the VIN in the new directory:
    unlink("/var/opt/bosch/persistent/lmm_pers/FC_SPM_CRY.dat");
    unlink("/var/opt/bosch/persistent/lmm_pers/FC_SPM_CMM.dat");
    unlink("/var/opt/bosch/persistent/lmm_pers/SPMVH.BIN");

    //stop measuring
	measuringEnd = OSAL_ClockGetElapsedTime() - measuringBegin;

	UpdateLog::vUpdateLogTrace("vinDelete, duration=%d", measuringEnd);
    lua_pushnumber (ioLState, METHOD_SUCCESS);
	ETG_TRACE_USR4 (("%s was left", methodName));

	return 1;
}


#if 0
    // Example from old script.lua:
	remoteExec("echo \"Remove SPM security files\" >> /tmp/downloadPipe");
	remoteExec("umount /dev/"..MMCBLK_X.."p6");
	_remoteExec("mkdir -p /ext/var/opt/bosch/dynamic");
	remoteExec("echo \"Mounting /dev/"..MMCBLK_X.."p6...\" >> /tmp/downloadPipe");
	_remoteExec("mount -t ext4 /dev/"..MMCBLK_X.."p6 /ext/var/opt/bosch/dynamic");
	remoteExec("echo \"Remove VIN Info if available...\" >> /tmp/downloadPipe");
	if fileAvailable ("/ext/var/opt/bosch/dynamic/ffs/lmm_pers/FC_SPM_CRY.dat") == 1 then
  		remoteExec("echo \"Remove FC_SPM_CRY.dat...\" >> /tmp/downloadPipe");
  		remoteExec("rm -f /ext/var/opt/bosch/dynamic/ffs/lmm_pers/FC_SPM_CRY.dat");
  		remoteExec("echo \"FC_SPM_CRY.dat is removed...\" >> /tmp/downloadPipe");
	end
	if fileAvailable ("/ext/var/opt/bosch/dynamic/ffs/lmm_pers/SPMVH.BIN") == 1 then
  		remoteExec("echo \"Remove SPMVH.BIN...\" >> /tmp/downloadPipe");
  		remoteExec("rm -f /ext/var/opt/bosch/dynamic/ffs/lmm_pers/SPMVH.BIN");
  		remoteExec("echo \"SPMVH.BIN is removed...\" >> /tmp/downloadPipe");
	end
	remoteExec("sync");
	waitTime("3000");
	remoteExec("umount /dev/"..MMCBLK_X.."p6");

    // Example from old download core:
	//tCString FC_SPM_CMM_DAT = "/dev/ffs2/lmm_pers/FC_SPM_CMM.dat";
	//tCString SPMVH_BIN      = "/dev/ffs2/lmm_pers/SPMVH.BIN";
	//tCString FC_SPM_CRY_DAT = "/dev/ffs2/lmm_pers/FC_SPM_CRY.dat";
    //process the file
    //delete two files only in the case
    //of Bosch certificate in target
    //if (dl_tclCoreUtilities::isBoschDevice (dl_LuaHandler::certType))
    //{
    //    ETG_TRACE_USR4 (("start vin deletion commands"));
    //    returnValue = dl_tclFlashOperations::bRemoveFileFromFS (FC_SPM_CMM_DAT);
    //    returnValue = dl_tclFlashOperations::bRemoveFileFromFS (SPMVH_BIN);
    //    returnValue = dl_tclFlashOperations::bRemoveFileFromFS (FC_SPM_CRY_DAT);
    //    ETG_TRACE_USR4 (("end vin deletion commands"));
    //}
    //else
    //{
    //    pushAndCheckReturnValueOnStack (ioLState, FALSE);
    //    return 1;
    //}
#endif

/*************************************************************
*FUNCTION:    LuaInterpreter::setError
*DESCRIPTION  set the error for the ErrorHandler, update.log and DTC
*
*PARAMETER    lua_State *lState
*       contains the parameter
*RETURNVALUE  gives back the count of return values
*       the method. In this case it is 1
*
*
*HISTORY   30.07.14 CM-AI/ECA2-Stautz
*
*Initial Revision
**************************************************************/
int LuaInterpreter::setError(lua_State *ioLState)
{
  // get parameter 'address' from lua runtime
  unsigned int iErrorDevice = 0;
  unsigned int iErrorType   = 0;
  ai_sw_update::common::core::imp::AppProxy::dispatchText("","","setError");

  if (false == getLuaStackValue(ioLState, 1, &iErrorDevice))
  {
	ETG_TRACE_ERR(("LuaInterpreter :: setError [non-critical] iErrorDevice read error "));
  }

  if (false == getLuaStackValue(ioLState, 2, &iErrorType))
  {
	ETG_TRACE_ERR(("LuaInterpreter :: setError [non-critical] iErrorType read error "));
  }
  
  // get parameter 'errorText' from lua runtime
  std::string errorText;
  if (false == getLuaStackValue(ioLState, 3, errorText))
  {
	ETG_TRACE_ERR(("LuaInterpreter :: setError [non-critical] errorText read error "));
  }
  // Set software update error
  iSetSwUpdateError((int)iErrorDevice,
                    (int)iErrorType,
                    errorText.c_str(),
                    "LuaInterpreter::setError");
  ETG_TRACE_USR4(("LuaInterpreter::setError: %s", "METHOD_SUCCESS"));
  lua_pushnumber(ioLState, METHOD_SUCCESS);

  return 1;
}

/*************************************************************
*FUNCTION:    LuaInterpreter::setErrorFR
*DESCRIPTION  set the error for the ErrorHandler, update.log and DTC
*	      critical version, unconditionally aborts the lua script.
*PARAMETER    lua_State *lState
*       contains the parameter
*RETURNVALUE  gives back the count of return values
*       the method. In this case it is 1
*
*
*HISTORY   14.08.14 CM-AI/ECA2-Multani
*
*Initial Revision
**************************************************************/
int LuaInterpreter::setErrorFR(lua_State *ioLState)
{
  // get parameter 'address' from lua runtime
  unsigned int iErrorDevice = 0;
  unsigned int iErrorType   = 0;

  // abort download in any case  
  bool paramAvail = true;
  
  ai_sw_update::common::core::imp::AppProxy::dispatchText("","","_setError");
  if (false == getLuaStackValue(ioLState, 1, &iErrorDevice))
  {
  	ETG_TRACE_FATAL(("LuaInterpreter :: setError [critical] iErrorDevice read error "));
	paramAvail = false;
  }
  if ((false == getLuaStackValue(ioLState, 2, &iErrorType)) && paramAvail)
  {
	ETG_TRACE_FATAL(("LuaInterpreter :: setError [critical] iErrorType read error "));
	paramAvail = false;
  }
  // get parameter 'errorText' from lua runtime
  std::string errorText;
  if ((false == getLuaStackValue(ioLState, 3, errorText)) && paramAvail)
  {
	ETG_TRACE_FATAL(("LuaInterpreter :: setError [critical] errorText read error "));
	paramAvail=false;
  }

  if(paramAvail)
  {
  	// Set software update error
	iSetSwUpdateError((int)iErrorDevice,
                      (int)iErrorType,
                      errorText.c_str(),
                      "LuaInterpreter::setError");
  }
  else
  {
	ETG_TRACE_FATAL(("LuaInterpreter:Parameter unavialble in Lua command set_error "));
  }
  lua_pushnumber(ioLState, METHOD_SUCCESS);
  // abort the download as critically lua command explicit abort received
  abortDownload (ioLState,"setErrorFR",SW_UPDATE_ERROR_TYPE_SCRIPT_EXECUTE);
  return 1;
}

/*************************************************************
*FUNCTION:    LuaInterpreter::swu_setError
*DESCRIPTION  set the error for the ErrorHandler, update.log and DTC
*
*PARAMETER    lua_State *lState
*       contains the parameter
*RETURNVALUE  gives back the count of return values
*       the method. In this case it is 1
*
*
*HISTORY   05.08.14 CM-AI/ECA2-Stautz
*
*Initial Revision
**************************************************************/
int LuaInterpreter::swu_setError(lua_State *ioLState)
{
  // get parameter 'address' from lua runtime
  std::string str_errorType;
  std::string str_errorDevice;
  std::string str_errorText; 
  bool paramAvail = true;
  std::string acText2;
  std::string displayLine1;
  std::string displayLine2;

  if (false == getLuaStackValue(ioLState, 1, str_errorType))
  {
	ETG_TRACE_ERR(("LuaInterpreter:: swu_setError[non-critical] errorType read error"));
        str_errorType="UNKNOWN_ERROR";
  }

  if ((false == getLuaStackValue(ioLState, 2, str_errorDevice)) && paramAvail)
  {
	ETG_TRACE_ERR(("LuaInterpreter:: swu_setError[non-critical] errorDevice read error"));
        str_errorDevice="UNKNOWN";
  }

  // get parameter 'filePath' from lua runtime
  if ((false == getLuaStackValue(ioLState, 3, str_errorText)) && paramAvail)
  {
	ETG_TRACE_ERR(("LuaInterpreter:: swu_setError[non-critical] errText read error"));
        str_errorText="UNKNOWN_ERROR";
  }

	  
  acText2.assign("LuaInterpreter::swu_setError=");	  
  acText2+=str_errorType;
  
  displayLine1.assign(" Download Error, Error Type = ");
  displayLine1+=str_errorType;    
  
  displayLine2.assign(" Error Device = ");	
  displayLine2+=str_errorDevice;
  
  ai_sw_update::common::core::imp::AppProxy::dispatchText(displayLine1,displayLine2,"swu_setError");
  // Set software update error
  iSetSwUpdateErrorString(str_errorDevice.c_str(),
                          str_errorType.c_str(),
                          str_errorText.c_str(),
                          acText2.c_str());


  ETG_TRACE_USR4(("LuaInterpreter::setError: %s", "METHOD_SUCCESS" ));
  lua_pushnumber(ioLState, METHOD_SUCCESS);

  return 1;
}

/*************************************************************
*FUNCTION:    LuaInterpreter::swu_setErrorFR
*DESCRIPTION  set the error for the ErrorHandler, update.log and DTC
*             critical version, unconditionally aborts the execution.
*
*PARAMETER    lua_State *lState
*       contains the parameter
*RETURNVALUE  gives back the count of return values
*       the method. In this case it is 1
*
*
*HISTORY   14.08.14 CM-AI/ECA2-Multani
*
*Initial Revision
**************************************************************/
int LuaInterpreter::swu_setErrorFR(lua_State *ioLState)
{
  // get parameter 'address' from lua runtime
  std::string str_errorType;
  std::string str_errorDevice;
  std::string str_errorText; 
  bool paramAvail = true;
  
  std::string acText2;
  std::string displayLine1;
  std::string displayLine2;

  if (false == getLuaStackValue(ioLState, 1, str_errorType))
  {
	ETG_TRACE_FATAL(("LuaInterpreter:: swu_setError[critical] errorType read error"));
        str_errorType="UNKNOWN_ERROR";
  }

  if ((false == getLuaStackValue(ioLState, 2, str_errorDevice)) && paramAvail)
  {
	ETG_TRACE_FATAL(("LuaInterpreter:: swu_setError[critical] errorDevice read error"));
        str_errorDevice="UNKNOWN";
  }

  // get parameter 'filePath' from lua runtime
  if ((false == getLuaStackValue(ioLState, 3, str_errorText)) && paramAvail)
  {
	ETG_TRACE_FATAL(("LuaInterpreter:: swu_setError[critical] errText read error"));
        str_errorText="UNKNOWN_ERROR";
  }

  acText2.assign("LuaInterpreter::swu_setError=");
  acText2+=str_errorType;

  displayLine1.assign(" Download Aborted, Error Type = ");
  displayLine1+=str_errorType;
  displayLine2.assign(" Error Device = ");
  displayLine2+=str_errorDevice;

  // Set software update error
  if(paramAvail){
	  iSetSwUpdateErrorString(str_errorDevice.c_str(),
        	                  str_errorType.c_str(),
                	          str_errorText.c_str(),
                        	  acText2.c_str());

  	ai_sw_update::common::core::imp::AppProxy::dispatchText(displayLine1,displayLine2,"_swu_setError");
  }
  else
  {
	ETG_TRACE_FATAL(("\n LuaInterpreter::parameter unavailable in lua command swu_setErrorFR"));
  }

  ETG_TRACE_USR4(("LuaInterpreter::setError: %s", "METHOD_SUCCESS"));

  lua_pushnumber(ioLState, METHOD_SUCCESS);
  abortDownload (ioLState,"setErrorFR",SW_UPDATE_ERROR_TYPE_SCRIPT_EXECUTE);

  return 1;
}
/*******************************************************************************************************/
void LuaInterpreter::luaHook(lua_State *ioLState, lua_Debug *ioLDebug)
{
  // check type of event that triggered the hook
  switch (ioLDebug->event)
  {
    case LUA_HOOKCALL: // fall through
    case LUA_HOOKRET:
      // 'n': fills in the field name and namewhat;
      // 'S': fills in the fields source, short_src, linedefined, lastlinedefined, and what;
      // 'l': fills in the field currentline;
      if(0 == lua_getinfo(ioLState, "Snl", ioLDebug))
      {
        // error case
        ETG_TRACE_ERR(("LuaInterpreter::luaHook: lua_getinfo"));
        // @todo assert
      }
      else
      {
        /*ETG_TRACE_USR4(("LuaInterpreter::luaHook: event (%10s)1: what (%20s), currentline (%i), name (%20s), namewhat (%s)",
            luaHookEventToString(ioLDebug->event).c_str(),
            ioLDebug->what,
            ioLDebug->currentline,
            ioLDebug->name == 0 ? "unknown":ioLDebug->name,
            ioLDebug->namewhat));

        ETG_TRACE_USR4(("LuaInterpreter::luaHook: event (%10s)2: source (%20s), short_src (%20s), startline (%i), endline (%i)",
            luaHookEventToString(ioLDebug->event).c_str(),
            ioLDebug->source,
            ioLDebug->short_src,
            ioLDebug->linedefined,
            ioLDebug->lastlinedefined));*/
      }
      break;

    case LUA_HOOKLINE:
      // diabled trace due to big trace output
      // ETG_TRACE_USR4(("LuaInterpreter::luaHook: event (LINE): %i", ioLDebug->currentline));
	  // UpdateLog::vUpdateLogTrace("LuaInterpreter::luaHook currentline=%d ",ioLDebug->currentline);
      break;

    default:
      break;
  }
}

//@todo: check parameter 'luaErrorCode' usage
void LuaInterpreter::abortDownload (lua_State *lState, const char *command, enum SwUpdateErrorType luaErrorCode)
{
  tCString message = "LuaInterpreter::abortDownload was entered\n";
  printf("%s\n", message);
  // for the time being only one error type
	if(command != NULL)
    luaL_error(lState, "%s\n", command);
  else
    luaL_error(lState, "%s\n", "exec failed");
  ETG_TRACE_USR4(("LuaInterpreter::abortDownload luaErrorCode=%d", luaErrorCode));

}

int LuaInterpreter::traceback_handler(lua_State *lState)
{
#if LUA_VERSION_NUM < 502
	lua_getfield(lState, LUA_GLOBALSINDEX, "debug");
	lua_getfield(lState, -1, "traceback");
	lua_pushvalue(lState, 1);
	lua_pushinteger(lState, 2);
	lua_call(lState, 2, 1);
#else
	luaL_traceback(lState, lState, lua_tostring(lState, 1), 0);
#endif
	return 1;
}

void LuaInterpreter::luaInitEnv(const char *saveEnvName)
{
	envVar.assign(saveEnvName);
    env_map.clear();
}

void LuaInterpreter::luaSaveField(const char *key, const char *value)
{
	env_map.insert(pair<std::string, std::string>(key, value));
}

void LuaInterpreter::luaSaveEnv()
{
	 lua_newtable(_LState);
	 for (std::map<std::string,std::string>::iterator it=env_map.begin(); it!=env_map.end(); ++it)
	 {
		lua_pushstring(_LState,it->first.c_str());
		lua_pushstring(_LState,it->second.c_str());
	    lua_settable(_LState, -3);
	 }
	 lua_setglobal(_LState, envVar.c_str());
}
}// namespace common
} // namespace ai_sw_update
