/*************************************************************************
* FILE:         UpdateLog.cppUpdateLogFileName
* PROJECT:      Gen3
* SW-COMPONENT: SW Update
*----------------------------------------------------------------------
*
* DESCRIPTION:  Gen3 Software Update Logging
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author        | Modification
*
* 2014-03-04  W.Stautz		  01.00
* 2014-08-27 Neera.K.Ponnappa,Apoorva K R  Modified for :CMG3G-5311  IS1604_Optimize Logging (update.log)@V850 Update Ctrl
                                                         CMG3G-5318  IS1604_Optimize Logging (update.log)@Teseo Update Ctrl
**************************************************************************/

// @todo:
// - improve tracing
// - remove OSAL
// - refactor all implementations here in a second step
// - use filesystem functions from fcswupdate, they need to be relocated from the VAG MIB to the common part

// ----------------------------------------------------------------------------
// includes
// ----------------------------------------------------------------------------

#include "UpdateLog.h"

#include <errno.h>
//#include <stdio.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <dirent.h>
#include <fcntl.h>
#include <unistd.h>
#include <string.h>

#include "swupd_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
 #define ETG_DEFAULT_TRACE_CLASS 		TR_CLASS_SWUPDATE_BASE
 #include "trcGenProj/Header/UpdateLog.cpp.trc.h"
#endif


namespace ai_sw_update {
namespace common {

// ----------------------------------------------------------------------------
// data definition
// ----------------------------------------------------------------------------

#define DL_BOSCH      "/var/opt/bosch/"
#define DL_BOSCH_PERS "persistent/"
#define DL_MMC        "/var/opt/bosch/persistent/"
#define SWUPDATE_PERS "swupdate/"
#define SWUPDATE_PERS_OLD "sw_update/"
#define SWUPDATE_PERS_LOG "swupdate/log"
#define DL_UPDATE_LOG_FILE "update.log"
#define DL_VERIFY_LOG_FILE "verify.log"
//#define DL_OLD_UPDATE_LOG  DL_MMC SWUPDATE_PERS DL_UPDATE_LOG_FILE
#define DL_OLD_VERIFY_LOG  DL_MMC SWUPDATE_PERS DL_VERIFY_LOG_FILE

#define DL_UPDATE_LOG      DL_MMC SWUPDATE_PERS DL_UPDATE_LOG_FILE
#define DL_VERIFY_LOG      DL_MMC SWUPDATE_PERS DL_VERIFY_LOG_FILE
//#define DL_UPDATE_LOG_OLD "/var/opt/bosch/persistent/sw_update/last_update.log"

#define DL_UPDATE_LOG_FILE_MAX_SIZE (200 * 1024)
//#define DL_UPDATE_LOG_FILE_MAX_SIZE (20 * 1024)    // Only for test

#define DL_UPDATE_LOG_CUT_SIZE (DL_UPDATE_LOG_FILE_MAX_SIZE / 2)
#define DL_UPDATE_MARK        "---------------------------------------------\n"
#define DL_UPDATE_MARK_CUTOFF "------------------ cutoff -------------------\n"

#define DEFAULT_MODE      (S_IRWXU | S_IRWXG |  S_IROTH | S_IWOTH)

tInt UpdateLog::UpdateLogFileTrace = OSAL_ERROR;
tInt UpdateLog::verifyLogFileTrace = OSAL_ERROR;
UpdateLog *UpdateLog::oneInstance = NULL;
tString UpdateLog::updateLogFile = const_cast<tString> (DL_UPDATE_LOG);
tString UpdateLog::verifyLogFile = const_cast<tString> (DL_VERIFY_LOG);
char UpdateLog::cUpdateLogFilePath[DL_FILE_PATH_MAX] = DL_UPDATE_LOG;

static pthread_mutex_t mMutexLogTrace = PTHREAD_MUTEX_INITIALIZER;
static pthread_mutex_t mMutexLogOpen = PTHREAD_MUTEX_INITIALIZER;
// ----------------------------------------------------------------------------
// functions
// ----------------------------------------------------------------------------


/*************************************************************
*FUNCTION:    getInstance
*DESCRIPTION  the function return the only one
*             object of class UpdateLog
*			  by calling of the constructor
*
*PARAMETER    U32 devEnvironmentType
*				one can decide if we are working with
*				target environment or pc environment
*
*RETURNVALUE the static UpdateLog* instance
*
*HISTORY 	 30.07.10 TMS CM-AI/PJ-GM54 Weimann
*
*Initial Revision
**************************************************************/
UpdateLog* UpdateLog::vGetInstance (tU32 devEnvironmentType)
{
	if (NULL == UpdateLog::oneInstance)
	{
		UpdateLog::oneInstance = OSAL_NEW UpdateLog (devEnvironmentType);
	}

	return oneInstance;
}
/*************************************************************
*FUNCTION:    vSetLogFile
*DESCRIPTION  This function sets the file name and path for logging V850 and Teseo
              component traces.
*
*PARAMETER    tCString
*
*RETURNVALUE  tVoid
*
*HISTORY 	 27.08.2014  Neera.K.Ponnappa , Apoorva K R
*
*Initial Revision
**************************************************************/
tVoid UpdateLog::vSetLogFile (tCString cStrLogFileName)
{
	if(NULL != cStrLogFileName)
	{
		memset(UpdateLog::cUpdateLogFilePath, 0, DL_FILE_PATH_MAX);
		strcpy(UpdateLog::cUpdateLogFilePath, DL_MMC);
		strcat(UpdateLog::cUpdateLogFilePath, SWUPDATE_PERS);
      if ((strlen(UpdateLog::cUpdateLogFilePath) + strlen(cStrLogFileName)) < DL_FILE_PATH_MAX)
      {         
         strcat(UpdateLog::cUpdateLogFilePath, cStrLogFileName);
      }
      else
      {
         strcat(UpdateLog::cUpdateLogFilePath, "swupd.log");
         ETG_TRACE_USR4 (("%s LogFile name too long ERROR", cStrLogFileName));
         NORMAL_M_ASSERT_ALWAYS();
      }
      UpdateLog::updateLogFile = cUpdateLogFilePath;
	}

}

/*************************************************************
*FUNCTION:    Constructor dl_DownnloadLog
*DESCRIPTION  the only constructor
*
*PARAMETER    U32 devEnvironmentType
*				one can decide if we are working with
*				target environment or pc environment
*
*RETURNVALUE the object
*
*HISTORY 	 30.07.10 TMS CM-AI/PJ-GM54 Weimann
*
*Initial Revision
*            27.08.2014  Neera.K.Ponnappa : Implemented Update Log for V850 Component
**************************************************************/
UpdateLog::UpdateLog (tU32 devEnvironmentType)
{
	    ETG_TRACE_USR3(("UpdateLog::UpdateLog devEnvironmentType=%d ", devEnvironmentType));
		UpdateLog::updateLogFile = const_cast<tString> (cUpdateLogFilePath);
		UpdateLog::verifyLogFile = const_cast<tString> (DL_VERIFY_LOG);
}

/*************************************************************
*FUNCTION:    UpdateLog::vUpdateLogOpen
*DESCRIPTION  opens the update log file
*
*PARAMETER
*RETURNVALUE tVoid
*
*HISTORY 	 30.07.10 TMS CM-AI/PJ-GM54 Weimann
*
*					Initial Revision
*            29.11.13 Aditya Kumar Jha (RBEI/ECG4)
*				Replaced OSAL function with linux native function.
**************************************************************/
tVoid UpdateLog::vUpdateLogOpen ()
{
	//pthread_mutex_init(&mMutexLogOpen, NULL);
	pthread_mutex_lock(&mMutexLogOpen);
	ETG_TRACE_USR3(("UpdateLog::vUpdateLogOpen entered"));
	UpdateLog::UpdateLogFileTrace = OSAL_ERROR;

	// Delete old log file
    //remove(DL_OLD_UPDATE_LOG);

	//create directory if not available
	bMkdirInFS (DL_BOSCH, DL_BOSCH_PERS ); // "/var/opt/bosch/persistent/"
	bMkdirInFS (DL_MMC,   SWUPDATE_PERS ); // "/var/opt/bosch/persistent/swupdate/"
	bMkdirInFS (DL_MMC,   SWUPDATE_PERS_OLD ); // "/var/opt/bosch/persistent/sw_update/"
	bMkdirInFS (SWUPDATE_PERS,   SWUPDATE_PERS_LOG ); // "/var/opt/bosch/persistent/swupdate/log/"

	// get the current file size!
	long int fileSize = dl_getFileSize (UpdateLog::updateLogFile);
	// is the file size bigger then 2 * DL_UPDATE_LOG_FILE_MAX_SIZE
	if (fileSize > (2 * DL_UPDATE_LOG_FILE_MAX_SIZE))
	{
		//print it to TTFIS TRACE
		ETG_TRACE_USR4 (("remove update.log file because it has exceed %d Bytes", 2 * DL_UPDATE_LOG_FILE_MAX_SIZE));
    	//remove the update.log
		if (0 != remove(UpdateLog::updateLogFile)) 
      {
         ETG_TRACE_USR4 (("remove updateLogFile not possible"));         
      }   
	}

	// Check whether the log file is already existing
	ETG_TRACE_USR4 (("open update Log file %s", UpdateLog::updateLogFile));
	UpdateLog::UpdateLogFileTrace = open(UpdateLog::updateLogFile, O_RDWR, S_IRUSR | S_IRGRP | S_IROTH);
	ETG_TRACE_USR4 (("update Log descriptor %d", UpdateLog::UpdateLogFileTrace));

	if (OSAL_ERROR == UpdateLog::UpdateLogFileTrace)
	{
		// Log file is not existing: create it
		// UpdateLog::UpdateLogFileTrace = OSAL_IOCreate(UpdateLog::updateLogFile, OSAL_EN_READWRITE);
		UpdateLog::UpdateLogFileTrace = open(UpdateLog::updateLogFile, O_RDWR | O_CREAT, S_IRUSR | S_IRGRP | S_IROTH);
		if (OSAL_ERROR ==  UpdateLog::UpdateLogFileTrace)
		{
			ETG_TRACE_USR3(("file %s could not be created", UpdateLog::updateLogFile));
		}
	}
	else
	{
		// Log file is already existing
	}

	//jha1kor: Close update file if opend
	if (OSAL_ERROR != UpdateLog::UpdateLogFileTrace)
	{
		close(UpdateLog::UpdateLogFileTrace);
		UpdateLog::UpdateLogFileTrace = OSAL_ERROR;
	}

	/* is the file size bigger then DL_DOWNLOAD_LOG_FILE_MAX_SIZE */
	if ( needToCutOff () )
	{
		/* print it to TTFIS TRACE */
		ETG_TRACE_USR4 (("Trim update.log file because it has exceed %d Bytes", DL_UPDATE_LOG_FILE_MAX_SIZE));
		/* trim the update.log */
		cutOff();
	}

	ETG_TRACE_USR3(("UpdateLog::vUpdataLogOpen left"));
	pthread_mutex_unlock(&mMutexLogOpen);
	//pthread_mutex_destroy(&mMutexLogOpen);
	return;
}

tBool UpdateLog::needToCutOff ()
{
    tCString method = "UpdateLog::needToCutOff";
    ETG_TRACE_USR4 (("%s was entered", method));
    return (dl_getFileSize(UpdateLog::updateLogFile) > DL_UPDATE_LOG_FILE_MAX_SIZE);
}

tVoid UpdateLog::cutOff ()
{
  tCString method = "UpdateLog::cutOff ()";
  ETG_TRACE_USR4 (("%s was entered", method));

  tS8  *s8NewUpdateLog = NULL; //buffer for the file to read into
  FILE *fd = NULL;

  // pathological case
  long int fileSize = 0;
  if ((fileSize = dl_getFileSize (cUpdateLogFilePath)) == OSAL_ERROR)
  {
    ETG_TRACE_ERR (("getFileSize of %s is not defined", cUpdateLogFilePath));
    //NORMAL_M_ASSERT_ALWAYS ();
    return;
  }
  ETG_TRACE_USR4 (("file size of update log %d", (int)fileSize));

  // pathological case do we really need to resize?
  if (fileSize < DL_UPDATE_LOG_FILE_MAX_SIZE)
  {
    ETG_TRACE_USR4 (("resize not neccessary. File size is smaller than %d", DL_UPDATE_LOG_FILE_MAX_SIZE));
    return;
  }

  // Open the file for read
  if ((fd = fopen (UpdateLog::updateLogFile, "r")) == NULL)
  {
    ETG_TRACE_USR4 (("%s File open ERROR", method));
    return;
  }

  // seek to end of file - size to be cutoff
  if (0 != fseek(fd, fileSize - DL_UPDATE_LOG_CUT_SIZE, SEEK_SET))
  {
    ETG_TRACE_USR4 (("%s File seek ERROR", method));
    fclose(fd);
    return;
  }

  //instanciate and initialise new updateLogFile
  tU32 newFileSize =  DL_UPDATE_LOG_CUT_SIZE + 1;
  s8NewUpdateLog = OSAL_NEW tS8[newFileSize];
  if (s8NewUpdateLog == NULL)
  {
    ETG_TRACE_USR4 (("%s Got no memory!!! ERROR", method));
    ETG_TRACE_ERR (("Got no memory!!! Exiting"));
    fclose(fd);
    //FATAL_M_ASSERT_ALWAYS ();
    return;
  }
  else
  {
    //make everything to 0 set
    OSAL_pvMemorySet (s8NewUpdateLog, 0, newFileSize);

    //read the rest of the file into buffer
    size_t readBytes = fread(s8NewUpdateLog, 1, newFileSize - 1, fd);
    if ( (0 == readBytes) || (readBytes > (newFileSize - 1)) )
    {
      ETG_TRACE_ERR (("could not read from the file buffer"));
      ETG_TRACE_USR4 (("%s File read ERROR", method));
      fclose(fd);
    }
    else
    {
      //reading was ok!
      //Close and delete file
      fclose(fd);
      if (0 != remove(UpdateLog::updateLogFile))
      {
         ETG_TRACE_USR4 (("remove updateLogFile not possible"));         
      }   

      //Create update.log file one more time
      fd = fopen(UpdateLog::updateLogFile, "w");
      if (NULL == fd)
      {
        ETG_TRACE_USR3(("file %s could not be created", UpdateLog::updateLogFile));
      }
      else
      {
        //write the mark into the file to show the cutoff place
        fprintf (fd, "%s\n", DL_UPDATE_MARK_CUTOFF);
        //write the file
        fwrite(s8NewUpdateLog, 1, readBytes, fd);
        fclose(fd);
				//fd = NULL;
      }

    }
    // release the buffer
    OSAL_DELETE [] s8NewUpdateLog;
  }

  //jha1kor: Close update file if opend
	//if (NULL != fd)
	//{
	//	fclose(fd);
	//}

  ETG_TRACE_USR4 (("%s was left", method));
  return;
}

/*************************************************************
*FUNCTION:    UpdateLog::updateLogTrace
*DESCRIPTION  traces the given parameters into
*			  the updateLog file
*
*PARAMETER    char *format
*				the format string similar to
*				format string used in printf
*
*            ...
*               all possible parameters
*RETURNVALUE tVoid

*HISTORY 	 30.07.10 TMS CM-AI/PJ-GM54 Weimann
*
*					Initial Revision
*            29.11.13 Aditya Kumar Jha (RBEI/ECG4)
*				Replaced OSAL function with linux native function.
**************************************************************/
tVoid UpdateLog::vUpdateLogTrace (const char *format, ...)
{
	// pathological case
	if (!format)
	{
		return;
	}

	//pthread_mutex_init(&mMutexLogTrace, NULL);
	pthread_mutex_lock(&mMutexLogTrace);

	tCU32 bufferSize = OSAL_C_U32_TRACE_MAX_BUFFERSIZE;
	tCU32 bufferSizeCRLF = 4;

	tChar buffer[bufferSize];
	tChar bufferCRLF[bufferSizeCRLF];
	tPChar pcPos1 = NULL;
	tPChar pcPos2 = NULL;
	OSAL_pvMemorySet(buffer, 0, bufferSize);
	OSAL_pvMemorySet(bufferCRLF, 0, bufferSizeCRLF);
	bufferCRLF[0] = 13; // CR
	bufferCRLF[1] = 10; // LF

	OSAL_tVarArgList args;

	/* prepare the message buffer */
	OSAL_VarArgStart(args, format);

	UpdateLog::UpdateLogFileTrace = open(UpdateLog::updateLogFile, O_RDWR | O_APPEND , S_IRUSR | S_IRGRP | S_IROTH);

	if (UpdateLog::UpdateLogFileTrace == OSAL_ERROR)
	{
		//create the folder and file
		UpdateLog::vUpdateLogOpen ();
	}

	if (UpdateLog::UpdateLogFileTrace != OSAL_ERROR)
	{

		OSAL_s32VarNPrintFormat (buffer,
			bufferSize,
			format,
			args);

	    /* Special handling for remote exec logging: Avoid double strings in update.log */
		pcPos1 = strstr (buffer, "remoteExec echo -e");
		pcPos2 = strstr (buffer, ";echo -e");
		if ( (pcPos1 != NULL) && (pcPos2 != NULL) && (pcPos2 < &buffer[bufferSize] ))
		{
			*pcPos2 = 0; // Terminate string to avoid double string logging
		}

		write(UpdateLog::UpdateLogFileTrace, (tPS8) buffer, OSAL_u32StringLength (buffer));
		write(UpdateLog::UpdateLogFileTrace, (tPS8) bufferCRLF, OSAL_u32StringLength (bufferCRLF));	// Write CR LF at the end of line
		//OSAL_s32IOWrite (UpdateLog::UpdateLogFileTrace, (tPS8) buffer, OSAL_u32StringLength (buffer));
	}

	//jha1kor: Close update file if opend
	if (OSAL_ERROR != UpdateLog::UpdateLogFileTrace)
	{
		close(UpdateLog::UpdateLogFileTrace);
		UpdateLog::UpdateLogFileTrace = OSAL_ERROR;
	}

	OSAL_VarArgEnd(args);
	pthread_mutex_unlock(&mMutexLogTrace);
	//pthread_mutex_destroy(&mMutexLogTrace);
}

/*************************************************************
*FUNCTION:    UpdateLog::vUpdateLogClose
*DESCRIPTION  closes the UpdateLog
*
*PARAMETER    no parameter
*
*RETURNVALUE tVoid
*
*HISTORY 	 30.07.10 TMS CM-AI/PJ-GM54 Weimann
*
*					Initial Revision
*            29.11.13 Aditya Kumar Jha (RBEI/ECG4)
*				Replaced OSAL function with linux native function.
**************************************************************/
tVoid UpdateLog::vUpdateLogClose ()
{
	ETG_TRACE_USR3(("UpdateLog::vUpdateLogClose entered"));
	if (UpdateLog::UpdateLogFileTrace != OSAL_ERROR)
	{
		//write the mark into the end of update
		OSALUTIL_s32FPrintf (UpdateLog::UpdateLogFileTrace,
							 "%s\n",
							 DL_UPDATE_MARK);
		//OSAL_s32IOClose (UpdateLog::UpdateLogFileTrace);
		close(UpdateLog::UpdateLogFileTrace);
		UpdateLog::UpdateLogFileTrace = OSAL_ERROR;
	}
	ETG_TRACE_USR3(("UpdateLog::vUpdateLogClose left"));
}

/**************************************************************/
tString UpdateLog::vGetUpdateLog ()
{
	return updateLogFile;
}

/*tString UpdateLog::vGetOldUpdateLog ()
{
	return DL_UPDATE_LOG_OLD;
}*/

/*************************************************************
*FUNCTION:    UpdateLog::vUpdateLogTest
*DESCRIPTION  Tests the UpdateLog
*
*PARAMETER    no parameter
*
*RETURNVALUE tVoid
*
*HISTORY 	 04.03.2014 CM-AI/ECA2-Stautz
*
*					Initial Revision
**************************************************************/
tVoid UpdateLog::vUpdateLogTest ()
{
	ETG_TRACE_USR3(("UpdateLog::vUpdateLogTest entered"));

	tCString LogMessage1 = "UPD: Text1\n";
	tCString LogMessage2 = "UPD: Text2\n";
	tCString LogMessage3 = "UPD: Text3\n";

	if (UpdateLog::UpdateLogFileTrace != OSAL_ERROR)
	{

		UpdateLog::vUpdateLogOpen();
		// Test the logging:
   		UpdateLog::vUpdateLogTrace ("%s %s", DL_LOG_ERROR,   LogMessage1);
   		UpdateLog::vUpdateLogTrace ("%s %s", DL_LOG_SERVICE, LogMessage2);
   		UpdateLog::vUpdateLogTrace ("%s %s", DL_LOG_INFO,    LogMessage3);
   		UpdateLog::vUpdateLogTrace ("%s %s", DL_LOG_ERROR,   LogMessage1);
   		UpdateLog::vUpdateLogTrace ("%s %s", DL_LOG_SERVICE, LogMessage2);
   		UpdateLog::vUpdateLogTrace ("%s %s", DL_LOG_INFO,    LogMessage3);
		//UpdateLog::vUpdateLogTrace (("%s Test02 DNL:ABORTED, ERROR OVERTEMPERATURE !!!\n", DL_LOG_SERVICE));
		//UpdateLog::vUpdateLogTrace (("%s Test03 DNL:information\n", DL_LOG_INFO));
		UpdateLog::vUpdateLogClose();
	}
	ETG_TRACE_USR3(("UpdateLog::vUpdateLogTest left"));
}

/*************************************************************
*FUNCTION:    UpdateLog::vVerifyLogOpen
*DESCRIPTION  opens the verify log
*
*PARAMETER    no parameters
*RETURNVALUE tVoid
*
*HISTORY 	 30.07.10 TMS CM-AI/PJ-GM54 Weimann
*
*					Initial Revision
*            29.11.13 Aditya Kumar Jha (RBEI/ECG4)
*				Replaced OSAL function with linux native function.
**************************************************************/
tVoid UpdateLog::vVerifyLogOpen  ()
{
	ETG_TRACE_USR3(("UpdateLog::vVerifyLogOpen entered"));

	// Delete old log file
	//OSAL_s32IORemove (DL_OLD_VERIFY_LOG);
	if (0 != remove(DL_OLD_VERIFY_LOG))
   {
         ETG_TRACE_USR4 (("remove DL_OLD_VERIFY_LOG not possible"));         
   }   

	UpdateLog::verifyLogFileTrace = OSAL_ERROR;
	//OSAL_s32IORemove (UpdateLog::verifyLogFile);
	if (0 != remove(UpdateLog::verifyLogFile))
   {
         ETG_TRACE_USR4 (("remove verifyLogFile not possible"));         
   }   
	//UpdateLog::verifyLogFileTrace = OSAL_IOCreate(UpdateLog::verifyLogFile, OSAL_EN_WRITEONLY);
	UpdateLog::verifyLogFileTrace = open(UpdateLog::verifyLogFile, O_RDWR | O_CREAT, S_IRUSR | S_IRGRP | S_IROTH);
	if (OSAL_ERROR == UpdateLog::verifyLogFileTrace)
	{
		ETG_TRACE_USR3(("file %s could not be created", UpdateLog::verifyLogFile));
	}

	//jha1kor: Close update file if opend
	if (OSAL_ERROR != UpdateLog::verifyLogFileTrace)
	{
		close(UpdateLog::verifyLogFileTrace);
		UpdateLog::verifyLogFileTrace = OSAL_ERROR;
	}

	ETG_TRACE_USR3(("UpdateLog::vVerifyLogOpen left"));
}

/*************************************************************
*FUNCTION:    UpdateLog::vVerifyLogTrace
*DESCRIPTION  trace the given buffer string into
*			  the file
*
*PARAMETER    tCString buffer
*				bufdfer to write into file
*RETURNVALUE tVoid
*
*HISTORY 	 30.07.10 TMS CM-AI/PJ-GM54 Weimann
*
*					Initial Revision
*            29.11.13 Aditya Kumar Jha (RBEI/ECG4)
*				Replaced OSAL function with linux native function.
**************************************************************/

tVoid UpdateLog::vVerifyLogTrace (tCString buffer)
{
	if (UpdateLog::verifyLogFileTrace != OSAL_ERROR)
	{
		OSALUTIL_s32FPrintf (UpdateLog::verifyLogFileTrace, "%s",buffer);
	}
	//jha1kor: Close update file if opend
	if (OSAL_ERROR != UpdateLog::verifyLogFileTrace)
	{
		close(UpdateLog::verifyLogFileTrace);
		UpdateLog::verifyLogFileTrace = OSAL_ERROR;
	}
}
/*************************************************************
*FUNCTION:    UpdateLog::vVerifyLogClose
*DESCRIPTION  closes the verify Log
*
*PARAMETER
*
*RETURNVALUE tVoid
*
*HISTORY 	 30.07.10 TMS CM-AI/PJ-GM54 Weimann
*
*Initial Revision
**************************************************************/
tVoid UpdateLog::vVerifyLogClose ()
{
	ETG_TRACE_USR3(("UpdateLog::vVerifyLogClose entred"));
	if (UpdateLog::verifyLogFileTrace != OSAL_ERROR)
	{
		close(UpdateLog::verifyLogFileTrace);
		UpdateLog::verifyLogFileTrace = OSAL_ERROR;
		//OSAL_s32IOClose (UpdateLog::verifyLogFileTrace);
	}
	ETG_TRACE_USR3(("UpdateLog::vVerifyLogClose left"));
}

/*************************************************************************/
/*
* FUNCTION: 	tBool dl_tclFlashOperations::bMkdirInFS
*
* DESCRIPTION:  Make a new subdirectory in the flash file system
*
*
* PARAMETER:    tCString CStrFileSystem 	e.g. "/maindir"
*   			tCString CStrPath       	e.g. "subdir"
*
* RETURNVALUE:  TRUE   in case of success
*               FALSE  in case of error
* HISTORY 	  29.11.13 Aditya Kumar Jha (RBEI/ECG4)
*
* Initial Revision
*/
/*************************************************************************/
tBool UpdateLog::bMkdirInFS (tCString CStrFileSystem,
                             tCString CStrPath)
{
	 DIR *dir;
	 struct dirent * rDirent;
	 tBool bSubDirAvailable = FALSE;
    tBool               bRet   = FALSE;
    tChar acDirName[OSAL_C_U32_MAX_PATHLENGTH];

    ETG_TRACE_USR4(("DNL: ------------------------"));
    ETG_TRACE_USR4(("DNL: bMkdirInFS"));
    ETG_TRACE_USR4(("%s", CStrFileSystem));
    ETG_TRACE_USR4(("%s", CStrPath));

    /* Open the root directory */
	 dir = opendir(CStrFileSystem);

	 if(!dir)
	 {
		  // can not open the dir
        ETG_TRACE_ERR (("Can not mkdir"));
	 }
	 else
	 {
		 //read each content of the directory
		 while( (rDirent = readdir(dir)) != NULL )
		 {
			 if ( strcmp(rDirent->d_name, CStrPath) == 0)
			 {
				 //Sub directory is present
				 bSubDirAvailable = TRUE;
				 break;
			 }
		 }

		 //check if sub folder is not available, then create
		 if( bSubDirAvailable == FALSE )
		 {
			 // Create the sub directory
          memset(acDirName, 0, OSAL_C_U32_MAX_PATHLENGTH);
			 strncpy(acDirName, CStrFileSystem, OSAL_C_U32_MAX_PATHLENGTH-2);
			 strcat(acDirName,  "/");
          if ((strlen(acDirName) + strlen(CStrPath)) < OSAL_C_U32_MAX_PATHLENGTH)
         {         
            strcat(acDirName,  CStrPath);
         }
         else
         {
            strcat(acDirName,  "swupd");
            ETG_TRACE_USR4 (("%s CStrPath name too long ERROR", CStrPath));
            NORMAL_M_ASSERT_ALWAYS();
         }
			 if ( mkdir(acDirName, DEFAULT_MODE) != OSAL_ERROR )
			 {
				 //Sub directory is created
				 bRet = TRUE;
			 }
		 }

		 //Close the directory
		 closedir(dir);
	 }

    ETG_TRACE_USR4 (("bMkdirInFS was left"));

    return bRet;
}
/*************************************************************************/

long int dl_getFileSize (tCString fileName)
{
  long int fileSize = 0;
  FILE *fd = NULL;

  /*pathological cases */
  if (!fileName)
  {
    NORMAL_M_ASSERT_ALWAYS();
    return OSAL_ERROR;
  }

  if ((fd = fopen (fileName, "r")) == NULL)
  {
    return OSAL_ERROR;
  }

  // seek to end of file
  if (0 == fseek(fd, 0, SEEK_END))
  {
    // get current file pointer
    fileSize = ftell(fd);
    fseek(fd, 0, SEEK_SET); // seek back to beginning of file
  }

  fclose(fd);

  ETG_TRACE_USR4 (("dl_getFileSize=%d bytes", (int)fileSize));
	return fileSize;
}

/*************************************************************************/

} //namespace ai_sw_update {
} //namespace common {

/*************************************************************************/
//EOF
