/**
 * @file IsoContainer.cpp
 * @author kut2hi
 * @copyright (c) 2013 Robert Bosch Car Multimedia GmbH
 * @addtogroup ai_sw_update/common
 * @{
 */


#include "IsoContainer.h"
#include "SystemCall.h"
#include "ErrorHandler.h"
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>


#include "swupd_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  #define ETG_DEFAULT_TRACE_CLASS 		TR_CLASS_SWUPDATE_BASE
  #include "trcGenProj/Header/IsoContainer.cpp.trc.h"
#endif
//************ TRACE IF ***************************************************

//#include "swupd_trace.h"
//#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
//#define ETG_DEFAULT_TRACE_CLASS     TR_CLASS_SWUPDATE_MAIN
//#include "trcGenProj/Header/IsoContainer.cpp.trc.h"
//#endif




namespace ai_sw_update {
namespace common {


IsoContainer::IsoContainer()
: _isoContainerFile()
, _mountPoint()
, _isMounted(false)
{
}


IsoContainer::~IsoContainer()
{
  umount();
}

static bool exists(std::string pathname, bool &existing) {
   struct stat path_stat;
   memset(&path_stat, 0, sizeof(path_stat));
   int r = stat(pathname.c_str(), &path_stat);
   if (r == 0) {
      existing = true;
      return true;
   } else if (errno == ENOENT) {
      existing = false;
      return true;
   }
   ETG_TRACE_ERR(("error trying to get stat (errno %8d)of file %s", errno, pathname.c_str()));
   return false;
}

//@todo:
//- Check race condition between losetup and mount, This also happens when used directly on a shell.
//  As workaround sleeps have been introduced. TBC with CF3.
// -----------------------------------------------------------------------------------
const bool IsoContainer::mount(
    const std::string & inIsoContainerFile,
    const std::string & inMountPoint)
// -----------------------------------------------------------------------------------
{
  ETG_TRACE_USR4 (("IsoContainer: inIsoContainerFile (%s)", inIsoContainerFile.c_str()));
  ETG_TRACE_USR4 (("IsoContainer: inMountPoint (%s)", inMountPoint.c_str()));


  if((_isoContainerFile == inIsoContainerFile) && (_mountPoint == inMountPoint)) {
	return true;
  }
  else if((!_isoContainerFile.empty()) && (!_mountPoint.empty()))
  {
   	// unmount ISO container -> OK: IsoContainer
	umount();
	unlink("/tmp/swupdate/container.iso");
  }
	

  _isoContainerFile = inIsoContainerFile;
  _mountPoint = inMountPoint;


  bool existing;
  if (!exists(_isoContainerFile, existing)) {
	  ETG_TRACE_ERR(("IsoContainer: could not check if continer exists: %s", _isoContainerFile.c_str()));
	  return false;
  }
  if (!existing) {
	  ETG_TRACE_ERR(("IsoContainer: continer does not exist: %s", _isoContainerFile.c_str()));
	  return false;
  }

  std::string sysCmd;
  bool sysCmdResult = false;

  // Install kernel module for loop block devices.
  // This is required since the switch to kernel 3.8 as the
  // default configuration for loop back device support is build as module.
  // @todo Decide whether to install the module or change the kernel configuration
  // in order compile loop device support into the kernel. Then the explicit installation
  // of the loop kernel module will not be required. TBC with CF3.
  sysCmd = "modprobe loop";
  sysCmdResult = SystemCall::exec(sysCmd);


  // create mount point
  // @todo TBC if point point needs to be removed if not needed anymore
  sysCmd = "mkdir -p " + _mountPoint;
  sysCmdResult = SystemCall::exec(sysCmd);

  // We had some problems with losetup. During SW Update in normal mode it returned with 127 in some
  // situations. To find out what is going wrong we print out now a list of used loopback device.
  sysCmdResult = SystemCall::exec("echo lopback devices before first try >> /tmp/downloadPipe");
  sysCmdResult = SystemCall::exec("losetup >> /tmp/downloadPipe");


  if (SystemCall::hasError()) {
    ETG_TRACE_ERR(("IsoContainer: mount failed because of previous error"));
    return false;
  }


  // @todo mount iso-container to mountpoint
  sysCmd = "mount " + _isoContainerFile + " " + _mountPoint;
  if (SystemCall::exec(sysCmd))
  {
    //@todo: workaround, sleep 1 second in order to prevent potential race condition (assumption, TBC with CF3) between losetup and following mount
    ETG_TRACE_USR4(("IsoContainer: sleep 1 second after losetup"));
    sleep(5);

    // mark that iso-container is mounted
    _isMounted = true;
  }
  else
  {
    // error case
    ETG_TRACE_ERR(("IsoContainer: %s failed", sysCmd.c_str()));
    iSetSwUpdateError(SW_UPDATE_ERROR_DEVICE_SOURCE_MEDIA, SW_UPDATE_ERROR_TYPE_NOT_AVAILABLE, "IsoContainer losetup", _isoContainerFile.c_str());
    umount();
    return false;
  }
  return true;
}






// -----------------------------------------------------------------------------------
void IsoContainer::umount()
// -----------------------------------------------------------------------------------
{
  // unmount iso-container if mounted
  if (_isMounted)
  {
    ETG_TRACE_COMP(("IsoContainer: umount %s", _mountPoint.c_str()));
    SystemCall::exec("umount " + _mountPoint);
    //@todo add assert for systemcall, it is expected to succeed
  }

  //@todo sleep is currently used for working around race conditions when executing linux commands.
  sleep(2);


  // reset internal control information
  _isoContainerFile = "";
  _mountPoint = "";
  _isMounted = false;
}


} // namespace core
} // namespace imp
