/*************************************************************************
* FILE:         ErrorHandler.cpp
* PROJECT:      Gen3
* SW-COMPONENT: SW Update
*----------------------------------------------------------------------
*
* DESCRIPTION:  Gen3 Software Update Error Handler
*
*----------------------------------------------------------------------
* COPYRIGHT:    (c) 2014 Robert Bosch GmbH, Hildesheim
* HISTORY:
* Date      | Author        | Modification
*
* 2014-02-12  W.Stautz		  01.00
* 2014-02-27  W.Stautz		  01.01
**************************************************************************/

// @todo:
// - improve tracing
// - remove OSAL
// - refactor all implementations here in a second step
// - use filesystem functions from fcswupdate, they need to be relocated from the VAG MIB to the common part

// ----------------------------------------------------------------------------
// includes
// ----------------------------------------------------------------------------

#define  DL_ERROR_HANDLER_DATA_INCLUDE
#include "ErrorHandler.h"
#include "UpdateLog.h"
#include "DownloadPipe.h"

#include <iostream>
#include <fstream>

#include <errno.h>
//#include <stdio.h>
#include <unistd.h>

// Workaround:
// #include "/home/szw2hi/bosch/szw2hi_GMG3G_LINUX_14.0A085P2.2.vws_GEN/ai_projects/generated/components/asf/asf/swupdate/cmt/src-gen/ai_sw_update/common/core/imp/SWUCoreTypesConst.h"


#include "swupd_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
 #define ETG_DEFAULT_TRACE_CLASS 		TR_CLASS_SWUPDATE_BASE
 #include "trcGenProj/Header/ErrorHandler.cpp.trc.h"
#endif


namespace ai_sw_update {
namespace common {

// ----------------------------------------------------------------------------
// data definition
// ----------------------------------------------------------------------------

// Pointer to the error structure
struct structSwUpdateErrors *paStructSwUpdateErrorsStart = NULL; // Start of the array of error structures
struct structSwUpdateErrors *paStructSwUpdateErrorsRead  = NULL; // Current read  pointer in the array
struct structSwUpdateErrors *paStructSwUpdateErrorsWrite = NULL; // Current write pointer in the array

// Error counter
int iSwUpdateErrorCounter = 0;

// ----------------------------------------------------------------------------
// function implementation
// ----------------------------------------------------------------------------

/*************************************************************************/
/*
* FUNCTION: 	int iInitializeSwUpdateErrorArray
*
* DESCRIPTION:  Allocates memory and initializes the software update error array
*
*
* PARAMETER:    none
*
*
*
* RETURNVALUE:  int iSwUpdateErrorCounter
*
*/
/*************************************************************************/
int iInitializeSwUpdateErrorArray (void)
{
    ETG_TRACE_USR4(("DNL: iInitializeSwUpdateErrorArray enter"));

	// Avoid multiple initializations
	// Error array is not existing: create the array data structure
	if (paStructSwUpdateErrorsStart == NULL)
	{
		// Reset the error counter
		iSwUpdateErrorCounter = 0;

		// Allocate memory for the array of error structures
		paStructSwUpdateErrorsStart = (struct structSwUpdateErrors *)malloc(sizeof(structSwUpdateErrors) * SW_UPDATE_ERROR_ARRAY_SIZE);
		paStructSwUpdateErrorsRead  = paStructSwUpdateErrorsStart;
		paStructSwUpdateErrorsWrite = paStructSwUpdateErrorsStart;

		// Clear all errors
		for (int i = 0; i < SW_UPDATE_ERROR_ARRAY_SIZE; i++)
		{
			// Clear the error in the array
			paStructSwUpdateErrorsWrite->iCount 	= 0;
			paStructSwUpdateErrorsWrite->iDevice 	= 0;
			paStructSwUpdateErrorsWrite->iErrorType = 0;
			strncpy(paStructSwUpdateErrorsWrite->acText1, "---", SW_UPDATE_ERROR_TEXT1_SIZE);
			strncpy(paStructSwUpdateErrorsWrite->acText2, "---", SW_UPDATE_ERROR_TEXT2_SIZE);

			paStructSwUpdateErrorsWrite++;
		}

		// set write pointer back to the start
		paStructSwUpdateErrorsWrite = paStructSwUpdateErrorsStart;
    	ETG_TRACE_USR4(("DNL: iInitializeSwUpdateErrorArray done"));

		// Open the update log file
		UpdateLog::vUpdateLogOpen();
	}
	else // Error array already existing, clear all error entries
	{
  		// Clear all errors in the error handler array
  		iClearSwUpdateError(0, 0);
	}

	// free((void *)paStructSwUpdateErrorsStart); // do be done at the end of the download

    ETG_TRACE_USR4(("DNL: iInitializeSwUpdateErrorArray was left"));

	return iSwUpdateErrorCounter;
}
/*************************************************************************/


/*************************************************************************/
/*
* FUNCTION: 	int iSetSwUpdateError
*
* DESCRIPTION:  Set a software update error in the error array
*
*
* PARAMETER:    int         iUpdateDevice : Device in which the error occured, see ErrorHandler.h
*				int         iUpdateError  : Error type						 , see ErrorHandler.h
*				const char* pcText1		  :	Error text1 from lua script execution
*				const char* pcText2		  :	Error text2 from lua script execution
*
* RETURNVALUE:  int iUpdateErrorCount
*
*/
/*************************************************************************/

// to solve lint warnings:
int iSetSwUpdateError (enum SwUpdateDevice iUpdateDevice, enum SwUpdateErrorType iUpdateError, const char* pcText1, const char* pcText2)
{
 	return iSetSwUpdateError ((int)iUpdateDevice, (int)iUpdateError, pcText1, pcText2);
}

int iSetSwUpdateError (int iUpdateDevice, int iUpdateError, const char* pcText1, const char* pcText2)
{
	// Pointer to the last written error
	struct structSwUpdateErrors *paStructSwUpdateErrorsLast = paStructSwUpdateErrorsWrite - 1;

    ETG_TRACE_USR4(("DNL: iSetSwUpdateError Device=%d Error=%d", iUpdateDevice, iUpdateError));

	// Check function parameters
	if ( (iUpdateDevice >= 0) && (iUpdateDevice < (int)SW_UPDATE_ERROR_DEVICE_LAST) &&
		 (iUpdateError  >= 0) && (iUpdateError  < (int)SW_UPDATE_ERROR_TYPE_LAST) &&
		 (NULL != pcText1) && (NULL != pcText2)
	   )
	{
	    // Parameter OK:
		if (iSwUpdateErrorCounter < SW_UPDATE_ERROR_ARRAY_SIZE)
		{
			// Store the error in the array
			paStructSwUpdateErrorsWrite->iCount    += 1; // Increment the counter
			paStructSwUpdateErrorsWrite->iDevice 	= iUpdateDevice;
			paStructSwUpdateErrorsWrite->iErrorType = iUpdateError;
			strncpy(paStructSwUpdateErrorsWrite->acText1, pcText1, SW_UPDATE_ERROR_TEXT1_SIZE);
			strncpy(paStructSwUpdateErrorsWrite->acText2, pcText2, SW_UPDATE_ERROR_TEXT2_SIZE);

    		// Increment the error counter
    		iSwUpdateErrorCounter++;
			// Set the write pointer to the next position
			paStructSwUpdateErrorsWrite++;
		}
		else // Error array is full:
		{
    		ETG_TRACE_USR4(("DNL: iSetSwUpdateError ERROR ARRAY FULL"));
			// Store the full message in the last position of the array
			paStructSwUpdateErrorsLast->iCount     += 1; // Increment the counter
			paStructSwUpdateErrorsLast->iDevice 	= (int)SW_UPDATE_ERROR_DEVICE_RAM;
			paStructSwUpdateErrorsLast->iErrorType  = (int)SW_UPDATE_ERROR_TYPE_MEMORY_SIZE;
			strncpy(paStructSwUpdateErrorsLast->acText1, "ErrorHandler array full", SW_UPDATE_ERROR_TEXT1_SIZE);
			strncpy(paStructSwUpdateErrorsLast->acText2, " ", SW_UPDATE_ERROR_TEXT2_SIZE);
		}

   		// Write this error to the update.log
		{
			tCString LogMessage1 = "*** ErrorHandler ***\n";
			tCString LogMessage2 = "      ********************\n";
   			UpdateLog::vUpdateLogTrace ("%s %s", DL_LOG_ERROR, LogMessage1);
   			UpdateLog::vUpdateLogTrace ("%s %s", " ",  StrUpdateDevice   [iUpdateDevice]);
   			UpdateLog::vUpdateLogTrace ("%s %s", " ",  StrUpdateErrorType[iUpdateError]);
   			UpdateLog::vUpdateLogTrace ("%s %s", " ",  pcText1);
   			UpdateLog::vUpdateLogTrace ("%s %s", " ",  pcText2);
   			UpdateLog::vUpdateLogTrace ("%s %s", "\n", LogMessage2);
		}

        // Write this error also to the error memory /dev/errmem
        {
			tCString LogMessage1 = "\n*** SWupdateErrorHandler ***\n";
			tCString LogMessage2 = "****************************\n";
			unsigned int iErrMemStringSize =static_cast<unsigned int> (strlen(LogMessage1) + 									
									         strlen(StrUpdateDevice[iUpdateDevice]) +
									         strlen(StrUpdateErrorType[iUpdateError]) +
									         strlen(pcText1) +
									         strlen(pcText2) +
									         strlen(LogMessage2) );

			char acErrMemBuffer[iErrMemStringSize+64];
		    memset(acErrMemBuffer, 0, iErrMemStringSize);
			if (iErrMemStringSize <= SW_UPDATE_ERROR_MEM_MAX_LENGTH)
			{
				sprintf(acErrMemBuffer, "%s%s\n%s\n%s\n%s\n%s", LogMessage1,
													        StrUpdateDevice[iUpdateDevice],
													        StrUpdateErrorType[iUpdateError],
													        pcText1,
													        pcText2,
													        LogMessage2);
        		vWriteErrmem(acErrMemBuffer,static_cast<tU32> (strlen(acErrMemBuffer) ));
			}
			else // entry is longer:
			{
				sprintf(acErrMemBuffer, "%s%s\n%s\n",   LogMessage1,
													    StrUpdateDevice[iUpdateDevice],
												        StrUpdateErrorType[iUpdateError]);
        		vWriteErrmem(acErrMemBuffer,static_cast<tU32> (strlen(acErrMemBuffer) ));
        		vWriteErrmem(pcText1,static_cast<tU32> (strlen(pcText1) ));
        		vWriteErrmem(pcText2,static_cast<tU32> (strlen(pcText2) ));
        		vWriteErrmem(LogMessage2,static_cast<tU32> (strlen(LogMessage2) ));
			}
		}

        // Write this error also to the /tmp/downloadPipe
        {
			tCString LogMessage1 = "\nERROR *** SWupdateErrorHandler ***\n";
			tCString LogMessage2 = "**********************************\n";
			unsigned int iErrMemStringSize =static_cast<unsigned int> (strlen(LogMessage1) + 									
									         strlen(StrUpdateDevice[iUpdateDevice]) +
									         strlen(StrUpdateErrorType[iUpdateError]) +
									         strlen(pcText1) +
									         strlen(pcText2) +
									         strlen(LogMessage2) + 64 );
			char acErrMemBuffer[iErrMemStringSize];

			memset(acErrMemBuffer, 0, iErrMemStringSize);
			sprintf(acErrMemBuffer, "%s%s\n%s\n%s\n%s\n%s", LogMessage1,
													        StrUpdateDevice[iUpdateDevice],
													        StrUpdateErrorType[iUpdateError],
													        pcText1,
													        pcText2,
													        LogMessage2);
        	vWriteDownloadPipe(acErrMemBuffer,static_cast<tU32> (strlen(acErrMemBuffer) ));
		}
	}
	else // Parameter error:
	{
    	ETG_TRACE_USR4(("DNL: iSetSwUpdateError parameter ERROR"));
	}

    ETG_TRACE_USR4(("DNL: iSetSwUpdateError was left"));

	return iSwUpdateErrorCounter;
}
/*************************************************************************/

/*************************************************************************/
/*
* FUNCTION: 	int iSetSwUpdateErrorString
*
* DESCRIPTION:  Set a software update error in the error array
*
*
* PARAMETER:    const char* pcErrorType   : Error device
*				const char* pcErrorDevice : Error type
*				const char* pcText1		  :	Error text1 from lua script execution
*				const char* pcText2		  :	Error text2 from lua script execution
*
* RETURNVALUE:  int iUpdateErrorCount
*
*/
/*************************************************************************/
int iSetSwUpdateErrorString (const char* pcErrorDevice, const char* pcErrorType, const char* pcText1, const char* pcText2)
{
	int iIndexDevice = 0;
	int iIndexType   = 0;
	const char * pcFindStringDevice = NULL;
	const char * pcFindStringType   = NULL;

    ETG_TRACE_USR4(("DNL: iSetSwUpdateErrorString enter"));

	// Check parameters
	if( (pcErrorDevice != NULL) && (pcErrorType != NULL) && (pcText1 != NULL) && (pcText2 != NULL) )
	{
		// Search error type in string array
		for (iIndexType=0; iIndexType<(int)SW_UPDATE_ERROR_TYPE_LAST; iIndexType++)
		{
			pcFindStringType = strstr((const char*)StrUpdateErrorType[iIndexType], pcErrorType);
			if (pcFindStringType != NULL) break;
		}
		if (strlen(pcErrorType) == 0) pcFindStringType   = NULL;

		// Search error device in string array
		for (iIndexDevice=0; iIndexDevice<(int)SW_UPDATE_ERROR_DEVICE_LAST; iIndexDevice++)
		{
			pcFindStringDevice = strstr((const char*)StrUpdateDevice[iIndexDevice], pcErrorDevice);
			if (pcFindStringDevice != NULL) break;
		}
		if (strlen(pcErrorDevice) == 0) pcFindStringDevice = NULL;

		// Both strings found:
		if      ((pcFindStringType != NULL) && (pcFindStringDevice != NULL) )
		{
    		iSetSwUpdateError(iIndexDevice, iIndexType,  pcText1, pcText2);
		}

		// Only type string found:
		else if ((pcFindStringType != NULL) && (pcFindStringDevice == NULL) )
		{
    		iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_UNKNOWN, iIndexType,  pcText1, pcText2);
		}

		// Only device string found:
		else if ((pcFindStringType == NULL) && (pcFindStringDevice != NULL) )
		{
    		iSetSwUpdateError(iIndexDevice, (int)SW_UPDATE_ERROR_TYPE_UNKNOWN_ERROR,  pcText1, pcText2);
		}

		// None of both strings found:
		else if ((pcFindStringType == NULL) && (pcFindStringDevice == NULL) )
		{
    		iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_UNKNOWN, (int)SW_UPDATE_ERROR_TYPE_UNKNOWN_ERROR,  pcText1, pcText2);
		}
	}
	else
	{
    	ETG_TRACE_USR4(("DNL: iSetSwUpdateErrorString parameter ERROR"));
	}

    ETG_TRACE_USR4(("DNL: iSetSwUpdateErrorString left"));
	return iSwUpdateErrorCounter;
}
/*************************************************************************/

/*************************************************************************/
/*
* FUNCTION: 	int iGetSwUpdateError
*
* DESCRIPTION:  Get all software update errors
*
*
* PARAMETER:    const char * acFileName : The name of a text file in which all error messages shall be printed;
*										  if NULL is given, no printing is done
*
*
* RETURNVALUE:  int iUpdateErrorCount   : Number of all errors stored in the error array
*
*/
/*************************************************************************/
int iGetSwUpdateError (const char * acFileName)
{
	FILE * fFile = NULL;
    struct structSwUpdateErrors *paStructSwUpdateErrorsGet = NULL; // Current read  pointer in the array

    ETG_TRACE_USR4(("DNL: iGetSwUpdateError enter"));
	ETG_TRACE_USR4(("DNL: iGetSwUpdateError iSwUpdateErrorCounter=%d", iSwUpdateErrorCounter));

	// Simulate some errors, if a special file exists in /tmp
	vSwUpdateErrorSimulation ();

	// Write error messages to a text file
	if (acFileName != NULL)
	{
		fFile = fopen(acFileName, "wt");
		if (fFile != NULL)
		{
			fprintf(fFile, "ErrorArray size %d\n", (int)(sizeof(structSwUpdateErrors) * SW_UPDATE_ERROR_ARRAY_SIZE));
			fprintf(fFile, "---------------------------------\n");
			fprintf(fFile, "--- ErrorHandlerList  begin ----- %s\n", acFileName);
			fprintf(fFile, "No. Device            ErrorType  \n");
			fprintf(fFile, "---------------------------------\n");
			// Iterate through all errors
			paStructSwUpdateErrorsGet = paStructSwUpdateErrorsStart;
			for (int i = 0; i < iSwUpdateErrorCounter; i++)
			{
				// Print all errors in the file
				fprintf(fFile, "%3.3d %s, %s, %s, %s.\n",
		                        i + 1,
						    	StrUpdateDevice   [paStructSwUpdateErrorsGet->iDevice],
						    	StrUpdateErrorType[paStructSwUpdateErrorsGet->iErrorType],
						        paStructSwUpdateErrorsGet->acText1,
						        paStructSwUpdateErrorsGet->acText2);

				paStructSwUpdateErrorsGet++;
			}
			fprintf(fFile, "--- ErrorHandlerList    end -----\n");
			fclose(fFile);
		}
	}

	// Trace all errors
	paStructSwUpdateErrorsGet = paStructSwUpdateErrorsStart;
	for (int i = 0; i < iSwUpdateErrorCounter; i++)
	{
		// Trace all errors in the array
        ETG_TRACE_USR4(("iGetSwUpdateError %3.3d %3.3d %3.3d %10s, %s",
		                 i,
						 paStructSwUpdateErrorsGet->iDevice,
						 paStructSwUpdateErrorsGet->iErrorType,
						 paStructSwUpdateErrorsGet->acText1,
						 paStructSwUpdateErrorsGet->acText2));

		 paStructSwUpdateErrorsGet++;
	}

    ETG_TRACE_USR4(("DNL: iGetSwUpdateError was left"));

	return iSwUpdateErrorCounter;
}
/*************************************************************************/

/*************************************************************************/
/*
* FUNCTION: 	struct structSwUpdateErrors * paGetSwUpdateErrorArray
*
* DESCRIPTION:  Provides a pointer to the error structures
*
*
* INPUT PARAMETER:   const char * acFileName  : The name of a text file in which all error messages shall be printed;
*										        if NULL is given, no printing is done
*
* OUTPUT PARAMETER:	 int * piErrorCounter	  : number of errors
*
*
* RETURNVALUE:       struct structSwUpdateErrors * : Pointer to the array of error structures
*
*/
/*************************************************************************/
struct structSwUpdateErrors *paGetSwUpdateErrorArray (const char * acFileName, int * piErrorCounter)
{
	FILE * fFile = NULL;
    struct structSwUpdateErrors *paStructSwUpdateErrorsGet = NULL; // Current read  pointer in the array

    ETG_TRACE_USR4(("DNL: iGetSwUpdateError enter"));
	ETG_TRACE_USR4(("DNL: iGetSwUpdateError iSwUpdateErrorCounter=%d", iSwUpdateErrorCounter));

	*piErrorCounter = iSwUpdateErrorCounter;

	fFile = fopen(acFileName, "wt");
	if (fFile != NULL) {
	   if (paStructSwUpdateErrorsStart != NULL)
	   {
         fprintf(fFile, "ErrorArray size %d\n", (int)(sizeof(structSwUpdateErrors) * SW_UPDATE_ERROR_ARRAY_SIZE));
         fprintf(fFile, "---------------------------------\n");
         fprintf(fFile, "--- ErrorHandlerList  begin ----- %s\n", acFileName);
         fprintf(fFile, "No. Device            ErrorType  \n");
         fprintf(fFile, "---------------------------------\n");

         // Iterate through all errors
         paStructSwUpdateErrorsGet = paStructSwUpdateErrorsStart;
         for (int i = 0; i < iSwUpdateErrorCounter; i++)
         {
            // Print all errors in the file
            fprintf(fFile, "%3.3d %s, %s, %s, %s.\n",
                             i + 1,
                         StrUpdateDevice   [paStructSwUpdateErrorsGet->iDevice],
                         StrUpdateErrorType[paStructSwUpdateErrorsGet->iErrorType],
                         paStructSwUpdateErrorsGet->acText1,
                         paStructSwUpdateErrorsGet->acText2);

            paStructSwUpdateErrorsGet++;
         }
         fprintf(fFile, "--- ErrorHandlerList    end -----\n");
      }
	   fclose(fFile);
	}
   ETG_TRACE_USR4(("DNL: iGetSwUpdateError was left"));

	return paStructSwUpdateErrorsStart;
}
/*************************************************************************/
/*************************************************************************/
/*
* FUNCTION: 	const char * pcGetSwUpdateErrorDeviceText
*
* DESCRIPTION:  Get the text for the error device
*
*
* PARAMETER:   	int iErrorDevice							: Index to the error text array
*
*
* RETURNVALUE:  const char * pcGetSwUpdateErrorDeviceText	: Error text
*
*/
/*************************************************************************/
const char * pcGetSwUpdateErrorDeviceText (int iErrorDevice)
{
	return StrUpdateDevice[iErrorDevice];
}

/*************************************************************************/
/*
* FUNCTION: 	const char * pcGetSwUpdateErrorTypeText
*
* DESCRIPTION:  Get the text for the error type
*
*
* PARAMETER:   	int iErrorType							: Index to the error text array
*
*
* RETURNVALUE:  const char * pcGetSwUpdateErrorTypeText	: Error text
*
*/
/*************************************************************************/
const char * pcGetSwUpdateErrorTypeText (int iErrorType)
{
	return StrUpdateErrorType[iErrorType];
}
/*************************************************************************/

/*************************************************************************/
/*
* FUNCTION: 	int iClearSwUpdateError
*
* DESCRIPTION:  Clear all software update errors
*
*
* PARAMETER:    int   iUpdateDevice : 	0
*										Device in which the error occured, see ErrorHandler.h
*
*				int   iUpdateError  : 	0
*										Error type					     , see ErrorHandler.h
*
*
* RETURNVALUE:  int iSwUpdateErrorCounter
*
*/
/*************************************************************************/
int iClearSwUpdateError (int iUpdateDevice, int iUpdateError)
{
    struct structSwUpdateErrors *paStructSwUpdateErrorsClear = NULL; // Current read  pointer in the array

    ETG_TRACE_USR4(("DNL: iClearSwUpdateError Device=%d Error=%d", iUpdateDevice, iUpdateError));

	paStructSwUpdateErrorsClear = paStructSwUpdateErrorsStart;
	for (int i = 0; i < SW_UPDATE_ERROR_ARRAY_SIZE; i++)
	{
		// Clear the error in the array
		paStructSwUpdateErrorsClear->iCount 		= 0;
		paStructSwUpdateErrorsClear->iDevice 	    = 0;
		paStructSwUpdateErrorsClear->iErrorType 	= 0;
		strncpy(paStructSwUpdateErrorsClear->acText1, "---", SW_UPDATE_ERROR_TEXT1_SIZE);
		strncpy(paStructSwUpdateErrorsClear->acText2, "---", SW_UPDATE_ERROR_TEXT2_SIZE);

		paStructSwUpdateErrorsClear++;
	}

	// set write pointer back to the start
	paStructSwUpdateErrorsWrite = paStructSwUpdateErrorsStart;
	// reset the error counter
	iSwUpdateErrorCounter = 0;

    ETG_TRACE_USR4(("DNL: iClearSwUpdateError was left"));

	return iSwUpdateErrorCounter;
}
/*************************************************************************/
/*
* FUNCTION: 	int iGetSwUpdateCoreInstallResult
*
* DESCRIPTION:  Get the Core Install Result Code
*
*
* PARAMETER:    void
*
*
*
* RETURNVALUE:  int iUpdateCoreInstallResult
*
*/
/*************************************************************************/
// todo: hpe2hi: put this mapping to core-component
int iGetSwUpdateCoreInstallResult (void)
{
	struct structSwUpdateErrors *paStructSwUpdateErrorsReadResult = NULL; // Current read  pointer in the array
	int iSwUpdateCoreInstallResult = -1;
    tChar tmpBuffer [256] = {0};

    ETG_TRACE_USR4(("DNL: iGetSwUpdateCoreInstallResult enter"));

	// Process all errors which occurred
	paStructSwUpdateErrorsReadResult = paStructSwUpdateErrorsStart;
	for (int i = 0; i < iSwUpdateErrorCounter; i++)
	{
           // Trace all errors in the array
           ETG_TRACE_USR4(("iGetSwUpdateCoreInstallResult %3.3d %3.3d %3.3d %10s, %s",
		                 i,
                           paStructSwUpdateErrorsReadResult->iDevice,
                           paStructSwUpdateErrorsReadResult->iErrorType,
                           paStructSwUpdateErrorsReadResult->acText1,
                           paStructSwUpdateErrorsReadResult->acText2));
           
           // If a valid error type was found:
           if ((iSwUpdateCoreInstallResult == -1)
               && (paStructSwUpdateErrorsReadResult->iErrorType <(int)SW_UPDATE_ERROR_TYPE_LAST) 
               &&  (paStructSwUpdateErrorsReadResult->iErrorType >=0)) 
           {
              iSwUpdateCoreInstallResult = paStructSwUpdateErrorsReadResult->iErrorType;
              break;
           }
		
           // Set pointer to next entry
           paStructSwUpdateErrorsReadResult++;
	}

    ETG_TRACE_USR4(("iGetSwUpdateCoreInstallResult=%3.3d",iSwUpdateCoreInstallResult));

	// No error found:
	if (iSwUpdateCoreInstallResult < 0)
	{
    	ETG_TRACE_USR4(("DNL: iGetSwUpdateCoreInstallResult ERROR_TYPE invalid !!!"));	   
	    iSwUpdateCoreInstallResult = (int)SW_UPDATE_ERROR_TYPE_UNKNOWN_ERROR;
	}
    // Write Install Result to console and downloadPipe
    //sprintf(tmpBuffer, "InstallResultCore: ErrorCounter=%d ErrorCode=%d", iSwUpdateErrorCounter, iSwUpdateCoreInstallResult);
    //cout << tmpBuffer << "\n;
    printf("InstallResultCore: ErrorCounter=%d ErrorCode=%d\n", iSwUpdateErrorCounter, iSwUpdateCoreInstallResult);
    sprintf(tmpBuffer, "InstallResultCore: ErrorCounter=%d ErrorCode=%d", iSwUpdateErrorCounter, iSwUpdateCoreInstallResult);
    printToDownloadPipe(tmpBuffer);

    ETG_TRACE_USR4(("iGetSwUpdateCoreInstallResult=%3.3d",iSwUpdateCoreInstallResult));
    ETG_TRACE_USR4(("DNL: iGetSwUpdateCoreInstallResult was left"));

	return iSwUpdateCoreInstallResult;
}


/*************************************************************************/
/*
* FUNCTION: 	int iTestErrorCodes
*
* DESCRIPTION:  Test the download error code functions
*
*
* PARAMETER INPUT:  none
*
* PARAMETER OUTPUT:	none
*
* RETURNVALUE:      int iErrorCount
*
*/
/*************************************************************************/
int iTestErrorCodes (void)
{
	int iErrorCount = 0;
	int iUpdateInstallResult = 0;
	FILE * fFile    = NULL;
	struct structSwUpdateErrors *paSwUpdateErrorArray = NULL; // Pointer to the error structure array


	// Allocate and initialize the memory to store the errors
	iInitializeSwUpdateErrorArray ();

	// Set some errors, for example
	iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_FLASH_NOR_RAW,  (int)SW_UPDATE_ERROR_TYPE_WRITE,  NULL, NULL);

    iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_FLASH_NAND_FFS, (int)SW_UPDATE_ERROR_TYPE_WRITE,  NULL, NULL);

    iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_SOURCE_MEDIA,   (int)SW_UPDATE_ERROR_TYPE_VERIFY, NULL, NULL);

    iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_SCRIPT,   	  (int)SW_UPDATE_ERROR_TYPE_SCRIPT_EXECUTE, "Error1", "Error2");

  	// First implementation: Get the error count from ErrorHandler and print all errors to a text file
  	iErrorCount = iGetSwUpdateError ("/tmp/ErrorHandler1.txt");

	// Second implementation: Get the pointer to the error array and print all errors to a text file
	paSwUpdateErrorArray = paGetSwUpdateErrorArray ("/tmp/ErrorHandler2.txt", &iErrorCount);

	// Optional: Print the errors in an other text format
	fFile = fopen("/tmp/ErrorHandler3.txt", "wt");
	if (fFile != NULL) {
	   if (paSwUpdateErrorArray != NULL)
      {
         fprintf(fFile, "ErrorArray size %d\n", (int)(sizeof(structSwUpdateErrors) * SW_UPDATE_ERROR_ARRAY_SIZE));
         fprintf(fFile, "---------------------------------\n");
         fprintf(fFile, "--- ErrorHandlerList  begin -----\n");
         fprintf(fFile, "No. Device            ErrorType  \n");
         fprintf(fFile, "---------------------------------\n");
         for (int i = 0; i < iErrorCount; i++)
         {
            // Print all errors in the file
            fprintf(fFile, "%3.3d %s, %s, %s, %s.\n",
                             i + 1,
                        pcGetSwUpdateErrorDeviceText(paSwUpdateErrorArray->iDevice),
                        pcGetSwUpdateErrorTypeText  (paSwUpdateErrorArray->iErrorType),
                         paSwUpdateErrorArray->acText1,
                         paSwUpdateErrorArray->acText2);

            paSwUpdateErrorArray++;
         }
         fprintf(fFile, "--- ErrorHandlerList    end -----\n");

      }
	   fclose(fFile);
	}

	// Get the diaglog DTC
	//iDiaglogDtc = iGetSwUpdateDiaglogDtc();
	iUpdateInstallResult = iGetSwUpdateCoreInstallResult();
    ETG_TRACE_USR4(("iUpdateInstallResult=%3.3d", iUpdateInstallResult));

	// Clear all errors
    iClearSwUpdateError(0, 0);

    return iErrorCount;
}

/*************************************************************************/
/*
* FUNCTION: 	void vSwUpdateErrorSimulation
*
* DESCRIPTION:  Simulate some errors, if a special file exists in the directory /tmp
*
*
* PARAMETER:    none
*
*
* RETURNVALUE:  none
*
*/
/*************************************************************************/
void vSwUpdateErrorSimulation (void)
{
	FILE * fFile = NULL;

    ETG_TRACE_USR4(("DNL: vSwUpdateErrorSimulation enter"));

	// Error simulations
	fFile = fopen("/tmp/errsim1.txt", "rt");
	if (fFile != NULL)
	{
        ETG_TRACE_USR4(("DNL: iGetSwUpdateError ErrorSimulation1 active"));
	    iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_FLASH_NOR_RAW,  (int)SW_UPDATE_ERROR_TYPE_WRITE,               "ErrSimFlashWrite", "1");
		fclose(fFile);
		unlink("/tmp/errsim1.txt"); // Simulate the error only once
	}

	fFile = fopen("/tmp/errsim2.txt", "rt");
	if (fFile != NULL)
	{
        ETG_TRACE_USR4(("DNL: iGetSwUpdateError ErrorSimulation2 active"));
	    iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_FLASH_NAND_FFS, (int)SW_UPDATE_ERROR_TYPE_CHECKSUM,            "ErrSimCheckSum", "2");
		fclose(fFile);
		unlink("/tmp/errsim2.txt"); // Simulate the error only once
	}

	fFile = fopen("/tmp/errsim3.txt", "rt");
	if (fFile != NULL)
	{
        ETG_TRACE_USR4(("DNL: iGetSwUpdateError ErrorSimulation3 active"));
	    iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_SOURCE_MEDIA,   (int)SW_UPDATE_ERROR_TYPE_NOT_AVAILABLE,     	"ErrSimNotAvail", "3");
		fclose(fFile);
	}

	fFile = fopen("/tmp/errsim4.txt", "rt");
	if (fFile != NULL)
	{
        ETG_TRACE_USR4(("DNL: iGetSwUpdateError ErrorSimulation4 active"));
	    iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_IMAGE,  (int)SW_UPDATE_ERROR_TYPE_COMPATIBLE_MISMATCH,         	"ErrSimCompatMis", "4");
		fclose(fFile);
	}

	fFile = fopen("/tmp/errsim5.txt", "rt");
	if (fFile != NULL)
	{
        ETG_TRACE_USR4(("DNL: iGetSwUpdateError ErrorSimulation5 active"));
	    iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_SOURCE_MEDIA,  (int)SW_UPDATE_ERROR_TYPE_MSG_DIG_SIG_MISMATCH, 	"ErrSimMsgDigSigMis", "5");
		fclose(fFile);
	}

	fFile = fopen("/tmp/errsim6.txt", "rt");
	if (fFile != NULL)
	{
        ETG_TRACE_USR4(("DNL: iGetSwUpdateError ErrorSimulation6 active"));
	    iSetSwUpdateError((int)SW_UPDATE_ERROR_DEVICE_SOURCE_MEDIA,  (int)SW_UPDATE_ERROR_TYPE_LIC_DIG_SIG_MISMATCH, 	"ErrSimLicDigSigMis", "6");
		fclose(fFile);
	}

    ETG_TRACE_USR4(("DNL: vSwUpdateErrorSimulation leave"));
	return;
}
/*************************************************************************/
/*
* FUNCTION: 	bool bSwUpdateErrorSimulation
*
* DESCRIPTION:  Simulate one error, if a special file exists in the directory /tmp
*
*
* PARAMETER:    none
*
*
* RETURNVALUE:  true	Error simulation active
*				false   No error simulation
*
*/
/*************************************************************************/
bool bSwUpdateErrorSimulation (void)
{
	FILE * fFile = NULL;
    bool bReturnValue = false; // No error simulation active

    ETG_TRACE_USR4(("DNL: bSwUpdateErrorSimulation enter"));

	// Permanent error simulation
	fFile = fopen("/tmp/errsim8.txt", "rt");
	if (fFile != NULL)
	{
        ETG_TRACE_USR4(("DNL: ErrorSimulation8 active"));
		fclose(fFile);
		bReturnValue = true; // Error simulation active
	}

	// Temporary error simulation
	fFile = fopen("/tmp/errsim9.txt", "rt");
	if (fFile != NULL)
	{
        ETG_TRACE_USR4(("DNL: ErrorSimulation9 active"));
		fclose(fFile);
		unlink("/tmp/errsim9.txt"); // Simulate the error only once
		bReturnValue = true; // Error simulation active
	}

    ETG_TRACE_USR4(("DNL: bSwUpdateErrorSimulation=%d", bReturnValue));

	return (bReturnValue);
}
/*************************************************************************/
/*
* FUNCTION: 	void vWriteErrmem
*
* DESCRIPTION:  Write buffer to the error memory
*
*
* PARAMETER:
*				tChar *acBuf
*				tU16  u16Len
*
* RETURNVALUE:  none
*
*/
/*************************************************************************/
tVoid vWriteErrmem(const tChar *acBuf, tU32 u32Len)
{
    FILE * fFile = NULL;
    tChar acErrMemText[SW_UPDATE_ERROR_MEM_MAX_LENGTH + 1] = {0};

	ETG_TRACE_USR4(("SWU: ERRMEM write length %d", u32Len));
	ETG_TRACE_USR4(("%s", acBuf));
	ETG_TRACE_USR4(("%d", u32Len));

    // Limit string for error memory to max. 200 chars
	memset(acErrMemText,  0, SW_UPDATE_ERROR_MEM_MAX_LENGTH + 1);
	strncpy(acErrMemText, acBuf, SW_UPDATE_ERROR_MEM_MAX_LENGTH);
	ETG_TRACE_USR4(("%s", acErrMemText));
	ETG_TRACE_USR4(("%d", strlen(acErrMemText)));
    
	printToErrMemPipe(acErrMemText);

	fFile = fopen(DNL_ERR_TMP_FILE, "wt");
	if (fFile != NULL)
	{
			fprintf(fFile, "%s", acErrMemText);
			fclose(fFile);
			std::ofstream downloadPipe("/dev/errmem", std::ofstream::binary);
			std::ifstream source(DNL_ERR_TMP_FILE, std::ofstream::binary);

			downloadPipe << source.rdbuf();
			downloadPipe << "\n";

			source.close();
			downloadPipe.close();
	}
  }

/*************************************************************************/
/*
* FUNCTION: 	void vWriteDownloadPipe
*
* DESCRIPTION:  Write buffer to the /tmp/downloadPipe
*
*
* PARAMETER:
*				tChar *acBuf
*				tU16  u16Len
*
* RETURNVALUE:  none
*
*/
/*************************************************************************/
tVoid vWriteDownloadPipe(const tChar *acBuf, tU32 u32Len)
{
    FILE * fFile = NULL;

	ETG_TRACE_USR4(("SWU: downloadPipe write length %d", u32Len));
	ETG_TRACE_USR4(("%s", acBuf));
	ETG_TRACE_USR4(("%d", u32Len));

	fFile = fopen(DNL_ERR_TMP_FILE, "wt");
	if (fFile != NULL)
	{
			fprintf(fFile, "%s", acBuf);
			fclose(fFile);
			attachFileToDownloadPipe(DNL_ERR_TMP_FILE);
	}
	fprintf(stderr, "%s", acBuf);  // output to the error console
}

/*************************************************************************/

} //namespace ai_sw_update {
} //namespace common {

/*************************************************************************/
//EOF
