#include "BootChainMgr.h"

#define OSAL_S_IMPORT_INTERFACE_GENERIC
#include "osal_if.h"

#include "MTD_IO.h"
#include "SystemCall.h"
#include "DownloadPipe.h"
#include <fcntl.h>        // for open() command and flags.
#include <unistd.h>       // for pwrite32/pread64.
#include <mtd/mtd-abi.h>  // for MTD ioctl commands and structs.
#include <string>



#include "swupd_trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
  #define ETG_DEFAULT_TRACE_CLASS     TR_CLASS_SWUPDATE_BASE
  #include "trcGenProj/Header/BootChainMgr.cpp.trc.h"
#endif


namespace ai_sw_update {
namespace common {

//FIXME: Bad solution, but ETG_TRACE_ERRMEM is unreadable
static void print2errmem(const char *Format, ...)
{
  int     fd=open("/dev/errmem", O_WRONLY);//O_RDWR);
  char    msg[1024];
  va_list arglst;

  if (fd >= 0) {
     va_start(arglst, Format);
     vsnprintf(msg, sizeof(msg), Format, arglst);
     va_end(arglst);
     write(fd, msg, strlen(msg));
     close(fd);
  }
}


/*
* FUNCTION:   tBool dl_tclFlashOperations::bWriteMagicToRawFlash
*
* DESCRIPTION:  Write the magic to the raw nor flash
*
*
* PARAMETER INPUT:  tU32     u32FlashMagicOffset e.g. 0x00000020 for DNL Magic
*                   tU32     u32Value        e.g  0x2342ABCD or  0xFFFFFFFF
*
* PARAMETER OUTPUT: none
*
* RETURNVALUE:      TRUE   in case of success
*                   FALSE  in case of error
*/
tBool bWriteMagicToRawFlash (unsigned int u32FlashMagicOffset, unsigned int u32Value)
{
  // Read sector 1 and 2
  // Determine selected sector (A)
  // Update the other sector (B) with new data:
  // - copy all data from selected sector (A)
  // - increment the counter by 1
  // - set the sector to valid
  //
  // Clear sector A:
  // - invalidate sector (flag)
  // - counter = 0
  // - copy bootchain from sector B
  //
  // Serialize active sector to NOR flash with MTD_IO functions. Erase existing NOR storage before.


  tBool  bRetVal  = FALSE;
  tU32   u32FlashAdressToSet  = 0;
  tU32   u32FlashAdressToClear= 0;
  tU32   u32MagicAreaSelected = 0;
  tU32   u32MagicIndex        = 0;
  tU32   au32MagicArea1[DNL_MAGIC_ARRAY_SIZE] = {0};
  tU32   au32MagicArea2[DNL_MAGIC_ARRAY_SIZE] = {0};
  tPVoid pvMagicBufferToSet                   = NULL;
  tPVoid pvMagicBufferToClear                 = NULL;

  ETG_TRACE_USR4 (("DNL: ------------------------"));
  ETG_TRACE_USR4 (("DNL: bWriteMagicToRawFlash Offset %d", u32FlashMagicOffset));
  print2errmem("DNL: bWriteMagicToRawFlash %08X at Offset %08X\n", u32Value, u32FlashMagicOffset);

  // Read both magic areas from flash
  bRetVal = bReadMagicAreas (
      u32FlashMagicOffset,
      &u32MagicIndex,
      &au32MagicArea1[0],
      &au32MagicArea2[0],
      &u32MagicAreaSelected);

  if (u32MagicIndex >= DNL_MAGIC_ARRAY_SIZE) {
     ETG_TRACE_FATAL(("Magic Value Index out of range while writing (%d)", u32MagicIndex));
     return false;
  }

  if(u32MagicAreaSelected == 1)  	// First magic area selected:
  {
    // Write the new data to the second magic area
    pvMagicBufferToSet    			= &au32MagicArea2[0];
    u32FlashAdressToSet   			= DNL_MAGIC2_BASE_ADDRESS;
    au32MagicArea2[0]				= DL_MAGIC_SECTOR_VALID;	// Magic valid
    au32MagicArea2[4]				= au32MagicArea1[4] + 1;	// Increment the counter
    au32MagicArea2[8]				= au32MagicArea1[8];		// Copy the same DNL state
    au32MagicArea2[12]				= au32MagicArea1[12];		// Copy the same DNL source
    au32MagicArea2[16]				= au32MagicArea1[16];		// Copy the same BootChain
    au32MagicArea2[u32MagicIndex] 	= u32Value;					// Set the new value
    // Clear the old magic:
    pvMagicBufferToClear  			= &au32MagicArea1[0];
    u32FlashAdressToClear 			= DNL_MAGIC1_BASE_ADDRESS;
    au32MagicArea1[ 0]	  			= DL_MAGIC_SECTOR_INVALID;  // Magic invalid
    au32MagicArea1[ 4]	  			= 0;  						// Reset counter
    au32MagicArea1[16]				= au32MagicArea2[16];		// Copy the same BootChain
  }
  else							// Second magic area selected:
  {
    // Write the new data to the first magic area
    pvMagicBufferToSet      		= &au32MagicArea1[0];
    u32FlashAdressToSet   			= DNL_MAGIC1_BASE_ADDRESS;
    au32MagicArea1[0]				= DL_MAGIC_SECTOR_VALID;    // Magic valid
    au32MagicArea1[4]				= au32MagicArea2[4] + 1;    // Increment the counter
    au32MagicArea1[8]				= au32MagicArea2[8];    	// Copy the same DNL state
    au32MagicArea1[12]				= au32MagicArea2[12];    	// Copy the same DNL source
    au32MagicArea1[16]				= au32MagicArea2[16];		// Copy the same BootChain
    au32MagicArea1[u32MagicIndex] 	= u32Value;					// Set the new value
    // Clear the old magic:
    pvMagicBufferToClear  			= &au32MagicArea2[0];
    u32FlashAdressToClear 			= DNL_MAGIC2_BASE_ADDRESS;
    au32MagicArea2[ 0]	  			= DL_MAGIC_SECTOR_INVALID;  // Magic invalid
    au32MagicArea2[ 4]	  			= 0;						// Reset the counter
    au32MagicArea2[16]				= au32MagicArea1[16];		// Copy the same BootChain
  }

  // Write the new magic area to raw flash
  bRetVal = bProgramDataToRawFlash(
      (tPU8)pvMagicBufferToSet,
      u32FlashAdressToSet,
      DNL_MAGIC_AREA_SIZE,
      TRUE // Erase flash sectors before programming
  );

  if (bRetVal) // if setting of new magic was OK:
  {
    // Clear the old magic area
    bRetVal = bProgramDataToRawFlash(
        (tPU8)pvMagicBufferToClear,
        u32FlashAdressToClear,
        DNL_MAGIC_AREA_SIZE,
        TRUE // Erase flash sectors before programming
    );
  }
  else
  {
    //@todo: handle error case
  }


  ETG_TRACE_USR4 (("bWriteMagicToRawFlash was left"));
  return bRetVal;
}


/*************************************************************************/
/*
* FUNCTION:   tBool dl_tclFlashOperations::bReadMagicAreas
*
* DESCRIPTION:  Write the magic to the raw nor flash
*
*
* PARAMETER INPUT:  tU32   u32FlashMagicOffset e.g. 0x00000020 for DNL MagicRecoveryDownload
*                          0x00000030 for DNL MagicEthernetSource
*                          0x00000040 for DNL MagicBootChainSelection
*
* PARAMETER OUTPUT: tPU32  pu32MagicIndex     Index of DWORD in u32FlashMagicArea1[] :  0, 4, 8, 12, 16 if OK
*                                                            20 in case of error
*         tPU32  pu32FlashMagicArea1    u32FlashMagicArea1[20]
*         tPU32  pu32FlashMagicArea2    u32FlashMagicArea2[20]
*         tPU32  pu32MagicAreaSelected  0: none, 1: Area1, 2: Area2
*
*
* RETURNVALUE:      TRUE   in case of success
*                   FALSE  in case of error
*/
/*************************************************************************/
tBool bReadMagicAreas (
    tU32  u32FlashMagicOffset,
    tPU32 pu32MagicIndex,
    tPU32 pu32FlashMagicArea1,
    tPU32 pu32FlashMagicArea2,
    tPU32 pu32MagicAreaSelected)
{
  // Calculate index of flash magic (4 byte alligned) -> OBSOLETE
  // Read Sector 1 from flash
  // - MTD_IO: Map sector base address to device name, get base address offset to flash device
  // - Read (size of sector) bytes
  // - dump read raw bytes into file /tmp/downloadPipe -> ABSTRACT
  //
  // Read sector 2 from flash (same implementation as above)
  //
  // Check whether number of read bytes equals size of defined bytes per sector (#define)
  // - error case: write to /tmp/downloadPipe -> ABSTRACT
  //
  // Check which sector is valid, return selected sector to caller(1 or 2), return 0 in error case


  ETG_TRACE_USR4 (("bReadMagicAreas was entered"));
  tBool  bRetVal        = FALSE;
  int    fd             = -1;
  int    retcode1       = 0;
  int    retcode2       = 0;
  std::string devname;
  tChar  tmpBuffer [256]   = {0};

  ETG_TRACE_USR4 (("DNL: ------------------------"));
  ETG_TRACE_USR4 (("DNL: bReadMagicAreas"));

  *pu32MagicAreaSelected = 0;   // Initialze with invalid value
  *pu32MagicIndex        = 20;  // Initialze with invalid value

  // Check magic offset and set the index
  // e.g. u32FlashMagicOffset is 0x00, 0x10, 0x20, 0x30, 0x40
  switch (u32FlashMagicOffset)
  {
    // @todo: optimize: u32FlashMagicOffset can be directly mapped to pu32MagicIndex, no subtractions are required.
    case DL_MAGIC_SECTOR_VALID_FLAG   - DNL_MAGIC1_BASE_ADDRESS:  // 0x00160000 - 0x00160000 = 0x00000000  @todo: lint: Info 778: Constant expression evaluates to 0 in operation '-': disable warning
    *pu32MagicIndex = 0;
    break;

    case DL_MAGIC_SECTOR_COUNTER_FLAG - DNL_MAGIC1_BASE_ADDRESS:  // 0x00160010 - 0x00160000 = 0x00000010
    *pu32MagicIndex = 4;
    break;

    case DL_UPDATE_STATUS_FLAG        - DNL_MAGIC1_BASE_ADDRESS:  // 0x00160020 - 0x00160000 = 0x00000020
    *pu32MagicIndex = 8;
    break;

    case DL_ETHERNET_FLAG       - DNL_MAGIC1_BASE_ADDRESS:  // 0x00160030 - 0x00160000 = 0x00000030
    *pu32MagicIndex = 12;
    break;

    case DL_ROOTFS_FLAG         - DNL_MAGIC1_BASE_ADDRESS:  // 0x00160040 - 0x00160000 = 0x00000040
    *pu32MagicIndex = 16;
    break;

    default: // Error
      *pu32MagicIndex = 20;
      break;
  }

  off_t u32FlashAdress = 0;
  off_t u32FlashDevOffset = 0;

  // Read magic area 1 from the flash
  u32FlashAdress = DNL_MAGIC1_BASE_ADDRESS;
  // Set the raw flash device name for the linux driver and adjust the flash address to the flash device offset
  bSetRawFlashDevName(u32FlashAdress, devname, &u32FlashDevOffset);
  u32FlashAdress = u32FlashAdress - u32FlashDevOffset;
  fd = open(devname.c_str(),O_RDONLY,0x1A4);
  if(fd >= 0)
  {
    retcode1 =static_cast<int> (pread(fd, pu32FlashMagicArea1, DNL_MAGIC_AREA_SIZE, u32FlashAdress) );
    close(fd);
  }

  // @todo: replace tracing
  sprintf(tmpBuffer, "ReadMagicArea1=0x%8.8X 0x%8.8X 0x%8.8X 0x%8.8X 0x%8.8X",
    (unsigned int) *pu32FlashMagicArea1,
    (unsigned int) *(pu32FlashMagicArea1+4),
    (unsigned int) *(pu32FlashMagicArea1+8),
    (unsigned int) *(pu32FlashMagicArea1+12),
    (unsigned int) *(pu32FlashMagicArea1+16) );
  printToDownloadPipe(tmpBuffer);

  // Read magic area 2 from the flash
  u32FlashAdress = DNL_MAGIC2_BASE_ADDRESS;
  // Set the raw flash device name for the linux driver and adjust the flash address to the flash devive offset
  bSetRawFlashDevName(u32FlashAdress, devname, &u32FlashDevOffset);
  u32FlashAdress = u32FlashAdress - u32FlashDevOffset;
  fd = open(devname.c_str(), O_RDONLY, 0x1A4);
  if(fd >= 0)
  {
    retcode2 =static_cast<int> (pread(fd, pu32FlashMagicArea2, DNL_MAGIC_AREA_SIZE, u32FlashAdress) );
    close(fd);
  }
  // @todo: replace tracing
  sprintf(tmpBuffer, "ReadMagicArea2=0x%8.8X 0x%8.8X 0x%8.8X 0x%8.8X 0x%8.8X",
    (unsigned int) *pu32FlashMagicArea2,
    (unsigned int) *(pu32FlashMagicArea2+4),
    (unsigned int) *(pu32FlashMagicArea2+8),
    (unsigned int) *(pu32FlashMagicArea2+12),
    (unsigned int) *(pu32FlashMagicArea2+16) );
  printToDownloadPipe(tmpBuffer);

  // Check both magic areas
  if( (retcode1 == DNL_MAGIC_AREA_SIZE) && (retcode2 == DNL_MAGIC_AREA_SIZE) ) // read OK?
  {
    bRetVal = TRUE;
  }
  else // read error
  {
    // @todo: replace tracing
    ETG_TRACE_USR4 (("DNL: bReadMagicAreas: read flash device ERROR"));
    printToDownloadPipe("read flash device ERROR");
  }

  // Check which magic area is valid
  if ( (*pu32FlashMagicArea1 == DL_MAGIC_SECTOR_VALID) && (*pu32FlashMagicArea2 != DL_MAGIC_SECTOR_VALID))
  {
    *pu32MagicAreaSelected = 1; // only area 1 is valid
  }
  if ( (*pu32FlashMagicArea1 != DL_MAGIC_SECTOR_VALID) && (*pu32FlashMagicArea2 == DL_MAGIC_SECTOR_VALID))
  {
    *pu32MagicAreaSelected = 2; // only area 2 is valid
  }
  if ( (*pu32FlashMagicArea1 == DL_MAGIC_SECTOR_VALID) && (*pu32FlashMagicArea2 == DL_MAGIC_SECTOR_VALID))
  {
    // both areas are valid
    // Check also the counters; magic with higher counter is selected
    if ( *(pu32FlashMagicArea1 + 4) >= *(pu32FlashMagicArea2 + 4) )
    {
      *pu32MagicAreaSelected = 1;
    }
    else
    {
      *pu32MagicAreaSelected = 2;
    }
  }
  if ( (*pu32FlashMagicArea1 != DL_MAGIC_SECTOR_VALID) && (*pu32FlashMagicArea2 != DL_MAGIC_SECTOR_VALID))
  {
    *pu32MagicAreaSelected = 0; // both are invalid  //@todo: check how this error case is handled in calling functions
  }

  if (*pu32MagicIndex == 20) bRetVal  = FALSE;

  ETG_TRACE_USR4 (("bReadMagicAreas was left"));
  return bRetVal;
}


/*************************************************************************/
/*
* FUNCTION:   tBool dl_tclFlashOperations::bReadMagicFromRawFlash
*
* DESCRIPTION:  Read the flag content from raw nor flash
*
*
* PARAMETER INPUT:  tU32     u32FlashMagicOffset e.g. 0x00000020 for DNL Magic, 0x00000040 for BootChain
*
* PARAMETER OUTPUT: tPU32    pu32Value         e.g  0x2342ABCD or  0xFFFFFFFF
*
* RETURNVALUE:      TRUE   in case of success
*                   FALSE  in case of error
*/
/*************************************************************************/
tBool bReadMagicFromRawFlash (const unsigned int inFlashAddress, unsigned int& outValue)
{
  // in: inFlashAddress is semantically the flash magic offset, e.g. 0x00, 0x10, 0x20, ...


  //u32MagicIndex: Index of DWORD in u32FlashMagicArea1[] :  0, 4, 8, 12, 16 if OK, 20 in case of error


  // Read sector 1 and 2
  // Determine selected sector
  // out: 4 bytes stored at the address of the requested magic offset of the selected sector, in error case 0
  // Write read raw bytes to file /tmp/downloadPipe -> ABSTRACT



  ETG_TRACE_USR4 (("bReadMagicFromRawFlash was entered"));
  tBool   bRetVal              = FALSE;
  tU32    u32MagicIndex        = 0;
  tU32  u32MagicAreaSelected = 0;
  tU32    au32MagicArea1[DNL_MAGIC_ARRAY_SIZE] = {0}; //@todo: only first element is set to 0, init also the other ones
  tU32    au32MagicArea2[DNL_MAGIC_ARRAY_SIZE] = {0}; //@todo: only first element is set to 0, init also the other ones
  tChar  tmpBuffer [256]     = {0};

  // Read both magic areas from flash
  bRetVal = bReadMagicAreas (inFlashAddress, &u32MagicIndex, &au32MagicArea1[0], &au32MagicArea2[0], &u32MagicAreaSelected);

  if (u32MagicIndex >= DNL_MAGIC_ARRAY_SIZE) {
     ETG_TRACE_FATAL(("Magic Value Index out of range (%d)", u32MagicIndex));
     return false;
  }

  if (u32MagicAreaSelected == 1)
  {
    outValue = au32MagicArea1[u32MagicIndex];
    bRetVal = TRUE;
  }
  else if (u32MagicAreaSelected == 2)
  {
    outValue = au32MagicArea2[u32MagicIndex];
    bRetVal = TRUE;
  }
  else
  {
    outValue = 0; // invalid
  }

  ETG_TRACE_USR4 (("DNL: bReadMagic bRet=%d Area=%d Value=0x%8X.X", bRetVal, u32MagicAreaSelected, outValue));

  sprintf(tmpBuffer, "ReadMagic Area=%d Offset=%d Index=%d Value=0x%8.8X",
      (int)u32MagicAreaSelected, (int)inFlashAddress,(int)u32MagicIndex, outValue);
  printToDownloadPipe(tmpBuffer);

  ETG_TRACE_USR4 (("bReadMagicFromRawFlash was left"));
  return bRetVal;
}


/*************************************************************************/
/*
* FUNCTION:   tBool dl_tclFlashOperations::bWriteMagicBootChain
*
* DESCRIPTION:  Write the magic to the raw nor flash
*       and set the active boot partition register in the eMMC
*
*
* PARAMETER INPUT:  tU32     u32FlashMagicOffset e.g. 0x00000020 for DNL Magic
*                   tU32     u32Value        e.g  0x2342ABCD or  0xFFFFFFFF
*
* PARAMETER OUTPUT: none
*
* RETURNVALUE:      TRUE   in case of success
*                   FALSE  in case of error
*/
/*************************************************************************/
tBool bWriteMagicBootChain(const unsigned int inFlashAddress, const unsigned int inValue)
{

  // inFlashAddress = BOOTCHAIN_MAGIC_ADDRESS (0x40, set from lua script)
  // update bootchain flag in NOR, see above
  // additionally:
  // change mmc to use the correct fast boot area (must be synced to the boot chain configuration in NOR



  ETG_TRACE_USR4 (("bWriteMagicBootChain was entered"));

  tBool  bRetVal  = FALSE;

  /*
  #Boot configuration bytes [PARTITION_CONFIG: 0x08]
  mmc bootpart enable 1 0 /dev/mmcblk1
  mmc extcsd read /dev/mmcblk1 | grep PARTITION_CONFIG
   */

  // Write selected boot chain to eMMC boot partition register
  if (inFlashAddress == DL_ROOTFS_FLAG_OFFSET)
  {
    if (inValue == DL_ROOTFS_1)
    {
      //system("mmc bootpart enable 1 0 /dev/mmcblk1");
      SystemCall::execCommand("/usr/bin/mmc bootpart enable 1 0 /dev/mmcblk1");
    }
    else if (inValue == DL_ROOTFS_2)
    {
      //system("mmc bootpart enable 2 0 /dev/mmcblk1");
       SystemCall::execCommand("/usr/bin/mmc bootpart enable 2 0 /dev/mmcblk1");
    }
    else
    {
      ETG_TRACE_USR4 (("bWriteMagicBootChain ERROR wrong parameter"));
      bRetVal  = FALSE;
    }
  }

  // Write magic to raw nor flash
  bRetVal = bWriteMagicToRawFlash (inFlashAddress, inValue);

  ETG_TRACE_USR4 (("bWriteMagicBootChain was left"));
  return bRetVal;
}


/*************************************************************************/
/*
* FUNCTION:   tBool dl_tclFlashOperations::bReadFlagFromRawFlash
*
* DESCRIPTION:  Read the flag content from raw nor flash
*
*
* PARAMETER INPUT:  tU32     u32FlashAdress e.g. 0x400E0000 for RootPartitionNumber, 0x400E0010 for DNL Magic
*                   tPU32    pu32Value    e.g  0x2342ABCD or  0xFFFFFFFF
*
* PARAMETER OUTPUT: none
*
* RETURNVALUE:      TRUE   in case of success
*                   FALSE  in case of error
*/
/*************************************************************************/
tBool bReadFlagFromRawFlash (off_t u32FlashAdress, tPU32 pu32Value)
{
  // MTD_IO: Set the raw flash device name for the linux driver and adjust the flash address to the flash device offset
  // calculate the flash address which shall be read
  // open the device
  // read 4 bytes from flash address
  //   log error case to /tmp/downloadPipe
  // caution: no proper error indication, no error handling

  ETG_TRACE_USR4(("bReadFlagFromRawFlash was entered"));

  tBool bRetVal = FALSE;

  tChar tmpBuffer [256] = {0};
  int fd;
  int retcode;
  char *buffer;
  std::string devname;
  fd      = -1;
  buffer  = 0;
  off_t u32FlashDevOffset = 0;
  off_t flashAddressOffset = 0;

  // Set the raw flash device name for the linux driver and adjust the flash address to the flash device offset
  bSetRawFlashDevName(u32FlashAdress, devname, &u32FlashDevOffset);
  flashAddressOffset = u32FlashAdress - u32FlashDevOffset;

  ETG_TRACE_USR4 (("DNL: ------------------------"));
  ETG_TRACE_USR4 (("DNL: bReadFlagFromRawFlash Address %d", flashAddressOffset));

  *pu32Value = 0;

  fd = open(devname.c_str(), O_RDONLY, 0x1A4);
  if(fd >= 0)
  {
    retcode =static_cast<int> (pread(fd, pu32Value, 4, flashAddressOffset) );
    if(retcode != 4)
    {
      printToDownloadPipe("read flash device ERROR");
    }
    else
    {
      sprintf(tmpBuffer, "read flag from flash adr=0x%8.8X val=0x%8.8X", (unsigned int)flashAddressOffset, (unsigned int)*pu32Value);
      printToDownloadPipe(tmpBuffer);
      bRetVal = TRUE; // Read OK
    }
    close(fd);
  }
  else
  {
    ETG_TRACE_USR4(("DNL: Flash open ERROR"));
  }

  ETG_TRACE_USR4 (("bReadFlagFromRawFlash was left"));
  return bRetVal;
}


const bool BootChainMgr::activateRecoveryMode()
{
  print2errmem("BootChainMgr::activateRecoveryMode executed\n");
  return bWriteMagicToRawFlash(DL_UPDATE_STATUS_FLAG_OFFSET, DL_UPDATE_STATE_RECOVERY);
}


const bool BootChainMgr::activateApplicationMode()
{
  print2errmem("BootChainMgr::activateApplicationMode executed\n");
  return bWriteMagicToRawFlash(DL_UPDATE_STATUS_FLAG_OFFSET, DL_UPDATE_STATE_SUCCESS);
}

const bool BootChainMgr::switchBootChainOne()
{
  print2errmem("BootChainMgr::switchBootChainOne executed\n");
  return bWriteMagicToRawFlash(DL_ROOTFS_FLAG_OFFSET, DL_ROOTFS_1);
}

const bool BootChainMgr::switchBootChainTwo()
{
  print2errmem("BootChainMgr::switchBootChainTwo executed\n");
  return bWriteMagicToRawFlash(DL_ROOTFS_FLAG_OFFSET, DL_ROOTFS_2);
}


} // namespace ai_sw_update {
} // namespace common {
