/**********************************************************************************************
* FILE:           dldab_ProcessCmd.cpp
* PROJECT:        NISSAN
* SW-COMPONENT:   Download
*-----------------------------------------------------------------------------------------------
* DESCRIPTION:  Process Command Class
*
* This file is the class dldab_tclProcessCmd defined. An object of this class is created by
* the dl_tclDABPlusMain class. This class is a singletone class. 
*------------------------------------------------------------------------------------------------
* COPYRIGHT: (c) 2011 Robert Bosch Car Multimedia GmbH, Hildesheim
* HISTORY:
* Date       |Author								  |Modification
* 23.02.11   |Ghatikar Nageetha (CM-AI/PJ-VW36 RBEI)  |Initial
*************************************************************************************************/

//----------------------------------------------------------------------------------------
// includes
//----------------------------------------------------------------------------------------
#include "dldab_ProcessCmd.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
/* Needed for Trace */
#define ETG_S_IMPORT_INTERFACE_GENERIC
//#define ET_TRACE_INFO_ON
#include "etg_if.h"
#endif

/* Needed for ETG trace */
#ifdef DL_ADR3_GEN2_CORE
#include "dl_Adr3TraceMacros.h"
#else
#include "ai_sw_update/common/base/imp/swupd_trace.h"
#endif

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SWUPDATE_ADR3
#include "trcGenProj/Header/dldab_ProcessCmd.cpp.trc.h"
#endif

#include "dl_tclHelp.h"

//-----------------------------------------------------------------------------------------
// static variables
//-----------------------------------------------------------------------------------------
tU32 dldab_tclProcessCmd::u32LastReportedProgress = 0xffffffff;
tU8 dldab_tclProcessCmd::u8LastError = 0xff;

/*******************************************************************************************/
#ifndef __SW_UPDATE_UNIT_TESTING__
// Implementation of the methods getInstanceOfdldab_tclProcessCmd and releaseInstanceOfdldab_tclProcessCmd
// In case of __SW_UPDATE_UNIT_TESTING__ they are implemented in mock!
SWU_IMPL_GET_AND_RELEASE_INSTANCE(dldab_tclProcessCmd);
#endif

// Implementation of the singleton methods
SWU_IMPL_SINGLETON(dldab_tclProcessCmd);
/*******************************************************************************************/

/******************************************************************************************
*
* FUNCTION:		dldab_tclProcessCmd::dldab_tclProcessCmd()
* 
* DESCRIPTION:  Constructor for dldab_tclProcessCmd class
*
* PARAMETER: void
*
* RETURN VALUE: none
*
******************************************************************************************/
dldab_tclProcessCmd::dldab_tclProcessCmd():pDataBuffer(OSAL_NULL),
                                           s32FileSize(0),
										   u32VersionAddr(0),
										   u32VersionSize(0),
										   u32FileChecksum(0x00000000),
										   m_poDnlFsmActions(OSAL_NULL)
{	
	ETG_TRACE_USR3(("Constructor dldab_tclProcessCmd"));


	// Init the static variable
	u8LastError = 0;
}
/******************************************************************************************
*
* FUNCTION:		dldab_tclProcessCmd::~dldab_tclProcessCmd()
* 
* DESCRIPTION:  Destructor for dldab_tclProcessCmd class
*
* PARAMETER: void
*
* RETURN VALUE: none
*
******************************************************************************************/
dldab_tclProcessCmd::~dldab_tclProcessCmd()
{	
	ETG_TRACE_USR3(("dldab_tclProcessCmd::Destructor"));

	if(m_poDnlFsmActions != OSAL_NULL)	{
		releaseInstanceOfdldab_DnlFsmActions();
		m_poDnlFsmActions = OSAL_NULL;
	}
	s32FileSize = 0;
	pDataBuffer = OSAL_NULL;
}

/******************************************************************************************
*
* FUNCTION:		dldab_tclProcessCmd::u32GetProgressOfAction()
* 
* DESCRIPTION:  Function to report the current progress of the ADR3-flashing  
*
* PARAMETER:	tVoid
*
* RETURN VALUE: tU32 u32Progress
*
******************************************************************************************/
tU32 dldab_tclProcessCmd::u32GetProgressOfAction() 
{
	ETG_TRACE_USR4(("dldab_tclProcessCmd::u32GetProgressOfAction"));
	return u32LastReportedProgress;
}

/******************************************************************************************
*
* FUNCTION:		dldab_tclProcessCmd::vSetProgressOfAction()
* 
* DESCRIPTION:  Function to report the current progress of the ADR3-flashing  
*
* PARAMETER:	tU32 u32Progress
*
* RETURN VALUE: tVoid
*
******************************************************************************************/
tVoid dldab_tclProcessCmd::vSetProgressOfAction(tU32 u32Progress) 
{
	ETG_TRACE_USR4(("dldab_tclProcessCmd::vSetProgressOfAction (%d)",u32Progress));

	if (u32LastReportedProgress != u32Progress)
	{
		ETG_TRACE_USR2(("ADR3-Progress: %d",u32Progress));
		u32LastReportedProgress = u32Progress;
	}
}

/******************************************************************************************
*
* FUNCTION:		dldab_tclProcessCmd::vBreakADR3Download()
* 
* DESCRIPTION:  Function break a running ADR3 download  
*
* PARAMETER:	tVoid
*
* RETURN VALUE: tVoid
*
******************************************************************************************/
tVoid dldab_tclProcessCmd::vBreakADR3Download() 
{
	ETG_TRACE_FATAL(("dldab_tclProcessCmd::vBreakADR3Download"));
	dl_tclHelp::vTrace("dldab_tclProcessCmd::vBreakADR3Download");
	dldab_DnlFsmActions::vBreakADR3Download();
}

/******************************************************************************************
*
* FUNCTION:		dldab_tclProcessCmd::WriteXLoaderCmd()
* 
* DESCRIPTION:  Function to store XLoader Buffer ptr and xloader size.  
*
* PARAMETER:	None
*
* RETURN VALUE: tBool
*
******************************************************************************************/
tBool dldab_tclProcessCmd::WriteXLoaderCmd(tPU8 pu8RamBuffer, tU32 u32SizeOfFile) 
{
	ETG_TRACE_USR3(("dldab_tclProcessCmd::WriteXLoaderCmd"));
	pDataBuffer = pu8RamBuffer;//Stores xloader Buffer Pointer
	s32FileSize = u32SizeOfFile;// and size , which is read in Main class
	return TRUE;	
}

/******************************************************************************************
*
* FUNCTION:		dldab_tclProcessCmd::bHandleInitilization()
* 
* DESCRIPTION:  Function to start uploading xloader , invoked from Main Class
*
* PARAMETER:	None
*
* RETURN VALUE: tBool
*
******************************************************************************************/
tBool dldab_tclProcessCmd::bHandleInitilization()
{	
	tBool bRetSt = FALSE;
	ETG_TRACE_USR3(("dldab_tclProcessCmd::bHandleInitilization"));

    m_poDnlFsmActions = getInstanceOfdldab_DnlFsmActions();

	if(OSAL_NULL == m_poDnlFsmActions)
	{
		return bRetSt;
	}
	m_poDnlFsmActions->vStoreReference(mpInstance);

	if(m_poDnlFsmActions->bPortOpened)//Once port is opened dnl is triggered
	{
		ETG_TRACE_USR3(("evDownload Trigger"));
		m_poDnlFsmActions->vAcceptEvent(DABPlusDNLFsm::evDownload,0);
	}

	if(m_poDnlFsmActions->bxLoaderDnlSuccess)
	{
		bRetSt = TRUE;//Ctrl goes back to Main class , Where Data dnl is triggered
	}	

	return bRetSt;
}

/******************************************************************************************
*
* FUNCTION:		dldab_tclProcessCmd::bHandleErase()
* 
* DESCRIPTION:  Function to start ERASE, invoked from Main Class
*
* PARAMETER:	tU32 u32SizeOfFlashFile
*
* RETURN VALUE: tBool
*
******************************************************************************************/
tBool dldab_tclProcessCmd::bHandleErase(tPU8 pu8RamBuffer, tU32 u32SizeOfFlashFile)
{	
	tBool bRetSt = FALSE;
	ETG_TRACE_USR3(("dldab_tclProcessCmd::bHandleErase"));

	s32FileSize = u32SizeOfFlashFile;// Size of the File to be flashed
	pDataBuffer = pu8RamBuffer;// Data buffer has to be set here because a READ is sent before ERASE (and response has to be received)

	if(OSAL_NULL == m_poDnlFsmActions)
	{
		return bRetSt;
	}

	if(m_poDnlFsmActions->bxLoaderDnlSuccess)
	{//Once Xloader is downloaded , Erase Flash
		//Event to trigger the Read cmd , ensures that xloader is downloaded 
		ETG_TRACE_USR3(("evxLdrRead Trigger"));
		m_poDnlFsmActions->vAcceptEvent(DABPlusDNLFsm::evxLdrRead,0);
	}
	if(m_poDnlFsmActions->bSecEraseSuccess)//Check Erase cmd is executed
	{
		bRetSt = TRUE;//Ctrl goes back to Main class , Where Data dnl is triggered
	}	

	return bRetSt;
}

/******************************************************************************************
*
* FUNCTION:		dldab_tclProcessCmd::bHandleRead()
*
* DESCRIPTION:  Function to read data from Flash, Invoked from the Main Class
*
* PARAMETER:	tPU8 pu8RamBuffer, tU32 u32SizeOfFile
*
* RETURN VALUE: tBool
*
******************************************************************************************/
tBool dldab_tclProcessCmd::bHandleRead(tPU8 pu8RamBuffer, tU32 u32SizeOfFlashFile)
{
	tBool bRetSt = FALSE;
	ETG_TRACE_USR3(("dldab_tclProcessCmd::bHandleRead"));

	pDataBuffer = pu8RamBuffer;// Data buffer for the data file to be Flashed
	s32FileSize = u32SizeOfFlashFile;// Size of the File to filled with flash content
	memset(pDataBuffer, 0x00, s32FileSize );	

	if(OSAL_NULL == m_poDnlFsmActions)
	{
		return bRetSt;
	}

	if(m_poDnlFsmActions->bxLoaderDnlSuccess)
	{//Once Xloader is downloaded , Read Flash
		//Event to trigger the Read cmd , ensures that xloader is downloaded
		ETG_TRACE_USR3(("evReadFlash Trigger"));
		m_poDnlFsmActions->vAcceptEvent(DABPlusDNLFsm::evReadFlash,0);
	}
	if(m_poDnlFsmActions->bReadFlashSuccess) //Check ReadFlash is executed completely
	{
		bRetSt = TRUE; //Ctrl goes back to Main class
	}	
	return bRetSt;
}

/******************************************************************************************
*
* FUNCTION:		dldab_tclProcessCmd::bHandleReadVersion()
*
* DESCRIPTION:  Function to read data from Flash, Invoked from the Main Class
*
* PARAMETER:	tPU8 pu8RamBuffer, tU32 u32AddressOfVersion, tU32 u32SizeOfVersion
*
* RETURN VALUE: tBool
*
******************************************************************************************/
tBool dldab_tclProcessCmd::bHandleReadVersion(tPU8 pu8RamBuffer, tU32 u32AddressOfVersion, tU32 u32SizeOfVersion)
{
	tBool bRetSt = FALSE;
	ETG_TRACE_USR3(("dldab_tclProcessCmd::bHandleReadVersion"));

	pDataBuffer = pu8RamBuffer;// Data buffer for the version "string"
	u32VersionAddr = u32AddressOfVersion;
	u32VersionSize = u32SizeOfVersion;
	memset(pDataBuffer, 0x00, u32VersionSize);

	if(OSAL_NULL == m_poDnlFsmActions)
	{
		return bRetSt;
	}

	if(m_poDnlFsmActions->bxLoaderDnlSuccess)
	{//Once Xloader is downloaded , Read Version
		//Event to trigger the Read Version cmd, ensures that xloader is downloaded
		ETG_TRACE_USR3(("evReadVersion Trigger"));
		m_poDnlFsmActions->vAcceptEvent(DABPlusDNLFsm::evReadVersion,0);
	}
	if(m_poDnlFsmActions->bReadFlashSuccess) //Check ReadFlash is executed completely
	{
		bRetSt = TRUE; //Ctrl goes back to Main class
	}	
	return bRetSt;
}

/******************************************************************************************
*
* FUNCTION:		dldab_tclProcessCmd::bHandleDataTransfer()
* 
* DESCRIPTION:  Function to upload data file to the Flash, Invoked from the Main Class
*
* PARAMETER:	tPU8 pu8RamBuffer, tU32 u32SizeOfFile
*
* RETURN VALUE: tBool
*
******************************************************************************************/
tBool dldab_tclProcessCmd::bHandleDataTransfer(tPU8 pu8RamBuffer, tU32 u32SizeOfFile)
{		
	ETG_TRACE_USR3(("dldab_tclProcessCmd::bHandleDataTransfer"));

	tBool bRetSt = FALSE;

	pDataBuffer = pu8RamBuffer;// Data buffer for the data file to be Flashed
	s32FileSize = u32SizeOfFile;// Size of the File to be flashed

	if(m_poDnlFsmActions == OSAL_NULL)	
	{
		ETG_TRACE_FATAL (("m_poDnlFsmActions == OSAL_NULL")); 	
		dl_tclHelp::vTrace("m_poDnlFsmActions == OSAL_NULL");
		return	bRetSt;	
	}
	if(m_poDnlFsmActions->bSecEraseSuccess)
	{	//Once Erasing is done , start Data Transfer
		ETG_TRACE_USR3(("Flash Erase Done !!"));
		//Event to Trigger the Data Cmd to Upload the Data to Flash
		m_poDnlFsmActions->vAcceptEvent(DABPlusDNLFsm::evDataTnfrCmd,0);							
	}
	if(m_poDnlFsmActions->bChecksumCalcDone)//Check whether Checksum is calculated
	{
		ETG_TRACE_USR3(("Calc Checksum Done !!"));
		bRetSt = TRUE;//Crtl Goes back to Main Class and the dnl is finished
	}		

	return bRetSt;
}

/******************************************************************************************
*
* FUNCTION:		dldab_tclProcessCmd::u32ComputeBufCRC2007
* 
* DESCRIPTION:  CRC_2007: CRC computation (simplest implementation) running on a buffer
*
* PARAMETER:	tU8* data_buf, tU32 num_bytes
*
* RETURN VALUE: none
*
******************************************************************************************/
tU32 dldab_tclProcessCmd::u32ComputeBufCRC2007(tPU8 data_buf, tU32 num_bytes) 
{
	tU32 u32Count;
	tU32 crc32 = CRC32_2007_DEFAULT;
	for (u32Count = 0; u32Count < num_bytes; u32Count++)
	{
		crc32 = CRC_ulwGetCRC32(data_buf[u32Count],crc32);
	}	
	u32FileChecksum = ~crc32;
	ETG_TRACE_USR3(("Checksum Calculated for Complete File: 0x%x",u32FileChecksum));
	return (~crc32);	
}		

/******************************************************************************************
*
* FUNCTION:		dldab_tclProcessCmd::CreateChecksumCmd( void* pArg)
* 
* DESCRIPTION:  CRC_2007: CRC computation (simplest implementation) running on a buffer
*
* PARAMETER: void
*
* RETURN VALUE: none
*
******************************************************************************************/
tU32 dldab_tclProcessCmd::CRC_ulwGetCRC32(tU8 ubData, tU32 ulwPrevCRC) const
{
	tU32 umCnt;
	tU32 ulwTemp = (ulwPrevCRC^ubData)&0x000000FFUL;
	for (umCnt = 8U; umCnt > 0U; umCnt--) /* Loop for each bit in char */
	{
		if ((ulwTemp & 1UL) != 0UL) {
			ulwTemp = (ulwTemp >> 1) ^ TROJAN_CRC_2007_POLYNOM;
		}
		else {
			ulwTemp >>= 1;
		}
	}
	ulwPrevCRC = (ulwPrevCRC>>8)^ulwTemp;
	return ulwPrevCRC; /* return new Checksum-value */
}

/******************************************************************************************
*
* FUNCTION:		dldab_tclProcessCmd::SetErrorInfo()
* 
* DESCRIPTION:  Function to evaluate and Display the error
*
* PARAMETER:	None
*
* RETURN VALUE: tVoid
*
******************************************************************************************/
tVoid dldab_tclProcessCmd::SetErrorInfo()
{
	tU8 errorCode = 0;
	const char* tmp = "Unknown";
	ETG_TRACE_USR3(("dldab_tclProcessCmd::SetErrorInfo"));
	if(OSAL_NULL != m_poDnlFsmActions)
		 errorCode = m_poDnlFsmActions->u8errorCode;
		 
	switch(errorCode){
		case DAB_ERASE_ERROR:			 tmp = "DAB_ERASE_ERROR";			u8LastError = DL_ADR3_ERASE_ERROR;            break;
		case DAB_WRITE_DATA_ERROR:		 tmp = "DAB_WRITE_DATA_ERROR";		u8LastError = DL_ADR3_WRITE_DATA_ERROR;       break;
		case DAB_READ_CMD_ERROR:		 tmp = "DAB_READ_CMD_ERROR";		u8LastError = DL_ADR3_READ_CMD_ERROR;         break;
		case DAB_CRC_CALC_ERROR:		 tmp = "DAB_CRC_CALC_ERROR";		u8LastError = DL_ADR3_CRC_CALC_ERROR;         break;
		case DAB_CRC_CALC_NOT_SUPORTED:	 tmp = "DAB_CRC_CALC_NOT_SUPORTED";	u8LastError = DL_ADR3_CRC_CALC_NOT_SUPORTED;  break;
		case DAB_ABORTED_BY_BREAK:		 tmp = "DAB_ABORTED_BY_BREAK";		u8LastError = DL_ADR3_ABORTED_BY_BREAK;       break;
		case DAB_NO_XLOADER_RESPONSE:    tmp = "DAB_NO_XLOADER_RESPONSE";   u8LastError = DL_ADR3_NO_XLOADER_RESPONSE;    break;
		case DAB_RESPONSE_TIMEOUT:       tmp = "DAB_RESPONSE_TIMEOUT";      u8LastError = DL_ADR3_RESPONSE_TIMEOUT;       break;
		case DAB_UNKNOWN_MSG_ID_ERROR:	 tmp = "DAB_UNKNOWN_MSG_ID_ERROR";	u8LastError = DL_ADR3_UNKNOWN_MSG_ID_ERROR;   break;				
		default:						 tmp = "Unknown";					u8LastError = DL_ADR3_UNKNOWN_ERROR;          break;
	}
	ETG_TRACE_USR3(("SetErrorInfo: %s",tmp));
	return;
}

/******************************************************************************************
*
* FUNCTION:		dldab_tclProcessCmd::u8GetLastError()
* 
* DESCRIPTION:  Get the code of the last error
*
* PARAMETER:	None
*
* RETURN VALUE: tU8
*
******************************************************************************************/
tU8 dldab_tclProcessCmd::u8GetLastError()
{
	ETG_TRACE_USR3(("dldab_tclProcessCmd::u8GetLastError"));
	return u8LastError;
}

/******************************************************************************************
*
* FUNCTION:		dldab_tclProcessCmd::vResetADR3()
* 
* DESCRIPTION:  Reset ADR3
*
* PARAMETER:	None
*
* RETURN VALUE: tVoid
*
******************************************************************************************/
tVoid dldab_tclProcessCmd::vResetADR3()
{
	if(OSAL_NULL != m_poDnlFsmActions)
		m_poDnlFsmActions->vResetADR3();
}


//EOF


