/****************************************************************************************
* FILE:			tclADR3Driver.cpp
* PROJECT:      Nissan
* SW-COMPONENT: Download
*-----------------------------------------------------------------------------------------
*
* DESCRIPTION: tclADR3Driver class for the ADR3 communication
*              
*-----------------------------------------------------------------------------------------
* AUTHOR:		Ghatikar Nageetha (CM-AI/PJ-VW36 RBEI)
* COPYRIGHT:    (c) 2011 Robert Bosch Car Multimedia GmbH, Hildesheim
* HISTORY:      Rev 1.0 
*				Initial version
******************************************************************************************/

//-----------------------------------------------------------------------------------------
// includes
//-----------------------------------------------------------------------------------------
#include "../dldab_includes.h"
#include "tclADR3Driver.h"
#include "dev_adr3ctrl.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
/* Needed for Trace */
#define ETG_S_IMPORT_INTERFACE_GENERIC
//#define ET_TRACE_INFO_ON
#include "etg_if.h"
#endif

/* Needed for ETG trace */
#ifdef DL_ADR3_GEN2_CORE
#include "../dl_Adr3TraceMacros.h"
#else
#include "ai_sw_update/common/base/imp/swupd_trace.h"
#endif

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_I_TRACE_CHANNEL TR_TTFIS_SWUPDATE_CTRL
#define ETG_I_TTFIS_CMD_PREFIX "DNL_"
#define ETG_I_FILE_PREFIX tclADR3Driver::
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SWUPDATE_ADR3
#include "trcGenProj/Header/tclADR3Driver.cpp.trc.h"
#endif

#include "../dl_tclHelp.h"

//--------------------------------------------------------------------------------------------
// static initialization
//--------------------------------------------------------------------------------------------
tU32 tclADR3Driver::u32ADR3CtrlState = 0;

/******************************************************************************************
*
* FUNCTION:		tclADR3Driver::vCallback()
* 
* DESCRIPTION:  Callback function for the state of ADR3Ctrl
*
* PARAMETER:	tU32 u32State - the current state of ADR3Ctrl
*
* RETURN VALUE: tVoid
*
******************************************************************************************/
tVoid tclADR3Driver::vCallback(tU32 u32State)
{
	ETG_TRACE_USR3(("----------------------------vCallback---------------%d-----------------", u32State));
	u32ADR3CtrlState=u32State;
}

/******************************************************************************************
*
* FUNCTION:		tclADR3Driver::bOpen()
* 
* DESCRIPTION:  Opens ADR3 device port and sets IOControl to BOOTMODE SPI
*
* PARAMETER:	None
*
* RETURN VALUE: TRUE = No error, False = Error
*
******************************************************************************************/
tBool tclADR3Driver::bOpen ()
{
	tS32 s32Return = OSAL_ERROR;
	tU32 u32WaitCounter = 0;
	ETG_TRACE_USR4(("tclADR3Driver::bOpen"));

	fd = OSAL_IOOpen("/dev/adr3ctrl", OSAL_EN_READWRITE);
	u32ADR3CtrlState = 0;
    if (fd == OSAL_ERROR)
	{
		ETG_TRACE_FATAL(("Open failure: %d", fd));
		dl_tclHelp::vTrace("Open failure for /dev/adr3ctrl");
	}
	else
	{
		s32Return = OSAL_s32IOControl( fd, OSAL_C_S32_IOCTRL_ADR3CTRL_REGISTER_RESET_CALLBACK, (intptr_t) &tclADR3Driver::vCallback );
		if (OSAL_ERROR == s32Return)
		{
			NORMAL_M_ASSERT_ALWAYS();
		}
		s32Return = OSAL_s32IOControl(fd, OSAL_C_S32_IOCTRL_ADR3CTRL_SET_BOOTMODE_SPI, OSAL_NULL);
		while ((u32WaitCounter<99) && (u32ADR3CtrlState!=ADR3CTRL_ADR3_STATE_DNL))
		{
			OSAL_s32ThreadWait(100); // Wait some time after setting ADR3 Bootmode
			u32WaitCounter++;
			ETG_TRACE_USR3(("-----u32ADR3CtrlState------%d-------", u32ADR3CtrlState));
		}
	}
	return ((fd != OSAL_ERROR) && (s32Return != OSAL_ERROR) && (u32ADR3CtrlState==ADR3CTRL_ADR3_STATE_DNL));
}

/*******************************************************************************************
*
* FUNCTION:		tclADR3Driver::bClose()
* 
* DESCRIPTION:  Closes OSAL_C_STRING_DEVICE_ADR3_DOWNLOAD device port
*
* PARAMETER:	None
*
* RETURN VALUE: TRUE = No error, False = Error
*
*******************************************************************************************/
tBool tclADR3Driver::bClose()
{
	tS32 s32Return = OSAL_ERROR;
	tU32 u32WaitCounter = 0;
	ETG_TRACE_USR4(("tclADR3Driver::bClose"));

	u32ADR3CtrlState = 0;

	s32Return = OSAL_s32IOControl(fd, OSAL_C_S32_IOCTRL_ADR3CTRL_SET_BOOTMODE_NORMAL, OSAL_NULL);
	while ((u32WaitCounter<99) && (u32ADR3CtrlState!=ADR3CTRL_ADR3_STATE_ALIVE))
	{
		OSAL_s32ThreadWait(100); // Wait some time after setting ADR3 Bootmode
		u32WaitCounter++;
		ETG_TRACE_USR3(("-----u32ADR3CtrlState------%d-------", u32ADR3CtrlState));
	}
	
	if ((s32Return == OSAL_ERROR) || (u32ADR3CtrlState!=ADR3CTRL_ADR3_STATE_ALIVE))
	{
		ETG_TRACE_FATAL(("tclADR3Driver::bClose Cannot reset ADR back to NORMAL MODE"));
		dl_tclHelp::vTrace("tclADR3Driver::bClose Cannot reset ADR back to NORMAL MODE");
		NORMAL_M_ASSERT_ALWAYS();
	}

   s32Return = OSAL_s32IOClose(fd);
   fd = OSAL_ERROR;
   return (s32Return != OSAL_ERROR);
}

/*******************************************************************************************
*
* FUNCTION:		tclADR3Driver::s32Write()
* 
* DESCRIPTION:  Writes the buffer data send
*
* PARAMETER:	u8Buffer - > Buffer Ptr , u32Count - no of Bytes to be written
*
* RETURN VALUE: No of Bytes wrote in case of success ,In case of Error Returns OSAL_ERROR(-1)
*
*********************************************************************************************/
tS32 tclADR3Driver::s32Write (tPCS8 pBuffer, tU32 u32Count)
{
   tS32 s32Count = 0;
   ETG_TRACE_USR4(("tclADR3Driver::s32Write"));
   s32Count = OSAL_s32IOWrite(fd, pBuffer, u32Count);
   return s32Count;
}

/*******************************************************************************************
*
* FUNCTION:		tclADR3Driver::s32Read()
* 
* DESCRIPTION:  Read the data into the buffer
*
* PARAMETER:	u8Buffer - > Buffer Ptr , u32Count - no of Bytes to be written
*
* RETURN VALUE: No of Bytes read in case of success ,In case of Error Returns OSAL_ERROR(-1)
*
*********************************************************************************************/
tS32 tclADR3Driver::s32Read (tPS8 pBuffer, tU32 u32Count)
{
   tS32 s32Count = 0;
   ETG_TRACE_USR4(("tclADR3Driver::s32Read"));
   s32Count = OSAL_s32IORead(fd, pBuffer, u32Count);
   return s32Count;
}

/*********************************************************************************************
*
* FUNCTION:		tclADR3Driver::bSetReadTimeout()
* 
* DESCRIPTION:  Time till which rsp is polled
*
* PARAMETER:	None
*
* RETURN VALUE: TRUE = No error, False = Error
*
**********************************************************************************************/
tBool tclADR3Driver::bSetReadTimeout()
{
   ETG_TRACE_USR4(("tclADR3Driver::bSetReadTimeout"));
   tS32 s32Return = OSAL_s32IOControl(fd, OSAL_C_S32_IOCTRL_ADR3CTRL_SET_BLOCKINGMODE_READ_TIMEOUT, 1000);
   return (s32Return != OSAL_ERROR);
}

/**********************************************************************************************
*
* FUNCTION:		tclADR3Driver::bReset()
* 
* DESCRIPTION:  Resets the Chip , invoked upon Error
*
* PARAMETER:	None
*
* RETURN VALUE: TRUE = No error, False = Error
*
**********************************************************************************************/
tBool tclADR3Driver::bReset()
{
   ETG_TRACE_USR4(("tclADR3Driver::bReset"));
   tS32 s32Return = OSAL_s32IOControl(fd, OSAL_C_S32_IOCTRL_ADR3CTRL_RESET_ADR3, OSAL_NULL);
   return (s32Return != OSAL_ERROR);
}


//EOF




