/***********************************************************************/
/*!
* \file  DynamicLibHandler.cpp
* \brief Wrapper class for shared library usage
*************************************************************************
\verbatim

PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    Wrapper class for shared library usage
AUTHOR:         Unmukt Jain
COPYRIGHT:      &copy; RBEI

HISTORY:
Date        | Author                | Modification
27.02.2018  | Unmukt Jain           | Initial Version

\endverbatim
*************************************************************************/

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/
#include <dlfcn.h>
#include "DynamicLibHandler.h"


//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
   #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_UTILS
      #include "trcGenProj/Header/DynamicLibHandler.cpp.trc.h"
   #endif
#endif


/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/


/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

/***************************************************************************
*********************************PUBLIC*************************************
***************************************************************************/

/***************************************************************************
** FUNCTION:  DynamicLibHandler::DynamicLibHandler(const t_String &szLibName)
***************************************************************************/
DynamicLibHandler::DynamicLibHandler(const t_String &szLibName): m_vHandle(NULL)
{
   ETG_TRACE_USR1(("DynamicLibHandler() entered with %s",szLibName.c_str()));
   m_vHandle = dlopen(szLibName.c_str(),RTLD_LAZY | RTLD_NOLOAD); //check if the library is already loaded.
   if(NULL == m_vHandle)
   {
       //libary is not loaded, try to load the library
       m_vHandle = dlopen(szLibName.c_str(),RTLD_NOW);
       if(NULL == m_vHandle)
       {
           ETG_TRACE_USR1(("[DESC]::DynamicLibHandler failed to load library!!"));
       }
   }
}

/***************************************************************************
** FUNCTION: DynamicLibHandler::~DynamicLibHandler()
***************************************************************************/
DynamicLibHandler::~DynamicLibHandler()
{
    m_vHandle = NULL;
}

/***************************************************************************
** FUNCTION: DynamicLibHandler::u32GetSymbol(..)
***************************************************************************/
t_U32 DynamicLibHandler::u32GetSymbol(const t_String &szSymbolName, t_Void **vpoSymbol)
{
    ETG_TRACE_USR1(("[FUNC]::DynamicLibHandler u32GetSymbol entered"));
    t_U32 u32res = 0;
    *vpoSymbol = NULL;
    if(NULL != m_vHandle)
    {
        dlerror(); //clear previous errors
        *vpoSymbol = dlsym(m_vHandle,szSymbolName.c_str());
        char *cError = dlerror();
        if(NULL != cError)
        {
            //error has occurred.
            ETG_TRACE_USR1(("[DESC]::DynamicLibHandler u32GetSymbol error = %s",cError));
        }
        else
        {
            u32res = 1;
        }
    }
    return u32res;
}

/***************************************************************************
** FUNCTION: DynamicLibHandler::u32CloseLib()
***************************************************************************/
t_U32 DynamicLibHandler::u32CloseLib()
{
    ETG_TRACE_USR1(("[FUNC]::DynamicLibHandler u32CloseLib entered"));
    t_U32 u32res = 0;
    if(NULL != m_vHandle)
    {
        u32res = ( dlclose(m_vHandle) == 0 ) ? 1 : 0;
    }
    return u32res;
}

///////////////////////////////////////////////////////////////////////////////
// <EOF>
