/*!
 *******************************************************************************
 * \file              spi_tclMPlayClientHandler.cpp
 * \brief             DiPO Client handler class
 *******************************************************************************
 \verbatim
 PROJECT:        G3G
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    DiPO Client handler class for Media player service
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 08.1.2014  |  Shihabudheen P M            | Initial Version
 02.04.2014 | Hari Priya E R               | Implemented Timer for triggering
                                             transfer of GPS Data
 15.04.2014 | Ramya Muthy                  | Implemented Location data transfer
                                             and removed timer implementation.
 13.06.2014 | Ramya Murthy                 | Implementation for MPlay FI extn 
                                             for location info.
 31.07.2014 | Ramya Murthy                 | Changes for dynamic registration of properties
 05.11.2014 | Ramya Murthy                 | Implementation for Application metadata.
 05.11.2014 | Ramya Murthy                 | Implementation of revised CarPlay media concept 
 23.04.2015 | Ramya Murthy                 | Changes to provide PASCD and PAGCD sentences
 06.05.2015 | Tejaswini HB                 | Lint Fix
 26.05.2015 | Tejaswini H B(RBEI/ECP2)     | Added Lint comments to suppress C++11 Errors
 14.12.2015 | SHITANSHU SHEKHAR(RBEI/ECP2) | Implemented interfaces as per new Media player FI:
                                           | vOnDiPONowPlayingStatus(), vOnDiPOPlayBackStatus()
                                           | vOnDiPOPlayBackShuffleModeStatus(),
                                           | vOnDiPOPlayBackRepeatModeStatus(),
                                           | enGetPhoneCallState(), enGetPhoneCallDirection(),
                                           | enGetCallStateUpdateService(),
                                           | enGetCallStateUpdateDisconnectReason(),
                                           | vOnDiPOCallStateStatus(), enGetRegistrationStatus()
                                           | vOnDiPOCommunicationsStatus()
 01.03.2017 |  Shiva Kumar G               | Adaptations for CarPlay Wireless
 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 | 1)system- and project- includes
 | 2)needed interfaces from external components
 | 3)internal and external interfaces from this component
 |----------------------------------------------------------------------------*/
#define AHL_S_IMPORT_INTERFACE_GENERIC
#define AHL_S_IMPORT_INTERFACE_CCA_EXTENSION
#include "ahl_if.h"
//Include common fi interface
#define FI_S_IMPORT_INTERFACE_BASE_TYPES
#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
#include "common_fi_if.h"

#define GENERICMSGS_S_IMPORT_INTERFACE_GENERIC
#include "generic_msgs_if.h"

#include <cstdlib>
#include <algorithm>
#include <utility>
#include "FIMsgDispatch.h"
#include "XFiObjHandler.h"
using namespace std;
using namespace shl::msgHandler;

#define DIPO_MPLY_MAJORVERSION MPLAY_APPCONTROLFI_C_U16_SERVICE_MAJORVERSION
#include "ExtInterfacesTypes.h"
#include "spi_tclFactory.h"
#include "spi_tclAudio.h"
#include "spi_tclRespInterface.h"
#include "spi_tclMplayResp.h"
#include "spi_tclDataServiceTypes.h"
#include "spi_LoopbackTypes.h"
#include "spi_tclMPlayClientHandler.h"
#include "spi_tclDeviceIDDataIntf.h"


#include "Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_CONNECTIVITY
#include "trcGenProj/Header/spi_tclMPlayClientHandler.cpp.trc.h"
#endif

//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e19 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e58 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e48 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e808 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e64 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e746 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e515 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e516 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported	

/******************************************************************************
 | defines and macros and constants(scope: module-local)
 |----------------------------------------------------------------------------*/
#define MPLAYFI_CHAR_SET_UTF8   (mplay_fi_tclString::FI_EN_UTF8)
static const t_U16 MAXANGLE = 180;
/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/

/*! 
 * \XFiObjHandler<mplay_appcontrolfi_tclMsgDiPODeviceConnectionsStatus> spi_tDevConnectionStatus
 * \brief  Property status, with auto extract & auto destroy feature.
 */
typedef XFiObjHandler<mplay_appcontrolfi_tclMsgDiPODeviceConnectionsStatus>
         spi_tDevConnectionStatus;

/*! 
 * \XFiObjHandler<mplay_appcontrolfi_tclMsgDiPOPlaytimeStatus> spi_tMsgPlayTimeStatus
 * \brief  Property status, with auto extract & auto destroy feature.
 */
typedef XFiObjHandler<mplay_appcontrolfi_tclMsgDiPOPlaytimeStatus>
         spi_tMsgPlayTimeStatus;

/*!
 * \XFiObjHandler<mplay_appcontrolfi_tclMsgDiPONowPlayingStatus> spi_tMsgNowPlayingStatus
 * \brief  Property status, with auto extract & auto destroy feature.
 */
typedef XFiObjHandler<mplay_appcontrolfi_tclMsgDiPONowPlayingStatus>
         spi_tMsgNowPlayingStatus;

/*!
 * \XFiObjHandler<mplay_appcontrolfi_tclMsgDiPOPlaybackStatusStatus> spi_tMsgPlayBackStatus
 * \brief  Property status, with auto extract & auto destroy feature.
 */
typedef XFiObjHandler<mplay_appcontrolfi_tclMsgDiPOPlaybackStatusStatus>
         spi_tMsgPlayBackStatus;

/*!
 * \XFiObjHandler<mplay_appcontrolfi_tclMsgDiPOPlaybackShuffleModeStatus> spi_tMsgPlayBackShuffleModeStatus
 * \brief  Property status, with auto extract & auto destroy feature.
 */
typedef XFiObjHandler<mplay_appcontrolfi_tclMsgDiPOPlaybackShuffleModeStatus>
         spi_tMsgPlayBackShuffleModeStatus;

/*!
 * \XFiObjHandler<mplay_appcontrolfi_tclMsgDiPOPlaybackRepeatModeStatus> spi_tMsgPlayBackRepeatModeStatus
 * \brief  Property status, with auto extract & auto destroy feature.
 */
typedef XFiObjHandler<mplay_appcontrolfi_tclMsgDiPOPlaybackRepeatModeStatus>
         spi_tMsgPlayBackRepeatModeStatus;

/*!
 * \XFiObjHandler<mplay_appcontrolfi_tclMsgDiPOCallStateStatus> spi_tMsgCallStateStatus
 * \brief  Property status, with auto extract & auto destroy feature.
 */
typedef XFiObjHandler<mplay_appcontrolfi_tclMsgDiPOCallStateStatus>
         spi_tMsgCallStateStatus;
		 
/*!
 * \XFiObjHandler<mplay_appcontrolfi_tclMsgDiPOCallDurationStatus> spi_tMsgCallDurationStatus
 * \brief  Property status, with auto extract & auto destroy feature.
 */
typedef XFiObjHandler<mplay_appcontrolfi_tclMsgDiPOCallDurationStatus>
         spi_tMsgCallDurationStatus;

/*!
 * \XFiObjHandler<mplay_appcontrolfi_tclMsgDiPOCommunicationsStatus> spi_tMsgCommunicationsStatus
 * \brief  Property status, with auto extract & auto destroy feature.
 */
typedef XFiObjHandler<mplay_appcontrolfi_tclMsgDiPOCommunicationsStatus>
         spi_tMsgCommunicationsStatus;

/*!
 * \XFiObjHandler<mplay_appcontrolfi_tclMsgCloseMethodResult> spi_tMsgDipoActiveResult
 * \brief  Method result, with auto extract & auto destroy feature.
 */
typedef XFiObjHandler<mplay_appcontrolfi_tclMsgDiPOActiveDeviceMethodResult>
         spi_tMsgDipoActiveResult;

/*! 
 * \XFiObjHandler<mplay_appcontrolfi_tclMsgDiPORoleSwitchRequestMethodResult> spi_tMsgDipoRoleSwitchResult
 * \brief  Method result, with auto extract & auto destroy feature.
 */
typedef XFiObjHandler<
         mplay_appcontrolfi_tclMsgDiPORoleSwitchRequestMethodResult>
         spi_tMsgDipoRoleSwitchResult;

/*! 
 * \XFiObjHandler<mplay_appcontrolfi_tclMsgDiPOGetMediaObjectAlbumArtInfoMethodResult> spi_tMsgDipoGetAlbArtInfoResult
 * \brief  Method result, with auto extract & auto destroy feature.
 */
typedef XFiObjHandler<
         mplay_appcontrolfi_tclMsgDiPOGetMediaObjectAlbumArtInfoMethodResult>
         spi_tMsgDipoGetAlbArtInfoResult;

/*! 
 * \XFiObjHandler<mplay_appcontrolfi_tclMsgDiPOGetMediaObjectAlbumArtMethodResult> spi_tMsgDiPOGetAlbumArt
 * \brief  Method result, with auto extract & auto destroy feature.
 */
typedef XFiObjHandler<
         mplay_appcontrolfi_tclMsgDiPOGetMediaObjectAlbumArtMethodResult>
         spi_tMsgDiPOGetAlbumArt;

/*! 
 * \XFiObjHandler<mplay_appcontrolfi_tclMsgDiPOTransferGPSDataMethodResult> spi_tMsgDipoTransGPSDataReult
 * \brief  Method result, with auto extract & auto destroy feature.
 */
typedef XFiObjHandler<mplay_appcontrolfi_tclMsgDiPOTransferGPSDataMethodResult>
         spi_tMsgDipoTransGPSDataReult;

/*! 
 * \XFiObjHandler<mplay_appcontrolfi_tclMsgDiPOTransferDRDataMethodResult> spi_tMsgDipoTransDRDataResult
 * \brief  Method result, with auto extract & auto destroy feature.
 */
typedef XFiObjHandler<mplay_appcontrolfi_tclMsgDiPOTransferDRDataMethodResult>
         spi_tMsgDipoTransDRDataResult;

/*!
 * \XFiObjHandler<mplay_appcontrolfi_tclMsgDiPOLocationInfoStatus> spi_tMsgDipoLocInfoStatus
 * \brief  Method result, with auto extract & auto destroy feature.
 */
typedef XFiObjHandler<mplay_appcontrolfi_tclMsgDiPOLocationInfoStatus>
         spi_tMsgDipoLocInfoStatus;

/*!
 * \XFiObjHandler<mplay_appcontrolfi_tclMsgDiPORequestAudioDeviceMethodResult> spi_tMsgDipoReqAudioDevResult
 * \brief  Method result, with auto extract & auto destroy feature.
 */
typedef XFiObjHandler<mplay_appcontrolfi_tclMsgDiPORequestAudioDeviceMethodResult>
         spi_tMsgDipoReqAudioDevResult;

/*!
 * \XFiObjHandler<mplay_appcontrolfi_tclMsgDiPOReleaseAudioDeviceMethodResult> spi_tMsgDipoRelAudioDevResult
 * \brief  Method result, with auto extract & auto destroy feature.
 */
typedef XFiObjHandler<mplay_appcontrolfi_tclMsgDiPOReleaseAudioDeviceMethodResult>
         spi_tMsgDipoRelAudioDevResult;

//! Connection get
typedef mplay_appcontrolfi_tclMsgDiPODeviceConnectionsGet MplayConnectionsGet; 

//! DiPoNowPlaying get
typedef mplay_appcontrolfi_tclMsgDiPONowPlayingGet spi_tMsgDiPoNowPlayingGet;

/*!
 * \XFiObjHandler<mplay_appcontrolfi_tclMsgNativeTransportStartStatus> spi_tMsgNativeTransStartStatus
 * \brief Property status, with auto extract & auto destroy feature.
 */
typedef XFiObjHandler<mplay_appcontrolfi_tclMsgNativeTransportStartStatus>
                spi_tMsgNativeTransStartStatus;

/*!
 * \XFiObjHandler<mplay_appcontrolfi_tclMsgNativeTransportStopStatus> spi_tMsgNativeTransStopStatus
 * \brief Property status, with auto extract & auto destroy feature.
 */
typedef XFiObjHandler<mplay_appcontrolfi_tclMsgNativeTransportStopStatus>
                spi_tMsgNativeTransStopStatus;

/*!
 * \XFiObjHandler<mplay_appcontrolfi_tclMsgConnectedBluetoothProfilesStatus> spi_tConnectedBtProfilesStatus
 * \brief  Property status, with auto extract & auto destroy feature.
 */
typedef XFiObjHandler<mplay_appcontrolfi_tclMsgConnectedBluetoothProfilesStatus>
                spi_tConnectedBtProfilesStatus;

/*!
 * \XFiObjHandler<mplay_appcontrolfi_tclMsgConnectMethodResult> spi_tMsgConnectResult
 * \brief  Method result, with auto extract & auto destroy feature.
 */
typedef XFiObjHandler<mplay_appcontrolfi_tclMsgConnectMethodResult>
                spi_tMsgConnectResult;

/*!
* \XFiObjHandler<mplay_appcontrolfi_tclMsgDiPOStartIAP2OverCarPlayWifiMethodResult> spi_tMsgStartiAP2OverCarPlayResult
* \brief  Method result, with auto extract & auto destroy feature for DiPOStartiAP2OverCarPlayWifi method.
*/
typedef XFiObjHandler<mplay_appcontrolfi_tclMsgDiPOStartIAP2OverCarPlayWifiMethodResult>
spi_tMsgStartiAP2OverCarPlayResult;

/*!
 * \XFiObjHandler<mplay_appcontrolfi_tclMsgDiPOPowerStatus> spi_tMsgDiPOPowerStatus
 * \brief  Property status, with auto extract & auto destroy feature.
 */
typedef XFiObjHandler<mplay_appcontrolfi_tclMsgDiPOPowerStatus>
         spi_tMsgDiPOPowerStatus;

/*!
* \XFiObjHandler<mplay_appcontrolfi_tclMsgDiPORouteGuidanceStatus> spi_tMsgDiPORouteGuidanceStatus
* \brief  Method result, with auto extract & auto destroy feature.
*/
typedef XFiObjHandler<mplay_appcontrolfi_tclMsgDiPORouteGuidanceStatus>
spi_tMsgDiPORouteGuidanceStatus;

/*!
* \XFiObjHandler<mplay_appcontrolfi_tclMsgDiPORouteGuidanceManeuverStatus> spi_tMsgDiPORouteGuidanceManeuverStatus
* \brief  Method result, with auto extract & auto destroy feature.
*/
typedef XFiObjHandler<mplay_appcontrolfi_tclMsgDiPORouteGuidanceManeuverStatus>
spi_tMsgDiPORouteGuidanceManeuverStatus;
/*
* \brief  Provides error status for roleswitch
*/
typedef XFiObjHandler<mplay_appcontrolfi_tclMsgDiPOCommunicationErrorStatus>
spi_tMsgDiPOCommErrorStatus;


/*!
* \XFiObjHandler<mplay_appcontrolfi_tclMsgDiPOGPRMCDataStatusValuesStatus> spi_tMsgDiPORouteGuidanceManeuverStatus
* \brief  Method result, with auto extract & auto destroy feature.
*/
typedef XFiObjHandler<mplay_appcontrolfi_tclMsgDiPOGPRMCDataStatusValuesStatus>
spi_tMsgDiPOGPRMCDataStatusValuesStatus;

/*!
 * \XFiObjHandler<mplay_appcontrolfi_tclMsgDiPOSetRouteGuidanceDisplayComponentInfoMethodStart> spi_tMsgDiPOSetRouteGuidanceDispComponentInfoMS
 * \brief  Set RouteGuidanceDisplayComponent Info Method Start
 */
typedef mplay_appcontrolfi_tclMsgDiPOSetRouteGuidanceDisplayComponentInfoMethodStart spi_tMsgDiPOSetRouteGuidanceDispComponentInfoMS;

/*!
 * \XFiObjHandler<mplay_appcontrolfi_tclMsgDiPORouteGuidance2Status> spi_tMsgDiPORouteGuidanceStatus
 * \brief  Method result, with auto extract & auto destroy feature.
 */
typedef XFiObjHandler<mplay_appcontrolfi_tclMsgDiPORouteGuidance2Status> spi_tMsgCarPlayRouteGuidanceStatus;

/*!
 * \XFiObjHandler<mplay_appcontrolfi_tclMsgDiPORouteGuidanceManeuver2Status> spi_tMsgCarPlayRouteGuidanceManeuverStatus
 * \brief  Method result, with auto extract & auto destroy feature.
 */
typedef XFiObjHandler<mplay_appcontrolfi_tclMsgDiPORouteGuidanceManeuver2Status> spi_tMsgCarPlayRouteGuidanceManeuverStatus;

/*!
* \mplay_appcontrolfi_tclMsgDiPOSetCallControlsConfigurationMethodStart spi_tMsgSetCallControlsConfigMS
* \brief  DiPOSetCallControlsConfiguration Method Start
*/
typedef mplay_appcontrolfi_tclMsgDiPOSetCallControlsConfigurationMethodStart spi_tMsgSetCallControlsConfigMS;

/*!
* \mplay_appcontrolfi_tclMsgDiPOAcceptCallMethodStart spi_tMsgDiPOAcceptCallMS
* \brief  DiPOAcceptCall Method Start
*/
typedef mplay_appcontrolfi_tclMsgDiPOAcceptCallMethodStart spi_tMsgDiPOAcceptCallMS;

/*!
* \mplay_appcontrolfi_tclMsgDiPOEndCallMethodStart spi_tMsgDiPOEndCallMS
* \brief  DiPOEndCall Method Start
*/
typedef mplay_appcontrolfi_tclMsgDiPOEndCallMethodStart spi_tMsgDiPOEndCallMS;

/*!
 * \XFiObjHandler<mplay_appcontrolfi_tclMsgDiPODeviceTimeUpdateStatus> spi_tMsgDiPODeviceTimeUpdate
 * \brief  Property status, with auto extract & auto destroy feature.
 */
typedef XFiObjHandler<mplay_appcontrolfi_tclMsgDiPODeviceTimeUpdateStatus> spi_tMsgDiPODeviceTimeUpdate;

static trIAPKeyCode aIapKeyCode [] =
{
   {e32MULTIMEDIA_PLAY, mplay_fi_tcl_e8_DiPOPlaypackAction::FI_EN_E8DIPO_PBA_PLAY},
   {e32MULTIMEDIA_PAUSE, mplay_fi_tcl_e8_DiPOPlaypackAction::FI_EN_E8DIPO_PBA_PAUSE},
   {e32MULTIMEDIA_STOP, mplay_fi_tcl_e8_DiPOPlaypackAction::FI_EN_E8DIPO_PBA_STOP},
   {e32MULTIMEDIA_NEXT, mplay_fi_tcl_e8_DiPOPlaypackAction::FI_EN_E8DIPO_PBA_NEXT},
   {e32MULTIMEDIA_PREVIOUS,mplay_fi_tcl_e8_DiPOPlaypackAction::FI_EN_E8DIPO_PBA_PREV},
   {e32MULTIMEDIA_FORWARD,mplay_fi_tcl_e8_DiPOPlaypackAction::FI_EN_E8DIPO_PBA_FFWD_START},
   {e32MULTIMEDIA_REWIND,mplay_fi_tcl_e8_DiPOPlaypackAction::FI_EN_E8DIPO_PBA_FREV_START}
};

t_U32 spi_tclMPlayClientHandler::su32RequestID = 0;

/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/
   static const t_U32 cou32SPI_SERVICE_ID = CCA_C_U16_SRV_SMARTPHONEINTEGRATION;

/******************************************************************************
 ** CCA MESSAGE MAP
******************************************************************************/
// Message map for the media player fi
BEGIN_MSG_MAP(spi_tclMPlayClientHandler, ahl_tclBaseWork)

ON_MESSAGE_SVCDATA( MPLAY_APPCONTROLFI_C_U16_DIPODEVICECONNECTIONS,
         AMT_C_U8_CCAMSG_OPCODE_STATUS, vOnDiPODeviceConnectionsStatus)

ON_MESSAGE_SVCDATA( MPLAY_APPCONTROLFI_C_U16_DIPOPLAYTIME,
         AMT_C_U8_CCAMSG_OPCODE_STATUS, vOnDiPOPlaytimeStatus)

ON_MESSAGE_SVCDATA( MPLAY_APPCONTROLFI_C_U16_DIPONOWPLAYING,
         AMT_C_U8_CCAMSG_OPCODE_STATUS, vOnDiPONowPlayingStatus)

ON_MESSAGE_SVCDATA( MPLAY_APPCONTROLFI_C_U16_DIPOPLAYBACKSTATUS,
         AMT_C_U8_CCAMSG_OPCODE_STATUS, vOnDiPOPlayBackStatus)

ON_MESSAGE_SVCDATA( MPLAY_APPCONTROLFI_C_U16_DIPOPLAYBACKSHUFFLEMODE,
         AMT_C_U8_CCAMSG_OPCODE_STATUS, vOnDiPOPlayBackShuffleModeStatus)

ON_MESSAGE_SVCDATA( MPLAY_APPCONTROLFI_C_U16_DIPOPLAYBACKREPEATMODE,
         AMT_C_U8_CCAMSG_OPCODE_STATUS, vOnDiPOPlayBackRepeatModeStatus)

ON_MESSAGE_SVCDATA( MPLAY_APPCONTROLFI_C_U16_DIPOCALLSTATE,
         AMT_C_U8_CCAMSG_OPCODE_STATUS, vOnDiPOCallStateStatus)

ON_MESSAGE_SVCDATA( MPLAY_APPCONTROLFI_C_U16_DIPOCALLDURATION,
         AMT_C_U8_CCAMSG_OPCODE_STATUS, vOnDiPoCallDuration)

ON_MESSAGE_SVCDATA( MPLAY_APPCONTROLFI_C_U16_DIPOCOMMUNICATIONS,
         AMT_C_U8_CCAMSG_OPCODE_STATUS, vOnDiPOCommunicationsStatus)

ON_MESSAGE_SVCDATA( MPLAY_APPCONTROLFI_C_U16_DIPOACTIVEDEVICE,
         AMT_C_U8_CCAMSG_OPCODE_METHODRESULT, vOnDiPOActiveDevice)

ON_MESSAGE_SVCDATA( MPLAY_APPCONTROLFI_C_U16_DIPOROLESWITCHREQUEST,
         AMT_C_U8_CCAMSG_OPCODE_METHODRESULT, vOnDiPORoleSwitchResult)

ON_MESSAGE_SVCDATA( MPLAY_APPCONTROLFI_C_U16_DIPOGETMEDIAOBJECTALBUMARTINFO,
         AMT_C_U8_CCAMSG_OPCODE_METHODRESULT, vOnDiPOGetMediaObjectAlbumArtInfo)

ON_MESSAGE_SVCDATA( MPLAY_APPCONTROLFI_C_U16_DIPOGETMEDIAOBJECTALBUMART,
         AMT_C_U8_CCAMSG_OPCODE_METHODRESULT, vOnDiPOGetMediaObjectAlbumArt)

ON_MESSAGE_SVCDATA( MPLAY_APPCONTROLFI_C_U16_DIPOTRANSFERGPSDATA,
         AMT_C_U8_CCAMSG_OPCODE_METHODRESULT, vOnDiPOTransferGPSData)

ON_MESSAGE_SVCDATA( MPLAY_APPCONTROLFI_C_U16_DIPOTRANSFERDRDATA,
         AMT_C_U8_CCAMSG_OPCODE_METHODRESULT, vOnDiPOTransferDRData)

ON_MESSAGE_SVCDATA( MPLAY_APPCONTROLFI_C_U16_DIPOLOCATIONINFO,
         AMT_C_U8_CCAMSG_OPCODE_STATUS, vOnStatusDiPOLocationInfo)

ON_MESSAGE_SVCDATA( MPLAY_APPCONTROLFI_C_U16_DIPOREQUESTAUDIODEVICE,
         AMT_C_U8_CCAMSG_OPCODE_METHODRESULT, vOnDiPOReqAudioDevice)

ON_MESSAGE_SVCDATA( MPLAY_APPCONTROLFI_C_U16_DIPORELEASEAUDIODEVICE,
         AMT_C_U8_CCAMSG_OPCODE_METHODRESULT, vOnDiPORelAudioDevice)

ON_MESSAGE_SVCDATA( MPLAY_APPCONTROLFI_C_U16_NATIVETRANSPORTSTART,
         AMT_C_U8_CCAMSG_OPCODE_STATUS, vOnNativeTransportStartStatus)

ON_MESSAGE_SVCDATA( MPLAY_APPCONTROLFI_C_U16_NATIVETRANSPORTSTOP,
         AMT_C_U8_CCAMSG_OPCODE_STATUS, vOnNativeTransportStopStatus)

ON_MESSAGE_SVCDATA( MPLAY_APPCONTROLFI_C_U16_CONNECTEDBLUETOOTHPROFILES,
         AMT_C_U8_CCAMSG_OPCODE_STATUS, vOnConnectedBTProfilesUpdate)

ON_MESSAGE_SVCDATA( MPLAY_APPCONTROLFI_C_U16_CONNECT,
         AMT_C_U8_CCAMSG_OPCODE_METHODRESULT, vOnConnectResult)

ON_MESSAGE_SVCDATA(MPLAY_APPCONTROLFI_C_U16_DIPOSTARTIAP2OVERCARPLAYWIFI,
         AMT_C_U8_CCAMSG_OPCODE_METHODRESULT, vOnStartiAP2OverCarPlayResult)

ON_MESSAGE_SVCDATA( MPLAY_APPCONTROLFI_C_U16_DIPOPOWER,
         AMT_C_U8_CCAMSG_OPCODE_STATUS, vOnDiPOPowerStatus)

ON_MESSAGE_SVCDATA(MPLAY_APPCONTROLFI_C_U16_DIPOROUTEGUIDANCE,
         AMT_C_U8_CCAMSG_OPCODE_STATUS, vOnDipoRouteGuidanceStatus)

ON_MESSAGE_SVCDATA(MPLAY_APPCONTROLFI_C_U16_DIPOROUTEGUIDANCEMANEUVER,
         AMT_C_U8_CCAMSG_OPCODE_STATUS, vOnDipoRouteGuidanceManeuverStatus)

ON_MESSAGE_SVCDATA(MPLAY_APPCONTROLFI_C_U16_DIPOCOMMUNICATIONERROR,
         AMT_C_U8_CCAMSG_OPCODE_STATUS, vOnDipoCommErrorStatus)

ON_MESSAGE_SVCDATA(MPLAY_APPCONTROLFI_C_U16_DIPOGPRMCDATASTATUSVALUES,
         AMT_C_U8_CCAMSG_OPCODE_STATUS, vOnDipoGPRMCDataStatusValuesStatus)

ON_MESSAGE_SVCDATA(MPLAY_APPCONTROLFI_C_U16_DIPOROUTEGUIDANCE2,
         AMT_C_U8_CCAMSG_OPCODE_STATUS, vOnDipoRouteGuidanceStatus2)

ON_MESSAGE_SVCDATA(MPLAY_APPCONTROLFI_C_U16_DIPOROUTEGUIDANCEMANEUVER2,
         AMT_C_U8_CCAMSG_OPCODE_STATUS, vOnDipoRouteGuidanceManeuverStatus2)

ON_MESSAGE_SVCDATA(MPLAY_APPCONTROLFI_C_U16_DIPOSETCALLCONTROLSCONFIGURATION,
         AMT_C_U8_CCAMSG_OPCODE_STATUS, bSetCallControlConfig)

ON_MESSAGE_SVCDATA(MPLAY_APPCONTROLFI_C_U16_DIPODEVICETIMEUPDATE,
         AMT_C_U8_CCAMSG_OPCODE_STATUS, vOnDipoDeviceTimeUpdate)

END_MSG_MAP()

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::spi_tclMPlayClientHandler()
 ***************************************************************************/
spi_tclMPlayClientHandler::spi_tclMPlayClientHandler(ahl_tclBaseOneThreadApp* poMainApp,
         spi_tclMplayResp* m_poMplayResp) :
                  ahl_tclBaseOneThreadClientHandler(poMainApp, /* Application Pointer */
                  CCA_C_U16_SRV_IPOD_APP_CONTROL, /* ID of used Service */
                  MPLAY_APPCONTROLFI_C_U16_SERVICE_MAJORVERSION, /* MajorVersion of used Service */
                  MPLAY_APPCONTROLFI_C_U16_SERVICE_MINORVERSION, /* MinorVersion of used Service */
                  CCA_C_U16_APP_MEDIAPLAYER),
                  m_poMainApp(poMainApp),
                  m_poMplayResp(m_poMplayResp),
                  m_u32DeviceTag(0),
                  m_enDevConnReq(e8DEVCONNREQ_SELECT),
                  m_fvBTProfileCallback(NULL),
                  m_bIsMetaDataPropertyRegistered(false),
                  m_poMplayDevList(NULL),
                  m_fvDeviceListUpdate(NULL)
{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::spi_tclMPlayClientHandler(..)
 ***************************************************************************/
spi_tclMPlayClientHandler::spi_tclMPlayClientHandler(ahl_tclBaseOneThreadApp* poMainApp) :
                  ahl_tclBaseOneThreadClientHandler(poMainApp, /* Application Pointer */
                  CCA_C_U16_SRV_IPOD_APP_CONTROL, /* ID of used Service */
                  MPLAY_APPCONTROLFI_C_U16_SERVICE_MAJORVERSION, /* MajorVersion of used Service */
                  MPLAY_APPCONTROLFI_C_U16_SERVICE_MINORVERSION, /* MinorVersion of used Service */
                  CCA_C_U16_APP_MEDIAPLAYER),
                  m_poMainApp(poMainApp),
                  m_poMplayResp(NULL),
                  m_u32DeviceTag(0),
                  m_enDevConnReq(e8DEVCONNREQ_SELECT),
                  m_fvBTProfileCallback(NULL),
                  m_poMplayDevList(NULL),
                  m_bIsMPServiceAvailable(false)

{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));
   //@Note: This constructor is not used. Only created since invoking getInstance()
   //method without parameters causes compilation error since ahl_tclBaseOneThreadClientHandler()
   //default ctor is not implemented.
   //To avoid this, getInstance(ahl_tclBaseOneThreadApp*) is called.
   
   for (t_U32 u32KeyIndex = 0; u32KeyIndex < ((sizeof(aIapKeyCode)) / sizeof(trIAPKeyCode)); ++u32KeyIndex)
   {
      m_KeyActionMap.insert(std::pair<tenKeyCode,mplay_fi_tcl_e8_DiPOPlaypackAction::tenType >(
            aIapKeyCode[u32KeyIndex].enKeyCode, aIapKeyCode[u32KeyIndex].enActionType));
   }
   m_poMplayDevList = new spi_tclMplayDeviceListHandler();
 
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::~spi_tclMPlayClientHandler()
 ***************************************************************************/
spi_tclMPlayClientHandler::~spi_tclMPlayClientHandler()
{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));

   m_poMplayResp               = NULL;
   m_poMainApp                = NULL;
   m_u32DeviceTag             = 0;
   if(NULL != m_poMplayDevList)
   {
      m_poMplayDevList->vSetRoleSwitchTriggerStatus(false);
   }
   RELEASE_MEM(m_poMplayDevList);
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::vOnServiceAvailable()
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vOnServiceAvailable()
{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));
   m_bIsMPServiceAvailable = true;
   tBool bResult = false;

   // Send the pending RouteGuidanceDisplayComponentRegistrations and clear the queue once sent.
   // Mediaplayer will store the registration sent and use the same persistently using datapool
   m_queueRouteGuidanceComponentRegLock.s16Lock();
   if (false == m_queueRouteGuidanceComponentReg.empty())
   {
      ETG_TRACE_USR1(("[DESC]: spi_tclMPlayClientHandler::vOnServiceAvailable - Sending pending route guidance registrations to mediaplayer"));
      // populate message context
      trMsgContext rMsgContext;
      vPopulateMessageContext(rMsgContext);
      FIMsgDispatch msgDispatcher(m_poMainApp);
      spi_tMsgDiPOSetRouteGuidanceDispComponentInfoMS oMPlayDiPOSetRouteGuidanceDispComponentInfo;
      ETG_TRACE_USR1(("[DESC]: spi_tclMPlayClientHandler::vOnServiceAvailable - Number of pending route guidance registration requests = %d", m_queueRouteGuidanceComponentReg.size()));
      for (tU16 u16Index = 0; u16Index < m_queueRouteGuidanceComponentReg.size(); ++u16Index)
      {
         ETG_TRACE_USR1(("[DESC]: spi_tclMPlayClientHandler::vOnServiceAvailable - Request No. %d", (u16Index + 1)));
         oMPlayDiPOSetRouteGuidanceDispComponentInfo = m_queueRouteGuidanceComponentReg.front();
         for (tU16 u16Index2 = 0; u16Index2 < oMPlayDiPOSetRouteGuidanceDispComponentInfo.RouteGuidanceDisplayComponents.oItems.size(); ++u16Index2)
         {
            ETG_TRACE_USR1(("[DESC]: spi_tclMPlayClientHandler::vOnServiceAvailable - Display component ID = %d", ETG_ENUM(DIPO_COMPONENT_DISPLAYID, oMPlayDiPOSetRouteGuidanceDispComponentInfo.RouteGuidanceDisplayComponents.oItems[u16Index2].Identifier)));
            ETG_TRACE_USR1(("[DESC]: spi_tclMPlayClientHandler::vOnServiceAvailable - Display name = %s",
                              oMPlayDiPOSetRouteGuidanceDispComponentInfo.RouteGuidanceDisplayComponents.oItems[u16Index2].Name.szValue));
            ETG_TRACE_USR1(("[DESC]: spi_tclMPlayClientHandler::vOnServiceAvailable - Max current road name length = %d",
                              oMPlayDiPOSetRouteGuidanceDispComponentInfo.RouteGuidanceDisplayComponents.oItems[u16Index2].MaxCurrentRoadNameLength));
            ETG_TRACE_USR1(("[DESC]: spi_tclMPlayClientHandler::vOnServiceAvailable - Max Destination name length = %d",
                              oMPlayDiPOSetRouteGuidanceDispComponentInfo.RouteGuidanceDisplayComponents.oItems[u16Index2].MaxDestinationNameLength));
            ETG_TRACE_USR1(("[DESC]: spi_tclMPlayClientHandler::vOnServiceAvailable - Max After Maneuver road name length = %d",
                              oMPlayDiPOSetRouteGuidanceDispComponentInfo.RouteGuidanceDisplayComponents.oItems[u16Index2].MaxAfterManeuverRoadNameLength));
            ETG_TRACE_USR1(("[DESC]: spi_tclMPlayClientHandler::vOnServiceAvailable - Max Maneuver description length = %d",
                              oMPlayDiPOSetRouteGuidanceDispComponentInfo.RouteGuidanceDisplayComponents.oItems[u16Index2].MaxManeuverDescriptionLength));
            ETG_TRACE_USR1(("[DESC]: spi_tclMPlayClientHandler::vOnServiceAvailable - Max guidance maneuver capacity = %d",
                              oMPlayDiPOSetRouteGuidanceDispComponentInfo.RouteGuidanceDisplayComponents.oItems[u16Index2].MaxGuidanceManeuverCapacity));
         }

         msgDispatcher.bSendMessage(oMPlayDiPOSetRouteGuidanceDispComponentInfo, rMsgContext, DIPO_MPLY_MAJORVERSION);
         m_queueRouteGuidanceComponentReg.pop();
      }
   }
   m_queueRouteGuidanceComponentRegLock.vUnlock();

   m_qCallControlsConfigLock.s16Lock();
   if(false == m_qCallControlsConfig.empty())
   {
      ETG_TRACE_USR1(("[DESC]: spi_tclMPlayClientHandler::vOnServiceAvailable - Sending pending call Controls Configuration"));
      // populate message context
      trMsgContext rMsgContext;
      vPopulateMessageContext(rMsgContext);
      FIMsgDispatch msgDispatcher(m_poMainApp);
      spi_tMsgSetCallControlsConfigMS oMplayCallControlsConfig;
      for(tU16 u16Index = 0; u16Index < m_qCallControlsConfig.size(); ++u16Index)
      {
         oMplayCallControlsConfig = m_qCallControlsConfig.front();
         bResult = msgDispatcher.bSendMessage(oMplayCallControlsConfig, rMsgContext, DIPO_MPLY_MAJORVERSION);
         m_qCallControlsConfig.pop();
         ETG_TRACE_USR4(("[PARAM]: spi_tclMPlayClientHandler::vOnServiceAvailable CallControls-bResult = %d ",ETG_ENUM(BOOL,bResult)));
      }
   }
   m_qCallControlsConfigLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::vOnServiceUnavailable()
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vOnServiceUnavailable()
{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));
   m_bIsMPServiceAvailable = false;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::vRegisterForDiscovererProperties()
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vRegisterForDiscovererProperties()
{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));
   vAddAutoRegisterForProperty( MPLAY_APPCONTROLFI_C_U16_DIPODEVICECONNECTIONS);
   vAddAutoRegisterForProperty( MPLAY_APPCONTROLFI_C_U16_DIPOCOMMUNICATIONERROR );
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::vRegisterForNavigationProperties()
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vRegisterForNavigationProperties()
{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));

   vAddAutoRegisterForProperty(MPLAY_APPCONTROLFI_C_U16_DIPOLOCATIONINFO);
   vAddAutoRegisterForProperty(MPLAY_APPCONTROLFI_C_U16_DIPOGPRMCDATASTATUSVALUES);
#ifdef VARIANT_S_FTR_ENABLE_TBT_OLD
   vAddAutoRegisterForProperty(MPLAY_APPCONTROLFI_C_U16_DIPOROUTEGUIDANCE);
   vAddAutoRegisterForProperty(MPLAY_APPCONTROLFI_C_U16_DIPOROUTEGUIDANCEMANEUVER);
#else
   vAddAutoRegisterForProperty (MPLAY_APPCONTROLFI_C_U16_DIPOROUTEGUIDANCE2);
   vAddAutoRegisterForProperty (MPLAY_APPCONTROLFI_C_U16_DIPOROUTEGUIDANCEMANEUVER2);
#endif
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::vRegisterForNativeTransportProperties()
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vRegisterForNativeTransportProperties()
{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));

   vAddAutoRegisterForProperty( MPLAY_APPCONTROLFI_C_U16_NATIVETRANSPORTSTART);
   vAddAutoRegisterForProperty( MPLAY_APPCONTROLFI_C_U16_NATIVETRANSPORTSTOP);
   vAddAutoRegisterForProperty( MPLAY_APPCONTROLFI_C_U16_CONNECTEDBLUETOOTHPROFILES);
   vAddAutoRegisterForProperty( MPLAY_APPCONTROLFI_C_U16_CONNECT);
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::vRegisterForMetadataProperties()
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vRegisterForMetadataProperties()
{
   ETG_TRACE_USR1(("spi_tclMPlayClientHandler::vRegisterForMetadataProperties entered "));
   vAddAutoRegisterForProperty( MPLAY_APPCONTROLFI_C_U16_DIPOCALLSTATE);
   vAddAutoRegisterForProperty( MPLAY_APPCONTROLFI_C_U16_DIPOCALLDURATION);
   vAddAutoRegisterForProperty (MPLAY_APPCONTROLFI_C_U16_DIPODEVICETIMEUPDATE);

   vAddAutoRegisterForProperty( MPLAY_APPCONTROLFI_C_U16_DIPOPLAYTIME);
   vAddAutoRegisterForProperty( MPLAY_APPCONTROLFI_C_U16_DIPONOWPLAYING);
   vAddAutoRegisterForProperty( MPLAY_APPCONTROLFI_C_U16_DIPOPLAYBACKSTATUS);
   vAddAutoRegisterForProperty( MPLAY_APPCONTROLFI_C_U16_DIPOPLAYBACKSHUFFLEMODE);
   vAddAutoRegisterForProperty( MPLAY_APPCONTROLFI_C_U16_DIPOPLAYBACKREPEATMODE);
   vAddAutoRegisterForProperty( MPLAY_APPCONTROLFI_C_U16_DIPOCOMMUNICATIONS);
   vAddAutoRegisterForProperty( MPLAY_APPCONTROLFI_C_U16_DIPOPOWER);

   m_bIsMetaDataPropertyRegistered = true;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::vUnregisterForProperties()
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vUnregisterForProperties()
{
   ETG_TRACE_USR1(("spi_tclMPlayClientHandler::vUnregisterForProperties entered "));

   //@TODO: Currently this function is unused. If required, this needs to be split into three
   //functions and MUST be called from ExtIntf only!!!
   vRemoveAutoRegisterForProperty(MPLAY_APPCONTROLFI_C_U16_DIPODEVICECONNECTIONS);
   vRemoveAutoRegisterForProperty(MPLAY_APPCONTROLFI_C_U16_DIPOLOCATIONINFO);
   vRemoveAutoRegisterForProperty(MPLAY_APPCONTROLFI_C_U16_DIPOGPRMCDATASTATUSVALUES);
   vRemoveAutoRegisterForProperty(MPLAY_APPCONTROLFI_C_U16_NATIVETRANSPORTSTART);
   vRemoveAutoRegisterForProperty(MPLAY_APPCONTROLFI_C_U16_NATIVETRANSPORTSTOP);
   vRemoveAutoRegisterForProperty(MPLAY_APPCONTROLFI_C_U16_CONNECTEDBLUETOOTHPROFILES);
   vRemoveAutoRegisterForProperty(MPLAY_APPCONTROLFI_C_U16_CONNECT);
   vRemoveAutoRegisterForProperty( MPLAY_APPCONTROLFI_C_U16_DIPOCOMMUNICATIONERROR);

#ifdef VARIANT_S_FTR_ENABLE_TBT_OLD
   vRemoveAutoRegisterForProperty(MPLAY_APPCONTROLFI_C_U16_DIPOROUTEGUIDANCE);
   vRemoveAutoRegisterForProperty(MPLAY_APPCONTROLFI_C_U16_DIPOROUTEGUIDANCEMANEUVER);
#else
   vRemoveAutoRegisterForProperty (MPLAY_APPCONTROLFI_C_U16_DIPOROUTEGUIDANCE2);
   vRemoveAutoRegisterForProperty (MPLAY_APPCONTROLFI_C_U16_DIPOROUTEGUIDANCEMANEUVER2);
#endif

}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::vUnregisterForMetadataProperties()
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vUnregisterForMetadataProperties()
{
   ETG_TRACE_USR1(("spi_tclMPlayClientHandler::vUnregisterForMetadataProperties entered "));
   vRemoveAutoRegisterForProperty( MPLAY_APPCONTROLFI_C_U16_DIPOCALLSTATE);
   vRemoveAutoRegisterForProperty( MPLAY_APPCONTROLFI_C_U16_DIPOCALLDURATION);
   vRemoveAutoRegisterForProperty (MPLAY_APPCONTROLFI_C_U16_DIPODEVICETIMEUPDATE);
   vRemoveAutoRegisterForProperty( MPLAY_APPCONTROLFI_C_U16_DIPOPLAYTIME);
   vRemoveAutoRegisterForProperty( MPLAY_APPCONTROLFI_C_U16_DIPONOWPLAYING);
   vRemoveAutoRegisterForProperty( MPLAY_APPCONTROLFI_C_U16_DIPOPLAYBACKSTATUS);
   vRemoveAutoRegisterForProperty( MPLAY_APPCONTROLFI_C_U16_DIPOPLAYBACKSHUFFLEMODE);
   vRemoveAutoRegisterForProperty( MPLAY_APPCONTROLFI_C_U16_DIPOPLAYBACKREPEATMODE);
   vRemoveAutoRegisterForProperty( MPLAY_APPCONTROLFI_C_U16_DIPOCOMMUNICATIONS);
   vRemoveAutoRegisterForProperty( MPLAY_APPCONTROLFI_C_U16_DIPOPOWER);

   m_bIsMetaDataPropertyRegistered = false;
   ETG_TRACE_USR1((" Registration Status: %d",m_bIsMetaDataPropertyRegistered));
   trAppMediaMetaData rAppMediaMetaData ;
   m_rAppMediaMetaDataLock.s16Lock();
   m_rAppMediaMetaData = rAppMediaMetaData;
   m_rAppMediaMetaDataLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::vOnDiPODeviceConnectionsStatus(..
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vOnDiPODeviceConnectionsStatus(
         amt_tclServiceData* poMessage/*mplay_appcontrolfi_tclMsgDiPODeviceConnectionsStatus &oDevConnectionStatus*/)
{

	/*lint -esym(40,nullptr)nullptr Undeclared identifier */
   ETG_TRACE_USR1(("[FUNC]: spi_tclMPlayClientHandler::vOnDiPODeviceConnectionsStatus entered"));
   spi_tDevConnectionStatus oDevConnectionStatus(*poMessage,
            DIPO_MPLY_MAJORVERSION);

   if ((oDevConnectionStatus.bIsValid()) && (NULL != m_poMplayDevList))
   {
      ETG_TRACE_USR2(("Number of Devices : %d", oDevConnectionStatus.oDeviceInfo.oItems.size()));
      m_poMplayDevList->vOnMplayDeviceListUpdate(oDevConnectionStatus.oDeviceInfo);
   }
   else //if (oDevConnectionStatus.bIsValid())
   {
      ETG_TRACE_ERR(("Message extraction failed."));
   } //if (oDevConnectionStatus.bIsValid()) 

   oDevConnectionStatus.vDestroy(); 
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::vOnDiPOPlaytimeStatus(amt_tclServiceData)
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vOnDiPOPlaytimeStatus(
         amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR1((" spi_tclMPlayClientHandler::vOnDiPOPlaytimeStatus entered "));

   spi_tMsgPlayTimeStatus oMsgPlayTimeStatus(*poMessage, DIPO_MPLY_MAJORVERSION);

   if(oMsgPlayTimeStatus.bIsValid())
   {
	  //Store the media Play time  info and send it to service.
     m_rAppMediaPlaytime.u32ElapsedPlayTime = oMsgPlayTimeStatus.u32ElapsedPlaytime;
     m_rAppMediaPlaytime.u32TotalPlayTime = oMsgPlayTimeStatus.u32TotalPlaytime;

      ETG_TRACE_USR4(("[PARAM]: spi_tclMPlayClientHandler::vOnDiPOPlaytimeStatus "
   		   "ElapsedTime = %d, Total Play Time = %d ",
   		   m_rAppMediaPlaytime.u32ElapsedPlayTime, m_rAppMediaPlaytime.u32TotalPlayTime));

      if (NULL != m_rMetaDataCbs.fvAppMediaPlaytime)
      {
         m_rMetaDataCbs.fvAppMediaPlaytime(m_rAppMediaPlaytime, corEmptyUsrContext);
      }//if (NULL != m_poMplayResp)
   }
   else
   {
      ETG_TRACE_ERR(("[ERR]:spi_tclMPlayClientHandler::vOnDiPOPlaytimeStatus : Message extraction failed. "));
   }

   oMsgPlayTimeStatus.vDestroy();
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclMPlayClientHandler::vOnDiPONowPlayingStatus()
***************************************************************************/
t_Void spi_tclMPlayClientHandler::vOnDiPONowPlayingStatus(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR1((" spi_tclMPlayClientHandler::vOnDiPONowPlayingStatus entered "));
   ETG_TRACE_USR1((" Registration Status: %d",m_bIsMetaDataPropertyRegistered));

   spi_tMsgNowPlayingStatus oMsgNowPlayingStatus(*poMessage, DIPO_MPLY_MAJORVERSION);

   if((oMsgNowPlayingStatus.bIsValid()) && (true == m_bIsMetaDataPropertyRegistered))
   {
      m_rAppMediaMetaDataLock.s16Lock();
      //Store the media meta data info and send it to service.
      m_rAppMediaMetaData.bMediaMetadataValid = true;

      GET_STRINGDATA_FROM_FI_STRINGOBJ(oMsgNowPlayingStatus.sAppName,
            MPLAYFI_CHAR_SET_UTF8, m_rAppMediaMetaData.szAppName);
      GET_STRINGDATA_FROM_FI_STRINGOBJ(oMsgNowPlayingStatus.sTitle,
            MPLAYFI_CHAR_SET_UTF8, m_rAppMediaMetaData.szTitle);
      GET_STRINGDATA_FROM_FI_STRINGOBJ(oMsgNowPlayingStatus.sArtist,
            MPLAYFI_CHAR_SET_UTF8, m_rAppMediaMetaData.szArtist);
      GET_STRINGDATA_FROM_FI_STRINGOBJ(oMsgNowPlayingStatus.sAlbum,
            MPLAYFI_CHAR_SET_UTF8, m_rAppMediaMetaData.szAlbum);
      GET_STRINGDATA_FROM_FI_STRINGOBJ(oMsgNowPlayingStatus.sGenre,
            MPLAYFI_CHAR_SET_UTF8, m_rAppMediaMetaData.szGenre);
      GET_STRINGDATA_FROM_FI_STRINGOBJ(oMsgNowPlayingStatus.sComposer,
            MPLAYFI_CHAR_SET_UTF8, m_rAppMediaMetaData.szComposerName);
      GET_STRINGDATA_FROM_FI_STRINGOBJ(oMsgNowPlayingStatus.sAlbumArt,
            MPLAYFI_CHAR_SET_UTF8, m_rAppMediaMetaData.szImageUrl);

      // Track number from media player
      m_rAppMediaMetaData.u32TrackNumber = static_cast<t_U32>(oMsgNowPlayingStatus.u16AlbumTrackNumber);
      m_rAppMediaMetaData.u32AlbumTrackCount = static_cast<t_U32>(oMsgNowPlayingStatus.u16AlbumTrackCount);

      // Disc Number from media player
      m_rAppMediaMetaData.u32AlbumDiscNumber = static_cast<t_U32>(oMsgNowPlayingStatus.u16AlbumDiscNumber);
      m_rAppMediaMetaData.u32AlbumDiscCount = static_cast<t_U32>(oMsgNowPlayingStatus.u16AlbumDiscCount);

      // Chapter count from media player
      m_rAppMediaMetaData.u32ChapterCount = static_cast<t_U32>(oMsgNowPlayingStatus.u16ChapterCount);

      // Now playing info according iAP2 Spec
      m_rAppMediaMetaData.bITunesRadioAd = oMsgNowPlayingStatus.biTunesRadioAd;
      GET_STRINGDATA_FROM_FI_STRINGOBJ(oMsgNowPlayingStatus.siTunesRadioStationName,
            MPLAYFI_CHAR_SET_UTF8, m_rAppMediaMetaData.szITunesRadioStationName);

      ETG_TRACE_USR4(("[PARAM]:spi_tclMPlayClientHandler::vOnDiPONowPlayingStatus entered: bMediaMetadataValid = %u ",
            ETG_ENUM(BOOL, m_rAppMediaMetaData.bMediaMetadataValid)));
      ETG_TRACE_USR4(("[PARAM]: vOnDiPONowPlayingStatus: Artist = %s ",
    		  m_rAppMediaMetaData.szArtist.c_str()));

      ETG_TRACE_USR4(("[PARAM]: vOnDiPONowPlayingStatus: Title = %s ",
    		  m_rAppMediaMetaData.szTitle.c_str()));

      ETG_TRACE_USR4(("[PARAM]: vOnDiPONowPlayingStatus: Album = %s ",
    		  m_rAppMediaMetaData.szAlbum.c_str()));

      ETG_TRACE_USR4(("[PARAM]: vOnDiPONowPlayingStatus: Genre = %s ",
    		  m_rAppMediaMetaData.szGenre.c_str()));

      ETG_TRACE_USR4(("[PARAM]: vOnDiPONowPlayingStatus: AppName = %s ",
    		  m_rAppMediaMetaData.szAppName.c_str()));

      ETG_TRACE_USR4(("[PARAM]: vOnDiPONowPlayingStatus: ImageUrl = %s ",
    		  m_rAppMediaMetaData.szImageUrl.c_str()));

      ETG_TRACE_USR4(("[PARAM]: vOnDiPONowPlayingStatus: Composer = %s ",
    		  m_rAppMediaMetaData.szComposerName.c_str()));

      ETG_TRACE_USR4(("[PARAM]: vOnDiPONowPlayingStatus: Track Number = %d, Album Track Count = %d ",
            m_rAppMediaMetaData.u32TrackNumber, m_rAppMediaMetaData.u32AlbumTrackCount));

      ETG_TRACE_USR4(("[PARAM]: vOnDiPONowPlayingStatus: Album Disc Number = %d, Album Disc Count = %d ",
            m_rAppMediaMetaData.u32AlbumDiscNumber, m_rAppMediaMetaData.u32AlbumDiscCount));

      ETG_TRACE_USR4(("[PARAM]: vOnDiPONowPlayingStatus: Chapter Count = %d ",
            m_rAppMediaMetaData.u32ChapterCount));

      ETG_TRACE_USR4(("[PARAM]: vOnDiPONowPlayingStatus: ITune Radio Ad status = %d ",
            ETG_ENUM(BOOL,m_rAppMediaMetaData.bITunesRadioAd)));

      ETG_TRACE_USR4(("[PARAM]: vOnDiPONowPlayingStatus: ITune Radio Name = %s ",
            m_rAppMediaMetaData.szITunesRadioStationName.c_str()));

      // Storing the AppMediaMetaData locally.
      trAppMediaMetaData rAppMediaMetaData = m_rAppMediaMetaData;

      m_rAppMediaMetaDataLock.vUnlock();

      if(!((rAppMediaMetaData.szImageUrl).empty()))
      {
         if(false == bDiPOGetAlbumArtInfo(rAppMediaMetaData.szImageUrl))
         {
            ETG_TRACE_ERR(("[ERR]: vOnDiPONowPlayingStatus: Album Art Info Request Failed"));
         }
      }

      // Posting meta data
      if (NULL !=  m_rMetaDataCbs.fvAppMediaMetaData)
      {
         m_rMetaDataCbs.fvAppMediaMetaData(rAppMediaMetaData, corEmptyUsrContext);
      }//if (NULL != m_poMplayResp)

   }
   else
   {
      ETG_TRACE_ERR(("[ERR]:spi_tclMPlayClientHandler::vOnDiPONowPlayingStatus : Message extraction failed. "));
   }

   oMsgNowPlayingStatus.vDestroy();
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclMPlayClientHandler::vOnDiPOPlayBackStatus()
***************************************************************************/
t_Void spi_tclMPlayClientHandler::vOnDiPOPlayBackStatus(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR1((" spi_tclMPlayClientHandler::vOnDiPOPlayBackStatus entered "));
   ETG_TRACE_USR1((" Registration Status: %d",m_bIsMetaDataPropertyRegistered));

   spi_tMsgPlayBackStatus oMsgPlayBackStatus(*poMessage, DIPO_MPLY_MAJORVERSION);

   if (oMsgPlayBackStatus.bIsValid()&& (true == m_bIsMetaDataPropertyRegistered))
   {
      m_rAppMediaMetaDataLock.s16Lock();
      //Retrieve the Play back state
      switch(oMsgPlayBackStatus.e8State.enType)
      {
      case mplay_fi_tcl_e8_DiPOPlaybackStatus::FI_EN_E8DIPO_PBS_STOPPED :
         m_rAppMediaMetaData.enMediaPlayBackState = e8PLAYBACK_STOPPED;
         break;
      case mplay_fi_tcl_e8_DiPOPlaybackStatus::FI_EN_E8DIPO_PBS_PLAYING :
         m_rAppMediaMetaData.enMediaPlayBackState = e8PLAYBACK_PLAYING;
         break;
      case mplay_fi_tcl_e8_DiPOPlaybackStatus::FI_EN_E8DIPO_PBS_PAUSED  :
         m_rAppMediaMetaData.enMediaPlayBackState = e8PLAYBACK_PAUSED;
         break;
      case mplay_fi_tcl_e8_DiPOPlaybackStatus::FI_EN_E8DIPO_PBS_SEEKFORWARD :
         m_rAppMediaMetaData.enMediaPlayBackState = e8PLAYBACK_SEEKFORWARD;
         break;
      case mplay_fi_tcl_e8_DiPOPlaybackStatus::FI_EN_E8DIPO_PBS_SEEKBACKWARD :
         m_rAppMediaMetaData.enMediaPlayBackState = e8PLAYBACK_SEEKBACKWARD;
         break;
      default :
         m_rAppMediaMetaData.enMediaPlayBackState = e8PLAYBACK_NOT_KNOWN;
         break;
      }

      ETG_TRACE_USR4(("[PARAM]: spi_tclMPlayClientHandler::vOnDiPOPlayBackStatus Play Back State = %d",
            ETG_ENUM(PLAYBACK_STATE,m_rAppMediaMetaData.enMediaPlayBackState)));

      // Storing the AppMediaMetaData locally.
      trAppMediaMetaData rAppMediaMetaData = m_rAppMediaMetaData;

      m_rAppMediaMetaDataLock.vUnlock();

      // Posting meta data
      if (NULL != m_rMetaDataCbs.fvAppMediaMetaData)
      {
         m_rMetaDataCbs.fvAppMediaMetaData(rAppMediaMetaData, corEmptyUsrContext);
      }//if (NULL != m_poMplayResp)

   }
   else
   {
      ETG_TRACE_ERR(("[ERR]:spi_tclMPlayClientHandler::vOnDiPOPlayBackStatus : Message extraction failed."));
   }
   oMsgPlayBackStatus.vDestroy();
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclMPlayClientHandler::vOnDiPOPlayBackShuffleModeStatus()
***************************************************************************/
t_Void spi_tclMPlayClientHandler::vOnDiPOPlayBackShuffleModeStatus(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR1((" spi_tclMPlayClientHandler::vOnDiPOPlayBackShuffleModeStatus entered "));
   ETG_TRACE_USR1((" Registration Status: %d",m_bIsMetaDataPropertyRegistered));

   spi_tMsgPlayBackShuffleModeStatus oMsgPlayBackShuffleStatus(*poMessage, DIPO_MPLY_MAJORVERSION);

   if (oMsgPlayBackShuffleStatus.bIsValid()&& (true == m_bIsMetaDataPropertyRegistered))
   {
      m_rAppMediaMetaDataLock.s16Lock();
      //Retrieve the Shuffle mode
      switch(oMsgPlayBackShuffleStatus.e8Mode.enType)
      {
      case mplay_fi_tcl_e8_DiPOPlaybackShuffleMode::FI_EN_E8DIPO_PBSM_OFF :
         m_rAppMediaMetaData.enMediaPlayBackShuffleState = e8SHUFFLE_OFF;
         break;
      case mplay_fi_tcl_e8_DiPOPlaybackShuffleMode::FI_EN_E8DIPO_PBSM_SONGS :
         m_rAppMediaMetaData.enMediaPlayBackShuffleState = e8SHUFFLE_SONGS;
         break;
      case mplay_fi_tcl_e8_DiPOPlaybackShuffleMode::FI_EN_E8DIPO_PBSM_ALBUMS :
         m_rAppMediaMetaData.enMediaPlayBackShuffleState = e8SHUFFLE_ALBUMS;
         break;
      default :
         m_rAppMediaMetaData.enMediaPlayBackShuffleState = e8SHUFFLE_NOT_KNOWN;
         break;
      }

      ETG_TRACE_USR4(("[PARAM]: spi_tclMPlayClientHandler::vOnDiPOPlayBackShuffleModeStatus Shuffle State = %d",
            ETG_ENUM(SHUFFLE_MODE, m_rAppMediaMetaData.enMediaPlayBackShuffleState)));

      // Storing the AppMediaMetaData locally.
      trAppMediaMetaData rAppMediaMetaData = m_rAppMediaMetaData;

      m_rAppMediaMetaDataLock.vUnlock();

      // Posting meta data
      if (NULL != m_rMetaDataCbs.fvAppMediaMetaData)
      {
         m_rMetaDataCbs.fvAppMediaMetaData(rAppMediaMetaData, corEmptyUsrContext);
      }//if (NULL != m_poMplayResp)
   }
   else
   {
      ETG_TRACE_ERR(("[ERR]:spi_tclMPlayClientHandler::vOnDiPOPlayBackShuffleModeStatus : Message extraction failed."));
   }
   oMsgPlayBackShuffleStatus.vDestroy();
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclMPlayClientHandler::vOnDiPOPlayBackRepeatModeStatus()
***************************************************************************/
t_Void spi_tclMPlayClientHandler::vOnDiPOPlayBackRepeatModeStatus(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR1((" spi_tclMPlayClientHandler::vOnDiPOPlayBackRepeatModeStatus entered "));
   ETG_TRACE_USR1((" Registration Status: %d",m_bIsMetaDataPropertyRegistered));

   spi_tMsgPlayBackRepeatModeStatus oMsgPlayBackRepeatStatus(*poMessage, DIPO_MPLY_MAJORVERSION);

   if (oMsgPlayBackRepeatStatus.bIsValid()&& (true == m_bIsMetaDataPropertyRegistered))
   {
      m_rAppMediaMetaDataLock.s16Lock();
      // Retrieve Repeat Mode
      switch(oMsgPlayBackRepeatStatus.e8Mode.enType)
      {
      case mplay_fi_tcl_e8_DiPOPlaybackRepeatMode::FI_EN_E8DIPO_PBRM_OFF :
         m_rAppMediaMetaData.enMediaPlayBackRepeatState = e8REPEAT_OFF;
         break;
      case mplay_fi_tcl_e8_DiPOPlaybackRepeatMode::FI_EN_E8DIPO_PBRM_ONE :
         m_rAppMediaMetaData.enMediaPlayBackRepeatState = e8REPEAT_ONE;
         break;
      case mplay_fi_tcl_e8_DiPOPlaybackRepeatMode::FI_EN_E8DIPO_PBRM_ALL :
         m_rAppMediaMetaData.enMediaPlayBackRepeatState = e8REPEAT_ALL;
         break;
      default :
         m_rAppMediaMetaData.enMediaPlayBackRepeatState = e8REPEAT_NOT_KNOWN;
         break;
      }

      ETG_TRACE_USR4(("[PARAM]: spi_tclMPlayClientHandler::vOnDiPOPlayBackRepeatModeStatus Repeat Mode = %d",
            ETG_ENUM(REPEAT_MODE, m_rAppMediaMetaData.enMediaPlayBackRepeatState)));

      // Storing the AppMediaMetaData locally.
      trAppMediaMetaData rAppMediaMetaData = m_rAppMediaMetaData;

      m_rAppMediaMetaDataLock.vUnlock();

      // Posting meta data
      if (NULL != m_rMetaDataCbs.fvAppMediaMetaData)
      {
         m_rMetaDataCbs.fvAppMediaMetaData(rAppMediaMetaData, corEmptyUsrContext);
      }//if (NULL != m_poMplayResp)
   }
   else
   {
      ETG_TRACE_ERR(("[ERR]:spi_tclMPlayClientHandler::vOnDiPOPlayBackRepeatModeStatus : Message extraction failed."));
   }
   oMsgPlayBackRepeatStatus.vDestroy();
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::enGetPhoneCallState()
 ***************************************************************************/
tenPhoneCallState spi_tclMPlayClientHandler::enGetPhoneCallState(
      mplay_fi_tcl_e8_DiPOCallStateUpdateStatus &rfrenPhoneCallState)
{
   ETG_TRACE_USR1(("spi_tclMPlayClientHandler::enGetPhoneCallState entered "));

   tenPhoneCallState enPhoneCallState = e8PHONE_CALL_STATE_NOT_KNOWN;
   switch(rfrenPhoneCallState.enType)
   {
   case mplay_fi_tcl_e8_DiPOCallStateUpdateStatus::FI_EN_E8DIPO_CSUS_DISCONNECTED :
      enPhoneCallState = e8CALL_DISCONNECTED;
      break;
   case mplay_fi_tcl_e8_DiPOCallStateUpdateStatus::FI_EN_E8DIPO_CSUS_SENDING :
      enPhoneCallState = e8CALL_SENDING;
      break;
   case mplay_fi_tcl_e8_DiPOCallStateUpdateStatus::FI_EN_E8DIPO_CSUS_RINGING :
      enPhoneCallState = e8CALL_RINGING;
      break;
   case mplay_fi_tcl_e8_DiPOCallStateUpdateStatus::FI_EN_E8DIPO_CSUS_CONNECTING :
      enPhoneCallState = e8CALL_CONNECTING;
      break;
   case mplay_fi_tcl_e8_DiPOCallStateUpdateStatus::FI_EN_E8DIPO_CSUS_ACTIVE :
      enPhoneCallState = e8CALL_ACTIVE;
      break;
   case mplay_fi_tcl_e8_DiPOCallStateUpdateStatus::FI_EN_E8DIPO_CSUS_HELD :
      enPhoneCallState = e8CALL_HELD;
      break;
   case mplay_fi_tcl_e8_DiPOCallStateUpdateStatus::FI_EN_E8DIPO_CSUS_DISCONNECTING :
      enPhoneCallState = e8CALL_DISCONNECTING;
      break;
   default :
      enPhoneCallState = e8PHONE_CALL_STATE_NOT_KNOWN;
   }
   return enPhoneCallState;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::enGetPhoneCallDirection()
 ***************************************************************************/
tenPhoneCallDirection spi_tclMPlayClientHandler::enGetPhoneCallDirection(
      mplay_fi_tcl_e8_DiPOCallStateUpdateDirection &rfenPhoneCallDirection)
{
   ETG_TRACE_USR1(("spi_tclMPlayClientHandler::enGetPhoneCallDirection entered "));

   tenPhoneCallDirection enPhoneCallDirection = e8PHONE_CALL_DIRECTION_NOT_KNOWN;
   switch(rfenPhoneCallDirection.enType)
   {
   case mplay_fi_tcl_e8_DiPOCallStateUpdateDirection::FI_EN_E8DIPO_CSUD_INCOMING :
      enPhoneCallDirection = e8CALL_INCOMING;
      break;
   case mplay_fi_tcl_e8_DiPOCallStateUpdateDirection::FI_EN_E8DIPO_CSUD_OUTGOING :
      enPhoneCallDirection = e8CALL_OUTGOING;
      break;
   default :
      enPhoneCallDirection = e8PHONE_CALL_DIRECTION_NOT_KNOWN;
   }

   return enPhoneCallDirection;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::enGetCallStateUpdateService()
 ***************************************************************************/
tenCallStateUpdateService spi_tclMPlayClientHandler::enGetCallStateUpdateService(
      mplay_fi_tcl_e8_DiPOCallStateUpdateService &rfenCallStateUpdateService)
{
   ETG_TRACE_USR1(("spi_tclMPlayClientHandler::enGetCallStateUpdateService entered "));

   tenCallStateUpdateService enCallStateUpdateService = e8CALLSTATE_SERVICE_UNKNOWN;
   switch(rfenCallStateUpdateService.enType)
   {
   case mplay_fi_tcl_e8_DiPOCallStateUpdateService::FI_EN_E8DIPO_CSEV_TELEPHONY :
      enCallStateUpdateService = e8CALLSTATE_SERVICE_TELEPHONY;
      break;
   case mplay_fi_tcl_e8_DiPOCallStateUpdateService::FI_EN_E8DIPO_CSEV_FACETIMEAUDIO :
      enCallStateUpdateService = e8CALLSTATE_SERVICE_FACETIMEAUDIO;
      break;
   case mplay_fi_tcl_e8_DiPOCallStateUpdateService::FI_EN_E8DIPO_CSEV_FACETIMEVIDEO :
      enCallStateUpdateService = e8CALLSTATE_SERVICE_FACETIMEVIDEO;
      break;
   default :
      enCallStateUpdateService = e8CALLSTATE_SERVICE_UNKNOWN;
   }
   return enCallStateUpdateService;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::enGetCallStateUpdateDisconnectReason()
 ***************************************************************************/
tenCallStateUpdateDisconnectReason spi_tclMPlayClientHandler::enGetCallStateUpdateDisconnectReason(
      mplay_fi_tcl_e8_DiPOCallStateUpdateDisconnectReason &rfenCallStateUpdateDisconnectReason)
{
   ETG_TRACE_USR1(("spi_tclMPlayClientHandler::enGetCallStateUpdateDisconnectReason entered "));

   tenCallStateUpdateDisconnectReason enCallStateUpdateDisconnectReason = e8CALLSTATE_DISCONNECT_REASON_UNKNOWN;
   switch(rfenCallStateUpdateDisconnectReason.enType)
   {
   case mplay_fi_tcl_e8_DiPOCallStateUpdateDisconnectReason::FI_EN_E8DIPO_CSDR_ENDED :
      enCallStateUpdateDisconnectReason = e8CALLSTATE_DISCONNECT_REASON_ENDED;
      break;
   case mplay_fi_tcl_e8_DiPOCallStateUpdateDisconnectReason::FI_EN_E8DIPO_CSDR_DECLINED :
      enCallStateUpdateDisconnectReason = e8CALLSTATE_DISCONNECT_REASON_DECLINED;
      break;
   case mplay_fi_tcl_e8_DiPOCallStateUpdateDisconnectReason::FI_EN_E8DIPO_CSDR_FAILED :
      enCallStateUpdateDisconnectReason = e8CALLSTATE_DISCONNECT_REASON_FAILED;
      break;
   default :
      enCallStateUpdateDisconnectReason = e8CALLSTATE_DISCONNECT_REASON_UNKNOWN;
   }
   return enCallStateUpdateDisconnectReason;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::vOnDiPOCallStateStatus(amt_tclServiceData)
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vOnDiPOCallStateStatus(
         amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR1(("spi_tclMPlayClientHandler::vOnDiPOCallStateStatus entered "));

   spi_tMsgCallStateStatus oMsgCallStateData(*poMessage, DIPO_MPLY_MAJORVERSION);

   if (oMsgCallStateData.bIsValid())
   {
      // clear the member variable vector list
      m_rAppPhoneDataLock.s16Lock();
      m_rAppPhoneData.tvecPhoneCallMetaDataList.clear();
      m_rAppPhoneData.bPhoneMetadataValid = true;

      for (tU16 u16Index = 0 ; u16Index < oMsgCallStateData.oCallStateUpdate.oItems.size() ; ++u16Index)
      {
         //Store the phone meta data info and send it to service.
         trPhoneCallMetaData rPhoneCallMetaDataList;

         GET_STRINGDATA_FROM_FI_STRINGOBJ(oMsgCallStateData.oCallStateUpdate.oItems[u16Index].sRemoteID,
               MPLAYFI_CHAR_SET_UTF8, rPhoneCallMetaDataList.szPhoneNumber);
         GET_STRINGDATA_FROM_FI_STRINGOBJ(oMsgCallStateData.oCallStateUpdate.oItems[u16Index].sDisplayName,
               MPLAYFI_CHAR_SET_UTF8, rPhoneCallMetaDataList.szDisplayName);

         rPhoneCallMetaDataList.enPhoneCallState = enGetPhoneCallState(oMsgCallStateData.oCallStateUpdate.oItems[u16Index].eStatus);
         rPhoneCallMetaDataList.enPhoneCallDirection = enGetPhoneCallDirection(oMsgCallStateData.oCallStateUpdate.oItems[u16Index].eDirection);

         GET_STRINGDATA_FROM_FI_STRINGOBJ(oMsgCallStateData.oCallStateUpdate.oItems[u16Index].sCallUUID,
               MPLAYFI_CHAR_SET_UTF8, rPhoneCallMetaDataList.szCallUUID);
         GET_STRINGDATA_FROM_FI_STRINGOBJ(oMsgCallStateData.oCallStateUpdate.oItems[u16Index].sLabel,
               MPLAYFI_CHAR_SET_UTF8, rPhoneCallMetaDataList.szCallerLabel);

         rPhoneCallMetaDataList.enCallStateUpdateService = enGetCallStateUpdateService(oMsgCallStateData.oCallStateUpdate.oItems[u16Index].eService);
         rPhoneCallMetaDataList.bConferencedCall = oMsgCallStateData.oCallStateUpdate.oItems[u16Index].bIsConferenced;
         rPhoneCallMetaDataList.u8ConferenceGroup = oMsgCallStateData.oCallStateUpdate.oItems[u16Index].u8ConferenceGroup;
         rPhoneCallMetaDataList.enCallStateUpdateDisconnectReason = enGetCallStateUpdateDisconnectReason(oMsgCallStateData.oCallStateUpdate.oItems[u16Index].eDisconnectReason);

         ETG_TRACE_USR4(("[PARAM]:vOnDiPOCallStateStatus : Phone number = %s ", rPhoneCallMetaDataList.szPhoneNumber.c_str()));
         ETG_TRACE_USR4(("[PARAM]:vOnDiPOCallStateStatus : Display name = %s ", rPhoneCallMetaDataList.szDisplayName.c_str()));

         ETG_TRACE_USR4(("[PARAM]:vOnDiPOCallStateStatus : Phone call state = %d, Phone call direction = %d ",
               ETG_ENUM(PHONE_CALL_STATE, rPhoneCallMetaDataList.enPhoneCallState),
               ETG_ENUM(PHONE_CALL_DIRECTION, rPhoneCallMetaDataList.enPhoneCallDirection)));

         ETG_TRACE_USR4(("[PARAM]:vOnDiPOCallStateStatus : Call UUID = %s ", rPhoneCallMetaDataList.szCallUUID.c_str()));
         ETG_TRACE_USR4(("[PARAM]:vOnDiPOCallStateStatus : Caller Label = %s ", rPhoneCallMetaDataList.szCallerLabel.c_str()));

         ETG_TRACE_USR4(("[PARAM]:vOnDiPOCallStateStatus : Call service of current device = %d, Is this call part of conference = %d ",
               ETG_ENUM(CALLSTATE_SERVICE, rPhoneCallMetaDataList.enCallStateUpdateService),
               ETG_ENUM(BOOL, rPhoneCallMetaDataList.bConferencedCall)));

         ETG_TRACE_USR4(("[PARAM]:vOnDiPOCallStateStatus : Conference group number = %d ",
               rPhoneCallMetaDataList.u8ConferenceGroup));

         ETG_TRACE_USR4(("[PARAM]:vOnDiPOCallStateStatus : Call disconnect reason = %d ",
               ETG_ENUM(CALLDISCONNECT_REASON, rPhoneCallMetaDataList.enCallStateUpdateDisconnectReason)));

         //push back data to the vector
         m_rAppPhoneData.tvecPhoneCallMetaDataList.push_back(rPhoneCallMetaDataList);
      }

      vPopulatePhoneCallData(oMsgCallStateData.oCallStateUpdate);

      // Storing the AppPhoneData locally.
      trAppPhoneData rAppPhoneData = m_rAppPhoneData;
      m_rAppPhoneDataLock.vUnlock();

      // post the data to service
      if (NULL != m_rMetaDataCbs.fvAppPhoneData)
      {
         m_rMetaDataCbs.fvAppPhoneData(rAppPhoneData, corEmptyUsrContext);
      }//if (NULL != m_poMplayResp)

      m_rAppPhoneCallDataLock.s16Lock();
      trAppPhoneCallData rAppPhoneCallData = m_rAppPhoneCallData;
      m_rAppPhoneCallDataLock.vUnlock();

      if (NULL != m_rMetaDataCbs.fvAppPhoneCallData)
      {
         m_rMetaDataCbs.fvAppPhoneCallData(rAppPhoneCallData, corEmptyUsrContext);
      }
   }
   else
   {
      ETG_TRACE_ERR(("[ERR]:spi_tclMPlayClientHandler::vOnDiPOCallStateStatus : Message extraction failed. "));
   }
   oMsgCallStateData.vDestroy();
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMPlayClientHandler::vPopulatePhoneCallData
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vPopulatePhoneCallData(mplay_fi_tcl_DiPOCallStateUpdate& rCallStateUpdate)
{
   ETG_TRACE_USR1(("spi_tclMPlayClientHandler::vPopulatePhoneCallData entered "));

   m_rAppPhoneCallDataLock.s16Lock();
   m_rAppPhoneCallData.tvecPhoneCallExtendedMetaDataList.clear();

   for (t_U16 u16Index = 0 ; u16Index < rCallStateUpdate.oItems.size() ; ++u16Index)
   {

      //Store the phone extended meta data info and send it to service.
      trPhoneCallExtendedMetaData rPhoneCallExtendedMetaDataList;

      GET_STRINGDATA_FROM_FI_STRINGOBJ(rCallStateUpdate.oItems[u16Index].sRemoteID,
            MPLAYFI_CHAR_SET_UTF8, rPhoneCallExtendedMetaDataList.szPhoneNumber);
      GET_STRINGDATA_FROM_FI_STRINGOBJ(rCallStateUpdate.oItems[u16Index].sDisplayName,
            MPLAYFI_CHAR_SET_UTF8, rPhoneCallExtendedMetaDataList.szDisplayName);

      rPhoneCallExtendedMetaDataList.enPhoneCallState = enGetPhoneCallState(rCallStateUpdate.oItems[u16Index].eStatus);
      rPhoneCallExtendedMetaDataList.enPhoneCallDirection = enGetPhoneCallDirection(rCallStateUpdate.oItems[u16Index].eDirection);

      GET_STRINGDATA_FROM_FI_STRINGOBJ(rCallStateUpdate.oItems[u16Index].sCallUUID,
            MPLAYFI_CHAR_SET_UTF8, rPhoneCallExtendedMetaDataList.szCallUUID);
      GET_STRINGDATA_FROM_FI_STRINGOBJ(rCallStateUpdate.oItems[u16Index].sLabel,
            MPLAYFI_CHAR_SET_UTF8, rPhoneCallExtendedMetaDataList.szCallerLabel);

      rPhoneCallExtendedMetaDataList.enCallStateUpdateService = enGetCallStateUpdateService(rCallStateUpdate.oItems[u16Index].eService);
      rPhoneCallExtendedMetaDataList.bConferencedCall = rCallStateUpdate.oItems[u16Index].bIsConferenced;
      rPhoneCallExtendedMetaDataList.u8ConferenceGroup = rCallStateUpdate.oItems[u16Index].u8ConferenceGroup;
      rPhoneCallExtendedMetaDataList.enCallStateUpdateDisconnectReason = enGetCallStateUpdateDisconnectReason(rCallStateUpdate.oItems[u16Index].eDisconnectReason);
      rPhoneCallExtendedMetaDataList.u64StartTimeStamp = rCallStateUpdate.oItems[u16Index].u64StartTimestamp;

      ETG_TRACE_USR4(("[PARAM]:vPopulatePhoneCallData : Phone number = %s ", rPhoneCallExtendedMetaDataList.szPhoneNumber.c_str()));
      ETG_TRACE_USR4(("[PARAM]:vPopulatePhoneCallData : Display name = %s ", rPhoneCallExtendedMetaDataList.szDisplayName.c_str()));

      ETG_TRACE_USR4(("[PARAM]:vPopulatePhoneCallData : Phone call state = %d, Phone call direction = %d ",
            ETG_ENUM(PHONE_CALL_STATE, rPhoneCallExtendedMetaDataList.enPhoneCallState),
            ETG_ENUM(PHONE_CALL_DIRECTION, rPhoneCallExtendedMetaDataList.enPhoneCallDirection)));

      ETG_TRACE_USR4(("[PARAM]:vPopulatePhoneCallData : Call UUID = %s ", rPhoneCallExtendedMetaDataList.szCallUUID.c_str()));
      ETG_TRACE_USR4(("[PARAM]:vOnDiPOCallStateStatus : Caller Label = %s ", rPhoneCallExtendedMetaDataList.szCallerLabel.c_str()));

      ETG_TRACE_USR4(("[PARAM]:vPopulatePhoneCallData : Call service of current device = %d, Is this call part of conference = %d ",
            ETG_ENUM(CALLSTATE_SERVICE, rPhoneCallExtendedMetaDataList.enCallStateUpdateService),
            ETG_ENUM(BOOL, rPhoneCallExtendedMetaDataList.bConferencedCall)));

      ETG_TRACE_USR4(("[PARAM]:vPopulatePhoneCallData : Conference group number = %d ",
            rPhoneCallExtendedMetaDataList.u8ConferenceGroup));

      ETG_TRACE_USR4(("[PARAM]:vPopulatePhoneCallData : Call disconnect reason = %d ",
            ETG_ENUM(CALLDISCONNECT_REASON, rPhoneCallExtendedMetaDataList.enCallStateUpdateDisconnectReason)));

      ETG_TRACE_USR4(("[PARAM]:vPopulatePhoneCallData : StartTimestamp = %s ", std::to_string(rPhoneCallExtendedMetaDataList.u64StartTimeStamp).c_str()));

      //push back data to the vector
      m_rAppPhoneCallData.tvecPhoneCallExtendedMetaDataList.push_back(rPhoneCallExtendedMetaDataList);

   }
   m_rAppPhoneCallDataLock.vUnlock();
}
/***************************************************************************
** FUNCTION:  spi_tclMPlayClientHandler::vOnDiPoCallDuration(amt_tclServiceData)
***************************************************************************/
t_Void spi_tclMPlayClientHandler::vOnDiPoCallDuration(amt_tclServiceData* poMessage)
{
ETG_TRACE_USR1(("spi_tclMPlayClientHandler::vOnDiPoCallDuration entered "));

spi_tMsgCallDurationStatus oMsgCallDurationStatusData(*poMessage, DIPO_MPLY_MAJORVERSION);

if (oMsgCallDurationStatusData.bIsValid())

{
   trPhoneCallMetaData rPhoneCallMetaDataList;
   m_rAppPhoneDataLock.s16Lock();
   if(!m_rAppPhoneData.tvecPhoneCallMetaDataList.empty())
   {
      rPhoneCallMetaDataList = m_rAppPhoneData.tvecPhoneCallMetaDataList.back();
   }
   rPhoneCallMetaDataList.u32CallDuration = oMsgCallDurationStatusData.u32CallDuration;
   m_rAppPhoneData.tvecPhoneCallMetaDataList.push_back(rPhoneCallMetaDataList);

   ETG_TRACE_USR4(("[PARAM]:vOnDiPoCallDuration : CallDuration = %d ",
                  rPhoneCallMetaDataList.u32CallDuration));

   // Storing the AppPhoneData locally.
   trAppPhoneData rAppPhoneData = m_rAppPhoneData;

   m_rAppPhoneDataLock.vUnlock();

   // post the data to service
   if (NULL != m_rMetaDataCbs.fvAppPhoneData)
   {
      m_rMetaDataCbs.fvAppPhoneData(rAppPhoneData, corEmptyUsrContext);
   }//if (NULL != m_poMplayResp)
}
else
{
   ETG_TRACE_ERR(("[ERR]:spi_tclMPlayClientHandler::vOnDiPoCallDuration : Message extraction failed. "));
}
oMsgCallDurationStatusData.vDestroy();
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::enGetRegistrationStatus()
 ***************************************************************************/
tenRegistrationStatus spi_tclMPlayClientHandler::enGetRegistrationStatus(
      mplay_fi_tcl_e8_DiPOCommunicationsUpdateRegistrationStatus &rfenRegistrationStatus)
{
   ETG_TRACE_USR1(("spi_tclMPlayClientHandler::enGetRegistrationStatus entered "));

   tenRegistrationStatus enRegistrationStatus = e8PHONE_REGISTRATION_NOT_KNOWN;
   switch(rfenRegistrationStatus.enType)
   {
   case mplay_fi_tcl_e8_DiPOCommunicationsUpdateRegistrationStatus::FI_EN_E8DIPO_CURS_UNKNONW :
      enRegistrationStatus = e8PHONE_REGISTRATION_NOT_KNOWN;
      break;
   case mplay_fi_tcl_e8_DiPOCommunicationsUpdateRegistrationStatus::FI_EN_E8DIPO_CURS_NOTREGISTERED :
      enRegistrationStatus = e8NOT_REGISTERED;
      break;
   case mplay_fi_tcl_e8_DiPOCommunicationsUpdateRegistrationStatus::FI_EN_E8DIPO_CURS_SEARCHING :
      enRegistrationStatus = e8SEARCHING;
      break;
   case mplay_fi_tcl_e8_DiPOCommunicationsUpdateRegistrationStatus::FI_EN_E8DIPO_CURS_DENIED :
      enRegistrationStatus = e8DENIED;
      break;
   case mplay_fi_tcl_e8_DiPOCommunicationsUpdateRegistrationStatus::FI_EN_E8DIPO_CURS_REGISTEREDHOME :
      enRegistrationStatus = e8REGISTERED_HOME;
      break;
   case mplay_fi_tcl_e8_DiPOCommunicationsUpdateRegistrationStatus::FI_EN_E8DIPO_CURS_REGISTEREDROAMING :
      enRegistrationStatus = e8REGISTERED_ROAMING;
      break;
   case mplay_fi_tcl_e8_DiPOCommunicationsUpdateRegistrationStatus::FI_EN_E8DIPO_CURS_EMERGENCYCALLONLY :
      enRegistrationStatus = e8EMERGENCY_CALLS_ONLY;
      break;
   default :
      enRegistrationStatus = e8PHONE_REGISTRATION_NOT_KNOWN;
   }
   return enRegistrationStatus;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::vOnDiPOCommunicationsStatus(amt_tclServiceData)
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vOnDiPOCommunicationsStatus(
         amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR1(("spi_tclMPlayClientHandler::vOnDiPOCommunicationsStatus entered "));

   spi_tMsgCommunicationsStatus oMsgCommunicationsData(*poMessage, DIPO_MPLY_MAJORVERSION);

   if (oMsgCommunicationsData.bIsValid())
   {
      //Store the phone meta data info and send it to service.
      m_rAppPhoneDataLock.s16Lock();
      m_rAppPhoneData.bPhoneMetadataValid = true;
      m_rAppPhoneData.enSignalStrength = static_cast<tenSignalStrength>(oMsgCommunicationsData.eSignalStrength.enType);
      m_rAppPhoneData.enRegistrationStatus = enGetRegistrationStatus(oMsgCommunicationsData.eRegistrationStatus);
      m_rAppPhoneData.enAirPlaneModeStatus = (true == oMsgCommunicationsData.bAirplaneModeStatus) ? (e8AIRPLANE_ON) : (e8AIRPLANE_OFF);

      GET_STRINGDATA_FROM_FI_STRINGOBJ(oMsgCommunicationsData.sCarrierName,
            MPLAYFI_CHAR_SET_UTF8, m_rAppPhoneData.szCarrierName);

      m_rAppPhoneData.enMute = (true == oMsgCommunicationsData.bMuteStatus) ? (e8PHONE_MUTE_ON):(e8PHONE_MUTE_OFF);
      m_rAppPhoneData.u8CurrentCallCount = oMsgCommunicationsData.u8CurrentCallCount;

      ETG_TRACE_USR4(("[PARAM]:vOnDiPOCommunicationsStatus : phone metadata validity = %d", ETG_ENUM(BOOL, m_rAppPhoneData.bPhoneMetadataValid)));

      ETG_TRACE_USR4(("[PARAM]:vOnDiPOCommunicationsStatus : Signal Strength = %d, Registration Status = %d ",
            ETG_ENUM(SIGNAL_STRENGTH, m_rAppPhoneData.enSignalStrength), ETG_ENUM(REGISTRATION_STATUS, m_rAppPhoneData.enRegistrationStatus)));

      ETG_TRACE_USR4(("[PARAM]:vOnDiPOCommunicationsStatus : Airplane Mode = %d",
            ETG_ENUM(AIRPLANE_MODE_STATUS, m_rAppPhoneData.enAirPlaneModeStatus)));

      ETG_TRACE_USR4(("[PARAM]:vOnDiPOCommunicationsStatus : Carrier Name on the phone = %s",
            m_rAppPhoneData.szCarrierName.c_str()));

      ETG_TRACE_USR4(("[PARAM]:vOnDiPOCommunicationsStatus : Mute Status = %d, Current Call Count = %d ",
            ETG_ENUM(PHONE_MUTE_STATUS, m_rAppPhoneData.enMute), m_rAppPhoneData.u8CurrentCallCount));


      // Storing the AppPhoneData locally.
      trAppPhoneData rAppPhoneData = m_rAppPhoneData;

      m_rAppPhoneDataLock.vUnlock();
      //post the data to service
      if (NULL != m_rMetaDataCbs.fvAppPhoneData)
      {
         m_rMetaDataCbs.fvAppPhoneData(rAppPhoneData, corEmptyUsrContext);
      }//if (NULL != m_poMplayResp)
   }
   else
   {
      ETG_TRACE_ERR(("[ERR]:spi_tclMPlayClientHandler::vOnDiPOCommunicationsStatus : Message extraction failed. "));
   }
   oMsgCommunicationsData.vDestroy();
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::vOnDiPOActiveDevice(amt_tclServiceData)
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vOnDiPOActiveDevice(
         amt_tclServiceData* poMessage) const
{

   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));
   spi_tMsgDipoActiveResult oDiPOActiveResult(*poMessage, DIPO_MPLY_MAJORVERSION);

   if (oDiPOActiveResult.bIsValid())
   {
      if (mplay_fi_tcl_e8_DiPOResponse::FI_EN_E8DIPO_OK != oDiPOActiveResult.e8DiPOResponse.enType)
      {
          ETG_TRACE_ERR(("DiPO Active operation failed."));         
      }
   }
   else //if (!oDiPOActiveResult.bIsValid())
   {
      ETG_TRACE_ERR(("Message extraction failed."));
   }

   oDiPOActiveResult.vDestroy();
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::vOnDiPORoleSwitchResult(amt_tclServiceData)
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vOnDiPORoleSwitchResult(
         amt_tclServiceData* poMessage/*mplay_appcontrolfi_tclMsgDiPORoleSwitchRequestMethodResult &oRoleSwitchResult*/)
{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));
   spi_tMsgDipoRoleSwitchResult oRoleSwitchResult(*poMessage,
            DIPO_MPLY_MAJORVERSION);
   trMsgContext oMsgContext;
   tenErrorCode enError = e8NO_ERRORS;
   tenResponseCode enResponseCode = e8FAILURE;
   CPY_TO_USRCNTXT(poMessage, oMsgContext.rUserContext);
   if (oRoleSwitchResult.bIsValid())
   {
      if(NULL != m_poMplayDevList)
      {
         m_poMplayDevList->vSetRoleSwitchTriggerStatus(false);
      }
      t_Bool bDeviceListUpdateRequired = true;
      if (mplay_fi_tcl_e8_DiPOResponse::FI_EN_E8DIPO_OK ==
         oRoleSwitchResult.e8DiPOResponse.enType)
      {
         bDeviceListUpdateRequired = false;
         enResponseCode = e8SUCCESS;
      }
      else if((mplay_fi_tcl_e8_DiPOResponse::FI_EN_E8DIPO_ERROR_UNKNOWN ==
         oRoleSwitchResult.e8DiPOResponse.enType) &&
         (e8DEVCONNREQ_DESELECT == m_enDevConnReq))
      {
         //This change is done to ensure clearing of device state "Renumeration in progress" when reverse role switch fails.
         enError = e8UNKNOWN_ERROR;
      }
      else if((mplay_fi_tcl_e8_DiPOResponse::FI_EN_E8DIPO_ERROR_UNKNOWN ==
         oRoleSwitchResult.e8DiPOResponse.enType) && (e8DEVCONNREQ_SELECT == m_enDevConnReq))
      {
         enError = e8UNKNOWN_ERROR;
      }
      else if(mplay_fi_tcl_e8_DiPOResponse::FI_EN_E8DIPO_ERROR_INVALIDHANLDE ==
         oRoleSwitchResult.e8DiPOResponse.enType)
      {
         enError = e8INVALID_DEV_HANDLE;
      }
      else
      {
         enError = e8UNSUPPORTED_OPERATION;
      } // end of if

      ETG_TRACE_USR1((" USB role switch result - [%d],[%d] ",enResponseCode, ETG_ENUM( DIPO_ROLE_SWITCH, oRoleSwitchResult.e8DiPOResponse.enType)));
   /*
      // Send the device selection response to connection manager
      if(nullptr != m_poMplayResp)
      {
         m_poMplayResp->vPostSelectDeviceResult(enResponseCode, enError);
      } // if(NULL != m_poMplayResp)
*/
      t_Bool bSwitchRes = (e8NO_ERRORS == enError);
      if(NULL !=  m_rDeviceSwitcherCbs.fvDeviceSwitchCompleteCb)
      {
         //! TODO RD send device ID + error code
         (m_rDeviceSwitcherCbs.fvDeviceSwitchCompleteCb)(0, bSwitchRes);
      }
	  
   
  
      // Explicitly requesting for an device list update from mediaplayer.
      // This logic is added because SPI is skipping any device disconnection update during a role switch.
      // So if there is a device disconnection during role switch,
      // We need a status update to update this to HMI.
      if(true == bDeviceListUpdateRequired)
      {
         trMsgContext rMsgContext;
         vPopulateMessageContext(rMsgContext);
         FIMsgDispatch msgDispatcher(m_poMainApp);
         MplayConnectionsGet oMplayConnGet;
         t_Bool bRetVal = msgDispatcher.bSendMessage(oMplayConnGet,
         rMsgContext,
         DIPO_MPLY_MAJORVERSION);
		 ETG_TRACE_USR1(("Message send status from vOnDiPORoleSwitchResult %d", bRetVal));

      }
   }
   else // if (!oRoleSwitchResult.bIsValid())
   {
      ETG_TRACE_ERR(("Message extraction failed."));
   }

   oRoleSwitchResult.vDestroy();
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::vOnDiPOGetMediaObjectAlbumArtInfo(amt_tclServiceData)
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vOnDiPOGetMediaObjectAlbumArtInfo(
         amt_tclServiceData* poMessage)
{
   //SPI_INTENTIONALLY_UNUSED(poMessage);
   ETG_TRACE_USR1((" spi_tclMPlayClientHandler::vOnDiPOGetMediaObjectAlbumArtInfo entered "));
   ETG_TRACE_USR1((" Registration Status: %d",m_bIsMetaDataPropertyRegistered));
   //SPI_INTENTIONALLY_UNUSED(poMessage);           //To remove Lint Warning

   spi_tMsgDipoGetAlbArtInfoResult oMsgDipoGetAlbArtInfoResult(*poMessage, DIPO_MPLY_MAJORVERSION);

   if(oMsgDipoGetAlbArtInfoResult.bIsValid()&& (true == m_bIsMetaDataPropertyRegistered))
   {
      m_rAppMediaMetaDataLock.s16Lock();
      m_rAppMediaMetaData.u32ImageSize = oMsgDipoGetAlbArtInfoResult.u32PhotoSize;
      GET_STRINGDATA_FROM_FI_STRINGOBJ(oMsgDipoGetAlbArtInfoResult.sMIMEImageSubtype,
            MPLAYFI_CHAR_SET_UTF8, m_rAppMediaMetaData.szImageMIMEType);

      ETG_TRACE_USR4(("[PARAM]: vOnDiPOGetMediaObjectAlbumArtInfo: ImageMIMEType = %s ",
      m_rAppMediaMetaData.szImageMIMEType.c_str()));

      ETG_TRACE_USR4(("[PARAM]: vOnDiPOGetMediaObjectAlbumArtInfo: ImageSize = %d ",
      m_rAppMediaMetaData.u32ImageSize));

      // Storing the AppMediaMetaData locally.
      trAppMediaMetaData rAppMediaMetaData = m_rAppMediaMetaData;

      m_rAppMediaMetaDataLock.vUnlock();
      // Posting meta data
      if (NULL != m_rMetaDataCbs.fvAppMediaMetaData)
      {
         m_rMetaDataCbs.fvAppMediaMetaData(rAppMediaMetaData, corEmptyUsrContext);
      }//if (NULL != m_poMplayResp)
   }
   else
   {
      ETG_TRACE_ERR(("[ERR]:spi_tclMPlayClientHandler::vOnDiPOGetMediaObjectAlbumArtInfo Message extraction failed."));
   }
   oMsgDipoGetAlbArtInfoResult.vDestroy();
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::vOnDiPOGetMediaObjectAlbumArt(amt_tclServiceData)
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vOnDiPOGetMediaObjectAlbumArt(
         amt_tclServiceData* poMessage)
{

   ETG_TRACE_USR1((" spi_tclMPlayClientHandler::vOnDiPOGetMediaObjectAlbumArt entered"));

   spi_tMsgDiPOGetAlbumArt oMsgDipoGetAlbumArtResult(*poMessage, DIPO_MPLY_MAJORVERSION);

   if(oMsgDipoGetAlbumArtResult.bIsValid())
   {
      //Set the image data of the album art
      m_rAppMediaAlbumArt.u8Items.clear();
      for (size_t u8Index = 0; u8Index
               < oMsgDipoGetAlbumArtResult.oImageData.u8Items.size(); u8Index++)
      {
         m_rAppMediaAlbumArt.u8Items.push_back(oMsgDipoGetAlbumArtResult.oImageData.u8Items[u8Index]);
      }

      trMsgContext rMsgContext = m_oAlbumArtMsgContext.rGetMsgContext(su32RequestID);
      su32RequestID--;

      // Posting albumart imagedata
      if (NULL != m_rMetaDataCbs.fvAppMediaAlbumArt)
      {
         (m_rMetaDataCbs.fvAppMediaAlbumArt)(m_rAppMediaAlbumArt, rMsgContext.rUserContext);
      }//if (NULL != m_poConnIntf)
   }
   else
   {
      ETG_TRACE_ERR(("spi_tclMPlayClientHandler::vOnDiPOGetMediaObjectAlbumArt Message extraction failed."));
   }
   oMsgDipoGetAlbumArtResult.vDestroy();
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::vOnDiPOTransferGPSData(amt_tclServiceData)
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vOnDiPOTransferGPSData(
         amt_tclServiceData* poMessage) const
{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));
   spi_tMsgDipoTransGPSDataReult oGPSDataResult(*poMessage,
            DIPO_MPLY_MAJORVERSION);
   if (oGPSDataResult.bIsValid())
   {
      if (mplay_fi_tcl_e8_DiPOResponse::FI_EN_E8DIPO_OK == oGPSDataResult.e8DiPOResponse.enType)
      {
         // TODO : send the response using response interface
      }
   }
   else
   {
       ETG_TRACE_ERR(("Message extraction failed."));
   }
   oGPSDataResult.vDestroy();
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::vOnDiPOTransferDRData(amt_tclServiceData)
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vOnDiPOTransferDRData(
         amt_tclServiceData* poMessage) const
{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));
   spi_tMsgDipoTransDRDataResult oDRDataResult(*poMessage,
            DIPO_MPLY_MAJORVERSION);

   if (oDRDataResult.bIsValid())
   {
      if (mplay_fi_tcl_e8_DiPOResponse::FI_EN_E8DIPO_OK == oDRDataResult.e8DiPOResponse.enType)
      {
         // TODO : send the response using response interface
      }
   }
   else
   {
       ETG_TRACE_ERR(("Message extraction failed."));
   }
   oDRDataResult.vDestroy();
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::vOnStatusDiPOLocationInfo(amt_tclServiceData)
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vOnStatusDiPOLocationInfo(
         amt_tclServiceData* poMessage) const
{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));

   spi_tMsgDipoLocInfoStatus oLocInfoStatus(*poMessage, DIPO_MPLY_MAJORVERSION);
   spi_tclFactory* poFactory = spi_tclFactory::getInstance();

   if ((true == oLocInfoStatus.bIsValid()) && (NULL != poFactory))
   {
         std::vector<tenNmeaSentenceType> NmeaSentencesList;
         spi_tclDataService* poDataService = poFactory->poGetDataServiceInstance();
         t_U8 u8SetFirstBit = 0;
         if (NULL != poDataService)
         {
            //! Populate required Nmea sentences into list for Start LocationInfo request
            if (oLocInfoStatus.bStartStopLocationInfo)
            {
               if (true == oLocInfoStatus.b8LocationInfoType.bGPGGA())
               {
                  ETG_TRACE_USR4(("vOnStatusDiPOLocationInfo: GPGGA sentence requested "));
                  NmeaSentencesList.push_back(e8NMEA_GPGGA);
               }
               if (true == oLocInfoStatus.b8LocationInfoType.bGPRMC())
               {
                  ETG_TRACE_USR4(("vOnStatusDiPOLocationInfo: GPRMC sentence requested "));
                  NmeaSentencesList.push_back(e8NMEA_GPRMC);
               }
               if (true == oLocInfoStatus.b8LocationInfoType.bPASCD())
               {
                  ETG_TRACE_USR4(("vOnStatusDiPOLocationInfo: PASCD sentence requested "));
                  NmeaSentencesList.push_back(e8NMEA_PASCD);
               }
               if (true == oLocInfoStatus.b8LocationInfoType.bPAGCD())
               {
                  ETG_TRACE_USR4(("vOnStatusDiPOLocationInfo: PAGCD sentence requested "));
                  NmeaSentencesList.push_back(e8NMEA_PAGCD);
               }
               /* Below formats are currently not supported
                  Code retained for future use.
               if (true == oLocInfoStatus.b8LocationInfoType.bGPGSV())
               {
                  NmeaSentencesList.push_back(e8NMEA_GPGSV);
               }
               if (true == oLocInfoStatus.b8LocationInfoType.bGPHDT())
               {
                  NmeaSentencesList.push_back(e8NMEA_GPHDT);
               }
               if (true == oLocInfoStatus.b8LocationInfoType.bPAACD())
               {
                  NmeaSentencesList.push_back(e8NMEA_PAACD);
               }*/
            } //if (oLocInfoStatus.bStartStopLocationInfo)

            //Set the bit position 1 to 1
            u8SetFirstBit |= 1<<1;
            //poDataService->vStartStopLocationData(
                  //oLocInfoStatus.bStartStopLocationInfo, NmeaSentencesList,u8SetFirstBit);
         }
         if (NULL != m_rDipoLocationInfoCallbacks.fvOnDipoLocationInfo)
            {
               (m_rDipoLocationInfoCallbacks.fvOnDipoLocationInfo)(oLocInfoStatus.bStartStopLocationInfo,NmeaSentencesList,u8SetFirstBit);

            }
   }
   else if (false == oLocInfoStatus.bIsValid())
   {
       ETG_TRACE_ERR(("Message extraction failed."));
   }
   oLocInfoStatus.vDestroy();
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::vOnDiPOReqAudioDevice(amt_tclServiceData)
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vOnDiPOReqAudioDevice(
         amt_tclServiceData* poMessage) const
{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));

   spi_tMsgDipoReqAudioDevResult oReqAudDevResult(*poMessage, DIPO_MPLY_MAJORVERSION);
   if (true == oReqAudDevResult.bIsValid())
   {
      //Retrieve Audio device name & error code
      t_String szAudioDevice;
      GET_STRINGDATA_FROM_FI_STRINGOBJ(oReqAudDevResult.sDeviceName,
            mplay_fi_tclString::FI_EN_UTF8, szAudioDevice);
      tenAudioError enAudioErr = enGetAudioError(oReqAudDevResult.e8Error);

      ETG_TRACE_USR3(("vOnDiPOReqAudioDevice: Received AudioError = %d, AudioDevice = %s ",
            ETG_ENUM(AUDIO_ERROR, enAudioErr), szAudioDevice.c_str()));

      //! Send Audio Device name & error code in loopback msg.
      tLbMplayAudioDevice oAudioDevLbMsg
            (
               m_poMainApp->u16GetAppId(),       // Source AppID
               m_poMainApp->u16GetAppId(),       // Target AppID
               0,                                // RegisterID
               (tU16)enAudioErr,                 // CmdCounter
               cou32SPI_SERVICE_ID,              // ServiceID
               SPI_C_U16_IFID_MPLAY_REQAUDIODEV_RESULT, // Function ID
               AMT_C_U8_CCAMSG_OPCODE_STATUS,    // Opcode
               static_cast<t_U32>(strlen(szAudioDevice.c_str()) + 1) // Buffer Size
            );

      //! Set the data & post message
      oAudioDevLbMsg.vSetData((const tChar*)(szAudioDevice.c_str()));

      if (true == oAudioDevLbMsg.bIsValid())
      {
         if (AIL_EN_N_NO_ERROR != m_poMainApp->enPostMessage(&oAudioDevLbMsg, TRUE))
         {
            ETG_TRACE_ERR(("vOnDiPOReqAudioDevice: Loopback message posting failed! "));
         }
      } //if (true == oAudioDevLbMsg.bIsValid())
      else
      {
        ETG_TRACE_ERR(("vOnDiPOReqAudioDevice: Loopback message creation failed! "));
      }
   }//if (true == oReqAudDevResult.bIsValid())
   else
   {
      ETG_TRACE_ERR(("vOnDiPOReqAudioDevice: Invalid message received! "));
   }
   oReqAudDevResult.vDestroy();
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::vOnDiPORelAudioDevice(amt_tclServiceData)
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vOnDiPORelAudioDevice(
         amt_tclServiceData* poMessage) const
{
   ETG_TRACE_USR1(("spi_tclMPlayClientHandler::vOnDiPORelAudioDevice() entered "));

   spi_tMsgDipoRelAudioDevResult oRelAudDevResult(*poMessage, DIPO_MPLY_MAJORVERSION);
   if (true == oRelAudDevResult.bIsValid())
   {
      //Retrieve error code
      tenAudioError enAudioErr = enGetAudioError(oRelAudDevResult.e8Error);

      ETG_TRACE_USR3(("vOnDiPORelAudioDevice: Received AudioError = %d ",
            ETG_ENUM(AUDIO_ERROR, enAudioErr)));

   }//if (true == oRelAudDevResult.bIsValid())
   else
   {
      ETG_TRACE_ERR(("vOnDiPORelAudioDevice: Invalid message received! "));
   }
   oRelAudDevResult.vDestroy();
}

/* Functions to interact with SPI design */
/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::bRoleSwitchRequest(tU8 deviceTag,.
 ***************************************************************************/
t_Bool spi_tclMPlayClientHandler::bRoleSwitchRequest(t_U32 u32DeviceID, tenDeviceConnectionReq enDevConnReq,trEAPAppInfo rEAPAppInfo,
         tenDeviceCategory endDeviceCat)
{
   ETG_TRACE_USR1((" spi_tclMPlayClientHandler::bRoleSwitchRequest DeviceID = %d Connectionrequest = %d endDeviceCat = %d ", u32DeviceID, enDevConnReq, endDeviceCat));
   t_Bool bRetVal = false;
   m_enDevConnReq = enDevConnReq;
   mplay_appcontrolfi_tclMsgDiPORoleSwitchRequestMethodStart oMplayRoleSwitch;

   // Generate message context
   trMsgContext rMsgContext;
   vPopulateMessageContext(rMsgContext);
   FIMsgDispatch msgDispatcher(m_poMainApp);
   if(NULL != m_poMplayDevList)
   {
      m_poMplayDevList->vSetRoleSwitchTriggerStatus(true);
   }

   t_U32 u32DeviceTag = 0;
   //! populate devicetag of mediaplayer before sending request
   if(NULL != m_poMplayDevList)
   {
      u32DeviceTag = m_poMplayDevList->u32GetMPDeviceTag(u32DeviceID, e8USB_CONNECTED);
   }

   if(e8DEVCONNREQ_SELECT == enDevConnReq)
   {
     ETG_TRACE_USR1(("From bDiPORoleSwitchRequest() connection called"));
     mplay_fi_tcl_e8_DiPORoleStatus::tenType e8DiPORoleSwitchType;
     if(e8DEV_TYPE_DIPO == endDeviceCat)
     {
        e8DiPORoleSwitchType = mplay_fi_tcl_e8_DiPORoleStatus::FI_EN_E8DIPO_CARPLAY_MODE;
     }else if (e8DEV_TYPE_CARLIFE == endDeviceCat)
     {
        e8DiPORoleSwitchType = mplay_fi_tcl_e8_DiPORoleStatus::FI_EN_E8DIPO_CARLIFE_NATIVE_TRANSPORT_MODE;
        oMplayRoleSwitch.oAppInfo.sAppName.bSet(rEAPAppInfo.szAppName.c_str(),MPLAYFI_CHAR_SET_UTF8);
        oMplayRoleSwitch.oAppInfo.sProtocol.bSet(rEAPAppInfo.szProtocol.c_str(),MPLAYFI_CHAR_SET_UTF8);
        oMplayRoleSwitch.oAppInfo.sBundleID.bSet(rEAPAppInfo.szBundleID.c_str(),MPLAYFI_CHAR_SET_UTF8);

     }else
     {
       e8DiPORoleSwitchType = mplay_fi_tcl_e8_DiPORoleStatus::FI_EN_E8DIPO_NATIVE_TRANSPORT_MODE;
     }
      // Message to do a role switch to the host mode
      oMplayRoleSwitch.u8DeviceTag = static_cast <tU8>(u32DeviceTag);
      //!For Dummy Implementation,will be removed later
      m_u32DeviceTag = u32DeviceTag;
      oMplayRoleSwitch.e8DiPORoleStatus.enType
         = e8DiPORoleSwitchType;
      bRetVal = msgDispatcher.bSendMessage(oMplayRoleSwitch,
         rMsgContext,
         DIPO_MPLY_MAJORVERSION);

   }
   else // if(e8DEV_CONNECT == enDevConnReq)
   {
      ETG_TRACE_USR1(("From bDiPORoleSwitchRequest() disconnection called"));
      oMplayRoleSwitch.u8DeviceTag = static_cast <tU8>(u32DeviceTag);
      oMplayRoleSwitch.e8DiPORoleStatus.enType
         = mplay_fi_tcl_e8_DiPORoleStatus::FI_EN_E8DIPO_CLIENTMODE;
      bRetVal = msgDispatcher.bSendMessage(oMplayRoleSwitch,
         rMsgContext,
         DIPO_MPLY_MAJORVERSION);

   } // if(e8DEV_CONNECT == enDevConnReq)
   if((false == bRetVal) && (NULL != m_poMplayDevList))
   {
      m_poMplayDevList->vSetRoleSwitchTriggerStatus(false);
   }
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::bDiPOGetAlbumArtInfo(tU8 deviceTag,DiPOImageURL url) 
 ***************************************************************************/
t_Bool spi_tclMPlayClientHandler::bDiPOGetAlbumArtInfo(t_String sAlbumArt)
{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));
   tBool bResult = false;

   if (nullptr != m_poMainApp)
   {
      mplay_appcontrolfi_tclMsgDiPOGetMediaObjectAlbumArtInfoMethodStart oMPlayAlbumArtInfo;
      oMPlayAlbumArtInfo.sAlbumArt.bSet(sAlbumArt.c_str(), MPLAYFI_CHAR_SET_UTF8);

      // Generate message context
      trMsgContext rMsgContext;
      vPopulateMessageContext(rMsgContext);
      FIMsgDispatch msgDispatcher(m_poMainApp);

      // Send method start to mplay fi
      bResult = msgDispatcher.bSendMessage(oMPlayAlbumArtInfo,
            rMsgContext,
            DIPO_MPLY_MAJORVERSION);
   } // if (nullptr != m_poMainApp)

   return bResult;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::bGetDiPOAlbumart(tU8 deviceTag,DiPOImageURL url)
 ***************************************************************************/
t_Bool spi_tclMPlayClientHandler::bGetDiPOAlbumArt(
        t_String szAlbumArtUrl, const trUserContext& corfrUsrCntxt)
{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));
   tBool bResult = false;

   if (corEmptyUsrContext != corfrUsrCntxt)
   {
      trMsgContext rAlbumArtMsgCtxt;
      rAlbumArtMsgCtxt.rUserContext = corfrUsrCntxt;
      su32RequestID++;
      m_oAlbumArtMsgContext.vAddMsgContext(su32RequestID, rAlbumArtMsgCtxt);

      if (nullptr != m_poMainApp)
      {
         mplay_appcontrolfi_tclMsgDiPOGetMediaObjectAlbumArtMethodStart oMPlayAlbumArt;
         oMPlayAlbumArt.sAlbumArt.bSet(szAlbumArtUrl.c_str(), MPLAYFI_CHAR_SET_UTF8);

      // populate message context
      trMsgContext rMsgContext;
      vPopulateMessageContext(rMsgContext);
      FIMsgDispatch msgDispatcher(m_poMainApp);

      // send method start to mplay fi
      bResult = msgDispatcher.bSendMessage(oMPlayAlbumArt,
               rMsgContext,
               DIPO_MPLY_MAJORVERSION);
      }
   }

   return bResult;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::bTransferGPSData(...)
 ***************************************************************************/
t_Bool spi_tclMPlayClientHandler::bTransferGPSData(t_U32 u32DeviceHandle, t_String szDeviceSerialNo,
      t_String szGPGGAData, t_String szGPRMCData,
      t_String szGPGSVData, t_String szGPHDTData)
{
   ETG_TRACE_USR1(("spi_tclMPlayClientHandler::bTransferGPSData() entered: DeviceHandle = %d, SerialNo = %s ",
            u32DeviceHandle, szDeviceSerialNo.c_str()));

   t_Bool bIsDeviceIDValid = (0 != u32DeviceHandle) || (false == szDeviceSerialNo.empty());
   t_Bool bResult = false;
   if ((nullptr != m_poMainApp) && (bIsDeviceIDValid))
   {
      mplay_appcontrolfi_tclMsgDiPOTransferGPSDataMethodStart oMPlayGPSData;

      //! populate MP devicetag or device serial number before sending request
      oMPlayGPSData.u8DeviceTag = u8GetMPDeviceTag(u32DeviceHandle);
      oMPlayGPSData.sDeviceUSBSerialNumber.bSet(szDeviceSerialNo.c_str(), MPLAYFI_CHAR_SET_UTF8);

      oMPlayGPSData.sGPGGAData.bSet(szGPGGAData.c_str(), MPLAYFI_CHAR_SET_UTF8);
      oMPlayGPSData.sGPRMCData.bSet(szGPRMCData.c_str(), MPLAYFI_CHAR_SET_UTF8);
      oMPlayGPSData.sGPGSVData.bSet(szGPGSVData.c_str(), MPLAYFI_CHAR_SET_UTF8);
      oMPlayGPSData.sGPHDTData.bSet(szGPHDTData.c_str(), MPLAYFI_CHAR_SET_UTF8);

      // populate message context
      trMsgContext rMsgContext;
      vPopulateMessageContext(rMsgContext);
      FIMsgDispatch msgDispatcher(m_poMainApp);

      // send the message to mplay fi
      bResult = msgDispatcher.bSendMessage(oMPlayGPSData,
               rMsgContext,
               DIPO_MPLY_MAJORVERSION);
   }
   else if (false == bIsDeviceIDValid)
   {
      ETG_TRACE_ERR(("[ERR]:spi_tclMPlayClientHandler::bTransferGPSData:: Device handle / Serial number is invalid"));
   }
   return bResult;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::bTransferDrData(...)
 ***************************************************************************/
t_Bool spi_tclMPlayClientHandler::bTransferDrData(t_U32 u32DeviceHandle, t_String szDeviceSerialNo,
      t_String szPASCDData, t_String szPAGCDData, t_String szPAACDData)
{
   ETG_TRACE_USR1(("spi_tclMPlayClientHandler::bTransferDrData() entered: DeviceHandle = %d, SerialNo = %s ",
            u32DeviceHandle, szDeviceSerialNo.c_str()));

   t_Bool bIsDeviceIDValid = (0 != u32DeviceHandle) || (false == szDeviceSerialNo.empty());
   t_Bool bResult = false;
   if ((nullptr != m_poMainApp) && (bIsDeviceIDValid))
   {
      mplay_appcontrolfi_tclMsgDiPOTransferDRDataMethodStart oMplayDrData;

      //! populate MP devicetag or device serial number before sending request
      oMplayDrData.u8DeviceTag = u8GetMPDeviceTag(u32DeviceHandle);
      oMplayDrData.sDeviceUSBSerialNumber.bSet(szDeviceSerialNo.c_str(), MPLAYFI_CHAR_SET_UTF8);

      oMplayDrData.sPASCDData.bSet(szPASCDData.c_str(), MPLAYFI_CHAR_SET_UTF8);
      oMplayDrData.sPAGCDData.bSet(szPAGCDData.c_str(), MPLAYFI_CHAR_SET_UTF8);
      oMplayDrData.sPAACDData.bSet(szPAACDData.c_str(), MPLAYFI_CHAR_SET_UTF8);

      // populate message context
      trMsgContext rMsgContext;
      vPopulateMessageContext(rMsgContext);
      FIMsgDispatch msgDispatcher(m_poMainApp);

      // send the message to mplay fi
      bResult = msgDispatcher.bSendMessage(oMplayDrData, rMsgContext,
               DIPO_MPLY_MAJORVERSION);
   }
   else if (false == bIsDeviceIDValid)
   {
      ETG_TRACE_ERR(("[ERR]:spi_tclMPlayClientHandler::bTransferDrData:: Device handle / Serial number is invalid"));
   }
   return bResult;
}

/***************************************************************************
 ** FUNCTION:  t_U8 spi_tclMPlayClientHandler::u8GetMPDeviceTag(...)
 ***************************************************************************/
t_U8 spi_tclMPlayClientHandler::u8GetMPDeviceTag(t_U32 u32DeviceHandle)
{
   t_U8 u8DeviceTag = 0;
   if ((0 != u32DeviceHandle) && (NULL != m_poMplayDevList))
   {
      spi_tclDeviceIDDataIntf oDeviceDataIntf;
      tenDeviceConnectionType enConnType = e8UNKNOWN_CONNECTION;
      tenSessionTransport enSessionTransPort = e8_SESSION_TRANSPORT_UNKNOWN;
      oDeviceDataIntf.vGetSessionTransport(u32DeviceHandle,enSessionTransPort);

      //Based on the Session Transport Type, set the Connection Type
      if(enSessionTransPort == e8_SESSION_TRANSPORT_USB)
      {
         enConnType = e8USB_CONNECTED;
      }
      else if(enSessionTransPort == e8_SESSION_TRANSPORT_WIFI)
      {
         enConnType = e8WIRELESS_CONNECTED;
      }
      u8DeviceTag = static_cast<t_U8>(m_poMplayDevList->u32GetMPDeviceTag(u32DeviceHandle, enConnType));
   }
   return u8DeviceTag;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::bRequestAudioDevice()
 ***************************************************************************/
t_Bool spi_tclMPlayClientHandler::bRequestAudioDevice(t_U32 u32DeviceHandle)
{
  ETG_TRACE_USR1(("spi_tclMPlayClientHandler::bRequestAudioDevice: DeviceTag %d ", u32DeviceHandle));

   t_Bool bResult = false;

   if (nullptr != m_poMainApp)
   {
      mplay_appcontrolfi_tclMsgDiPORequestAudioDeviceMethodStart oMplayReqAudDeviceMS;
      //! populate devicetag of mediaplayer before sending request
      t_U32 u32DeviceTag = 0;
      if(NULL != m_poMplayDevList)
      {
         u32DeviceTag = m_poMplayDevList->u32GetMPDeviceTag(u32DeviceHandle, e8USB_CONNECTED);
      }
      oMplayReqAudDeviceMS.u8DeviceTag = static_cast <tU8>(u32DeviceTag);
      
      //! populate message context
      trMsgContext rMsgContext;
      vPopulateMessageContext(rMsgContext);
      FIMsgDispatch msgDispatcher(m_poMainApp);

      //! send the message to mediaplayer
      bResult = msgDispatcher.bSendMessage(oMplayReqAudDeviceMS, rMsgContext, DIPO_MPLY_MAJORVERSION);

      if (false == bResult)
      {
         ETG_TRACE_ERR((" bRequestAudioDevice: Posting RequestAudioDevice MS message failed! "));
      }
   }//if (nullptr != m_poMainApp)
   return bResult;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::bReleaseAudioDevice()
 ***************************************************************************/
t_Bool spi_tclMPlayClientHandler::bReleaseAudioDevice(t_U32 u32DeviceHandle)
{
  ETG_TRACE_USR1(("spi_tclMPlayClientHandler::bReleaseAudioDevice: DeviceTag %d ", u32DeviceHandle));

   t_Bool bResult = false;

   if (nullptr != m_poMainApp)
   {
      mplay_appcontrolfi_tclMsgDiPOReleaseAudioDeviceMethodStart oMplayRelAudDeviceMS;
      //! populate devicetag of mediaplayer before sending request
      t_U32 u32DeviceTag = 0;
      if(NULL != m_poMplayDevList)
      {
         u32DeviceTag = m_poMplayDevList->u32GetMPDeviceTag(u32DeviceHandle, e8USB_CONNECTED);
      }
      oMplayRelAudDeviceMS.u8DeviceTag = static_cast <tU8>(u32DeviceTag);

      //! populate message context
      trMsgContext rMsgContext;
      vPopulateMessageContext(rMsgContext);
      FIMsgDispatch msgDispatcher(m_poMainApp);

      //! send the message to mediaplayer
      bResult = msgDispatcher.bSendMessage(oMplayRelAudDeviceMS, rMsgContext, DIPO_MPLY_MAJORVERSION);

      if (false == bResult)
      {
         ETG_TRACE_ERR((" bReleaseAudioDevice: Posting ReleaseAudioDevice MS message failed! "));
      }
   }//if (nullptr != m_poMainApp)
   return bResult;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::vPopulateMessageContext(trMsgContext
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vPopulateMessageContext(
         trMsgContext &rMsgContext) const
{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));
   rMsgContext.rUserContext.u32DestAppID = u16GetServerAppID();
   rMsgContext.rUserContext.u32SrcAppID = CCA_C_U16_APP_SMARTPHONEINTEGRATION;
   rMsgContext.rUserContext.u32RegID = u16GetRegID();
   rMsgContext.rUserContext.u32CmdCtr = 0;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::vPopulateDipoVersion(trMsgContext
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vPopulateDipoVersion(t_String szSource,
         t_U32 &u32XValue, t_U32 &u32YValue, t_U32 &u32ZValue) const
{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));

   t_U8 u8FirstSepPos = static_cast <t_U8>(szSource.find_first_of("."));
   t_U8 u8SecondSepPos = static_cast <t_U8>(szSource.find_first_of(".", u8FirstSepPos+1));
   t_String szXValue = szSource.substr(0, u8FirstSepPos);
   t_String szYValue = szSource.substr(u8FirstSepPos+1, u8SecondSepPos-1);
   t_String szZValue = szSource.substr(u8SecondSepPos+1);
   u32XValue = atoi(szXValue.c_str());
   u32YValue = atoi(szYValue.c_str());
   u32ZValue = atoi(szZValue.c_str());
   ETG_TRACE_USR1(("Function vPopulateDipoVersion() XValue == %s",szXValue.c_str()));
   ETG_TRACE_USR1(("Function vPopulateDipoVersion() XValue == %d",u32XValue ));
}

/***************************************************************************
 ** FUNCTION:  tenAudioError spi_tclMPlayClientHandler::enGetAudioError(...)
 ***************************************************************************/
tenAudioError spi_tclMPlayClientHandler::enGetAudioError(
      const mplay_tFiAudError& rfcpAudioError) const
{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));

   tenAudioError enAudError;
   switch (rfcpAudioError.enType)
   {
      case mplay_tFiAudError::FI_EN_E8NO_ERROR:
         enAudError = e8_AUDIOERROR_NONE;
         break;
      case mplay_tFiAudError::FI_EN_E8ACTIVATION_ERROR:
         enAudError = e8_AUDIOERROR_AVACTIVATION;
         break;
      case mplay_tFiAudError::FI_EN_E8ALLOCATE_ERROR:
         enAudError = e8_AUDIOERROR_ALLOCATE;
         break;
      case mplay_tFiAudError::FI_EN_E8SOURCEACT_ON_ERROR:
         enAudError = e8_AUDIOERROR_STARTSOURCEACT;
         break;
      case mplay_tFiAudError::FI_EN_E8DEACTIVATION_ERROR:
         enAudError = e8_AUDIOERROR_AVDEACTIVATION;
         break;
      case mplay_tFiAudError::FI_EN_E8DEALLOCATE_ERROR:
         enAudError = e8_AUDIOERROR_DEALLOCATE;
         break;
      case mplay_tFiAudError::FI_EN_E8SOURCEACT_OFF_ERROR:
         enAudError = e8_AUDIOERROR_STOPSOURCEACT;
         break;
      case mplay_tFiAudError::FI_EN_E8UNKNOWN_ERROR:
      default:
         enAudError = e8_AUDIOERROR_UNKNOWN;
         break;
   }//switch (rfcpAudioError.enType)

   ETG_TRACE_USR3(("enGetAudioError() left with %d ",
         ETG_ENUM(AUDIO_ERROR, enAudError)));
   return enAudError;
}

// code for testing
t_Void spi_tclMPlayClientHandler::vTestMediaPlayer()
{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));
   /*mplay_appcontrolfi_tclMsgDiPODeviceConnectionsStatus oDipoDevConnStatus;
   mplay_fi_tcl_DiPODeviceInfoItem oInfoItem;


   oInfoItem.u8DeviceTag = 1;
   oInfoItem.bDeviceConnected = true;
   oInfoItem.bDiPOActive = false;
   oInfoItem.sDeviceName.bSet("iphone",
      mplay_fi_tclString::FI_EN_UTF8);
   oInfoItem.sDeviceSerialNumber.bSet("abcd", mplay_fi_tclString::FI_EN_UTF8);
   oInfoItem.bDeviceActiveSource = true;
   oInfoItem.sDiPOVersion.bSet("1.1.1", mplay_fi_tclString::FI_EN_UTF8);
   oInfoItem.bDiPOCapable = true;
   oInfoItem.e8DeviceType.enType = mplay_fi_tcl_e8_DiPODeviceType::FI_EN_E8DTY_IPHONE;

   oDipoDevConnStatus.oDeviceInfo.oItems.push_back(oInfoItem);

   // device two data
   oInfoItem.u8DeviceTag = 2;
   oInfoItem.bDeviceConnected = false;
   oInfoItem.bDiPOActive = false;
   oInfoItem.sDeviceName.bSet("iphone",
      mplay_fi_tclString::FI_EN_UTF8);
   oInfoItem.sDeviceSerialNumber.bSet("efgh", mplay_fi_tclString::FI_EN_UTF8);
   oInfoItem.bDeviceActiveSource = true;
   oInfoItem.sDiPOVersion.bSet("1.1.1", mplay_fi_tclString::FI_EN_UTF8);
   oInfoItem.bDiPOCapable = true;
   oInfoItem.e8DeviceType.enType = mplay_fi_tcl_e8_DiPODeviceType::FI_EN_E8DTY_IPHONE;

   oDipoDevConnStatus.oDeviceInfo.oItems.push_back(oInfoItem);
   // device 3 data

   oInfoItem.u8DeviceTag = 3;
   oInfoItem.bDeviceConnected = false;
   oInfoItem.bDiPOActive = false;
   oInfoItem.sDeviceName.bSet("iphone",
      mplay_fi_tclString::FI_EN_UTF8);
   oInfoItem.sDeviceSerialNumber.bSet("efgh", mplay_fi_tclString::FI_EN_UTF8);
   oInfoItem.bDeviceActiveSource = true;
   oInfoItem.sDiPOVersion.bSet("1.1.1", mplay_fi_tclString::FI_EN_UTF8);
   oInfoItem.bDiPOCapable = true;
   oInfoItem.e8DeviceType.enType = mplay_fi_tcl_e8_DiPODeviceType::FI_EN_E8DTY_IPHONE;
   oDipoDevConnStatus.oDeviceInfo.oItems.push_back(oInfoItem);

   vOnDiPODeviceConnectionsStatus(oDipoDevConnStatus);*/
}


/***************************************************************************
** FUNCTION:  t_Bool spi_tclMPlayClientHandler::vRegisterCallbacks()
***************************************************************************/
t_Void spi_tclMPlayClientHandler::vRegisterCallbacks(const trDiscovererCbs &rfrDiscovererCbs)
{
   if(NULL != m_poMplayDevList)
   {
      m_poMplayDevList->vRegisterCallbacks(rfrDiscovererCbs);
   }
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclMPlayClientHandler::vRegisterCallbacks()
***************************************************************************/
t_Void spi_tclMPlayClientHandler::vRegisterCallbacks(const trDeviceSwitcherCbs &rfrDeviceSwitcherCbs)
{
   ETG_TRACE_USR1(("spi_tclMPlayClientHandler::vRegisterCallbacks for deviceswitcher "));
   m_rDeviceSwitcherCbs =  rfrDeviceSwitcherCbs;
}
/***************************************************************************
** FUNCTION:  t_Bool spi_tclMPlayClientHandler::vRegisterCallbacks()
***************************************************************************/
t_Void spi_tclMPlayClientHandler::vRegisterCallbacks(const tfvDeviceListUpdate &rfrDeviceListUpdate)
{
   ETG_TRACE_USR1(("spi_tclMPlayClientHandler::vRegisterCallbacks for devicelistupdate "));
   if(NULL != m_poMplayDevList)
   {
      m_poMplayDevList->vRegisterCallbacks(rfrDeviceListUpdate);
   }
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclMPlayClientHandler::vRegisterMetaDataCallbacks()
***************************************************************************/
t_Void spi_tclMPlayClientHandler::vRegisterMetaDataCallbacks(const trMetaDataCbs &rfrMetaDataCbs)
{
   ETG_TRACE_USR1(("spi_tclMPlayClientHandler::vRegisterMetaDataCallbacks for metadata "));
   m_rMetaDataCbs =  rfrMetaDataCbs;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclMPlayClientHandler::vRegisterDipoLocationCallbacks()
***************************************************************************/
t_Void spi_tclMPlayClientHandler::vRegisterDipoLocationCallbacks(const trDipoLocationInfoCallbacks &rfrDipoLocationInfoCallbacks)
{
   ETG_TRACE_USR1(("spi_tclMPlayClientHandler::vRegisterDipoLocationCallbacks entered "));
   m_rDipoLocationInfoCallbacks =  rfrDipoLocationInfoCallbacks;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclMPlayClientHandler::vUnregisterCallbacks()
***************************************************************************/
t_Void spi_tclMPlayClientHandler::vUnregisterCallbacks(const trDiscovererCbs &rfrDiscovererCbs)
{
   SPI_INTENTIONALLY_UNUSED(rfrDiscovererCbs);
   if(NULL != m_poMplayDevList)
   {
      m_poMplayDevList->vUnregisterCallbacks();
   }
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclMPlayClientHandler::vUnregisterCallbacks()
***************************************************************************/
t_Void spi_tclMPlayClientHandler::vUnregisterCallbacks(const trDeviceSwitcherCbs &rfrDeviceSwitcherCbs)
{
   SPI_INTENTIONALLY_UNUSED(rfrDeviceSwitcherCbs);
   ETG_TRACE_USR1(("spi_tclMPlayClientHandler::vUnregisterCallbacks for deviceswitcher "));
   //! Clear the callbacks
   m_rDeviceSwitcherCbs =  trDeviceSwitcherCbs();
}
/***************************************************************************
** FUNCTION:  t_Bool spi_tclMPlayClientHandler::vDiPORoleSwitchResultTest()
***************************************************************************/
t_Void spi_tclMPlayClientHandler::vDiPORoleSwitchResultTest()
{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));
  /* mplay_appcontrolfi_tclMsgDiPORoleSwitchRequestMethodResult oDiPORoleSwitchMR;
   oDiPORoleSwitchMR.e8DiPOResponse.enType = mplay_fi_tcl_e8_DiPOResponse::FI_EN_E8DIPO_OK;
   vOnDiPORoleSwitchResult(oDiPORoleSwitchMR);*/
}

/***************************************************************************
 ** FUNCTION:  tenAudioError spi_tclMPlayClientHandler::vOnNativeTransportStartStatus(...)
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vOnNativeTransportStartStatus(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));

   spi_tMsgNativeTransStartStatus oMsgNativeTransStart(*poMessage,DIPO_MPLY_MAJORVERSION);

   t_U32 u32DeviceId = 0;
   //! populate devicetag of SPI before sending request
   if(NULL != m_poMplayDevList)
   {
      u32DeviceId = m_poMplayDevList->u32GetSPIDeviceID(oMsgNativeTransStart.u32DeviceId, e8USB_CONNECTED);
   }
   t_U8 u8IAP2iOSAppIdentifier = oMsgNativeTransStart.u8IAP2iOSAppIdentifier;
   ETG_TRACE_USR1((" Device ID :[%d] iOSAppIdentifier:[%d] ",u32DeviceId,u8IAP2iOSAppIdentifier));

   if (NULL != m_rIAPDevInfoCallbacks.fvOnNativeTransportStart)
   {
      //Inform IAP connection, native transport has started.
      m_rIAPDevInfoCallbacks.fvOnNativeTransportStart(u32DeviceId);
   }
}

/***************************************************************************
 ** FUNCTION:  tenAudioError spi_tclMPlayClientHandler::vOnNativeTransportStopStatus(...)
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vOnNativeTransportStopStatus(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));
   spi_tMsgNativeTransStopStatus oMsgNativeTransStop(*poMessage,DIPO_MPLY_MAJORVERSION);

   t_U32 u32DeviceId = 0;
   //! populate devicetag of SPI before sending request
   if(NULL != m_poMplayDevList)
   {
      u32DeviceId = m_poMplayDevList->u32GetSPIDeviceID(oMsgNativeTransStop.u32DeviceId, e8USB_CONNECTED);
   }

   if (NULL != m_rIAPDevInfoCallbacks.fvOnNativeTransportStop)
   {
    //Send to iAP Connection
    m_rIAPDevInfoCallbacks.fvOnNativeTransportStop(u32DeviceId);
   }
}
/***************************************************************************
 ** FUNCTION:  tenAudioError spi_tclMPlayClientHandler::vOnConnectedBTProfilesUpdate(...)
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vOnConnectedBTProfilesUpdate(amt_tclServiceData* poMessage)
{

   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));

   spi_tConnectedBtProfilesStatus oConBtProfiles(*poMessage, DIPO_MPLY_MAJORVERSION);

   for (size_t u8Index = 0; u8Index < oConBtProfiles.oBTProfileInfo.oItems.size(); u8Index++)
   {
      t_Bool bIsHFP = oConBtProfiles.oBTProfileInfo.oItems[u8Index].b32BTProfile.bHandsFree();
      ETG_TRACE_USR2(("HFP is enabled [%d]", ETG_ENUM(BOOL, bIsHFP)));

      t_Bool bIsA2DP = oConBtProfiles.oBTProfileInfo.oItems[u8Index].b32BTProfile.bAdvancedAudioDistribution();
      ETG_TRACE_USR2(("A2DP is enabled [%d]", ETG_ENUM(BOOL, bIsA2DP)));
      t_U32 u32DeviceId = 0;
      //! populate devicetag of SPI before sending request
      if(NULL != m_poMplayDevList)
      {
         u32DeviceId = m_poMplayDevList->u32GetSPIDeviceID(oConBtProfiles.oBTProfileInfo.oItems[u8Index].u8DeviceId, e8USB_CONNECTED);
      }

      if (NULL != m_fvBTProfileCallback)
      {
         m_fvBTProfileCallback(u32DeviceId,bIsHFP,bIsA2DP,"");
      }
   }
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::bLaunchApp(...)
 ***************************************************************************/
t_Bool spi_tclMPlayClientHandler::bLaunchApp(const t_U32 cou32DevId, const tenDeviceCategory enDevcat, trEAPAppInfo rEAPAppInfo)
{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));
   ETG_TRACE_USR1(("bLaunchApp:  entered cou32DevId:%d, enDevcat:%d",cou32DevId,enDevcat));

   t_Bool bResult = false;

   // Generate message context
    trMsgContext rMsgContext;
    vPopulateMessageContext(rMsgContext);
    FIMsgDispatch msgDispatcher(m_poMainApp);

    mplay_appcontrolfi_tclMsgConnectMethodStart oLaucnhApp;

   if( enDevcat == e8DEV_TYPE_MYSPIN)
   {
      if (NULL != m_fvGetSelectedDevSerialCallback)
      {
         t_String szDeviceId;
         ETG_TRACE_USR1((" bLaunchApp: Myspin entered"));
         m_fvGetSelectedDevSerialCallback(cou32DevId,szDeviceId);
         oLaucnhApp.szDeviceId.bSet(szDeviceId.c_str(), MPLAYFI_CHAR_SET_UTF8);
         oLaucnhApp.szProtocol.bSet(rEAPAppInfo.szProtocol.c_str(), MPLAYFI_CHAR_SET_UTF8);
         oLaucnhApp.szBundleSeedID.bSet(rEAPAppInfo.szBundleID.c_str(), MPLAYFI_CHAR_SET_UTF8);
         oLaucnhApp.szAppName.bSet(rEAPAppInfo.szAppName.c_str(), MPLAYFI_CHAR_SET_UTF8);
         oLaucnhApp.e8AppLaunchOption.enType = static_cast<mplay_fi_tcl_e8_AppLaunchOption::tenType>(rEAPAppInfo.bLaunchType);
      }
   }else if (enDevcat == e8DEV_TYPE_CARLIFE)
   {
      ETG_TRACE_USR1((" bLaunchApp: Carlife entered"));
	   spi_tclDeviceIDDataIntf oDeviceDataIntf;
	   t_String szSerialNum;
	   ETG_TRACE_USR1(("spi_tclMPlayClientHandler::bLaunchApp() : szSerialNum:%s",szSerialNum.c_str()));
      oDeviceDataIntf.vGetSerialNumber(szSerialNum,cou32DevId);
	   ETG_TRACE_USR1(("spi_tclMPlayClientHandler::bLaunchApp() : szSerialNum:%s",szSerialNum.c_str()));
	   oLaucnhApp.szDeviceId.bSet(szSerialNum.c_str(), MPLAYFI_CHAR_SET_UTF8);
	   oLaucnhApp.szAppName.bSet(rEAPAppInfo.szAppName.c_str(),MPLAYFI_CHAR_SET_UTF8);
	   oLaucnhApp.szProtocol.bSet(rEAPAppInfo.szProtocol.c_str(),MPLAYFI_CHAR_SET_UTF8);
	   oLaucnhApp.szBundleSeedID.bSet(rEAPAppInfo.szBundleID.c_str(),MPLAYFI_CHAR_SET_UTF8);
	   oLaucnhApp.e8AppLaunchOption.enType = static_cast<mplay_fi_tcl_e8_AppLaunchOption::tenType>( rEAPAppInfo.bLaunchType);

   }
   // send the message to mplay fi
   bResult = msgDispatcher.bSendMessage(oLaucnhApp, rMsgContext,
            DIPO_MPLY_MAJORVERSION);
   

   return bResult;
}
/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::vOnConnectResult(...)
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vOnConnectResult(amt_tclServiceData* poMessage) const
{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));

   spi_tMsgConnectResult oConnectResult(*poMessage, DIPO_MPLY_MAJORVERSION);

   if (true == oConnectResult.bIsValid())
   {
      ETG_TRACE_USR1((" Session started with id [%d], for App [%s] ",oConnectResult.u16SessionId,
                                                                     oConnectResult.szAppName.szGet(mplay_fi_tclString::FI_EN_UTF8)));
   }

}
/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclMPlayClientHandler::bLaunchApp(...)
 ***************************************************************************/
t_Bool spi_tclMPlayClientHandler::bSendPlayBackCtrlEvent(const t_U32 cou32DevId, tenKeyCode enKeyCode, tenKeyMode enKeyMode)
{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));

   t_Bool bResult = false;
   t_Bool bSendReq = false;

   std::map<tenKeyCode, mplay_fi_tcl_e8_DiPOPlaypackAction::tenType>::const_iterator itrKeyCode;

   itrKeyCode = m_KeyActionMap.find(enKeyCode);

   if (m_KeyActionMap.end() != itrKeyCode)
   {
      mplay_appcontrolfi_tclMsgDiPORequestPlaybackActionMethodStart oPlayBackAct;

      // Generate message context
      trMsgContext rMsgContext;
      vPopulateMessageContext(rMsgContext);
      FIMsgDispatch msgDispatcher(m_poMainApp);

      //! populate devicetag of mediaplayer before sending request
      t_U32 u32DeviceTag = 0;
      if(NULL != m_poMplayDevList)
      {
         u32DeviceTag = m_poMplayDevList->u32GetMPDeviceTag(cou32DevId, e8USB_CONNECTED);
      }
      //!@TODO: Check for WIFI Case

      oPlayBackAct.u8DeviceTag = static_cast <tU8>(u32DeviceTag);
      if(e8KEY_PRESS == enKeyMode)
      {
          oPlayBackAct.e8Action.enType = itrKeyCode->second; //We need to react on KEY_PRESS always
          bSendReq = true;
      }
      else //We need to handle KEY_RELEASE for FFwd and FRev.
      {
          switch(enKeyCode)
          {
              case e32MULTIMEDIA_FORWARD:
                  oPlayBackAct.e8Action.enType = mplay_fi_tcl_e8_DiPOPlaypackAction::FI_EN_E8DIPO_PBA_FFWD_STOP;
                  bSendReq = true;
              break;
              case e32MULTIMEDIA_REWIND:
                  oPlayBackAct.e8Action.enType = mplay_fi_tcl_e8_DiPOPlaypackAction::FI_EN_E8DIPO_PBA_FREV_STOP;
                  bSendReq = true;
              break;
              default:
              break;
          }
      }

      if(true == bSendReq)
      {
          // send the message to mplay fi
          bResult = msgDispatcher.bSendMessage(oPlayBackAct, rMsgContext,
                   DIPO_MPLY_MAJORVERSION);
      }
    }

   return bResult;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::vRegisterCallbacks(trIAPDevInfoCallbacks)
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vRegisterIAPCallbacks(trIAPDevInfoCallbacks rIAPDevInfoCallbacks,
         tfvBTProfileCallback fvBTProfileCallback)
{
   if ((NULL != rIAPDevInfoCallbacks.fvOnNativeTransportStart)
            && (NULL != rIAPDevInfoCallbacks.fvOnNativeTransportStop))
   {
      m_rIAPDevInfoCallbacks = rIAPDevInfoCallbacks;
   }

   if (NULL != fvBTProfileCallback)
   {
      m_fvBTProfileCallback = fvBTProfileCallback;
   }
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclMPlayClientHandler::vOnStartiAP2OverCarPlayResult(amt_tclServiceData ..
***************************************************************************/
t_Void spi_tclMPlayClientHandler::vOnStartiAP2OverCarPlayResult(amt_tclServiceData* poMessage) const
{
	ETG_TRACE_USR1(("spi_tclMPlayClientHandler::vOnStartiAP2OverCarPlayResult()"));

	spi_tMsgStartiAP2OverCarPlayResult oStartiAP2OverCarPlayResult(*poMessage, DIPO_MPLY_MAJORVERSION);

	if (true == oStartiAP2OverCarPlayResult.bIsValid())
	{
		ETG_TRACE_USR1((" spi_tclMPlayClientHandler::vOnStartiAP2OverCarPlayResult() valid "));
	}
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclMPlayClientHandler::vStartiAP2OverCarPlay()
***************************************************************************/
t_Void spi_tclMPlayClientHandler::vStartiAP2OverCarPlay(t_String szBTMACAddress)
{
	ETG_TRACE_USR1(("spi_tclMPlayClientHandler::vOnStartiAP2OverCarPlay for device:%s", szBTMACAddress.c_str()));

	mplay_appcontrolfi_tclMsgDiPOStartIAP2OverCarPlayWifiMethodStart oStartiAP2OverCarPlayMS;
	std::transform(szBTMACAddress.begin(), szBTMACAddress.end(),szBTMACAddress.begin(), ::tolower);
	oStartiAP2OverCarPlayMS.sDeviceBTMacAddress.bSet(szBTMACAddress.c_str(), mplay_fi_tclString::FI_EN_UTF8);
	// generate message context
	trMsgContext rMsgContext;
	vPopulateMessageContext(rMsgContext);
	FIMsgDispatch msgDispatcher(m_poMainApp);

	// Send method start to mplay fi
	t_Bool bStatus = msgDispatcher.bSendMessage(oStartiAP2OverCarPlayMS,
		rMsgContext,
		DIPO_MPLY_MAJORVERSION);

	ETG_TRACE_USR4(("spi_tclMPlayClientHandler::vOnStartiAP2OverCarPlay send status:%d", ETG_ENUM(BOOL, bStatus)));
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::vRegisterGetSelectedDevSerialCb(tfvGetSelectedDevSerial)
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vRegisterGetSelectedDevSerialCb(
                                      tfvGetSelectedDevSerialCallback fvGetSelectedDevSerialCallback)
{
   if(NULL != fvGetSelectedDevSerialCallback)
   {
       m_fvGetSelectedDevSerialCallback = fvGetSelectedDevSerialCallback;
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::vOnDiPOPowerStatus(amt_tclServiceData)
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vOnDiPOPowerStatus(amt_tclServiceData* poMessage)
{
   ETG_TRACE_USR1(("spi_tclMPlayClientHandler::vOnDiPOPowerStatus entered"));

   spi_tMsgDiPOPowerStatus oMsgDiPOPowerData(*poMessage, DIPO_MPLY_MAJORVERSION);

   if (oMsgDiPOPowerData.bIsValid())
   {
      //Store the phone meta data info and send it to service.
      m_rAppPhoneDataLock.s16Lock();
      m_rAppPhoneData.bPhoneMetadataValid = true;
      m_rAppPhoneData.u8BatteryChargeLevel = static_cast<t_U8>(oMsgDiPOPowerData.u16BatteryChargeLevel);
      m_rAppPhoneData.enBatteryChargingState = static_cast<tenBatteryChargingState>(oMsgDiPOPowerData.e8BatteryChargingState.enType);

      ETG_TRACE_USR4(("[PARAM]:vOnDiPOPowerStatus: phone metadata validity = %d", ETG_ENUM(BOOL, m_rAppPhoneData.bPhoneMetadataValid)));

      ETG_TRACE_USR4(("[PARAM]:vOnDiPOPowerStatus : battery charge level = %d, battery charging state = %d ",
            m_rAppPhoneData.u8BatteryChargeLevel, ETG_ENUM(BATTERY_CHARGING_STATE, m_rAppPhoneData.enBatteryChargingState)));

      // Storing the AppPhoneData locally.
      trAppPhoneData rAppPhoneData = m_rAppPhoneData;

      m_rAppPhoneDataLock.vUnlock();

      //post the data to service
      if (NULL != m_rMetaDataCbs.fvAppPhoneData)
      {
         m_rMetaDataCbs.fvAppPhoneData(rAppPhoneData, corEmptyUsrContext);
      }//if (NULL != m_poMplayResp)
   }
   else
   {
      ETG_TRACE_ERR(("[ERR]:spi_tclMPlayClientHandler::vOnDiPOPowerStatus : Message extraction failed. "));
   }
   oMsgDiPOPowerData.vDestroy();
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::vRegisterTurnByTurnInfoUpdateCb(tfvGetSelectedDevSerial)
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vRegisterTBTCallback(const trTBTCallbacks &corfrTBTCallbacks)
{
    m_rTBTCallbacks = corfrTBTCallbacks;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::vOnDipoRouteGuidanceStatus()
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vOnDipoRouteGuidanceStatus(amt_tclServiceData* poMessage) const
{
    ETG_TRACE_USR1(("spi_tclMPlayClientHandler::vOnDipoRouteGuidanceStatus entered"));
    spi_tMsgDiPORouteGuidanceStatus oDiPORouteGuidanceStatus(*poMessage, DIPO_MPLY_MAJORVERSION);
    if(oDiPORouteGuidanceStatus.bIsValid())
    {
        trDiPORouteGuidanceUpdate rDiPORouteGuidanceUpdate;

        rDiPORouteGuidanceUpdate.enRouteGuidanceState = static_cast<tenDiPoRouteGuidanceState>(oDiPORouteGuidanceStatus.e8RouteGuidanceState.enType);
        rDiPORouteGuidanceUpdate.enManeuverState = static_cast<tenDiPoRouteManeuverState>(oDiPORouteGuidanceStatus.e8ManeuverState.enType);
        GET_STRINGDATA_FROM_FI_STRINGOBJ(oDiPORouteGuidanceStatus.szCurrentRoadName,mplay_fi_tclString::FI_EN_UTF8,
                                         rDiPORouteGuidanceUpdate.szCurrentRoadName);
        GET_STRINGDATA_FROM_FI_STRINGOBJ(oDiPORouteGuidanceStatus.szDestinationName,mplay_fi_tclString::FI_EN_UTF8,
                                         rDiPORouteGuidanceUpdate.szDestinationName);
        rDiPORouteGuidanceUpdate.u64EstimatedTimeOfArrival = oDiPORouteGuidanceStatus.u64EstimatedTimeOfArrival;
        rDiPORouteGuidanceUpdate.u64TimeRemainingToDestination = oDiPORouteGuidanceStatus.u64TimeRemainingToDestination;
        rDiPORouteGuidanceUpdate.u32DistanceRemaining = oDiPORouteGuidanceStatus.u32DistanceRemaining;
        GET_STRINGDATA_FROM_FI_STRINGOBJ(oDiPORouteGuidanceStatus.szDistanceRemainingDisplayStr,mplay_fi_tclString::FI_EN_UTF8,
                                         rDiPORouteGuidanceUpdate.szDistanceRemainingDisplayStr);
        rDiPORouteGuidanceUpdate.enDistanceRemainingDisplayUnits = enGetTBTDistanceUnits(oDiPORouteGuidanceStatus.e8DistanceRemainingDisplayUnits);
        rDiPORouteGuidanceUpdate.u32DistanceToNextManeuver = oDiPORouteGuidanceStatus.u32DistanceToNextManeuver;
        GET_STRINGDATA_FROM_FI_STRINGOBJ(oDiPORouteGuidanceStatus.szDistanceToNextManeuverDisplayStr,mplay_fi_tclString::FI_EN_UTF8,
                                         rDiPORouteGuidanceUpdate.szDistanceToNextManeuverDisplayStr);
        rDiPORouteGuidanceUpdate.enDistanceToNextManeuverDisplayUnits = enGetTBTDistanceUnits(oDiPORouteGuidanceStatus.e8DistanceToNextManeuverDisplayUnits);

        for(t_U16 u16Index = 0; u16Index < oDiPORouteGuidanceStatus.u16RouteGuidanceManeuverCurrentList.u16Items.size();
                u16Index = static_cast<t_U16>(u16Index+1))
        {
            rDiPORouteGuidanceUpdate.vecRouteGuidanceManeuverCurrentList.push_back(
                             oDiPORouteGuidanceStatus.u16RouteGuidanceManeuverCurrentList.u16Items[u16Index] );
        }

        rDiPORouteGuidanceUpdate.u16RouteGuidanceManeuverCount = oDiPORouteGuidanceStatus.u16RouteGuidanceManeuverCount;
        rDiPORouteGuidanceUpdate.bIsRouteGuidanceVisibleInApp = oDiPORouteGuidanceStatus.bIsRouteGuidanceVisibleInApp;

        ETG_TRACE_USR4(("[PARAM]::vOnDipoRouteGuidanceStatus RouteGuidanceState = %d, ManeuverState = %d",
                         ETG_ENUM(DIPO_ROUTE_GUIDANCE_STATE,oDiPORouteGuidanceStatus.e8RouteGuidanceState.enType),
                         ETG_ENUM(DIPO_MANEUVER_STATE,oDiPORouteGuidanceStatus.e8ManeuverState.enType)));
        ETG_TRACE_USR4(("[PARAM]::vOnDipoRouteGuidanceStatus CurrentRoadName = %s",rDiPORouteGuidanceUpdate.szCurrentRoadName.c_str()));
        ETG_TRACE_USR4(("[PARAM]::vOnDipoRouteGuidanceStatus DestinationName = %s",rDiPORouteGuidanceUpdate.szDestinationName.c_str()));
        ETG_TRACE_USR4(("[PARAM]::vOnDipoRouteGuidanceStatus EstimatedTimeOfArrival = %s", std::to_string(oDiPORouteGuidanceStatus.u64EstimatedTimeOfArrival).c_str()));
        ETG_TRACE_USR4(("[PARAM]::vOnDipoRouteGuidanceStatus TimeRemainingToDestination = %s",std::to_string(oDiPORouteGuidanceStatus.u64TimeRemainingToDestination).c_str()));
        ETG_TRACE_USR4(("[PARAM]::vOnDipoRouteGuidanceStatus DistanceRemaining = %d, DistanceRemainingDisplayUnits = %d, DistanceRemainingDisplayStr = %s",
                         oDiPORouteGuidanceStatus.u32DistanceRemaining,
                         ETG_ENUM(TBT_DISTANCE_UNITS,rDiPORouteGuidanceUpdate.enDistanceRemainingDisplayUnits),
                         rDiPORouteGuidanceUpdate.szDistanceRemainingDisplayStr.c_str()));
        ETG_TRACE_USR4(("[PARAM]::vOnDipoRouteGuidanceStatus DistanceToNextManeuver = %d, DistanceToNextManeuverDisplayUnits = %d, DistanceToNextManeuverDisplayStr = %s",
                         oDiPORouteGuidanceStatus.u32DistanceToNextManeuver,
                         ETG_ENUM(TBT_DISTANCE_UNITS,rDiPORouteGuidanceUpdate.enDistanceToNextManeuverDisplayUnits),
                         rDiPORouteGuidanceUpdate.szDistanceToNextManeuverDisplayStr.c_str()));
        ETG_TRACE_USR4(("[PARAM]::vOnDipoRouteGuidanceStatus RouteGuidanceManeuverCount = %d, IsRouteGuidanceVisibleInApp = %d",
                         oDiPORouteGuidanceStatus.u16RouteGuidanceManeuverCount,
                         ETG_ENUM(BOOL,oDiPORouteGuidanceStatus.bIsRouteGuidanceVisibleInApp)));

        if(NULL != m_rTBTCallbacks.fvOnDiPORouteGuidanceUpdateCallback)
        {
            m_rTBTCallbacks.fvOnDiPORouteGuidanceUpdateCallback(rDiPORouteGuidanceUpdate);
        }
    }//end of if(oDiPORouteGuidanceStatus.bIsValid())
}//end of vOnDipoRouteGuidanceStatus()

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::vOnDipoRouteGuidanceManeuverStatus()
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vOnDipoRouteGuidanceManeuverStatus(amt_tclServiceData* poMessage) const
{
    ETG_TRACE_USR1(("spi_tclMPlayClientHandler::vOnDipoRouteGuidanceManeuverStatus entered"));
    spi_tMsgDiPORouteGuidanceManeuverStatus oDiPORouteGuidanceManeuverStatus(*poMessage, DIPO_MPLY_MAJORVERSION);
    if(oDiPORouteGuidanceManeuverStatus.bIsValid())
    {
        trDiPORouteGuidanceManeuverUpdate rDiPORouteGuidanceManeuverUpdate;

        rDiPORouteGuidanceManeuverUpdate.u16Index = oDiPORouteGuidanceManeuverStatus.u16Index;
        GET_STRINGDATA_FROM_FI_STRINGOBJ(oDiPORouteGuidanceManeuverStatus.szManeuverDescription,mplay_fi_tclString::FI_EN_UTF8,
                                         rDiPORouteGuidanceManeuverUpdate.szManeuverDescription);
        rDiPORouteGuidanceManeuverUpdate.enManeuverType = static_cast<tenDiPoManeuverType>(oDiPORouteGuidanceManeuverStatus.e8ManeuverType.enType);
        GET_STRINGDATA_FROM_FI_STRINGOBJ(oDiPORouteGuidanceManeuverStatus.szAfterManeuverRoadName,mplay_fi_tclString::FI_EN_UTF8,
                                         rDiPORouteGuidanceManeuverUpdate.szAfterManeuverRoadName);
        rDiPORouteGuidanceManeuverUpdate.u32DistanceBetweenManeuver = oDiPORouteGuidanceManeuverStatus.u32DistanceBetweenManeuver;
        GET_STRINGDATA_FROM_FI_STRINGOBJ(oDiPORouteGuidanceManeuverStatus.szDistanceBetweenManeuverDisplayStr,mplay_fi_tclString::FI_EN_UTF8,
                                         rDiPORouteGuidanceManeuverUpdate.szDistanceBetweenManeuverDisplayStr);
        rDiPORouteGuidanceManeuverUpdate.enDistanceBetweenManeuverDisplayUnits = enGetTBTDistanceUnits(oDiPORouteGuidanceManeuverStatus.e8DistanceBetweenManeuverDisplayUnits);
        rDiPORouteGuidanceManeuverUpdate.enDrivingSide = static_cast<tenDiPoDrivingSide>(oDiPORouteGuidanceManeuverStatus.e8DrivingSide.enType);
        rDiPORouteGuidanceManeuverUpdate.enJunctionType = static_cast<tenDiPoJunctionType>(oDiPORouteGuidanceManeuverStatus.e8JunctionType.enType);
        rDiPORouteGuidanceManeuverUpdate.vecJunctionElementAngleList.push_back(static_cast<t_S16>(oDiPORouteGuidanceManeuverStatus.s16JunctionElementAngle));
        rDiPORouteGuidanceManeuverUpdate.s16JunctionElementExitAngle = oDiPORouteGuidanceManeuverStatus.s16JunctionElementExitAngle;

        ETG_TRACE_USR4(("[PARAM]::vOnDipoRouteGuidanceManeuverStatus Index = %d, ManeuverType = %d",
                         oDiPORouteGuidanceManeuverStatus.u16Index,
                         oDiPORouteGuidanceManeuverStatus.e8ManeuverType.enType));
        ETG_TRACE_USR4(("[PARAM]::vOnDipoRouteGuidanceManeuverStatus ManeuverDescription = %s",rDiPORouteGuidanceManeuverUpdate.szManeuverDescription.c_str()));
        ETG_TRACE_USR4(("[PARAM]::vOnDipoRouteGuidanceManeuverStatus AfterManeuverRoadName = %s",rDiPORouteGuidanceManeuverUpdate.szAfterManeuverRoadName.c_str()));
        ETG_TRACE_USR4(("[PARAM]::vOnDipoRouteGuidanceManeuverStatus DistanceBetweenManeuver=%d, DistanceBetweenManeuverDisplayUnits=%d, DistanceBetweenManeuverDisplayStr=%s",
                         oDiPORouteGuidanceManeuverStatus.u32DistanceBetweenManeuver,
                         ETG_ENUM(TBT_DISTANCE_UNITS,rDiPORouteGuidanceManeuverUpdate.enDistanceBetweenManeuverDisplayUnits),
                         rDiPORouteGuidanceManeuverUpdate.szDistanceBetweenManeuverDisplayStr.c_str()));
        ETG_TRACE_USR4(("[PARAM]::vOnDipoRouteGuidanceManeuverStatus DrivingSide = %d, JunctionType = %d, JunctionElementAngle = %d, JunctionElementExitAngle = %d",
                         ETG_ENUM(DIPO_DRIVING_SIDE,oDiPORouteGuidanceManeuverStatus.e8DrivingSide.enType),
                         ETG_ENUM(DIPO_JUNCTION_TYPE,oDiPORouteGuidanceManeuverStatus.e8JunctionType.enType),
                         oDiPORouteGuidanceManeuverStatus.s16JunctionElementAngle,
                         oDiPORouteGuidanceManeuverStatus.s16JunctionElementExitAngle));

        if(NULL != m_rTBTCallbacks.fvOnDiPORouteGuidanceManeuverUpdateCallback)
        {
            m_rTBTCallbacks.fvOnDiPORouteGuidanceManeuverUpdateCallback(rDiPORouteGuidanceManeuverUpdate);
        }
    }//end of if(oDiPORouteGuidanceManeuverStatus())
}//end of vOnDipoRouteGuidanceManeuverStatus()

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::vOnMock_DipoRouteGuidanceManeuverStatus(....)
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vOnMock_DipoRouteGuidanceManeuverStatus(tenDiPoManeuverType enManeuverType,
         tenDiPoDrivingSide enDrivingSide, std::vector<t_S16> vecJunctionElementAngleList,
         t_S16 s16JunctionElementExitAngle)
{
   ETG_TRACE_USR1(("spi_tclMPlayClientHandler::vOnMock_DipoRouteGuidanceManeuverStatus entered"));

   trDiPORouteGuidanceManeuverUpdate rDiPORouteGuidanceManeuverUpdate;
   //Hard coding the values
   rDiPORouteGuidanceManeuverUpdate.u16Index = 1;
   rDiPORouteGuidanceManeuverUpdate.enManeuverType = enManeuverType;
   rDiPORouteGuidanceManeuverUpdate.u32DistanceBetweenManeuver = 1;
   rDiPORouteGuidanceManeuverUpdate.enDistanceBetweenManeuverDisplayUnits = static_cast<tenTBTDistanceUnits>(1);
   rDiPORouteGuidanceManeuverUpdate.enDrivingSide = enDrivingSide;
   rDiPORouteGuidanceManeuverUpdate.enJunctionType = static_cast<tenDiPoJunctionType>(1);
   //If the angle is more than 180 you need to send a -ve value.
   for (t_U16 u16Index = 0; u16Index < rDiPORouteGuidanceManeuverUpdate.vecJunctionElementAngleList.size(); u16Index =
            static_cast<t_U16>(u16Index + 1))
   {
      if (vecJunctionElementAngleList[u16Index] > MAXANGLE)
      {
         rDiPORouteGuidanceManeuverUpdate.vecJunctionElementAngleList[u16Index] =
                  static_cast<t_S16>(-((vecJunctionElementAngleList[u16Index]) - MAXANGLE));
      }
      else
      {
         rDiPORouteGuidanceManeuverUpdate.vecJunctionElementAngleList[u16Index] = vecJunctionElementAngleList[u16Index];
      }
      ETG_TRACE_USR4(("[PARAM]::vOnMock_DipoRouteGuidanceManeuverStatus - JunctionElementAngle[%d] = %d", u16Index, rDiPORouteGuidanceManeuverUpdate.vecJunctionElementAngleList[u16Index]));
   }
   if (s16JunctionElementExitAngle > MAXANGLE)
   {
      rDiPORouteGuidanceManeuverUpdate.s16JunctionElementExitAngle = static_cast<t_S16>(-(s16JunctionElementExitAngle - MAXANGLE));
   }
   else
   {
      rDiPORouteGuidanceManeuverUpdate.s16JunctionElementExitAngle = s16JunctionElementExitAngle;
   }
   ETG_TRACE_USR4(("[PARAM]::vOnMock_DipoRouteGuidanceManeuverStatus Driving side = %d, ManeuverType = %d , JunctionElementExitAngle = %d ", rDiPORouteGuidanceManeuverUpdate.enDrivingSide, rDiPORouteGuidanceManeuverUpdate.enManeuverType, rDiPORouteGuidanceManeuverUpdate.s16JunctionElementExitAngle));

   if (NULL != m_rTBTCallbacks.fvOnDiPORouteGuidanceManeuverUpdateCallback)
   {
      ETG_TRACE_USR1(("spi_tclMPlayClientHandler::vOnMock_DipoRouteGuidanceManeuverStatus calling the callback function"));
      m_rTBTCallbacks.fvOnDiPORouteGuidanceManeuverUpdateCallback(rDiPORouteGuidanceManeuverUpdate);
   }
}
/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::vDipoStartRouteGuidanceUpdate()
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vDipoStartRouteGuidanceUpdate(const t_String& corfszDeviceSerialNumber)
{
   ETG_TRACE_USR1(("spi_tclMPlayClientHandler::vDipoStartRouteGuidanceUpdate entered with Device Serial Number = %s", corfszDeviceSerialNumber.c_str()));
   mplay_appcontrolfi_tclMsgDiPOStartRouteGuidanceMethodStart oDiPOStartRouteGuidanceMethodStart;
   oDiPOStartRouteGuidanceMethodStart.sDeviceUSBSerialNumber.bSet(corfszDeviceSerialNumber.c_str(), MPLAYFI_CHAR_SET_UTF8);
   oDiPOStartRouteGuidanceMethodStart.SourceName = true;
   oDiPOStartRouteGuidanceMethodStart.SourceSupportsRouteGuidance = true;

   trMsgContext rMsgContext;
   vPopulateMessageContext(rMsgContext);
   FIMsgDispatch msgDispatcher(m_poMainApp);

   // send the message to mplay fi
   t_Bool bResult = msgDispatcher.bSendMessage(oDiPOStartRouteGuidanceMethodStart, rMsgContext,
                                                DIPO_MPLY_MAJORVERSION);
   SPI_INTENTIONALLY_UNUSED(bResult);
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::vDipoStopRouteGuidanceUpdate()
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vDipoStopRouteGuidanceUpdate(const t_String& corfszDeviceSerialNumber)
{
    ETG_TRACE_USR1(("spi_tclMPlayClientHandler::vDipoStopRouteGuidanceUpdate entered with Device Serial Number = %s",corfszDeviceSerialNumber.c_str()));
    mplay_appcontrolfi_tclMsgDiPOStopRouteGuidanceMethodStart oDiPOStopRouteGuidanceMethodStart;
    oDiPOStopRouteGuidanceMethodStart.sDeviceUSBSerialNumber.bSet(corfszDeviceSerialNumber.c_str(), MPLAYFI_CHAR_SET_UTF8);

    trMsgContext rMsgContext;
    vPopulateMessageContext(rMsgContext);
    FIMsgDispatch msgDispatcher(m_poMainApp);

    // send the message to mplay fi
    t_Bool bResult = msgDispatcher.bSendMessage(oDiPOStopRouteGuidanceMethodStart, rMsgContext,
                                                DIPO_MPLY_MAJORVERSION);
    SPI_INTENTIONALLY_UNUSED(bResult);
}

/***************************************************************************
 ** FUNCTION:  tenTBTDistanceUnits spi_tclMPlayClientHandler::enGetTBTDistanceUnits(...)
 ***************************************************************************/
tenTBTDistanceUnits spi_tclMPlayClientHandler::enGetTBTDistanceUnits(const mplay_tFiDistanceDisplayUnits& rfcpDistanceUnits) const
{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));

   tenTBTDistanceUnits enTBTDistanceUnits;
   switch(rfcpDistanceUnits.enType)
   {
      case mplay_tFiDistanceDisplayUnits::FI_EN_E8DIPO_DISTANCE_DISPLAY_UNIT_KM:
          enTBTDistanceUnits = e8TBT_DISTANCE_UNITS_KILOMETERS;
      break;
      case mplay_tFiDistanceDisplayUnits::FI_EN_E8DIPO_DISTANCE_DISPLAY_UNIT_MILES:
          enTBTDistanceUnits = e8TBT_DISTANCE_UNITS_MILES;
      break;
      case mplay_tFiDistanceDisplayUnits::FI_EN_E8DIPO_DISTANCE_DISPLAY_UNIT_M:
          enTBTDistanceUnits = e8TBT_DISTANCE_UNITS_METERS;
      break;
      case mplay_tFiDistanceDisplayUnits::FI_EN_E8DIPO_DISTANCE_DISPLAY_UNIT_YARDS:
          enTBTDistanceUnits = e8TBT_DISTANCE_UNITS_YARDS;
      break;
      case mplay_tFiDistanceDisplayUnits::FI_EN_E8DIPO_DISTANCE_DISPLAY_UNIT_FT:
          enTBTDistanceUnits = e8TBT_DISTANCE_UNITS_FEET;
      break;
      default:
          enTBTDistanceUnits = e8TBT_DISTANCE_UNITS_UNKNOWN;
          ETG_TRACE_ERR(("[ERR]::enGetTBTDistanceUnits unsupported distance unit!!"));
      break;
   }

   ETG_TRACE_USR1(("enGetTBTDistanceUnits() left with %d ", ETG_ENUM(TBT_DISTANCE_UNITS, enTBTDistanceUnits)));
   return enTBTDistanceUnits;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vTriggerNowPlayingStatus
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vTriggerNowPlayingStatus() const
{
   ETG_TRACE_USR1((" spi_tclMPlayClientHandler::vTriggerNowPlayingStatus entered "));

   trMsgContext rMsgContext;
   vPopulateMessageContext(rMsgContext);
   FIMsgDispatch msgDispatcher(m_poMainApp);
   spi_tMsgDiPoNowPlayingGet oDiPoNowPlayingGet;
   t_Bool bRetVal = msgDispatcher.bSendMessage(oDiPoNowPlayingGet, rMsgContext, DIPO_MPLY_MAJORVERSION);

   ETG_TRACE_USR1((" spi_tclMPlayClientHandler::vTriggerNowPlayingStatus left with bRetVal: %d", bRetVal));
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclMPlayClientHandler::vOnDipoCommErrorStatus(amt_tclServiceData ..
***************************************************************************/
t_Void spi_tclMPlayClientHandler::vOnDipoCommErrorStatus(amt_tclServiceData* poMessage) const
{
   ETG_TRACE_USR1((" spi_tclMPlayClientHandler::vOnDipoCommErrorStatus entered "));
   spi_tMsgDiPOCommErrorStatus oDiPOCommErrorStatus(*poMessage, DIPO_MPLY_MAJORVERSION);
   if(oDiPOCommErrorStatus.bIsValid())
   {
       t_String szDeviceSerialNo;
       GET_STRINGDATA_FROM_FI_STRINGOBJ(oDiPOCommErrorStatus.sDeviceUSBSerialNumber,
       MPLAYFI_CHAR_SET_UTF8, szDeviceSerialNo);
       spi_tclDeviceIDDataIntf oDeviceDataIntf;
       t_U32 u32DeviceId = oDeviceDataIntf.u32GetDeviceIDFromSerialNo(szDeviceSerialNo);
       ETG_TRACE_USR4((" spi_tclMPlayClientHandler::vOnDipoCommErrorStatus  Received Error code %d", oDiPOCommErrorStatus.e8DiPOCommunicationError.enType));
       if((mplay_fi_tcl_e8_DiPOCommunicationError::FI_EN_E8IPOD_ROLESWITCH_ERROR == oDiPOCommErrorStatus.e8DiPOCommunicationError.enType)
                || (mplay_fi_tcl_e8_DiPOCommunicationError::FI_EN_E8IPOD_COMM_ERROR == oDiPOCommErrorStatus.e8DiPOCommunicationError.enType))
       {
         if(NULL !=  m_rDeviceSwitcherCbs.fvDeviceSwitchCompleteCb)
         {
            (m_rDeviceSwitcherCbs.fvDeviceSwitchCompleteCb)(u32DeviceId, false);
         }
       }
    }
}
/***************************************************************************
** FUNCTION:  t_Bool spi_tclMPlayClientHandler::vOnDipoGPRMCDataStatusValuesStatus(amt_tclServiceData ..
***************************************************************************/
t_Void spi_tclMPlayClientHandler::vOnDipoGPRMCDataStatusValuesStatus(amt_tclServiceData* poMessage) const
{
   ETG_TRACE_USR1(("spi_tclMPlayClientHandler::vOnDipoGPRMCDataStatusValuesStatus entered"));
   spi_tMsgDiPOGPRMCDataStatusValuesStatus oDiPOGPRMCDataStatusValuesStatus(*poMessage, DIPO_MPLY_MAJORVERSION);
   if(oDiPOGPRMCDataStatusValuesStatus.bIsValid())
   {
      t_U8 u8GPRMCDataStatusValues = static_cast<t_U8>(oDiPOGPRMCDataStatusValuesStatus.b8GPRMCDataStatusValues.u8Value);
      t_U32 u32DeviceId = static_cast<t_U32>(oDiPOGPRMCDataStatusValuesStatus.u8DeviceId);

      ETG_TRACE_USR1(("[PARAM]::spi_tclMPlayClientHandler::vOnDipoGPRMCDataStatusValuesStatus GPRMCData Status Values=%d, Device id = %d",
               u8GPRMCDataStatusValues, u32DeviceId));

      if (NULL != m_rDipoLocationInfoCallbacks.fvOnDipoGPRMCDataStatusValues)
      {
         (m_rDipoLocationInfoCallbacks.fvOnDipoGPRMCDataStatusValues)(u32DeviceId, u8GPRMCDataStatusValues);
      }
   }
   else
   {
      ETG_TRACE_ERR(("[ERR]:spi_tclMPlayClientHandler::vOnDipoGPRMCDataStatusValuesStatus : Message extraction failed. "));
   }
   oDiPOGPRMCDataStatusValuesStatus.vDestroy();
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclMPlayClientHandler::vRegisterCallbacks()
***************************************************************************/
t_Void spi_tclMPlayClientHandler::vRegisterCallbacks(const trMediaDeviceInfoCbs& corMediaDeviceInfoCbs)
{
   ETG_TRACE_USR1(("spi_tclMPlayClientHandler::vRegisterCallbacks(const trMediaDeviceInfoCbs& corMediaDeviceInfoCbs)"));
   if(m_poMplayDevList)
   {
      m_poMplayDevList->vRegisterCallbacks(corMediaDeviceInfoCbs);
   }
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclMPlayClientHandler::vSetCPWEnabledStatus()
***************************************************************************/
t_Void spi_tclMPlayClientHandler::vSetFeatureSupport(const trSpiFeatureSupport& corfrSpiFeatureSupport)
{
   if(m_poMplayDevList)
   {
      m_poMplayDevList->vSetFeatureSupport(corfrSpiFeatureSupport);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vRestoreSettings()
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vRestoreSettings()
{
   ETG_TRACE_USR1(("[FUNC]spi_tclMPlayClientHandler::vRestoreSettings() entered "));

   if (NULL != m_poMplayDevList)
   {
      m_poMplayDevList->vRestoreSettings();
   }//if (nullptr != m_pSpiCmdIntf)
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::bOnRegRouteGuidanceDisplayConf
 ***************************************************************************/
t_Bool spi_tclMPlayClientHandler::bOnRegRouteGuidanceDisplayConf(
         const std::vector<trDiPORouteGuidanceDisplayComponent>& corfvecRouteGuidanceDisplayComponent)
{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));
   tBool bResult = false;

   if (nullptr != m_poMainApp)
   {
      spi_tMsgDiPOSetRouteGuidanceDispComponentInfoMS oMPlayDiPOSetRouteGuidanceDisplayComponentInfo;
      mplay_fi_tcl_RouteGuidanceDisplayComponentItem oRouteGuidanceDisplayComponentItem;

      ETG_TRACE_USR4(("[DESC]: spi_tclMPlayClientHandler::bOnRegRouteGuidanceDisplayConf - Number of components = %d", corfvecRouteGuidanceDisplayComponent.size()));
      std::vector<trDiPORouteGuidanceDisplayComponent>::const_iterator itrRGDisplayCompList;
      tU16 u16Index = 0;
      for (itrRGDisplayCompList = corfvecRouteGuidanceDisplayComponent.begin(); itrRGDisplayCompList != corfvecRouteGuidanceDisplayComponent.end(); itrRGDisplayCompList++)
      {
         ETG_TRACE_USR4(("[DESC]: spi_tclMPlayClientHandler::bOnRegRouteGuidanceDisplayConf - Component No. %d", (u16Index++)));
         oRouteGuidanceDisplayComponentItem.Identifier =
                  static_cast<tU16>(itrRGDisplayCompList->enDisplayComponent);
         ETG_TRACE_USR4(("[DESC]: spi_tclMPlayClientHandler::bOnRegRouteGuidanceDisplayConf - Display Component ID = %d", ETG_ENUM(DIPO_COMPONENT_DISPLAYID, oRouteGuidanceDisplayComponentItem.Identifier)));
         oRouteGuidanceDisplayComponentItem.Name.bSet(itrRGDisplayCompList->szDisplayName.c_str(),
                  MPLAYFI_CHAR_SET_UTF8);
         ETG_TRACE_USR4(("[DESC]: spi_tclMPlayClientHandler::bOnRegRouteGuidanceDisplayConf - Display name = %s", oRouteGuidanceDisplayComponentItem.Name.szValue));
         oRouteGuidanceDisplayComponentItem.MaxCurrentRoadNameLength =
                  static_cast<tU16>(itrRGDisplayCompList->u16MaxCurrentRoadNameLength);
         ETG_TRACE_USR4(("[DESC]: spi_tclMPlayClientHandler::bOnRegRouteGuidanceDisplayConf - Max Current road name length = %d", oRouteGuidanceDisplayComponentItem.MaxCurrentRoadNameLength));
         oRouteGuidanceDisplayComponentItem.MaxDestinationNameLength =
                  static_cast<tU16>(itrRGDisplayCompList->u16MaxDestinationNameLength);
         ETG_TRACE_USR4(("[DESC]: spi_tclMPlayClientHandler::bOnRegRouteGuidanceDisplayConf - Max Destination name length = %d", oRouteGuidanceDisplayComponentItem.MaxDestinationNameLength));
         oRouteGuidanceDisplayComponentItem.MaxAfterManeuverRoadNameLength =
                  static_cast<tU16>(itrRGDisplayCompList->u16MaxAfterManeuverRoadNameLength);
         ETG_TRACE_USR4(("[DESC]: spi_tclMPlayClientHandler::bOnRegRouteGuidanceDisplayConf - Max After Maneuver road name length = %d", oRouteGuidanceDisplayComponentItem.MaxAfterManeuverRoadNameLength));
         oRouteGuidanceDisplayComponentItem.MaxManeuverDescriptionLength =
                  static_cast<tU16>(itrRGDisplayCompList->u16MaxManeuverDescriptionLength);
         ETG_TRACE_USR4(("[DESC]: spi_tclMPlayClientHandler::bOnRegRouteGuidanceDisplayConf - Max maneuver description length = %d", oRouteGuidanceDisplayComponentItem.MaxManeuverDescriptionLength));
         oRouteGuidanceDisplayComponentItem.MaxGuidanceManeuverCapacity =
                  static_cast<tU16>(itrRGDisplayCompList->u16MaxGuidanceManeuverCapacity);
         ETG_TRACE_USR4(("[DESC]: spi_tclMPlayClientHandler::bOnRegRouteGuidanceDisplayConf - Max guidance maneuver capacity = %d", oRouteGuidanceDisplayComponentItem.MaxGuidanceManeuverCapacity));

         oMPlayDiPOSetRouteGuidanceDisplayComponentInfo.RouteGuidanceDisplayComponents.oItems.push_back(oRouteGuidanceDisplayComponentItem);
      }

      // populate message context
      trMsgContext rMsgContext;
      vPopulateMessageContext(rMsgContext);
      FIMsgDispatch msgDispatcher(m_poMainApp);

      if (true == m_bIsMPServiceAvailable)
      {
         //send method DiPOSetRouteGuidanceDisplayComponentInfo to Mediaplayer
         bResult = msgDispatcher.bSendMessage(oMPlayDiPOSetRouteGuidanceDisplayComponentInfo,
                  rMsgContext,
                  DIPO_MPLY_MAJORVERSION);
      }
      else
      {
         // Push the registration request into a queue and send the request to MediaPlayer once the service becomes available.
         m_queueRouteGuidanceComponentRegLock.s16Lock();
         m_queueRouteGuidanceComponentReg.push(oMPlayDiPOSetRouteGuidanceDisplayComponentInfo);
         m_queueRouteGuidanceComponentRegLock.vUnlock();
         ETG_TRACE_USR2(("[DESC]: bOnRegRouteGuidanceDisplayConf - MediaPlayer service unavailable. RouteGuidanceComponentRegistration pending until service is available"));
      }
   }
   return bResult;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::bOnStartRouteGuidanceUpdates
 ***************************************************************************/
t_Bool spi_tclMPlayClientHandler::bOnStartRouteGuidanceUpdates(
         trStartRouteGuidanceUpdates& corfStartRouteGuidanceUpdates)
{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));
   tBool bResult = false;

   if (nullptr != m_poMainApp)
   {
      mplay_appcontrolfi_tclMsgDiPOStartRouteGuidance2MethodStart oMPlayStartRouteGuidance;

      std::vector<tenDisplayComponentId>::const_iterator itrDisplayComponentList;
      for (itrDisplayComponentList = corfStartRouteGuidanceUpdates.vecDisplayComponent.begin(); itrDisplayComponentList != corfStartRouteGuidanceUpdates.vecDisplayComponent.end(); itrDisplayComponentList++)
      {
         oMPlayStartRouteGuidance.RouteGuidanceDisplayComponentIDs.Items.push_back(static_cast<tU16>(*itrDisplayComponentList));
         ETG_TRACE_USR4(("[DESC]: spi_tclMPlayClientHandler::bOnStartRouteGuidanceUpdates - Display component ID = %d", ETG_ENUM(DIPO_COMPONENT_DISPLAYID, *itrDisplayComponentList)));
      }
      oMPlayStartRouteGuidance.sDeviceUSBSerialNumber.bSet(corfStartRouteGuidanceUpdates.szDeviceSerialNumber.c_str(),
               MPLAYFI_CHAR_SET_UTF8);
      ETG_TRACE_USR4(("[DESC]: spi_tclMPlayClientHandler::bOnStartRouteGuidanceUpdates - Device USB Serial Number = %s", oMPlayStartRouteGuidance.sDeviceUSBSerialNumber.szValue));
      oMPlayStartRouteGuidance.SourceName = static_cast<tBool>(corfStartRouteGuidanceUpdates.bSourceName);
      ETG_TRACE_USR4(("[DESC]: spi_tclMPlayClientHandler::bOnStartRouteGuidanceUpdates - Source name = %d", ETG_ENUM(BOOL, oMPlayStartRouteGuidance.SourceName)));
      oMPlayStartRouteGuidance.SourceSupportsRouteGuidance =
               static_cast<tBool>(corfStartRouteGuidanceUpdates.bSourceSupportsRouteGuidance);
      ETG_TRACE_USR4(("[DESC]: spi_tclMPlayClientHandler::bOnStartRouteGuidanceUpdates - Source supports route guidance = %d",
                        ETG_ENUM(BOOL,oMPlayStartRouteGuidance.SourceSupportsRouteGuidance)));

      // populate message context
      trMsgContext rMsgContext;
      vPopulateMessageContext(rMsgContext);
      FIMsgDispatch msgDispatcher(m_poMainApp);

      // send method DiPOStartRouteGuidance to mplay fi
      bResult = msgDispatcher.bSendMessage(oMPlayStartRouteGuidance, rMsgContext, DIPO_MPLY_MAJORVERSION);
   }
   return bResult;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::bOnStartRouteGuidanceUpdates
 ***************************************************************************/
t_Bool spi_tclMPlayClientHandler::bOnStopRouteGuidanceUpdates(trStopRouteGuidanceUpdates& corfStopRouteGuidanceUpdates)
{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));
   tBool bResult = false;

   if (nullptr != m_poMainApp)
   {
      mplay_appcontrolfi_tclMsgDiPOStopRouteGuidance2MethodStart oMPlayStopRouteGuidance;
      oMPlayStopRouteGuidance.sDeviceUSBSerialNumber.bSet(corfStopRouteGuidanceUpdates.szDeviceSerialNumber.c_str(),
               MPLAYFI_CHAR_SET_UTF8);
      ETG_TRACE_USR4(("[DESC]: spi_tclMPlayClientHandler::bOnStopRouteGuidanceUpdates - Device USB Serial Number = %s", oMPlayStopRouteGuidance.sDeviceUSBSerialNumber.szValue));

      std::vector<tenDisplayComponentId>::const_iterator itrDisplayComponentList;
      for (itrDisplayComponentList = corfStopRouteGuidanceUpdates.vecDisplayComponent.begin(); itrDisplayComponentList != corfStopRouteGuidanceUpdates.vecDisplayComponent.end(); itrDisplayComponentList++)
      {
         oMPlayStopRouteGuidance.RouteGuidanceDisplayComponentIDs.Items.push_back(static_cast<tU16>(*itrDisplayComponentList));
         ETG_TRACE_USR4(("[DESC]: spi_tclMPlayClientHandler::bOnStopRouteGuidanceUpdates - Display component ID = %d", ETG_ENUM(DIPO_COMPONENT_DISPLAYID, *(itrDisplayComponentList))));
      }

         // populate message context
      trMsgContext rMsgContext;
      vPopulateMessageContext(rMsgContext);
      FIMsgDispatch msgDispatcher(m_poMainApp);

      // send method StopRouteGuidanceUpdates to mplay fi
      bResult = msgDispatcher.bSendMessage(oMPlayStopRouteGuidance, rMsgContext, DIPO_MPLY_MAJORVERSION);
   }
   return bResult;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::vOnDipoRouteGuidanceStatus2()
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vOnDipoRouteGuidanceStatus2(amt_tclServiceData* poMessage) const
{
   ETG_TRACE_USR1(("spi_tclMPlayClientHandler::vOnDipoRouteGuidanceStatus2 entered"));
   trDiPORouteGuidanceUpdate rDiPORouteGuidanceUpdate;
   spi_tMsgCarPlayRouteGuidanceStatus oDiPORouteGuidanceStatus(*poMessage, DIPO_MPLY_MAJORVERSION);
   if (oDiPORouteGuidanceStatus.bIsValid())
   {
      rDiPORouteGuidanceUpdate.szDeviceSerialNumber = t_String(oDiPORouteGuidanceStatus.sDeviceUSBSerialNumber.szValue);

      ETG_TRACE_USR4(("[DESC]:vOnDipoRouteGuidanceStatus2 - Is DisplayComponentID list available = %d", ETG_ENUM(BOOL, oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bRouteGuidanceDisplayComponentIDsAvailable)));
      rDiPORouteGuidanceUpdate.bIsDisplayComponentListAvailable =
               oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bRouteGuidanceDisplayComponentIDsAvailable;

      std::vector<tU16, std::allocator<tU16> >::const_iterator itrRGDisplayComponentIDList;
      for (itrRGDisplayComponentIDList = oDiPORouteGuidanceStatus.RouteGuidanceUpdate.RouteGuidanceDisplayComponentIDs.Items.begin();
               itrRGDisplayComponentIDList != oDiPORouteGuidanceStatus.RouteGuidanceUpdate.RouteGuidanceDisplayComponentIDs.Items.end();
               itrRGDisplayComponentIDList++)
      {
         rDiPORouteGuidanceUpdate.vecDisplayComponentList.push_back(static_cast<tenDisplayComponentId>(*itrRGDisplayComponentIDList));
         ETG_TRACE_USR4(("[DESC]:vOnDipoRouteGuidanceStatus2 - Display component ID = %d", ETG_ENUM(DIPO_COMPONENT_DISPLAYID, *itrRGDisplayComponentIDList)));
      }

      ETG_TRACE_USR4(("[DESC]:vOnDipoRouteGuidanceStatus2 - Is RouteGuidanceState available = %d, RouteGuidanceState = %d", 
        ETG_ENUM(BOOL, oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bRouteGuidanceStateAvailable), ETG_ENUM(DIPO_ROUTE_GUIDANCE_STATE,oDiPORouteGuidanceStatus.RouteGuidanceUpdate.e8RouteGuidanceState.enType)));
      rDiPORouteGuidanceUpdate.bIsRouteGuidanceStateAvailable = oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bRouteGuidanceStateAvailable;
      rDiPORouteGuidanceUpdate.enRouteGuidanceState = static_cast<tenDiPoRouteGuidanceState>(oDiPORouteGuidanceStatus.RouteGuidanceUpdate.e8RouteGuidanceState.enType);

      ETG_TRACE_USR4(("[DESC]:vOnDipoRouteGuidanceStatus2 - Is ManeuverState available = %d, ManeuverState = %d", 
        ETG_ENUM(BOOL, oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bManeuverStateAvailable), ETG_ENUM(DIPO_MANEUVER_STATE,oDiPORouteGuidanceStatus.RouteGuidanceUpdate.e8ManeuverState.enType)));
      rDiPORouteGuidanceUpdate.bIsManeuverStateAvailable = oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bManeuverStateAvailable;
      rDiPORouteGuidanceUpdate.enManeuverState = static_cast<tenDiPoRouteManeuverState>(oDiPORouteGuidanceStatus.RouteGuidanceUpdate.e8ManeuverState.enType);

      ETG_TRACE_USR4(("[DESC]:vOnDipoRouteGuidanceStatus2 - Is CurrentRoadName available = %d, Current Road Name = %s", 
        ETG_ENUM(BOOL, oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bCurrentRoadNameAvailable), oDiPORouteGuidanceStatus.RouteGuidanceUpdate.CurrentRoadName.szValue));
      rDiPORouteGuidanceUpdate.bIsCurrentRoadNameAvailable = oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bCurrentRoadNameAvailable;
      rDiPORouteGuidanceUpdate.szCurrentRoadName = t_String(oDiPORouteGuidanceStatus.RouteGuidanceUpdate.CurrentRoadName.szValue);

      ETG_TRACE_USR4(("[DESC]:vOnDipoRouteGuidanceStatus2 - Is DestinationName available = %d, DestinationName = %s", 
        ETG_ENUM(BOOL, oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bDestinationNameAvailable), oDiPORouteGuidanceStatus.RouteGuidanceUpdate.DestinationName.szValue));
      rDiPORouteGuidanceUpdate.bIsDestinationNameAvailable = oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bDestinationNameAvailable;
      rDiPORouteGuidanceUpdate.szDestinationName = t_String(oDiPORouteGuidanceStatus.RouteGuidanceUpdate.DestinationName.szValue);

      ETG_TRACE_USR4(("[DESC]:vOnDipoRouteGuidanceStatus2 - Is EstimatedTimeOfArrival available = %d, EstimatedTimeOfArrival = %s",
        ETG_ENUM(BOOL, oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bEstimatedTimeOfArrivalAvailable), std::to_string(oDiPORouteGuidanceStatus.RouteGuidanceUpdate.EstimatedTimeOfArrival).c_str()));
      rDiPORouteGuidanceUpdate.bIsEstimatedTimeOfArrivalAvailable = oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bEstimatedTimeOfArrivalAvailable;
      rDiPORouteGuidanceUpdate.u64EstimatedTimeOfArrival = oDiPORouteGuidanceStatus.RouteGuidanceUpdate.EstimatedTimeOfArrival;

      ETG_TRACE_USR4(("[DESC]:vOnDipoRouteGuidanceStatus2 - Is TimeRemainingToDestination available = %d, TimeRemainingToDestination=%s",
        ETG_ENUM(BOOL, oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bTimeRemainingToDestinationAvailable), std::to_string(oDiPORouteGuidanceStatus.RouteGuidanceUpdate.TimeRemainingToDestination).c_str()));
      rDiPORouteGuidanceUpdate.bIsTimeRemainingToDestAvailable = oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bTimeRemainingToDestinationAvailable;
      rDiPORouteGuidanceUpdate.u64TimeRemainingToDestination = oDiPORouteGuidanceStatus.RouteGuidanceUpdate.TimeRemainingToDestination;

      ETG_TRACE_USR4(("[DESC]:vOnDipoRouteGuidanceStatus2 - Is DistanceRemaining available = %d, DistanceRemaining = %d",
        ETG_ENUM(BOOL, oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bDistanceRemainingAvailable), oDiPORouteGuidanceStatus.RouteGuidanceUpdate.DistanceRemaining));
      rDiPORouteGuidanceUpdate.bIsDistanceRemainingAvailable = oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bDistanceRemainingAvailable;
      rDiPORouteGuidanceUpdate.u32DistanceRemaining = oDiPORouteGuidanceStatus.RouteGuidanceUpdate.DistanceRemaining;

      ETG_TRACE_USR4(("[DESC]:vOnDipoRouteGuidanceStatus2 - Is DistanceRemainingDisplayStr available = %d, DistanceRemaining DisplayStr = %s", 
        ETG_ENUM(BOOL, oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bDistanceRemainingDisplayStrAvailable), oDiPORouteGuidanceStatus.RouteGuidanceUpdate.DistanceRemainingDisplayStr.szValue));
      rDiPORouteGuidanceUpdate.bIsDistanceRemainingDisplayStrAvailable = oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bDistanceRemainingDisplayStrAvailable;
      rDiPORouteGuidanceUpdate.szDistanceRemainingDisplayStr = t_String(oDiPORouteGuidanceStatus.RouteGuidanceUpdate.DistanceRemainingDisplayStr.szValue);

      ETG_TRACE_USR4(("[DESC]:vOnDipoRouteGuidanceStatus2 - Is DistanceRemainingDisplayUnits available = %d, DistanceRemaining DisplayUnits  = %d", 
        ETG_ENUM(BOOL, oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bDistanceRemainingDisplayUnitsAvailable), ETG_ENUM(TBT_DISTANCE_UNITS,oDiPORouteGuidanceStatus.RouteGuidanceUpdate.e8DistanceRemainingDisplayUnits.enType)));
      rDiPORouteGuidanceUpdate.bIsDistanceRemainingDisplayUnitsAvailable = oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bDistanceRemainingDisplayUnitsAvailable;
      rDiPORouteGuidanceUpdate.enDistanceRemainingDisplayUnits = enGetTBTDistanceUnits(oDiPORouteGuidanceStatus.RouteGuidanceUpdate.e8DistanceRemainingDisplayUnits);

      ETG_TRACE_USR4(("[DESC]:vOnDipoRouteGuidanceStatus2 - Is DistanceToNextManeuver available = %d, Distance to Next Maneuver = %d", 
        ETG_ENUM(BOOL, oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bDistanceToNextManeuverAvailable), oDiPORouteGuidanceStatus.RouteGuidanceUpdate.DistanceToNextManeuver));
      rDiPORouteGuidanceUpdate.bIsDistanceToNextManeuverAvailable = oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bDistanceToNextManeuverAvailable;
      rDiPORouteGuidanceUpdate.u32DistanceToNextManeuver = oDiPORouteGuidanceStatus.RouteGuidanceUpdate.DistanceToNextManeuver;

      ETG_TRACE_USR4(("[DESC]:vOnDipoRouteGuidanceStatus2 - Is DistanceToNextManeuverDisplayStr available = %d, DistanceToNextManeuverDisplayStr = %s", 
        ETG_ENUM(BOOL, oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bDistanceToNextManeuverDisplayStrAvailable), oDiPORouteGuidanceStatus.RouteGuidanceUpdate.DistanceToNextManeuverDisplayStr.szValue));
      rDiPORouteGuidanceUpdate.bIsDistanceToNextManeuverDispStrAvailable = oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bDistanceToNextManeuverDisplayStrAvailable;
      rDiPORouteGuidanceUpdate.szDistanceToNextManeuverDisplayStr = t_String(oDiPORouteGuidanceStatus.RouteGuidanceUpdate.DistanceToNextManeuverDisplayStr.szValue);

      ETG_TRACE_USR4(("[DESC]:vOnDipoRouteGuidanceStatus2 - Is DistanceToNextManeuverDisplayUnits available = %d, DistanceToNextManeuverDisplayUnits = %d", 
        ETG_ENUM(BOOL, oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bDistanceToNextManeuverDisplayUnitsAvailable), ETG_ENUM(TBT_DISTANCE_UNITS,oDiPORouteGuidanceStatus.RouteGuidanceUpdate.e8DistanceToNextManeuverDisplayUnits.enType)));
      rDiPORouteGuidanceUpdate.bIsDistanceToNextManeuverDIspUnitsAvailable = oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bDistanceToNextManeuverDisplayUnitsAvailable;
      rDiPORouteGuidanceUpdate.enDistanceToNextManeuverDisplayUnits = enGetTBTDistanceUnits(oDiPORouteGuidanceStatus.RouteGuidanceUpdate.e8DistanceToNextManeuverDisplayUnits);

      ETG_TRACE_USR4(("[DESC]:vOnDipoRouteGuidanceStatus2 - Is RouteGuidanceManeuverCurrentList available = %d", ETG_ENUM(BOOL, oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bRouteGuidanceManeuverCurrentListAvailable)));
      rDiPORouteGuidanceUpdate.bIsRouteGuidanceManeuverCurrentListAvailable = oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bRouteGuidanceManeuverCurrentListAvailable;

      std::vector<tU16, std::allocator<tU16> >::const_iterator itrRGManeuverCurrentList;
      for (itrRGManeuverCurrentList = oDiPORouteGuidanceStatus.RouteGuidanceUpdate.RouteGuidanceManeuverCurrentList.u16Items.begin();
               itrRGManeuverCurrentList != oDiPORouteGuidanceStatus.RouteGuidanceUpdate.RouteGuidanceManeuverCurrentList.u16Items.end();
               itrRGManeuverCurrentList++)
      {
         rDiPORouteGuidanceUpdate.vecRouteGuidanceManeuverCurrentList.push_back(*itrRGManeuverCurrentList);
      }

      ETG_TRACE_USR4(("[DESC]:vOnDipoRouteGuidanceStatus2 - Is RouteGuidanceManeuverCount available = %d, RouteGuidanceManeuverCount = %d", 
        ETG_ENUM(BOOL, oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bRouteGuidanceManeuverCountAvailable), oDiPORouteGuidanceStatus.RouteGuidanceUpdate.RouteGuidanceManeuverCount));
      rDiPORouteGuidanceUpdate.bIsRouteGuidanceManeuverCountAvailable = oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bRouteGuidanceManeuverCountAvailable;
      rDiPORouteGuidanceUpdate.u16RouteGuidanceManeuverCount = oDiPORouteGuidanceStatus.RouteGuidanceUpdate.RouteGuidanceManeuverCount;

      ETG_TRACE_USR4(("[DESC]:vOnDipoRouteGuidanceStatus2 - Is RouteGuidanceVisibleInApp available = %d, RouteGuidanceVisibleInApp = %d", 
        ETG_ENUM(BOOL, oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bRouteGuidanceVisibleInAppAvailable), ETG_ENUM(BOOL,oDiPORouteGuidanceStatus.RouteGuidanceUpdate.RouteGuidanceVisibleInApp)));
      rDiPORouteGuidanceUpdate.bIsRouteGuidanceVisibleInAppAvailable = oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bRouteGuidanceVisibleInAppAvailable;
      rDiPORouteGuidanceUpdate.bIsRouteGuidanceVisibleInApp = oDiPORouteGuidanceStatus.RouteGuidanceUpdate.RouteGuidanceVisibleInApp;

      ETG_TRACE_USR4(("[DESC]:vOnDipoRouteGuidanceStatus2 - Is SourceName available = %d, Source Name = %s", 
        ETG_ENUM(BOOL, oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bSourceNameAvailable), oDiPORouteGuidanceStatus.RouteGuidanceUpdate.SourceName.szValue));
      rDiPORouteGuidanceUpdate.bIsSourceNameAvailable = oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bSourceNameAvailable;
      rDiPORouteGuidanceUpdate.szSourceName = t_String(oDiPORouteGuidanceStatus.RouteGuidanceUpdate.SourceName.szValue);

      ETG_TRACE_USR4(("[DESC]:vOnDipoRouteGuidanceStatus2 - Is SourceSupportsRouteGuidance available = %d, SourceSupportsRouteGuidance = %d", 
        ETG_ENUM(BOOL, oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bSourceSupportsRouteGuidanceAvailable), ETG_ENUM(BOOL, oDiPORouteGuidanceStatus.RouteGuidanceUpdate.SourceSupportsRouteGuidance)));
      rDiPORouteGuidanceUpdate.bIsSourceSupportsRouteGuidanceAvailable = oDiPORouteGuidanceStatus.RouteGuidanceUpdate.bSourceSupportsRouteGuidanceAvailable;
      rDiPORouteGuidanceUpdate.bIsSourceSupportsRouteGuidance = oDiPORouteGuidanceStatus.RouteGuidanceUpdate.SourceSupportsRouteGuidance;
   }
   if (NULL != m_rTBTCallbacks.fvOnDiPORouteGuidanceUpdateCallback)
   {
      m_rTBTCallbacks.fvOnDiPORouteGuidanceUpdateCallback(rDiPORouteGuidanceUpdate);
   }
} //end of vOnDipoRouteGuidanceStatus2()

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::vOnDipoRouteGuidanceManeuverStatus2()
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vOnDipoRouteGuidanceManeuverStatus2(amt_tclServiceData* poMessage) const
{
   ETG_TRACE_USR1(("spi_tclMPlayClientHandler::vOnDipoRouteGuidanceManeuverStatus2 entered"));
   trDiPORouteGuidanceManeuverUpdate rDiPORouteGuidanceManeuverUpdate;

   spi_tMsgCarPlayRouteGuidanceManeuverStatus oDiPORouteGuidanceManeuverStatus(*poMessage, DIPO_MPLY_MAJORVERSION);
   if (oDiPORouteGuidanceManeuverStatus.bIsValid())
   {
      rDiPORouteGuidanceManeuverUpdate.szDeviceSerialNumber = t_String(oDiPORouteGuidanceManeuverStatus.sDeviceUSBSerialNumber.szValue);

      ETG_TRACE_USR4(("[DESC]:vOnDipoRouteGuidanceManeuverStatus2 - Is DisplayComponentID list available = %d", ETG_ENUM(BOOL, oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.bRouteGuidanceDisplayComponentIDsAvailable)));
      rDiPORouteGuidanceManeuverUpdate.bIsDisplayComponentListAvailable =
               oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.bRouteGuidanceDisplayComponentIDsAvailable;
      std::vector<tU16, std::allocator<tU16> >::const_iterator itrRGDisplayComponentIDList;
      for (itrRGDisplayComponentIDList = oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.RouteGuidanceDisplayComponentIDs.Items.begin();
               itrRGDisplayComponentIDList != oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.RouteGuidanceDisplayComponentIDs.Items.end();
               itrRGDisplayComponentIDList++)
      {
         rDiPORouteGuidanceManeuverUpdate.vecDisplayComponentList.push_back(static_cast<tenDisplayComponentId>(*itrRGDisplayComponentIDList));
         ETG_TRACE_USR4(("[DESC]:vOnDipoRouteGuidanceManeuverStatus2 - Display component ID = %d", ETG_ENUM(DIPO_COMPONENT_DISPLAYID, *itrRGDisplayComponentIDList)));
      }

      ETG_TRACE_USR4(("[DESC]:vOnDipoRouteGuidanceManeuverStatus2 - Is Index available = %d, Index = %d", 
        ETG_ENUM(BOOL, oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.bIndexAvailable), oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.Index));
      rDiPORouteGuidanceManeuverUpdate.bIsIndexAvailable = oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.bIndexAvailable;
      rDiPORouteGuidanceManeuverUpdate.u16Index = oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.Index;

      ETG_TRACE_USR4(("[DESC]:vOnDipoRouteGuidanceManeuverStatus2 - Is ManeuverDescription available = %d, ManeuverDescription = %s", 
        ETG_ENUM(BOOL, oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.bManeuverDescriptionAvailable), oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.ManeuverDescription.szValue));
      rDiPORouteGuidanceManeuverUpdate.bIsManeuverDescriptionAvailable = oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.bManeuverDescriptionAvailable;
      rDiPORouteGuidanceManeuverUpdate.szManeuverDescription = t_String(oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.ManeuverDescription.szValue);

      ETG_TRACE_USR4(("[DESC]:vOnDipoRouteGuidanceManeuverStatus2 - Is ManeuverType available = %d, ManeuverType = %d", 
        ETG_ENUM(BOOL, oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.bManeuverTypeAvailable), oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.e8ManeuverType.enType));
      rDiPORouteGuidanceManeuverUpdate.bIsManeuverTypeAvailable = oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.bManeuverTypeAvailable;
      rDiPORouteGuidanceManeuverUpdate.enManeuverType = static_cast<tenDiPoManeuverType>(oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.e8ManeuverType.enType);

      ETG_TRACE_USR4(("[DESC]:vOnDipoRouteGuidanceManeuverStatus2 - Is AfterManeuverRoadname available = %d, AfterManeuverRoadName = %s", 
        ETG_ENUM(BOOL, oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.bAfterManeuverRoadNameAvailable), oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.AfterManeuverRoadName.szValue));
      rDiPORouteGuidanceManeuverUpdate.bIsAfterManeuverRoadNameAvailable = oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.bAfterManeuverRoadNameAvailable;
      rDiPORouteGuidanceManeuverUpdate.szAfterManeuverRoadName = t_String(oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.AfterManeuverRoadName.szValue);

      ETG_TRACE_USR4(("[DESC]:vOnDipoRouteGuidanceManeuverStatus2 - Is DistanceBetweenManeuver available = %d, Distance Between Maneuver = %d", 
        ETG_ENUM(BOOL, oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.bDistanceBetweenManeuverAvailable), oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.DistanceBetweenManeuver));
      rDiPORouteGuidanceManeuverUpdate.bIsDistanceBetweenManeuverAvailable = oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.bDistanceBetweenManeuverAvailable;
      rDiPORouteGuidanceManeuverUpdate.u32DistanceBetweenManeuver = oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.DistanceBetweenManeuver;

      ETG_TRACE_USR4(("[DESC]:vOnDipoRouteGuidanceManeuverStatus2 - Is DistanceBetweenManeuverDisplayStr available = %d, Distance Between Maneuver DisplayStr = %s", 
        ETG_ENUM(BOOL, oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.bDistanceBetweenManeuverDisplayStrAvailable), oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.DistanceBetweenManeuverDisplayStr.szValue));
      rDiPORouteGuidanceManeuverUpdate.bIsDistanceBetweenManeuverDispStrAvailable = oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.bDistanceBetweenManeuverDisplayStrAvailable;
      rDiPORouteGuidanceManeuverUpdate.szDistanceBetweenManeuverDisplayStr = oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.DistanceBetweenManeuverDisplayStr.szValue;

      ETG_TRACE_USR4(("[DESC]:vOnDipoRouteGuidanceManeuverStatus2 - Is DistanceBetweenManeuverDisplayUnits available = %d, Distance Between Maneuver DisplayUnits = %d", 
        ETG_ENUM(BOOL, oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.bDistanceBetweenManeuverDisplayUnitsAvailable), ETG_ENUM(TBT_DISTANCE_UNITS,oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.e8DistanceBetweenManeuverDisplayUnits.enType)));
      rDiPORouteGuidanceManeuverUpdate.bIsDistanceBetweenManeuverDispUnitsAvailable = oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.bDistanceBetweenManeuverDisplayUnitsAvailable;
      rDiPORouteGuidanceManeuverUpdate.enDistanceBetweenManeuverDisplayUnits = enGetTBTDistanceUnits(oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.e8DistanceBetweenManeuverDisplayUnits);

      ETG_TRACE_USR4(("[DESC]:vOnDipoRouteGuidanceManeuverStatus2 - Is DrivingSide available = %d, DrivingSide = %d", 
        ETG_ENUM(BOOL, oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.bDrivingSideAvailable), ETG_ENUM(DIPO_DRIVING_SIDE,oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.e8DrivingSide.enType)));
      rDiPORouteGuidanceManeuverUpdate.bIsDrivingSideAvailable = oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.bDrivingSideAvailable;
      rDiPORouteGuidanceManeuverUpdate.enDrivingSide = static_cast<tenDiPoDrivingSide>(oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.e8DrivingSide.enType);

      ETG_TRACE_USR4(("[DESC]:vOnDipoRouteGuidanceManeuverStatus2 - Is JunctionType available = %d, JunctionType = %d", 
        ETG_ENUM(BOOL, oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.bJunctionTypeAvailable), ETG_ENUM(DIPO_JUNCTION_TYPE,oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.e8JunctionType.enType)));
      rDiPORouteGuidanceManeuverUpdate.bIsJunctionTypeAvailable = oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.bJunctionTypeAvailable;
      rDiPORouteGuidanceManeuverUpdate.enJunctionType = static_cast<tenDiPoJunctionType>(oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.e8JunctionType.enType);

      ETG_TRACE_USR4(("[DESC]:vOnDipoRouteGuidanceManeuverStatus2 - Is JunctionElementAngle available = %d", ETG_ENUM(BOOL, oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.bJunctionElementAngleAvailable)));
      rDiPORouteGuidanceManeuverUpdate.bIsJunctionElementAngleAvailable =
               oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.bJunctionElementAngleAvailable;

      std::vector<tS16, std::allocator<tS16> >::const_iterator itrJunctionElementAngleList;
      tU16 u16Index = 0;
      for (itrJunctionElementAngleList = oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.JunctionElementAngle.Items.begin();
            itrJunctionElementAngleList != oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.JunctionElementAngle.Items.end();
            itrJunctionElementAngleList++)
      {
         ETG_TRACE_USR4(("[DESC]:vOnDipoRouteGuidanceManeuverStatus2 - JunctionElementAngle[%d] = %d", u16Index++, *itrJunctionElementAngleList));
         rDiPORouteGuidanceManeuverUpdate.vecJunctionElementAngleList.push_back(*itrJunctionElementAngleList);
      }

      ETG_TRACE_USR4(("[DESC]:vOnDipoRouteGuidanceManeuverStatus2 - Is JunctionElementExitAngle available = %d, JunctionElementExitAngle = %d", 
        ETG_ENUM(BOOL, oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.bJunctionElementExitAngleAvailable), oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.JunctionElementExitAngle));
      rDiPORouteGuidanceManeuverUpdate.bIsJunctionElementExitAngleAvailable = oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.bJunctionElementExitAngleAvailable;
      rDiPORouteGuidanceManeuverUpdate.s16JunctionElementExitAngle = oDiPORouteGuidanceManeuverStatus.RouteGuidanceManeuverUpdate.JunctionElementExitAngle;
   }

   if (NULL != m_rTBTCallbacks.fvOnDiPORouteGuidanceManeuverUpdateCallback)
   {
      m_rTBTCallbacks.fvOnDiPORouteGuidanceManeuverUpdateCallback(rDiPORouteGuidanceManeuverUpdate);
   }
} //end of vOnDipoRouteGuidanceManeuverStatus2()

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::bOnRegRouteGuidanceDisplayConf
 ***************************************************************************/
t_Bool spi_tclMPlayClientHandler::bSetCallControlConfig(const std::set<tenKeyCode> &rsetKeyCodes)
{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));
   tBool bResult = false;
   SPI_INTENTIONALLY_UNUSED(rsetKeyCodes);
   if (nullptr != m_poMainApp)
   {
      spi_tMsgSetCallControlsConfigMS oMPlayDiPOSetCallControlsConfig;

      // Set AcceptCall and EndCall bits by default in the CallControls config
      for(auto enPhnKeys : rsetKeyCodes)
      {
         switch(enPhnKeys)
         {
             case e32DEV_PHONE_CALL :
                  oMPlayDiPOSetCallControlsConfig.callControlsConfiguration.vSetAcceptCall(true);
                  break;
             case e32DEV_PHONE_END  :
                  oMPlayDiPOSetCallControlsConfig.callControlsConfiguration.vSetEndCall(true);
                  break;
             case e32MERGE_CALLS    :
                  oMPlayDiPOSetCallControlsConfig.callControlsConfiguration.vSetMergeCalls(true);
                  break;
             case e32SWAP_CALLS     :
                  oMPlayDiPOSetCallControlsConfig.callControlsConfiguration.vSetSwapCalls(true);
                  break;
             default :
                  ETG_TRACE_USR4(("KeyCode Not Supported = %d",ETG_ENUM(SPI_KEY_CODE,enPhnKeys)));
                  break;
         }
       }

      ETG_TRACE_USR4(("[PARAM]: spi_tclMPlayClientHandler::bSetCallControlsConfig - AcceptCall set status = %d, endCall set status = %d",
      ETG_ENUM(BOOL, oMPlayDiPOSetCallControlsConfig.callControlsConfiguration.bAcceptCall()),
      ETG_ENUM(BOOL, oMPlayDiPOSetCallControlsConfig.callControlsConfiguration.bEndCall())));

      // populate message context
      trMsgContext rMsgContext;
      vPopulateMessageContext(rMsgContext);
      FIMsgDispatch msgDispatcher(m_poMainApp);

      //send method DiPOSetCallControlsConfiguration to Mediaplayer
      if (true == m_bIsMPServiceAvailable)
      {
         bResult = msgDispatcher.bSendMessage(oMPlayDiPOSetCallControlsConfig, rMsgContext, DIPO_MPLY_MAJORVERSION);
      }
      else
      {
         // Push the registration request into a queue and send the request to MediaPlayer once the service becomes available.
         m_qCallControlsConfigLock.s16Lock();
         m_qCallControlsConfig.push(oMPlayDiPOSetCallControlsConfig);
         m_qCallControlsConfigLock.vUnlock();
         ETG_TRACE_USR2(("[DESC]: bOnRegRouteGuidanceDisplayConf - MediaPlayer service unavailable. CallControls Configuration pending until service is available"));
      }

   }
   ETG_TRACE_USR4(("[PARAM]: spi_tclMPlayClientHandler::bSetCallControlConfig -bResult = %d ",ETG_ENUM(BOOL,bResult)));
   return bResult;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::bDiPOAcceptCall
 ***************************************************************************/
t_Bool spi_tclMPlayClientHandler::bDiPOAcceptCall(const t_String& corfszDeviceSerialNumber, const tenPhoneCallAcceptAction coenCallAcceptAction) const
{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));
   tBool bResult = false;
   SPI_INTENTIONALLY_UNUSED(corfszDeviceSerialNumber);
   SPI_INTENTIONALLY_UNUSED(coenCallAcceptAction);

   if (nullptr != m_poMainApp)
   {
      spi_tMsgDiPOAcceptCallMS oMPlayDiPOAcceptCallMS;

      oMPlayDiPOAcceptCallMS.deviceUSBSerialNumber.bSet(corfszDeviceSerialNumber.c_str(), MPLAYFI_CHAR_SET_UTF8);
      oMPlayDiPOAcceptCallMS.acceptAction.enType = static_cast<mplay_fi_tcl_e_AcceptAction::tenType>(coenCallAcceptAction);
      oMPlayDiPOAcceptCallMS.callUUID = "";

      ETG_TRACE_USR4(("[PARAM]: spi_tclMPlayClientHandler::bDiPOAcceptCall - Serial Number = %s", oMPlayDiPOAcceptCallMS.deviceUSBSerialNumber.szValue));

      // populate message context
      trMsgContext rMsgContext;
      vPopulateMessageContext(rMsgContext);
      FIMsgDispatch msgDispatcher(m_poMainApp);

      //send method DiPOAcceptCall to Mediaplayer
      bResult = msgDispatcher.bSendMessage(oMPlayDiPOAcceptCallMS, rMsgContext, DIPO_MPLY_MAJORVERSION);
   }
   ETG_TRACE_USR4(("[PARAM]: spi_tclMPlayClientHandler::bDiPOAcceptCall -bResult = %d ",ETG_ENUM(BOOL,bResult)));
   return bResult;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::bDiPOAcceptCall
 ***************************************************************************/
t_Bool spi_tclMPlayClientHandler::bDiPOEndCall(const t_String& corfszDeviceSerialNumber, const tenPhoneCallEndAction coenCallEndAction) const
{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));
   tBool bResult = false;
   SPI_INTENTIONALLY_UNUSED(corfszDeviceSerialNumber);
   SPI_INTENTIONALLY_UNUSED(coenCallEndAction);


   if (nullptr != m_poMainApp)
   {
      spi_tMsgDiPOEndCallMS oMPlayDiPOEndCallMS;

      oMPlayDiPOEndCallMS.deviceUSBSerialNumber.bSet(corfszDeviceSerialNumber.c_str(), MPLAYFI_CHAR_SET_UTF8);
      oMPlayDiPOEndCallMS.endAction.enType = static_cast<mplay_fi_tcl_e_EndAction::tenType>(coenCallEndAction);
      oMPlayDiPOEndCallMS.callUUID = "";

      ETG_TRACE_USR4(("[PARAM]: spi_tclMPlayClientHandler::bDiPOEndCall - Serial Number = %s", oMPlayDiPOEndCallMS.deviceUSBSerialNumber.szValue));

      // populate message context
      trMsgContext rMsgContext;
      vPopulateMessageContext(rMsgContext);
      FIMsgDispatch msgDispatcher(m_poMainApp);

      //send method DiPOAcceptCall to Mediaplayer
      bResult = msgDispatcher.bSendMessage(oMPlayDiPOEndCallMS, rMsgContext, DIPO_MPLY_MAJORVERSION);
   }
   ETG_TRACE_USR4(("[PARAM]: spi_tclMPlayClientHandler::bDiPOEndCall -bResult = %d",ETG_ENUM(BOOL,bResult)));
   return bResult;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vPrepareForRestoreSettings()
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vPrepareForRestoreSettings()
{
   ETG_TRACE_USR1(("[FUNC]spi_tclMPlayClientHandler::vPrepareForRestoreSettings() entered "));

   vRemoveAutoRegisterForProperty(MPLAY_APPCONTROLFI_C_U16_DIPODEVICECONNECTIONS);
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMPlayClientHandler::vRestoreSettingsComplete()
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vRestoreSettingsComplete()
{
   ETG_TRACE_USR1(("[FUNC]spi_tclMPlayClientHandler::vRestoreSettingsComplete() entered "));

   vAddAutoRegisterForProperty( MPLAY_APPCONTROLFI_C_U16_DIPODEVICECONNECTIONS);
}


/**************************************************************************
** FUNCTION   : tVoid spi_tclMPlayClientHandler:: vDeleteDevices()
***************************************************************************/
t_Void spi_tclMPlayClientHandler::vDeleteDevices(const std::vector<t_U32> &corfrvecDeletedDevices)
{

   ETG_TRACE_USR1(("spi_tclMPlayClientHandler::vDeleteDevices Entered"));

   if (NULL != m_poMplayDevList)
   {
      m_poMplayDevList->vDeleteDevices(corfrvecDeletedDevices);
   }//if (nullptr != m_poMplayDevList)
}

/***************************************************************************
 ** FUNCTION:  spi_tclMPlayClientHandler::vOnDipoDeviceTimeUpdate()
 ***************************************************************************/
t_Void spi_tclMPlayClientHandler::vOnDipoDeviceTimeUpdate(amt_tclServiceData* poMessage) const
{
   ETG_TRACE_USR1(("spi_tclMPlayClientHandler::vOnDipoDeviceTimeUpdate"));
   spi_tMsgDiPODeviceTimeUpdate oDiPODeviceTimeUpdate(*poMessage, DIPO_MPLY_MAJORVERSION);
   if (oDiPODeviceTimeUpdate.bIsValid())
   {
      t_U64 u64SecondsSinceReferenceDate = static_cast<t_U64>(oDiPODeviceTimeUpdate.DeviceTimeUpdate.u64SecondsSinceReferenceDate);
      t_S16 s16TimeZoneOffsetMinutes = static_cast<t_S16>(oDiPODeviceTimeUpdate.DeviceTimeUpdate.s16TimeZoneOffsetMinutes);
      t_S8 s8DaylightSavingsOffsetMinutes = static_cast<t_S8>(oDiPODeviceTimeUpdate.DeviceTimeUpdate.s8DaylightSavingsOffsetMinutes);
      ETG_TRACE_USR1(("spi_tclMPlayClientHandler::vOnDipoDeviceTimeUpdate - Curr Device Time = %s",std::to_string(u64SecondsSinceReferenceDate).c_str()));
      ETG_TRACE_USR1(("spi_tclMPlayClientHandler::vOnDipoDeviceTimeUpdate - s16TimeZoneOffsetMinutes = %d", s16TimeZoneOffsetMinutes));
      ETG_TRACE_USR1(("spi_tclMPlayClientHandler::vOnDipoDeviceTimeUpdate - s8DaylightSavingsOffsetMinutes = %d", s8DaylightSavingsOffsetMinutes));
      if(NULL != m_rMetaDataCbs.fvAppDeviceTimeUpdate)
      {
         trDeviceTime rDeviceTime;
         rDeviceTime.s16TimeZoneOffsetMinutes = s16TimeZoneOffsetMinutes;
         rDeviceTime.s8DaylightSavingsOffsetMinutes = s8DaylightSavingsOffsetMinutes;
         rDeviceTime.u64CurrDevTimeInSeconds = u64SecondsSinceReferenceDate;
         m_rMetaDataCbs.fvAppDeviceTimeUpdate(rDeviceTime);
      }
   }
}
//lint -restore
