/*!
*******************************************************************************
* \file              spi_tclBluetoothClient.h
* \brief             Bluetooth Settings Client handler class
*******************************************************************************
\verbatim
PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    Bluetooth Settings Client handler class
COPYRIGHT:      &copy; RBEI

HISTORY:
 Date       |  Author                           | Modifications
 21.02.2014 |  Ramya Murthy (RBEI/ECP2)         | Initial Version
 21.05.2014 |  Ramya Murthy (RBEI/ECP2)         | Revised logic for storing DeviceList
                                                  and validation of service responses.
 04.06.2014 |  Ramya Murthy (RBEI/ECP2)         | VehicleBTAddress implementation
 22.09.2014 |  Ramya Murthy (RBEI/ECP2)         | Implemented locks for DeviceList and
                                                  ChngDevRequestsList maps
 28.10.2014 |  Ramya Murthy (RBEI/ECP2)         | Implementation for setting BluetoothAudioSource
                                                  (Fix for SUZUKI-18263)
 19.05.2015 |  Ramya Murthy (RBEI/ECP2)         | Adaptation to DeviceListExtended property
 18.08.2015 |  Ramya Murthy (RBEI/ECP2)         | Adaptation to new BTSettings interfaces for AndroidAuto

\endverbatim
******************************************************************************/

#ifndef _SPI_TCLBLUETOOTHCLIENT_H_
#define _SPI_TCLBLUETOOTHCLIENT_H_

/******************************************************************************
| includes:
|----------------------------------------------------------------------------*/
#include <map>
#include <queue>

//!Include Application Help Library.
#define AHL_S_IMPORT_INTERFACE_GENERIC
#define AHL_S_IMPORT_INTERFACE_CCA_EXTENSION
#include "ahl_if.h"

//!Include common fi interface
#define FI_S_IMPORT_INTERFACE_BASE_TYPES
#define FI_S_IMPORT_INTERFACE_FI_MESSAGE
#include "common_fi_if.h"

//!Include Bluetooth Settings FI type defines
#define MOST_FI_S_IMPORT_INTERFACE_MOST_BTSETFI_FUNCTIONIDS
#define MOST_FI_S_IMPORT_INTERFACE_MOST_BTSETFI_TYPES
#define MOST_FI_S_IMPORT_INTERFACE_MOST_BTSETFI_ERRORCODES
#define MOST_FI_S_IMPORT_INTERFACE_MOST_BTSETFI_SERVICEINFO

 #include "conn_most_fi_if.h"

#include "XFiObjHandler.h"
using namespace shl::msgHandler;

#include "SPITypes.h"
#include "spi_BluetoothTypedefs.h"
#include "Lock.h"


/******************************************************************************
| defines and macros and constants(scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/
//! Type defines for iterators to DeviceList map.
typedef std::map<tU8, trBluetoothDeviceInfo>::iterator         tBTDevListMapItr;
typedef std::map<tU8, trBluetoothDeviceInfo>::const_iterator   tBTDevListMapConstItr;


//! BT Settings FI types
typedef most_btsetfi_tclMsgBaseMessage                   btset_FiMsgBase;
typedef most_fi_tcl_BTSetDateTimeStamp                   btset_tDateTimestamp;

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

/* Forward Declarations. */

/*!
* \class spi_tclBluetoothClient
* \brief Bluetooth Settings client handler class that realizes the BluetoothPolicy interface.
*/
class spi_tclBluetoothClient
   : public ahl_tclBaseOneThreadClientHandler
{

public:

   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  spi_tclBluetoothClient::spi_tclBluetoothClient(ahl_tclBaseOneThreadApp...)
   **************************************************************************/
   /*!
   * \fn      spi_tclBluetoothClient(ahl_tclBaseOneThreadApp* poMainAppl)
   * \brief   Overloaded Constructor
   * \param   [IN] poMainAppl : Pointer to main CCA application
   **************************************************************************/
   spi_tclBluetoothClient(ahl_tclBaseOneThreadApp* poMainAppl);

   /***************************************************************************
   ** FUNCTION:  spi_tclBluetoothClient::~spi_tclBluetoothClient()
   **************************************************************************/
   /*!
   * \fn      ~spi_tclBluetoothClient()
   * \brief   Destructor
   **************************************************************************/
   virtual ~spi_tclBluetoothClient();

   /**************************************************************************
   * Overriding ahl_tclBaseOneThreadService methods.
   **************************************************************************/

   /**************************************************************************
   ** FUNCTION:  tVoid spi_tclBluetoothClient::vOnServiceAvailable();
   **************************************************************************/
   /*!
   * \fn      vOnServiceAvailable()
   * \brief   This method is called by the framework if the service of our
   *          server becomes available, e.g. server has been started.
   * \param   None
   **************************************************************************/
   virtual tVoid vOnServiceAvailable();

   /**************************************************************************
   ** FUNCTION:  tVoid spi_tclBluetoothClient::vOnServiceUnavailable();
   **************************************************************************/
   /*!
   * \fn      vOnServiceUnavailable()
   * \brief   This method is called by the framework if the service of our
   *          server becomes unavailable, e.g. server has been shut down.
   * \param   None
   **************************************************************************/
   virtual tVoid vOnServiceUnavailable();

   /**************************************************************************
   * Overriding spi_tclBluetoothPolicyBase methods.
   **************************************************************************/

   /***************************************************************************
   ** FUNCTION:  tVoid spi_tclBluetoothClient::vRegisterForProperties()
   **************************************************************************/
   /*!
   * \fn      vRegisterForProperties()
   * \brief   Registers for interested properties to Bluetooth Service.
   **************************************************************************/
   virtual tVoid vRegisterForProperties();

   /***************************************************************************
   ** FUNCTION:  tVoid spi_tclBluetoothClient::vUnregisterForProperties()
   **************************************************************************/
   /*!
   * \fn      vUnregisterForProperties()
   * \brief   Registers for interested properties to Bluetooth Service.
   **************************************************************************/
   virtual tVoid vUnregisterForProperties();

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBluetoothClient::vRegisterCallbacks()
   ***************************************************************************/
   /*!
   * \fn      vRegisterCallbacks(trBluetoothCallbacks rBTRespCallbacks)
   * \brief   Interface to register for BT connection callbacks
   *          Optional interface to be implemented.
   * \param   [IN] rBTRespCallbacks: Callbacks structure
   * \retval  None
   **************************************************************************/
   virtual t_Void vRegisterCallbacks(trBluetoothCallbacks rBTRespCallbacks);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBluetoothClient::vRegisterCallbacks()
   ***************************************************************************/
   /*!
   * \fn      vRegisterOOBTCallbacks(trOOBTPairingCallbacks rOOBTRespCallbacks)
   * \brief   Interface to register for OOBT connection callbacks
   *          Optional interface to be implemented.
   * \param   [IN] rOOBTRespCallbacks: Callbacks structure
   * \retval  None
   **************************************************************************/
   virtual t_Void vRegisterOOBTCallbacks(trOOBTPairingCallbacks rOOBTRespCallbacks);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBluetoothClient::vRegisterPairingInfoCallbacks()
   ***************************************************************************/
   /*!
   * \fn      vRegisterPairingInfoCallbacks(trBluetoothPairingCallbacks rBTPairInfoCallbacks)
   * \brief   Interface to register for BT pairing info callbacks.
   *          Optional interface to be implemented.
   * \param   [IN] rBTPairInfoCallbacks: Callbacks structure
   * \retval  None
   **************************************************************************/
   virtual t_Void vRegisterPairingInfoCallbacks(trBluetoothPairingCallbacks rBTPairInfoCallbacks);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBluetoothClient::vRegisterBTMacAddressCallbacks()
   ***************************************************************************/
   /*!
   * \fn      vRegisterBTMacAddressCallbacks(trVehicleBTAddressCallback &rVehicleBTAddress)
   * \brief   Interface to register for BT pairing info callbacks.
   *          Optional interface to be implemented.
   * \param   [IN] rBTPairInfoCallbacks: Callbacks structure
   * \retval  None
   **************************************************************************/
   t_Void vRegisterBTMacAddressCallbacks(trVehicleBTAddressCallback &rVehicleBTAddress);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBluetoothClient::vRegisterDeviceNameCallback()
   ***************************************************************************/
   /*!
   * \fn      vRegisterDeviceNameCallback(trBTDeviceNameCbInfo rBTDeviceNameCbInfo)
   * \brief   Interface to register for device name callback of a specific device
   *          Optional interface to be implemented.
   * \param   [IN] rBTDeviceNameCbInfo: Callback info structure
   * \retval  None
   **************************************************************************/
   virtual t_Void vRegisterDeviceNameCallback(trBTDeviceNameCbInfo rBTDeviceNameCbInfo);

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclBluetoothClient::bGetPairingStatus(const...)
   ***************************************************************************/
   /*!
   * \fn      bGetPairingStatus(const t_String& rfcoszDeviceBTAddress)
   * \brief   Request from the Bluetooth Manager to the Bluetooth service
   *          for the pairing status of a BT device.
   *          Optional interface to be implemented.
   * \param   [IN] rfcoszDeviceBTAddress: BT address of device.
   * \retval  Bool value: TRUE - if Pairing is required, else FALSE
   **************************************************************************/
   virtual t_Bool bGetPairingStatus(const t_String& rfcoszDeviceBTAddress);

   /***************************************************************************
   ** FUNCTION:  t_U32 spi_tclBluetoothClient::u32GetBTDeviceHandle(const...)
   ***************************************************************************/
   /*!
   * \fn      u32GetBTDeviceHandle(const t_String& rfcoszDeviceBTAddress)
   * \brief   Request from Bluetooth Manager to the Bluetooth Client
   *          for the BT DeviceHandle of a device.
   *          Optional interface to be implemented.
   * \param   [IN] rfcoszDeviceBTAddress: BT address of device.
   * \retval  t_U32 : BT device handle of device with BT address = rfcoszDeviceBTAddress.
   *          If the device is not found in BT DeviceList, zero is returned.
   **************************************************************************/
   virtual t_U32 u32GetBTDeviceHandle(const t_String& rfcoszDeviceBTAddress);

   /***************************************************************************
   ** FUNCTION:  t_String spi_tclBluetoothClient::szGetBTDeviceName()
   ***************************************************************************/
   /*!
   * \fn      szGetBTDeviceName()
   * \brief   Interface to fetch name of a BT device.
   * \param   [IN] rfcszDeviceBTAddress: BT address of device.
   * \retval  t_String : BT device name of requested device.
   *            If device name is not available, returns empty string.
   **************************************************************************/
   virtual t_String szGetBTDeviceName(const t_String& rfcszDeviceBTAddress);

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclBluetoothClient::bGetBTServiceAvailability(...)
   ***************************************************************************/
   /*!
   * \fn      bGetBTServiceAvailability()
   * \brief   Interface to check Bluetooth Service Availability in HU.
   *          Optional interface to be implemented.
   * \param   None
   * \retval  t_Bool : True - if BT Service is available, else false.
   **************************************************************************/
   virtual t_Bool bGetBTServiceAvailability();
   
   /***************************************************************************`
   ** FUNCTION:  t_Bool spi_tclBluetoothClient::bSetBTLimitationMode(...)
   ***************************************************************************/
   /*!
   * \brief   Interface to set a Bluetooth limitation in HU.
   *          Optional interface to be implemented.
   * \param   [IN] rfcszBTAddress: BT address of device
   * \param   [IN] rfcszBTDeviceName: BT device name
   * \param   [IN] enTechnology: Technology for which BT limitation mode is required
   * \param   [IN] enCommChannel: Communication channel of the device
   * \param   [IN] enAction: Type of action for which BT limitation mode is required
   * \retval  t_Bool : True - if request to set limitation mode is sent successfully,
   *              else false.
   **************************************************************************/
   t_Bool bSetBTLimitationMode(const t_String& rfcszBTAddress,
         const t_String& rfcszBTDeviceName,
         tenBTTechnology enTechnology,
         tenBTCommunicationChannel enCommChannel,
         tenBTLimitationAction enAction);

   /***************************************************************************`
   ** FUNCTION:  t_Bool spi_tclBluetoothClient::bSetBTPairingResponse(...)
   ***************************************************************************/
   /*!
   * \brief   Interface to set a BT Pairing response
   * \param   [IN] enBTPairingResponse: BT pairing response
   * \retval  t_Bool : True - if BT pairing response is sent successfully,
   *              else false.
   **************************************************************************/
   t_Bool bSetBTPairingResponse(tenPairingResponseType enBTPairingResponse);

  /***************************************************************************
   ** FUNCTION:  t_String spi_tclBluetoothClient::vGetBtMacAddress()
   ***************************************************************************/
   /*!
   * \fn      vGetBtMacAddress()
   * \brief   Interface to fetch BT MAC Address from the friendly Name.
   * \param   [IN] rfcszDevName: BT name of device.
   * \param   [IN/OUT] rfszMACAddress: BT MAC Address of device.
   * \retval  NONE.
   **************************************************************************/
   virtual t_Void vGetBtMacAddress(const t_String& rfcszDevName,
                                     t_String& rfszMACAddress);

   /***************************************************************************
   ** FUNCTION:  t_String spi_tclBluetoothClient::szGetVehicleBTFriendlyName()
   ***************************************************************************/
   /*!
   * \fn      szGetVehicleBTFriendlyName()
   * \brief   Interface to fetch Vehicle BT Friendly name.
   * \retval  t_String : HU's BT friendly name.
   **************************************************************************/
   virtual t_String szGetVehicleBTFriendlyName();

   /***************************************************************************`
   ** FUNCTION:  t_Bool spi_tclBluetoothClient::bReplaceBTLimitationMode(...)
   ***************************************************************************/
   /*!
   * \brief   Interface to replace a Bluetooth limitation in HU.
   * \param   [IN] rCurrentBTLimitationMode: Current BT limitation mode to be replaced
   * \param   [IN] rReplacingBTLimitationMode: New BT limitation mode
   * \retval  t_Bool : True - if request to replace limitation mode is sent successfully,
   *                   else false.
   **************************************************************************/
   t_Bool bReplaceBTLimitationMode(trBTLimitationModeInfo rCurrentBTLimitationMode,
           trBTLimitationActionInfo rNewBTLimitationMode);
           
   /***************************************************************************`
   ** FUNCTION:  t_Bool spi_tclBluetoothClient::vGetBTProfileStatus(...)
   ***************************************************************************/
   /*!
   * \brief   Interface to get BT profile status
   * \param   [IN] u8DeviceHandle: BT device handle
   * \param   [OUT] rfbIsHFPConn : HFP connection status
   * \param   [OUT] rfbIsA2DPConn : A2DP connection status
   * \retval  NONE
   **************************************************************************/
   t_Void vGetBTProfileStatus(t_U8 u8DeviceHandle,
            t_Bool& rfbIsHFPConn, t_Bool& rfbIsA2DPConn);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclBluetoothClient::vTriggerBTNameUpdate()
   ***************************************************************************/
   /*!
   * \fn      vTriggerBTNameUpdate(const t_String& rfcszAAPBTAddress)
   * \brief   Interface to get device name associated with a specific device
   * \param   [IN] rfcszAAPBTAddress: BT Address of the device
   * \retval  None
   **************************************************************************/
   virtual t_Void vTriggerBTNameUpdate(const t_String& rfcszAAPBTAddress);

private:

   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/

   /***************************************************************************
   * ! Handler method declarations used by message map.
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  tVoid spi_tclBluetoothClient::vOnStatusDeviceListExtended(...)
   ***************************************************************************/
   /*!
   * \fn      vOnStatusDeviceListExtended(amt_tclServiceData* poMessage)
   * \brief   Called by framework when DeviceListStatus property update message is
   *          sent by Bluetooth Service.
   * \param   [IN] poMessage : Pointer to message
   * \retval  None
   **************************************************************************/
   tVoid vOnStatusDeviceListExtended(amt_tclServiceData* poMessage);

   /**************************************************************************
   ** FUNCTION:  tVoid spi_tclBluetoothClient::vOnMRChangeDevState(amt_...
   **************************************************************************/
   /*!
   * \fn      vOnMRChangeDevState(amt_tclServiceData* poMessage)
   * \brief   Called by framework when ChangeDeviceState method result is
   *          sent by Bluetooth Service.
   * \param   [IN] poMessage : Pointer to message
   * \retval  None
   **************************************************************************/
   tVoid vOnMRChangeDevState(amt_tclServiceData* poMessage);

   /**************************************************************************
   ** FUNCTION:  tVoid spi_tclBluetoothClient::vOnStatusPairingStatus(...)
   **************************************************************************/
   /*!
   * \fn      vOnStatusPairingStatus(amt_tclServiceData* poMessage)
   * \brief   Called by framework when PairingStatus property update is
   *          sent by Bluetooth Service.
   * \param   [IN] poMessage : Pointer to message
   * \retval  None
   **************************************************************************/
   tVoid vOnStatusPairingStatus(amt_tclServiceData* poMessage);

   /***************************************************************************
   ** FUNCTION:  spi_tclBluetoothClient::vOnVehicleBTAddressStatus(amt_tclServiceData...
   **************************************************************************/
   /*!
   * \fn      vOnVehicleBTAddressStatus(amt_tclServiceData* poMessage)
   * \brief   Called by framework when VehicleBTAddress property update is
   *          sent by Bluetooth Service.
   * \param   [IN] poMessage : Pointer to message
   * \retval  None
   **************************************************************************/
   tVoid vOnVehicleBTAddressStatus(amt_tclServiceData* poMessage);

   /***************************************************************************
   ** FUNCTION:  spi_tclBluetoothClient::vOnMRSetBTLimitationMode(amt_tclServiceData...
   **************************************************************************/
   /*!
   * \brief   Called by framework when SetBluetoothLimitationMode method result is
   *          sent by Bluetooth Service.
   * \param   [IN] poMessage : Pointer to message
   * \retval  None
   **************************************************************************/
   tVoid vOnMRSetBTLimitationMode(amt_tclServiceData* poMessage);

   /***************************************************************************
   ** FUNCTION:  spi_tclBluetoothClient::vOnErrorSetBTLimitationMode(amt_tclServiceData...
   **************************************************************************/
   /*!
   * \brief   Called by framework when SetBluetoothLimitationMode error is
   *          sent by Bluetooth Service.
   * \param   [IN] poMessage : Pointer to message
   * \retval  None
   **************************************************************************/
   tVoid vOnErrorSetBTLimitationMode(amt_tclServiceData* poMessage);

   /***************************************************************************
   ** FUNCTION:  spi_tclBluetoothClient::vOnConflictsDetectedStatus(amt_tclServiceData...
   **************************************************************************/
   /*!
   * \fn      vOnConflictsDetectedStatus(amt_tclServiceData* poMessage)
   * \brief   Called by framework when ConflictsDetected property update is
   *          sent by Bluetooth Service.
   * \param   [IN] poMessage : Pointer to message
   * \retval  None
   **************************************************************************/
   tVoid vOnConflictsDetectedStatus(amt_tclServiceData* poMessage);

   /***************************************************************************
   ** FUNCTION:  spi_tclBluetoothClient::vOnMRResolveConflicts(amt_tclServiceData...
   **************************************************************************/
   /*!
   * \brief   Called by framework when ResolveConflicts method result is
   *          sent by Bluetooth Service.
   * \param   [IN] poMessage : Pointer to message
   * \retval  None
   **************************************************************************/
   tVoid vOnMRResolveConflicts(amt_tclServiceData* poMessage);

   /***************************************************************************
   ** FUNCTION:  spi_tclBluetoothClient::vOnErrorResolveConflicts(amt_tclServiceData...
   **************************************************************************/
   /*!
   * \brief   Called by framework when ResolveConflicts method result is
   *          sent by Bluetooth Service.
   * \param   [IN] poMessage : Pointer to message
   * \retval  None
   **************************************************************************/
   tVoid vOnErrorResolveConflicts(amt_tclServiceData* poMessage);


   /***************************************************************************
   ** FUNCTION:  spi_tclBluetoothClient::vOnMRReplaceBTLimitationMode(amt_tclServiceData...
   **************************************************************************/
   /*!
   * \brief   Called by framework when ReplaceBluetoothLimitationMode method result is
   *          sent by Bluetooth Service.
   * \param   [IN] poMessage : Pointer to message
   * \retval  None
   **************************************************************************/
   tVoid vOnMRReplaceBTLimitationMode(amt_tclServiceData* poMessage);

   /***************************************************************************
   ** FUNCTION:  spi_tclBluetoothClient::vOnErrorReplaceBTLimitationMode(amt_tclServiceData...
   **************************************************************************/
   /*!
   * \brief   Called by framework when ReplaceBluetoothLimitationMode error is
   *          sent by Bluetooth Service.
   * \param   [IN] poMessage : Pointer to message
   * \retval  None
   **************************************************************************/
   tVoid vOnErrorReplaceBTLimitationMode(amt_tclServiceData* poMessage);

   /***************************************************************************
   ** FUNCTION:  spi_tclBluetoothClient::vOnStatusPairingRequest(amt_tclServiceData...
   **************************************************************************/
   /*!
   * \fn      vOnStatusPairingRequest(amt_tclServiceData* poMessage)
   * \brief   Called by framework when PairingRequest property update is
   *          sent by Bluetooth Service.
   * \param   [IN] poMessage : Pointer to message
   * \retval  None
   **************************************************************************/
   tVoid vOnStatusPairingRequest(amt_tclServiceData* poMessage);

   /***************************************************************************
   ** FUNCTION:  spi_tclBluetoothClient::vOnStatusVehicleBTFriendlyName(amt_tclServiceData...
   **************************************************************************/
   /*!
   * \fn      vOnStatusVehicleBTFriendlyName(amt_tclServiceData* poMessage)
   * \brief   Called by framework when VehicleBTFriendlyName property update is
   *          sent by Bluetooth Service.
   * \param   [IN] poMessage : Pointer to message
   * \retval  None
   **************************************************************************/
   tVoid vOnStatusVehicleBTFriendlyName(amt_tclServiceData* poMessage);

   /***************************************************************************
   ** FUNCTION:  spi_tclBluetoothClient::vOnStatusBTLimitationMode(amt_tclServiceData...
   **************************************************************************/
   /*!
   * \brief   Called by framework when BluetoothLimitationMode property update is
   *          sent by Bluetooth Service.
   * \param   [IN] poMessage : Pointer to message
   * \retval  None
   **************************************************************************/
   tVoid vOnStatusBTLimitationMode(amt_tclServiceData* poMessage);

   /**************************************************************************
   * BT settings message posting methods
   **************************************************************************/

   /**************************************************************************
   ** FUNCTION:  tBool spi_tclBluetoothClient::bPostMethodStart(const...
   **************************************************************************/
   /*!
   * \fn      bPostMethodStart(const btset_FiMsgBase& rfcooBTSetMS,
   *           t_U32 u32CmdCounter)
   * \brief   Posts a BT settings MethodStart message.
   * \param   [IN] rfcooBTSetMS : BT settings Method Start message type
   * \param   [IN] u32CmdCounter : Command counter value to be set in the message.
   * \retval  tBool: TRUE - if message posting is successful, else FALSE.
   **************************************************************************/
   tBool bPostMethodStart(const btset_FiMsgBase& rfcooBTSetMS,
         t_U32 u32CmdCounter) const;

   /**************************************************************************
   * ! Other methods
   **************************************************************************/

   /***************************************************************************
   ** FUNCTION:  tU8 spi_tclBluetoothClient::u8GetBTDeviceHandle(const t_String&...)
   ***************************************************************************/
   /*!
   * \fn      u8GetBTDeviceHandle(const t_String& rfcoszDeviceBTAddress)
   * \brief   Retrieves BT DeviceHandle of the device for input BT Address.
   *          If device does not exist in BT DeviceList, 0 (i.e. invalid handle)
   *          is returned.
   * \param   [IN] rfcoszDeviceBTAddress : BT Address of a device
   * \retval  tU8
   **************************************************************************/
   tU8 u8GetBTDeviceHandle(const t_String& rfcoszDeviceBTAddress);

    /**************************************************************************
   ** FUNCTION:  spi_tclBluetoothClient::vAddToDeviceList(tU8 u8DeviceHandle,...)
   **************************************************************************/
   /*!
   * \fn      vAddToDeviceList(tU8 u8DeviceHandle, tBool bDeviceConnStatus)
   * \brief   Adds a new device to internal BTDeviceList with its connection status
   *          information, and requests device info from Bluetooth Service using
   *          GetDeviceInfo message.
   * \param   [IN] u8DeviceHandle : Unique handle of device to be added.
   * \param   [IN] bDeviceConnStatus : Connection status of BT device
   *               (TRUE - if connected, else FALSE).
   * \param   [IN] rfcszDeviceName : Name of BT device
   * \param   [IN] rfcszBTAddress : BT Address of device
   * \param   [IN] bIsHFPConn : BT HFP connection status
   * \param   [IN] bIsA2DPConn : BT A2DP connection status
   * \retval  None
   **************************************************************************/
   tVoid vAddToDeviceList(tU8 u8DeviceHandle, tBool bDeviceConnStatus,
            const t_String& rfcszDeviceName, const t_String& rfcszBTAddress,
            t_Bool bIsHFPConn, t_Bool bIsA2DPConn);

   /**************************************************************************
   ** FUNCTION:  spi_tclBluetoothClient::vRemoveFromDeviceList(tU8 u8DeviceHandle)
   **************************************************************************/
   /*!
   * \fn      vRemoveFromDeviceList(tU8 u8DeviceHandle, tBool bDeviceConnStatus)
   * \brief   Removes a device from BTDeviceList map.
   * \param   [IN] u8DeviceHandle : Unique handle of device to be deleted.
   * \retval  None
   **************************************************************************/
   tVoid vRemoveFromDeviceList(tU8 u8DeviceHandle);

   /**************************************************************************
   ** FUNCTION:  spi_tclBluetoothClient::vClearBTDeviceList()
   **************************************************************************/
   /*!
   * \fn      vClearBTDeviceList()
   * \brief   Clears the BT device info list.
   * \param   None.
   * \retval  None
   **************************************************************************/
   tVoid vClearBTDeviceList();

   /**************************************************************************
   ** FUNCTION:  spi_tclBluetoothClient::vUpdateDeviceConnStatus(tU8...)
   **************************************************************************/
   /*!
   * \fn      vUpdateDeviceConnStatus(tU8 u8DeviceHandle, tBool bDeviceConnStatus)
   * \brief   Updates the connection status of a device in BTDeviceList map.
   * \param   [IN] u8DeviceHandle : Unique handle of device.
   * \param   [IN] bDeviceConnectionStatus : Connection status of device.
   * \param   [IN] bIsHFPConn : Connection status of HFP.
   * \param   [IN] bIsA2DPConn : Connection status of A2DP.
   * \retval  None
   **************************************************************************/
   tVoid vUpdateDeviceConnStatus(tU8 u8DeviceHandle,
         tBool bDeviceConnectionStatus,
         tBool bIsHFPConn, tBool bIsA2DPConn);

   /**************************************************************************
   ** FUNCTION:  spi_tclBluetoothClient::vUpdateDeviceName(tU8...)
   **************************************************************************/
   /*!
   * \fn      vUpdateDeviceName(tU8 u8DeviceHandle, const t_String& rfcszDeviceName)
   * \brief   Updates the BT device name in BTDeviceList map.
   * \param   [IN] u8DeviceHandle : Unique handle of device.
   * \param   [IN] rfcszDeviceName : Name of BT device
   * \retval  None
   **************************************************************************/
   tVoid vUpdateDeviceName(tU8 u8DeviceHandle, const t_String& rfcszDeviceName);

   /***************************************************************************
   ** FUNCTION:  tBool spi_tclBluetoothClient::bIsDeviceValid(tU8...)
   ***************************************************************************/
   /*!
   * \fn      bIsDeviceValid(tU8 u8DeviceHandle, tBTDevListMapItr& rfBTDevLstMapItr)
   * \brief   Checks if specified device exists in the internal BT DeviceList map
   * \param   [IN] u8DeviceHandle : BT DeviceHandle
   * \retval  tBool: TRUE - If Device is found in BT DeviceList, else FALSE
   **************************************************************************/
   tBool bIsDeviceValid(tU8 u8DeviceHandle);

   /**************************************************************************
   ** FUNCTION:  spi_tclBluetoothClient::vNotifyBTConnectionChanged(...)
   **************************************************************************/
   /*!
   * \fn      vNotifyBTConnectionChanged(const t_String& rfcoszDevBTAddress,
   *             tenBTConnectionResult enBTConnResult)
   * \brief   Notifies BT Manager about a device connection change.
   *          (Used to inform when a change in connection status of a device is
   *          detected by the client-handler. i.e, when a BT device changes to
   *          connected or disconnected state)
   * \param   [IN] rfcoszDevBTAddress : BT Address of changed device
   * \param   [IN] enBTConnResult : Connection status of changed device.
   * \retval  None
   **************************************************************************/
   tVoid vNotifyBTConnectionChanged(const t_String& rfcoszDevBTAddress,
         tenBTConnectionResult enBTConnResult) const;
         
   /**************************************************************************
   ** FUNCTION:  spi_tclBluetoothClient::vNotifyBTProfileStatus(...)
   **************************************************************************/
   /*!
   * \fn      vNotifyBTProfileStatus(std::map<tU8, trBluetoothDeviceInfo>& mapBTDeviceList)
   * \brief   Notifies BT Manager about BT profile connection status
   * \retval  None
   **************************************************************************/
   tVoid vNotifyBTProfileStatus();

   /**************************************************************************
   ** FUNCTION:  spi_tclBluetoothClient::vNotifyBTDeviceName(...)
   **************************************************************************/
   /*!
   * \fn      vNotifyBTDeviceName
   * \brief   Notifies BT device name of a BT device.
   * \param   [IN] szBTAddress : BT Address of device
   * \param   [IN] szBTDeviceName : BT Device name of device
   * \retval  None
   **************************************************************************/
   tVoid vNotifyBTDeviceName(t_String szBTAddress, t_String szBTDeviceName) const;

   /**************************************************************************
   ** FUNCTION:  t_String spi_tclBluetoothClient::szGetDateTimestamp(const...
   **************************************************************************/
   /*!
   * \fn      szGetDateTimestamp(const btset_tDateTimestamp& rfcooDateTimeStamp)
   * \brief   Converts a DateTimeStamp object to a single string.
   * \param   [IN] rfcooDateTimeStamp : BT Settings DateTimeStamp object.
   * \retval  t_String
   **************************************************************************/
   t_String szGetDateTimestamp(const btset_tDateTimestamp& rfcooDateTimeStamp) const;

   /**************************************************************************
   ** FUNCTION:  tVoid spi_tclBluetoothClient::szGetBTDeviceAddress(...)
   **************************************************************************/
   /*!
   * \fn      szGetBTDeviceAddress(tU8 u8DeviceHandle)
   * \brief   Retrieves BT address of a device from DeviceList
   * \param   [IN] u8DeviceHandle : BT Device Handle
   * \retval  t_String
   **************************************************************************/
   t_String szGetBTDeviceAddress(tU8 u8DeviceHandle);

   /***************************************************************************
   ** FUNCTION:  spi_tclBluetoothClient::vHandleLimitationMode(...)
   **************************************************************************/
   /*!
   * \fn      vHandleLimitationMode(const t_String& rfcszBTAddress, tenBTTechnology enTechnology,
   *            tenBTCommunicationChannel enCommChannel, tenBTLimitationAction enAction)
   * \brief   Handles BT Limitation mode to be sent based on the request and current state
   * \param   [IN] rfcszBTAddress: BT address of device
   * \param   [IN] enTechnology: Technology for which BT limitation mode is required
   * \param   [IN] enCommChannel: Communication channel of the device
   * \param   [IN] enAction: Type of action for which BT limitation mode is required
   * \retval  t_Bool : True - if request to set limitation mode should be sent.
   **************************************************************************/
   t_Bool vHandleLimitationMode(const t_String& rfcszBTAddress, const t_String& rfcszBTDeviceName, tenBTTechnology enTechnology,
         tenBTCommunicationChannel enCommChannel, tenBTLimitationAction enAction);

   /**************************************************************************
   ** FUNCTION:  tVoid spi_tclBluetoothClient::bIsCurrentBTLimMode(...)
   **************************************************************************/
   /*!
   * \fn      bIsCurrentBTLimMode(const t_String& rfcszBTAddress,
   *           tenBTTechnology enTechnology, tenBTCommunicationChannel enCommChannel)
   * \brief   Checks if the given device details is same as current BT limitation mode device
   * \param   [IN] rfcszBTAddress: BT address of device
   * \param   [IN] rfcszBTDeviceName: BT device name
   * \param   [IN] enTechnology: Technology for which BT limitation mode is required
   * \param   [IN] enCommChannel: Communication channel of the device
   * \retval  t_Bool
   **************************************************************************/
   t_Bool bIsCurrentBTLimMode(const t_String& rfcszBTAddress, const t_String& rfcszBTDeviceName,
            tenBTTechnology enTechnology, tenBTCommunicationChannel enCommChannel);

   /***************************************************************************
   ** FUNCTION:  spi_tclBluetoothClient::bHandlePrepareStateBTLim(...)
   **************************************************************************/
   /*!
   * \fn      bHandlePrepareStateBTLim(trBTLimitationActionInfo rNewBtLimMode)
   * \brief   Handles BT limitation mode when Prepare request to be sent
   * \param   [IN] rNewBtLimMode: New BT Limitation Mode Info
   * \retval  t_Bool : True - if request to set limitation mode should be sent.
   **************************************************************************/
   t_Bool bHandlePrepareStateBTLim(trBTLimitationActionInfo rNewBtLimMode);

   /***************************************************************************
   ** FUNCTION:  spi_tclBluetoothClient::bHandlePreactivateStateBTLim(...)
   **************************************************************************/
   /*!
   * \fn      bHandlePreactivateStateBTLim(trBTLimitationActionInfo rNewBtLimMode)
   * \brief   Handles BT limitation mode when Preactivate request to be sent
   * \param   [IN] rNewBtLimMode: New BT Limitation Mode Info
   * \retval  t_Bool : True - if request to set limitation mode should be sent.
   **************************************************************************/
   t_Bool bHandlePreactivateStateBTLim(trBTLimitationActionInfo rNewBtLimMode);

   /***************************************************************************
   ** FUNCTION:  spi_tclBluetoothClient::bHandleActivateStateBTLim(...)
   **************************************************************************/
   /*!
   * \fn      bHandleActivateStateBTLim(trBTLimitationActionInfo rNewBtLimMode)
   * \brief   Handles BT limitation mode when Activate request to be sent
   * \param   [IN] rNewBtLimMode: New BT Limitation Mode Info
   * \retval  t_Bool : True - if request to set limitation mode should be sent.
   **************************************************************************/
   t_Bool bHandleActivateStateBTLim(trBTLimitationActionInfo rNewBtLimMode);

   /***************************************************************************
   ** FUNCTION:  spi_tclBluetoothClient::bHandleDactivateStateBTLim(...)
   **************************************************************************/
   /*!
   * \fn      bHandleDactivateStateBTLim(trBTLimitationActionInfo rNewBtLimMode)
   * \brief   Handles BT limitation mode when Deactivate request to be sent
   * \param   [IN] rNewBtLimMode: New BT Limitation Mode Info
   * \retval  t_Bool : True - if request to set limitation mode should be sent.
   **************************************************************************/
   t_Bool bHandleDactivateStateBTLim(trBTLimitationActionInfo rNewBtLimMode);

   /***************************************************************************
   ** FUNCTION:  spi_tclBluetoothClient::vGetCurrentBTLimModeInfo(...)
   **************************************************************************/
   /*!
   * \fn      vGetCurrentBTLimModeInfo()
   * \brief   Returns the current BT Limitation Mode info
   * \retval  t_Void
   **************************************************************************/
   t_Void vGetCurrentBTLimModeInfo(trBTLimitationModeInfo& rfrBTLimitationModeInfo);

   /***************************************************************************
   ** FUNCTION:  spi_tclBluetoothClient::vOnStatusServiceAvailable(amt_tclServiceData...
   **************************************************************************/
   /*!
   * \fn      vOnStatusServiceAvailable(amt_tclServiceData* poMessage)
   * \brief   Called by framework when ServiceAvailable property update is
   *          sent by Bluetooth Service.
   * \param   [IN] poMessage : Pointer to message
   * \retval  None
   **************************************************************************/
   t_Void vOnStatusServiceAvailable(amt_tclServiceData* poMessage);

   /***************************************************************************
   ** FUNCTION:  spi_tclBluetoothClient::vPostBTServiceAvailability(t_Bool bStatus)
   **************************************************************************/
   /*!
   * \fn      vPostBTServiceAvailability(t_Bool bStatus)
   * \brief   Updates the registered clients about the BT service availability.
   * \param   [IN] bStatus : Status of the Bluetooth service
   *                         If true - available(normal mode) else unavailable
   * \retval  None
   **************************************************************************/
   t_Void vPostBTServiceAvailability(t_Bool bStatus);

   /**************************************************************************
   ** FUNCTION:  spi_tclBluetoothClient::spi_tclBluetoothClient()
   **************************************************************************/
   /*!
   * \fn      spi_tclBluetoothClient()
   * \brief   Default Constructor, will not be implemented.
   *          NOTE: This is a technique to disable the Default Constructor for
   *          this class. So if an attempt for the constructor is made compiler
   *          complains.
   **************************************************************************/
   spi_tclBluetoothClient();

   /**************************************************************************
   ** FUNCTION:  spi_tclBluetoothClient::spi_tclBluetoothClient(const...
   **************************************************************************/
   /*!
   * \fn      spi_tclBluetoothClient(const spi_tclBluetoothClient& oClient)
   * \brief   Copy Consturctor, will not be implemented.
   *          Avoids Lint Prio 3 warning: Info 1732: new in constructor for
   *          class'spi_tclBluetoothClient' which has no Copy Consturctor.
   *          NOTE: This is a technique to disable the Copy Consturctor for this
   *          class. So if an attempt for the copying is made linker complains.
   * \param   [IN] poMessage : Property to be set.
   **************************************************************************/
   spi_tclBluetoothClient(const spi_tclBluetoothClient& oClient);

   /**************************************************************************
   ** FUNCTION:  spi_tclBluetoothClient::spi_tclBluetoothClient& operator=(...
   **************************************************************************/
   /*!
   * \fn      spi_tclBluetoothClient& operator=(
   *          const spi_tclBluetoothClient& oClient)
   * \brief   Assingment Operater, will not be implemented.
   *          Avoids Lint Prio 3 warning: Info 1732: new in constructor for
   *          class 'spi_tclBluetoothClient' which has no assignment operator.
   *          NOTE: This is a technique to disable the assignment operator for this
   *          class. So if an attempt for the assignment is made compiler complains.
   **************************************************************************/
   spi_tclBluetoothClient& operator=(const spi_tclBluetoothClient& oClient);

   /***************************************************************************
   * ! Data members
   ***************************************************************************/

   /***************************************************************************
   ** Main Application pointer
   ***************************************************************************/
   ahl_tclBaseOneThreadApp*                     m_poMainAppl;

   /***************************************************************************
   ** Bluetooth Device List Map
   ***************************************************************************/
   std::map<tU8, trBluetoothDeviceInfo>         m_mapBTDeviceList;

   /***************************************************************************
   ** Bluetooth Response callbacks structure for BT Manager
   ***************************************************************************/
   trBluetoothCallbacks                         m_rBluetoothCallbacks;

   /***************************************************************************
   ** OOBT Response callbacks structure for OOBTBT Manager
   ***************************************************************************/
   trOOBTPairingCallbacks                         m_rOOBTCallbacks;


   /***************************************************************************
   ** Bluetooth pairing info callbacks structure for BT Manager
   ***************************************************************************/
   trBluetoothPairingCallbacks                  m_rBTPairInfoCallbacks;

   /***************************************************************************
   ** Bluetooth Device name callback for BT Manager
   ***************************************************************************/
   trBTDeviceNameCbInfo                         m_rBTDeviceNameCbInfo;

   /***************************************************************************
   ** Current BT Limitation Mode Info
   ***************************************************************************/
   trBTLimitationModeInfo                       m_rCurrentBTLimModeInfo;

   /***************************************************************************
   ** Current BT Mac Address callback
   ***************************************************************************/
   trVehicleBTAddressCallback                   m_rVehicleBTAddress;
   
   /***************************************************************************
   ** Current BT Conflict State Information
   ***************************************************************************/
   tenConflictState                              m_enConflictState;

   /***************************************************************************
   ** Lock object for DeviceList map
   ***************************************************************************/
   Lock  m_oDevLstLock;


   /***************************************************************************
   ** Bluetooth Service Availability info
   ***************************************************************************/
   t_Bool  m_bIsBTServiceAvailable;

   /***************************************************************************
   ** Lock object for Bluetooth Service Availability
   ***************************************************************************/
   Lock    m_oBTAvailLock;

   /***************************************************************************
   ** Lock object for Current BT Limitation Mode Info
   ***************************************************************************/
   Lock    m_oCurBTLimModeLock;

   /***************************************************************************
   ** HU Bluetooth Friendly name
   ***************************************************************************/
   t_String m_szVehicleBTFriendlyName;
   
   /***************************************************************************
   ** OOB Conflict Processed state
   ***************************************************************************/
   t_Bool m_bOOBConflictProcessed;

   /***************************************************************************
   ** BT Service Available status (BT SPM state)
   ***************************************************************************/  
   tenBTServiceAvailableStatus m_enBTServiceAvailableStatus;

   /***************************************************************************
   * Message map definition macro
   ***************************************************************************/
   DECLARE_MSG_MAP(spi_tclBluetoothClient)
};

#endif // _SPI_TCLBLUETOOTHCLIENT_H_

///////////////////////////////////////////////////////////////////////////////
// <EOF>
