/*!
 *******************************************************************************
 * \file             spi_tclAudioIn.cpp
 * \brief            Audio Input Handling using ECNR
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Audio Input Handling using ECNR
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 24.03.2014 |  Pruthvi Thej Nagaraju       | Initial Version

 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "spi_tclAudioIn.h"
/*Replaced with Synchronous APIs of ECNR and also to support Gen4 build
#include "spi_tclAudioDBusClient.h" */
//#include "spi_EcnrIntrospection.h"
#include <pthread.h>
//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_AUDIO
#include "trcGenProj/Header/spi_tclAudioIn.cpp.trc.h"
#endif
#endif

/******************************************************************************
 | defines and macros and constants(scope: module-local)
 |----------------------------------------------------------------------------*/
//! Defines for ECNR DBus service
#define ECNR_OBJECT_PATH "/"
#define ECNR_SERVICE_NAME "org.bosch.ecnr.service"
#define ECNR_INTERFACE_NAME "org.bosch.ecnr.service"
#define ECNR_ALSA_INTERFACE_NAME "org.bosch.ecnr.alsa"

//! TODO : Remove hardcoded value and use include file from ECNR
//! SPI AppID as defined in ECNR config
static const t_UChar cocECNRAppID = 2;
static const t_UChar cocECNRStartMode = 1;
//! SPI Dataset Offset as defined in ECNR config
static const t_U32 cou32SPIDatasetOffset = 200;

/***************************************************************************
 *********************************PUBLIC*************************************
 ***************************************************************************/

/***************************************************************************
 ** FUNCTION:  spi_tclAudioIn::spi_tclAudioIn
 ***************************************************************************/
spi_tclAudioIn::spi_tclAudioIn() :
         m_poEcnrService(NULL), m_poEcnrAlsa(NULL)//, m_poDBusConn(NULL)
{
}

/***************************************************************************
 ** FUNCTION:  spi_tclAudioIn::~spi_tclAudioIn
 ***************************************************************************/
spi_tclAudioIn::~spi_tclAudioIn()
{
   m_poEcnrService = NULL;
   m_poEcnrAlsa = NULL;
   //m_poDBusConn = NULL;
   //finalize_proxy_call_queue();(Replaced with Synchronous APIs of ECNR and also to support Gen4 build)
}

/***************************************************************************
 ** FUNCTION:  spi_tclAudioIn::bSubscribeForAudioIn
 ***************************************************************************/
t_Bool spi_tclAudioIn::bSubscribeForAudioIn()
{
   ETG_TRACE_USR1(("spi_tclAudioIn::bSubscribeForAudioIn entered "));
   t_Bool bResult= true;
   //! Get connection to the system bus (ECNR service is launched on system bus)
   //m_poDBusConn = dbus_g_bus_get(DBUS_BUS_SYSTEM, &poError);

   guint watcher_id = g_bus_watch_name ( G_BUS_TYPE_SYSTEM ,           // dbus type: system or session
                                ECNR_SERVICE_NAME,            // dbus service name to watch
                                G_BUS_NAME_WATCHER_FLAGS_NONE, // client, NO flag set
                                vDbusNameAppearedCb,           // dbus_name_appeared cb
                                vDbusNameVanishedCb,           // dbus_name_vanished cb
                                (void*)this,                   // user data to be passed
                                NULL);                         // user data free func
   if(0 == watcher_id)
   {
      ETG_TRACE_USR1(("spi_tclAudioIn:: bSubscribeForAudioIn g_bus_watch_name() failed!"));
      bResult= false;
   }
   return bResult;
}

/***************************************************************************
 ** FUNCTION:  spi_tclAudioIn::vDbusNameAppearedCb()
 ***************************************************************************/
t_Void spi_tclAudioIn::vDbusNameAppearedCb(GDBusConnection *poconnection, const gchar *name,
                                                 const gchar *name_owner, gpointer pUserData)
{
    ETG_TRACE_USR1(("spi_tclAudioIn::vDbusNameAppearedCb entered"));
    SPI_INTENTIONALLY_UNUSED(name_owner)
    spi_tclAudioIn* poAudioIn = static_cast<spi_tclAudioIn *>(pUserData);
    SPI_NORMAL_ASSERT(NULL == poAudioIn);
    if(NULL != poAudioIn)
    {
       GError *poGError = NULL;
       poAudioIn->m_poEcnrService = org_bosch_ecnr_service_proxy_new_sync( poconnection,
                                                  G_DBUS_PROXY_FLAGS_NONE,
                                                  name,
                                                  ECNR_OBJECT_PATH,
                                                  NULL,
                                                  &poGError);
       if(NULL == poAudioIn->m_poEcnrService)
       {
          ETG_TRACE_USR1(("spi_tclAudioIn::failed to create ECNR service proxy"));
          if(NULL != poGError)
          {
              ETG_TRACE_USR1(("proxy error message = %s",poGError->message));
          }
       }
       g_error_free(poGError);

       poAudioIn->m_poEcnrAlsa = org_bosch_ecnr_alsa_proxy_new_sync( poconnection,
                                                  G_DBUS_PROXY_FLAGS_NONE,
                                                  name,
                                                  ECNR_OBJECT_PATH,
                                                  NULL,
                                                  &poGError);
       if(NULL == poAudioIn->m_poEcnrAlsa)
       {
          ETG_TRACE_USR1(("spi_tclAudioIn::failed to create ECNR ALSA service proxy"));
          if(NULL != poGError)
          {
              ETG_TRACE_USR1(("proxy error message = %s",poGError->message));
          }
       }
       g_error_free(poGError);
    }
}

/***************************************************************************
 ** FUNCTION:  spi_tclAudioIn::vDbusNameVanishedCb()
 ***************************************************************************/
t_Void spi_tclAudioIn::vDbusNameVanishedCb(GDBusConnection *poconnection, const gchar *name, gpointer pUserData)
{
   ETG_TRACE_USR1(("spi_tclAudioIn::vDbusNameVanishedCb entered"));
   SPI_INTENTIONALLY_UNUSED(poconnection)
   SPI_INTENTIONALLY_UNUSED(name)
   spi_tclAudioIn* poAudioIn = static_cast<spi_tclAudioIn *>(pUserData);
   SPI_NORMAL_ASSERT(NULL == poAudioIn);
   if(NULL != poAudioIn)
   {
      poAudioIn->m_poEcnrService = NULL;
      poAudioIn->m_poEcnrAlsa = NULL;
   }
}


/***************************************************************************
 ** FUNCTION:  spi_tclAudioIn::vInitializeAudioIn
 ***************************************************************************/
t_Void spi_tclAudioIn::vInitializeAudioIn(tenAudioInDataSet enAudioDataSet)
{
   ETG_TRACE_USR1(("spi_tclAudioIn::vInitializeAudioIn: enAudioDataSet = %d",
            ETG_ENUM(AUDIO_ECNR_DATASET, enAudioDataSet)));
   //! changed to synchronous interface of ecnr
   if (NULL != m_poEcnrService)
   {
      GError *poError = NULL;
      t_Bool bResult = org_bosch_ecnr_service_call_ecnr_initialize_sync(
            m_poEcnrService, cocECNRAppID,  cou32SPIDatasetOffset + enAudioDataSet, NULL, &poError);

      ETG_TRACE_USR4(("org_bosch_ecnr_service_ecnr_initialize result = %d ", bResult));
      if (NULL != poError)
      {
         ETG_TRACE_ERR(("spi_tclAudioIn::vInitializeAudioIn:  Error = %s ",
                     poError->message));
         g_error_free (poError);
      }
   }
   ETG_TRACE_USR4(("spi_tclAudioIn::vInitializeAudioIn left "));
}

/***************************************************************************
 ** FUNCTION:  spi_tclAudioIn::vUnitializeAudioIn
 ***************************************************************************/
t_Void spi_tclAudioIn::vUnitializeAudioIn()
{
   ETG_TRACE_USR1(("spi_tclAudioIn::vUnitializeAudioIn entered "));
   //! changed to synchronous interface of ecnr
   if (NULL != m_poEcnrService)
   {
      GError *poError = NULL;
      t_Bool bResult =  org_bosch_ecnr_service_call_ecnr_destroy_sync(
            m_poEcnrService, cocECNRAppID, NULL, &poError);
      ETG_TRACE_USR4(("org_bosch_ecnr_service_ecnr_destroy result = %d ", bResult));

      if (NULL != poError)
      {
         ETG_TRACE_ERR(("spi_tclAudioIn::vUnitializeAudioIn:  Error = %s ",
                     poError->message));
         g_error_free (poError);
      }
   }
   ETG_TRACE_USR4(("spi_tclAudioIn::vUnitializeAudioIn left "));

}

/***************************************************************************
 ** FUNCTION:  spi_tclAudioIn::vStartAudioIn
 ***************************************************************************/
t_Void spi_tclAudioIn::vStartAudioIn()
{
   ETG_TRACE_USR1(("spi_tclAudioIn::vStartAudioIn entered "));
   //! As suggested by ECNR team, synchronous blocking call is used for start audio.
   //! @Note: Please don't pass the ecnr startMode as 0 for synchronous call as this would block for ~1 minute.
   //! instead use the asynchrnous interface if startMode has to be set to 0
   if (NULL != m_poEcnrService)
   {
      GError *poError = NULL;
      t_Bool bResult = org_bosch_ecnr_service_call_ecnr_start_audio_ext_sync(
            m_poEcnrService, cocECNRAppID, cocECNRStartMode, NULL, &poError);

      ETG_TRACE_USR4(("org_bosch_ecnr_service_ecnr_start_audio_ext result = %d ", bResult));
      if (NULL != poError)
      {
         ETG_TRACE_ERR(("spi_tclAudioIn::vStartAudioIn:  Error = %s ",
                     poError->message));
         g_error_free (poError);
      }
   }
   ETG_TRACE_USR4(("spi_tclAudioIn::vStartAudioIn left "));
}

/***************************************************************************
 ** FUNCTION:  spi_tclAudioIn::vStopAudioIn
 ***************************************************************************/
t_Void spi_tclAudioIn::vStopAudioIn()
{
   ETG_TRACE_USR1(("spi_tclAudioIn::vStopAudioIn entered "));
   //! changed to synchronous interface of ecnr
   if (NULL != m_poEcnrService)
   {
      GError *poError = NULL;
      t_Bool bResult = org_bosch_ecnr_service_call_ecnr_stop_audio_sync(m_poEcnrService, cocECNRAppID, NULL, &poError);

      ETG_TRACE_USR4(("org_bosch_ecnr_service_ecnr_stop_audio result = %d ", bResult));
      if (NULL != poError)
      {
         ETG_TRACE_ERR(("spi_tclAudioIn::vStopAudioIn:  Error = %s ", poError->message));
         g_error_free (poError);
      }
   }
   ETG_TRACE_USR4(("spi_tclAudioIn::vStopAudioIn left"));

}

/***************************************************************************
 ** FUNCTION:  spi_tclAudioIn::vSetAudioInConfig
 ***************************************************************************/
t_Void spi_tclAudioIn::vSetAudioInConfig(tenAudioInDataSet enAudioDataSet)
{
   ETG_TRACE_USR1((" vSetAudioInConfig enAudioDataSet = %d ", ETG_ENUM(AUDIO_ECNR_DATASET, enAudioDataSet)));

   //! changed to synchronous interface of ecnr
   GError *poError = NULL;
    t_Bool bResult = org_bosch_ecnr_service_call_ecnr_set_configuration_sync(m_poEcnrService, cocECNRAppID,cou32SPIDatasetOffset + enAudioDataSet, NULL, &poError);

   ETG_TRACE_USR4(("org_bosch_ecnr_service_ecnr_set_configuration result = %d ", bResult));
   if (NULL != poError)
   {
      ETG_TRACE_ERR(("spi_tclAudioIn::vSetAudioInConfig:  Error = %s ", poError->message));
      g_error_free (poError);
   }

   ETG_TRACE_USR4(("spi_tclAudioIn::vSetAudioInConfig left "));
}

/***************************************************************************
 ** FUNCTION:  spi_tclAudioIn::vSetWBAudio
 ***************************************************************************/
t_Void spi_tclAudioIn::vSetWBAudio(t_Bool bSettoWB)
{
   ETG_TRACE_USR1((" %s Not Implemented  ", __PRETTY_FUNCTION__));
   SPI_INTENTIONALLY_UNUSED(bSettoWB);
   //! TODO
}


/***************************************************************************
 ** FUNCTION:  spi_tclAudioIn::vSetWBAudio
 ***************************************************************************/
t_Void spi_tclAudioIn::vSetAlsaDevice(t_U32 u32AlsaDeviceSelector, t_String szAlsaDeviceName)
{
   ETG_TRACE_USR1(("spi_tclAudioIn::vSetAlsaDevice entered "));

   if ((NULL != m_poEcnrAlsa ) && (false == szAlsaDeviceName.empty()))
   {
      ETG_TRACE_USR4((" ECNR ALSA Device %s ", szAlsaDeviceName.c_str()));
      ETG_TRACE_USR4((" ECNR ALSA Device Selector %d ", u32AlsaDeviceSelector));
      //! Solution for  SUZUKI-25184 
      //! As suggested by ECNR team, synchronous blocking call is used for setting alsa device.
      //! If StartAudio is called as blocking and vSetAlsaDevice as asynchronous call the calls
      //! can reach ecnr in reverse order. Hence changing vSetAlsaDevice also to blocking call
      GError *poError = NULL;
       t_Bool bResult =org_bosch_ecnr_alsa_call_ecnr_alsa_set_device_sync(
            m_poEcnrAlsa, cocECNRAppID, u32AlsaDeviceSelector, szAlsaDeviceName.c_str(), NULL, &poError);

      ETG_TRACE_USR4(("org_bosch_ecnr_alsa_ecnr_alsa_set_device result = %d ", bResult));
      if (NULL != poError)
      {
         ETG_TRACE_ERR(("spi_tclAudioIn::vSetAlsaDevice:  Error = %s ",
                     poError->message));
         g_error_free (poError);
      }
   }//End of if (false == szAlsaDeviceName.empty())

   ETG_TRACE_USR4(("spi_tclAudioIn::vSetAlsaDevice left "));
}








