/***********************************************************************/
/*!
 * \file  spi_tclMySPINCmdAudio.h
 * \brief  Implementation of the Class spi_tclMySPINCmdAudio
 *************************************************************************
 \verbatim


 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:
 AUTHOR:         tch5kor
 COPYRIGHT:      &copy; 2015 Robert Bosch Car Multimedia GmbH
 HISTORY:
 Date        | Author                | Modification
 12.04.2016  | Chaitra Srinivasa     | Trace message cleanup
 \endverbatim
 *************************************************************************/

/******************************************************************************
 | includes:
 | 1)system- and project- includes
 | 2)needed interfaces from external components
 | 3)internal and external interfaces from this component
 |----------------------------------------------------------------------------*/

#include "mySPINTypes.h"
#include "spi_tclMySPINManager.h"
#include "spi_tclMySPINCmdAudio.h"
#include "spi_tclMySPINDbusHandler.h"
#include "spi_tclMySPINTADataIntf.h"
#include "spi_tclMySPINTAAudioIntf.h"
#include "spi_tclMySPINAudioDispatcher.h"
#include "spi_tclMySPINMsgQInterface.h"
#include <unistd.h>

#include "Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_AUDIO
#include "trcGenProj/Header/spi_tclMySPINCmdAudio.cpp.trc.h"
#endif

static const int A2DP_SET = 2;
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e746 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e515 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e516 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/
/******************************************************************************
 | defines and macros (scope: global)
 |----------------------------------------------------------------------------*/
/******************************************************************************
 | variable definition (scope: global)
 |----------------------------------------------------------------------------*/
/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/

static const t_String coszUAC2DevName = "plughw:UAC2Gadget,0";
static const t_String coszUAC2DevNameForVR = "plughw:UAC2Gadget,0";

/***************************************************************************
 ** FUNCTION:  spi_tclMySPINCmdAudio::spi_tclMySPINCmdAudio()
 ***************************************************************************/
spi_tclMySPINCmdAudio::spi_tclMySPINCmdAudio() :
   m_poMySPINDbusHandler(NULL), m_poAudioStreamer(NULL), 
   m_poVrStreamer(NULL),m_BTDevConStat(NULL),m_u32DeviceId(0),
   m_bIsMainAudioActive(false),m_bIsMixAudioActive(false),
   m_enA2DPStatus(e8MSPIN_A2DP_UNKNOWN),m_bA2DPStreamingStatus(false)
{
   ETG_TRACE_USR1(("Create CmdAudio for mySPIN "));

   m_poMySPINDbusHandler = new spi_tclMySPINDbusHandler();
   //Initialize DBUS handler
   if (NULL != m_poMySPINDbusHandler)
   {
      m_poMySPINDbusHandler->vInitialize();
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclMySPINCmdAudio::~spi_tclMySPINCmdAudio()
 ***************************************************************************/
spi_tclMySPINCmdAudio::~spi_tclMySPINCmdAudio()
{
   m_BtMacAddressMap.clear();
   delete m_poMySPINDbusHandler;
   m_poVrStreamer = NULL;
   m_poAudioStreamer = NULL;
   m_u32DeviceId = 0;
   m_bIsMainAudioActive = false;
   m_bIsMixAudioActive = false;
   m_enA2DPStatus = e8MSPIN_A2DP_UNKNOWN;
   m_bA2DPStreamingStatus = false;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclMySPINCmdAudio::bInitialize()
 ***************************************************************************/
t_Bool spi_tclMySPINCmdAudio::bInitialize(t_U32 u32DeviceId, tenDeviceSubCategory enDeviceSubCategory,
         tenAudioDir enAudioDir, tvBluetoothConStat fvBluetoothConStat)
{

   ETG_TRACE_USR2(("[FUNC]::spi_tclMySPINCmdAudio::bInitialize : Dev-0x%x, Audio Dir [%d]", u32DeviceId, ETG_ENUM(SPI_AUDIO_DIRECTION,
            enAudioDir)));
   // Get Source for Audio based on the device subcategory.
   t_Bool bInitStatus = false;
   audioStreamDeviceType enAudioStreamDeviceType = audioStreamDeviceType::AUDIO_DEVICE_TYPE_BT;
   trMySpinAudioConfig rMySpinAudioConfig;

   m_BTDevConStat = fvBluetoothConStat;
   m_u32DeviceId = u32DeviceId;

   // if ((e8AUD_VR_IN == enAudioDir) && (e8DEVTYPE_IAP == enDeviceSubCategory))
   // {

      // //Set VR Input device configuration
      // m_rVRInputDevConfig.setBufferSizeAndPeriodSize = false;
      // //deviceName will be populated later
      // m_rVRInputDevConfig.blocking = false;

      // //Set VR Playback device config
      // m_rVRPlayBackDevConfig.setBufferSizeAndPeriodSize = false;
      // m_rVRPlayBackDevConfig.deviceName = coszUAC2DevNameForVR;
      // m_rVRPlayBackDevConfig.blocking = false;
      // // m_rVRPlayBackDevConfig.bufferSize = 256;
      // // m_rVRPlayBackDevConfig.periodSize = 64;

      // bInitStatus = true;
   // }
   // else
   // {
      switch (enDeviceSubCategory)
      {
         case e8DEVTYPE_AOAP:
         {
            ETG_TRACE_USR2(("[DESC]::For Device type [%d] Initialize Audio Dir [%d]", u32DeviceId, ETG_ENUM(SPI_AUDIO_DIRECTION,
                     enAudioDir)));
            // Get the source from ALPS stack using DBUS Handler
            m_macAddLock.s16Lock();
            if ((m_BtMacAddressMap.end() != m_BtMacAddressMap.find(u32DeviceId)) && (NULL != m_poMySPINDbusHandler))
            {
               bInitStatus = m_poMySPINDbusHandler->bStartA2DPAudioPipe(m_BtMacAddressMap[u32DeviceId]);

               ETG_TRACE_USR4(("[PARAM]::bInitialize-A2DP Audio Pipe Start Status [%d]", ETG_ENUM(BOOL, bInitStatus)));
               enAudioStreamDeviceType = audioStreamDeviceType::AUDIO_DEVICE_TYPE_BT;

               m_poMySPINDbusHandler->vGetConfiguration(rMySpinAudioConfig);
            }
            m_macAddLock.vUnlock();
         }
            break;
         case e8DEVTYPE_IAP:
         {
            //Get the PCM device name from UAC2 gadget
            enAudioStreamDeviceType = audioStreamDeviceType::AUDIO_DEVICE_TYPE_ALSA_W_SNDCTL;

            rMySpinAudioConfig.szInputDev = coszUAC2DevName;
            rMySpinAudioConfig.bHasIndianNess = false;
            rMySpinAudioConfig.bIsSigned = true;
            rMySpinAudioConfig.u8SampleDepth = 16;
            rMySpinAudioConfig.u8SampleWidth = 16;
            rMySpinAudioConfig.u8Channels = 2;
            rMySpinAudioConfig.u32sampleRate = 44100;
            bInitStatus = true;
            ETG_TRACE_USR1(("Attributes set"));
         }
            break;
         default:
         {
            ETG_TRACE_ERR(("[ERROR]Invalid device sub category"));
         }
            break;
      }

      if (true == bInitStatus)
      {
         ETG_TRACE_USR1(("spi_tclMySPINCmdAudio::bInitialize:Audio Initialization done"));
         //Create the Plugin object now
         m_poAudioStreamer = new MySpinAudioStream();

         if (NULL != m_poAudioStreamer)
         {
            audioStreamCallbacks rAudioCbs;

            //! Populate the callback structure
            rAudioCbs.fvOnAudioStreamError = std::bind(&spi_tclMySPINCmdAudio::vOnAudioStreamErroCb,
                     this,
                     std::placeholders::_1,
                     std::placeholders::_2,
                     std::placeholders::_3);
            rAudioCbs.fvOnAudioStreamRateChange = std::bind(&spi_tclMySPINCmdAudio::vOnAudStreamRateChanedCb,
                     this,
                     std::placeholders::_1,
                     std::placeholders::_2);

            m_poAudioStreamer->registerCallbacks(u32DeviceId, rAudioCbs);

            audioStreamConfig audioStreamConfig;
            vCopyAudioConfig(rMySpinAudioConfig, audioStreamConfig);
            m_poAudioStreamer->setAudioStreamConfig(audioStreamConfig);

            t_S32 s32IErrorCode = m_poAudioStreamer->setAudioStreamInputDevice(rMySpinAudioConfig.szInputDev,
                     enAudioStreamDeviceType);
            bInitStatus = (0 < s32IErrorCode);
         }
      }
   // }

   return bInitStatus;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMySPINCmdAudio::vUnInitialize()
 ***************************************************************************/
t_Void spi_tclMySPINCmdAudio::vUnInitialize(tenDeviceSubCategory enDeviceSubCategory, tenAudioDir enAudioDir)
{
    ETG_TRACE_USR4(("[FUNC]:spi_tclMySPINCmdAudio::vUnInitialize entered"));
   // Get Source for Audio based on the device subcategory.
   if ((e8AUD_VR_IN != enAudioDir) && (NULL != m_poMySPINDbusHandler))
   {
      switch (enDeviceSubCategory)
      {
         case e8DEVTYPE_AOAP:
         {
            // Get the source from ALPS stack using DBUS Handler
            t_Bool bStopStatus = m_poMySPINDbusHandler->bStopA2DPAudioPipe();
            ETG_TRACE_USR4(("[PARAM]::vUnInitialize:A2DP Audio Pipe Start Status [%d]", ETG_ENUM(BOOL, bStopStatus)));

         }
            break;
         case e8DEVTYPE_IAP:
         {

            //Add Code
         }
            break;
         default:
         {
            ETG_TRACE_ERR(("[ERROR]Invalid device sub category"));
         }
            break;
      }
      //Delete Audio Streamer
      RELEASE_MEM(m_poAudioStreamer);
   }
}

/***************************************************************************
 ** FUNCTION: tenAudioStreamError spi_tclMySPINCmdAudio::enStartAudioStreaming()
 ***************************************************************************/
tenAudioStreamError spi_tclMySPINCmdAudio::enStartAudioStreaming(tenAudioDir enAudioDir, t_String szAudioDev)
{
    ETG_TRACE_USR2(("[DESC]:enStartAudioStreaming for : Audio Dir [%d], Audio Device[%s]", ETG_ENUM(SPI_AUDIO_DIRECTION,
             enAudioDir), szAudioDev.c_str()));
    // Set the configurations and
    // Trigger Start of Audio streaming via BSOT Interface.
    tenAudioStreamError enAudioStreamError = e8STREAMERROR;

    //m_szAudioDev = szAudioDev;
    switch (enAudioDir)
    {
        // Currently Myspin VR is handled by BT. Commenting for future use
        // case e8AUD_VR_IN:
        // {
            // m_rVRInputDevConfig.deviceName = "AdevAcousticinSpeech"; //set the input device
            // // Create object for VR Streaming
            // m_poVrStreamer = new MySpinVrStream();
            // ETG_TRACE_USR2(("[DESC]:enStartAudioStreaming object for vr created"));
            // if (NULL != m_poVrStreamer)
            // {
                // vrStreamCallbacks rVRCallbacks;
                // rVRCallbacks.fvOnMySpinVrStreamError = std::bind(&spi_tclMySPINCmdAudio::vOnMySpinVrStreamErrorCb,
                                                                // this,
                                                                // SPI_FUNC_PLACEHOLDERS_2);
                // m_poVrStreamer->registerCallbacks(m_u32DeviceId,rVRCallbacks);
                // t_S32 s32Ret = m_poVrStreamer->startVrStream(m_rVRInputDevConfig,
                                                            // m_rVRPlayBackDevConfig,
                                                            // SND_PCM_ACCESS_MMAP_INTERLEAVED,
                                                            // SND_PCM_FORMAT_S16_LE,
                                                            // 1,
                                                            // 16000);
                // enAudioStreamError = (0 > s32Ret) ? e8STREAMERROR : e8NOERROR;
                // ETG_TRACE_USR4(("[PARAM]:enStartAudioStreaming for : error:[%d]", ETG_ENUM(MYSPIN_AUDIO_STREAM_ERROR,enAudioStreamError)));
            // }
        // }
        // break;
        //TODO add all possible Audio Directions
        case e8AUD_MAIN_OUT:
        {
            if((NULL != m_poAudioStreamer) && (false == m_bIsMixAudioActive))
            {
                t_S32 s32OutErrorCode = m_poAudioStreamer->setAudioStreamOutputDevice(szAudioDev);
                t_S32 s32ErrorCode = m_poAudioStreamer->startAudioStream();
                enAudioStreamError = ((0 > s32OutErrorCode) || (0 > s32ErrorCode)) ? e8STREAMERROR : e8NOERROR;
                m_bIsMainAudioActive = (e8NOERROR == enAudioStreamError) ? true : false;
            }
            else if(true == m_bIsMixAudioActive)
            {
               m_bIsMainAudioActive = true;
               enAudioStreamError = e8NOERROR;
            }
        }
        break;
        case e8AUD_STEREO_MIX_OUT:
        {
            if((NULL != m_poAudioStreamer) && (false == m_bIsMainAudioActive))
            {
                t_S32 s32OutErrorCode = m_poAudioStreamer->setAudioStreamOutputDevice(szAudioDev);
                t_S32 s32ErrorCode = m_poAudioStreamer->startAudioStream();
                enAudioStreamError = ((0 > s32OutErrorCode) || (0 > s32ErrorCode)) ? e8STREAMERROR : e8NOERROR;
                m_bIsMixAudioActive = (e8NOERROR == enAudioStreamError) ? true : false;
            } 
            else if(true == m_bIsMainAudioActive)
            {
               m_bIsMixAudioActive = true;
               enAudioStreamError = e8NOERROR;
            }
        }
        break;
        default:
        {
           ETG_TRACE_ERR(("[ERROR]Audio type not supported"));
        }
        break;
    }

    return enAudioStreamError;
}

/***************************************************************************
 ** FUNCTION: tenAudioStreamError spi_tclMySPINCmdAudio::enStopAudioStreaming()
 ***************************************************************************/
tenAudioStreamError spi_tclMySPINCmdAudio::enStopAudioStreaming(tenAudioDir enAudDir)
{
    ETG_TRACE_USR2(("[FUNC]:spi_tclMySPINCmdAudio::enStopAudioStreaming entered enAudDir = %d",ETG_ENUM(SPI_AUDIO_DIRECTION,enAudDir)));
   // Trigger Start of Audio streaming via BSOT Interface.
   tenAudioStreamError enAudioStreamError = e8STREAMERROR;

    switch (enAudDir)
    {   
        // Currently Myspin VR is handled by BT. Commenting for future use
        // case e8AUD_VR_IN:
        // {
            // if (NULL != m_poVrStreamer)
            // {
                // vrStreamCallbacks rVRCallbacks;
                // rVRCallbacks.fvOnMySpinVrStreamError = NULL;
                // m_poVrStreamer->registerCallbacks(m_u32DeviceId,rVRCallbacks); //clear the callback
                // t_S32 s32ErrorCode = m_poVrStreamer->stopVrStream();
                // enAudioStreamError = (0 > s32ErrorCode) ? e8STREAMERROR : e8NOERROR;
                // delete m_poVrStreamer;
            // }
        // }
        // break;
        case e8AUD_MAIN_OUT:
        {
            if ((NULL != m_poAudioStreamer) && (false == m_bIsMixAudioActive))
            {
                ETG_TRACE_USR2(("[DESC]:Stopping Audio for [%d] ", ETG_ENUM(SPI_AUDIO_DIRECTION, enAudDir)));
                t_S32 s32ErrorCode = m_poAudioStreamer->stopAudioStream();
                enAudioStreamError = (0 > s32ErrorCode) ? e8STREAMERROR : e8NOERROR;
                m_bIsMainAudioActive = false;
            }
        }
        break;
        case e8AUD_STEREO_MIX_OUT:
        {
            if ((NULL != m_poAudioStreamer) && (false == m_bIsMainAudioActive))
            {
                ETG_TRACE_USR2(("[DESC]:Stopping Audio for [%d] ", ETG_ENUM(SPI_AUDIO_DIRECTION, enAudDir)));
                t_S32 s32ErrorCode = m_poAudioStreamer->stopAudioStream();
                enAudioStreamError = (0 > s32ErrorCode) ? e8STREAMERROR : e8NOERROR;
                m_bIsMixAudioActive = false;
            }
        }
        break;
        default:
        {
           ETG_TRACE_ERR(("[ERROR]Audio type not supported"));
        }
        break;
    }
    return enAudioStreamError;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINCmdAudio::vCopyAudioConfig()
 ***************************************************************************/
t_Void spi_tclMySPINCmdAudio::vCopyAudioConfig(trMySpinAudioConfig rMySpinAudioConfig,
         audioStreamConfig& rAudStreamConfig)
{
   rAudStreamConfig.codecName = rMySpinAudioConfig.szEncoding;
   rAudStreamConfig.sampleRate = rMySpinAudioConfig.u32sampleRate;
   rAudStreamConfig.sampleDepth = rMySpinAudioConfig.u8SampleDepth;
   rAudStreamConfig.sampleWidth = rMySpinAudioConfig.u8SampleWidth;
   rAudStreamConfig.sampleEndianness = rMySpinAudioConfig.bHasIndianNess;
   rAudStreamConfig.channels = rMySpinAudioConfig.u8Channels;
   rAudStreamConfig.sampleSignedness = rMySpinAudioConfig.bIsSigned;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINCmdAudio::vOnAudioStreamErroCb()
 ***************************************************************************/
t_Void spi_tclMySPINCmdAudio::vOnAudioStreamErroCb(t_U32 u32Context, audioStreamError enError, t_String szError)
{
   SPI_INTENTIONALLY_UNUSED(enError)
   ETG_TRACE_ERR(("[ERR]:Audio Stream Error for Device [%d],[%s]", u32Context, szError.c_str()));
   //TODO in case error occurs DeAllocate audio stop the streaming.
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINCmdAudio::vOnAudStreamRateChanedCb()
 ***************************************************************************/
t_Void spi_tclMySPINCmdAudio::vOnAudStreamRateChanedCb(t_U32 u32Context, t_U32 u32SampleRate)
{
   ETG_TRACE_USR2(("[DESC]:Stream Rate Changed for Device  : Dev-0x%x, New Sample rate[%d]", u32Context, u32SampleRate));
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINCmdAudio::vOnMySpinVrStreamErrorCb()
 ***************************************************************************/
t_Void spi_tclMySPINCmdAudio::vOnMySpinVrStreamErrorCb(t_U32 u32Context, vrStreamError enVRError)
{
   ETG_TRACE_USR2(("[FUNC]:spi_tclMySPINCmdAudio::vOnMySpinVrStreamErrorCb [%d],[%s] : ", u32Context, enVRError));
   MySPINVRStreamErrorMsg oSPINVRStreamErrorMsg;
   oSPINVRStreamErrorMsg.m_enError = static_cast<tenMySPINVRStreamError>(enVRError);

   trMySPINContext* rMySPINContext = (trMySPINContext*)u32Context;
   oSPINVRStreamErrorMsg.vSetDeviceHandle(rMySPINContext->u32DeviceHandle);

   spi_tclMySPINMsgQInterface *poMsgQinterface = spi_tclMySPINMsgQInterface::getInstance();

   if (NULL != poMsgQinterface)
   {
      poMsgQinterface->bWriteMsgToQ(&oSPINVRStreamErrorMsg, sizeof(oSPINVRStreamErrorMsg));
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINCmdAudio::vSetBtMacAddress()
 ***************************************************************************/
t_Void spi_tclMySPINCmdAudio::vSetBtMacAddress(t_U32 u32DeviceID, t_String rfszMACAddress)
{
   ETG_TRACE_USR2(("[DESC]:SetBtMacAddress for : Dev-0x%x, MAC Address [%s]", u32DeviceID, rfszMACAddress.c_str()));

   m_macAddLock.s16Lock();

   //Clear the map in unitialize
   m_BtMacAddressMap[u32DeviceID] = rfszMACAddress;
   m_macAddLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclMySPINCmdAudio::bSendAVPCtrlCmd()
 ***************************************************************************/
t_Bool spi_tclMySPINCmdAudio::bSendAVPCtrlCmd(t_U8 u8KeyCode, t_U8 szEventType)
{
   ETG_TRACE_USR1(("spi_tclMySPINCmdAudio::bSendAVPCtrlCmd:bSendAVPCtrlCmd"));
   t_Bool bRet = false;
   if (NULL != m_poMySPINDbusHandler)
   {
      bRet = m_poMySPINDbusHandler->bSendAVPCtrlCmd(u8KeyCode, szEventType);
   }
   return bRet;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclMySPINCmdAudio::bGetBTConnectionStatus()
 ***************************************************************************/
t_Bool spi_tclMySPINCmdAudio::bGetBTConnectionStatus(t_U32 u32DeviceID)
{
   t_Bool bIsBTConnected = false;

   m_btStatusLock.s16Lock();
   std::map<t_U32, t_Bool>::iterator itrBtConn;
   itrBtConn = m_BtConnStatusMap.find(u32DeviceID);

   if (m_BtConnStatusMap.end() != itrBtConn)
   {
      bIsBTConnected = itrBtConn->second;
   }
   m_btStatusLock.vUnlock();

   ETG_TRACE_USR2(("[DESC]:Is device Connnected via Bluetooth [%d] ", ETG_ENUM(BOOL, bIsBTConnected)));

   return bIsBTConnected;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINCmdAudio::bClearBTDevices()
 ***************************************************************************/
t_Void spi_tclMySPINCmdAudio::vClearBTDevices(t_U32 u32DeviceID)
{
   ETG_TRACE_USR1(("spi_tclMySPINCmdAudio::bClearBTDevices:Clear BT devices "));
   m_macAddLock.s16Lock();

   //Clear the map in unitialize
   m_BtMacAddressMap.erase(u32DeviceID);
   m_poMySPINDbusHandler->vClearDBUSProxies();
   m_macAddLock.vUnlock();
   
   m_bIsMainAudioActive = false;
   m_bIsMixAudioActive = false;
   m_enA2DPStatus = e8MSPIN_A2DP_UNKNOWN;
   m_bA2DPStreamingStatus = false;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINCmdAudio::vSetBluetoothStatus()
 ***************************************************************************/
t_Void spi_tclMySPINCmdAudio::vSetBluetoothStatus(t_U32 u32DeviceID, t_Bool bIsA2DPConn, 
                                                  tenA2DPStatus enA2DPStatus)
{
   ETG_TRACE_USR2(("[DESC]:Set BT Connection status for device [0x%x]", u32DeviceID));

   //Check if at least A2DP is connected, which is required for main audio
   // If bit pas [1] pos is set ?

   m_btStatusLock.s16Lock();
   m_BtConnStatusMap[u32DeviceID] = bIsA2DPConn;
   m_btStatusLock.vUnlock();
  
   //Based on connection status, trigger callbacks.
   m_oAudioCmdMapLock.s16Lock();
   ETG_TRACE_USR2(("[DESC]:BT Connection status[%d]",ETG_ENUM(BOOL,bIsA2DPConn)));
   if ((false == bIsA2DPConn) && (NULL != m_BTDevConStat))
   {
      // If BT Got disconnected trigger callback
      (m_BTDevConStat)(u32DeviceID);
   }
   else if (true == bIsA2DPConn)
   {
      std::list<trMySPINAudioCmd>::iterator itrCmdsList;
       for(itrCmdsList = m_AudioCmdsList.begin();m_AudioCmdsList.end()!= itrCmdsList;itrCmdsList++)
       {   
           ETG_TRACE_USR4(("vSetBluetoothStatus::Audio state:[%d], Audio dir:[%d]", ETG_ENUM(MYSPIN_AUDIO_CMD_STATE,itrCmdsList->enState), ETG_ENUM(SPI_AUDIO_DIRECTION,itrCmdsList->enAudioDir)));
           if (((e8_STATE_ACTIVE == itrCmdsList->enState) || (e8_STATE_INACTIVATED == itrCmdsList->enState)) 
               && (e8AUD_MAIN_OUT == itrCmdsList->enAudioDir)
               && (enA2DPStatus != m_enA2DPStatus))
           { 
                //Request for Audio Channels.
                if(NULL != m_rmySPINAudioCbs.fvRequestLaunchAudioCb)
                {
                    trMySPINAudioCmd rMySPINAudioCmd = *itrCmdsList;
                    m_rmySPINAudioCbs.fvRequestLaunchAudioCb(rMySPINAudioCmd);
                }
           }
       }
   }
   m_enA2DPStatus = enA2DPStatus;
   m_oAudioCmdMapLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINCmdAudio::vRegisterAudioCbs()
 ***************************************************************************/
t_Void spi_tclMySPINCmdAudio::vRegisterAudioCbs(t_U32 u32DeviceID,trmySPINAudioCbs rmySPINAudioCbs,
                               trmySPINAudioDuckCb rmySPINAudioDuckCb)
{
   ETG_TRACE_USR2(("[DESC]:Register audio callbacks for [0x%x]", u32DeviceID));

   spi_tclMySPINTADataIntf oTADataIntf;
   spi_tclMySPINTAAudioIntf* poMySPINAudioIntf = oTADataIntf.poGetTAAudioIntfInst();

   if (NULL != poMySPINAudioIntf)
   {
      trMYSPINAudioCbs rMYSPINAudioCbs;
      rMYSPINAudioCbs.vAudioRequestCb = &spi_tclMySPINCmdAudio::vAudioRequestCb;
      poMySPINAudioIntf->vRegisterAudioCbs(u32DeviceID,rMYSPINAudioCbs);

   }

   //Register for callbacks with mySPIN Audio to handle advacned audio requests.
   m_rmySPINAudioCbs = rmySPINAudioCbs;
   m_rmySPINAudioDuckCb = rmySPINAudioDuckCb;
}
/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINCmdAudio::vAudioRequestCb()
 ***************************************************************************/
t_Void spi_tclMySPINCmdAudio::vAudioRequestCb(t_Void *context, MSPIN_AUDIOCONTROL command, t_U32 requestID,
         MSPIN_AUDIOTYPE type)
{
   ETG_TRACE_USR2(("spi_tclMySPINCmdAudio::vAudioRequestCb entered command=%d type=%d requestID=%d",command,type,requestID));
   if (NULL != context)
   {
      //Process the request only
      MySPINAudioCmdMsg oMySPINAudioCmdMsg;
      oMySPINAudioCmdMsg.m_enCommand = static_cast<tenAudioCtrlCommand>(command);
      oMySPINAudioCmdMsg.m_u32RequestID = requestID;
      oMySPINAudioCmdMsg.m_enAudioType = static_cast<tenMySPINAudioType>(type);

      trMySPINContext* rMySPINContext = (trMySPINContext*)context;
      oMySPINAudioCmdMsg.vSetDeviceHandle(rMySPINContext->u32DeviceHandle);

      spi_tclMySPINMsgQInterface *poMsgQinterface = spi_tclMySPINMsgQInterface::getInstance();

      if (NULL != poMsgQinterface)
      {
         poMsgQinterface->bWriteMsgToQ(&oMySPINAudioCmdMsg, sizeof(oMySPINAudioCmdMsg));
      }
   }
}
/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINCmdAudio::vHandleAudioCmd()
 ***************************************************************************/
t_Void spi_tclMySPINCmdAudio::vHandleAudioCmd(trMySPINAudioCmd rMySPINAudioCmd)
{
   ETG_TRACE_USR2(("spi_tclMySPINCmdAudio::vHandleAudioCmd entered"));
   //In case the request is accepted then send to upper layers
   if (e8_REQUEST_REJECT == rMySPINAudioCmd.enAudioResponse)
   {
      vSendAudioRespose(rMySPINAudioCmd.u32DeviceHandle, rMySPINAudioCmd.u32RequestID, rMySPINAudioCmd.enAudioResponse);
   }
   else if(e8_AUDIOTYPE_IGNORE == rMySPINAudioCmd.enAudioType && e8COMMAND_RELEASE == rMySPINAudioCmd.enCommand)
   {
      ETG_TRACE_USR2(("spi_tclMySPINCmdAudio::vHandleAudioCmd ignore the release request"));
      rMySPINAudioCmd.enAudioResponse = e8_REQUEST_CLOSE;
      vSendAudioRespose(rMySPINAudioCmd.u32DeviceHandle, rMySPINAudioCmd.u32RequestID, rMySPINAudioCmd.enAudioResponse);
      m_bIsMainAudioActive = false;
      m_bIsMixAudioActive = false;
   }
   else if(e8COMMAND_REQUEST == rMySPINAudioCmd.enCommand)
   {
      vHandleAudioSourceCtrlRequest(rMySPINAudioCmd);
   }
   else
   {
      vHandleAudioSourceCtrlRelease(rMySPINAudioCmd);
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINCmdAudio::vSendAudioRespose()
 ***************************************************************************/
t_Void spi_tclMySPINCmdAudio::vSendAudioRespose(t_U32 u32DeviceHandle, t_U32 u32RequestID,
         tenAudioResponse enAudioResponse)
{

   spi_tclMySPINTADataIntf oMySPINTADataIntf;
   spi_tclMySPINTAAudioIntf* poAudioIntf = oMySPINTADataIntf.poGetTAAudioIntfInst();

   if (NULL != poAudioIntf)
   {
      poAudioIntf->vSetAudioStatus(u32DeviceHandle,u32RequestID,enAudioResponse);
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINCmdAudio::vHandleAudioSourceCtrlRequest()
 ***************************************************************************/
t_Void spi_tclMySPINCmdAudio::vHandleAudioSourceCtrlRequest(trMySPINAudioCmd& rfrMySPINAudioCmd)
{
   ETG_TRACE_USR1(("spi_tclMySPINCmdAudio::vHandleAudioSourceCtrlRequest entered for device ID: %d",rfrMySPINAudioCmd.u32DeviceHandle));

   tenDeviceSubCategory enDeviceSubCategory = e8DEVTYPE_UNKNWON;
   spi_tclMySPINManager::vGetDeviceSubCategory(enDeviceSubCategory);

   if(e8DEVTYPE_AOAP == enDeviceSubCategory)
   {
       //For android devices check if A2DP streaming is already active outside SPI
       m_macAddLock.s16Lock();
       if ((m_BtMacAddressMap.end() != m_BtMacAddressMap.find(rfrMySPINAudioCmd.u32DeviceHandle)) && (NULL != m_poMySPINDbusHandler))
       {
           m_bA2DPStreamingStatus = m_poMySPINDbusHandler->bGetA2DPStreamingStatus(m_BtMacAddressMap[rfrMySPINAudioCmd.u32DeviceHandle]);
       }
       m_macAddLock.vUnlock();
       if(m_bA2DPStreamingStatus && (!m_bIsMainAudioActive) && (!m_bIsMixAudioActive) )
       {
           //A2DP streaming is active and neither main audio nor mix channel is active in SPI
           ETG_TRACE_USR1(("[PARAM]::vHandleAudioSourceCtrlRequest A2DP streaming is already active outside SPI"));
           rfrMySPINAudioCmd.enAudioResponse = e8_REQUEST_OPEN;
           /*rfrMySPINAudioCmd.enState = e8_STATE_PRE_ACTIVATED;
           m_oAudioCmdMapLock.s16Lock();
           m_AudioCmdsList.push_back(rfrMySPINAudioCmd);
           m_oAudioCmdMapLock.vUnlock();
           vHandlePreActivedRequests();*/
           vSendAudioRespose(rfrMySPINAudioCmd.u32DeviceHandle,rfrMySPINAudioCmd.u32RequestID,rfrMySPINAudioCmd.enAudioResponse);
           //No more processing needed, just return
           return;
       }
   }

   t_Bool bProcessPreActivatedReq = false;

   std::list<trMySPINAudioCmd>::iterator itrCmdsList;
   rfrMySPINAudioCmd.enState = e8_STATE_PROCESSING;

   m_oAudioCmdMapLock.s16Lock();
   ETG_TRACE_USR1(("[PARAM]::vHandleAudioSourceCtrlRequest size: %d",m_AudioCmdsList.size()));
   for(itrCmdsList = m_AudioCmdsList.begin();m_AudioCmdsList.end()!= itrCmdsList;)
   {
      ETG_TRACE_USR2(("[PARAM]::vHandleAudioSourceCtrlRequest::RequestID = %d state = %d",itrCmdsList->u32RequestID, ETG_ENUM(MYSPIN_AUDIO_CMD_STATE,itrCmdsList->enState)));
      if (( itrCmdsList->enAudioType == rfrMySPINAudioCmd.enAudioType)
              && ((e8_STATE_PRE_ACTIVATED == itrCmdsList->enState)
              || (e8_STATE_ACTIVE == itrCmdsList->enState) 
              || (e8_STATE_SUSPENDED == itrCmdsList->enState) /*Check for suspended state is added to request for launch audio in case audio needs to be resumed for instance like BT reconnected*/
              || (e8_STATE_INACTIVATED == itrCmdsList->enState)
              || (e8_STATE_PROCESSING == itrCmdsList->enState))) /* This check was added in order to remove the requests that were in processing state when new requests for same audio type is received*/
      {
          if((e8_STATE_SUSPENDED != itrCmdsList->enState) && (e8_STATE_PROCESSING != itrCmdsList->enState))
          {
             //Found an ACTIVE request, consider this request as active as well
             rfrMySPINAudioCmd.enState = e8_STATE_PRE_ACTIVATED;
             rfrMySPINAudioCmd.enAudioResponse = e8_REQUEST_OPEN;
             bProcessPreActivatedReq = true;
          }
         itrCmdsList = m_AudioCmdsList.erase(itrCmdsList); //delete this request.
         ETG_TRACE_USR2(("[DESC]: Request Audio Source type is already activated"));
      }
      else
      {
          ++itrCmdsList;
      }
   }
   ETG_TRACE_USR1(("[PARAM]::vHandleAudioSourceCtrlRequest size: %d",m_AudioCmdsList.size()));
   // - If not even one is active then, SPI is waiting response from Audio Manager, put the request in PROCESSING.
   // - When Audio Manager allocates for previous request respond to all requests of the Type in PROCESSING state.
   // Add to a list
   m_AudioCmdsList.push_back(rfrMySPINAudioCmd);

   m_oAudioCmdMapLock.vUnlock();

   //In case the audio type is already active handle them
   if (true == bProcessPreActivatedReq)
   {
      vHandlePreActivedRequests();
   }
   else
   {
       //Request for Audio Channels.  
       if(e8AUD_MAIN_OUT == rfrMySPINAudioCmd.enAudioDir)
       {
           if((NULL != m_rmySPINAudioCbs.fvRequestLaunchAudioCb) && (false == m_bIsMixAudioActive))
           {
               m_rmySPINAudioCbs.fvRequestLaunchAudioCb(rfrMySPINAudioCmd);
           }
           else
           {
               vSendAudioRespose(rfrMySPINAudioCmd.u32DeviceHandle, rfrMySPINAudioCmd.u32RequestID, e8_REQUEST_REJECT);
           }
       }
       else if(e8AUD_STEREO_MIX_OUT == rfrMySPINAudioCmd.enAudioDir)
       {
           if((NULL != m_rmySPINAudioCbs.fvRequestLaunchAudioCb) && (false == m_bIsMainAudioActive))
           {
               m_rmySPINAudioCbs.fvRequestLaunchAudioCb(rfrMySPINAudioCmd);
           }
           else
           {
               vSendAudioRespose(rfrMySPINAudioCmd.u32DeviceHandle, rfrMySPINAudioCmd.u32RequestID, e8_REQUEST_OPEN);
           }
       }
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINCmdAudio::vHandleAudioSourceCtrlRelease()
 ***************************************************************************/
t_Void spi_tclMySPINCmdAudio::vHandleAudioSourceCtrlRelease(trMySPINAudioCmd& rfrMySPINAudioCmd)
{
    ETG_TRACE_USR1(("spi_tclMySPINCmdAudio::vHandleAudioSourceCtrlRelease entered for device ID: %d , %d",rfrMySPINAudioCmd.u32DeviceHandle, ETG_ENUM(SPI_AUDIO_DIRECTION,rfrMySPINAudioCmd.enAudioDir)));

    tenDeviceSubCategory enDeviceSubCategory = e8DEVTYPE_UNKNWON;
    spi_tclMySPINManager::vGetDeviceSubCategory(enDeviceSubCategory);

    if(e8DEVTYPE_AOAP == enDeviceSubCategory && m_bA2DPStreamingStatus
            && (!m_bIsMainAudioActive) && (!m_bIsMixAudioActive) 
            && (false == bIsAudioRequestAvailable(rfrMySPINAudioCmd)))
    {
         //A2DP streaming is active and neither main audio nor mix channel is active in SPI
         ETG_TRACE_USR1(("[PARAM]::vHandleAudioSourceCtrlRequest A2DP streaming is already active outside SPI"));
         rfrMySPINAudioCmd.enAudioResponse = e8_REQUEST_CLOSE;
         vSendAudioRespose(rfrMySPINAudioCmd.u32DeviceHandle,rfrMySPINAudioCmd.u32RequestID,rfrMySPINAudioCmd.enAudioResponse);
         //No more processing needed, just return
         return;
    }

   std::list<trMySPINAudioCmd>::iterator itrCmdsList;
   tenAudioResponse enAudioResponse = e8_REQUEST_INVALID;
   
   m_oAudioCmdMapLock.s16Lock();
   for (itrCmdsList = m_AudioCmdsList.begin(); m_AudioCmdsList.end() != itrCmdsList;)
   {
       if ((rfrMySPINAudioCmd.enAudioDir == itrCmdsList->enAudioDir) && ((e8_STATE_ACTIVE == itrCmdsList->enState)
           || (e8_STATE_PROCESSING == itrCmdsList->enState) || (e8_STATE_SUSPENDED == itrCmdsList->enState)))
       {
           ETG_TRACE_USR2(("[PARAM]::vHandleAudioSourceCtrlRelease::RequestID = %d state = %d",itrCmdsList->u32RequestID, ETG_ENUM(MYSPIN_AUDIO_CMD_STATE,itrCmdsList->enState)));
           if((e8AUD_STEREO_MIX_OUT == itrCmdsList->enAudioDir) && (true == m_bIsMainAudioActive))   
           {
               itrCmdsList->enState = e8_STATE_CLOSED;
               itrCmdsList->enAudioResponse = e8_REQUEST_CLOSE;
               enAudioResponse = e8_REQUEST_CLOSE;
               vSendAudioRespose(itrCmdsList->u32DeviceHandle, itrCmdsList->u32RequestID, itrCmdsList->enAudioResponse);
               m_bIsMixAudioActive = false;
               // m_rmySPINAudioDuckCb.fvAudioDuckingCb(e8_DUCKINGTYPE_UNDUCK);
               // t_String szAudioDev = "AdevEnt1Out";
               // if((NULL != m_poAudioStreamer) && (m_szAudioDev == szAudioDev))
               // {
                   // t_S32 s32ErrorCode = m_poAudioStreamer->stopAudioStream();
                   // bInitialize(rfrMySPINAudioCmd.u32DeviceHandle, e8DEVTYPE_IAP,
                          // rfrMySPINAudioCmd.enAudioDir, m_BTDevConStat);
                   // t_S32 s32OutErrorCode = m_poAudioStreamer->setAudioStreamOutputDevice("AdevEnt1Out");   
                   // s32ErrorCode = m_poAudioStreamer->startAudioStream();
               // }
           }    
           else if(e8AUD_MAIN_OUT == itrCmdsList->enAudioDir)
           {
               itrCmdsList->enState = e8_STATE_INACTIVATED;
               itrCmdsList->enAudioResponse = e8_REQUEST_CLOSE;
               vSendAudioRespose(itrCmdsList->u32DeviceHandle, itrCmdsList->u32RequestID, itrCmdsList->enAudioResponse);               
           }               
           else if((e8AUD_STEREO_MIX_OUT == itrCmdsList->enAudioDir) && (NULL != m_rmySPINAudioCbs.fvRequestTerminateAudioCb)) //Request for terminate audio
           {
               m_rmySPINAudioCbs.fvRequestTerminateAudioCb(rfrMySPINAudioCmd);
           }
           break;
       }
       else
       {
           ++itrCmdsList;
       }
   }  
   m_oAudioCmdMapLock.vUnlock();
   if(e8_REQUEST_CLOSE == enAudioResponse)
   {
       vRemoveClosedRequests();
   }    
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINCmdAudio::vHandlePreActivedRequests()
 ***************************************************************************/
t_Void spi_tclMySPINCmdAudio::vHandlePreActivedRequests()
{
   ETG_TRACE_USR1(("spi_tclMySPINCmdAudio::vHandlePreActivedRequests entered"));
   std::list<trMySPINAudioCmd>::iterator itrCmdsList;
   m_oAudioCmdMapLock.s16Lock();
   for(itrCmdsList = m_AudioCmdsList.begin();m_AudioCmdsList.end()!= itrCmdsList;++itrCmdsList)
   {
      //Check if there is any request in e8_STATE_PRE_ACTIVATED
      if (e8_STATE_PRE_ACTIVATED == itrCmdsList->enState)
      {
         // Since it is preactivated send the response as OPEN to the phone.
          vSendAudioRespose(itrCmdsList->u32DeviceHandle, itrCmdsList->u32RequestID, itrCmdsList->enAudioResponse);
          itrCmdsList->enState = e8_STATE_ACTIVE;
      }
   }
   m_oAudioCmdMapLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINAudioResourceMngr::vCleanup()
 ***************************************************************************/
t_Void spi_tclMySPINCmdAudio::vCleanup()
{
    ETG_TRACE_USR1(("spi_tclMySPINCmdAudio::vCleanup entered"));
    m_oAudioCmdMapLock.s16Lock();
    m_AudioCmdsList.clear();
    m_oAudioCmdMapLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINCmdAudio::vRemoveClosedRequests(...
 ***************************************************************************/
t_Void spi_tclMySPINCmdAudio::vRemoveClosedRequests()
{
   ETG_TRACE_USR1(("spi_tclMySPINCmdAudio::vRemoveClosedRequests entered"));
   std::list<trMySPINAudioCmd>::iterator itrCmdsList;
   m_oAudioCmdMapLock.s16Lock();
   for(itrCmdsList = m_AudioCmdsList.begin();m_AudioCmdsList.end()!= itrCmdsList;)
   {
      //Check if there is any request in e8_STATE_PRE_ACTIVATED
      if (e8_STATE_CLOSED == itrCmdsList->enState)
      {
          itrCmdsList = m_AudioCmdsList.erase(itrCmdsList);
      }
      else
      {
          ++itrCmdsList;
      }
   }
   m_oAudioCmdMapLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINCmdAudio::bSendAudioCmdResponse(...
 ***************************************************************************/
t_Bool spi_tclMySPINCmdAudio::bSendAudioCmdResponse(tenAudioDir enAudDir, t_Bool bIsAudioActive,
                                                    tenAudioResponse enAudioResponse,
                                                    const tenAudioContext coenAudioCntxt)
{
    ETG_TRACE_USR1(("spi_tclMySPINCmdAudio::bSendAudioCmdResponse entered enAudDir=%d, bIsAudioActive=%d, enAudioResponse=%d",
                    ETG_ENUM(SPI_AUDIO_DIRECTION,enAudDir),ETG_ENUM(BOOL,bIsAudioActive),enAudioResponse));
    t_Bool bRet = false;
    std::list<trMySPINAudioCmd>::iterator itrCmdsList;
    if(e8AUD_INVALID != enAudDir)
    {
        t_Bool isCleanRequired = false;
        m_oAudioCmdMapLock.s16Lock();
        for(itrCmdsList = m_AudioCmdsList.begin();m_AudioCmdsList.end()!= itrCmdsList;itrCmdsList++)
        {
            ETG_TRACE_USR2(("[PARAM]::bSendAudioCmdResponse requestID = %d, state = %d",itrCmdsList->u32RequestID,ETG_ENUM(MYSPIN_AUDIO_CMD_STATE,itrCmdsList->enState)));
            if ( ((e8_STATE_PROCESSING == itrCmdsList->enState) 
                || (e8_STATE_SUSPENDED == itrCmdsList->enState) 
                || (e8_STATE_ACTIVE == itrCmdsList->enState)
                || (e8_STATE_PRE_ACTIVATED == itrCmdsList->enState)
                || (e8_STATE_INACTIVATED == itrCmdsList->enState))
                && (enAudDir == itrCmdsList->enAudioDir) )
            {
                if(true == bIsAudioActive)
                {
                   itrCmdsList->enState = e8_STATE_ACTIVE;
                   itrCmdsList->enAudioResponse = e8_REQUEST_OPEN;
                   vSendAudioRespose(itrCmdsList->u32DeviceHandle, itrCmdsList->u32RequestID, itrCmdsList->enAudioResponse);
                }
                else
                {
                    itrCmdsList->enState = e8_STATE_CLOSED;
                    itrCmdsList->enAudioResponse = e8_REQUEST_CLOSE;
                    vSendAudioRespose(itrCmdsList->u32DeviceHandle, itrCmdsList->u32RequestID, itrCmdsList->enAudioResponse);
                    isCleanRequired = true;
                    if((e8AUD_MAIN_OUT == itrCmdsList->enAudioDir) && (m_bIsMixAudioActive == false))
                    {
                        m_bIsMainAudioActive = false;
                    }
                    else if((e8AUD_STEREO_MIX_OUT == itrCmdsList->enAudioDir) && (m_bIsMainAudioActive == false))
                    {
                        m_bIsMixAudioActive = false;
                    }
                }
                bRet = true;
            }
        }
        m_oAudioCmdMapLock.vUnlock();
        if(true == isCleanRequired)
        {
            vRemoveClosedRequests();
        }
    }
    else
    {
        bRet = bSendAudioResponse(enAudioResponse,coenAudioCntxt);
    }
        
    ETG_TRACE_USR2(("[PARAM]::bSendAudioCmdResponse left with bRet = %d",ETG_ENUM(BOOL,bRet)));
    return bRet;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINCmdAudio::bSendAudioResponse(...
 ***************************************************************************/
t_Bool spi_tclMySPINCmdAudio::bSendAudioResponse(tenAudioResponse enAudioResponse, const tenAudioContext coenAudioCntxt)
{
    ETG_TRACE_USR1(("spi_tclMySPINCmdAudio::bSendAudioResponse entered"));
    
    t_Bool bRet = false;
    std::list<trMySPINAudioCmd>::iterator itrCmdsList;
    
    if(e8_REQUEST_INVALID != enAudioResponse)
    {
        m_oAudioCmdMapLock.s16Lock();
        for(itrCmdsList = m_AudioCmdsList.begin();m_AudioCmdsList.end()!= itrCmdsList;itrCmdsList++)
        {
            ETG_TRACE_USR2(("[PARAM]::bSendAudioResponse requestID = %d, state = %d, response = %d ",
            itrCmdsList->u32RequestID,ETG_ENUM(MYSPIN_AUDIO_CMD_STATE,itrCmdsList->enState),itrCmdsList->enAudioResponse));
            if ( ((e8_STATE_ACTIVE == itrCmdsList->enState) ||
                  (e8_STATE_PRE_ACTIVATED == itrCmdsList->enState) ||
                  (e8_STATE_SUSPENDED == itrCmdsList->enState) ||
                  (e8_STATE_PROCESSING == itrCmdsList->enState) ||
                  (e8_STATE_INACTIVATED == itrCmdsList->enState)) )
            {
                 trMySPINAudioCmd rMySPINAudioCmd;
                 rMySPINAudioCmd.u32DeviceHandle = itrCmdsList->u32DeviceHandle;
                 rMySPINAudioCmd.enCommand = itrCmdsList->enCommand;
                 rMySPINAudioCmd.enAudioType = itrCmdsList->enAudioType;
                 rMySPINAudioCmd.u32RequestID = itrCmdsList->u32RequestID;
                 rMySPINAudioCmd.enAudioResponse = itrCmdsList->enAudioResponse;
                 rMySPINAudioCmd.enAudioDir = itrCmdsList->enAudioDir;
                 rMySPINAudioCmd.enState = itrCmdsList->enState;
                 
                if((e8_STATE_PROCESSING == itrCmdsList->enState) && (e8_REQUEST_OPEN == enAudioResponse))
                {
                    vSendAudioRespose(itrCmdsList->u32DeviceHandle, itrCmdsList->u32RequestID, enAudioResponse);
                }
                
                if(e8_REQUEST_SUSPEND == enAudioResponse)
                {
                    if((e8AUD_MAIN_OUT == itrCmdsList->enAudioDir) && (m_bIsMixAudioActive == false))
                    {
                        m_bIsMainAudioActive = false;
                        itrCmdsList->enState = e8_STATE_SUSPENDED;
                    }
                    else if(e8AUD_STEREO_MIX_OUT == itrCmdsList->enAudioDir)
                    {
                        continue;
                    }
                }
                else if(e8_REQUEST_CLOSE == enAudioResponse)
                {
                    // This check is added because when one of the audiotype is not active and the respective member
                    // variable is made false, when a new request is made for another audio type it would result in wrong handling
                    // during the starting of audio streaming.
                    if((e8AUD_MAIN_OUT == itrCmdsList->enAudioDir) && (m_bIsMixAudioActive == false))
                    {
                        m_bIsMainAudioActive = false;
                        itrCmdsList->enState = e8_STATE_CLOSED;
                    }      
                    else if(e8AUD_STEREO_MIX_OUT == itrCmdsList->enAudioDir)
                    {
                        if((e8SPI_AUDIO_MAIN == coenAudioCntxt) || (e8SPI_AUDIO_INTERNET_APP == coenAudioCntxt) ||
                        (e8SPI_AUDIO_MAIN_DEFAULT == coenAudioCntxt) || (e8SPI_AUDIO_PROJECTION == coenAudioCntxt))
                        {
                            ETG_TRACE_USR2(("[PARAM]::bSendAudioResponse entered continue case"));
                            continue;
                        }
                        else
                        {
                            m_bIsMixAudioActive = false;
                            itrCmdsList->enState = e8_STATE_CLOSED;
                            if((m_bIsMainAudioActive == false) && (NULL != m_rmySPINAudioCbs.fvRequestTerminateAudioCb))
                            {
                                m_rmySPINAudioCbs.fvRequestTerminateAudioCb(rMySPINAudioCmd);
                            }
                        }
                    }
                }
                else if(e8_REQUEST_OPEN == enAudioResponse)
                {
                    if(e8AUD_MAIN_OUT == itrCmdsList->enAudioDir)
                    {
                        itrCmdsList->enState = e8_STATE_ACTIVE;
                    }
                    else
                    {
                        continue;
                    }
                }
                else if(e8_REQUEST_REJECT == enAudioResponse)
                {
                    itrCmdsList->enState = e8_STATE_CLOSED;
                }
                
                if(enAudioResponse != itrCmdsList->enAudioResponse)
                {
                    ETG_TRACE_USR2(("[PARAM]::bSendAudioResponse entered send response case"));
                    itrCmdsList->enAudioResponse = enAudioResponse;
                    vSendAudioRespose(itrCmdsList->u32DeviceHandle, itrCmdsList->u32RequestID, enAudioResponse);
                }
                bRet = true;
            }
        }
        m_oAudioCmdMapLock.vUnlock();
        if((e8_REQUEST_CLOSE == enAudioResponse) || (e8_REQUEST_REJECT == enAudioResponse))
        {
            vRemoveClosedRequests();
        }
    }
    return bRet;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINCmdAudio::bIsAudioRequestAvailable(...
 ***************************************************************************/
t_Bool spi_tclMySPINCmdAudio::bIsAudioRequestAvailable(trMySPINAudioCmd rMySPINAudioCmd)
{
   std::list<trMySPINAudioCmd>::iterator itrCmdsList;
   t_Bool bIsAudReqAvailable = false;
   if(e8AUD_MAIN_OUT != rMySPINAudioCmd.enAudioDir)
   {
       m_oAudioCmdMapLock.s16Lock();
       ETG_TRACE_USR1(("[PARAM]::bIsAudioRequestAvailable size: %d",m_AudioCmdsList.size()));
       for(itrCmdsList = m_AudioCmdsList.begin();m_AudioCmdsList.end()!= itrCmdsList;++itrCmdsList)
       {
           if(itrCmdsList->enAudioDir == rMySPINAudioCmd.enAudioDir)
           {
               bIsAudReqAvailable = true;
           }
       }
       m_oAudioCmdMapLock.vUnlock();
   }
   return bIsAudReqAvailable;
}
//lint restore
///////////////////////////////////////////////////////////////////////////////
// <EOF>
