/***********************************************************************/
/*!
 * \file  spi_tclMySPINDbusHandler.h
 * \brief  Declaration of the Class spi_tclMySPINDbusHandler
 *************************************************************************
 \verbatim


 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:
 AUTHOR:         tch5kor
 COPYRIGHT:      &copy; 2015 Robert Bosch Car Multimedia GmbH
 HISTORY:
 Date        | Author                | Modification
 07.04.2016  | Chaitra Srinivasa     | Adding comments
 \endverbatim
 *************************************************************************/
#ifndef _SPI_TCLMYSPINDBUSHANDLER_H_
#define _SPI_TCLMYSPINDBUSHANDLER_H_

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/
/******************************************************************************
 | defines and macros (scope: global)
 |----------------------------------------------------------------------------*/

#include <glib.h>
#include <dbus/dbus-glib.h>
#include "mySPINTypes.h"

class spi_tclMySPINDbusHandler
{
   public:

	enum BTDBus
	{
	   BTDAEMON_SRV_NAME = 0,
	   BTDAEMON_BASEOBJECT_PATH,
	   BTDAEMON_AUDIO_STRING_INTERFACE,
	   BTDAEMON_AUDIO_INTERFACE,
	   BTDAEMON_AVP_INTERFACE,
	   BTDAEMON_DEVICE_ADDR
	};
      /***************************************************************************
       ** FUNCTION: spi_tclMySPINDbusHandler::spi_tclMySPINDbusHandler()
       ***************************************************************************/
      /*!
       * \fn    spi_tclMySPINDbusHandler()
       * \brief Constructor for MySPIN Dbus Handler class
       **************************************************************************/
      spi_tclMySPINDbusHandler();

      /***************************************************************************
       ** FUNCTION: spi_tclMySPINDbusHandler::~spi_tclMySPINDbusHandler()
       ***************************************************************************/
      /*!
       * \fn    ~spi_tclMySPINDbusHandler()
       * \brief Destructor for MySPIN Dbus Handler class
       **************************************************************************/
      virtual ~spi_tclMySPINDbusHandler();

      /***************************************************************************
       ** FUNCTION: spi_tclMySPINDbusHandler::t_Void vInitialize()
       ***************************************************************************/
      /*!
       * \fn
       * \brief Sets up connections with system bus and session bus.
       * \retval t_Void
       **************************************************************************/
      t_Void vInitialize();

      /***************************************************************************
       ** FUNCTION: spi_tclMySPINDbusHandler::t_Void vUnInitialize()
       ***************************************************************************/
      /*!
       * \fn    t_Void vUnInitialize()
       * \brief Destroys the connections with system bus and session bus
       * \retval t_Void
       **************************************************************************/
      t_Void vUnInitialize();

      /***************************************************************************
       ** FUNCTION: spi_tclMySPINDbusHandler::DBusGProxy* poGetDBusProxy()
       ***************************************************************************/
      /*!
       * \fn    DBusGProxy* poGetDBusProxy(tenProxyType enProxyType)
       * \brief creates/provides DBUS proxies
       * \param enProxyType: [IN]Stores the Proxy type of DBus
       * \retval DBusGProxy
       **************************************************************************/
      DBusGProxy* poGetDBusProxy(tenProxyType enProxyType);

      /***************************************************************************
       ** FUNCTION: spi_tclMySPINDbusHandler::t_Char *szGetPlayerInstance()
       ***************************************************************************/
      /*!
       * \fn    t_Char* szGetPlayerInstance()
       * \brief Method to get the player instance
       * \retval t_Char
       **************************************************************************/
      t_Char* szGetPlayerInstance();

      /***************************************************************************
       ** FUNCTION: spi_tclMySPINDbusHandler::t_Void vSetConfiguration()
       ***************************************************************************/
      /*!
       * \fn    t_Void vSetConfiguration(trMySpinAudioConfig rMySpinAudioConfig)
       * \brief Method to set configuration for audio
       * \param rMySpinAudioConfig: [IN] structure to store mySPIn
       *                           audio configuration
       * \retval t_Void
       **************************************************************************/
      t_Void vSetConfiguration(trMySpinAudioConfig rMySpinAudioConfig);

      /***************************************************************************
       ** FUNCTION: spi_tclMySPINDbusHandler::t_Void vGetConfiguration()
       ***************************************************************************/
      /*!
       * \fn    t_Void vGetConfiguration(trMySpinAudioConfig& rMySpinAudioConfig)
       * \brief Method to get the configuration for audio
       * \param rMySpinAudioConfig: [IN] structure to store mySPIn
       *                           audio configuration
       * \retval t_Void
       **************************************************************************/
      t_Void vGetConfiguration(trMySpinAudioConfig& rMySpinAudioConfig);

      /***************************************************************************
       ** FUNCTION: spi_tclMySPINDbusHandler::t_Bool bStartA2DPAudioPipe()
       ***************************************************************************/
      /*!
       * \fn    t_Bool bStartA2DPAudioPipe(t_String szDevBTMacAdd)
       * \brief Method to start A2DP audio pipe
       * \param szDevBTMacAdd: [IN] string for Device BT Mac address
       * \retval t_Bool
       **************************************************************************/
      t_Bool bStartA2DPAudioPipe(t_String szDevBTMacAdd);

      /***************************************************************************
       ** FUNCTION: spi_tclMySPINDbusHandler::t_Bool bStopA2DPAudioPipe()
       ***************************************************************************/
      /*!
       * \fn    t_Bool bStopA2DPAudioPipe()
       * \brief Method to stop A2DP audio pipe
       * \retval t_Bool
       **************************************************************************/
      t_Bool bStopA2DPAudioPipe();

      /***************************************************************************
       ** FUNCTION: spi_tclMySPINDbusHandler::t_Bool bSendAVPCtrlCmd()
       ***************************************************************************/
      /*!
       * \fn    t_Bool bSendAVPCtrlCmd(t_U8 u8AVPcmd, t_U8 szEventType)
       * \brief Method for MySPIN Dbus handler
       * \param u8AVPcmd: [IN]To store the AVP command
       * \param szEventType: [IN]String for event type
       * \retval t_Bool
       **************************************************************************/
      t_Bool bSendAVPCtrlCmd(t_U8 u8AVPcmd, t_U8 szEventType);

      /***************************************************************************
       ** FUNCTION: spi_tclMySPINDbusHandler::t_Void vClearDBUSProxies()
       ***************************************************************************/
      /*!
       * \fn    t_Void vClearDBUSProxies()
       * \brief Method for cleaning up DBUS proxies
       **************************************************************************/
      t_Void vClearDBUSProxies();

      /***************************************************************************
       ** FUNCTION: spi_tclMySPINDbusHandler::t_Void bGetA2DPStreamingStatus()
       ***************************************************************************/
      /*!
       * \fn    t_Void bGetA2DPStreamingStatus(t_String szDevBTMacAdd)
       * \brief Method for getting A2DP streaming status from BT component
       **************************************************************************/
      t_Bool bGetA2DPStreamingStatus(t_String szDevBTMacAdd);

   private:

      /***************************************************************************
       ** FUNCTION: spi_tclMySPINDbusHandler::t_String szGetDevicePath()
       ***************************************************************************/
      /*!
       * \fn     t_String szGetDevicePath()
       * \brief  Method to get the device path
       * \retval t_String
       **************************************************************************/
      t_String szGetDevicePath();
      /***************************************************************************
       ** FUNCTION: spi_tclMySPINDbusHandler::t_Bool bGetMediaPlayerPath()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetMediaPlayerPath()
       * \brief  Method to get the device path
       * \retval t_Bool
       **************************************************************************/
      t_Bool bGetMediaPlayerPath();
      /***************************************************************************
       ** FUNCTION: spi_tclMySPINDbusHandler::t_Void vLoadDBusConfiguration()
       ***************************************************************************/
      /*!
       * \fn     t_Void vLoadDBusConfiguration()
       * \brief  Load DBus interface const depending on BT stack
       * \retval t_Void
       **************************************************************************/
      t_Void vLoadDBusConfiguration();
      /***************************************************************************
       ** FUNCTION: spi_tclMySPINDbusHandler:: bStartBTEVOA2DPAudioPipe()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bStartBTALPSA2DPAudioPipe()
       * \brief  Start BT Alps stack audio pipe
       * \retval t_Bool
       **************************************************************************/
      t_Bool bStartBTALPSA2DPAudioPipe(t_String szDevBTMacAdd);
      /***************************************************************************
       ** FUNCTION: spi_tclMySPINDbusHandler:: bStartBTEVOA2DPAudioPipe()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bStartBTEVOA2DPAudioPipe()
       * \brief  Start BT Evolution stack audio pipe
       * \retval t_Bool
       **************************************************************************/
      t_Bool bStartBTEVOA2DPAudioPipe(t_String szDevBTMacAdd);
      
      /***************************************************************************
       ** FUNCTION:  spi_tclMySPINDbusHandler(const spi_tclMySPINDbusHandler...
       ***************************************************************************/
      /*!
       * \fn      spi_tclMySPINDbusHandler(
       *                             const spi_tclMySPINDbusHandler& corfoSrc)
       * \brief   Copy constructor - Do not allow the creation of copy constructor
       * \param   corfoSrc : [IN] reference to source data interface object
       * \retval
       * \sa      spi_tclMySPINDbusHandler()
       ***************************************************************************/
      spi_tclMySPINDbusHandler(const spi_tclMySPINDbusHandler &corfoSrc) = delete;
      
      /***************************************************************************
       ** FUNCTION:  spi_tclMySPINDbusHandler& operator=( const ...
       ***************************************************************************/
      /*!
       * \fn      spi_tclMySPINDbusHandler& operator=(
       *                          const spi_tclMySPINDbusHandler& corfoSrc))
       * \brief   Assignment operator
       * \param   corfoSrc : [IN] reference to source data interface object
       * \retval
       * \sa      spi_tclMySPINDbusHandler(const spi_tclMySPINDbusHandler& otrSrc)
       ***************************************************************************/
       spi_tclMySPINDbusHandler& operator=(const spi_tclMySPINDbusHandler& corfoSrc) = delete;
      

      // Configuration used by player engine
      trMySpinAudioConfig m_rMySpinAudioConfig;

      // DBUS proxy for ALPS BT Deamon
      DBusGProxy* m_poALPSBTDeamonProxy;

      // DBUS proxy for Audio Visual
      DBusGProxy* m_poAVProxy;

      //DBus proxy for media player
      DBusGProxy* m_poMPProxy;
      //Connection handler for System bus
      DBusGConnection *m_poSystemDBusConn;

      //Connection handler for Session bus
      DBusGConnection *m_poSessionDBusConn;

      //Player Instance
      t_Char* m_szPlayerInstance;
      //Check For Bt Evolution stack
      t_Bool m_bBTEvolutionAvailability;
      //Bt interface variable names
      t_String m_szServiceName;
      t_String m_szBaseObjectPath;
      t_String m_szAudioStringInterface;
      t_String m_szAudioInterface;
      t_String m_szAVPInterface;
      t_String m_szDeviceAddr;

};
#endif //_SPI_TCLMYSPINDBUSHANDLER_H_
