/***********************************************************************/
/*!
 * \file  spi_tclMySPINCmdAudio.h
 * \brief  Implementation of the Class spi_tclMySPINCmdAudio
 *************************************************************************
 \verbatim


 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:
 AUTHOR:         tch5kor
 COPYRIGHT:      &copy; 2015 Robert Bosch Car Multimedia GmbH
 HISTORY:
 Date        | Author                | Modification
 07.04.2016  | Chaitra Srinivasa     | Adding comments
 \endverbatim
 *************************************************************************/
#ifndef _SPI_TCLMYSPINCMDAUDIO_H
#define _SPI_TCLMYSPINCMDAUDIO_H

#include "spi_tclMySPINDbusHandler.h"
#include "mspin_audio_stream.h"
#include "mspin_vr_stream.h"
#include "Lock.h"

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/
/******************************************************************************
 | defines and macros (scope: global)
 |----------------------------------------------------------------------------*/

class spi_tclMySPINCmdAudio
{

   public:

      /***************************************************************************
       ** FUNCTION: spi_tclMySPINCmdAudio::spi_tclMySPINCmdAudio()
       ***************************************************************************/
      /*!
       * \fn    spi_tclMySPINCmdAudio()
       * \brief Constructor for mySPIN command audio class
       **************************************************************************/
      spi_tclMySPINCmdAudio();

      /***************************************************************************
       ** FUNCTION: spi_tclMySPINCmdAudio::~spi_tclMySPINCmdAudio()
       ***************************************************************************/
      /*!
       * \fn    ~spi_tclMySPINCmdAudio()
       * \brief  Destructor for mySPIN command audio class
       **************************************************************************/
      virtual ~spi_tclMySPINCmdAudio();

      /***************************************************************************
       ** FUNCTION: spi_tclMySPINCmdAudio::t_Bool bInitialize()
       ***************************************************************************/
      /*!
       * \fn    spi_tclMySPINCmdAudio::t_Bool bInitialize(t_U32 u32DeviceId,
       *                             tenDeviceSubCategory enDeviceSubCategory,
       *                             tenAudioDir enAudioDir,
       *                             tvBluetoothDiscon fvBluetoothDiscon)
       * \brief Method used to initialize the device id, category etc
       * \param u32DeviceId: [IN]Device ID
       * \param enDeviceSubCategory: [IN] To store device sub-category
       * \param enAudioDir: [IN] To store the audio stream direction
       * \param fvBluetoothDiscon: [IN] Bluetooth disconnection
       * \retval t_Bool
       **************************************************************************/
      t_Bool bInitialize(t_U32 u32DeviceId, tenDeviceSubCategory enDeviceSubCategory, tenAudioDir enAudioDir,
               tvBluetoothConStat fvBluetoothConStat);

      /***************************************************************************
       ** FUNCTION: spi_tclMySPINCmdAudio::t_Void vUnInitialize()
       ***************************************************************************/
      /*!
       * \fn    t_Void vUnInitialize(tenDeviceSubCategory enDeviceSubCategory,
       *                               tenAudioDir enAudioDir)
       * \brief Method to uninitialize the values of device sub category and
       *        audio stream direction
       * \param enDeviceSubCategory: [IN] To store device sub-category
       * \param enAudioDir: [IN] To store the audio stream direction
       * \retval t_Void
       **************************************************************************/
      t_Void vUnInitialize(tenDeviceSubCategory enDeviceSubCategory, tenAudioDir enAudioDir);

      /***************************************************************************
       ** FUNCTION: spi_tclMySPINCmdAudio::tenAudioStreamError enStartAudioStreaming()
       ***************************************************************************/
      /*!
       * \fn    tenAudioStreamError enStartAudioStreaming(tenAudioDir enAudioDir,
       *                                   t_String szAudioDev)
       * \brief Method to start audio streaming
       * \param enAudioDir: [IN] To store the audio stream direction
       * \param szAudioDev: [IN] Audio device
       * \retval tenAudioStreamError
       **************************************************************************/
      tenAudioStreamError enStartAudioStreaming(tenAudioDir enAudioDir, t_String szAudioDev);

      /***************************************************************************
       ** FUNCTION: spi_tclMySPINCmdAudio::tenAudioStreamError enStopAudioStreaming()
       ***************************************************************************/
      /*!
       * \fn    tenAudioStreamError enStopAudioStreaming(tenAudioDir enAudDir)
       * \brief Method to stop audio streaming
       * \param enAudDir: [IN] To store the audio stream direction
       * \retval tenAudioStreamError
       **************************************************************************/
      tenAudioStreamError enStopAudioStreaming(tenAudioDir enAudDir);

      /***************************************************************************
       ** FUNCTION: spi_tclMySPINCmdAudio::t_Void vSetBtMacAddress()
       ***************************************************************************/
      /*!
       * \fn      t_Void vSetBtMacAddress(t_U32 u32DeviceID,
       *                                  t_String rfszMACAddress)
       * \brief   Interface to set BT MAC Address from the friendly Name.
       * \param   [IN] u32DeviceID: Device ID.
       * \param   [IN/OUT] rfszMACAddress: BT MAC Address of device.
       * \retval  NONE.
       **************************************************************************/
      t_Void vSetBtMacAddress(t_U32 u32DeviceID, t_String rfszMACAddress);

      /***************************************************************************
       ** FUNCTION: spi_tclMySPINCmdAudio::t_Bool bSendAVPCtrlCmd()
       ***************************************************************************/
      /*!
       * \fn    t_Bool bSendAVPCtrlCmd(t_U8 u8KeyCode, t_UChar szEventType)
       * \brief Method to send AVP control command
       * \param u8KeyCode: [IN]Used to store key code
       * \param szEventType: [IN]Used to store type of event
       * \retval  NONE.
       **************************************************************************/
      t_Bool bSendAVPCtrlCmd(t_U8 u8KeyCode, t_UChar szEventType);

      /***************************************************************************
       ** FUNCTION: spi_tclMySPINCmdAudio::t_Bool bGetBTConnectionStatus()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetBTConnectionStatus(t_U32 u32DeviceID)
       * \brief  Method to get BT connection status
       * \param  u32DeviceID: [IN]Unique Device ID
       * \retval NONE.
       **************************************************************************/
      t_Bool bGetBTConnectionStatus(t_U32 u32DeviceID);

      /***************************************************************************
       ** FUNCTION: spi_tclMySPINCmdAudio::t_Void bClearBTDevices()
       ***************************************************************************/
      /*!
       * \fn     t_Void bClearBTDevices(t_U32 u32DeviceID)
       * \brief  Method to clear BT devices
       * \param  u32DeviceID: [IN]Unique device ID
       * \retval NONE.
       **************************************************************************/
      t_Void vClearBTDevices(t_U32 u32DeviceID);

      /***************************************************************************
       ** FUNCTION: spi_tclMySPINCmdAudio::t_Void vSetBluetoothStatus()
       ***************************************************************************/
      /*!
       * \fn      t_Void vSetBluetoothStatus(t_U32 u32DeviceID,
       *                           t_Bool bIsConnected)
       * \brief   Interface to set BT status.
       * \param   u32DeviceID: [IN]Device ID
       * \param   bIsA2DPConn: [IN]Used to get the status of A2DP connection
       * \param   enA2DPStatus: [IN]Used to get the status of A2DP connection
       * \retval  NONE.
       **************************************************************************/
      t_Void vSetBluetoothStatus(t_U32 u32DeviceID, t_Bool bIsA2DPConn, 
                                 tenA2DPStatus enA2DPStatus);


      /***************************************************************************
       ** FUNCTION: spi_tclMySPINCmdAudio::t_Void vRegisterAudioCbs()
       ***************************************************************************/
      /*!
       * \fn      t_Void vRegisterAudioCbs(t_U32 u32DeviceID)
       * \brief   Interface to Register for audio callbacks when advanced audio is
       *          enabled.
       * \param   u32DeviceID: [IN]Device ID
       * \param   rmySPINAudioCbs: [IN] Structure containing callbacks
       * \param   rmySPINAudioDuckCb: [IN] Structure containing callbacks
       * \retval  NONE.
       **************************************************************************/
      t_Void vRegisterAudioCbs(t_U32 u32DeviceID,trmySPINAudioCbs rmySPINAudioCbs,
      trmySPINAudioDuckCb rmySPINAudioDuckCb);

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclMySPINCmdAudio::vHandleAudioCmd(...
       ***************************************************************************/
      /*!
       * \fn     vHandleAudioCmd(trMySPINAudioCmd rMySPINAudioCmd)
       * \brief  Method to handle audio commands from phone
       * \param  [rMySPINAudioCmd]: Structure containing audio command information
       * \parma
       * \retval t_Void
       **************************************************************************/
      t_Void vHandleAudioCmd(trMySPINAudioCmd rMySPINAudioCmd);

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclMySPINCmdAudio::vHandleAudioCmd(...
       ***************************************************************************/
      /*!
       * \fn     vHandleAudioCmd(trMySPINAudioCmd rMySPINAudioCmd)
       * \brief  Method to handle audio commands from phone
       * \param  [rMySPINAudioCmd]: Structure containing audio command information
       * \parma
       * \retval t_Void
       **************************************************************************/
      t_Void vSendAudioRespose(t_U32 u32DeviceHandle, t_U32 u32RequestID,
               tenAudioResponse enAudioResponse);

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclMySPINCmdAudio::vCleanup(...
       ***************************************************************************/
      /*!
       * \fn     vCleanup()
       * \brief  Method to perform Cleanup when mySPIN session ends
       * \retval t_Void
       **************************************************************************/
      t_Void vCleanup();

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclMySPINCmdAudio::bSendAudioCmdResponse(...
       ***************************************************************************/
      /*!
       * \fn     bSendAudioCmdResponse()
       * \brief  Method to send response to audio requests sent by phone.
       *         if enAudDir != e8AUD_INVALID, then open is sent if bIsAudioActive
       *         is set to true else close is sent.
       *         if enAudioResponse != e8_REQUEST_INVALID then enAudioResponse is sent
       *         Only enAudDir or enAudioResponse can be valid at a time. If both are
       *         set then enAudDir is given priority
       * \param  enAudDir: Audio direction
       * \param  bIsAudioActive: Audio is active or not
       * \param  enAudioResponse: response to be sent
       * \retval t_Void
       **************************************************************************/
      t_Bool bSendAudioCmdResponse(tenAudioDir enAudDir, t_Bool bIsAudioActive, 
                                   tenAudioResponse enAudioResponse,
                                   const tenAudioContext coenAudioCntxt);
      
      /***************************************************************************
       ** FUNCTION: t_Void spi_tclMySPINCmdAudio::bIsAudioRequestAvailable(...
       ***************************************************************************/
      /*!
       * \fn     bIsAudioRequestAvailable()
       * \brief  Method to check whether audio request is available(Special handling for phonecall and VR)
       * \param  [rMySPINAudioCmd]: Structure containing audio command information
       * \retval t_Bool
       **************************************************************************/
      t_Bool bIsAudioRequestAvailable(trMySPINAudioCmd rMySPINAudioCmd);

   private:

      /***************************************************************************
       ** FUNCTION: spi_tclMySPINCmdAudio::t_Void vCopyAudioConfig()
       ***************************************************************************/
      /*!
       * \fn    t_Void vCopyAudioConfig(trMySpinAudioConfig rMySpinAudioConfig,
       *                       audioStreamConfig& rAudStreamConfig)
       * \brief Method to copy the audio configuration
       * \param rMySpinAudioConfig: [IN]Structure to store the mySPIN audio
       *                               configuration
       * \param rAudStreamConfig: [IN]Structure to store audio stream configuration
       * \retval NONE.
       **************************************************************************/
      t_Void vCopyAudioConfig(trMySpinAudioConfig rMySpinAudioConfig, audioStreamConfig& rAudStreamConfig);

      /***************************************************************************
       ** FUNCTION: spi_tclMySPINCmdAudio::t_Void vOnAudioStreamErroCb()
       ***************************************************************************/
      /*!
       * \fn    t_Void vOnAudioStreamErroCb(t_U32 u32Context,
       *                                audioStreamError enError,
       *                                t_String szError)
       * \brief Audio stream error call back
       * \param u32Context: [IN]Used to store context
       * \param enError: [IN]Used to store the error
       * \param szError: [IN]Used to store the type of error occured
       * \retval NONE.
       **************************************************************************/
      t_Void vOnAudioStreamErroCb(t_U32 u32Context, audioStreamError enError, t_String szError);

      /***************************************************************************
       ** FUNCTION: spi_tclMySPINCmdAudio::t_Void vOnAudStreamRateChanedCb()
       ***************************************************************************/
      /*!
       * \fn    t_Void vOnAudStreamRateChanedCb(t_U32 u32Context,
       *                                       t_U32 u32SampleRate)
       * \brief Call back for change in audio stream rate
       * \param u32Context: [IN]Used to store the context
       * \param u32SampleRate: [IN]Used to store the sample rate
       * \retval NONE.
       **************************************************************************/
      t_Void vOnAudStreamRateChanedCb(t_U32 u32Context, t_U32 u32SampleRate);

      /***************************************************************************
       ** FUNCTION: spi_tclMySPINCmdAudio::t_Void vOnMySpinVrStreamErrorCb()
       ***************************************************************************/
      /*!
       * \fn    t_Void vOnMySpinVrStreamErrorCb(t_U32 u32Context,
       *                                        vrStreamError enVRError)
       * \brief Call back for mySPIN VR error
       * \param u32Context: [IN]Used to store the context
       * \param enVRError: [IN]Used to store the error
       * \retval NONE.
       **************************************************************************/
      t_Void vOnMySpinVrStreamErrorCb(t_U32 u32Context, vrStreamError enVRError);

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclMySPINCmdAudio::vHandleAudioSourceRequest(...
       ***************************************************************************/
      /*!
       * \fn     vHandleAudioSourceRequest(trMySPINAudioCmd rMySPINAudioCmd)
       * \brief  Method to handle audio control request commands
       * \param  [rMySPINAudioCmd]: Structure containing audio command information
       * \retval t_Void
       **************************************************************************/
      t_Void vHandleAudioSourceCtrlRequest(trMySPINAudioCmd& rfrMySPINAudioCmd);


      /***************************************************************************
       ** FUNCTION: t_Void spi_tclMySPINCmdAudio::vHandleAudioSourceRelease(...
       ***************************************************************************/
      /*!
       * \fn     vHandleAudioSourceRelease(trMySPINAudioCmd rMySPINAudioCmd)
       * \brief  Method to handle audio control release commands
       * \param  [rMySPINAudioCmd]: Structure containing audio command information
       * \retval t_Void
       **************************************************************************/
      t_Void vHandleAudioSourceCtrlRelease(trMySPINAudioCmd& rfrMySPINAudioCmd);

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclMySPINCmdAudio::vHandlePreActivedRequests(...
       ***************************************************************************/
      /*!
       * \fn     vHandlePreActivedRequests(trMySPINAudioCmd rMySPINAudioCmd)
       * \brief  Method to handle preactivated control request commands
       * \param  NONE
       * \retval t_Void
       **************************************************************************/
      t_Void vHandlePreActivedRequests();

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclMySPINCmdAudio::vRemoveClosedRequests(...
       ***************************************************************************/
      /*!
       * \fn     vRemoveClosedRequests()
       * \brief  Method to remove closed audio requests
       * \param  NONE
       * \retval t_Void
       **************************************************************************/
      t_Void vRemoveClosedRequests();

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclMySPINCmdAudio::vAudioRequestCb(...
       ***************************************************************************/
      /*!
       * \fn     vAudioRequestCb()
       * \brief  Audio request callback
       * \param  context : Audio context
       * \param  command : Myspin audio control command
       * \param  requestID : request ID
       * \param  type : Myspin audio type
       * \retval t_Void
       **************************************************************************/
      static t_Void vAudioRequestCb(t_Void *context, MSPIN_AUDIOCONTROL command, t_U32 requestID, MSPIN_AUDIOTYPE type);
      
      /***************************************************************************
       ** FUNCTION: t_Void spi_tclMySPINCmdAudio::bSendAudioResponse(...
       ***************************************************************************/
      /*!
       * \fn     bSendAudioResponse(tenAudioResponse enAudioResponse)
       * \brief  Method to send audio response for invalid audio direction usecases
       * \param  NONE
       * \retval t_Bool
       **************************************************************************/
      t_Bool bSendAudioResponse(tenAudioResponse enAudioResponse,
                                const tenAudioContext coenAudioCntxt);
      
      /***************************************************************************
       ** FUNCTION:  spi_tclMySPINCmdAudio(const spi_tclMySPINCmdAudio...
       ***************************************************************************/
      /*!
       * \fn      spi_tclMySPINCmdAudio(
       *                             const spi_tclMySPINCmdAudio& corfoSrc)
       * \brief   Copy constructor - Do not allow the creation of copy constructor
       * \param   corfoSrc : [IN] reference to source data interface object
       * \retval
       * \sa      spi_tclMySPINCmdAudio()
       ***************************************************************************/
       spi_tclMySPINCmdAudio(const spi_tclMySPINCmdAudio& corfoSrc);


      /***************************************************************************
       ** FUNCTION:  spi_tclMySPINCmdAudio& operator=( const spi_tclAAP...
       ***************************************************************************/
      /*!
       * \fn      spi_tclMySPINCmdAudio& operator=(
       *                          const spi_tclMySPINCmdAudio& corfoSrc))
       * \brief   Assignment operator
       * \param   corfoSrc : [IN] reference to source data interface object
       * \retval
       * \sa      spi_tclMySPINCmdAudio(const spi_tclMySPINCmdAudio& otrSrc)
       ***************************************************************************/
       spi_tclMySPINCmdAudio& operator=(const spi_tclMySPINCmdAudio& corfoSrc);

      //Dbus Handler
      spi_tclMySPINDbusHandler* m_poMySPINDbusHandler;

      //Audio Streamer
      MySpinAudioStream* m_poAudioStreamer;

      //VR streamer
      MySpinVrStream* m_poVrStreamer;

      // Device Configuration for input device
      vrDeviceConfig m_rVRInputDevConfig;

      //Device Configuration for playback device
      vrDeviceConfig m_rVRPlayBackDevConfig;

      //Map Holding BT Friendly Name for the selected devices
      std::map<t_U32, t_String> m_BtMacAddressMap;

      //Map Holding Device Id and Connection Status
      std::map<t_U32, t_Bool> m_BtConnStatusMap;

      //Lock to protect Map containing M Addresses
      Lock m_macAddLock;

      //Lock to protect Map containing BT Ststu
      Lock m_btStatusLock;

      //Callback to propogate BT disconnection
      tvBluetoothConStat m_BTDevConStat;

      //Structure containg function pointers for callback
      trmySPINAudioCbs m_rmySPINAudioCbs;

      //Structure containing function pointer for cb
      trmySPINAudioDuckCb m_rmySPINAudioDuckCb;

      //hold the device id
      t_U32 m_u32DeviceId;

      //List containing audio requests
      std::list<trMySPINAudioCmd> m_AudioCmdsList;

      Lock m_oAudioCmdMapLock;

      //Variable to maintain status of Main audio
      t_Bool m_bIsMainAudioActive;
      
      //Variable to maintain status of mix audio
      t_Bool m_bIsMixAudioActive;
      
      //t_String m_szAudioDev;

      tenA2DPStatus m_enA2DPStatus;

      t_Bool m_bA2DPStreamingStatus;
};

#endif //_SPI_TCLMYSPINCMDAUDIO_H
