/***********************************************************************/
/*!
 * \file    spi_tclMySPINVideoResourceMngr.cpp
 * \brief   MySPIN Video Resource Manager
 *************************************************************************
 \verbatim

 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    MySPIN Video resource manager
 AUTHOR:         Shiva Kumar Gurija
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date         | Author                | Modification
 04.11.2015   | Shiva Kumar Gurija    | Initial Version


 \endverbatim
 *************************************************************************/

/******************************************************************************
 | includes:
 | 1)system- and project- includes
 | 2)needed interfaces from external components
 | 3)internal and external interfaces from this component
 |----------------------------------------------------------------------------*/
#include "spi_tclResorceMngrDefines.h"
#include "spi_tclMySPINRespVideo.h"
#include "spi_tclMySPINManager.h"
#include "spi_tclMySPINCmdVideo.h"
#include "spi_tclMySPINResourceMngr.h"
#include "spi_tclMySPINVideoResourceMngr.h"

//! Video tables
#define MSPIN_ACC_VIDEO_FOCUS
static const trMspinAccVideoFocusNoti sacoAccVideoFocusNoti[]=
#include "spi_tclMSPINContext.cfg"
#undef MSPIN_ACC_VIDEO_FOCUS

#define MSPIN_MD_VIDEO_FOCUS
static const trMspinVideoFocusState  sacoVideoFocusState[]=
#include "spi_tclMSPINContext.cfg"
#undef MSPIN_MD_VIDEO_FOCUS

#define MSPIN_VIDEO_REQ
static const trMspinMDVideoFocusReq  sacoMDVideoFocusReq[]=
#include "spi_tclMSPINContext.cfg"
#undef MSPIN_VIDEO_REQ

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_RSRCMNGR
#include "trcGenProj/Header/spi_tclMySPINVideoResourceMngr.cpp.trc.h"
#endif
#endif
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e746 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | defines and macros (scope: global)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: global)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/
static spi_tclMySPINManager* spoMySPINMngr = NULL;
static spi_tclMySPINCmdVideo* spoCmdVideo = NULL;

/***************************************************************************
 ** FUNCTION:  spi_tclMySPINVideoResourceMngr::spi_tclMySPINVideoResourceMngr()
 ***************************************************************************/
spi_tclMySPINVideoResourceMngr::spi_tclMySPINVideoResourceMngr(spi_tclMySPINResourceMngr* poMySPINRsrcMngr) :
   m_poMySPINRsrcMngr(poMySPINRsrcMngr), m_bMySPINSessionActive(false), m_bIsFirstFrameRendered(false),
   m_bIsPlaybackRequested(false), m_enCurMDFocusState(e8MSPIN_VIDEOFOCUSSTATE_LOSS),
   m_enCurAccDispCntxt(e8DISPLAY_CONTEXT_NATIVE), m_u32SelectedDeviceID(0),
   m_enGrantConstraint(e8DIPO_CONSTRAINT_NA),m_enPrevTransferType(e8DIPO_TRANSFERTYPE_NA),
   m_enSessionStatus(e8_SESSION_INACTIVE), m_u32SessionErrDevID(0)
{
   ETG_TRACE_USR1(("spi_tclMySPINVideoResourceMngr() entered "));
   //nothing to do
}

/***************************************************************************
 ** FUNCTION:  spi_tclMySPINVideoResourceMngr::~spi_tclMySPINVideoResourceMngr()
 ***************************************************************************/
spi_tclMySPINVideoResourceMngr::~spi_tclMySPINVideoResourceMngr()
{
   ETG_TRACE_USR1(("~spi_tclMySPINVideoResourceMngr() entered"));

   m_poMySPINRsrcMngr = NULL;
   spoMySPINMngr = NULL;
   spoCmdVideo = NULL;
   //nothing to do
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclMySPINVideoResourceMngr::bInitialize()
 ***************************************************************************/
t_Bool spi_tclMySPINVideoResourceMngr::bInitialize()
{
   ETG_TRACE_USR1(("spi_tclMySPINVideoResourceMngr::bInitialize() entered"));

   t_Bool bRet = false;

   spoMySPINMngr = spi_tclMySPINManager::getInstance();
   SPI_NORMAL_ASSERT(NULL == spoMySPINMngr);

   if (NULL != spoMySPINMngr)
   {
      spoCmdVideo = spoMySPINMngr->poGetVideoInstance();

      bRet = spoMySPINMngr->bRegisterObject((spi_tclMySPINRespVideo*) this);
      bRet = spoMySPINMngr->bRegisterObject((spi_tclMySPINRespSession*) this);

   }

   return bRet;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMySPINVideoResourceMngr::vUninitialize()
 ***************************************************************************/
t_Void spi_tclMySPINVideoResourceMngr::vUnInitialize()
{
   //Nothing to do
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMySPINVideoResourceMngr::vRegForCallbacks()
 ***************************************************************************/
t_Void spi_tclMySPINVideoResourceMngr::vRegForCallbacks(const trMySPINRsrcMngrCallback& corfrMySPINRsrcCbs)
{
   m_rMySPINRsrcMngrCb = corfrMySPINRsrcCbs;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclMySPINVideoResourceMngr::bGetAccVideoFocusState()
***************************************************************************/
t_Bool spi_tclMySPINVideoResourceMngr::bGetAccVideoFocusState(
         tenDisplayContext enDisplayContext,
         t_Bool bDisplayFlag,
         tenMspinVideoFocusState& rfenVideoFocusState)
{
   t_Bool bRet = false;

   t_U32 u32ContSize = (sizeof(sacoAccVideoFocusNoti))/(sizeof(trMspinAccVideoFocusNoti));
   for (t_U8 u8Index =0; u8Index < u32ContSize; u8Index++)
   {
      // Check for the matching Accessory context entry
      if (enDisplayContext == sacoAccVideoFocusNoti[u8Index].enAccDispCntxt)
      {
         bRet = true;

         rfenVideoFocusState = (true == bDisplayFlag)?
               (sacoAccVideoFocusNoti[u8Index].enAccFocusReqType):
               (sacoAccVideoFocusNoti[u8Index].enAccFocusRelType);

         ETG_TRACE_USR4(("[DESC] Derived Accessory Focus State based on display context = %d",
            ETG_ENUM(MYSPIN_VIDEOFOCUS_STATE, rfenVideoFocusState)));
         break;
      }
   }//for (t_U8 u8Index =0; u8Index < u32ContSize; u8Index++)

   ETG_TRACE_USR1(("spi_tclMySPINVideoResourceMngr::bGetAccVideoFocusState left with result %d "
            "(for DisplayContext = %d, DisplayFlag = %d)",
            ETG_ENUM(BOOL, bRet),
            ETG_ENUM(DISPLAY_CONTEXT, enDisplayContext),
            ETG_ENUM(BOOL, bDisplayFlag)));
   return bRet;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclMySPINVideoResourceMngr::bGetMDVideoFocusState()
***************************************************************************/
t_Bool spi_tclMySPINVideoResourceMngr::bGetMDVideoFocusState(
         tenMspinVideoFocusState enCurMDFocusState,
         tenMspinVideoFocusState enNewAccFocusState,
         tenMspinVideoFocusState& rfenNewMDFocusState)
{
   t_Bool bRet = false;

   t_U32 u32ContSize = (sizeof(sacoVideoFocusState)) / (sizeof(trMspinVideoFocusState));
   for (t_U8 u8Index = 0; u8Index < u32ContSize; u8Index++)
   {
      if ((enCurMDFocusState == sacoVideoFocusState[u8Index].enCurMDFocusState) &&
               (enNewAccFocusState == sacoVideoFocusState[u8Index].enReqFocusState))
      {
         bRet = true;

         rfenNewMDFocusState = sacoVideoFocusState[u8Index].enUpdatedMDFocusState;

         ETG_TRACE_USR4(("[PARAM]::bGetMDVideoFocusState: NewMDFocusState = %d",
                  ETG_ENUM(MYSPIN_VIDEOFOCUS_STATE, rfenNewMDFocusState)));
         break;
      } //if( (enCurMDFocusState == sacoVideoFocusState[u8Index]
   } //for(t_U8 u8Index =0; u8Index < u32ContSize; u8Index++)

   ETG_TRACE_USR1(("spi_tclMySPINVideoResourceMngr::bGetMDVideoFocusState left with result = %d "
            "(for CurMDFocusState = %d, NewAccFocusState = %d)",
            ETG_ENUM(BOOL, bRet),
            ETG_ENUM(MYSPIN_VIDEOFOCUS_STATE, enCurMDFocusState),
            ETG_ENUM(MYSPIN_VIDEOFOCUS_STATE, enNewAccFocusState)));
   return bRet;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclMySPINVideoResourceMngr::bGetAccRespType()
***************************************************************************/
t_Bool spi_tclMySPINVideoResourceMngr::bGetAccRespType(tenDisplayContext enCurDisplayContext,
         tenMspinVideoFocusState enCurMDFocusState,
         tenMspinVideoResponse& rfenRespType)
{
   t_Bool bRet = false;

   t_U32 u32ContSize = (sizeof(sacoMDVideoFocusReq)) / (sizeof(trMspinMDVideoFocusReq));
   for (t_U8 u8Index = 0; u8Index < u32ContSize; u8Index++)
   {
      if (enCurDisplayContext == sacoMDVideoFocusReq[u8Index].enAccDispCntxt)
      {
         bRet = true;

         rfenRespType = (e8MSPIN_VIDEOFOCUSSTATE_LOSS == enCurMDFocusState)?
              sacoMDVideoFocusReq[u8Index].enMDFocusReqResp_StateLoss:
              sacoMDVideoFocusReq[u8Index].enMDFocusReqResp_StateLossTransient;

         ETG_TRACE_USR4(("[PARAM]::bGetAccRespType: Accessory response for MD Video focus request = %d",
                  ETG_ENUM(MYSPIN_RESP_TYPE, rfenRespType)));
         break;
      } //if( (enCurMDFocusState == sacoMDVideoFocusReq[u8Index]
   } //for(t_U8 u8Index =0; u8Index < u32ContSize; u8Index++)

   ETG_TRACE_USR1(("spi_tclMySPINVideoResourceMngr::bGetAccRespType left with result = %d "
            "(for CurAccDispCntxt = %d, CurMDFocusState = %d)",
            ETG_ENUM(BOOL, bRet),
            ETG_ENUM(DISPLAY_CONTEXT, enCurDisplayContext),
            ETG_ENUM(MYSPIN_VIDEOFOCUS_STATE, enCurMDFocusState)));

   return bRet;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMySPINVideoResourceMngr::vSetAccessoryDisplayContext()
 ***************************************************************************/
t_Void spi_tclMySPINVideoResourceMngr::vSetAccessoryDisplayContext(const t_U32 cou32DevId, t_Bool bDisplayFlag,
         tenDisplayContext enDisplayContext)
{
   ETG_TRACE_USR1(("spi_tclMySPINVideoResourceMngr::vSetAccessoryDisplayContext:"
            "Device ID-0x%x Accessory Takes Focus-%d AccDisplayContext-%d",
            cou32DevId,ETG_ENUM(BOOL,bDisplayFlag),ETG_ENUM(DISPLAY_CONTEXT,enDisplayContext)));

   // do not process the update, if the DisplayFlag is false for a context, which is not active currently.
   if ((true == bDisplayFlag) || ((false == bDisplayFlag) && (m_enCurAccDispCntxt == enDisplayContext)))
   {
      m_enCurAccDispCntxt = (bDisplayFlag)? (enDisplayContext) : (e8DISPLAY_CONTEXT_NATIVE);
  
      //Updates can come even when session is not active. Check is added to prevent
      //forwarding the context when session is not active
      if(0 != m_u32SelectedDeviceID) 
      {
          tenMspinVideoFocusState enNewAccFocusState = e8MSPIN_VIDEOFOCUSSTATE_UNKNOWN;
          tenMspinVideoFocusState enNewMDFocusState = e8MSPIN_VIDEOFOCUSSTATE_UNKNOWN;
          
          m_oCurMDFocusStateLock.s16Lock();
          if ((true == bGetAccVideoFocusState(enDisplayContext, bDisplayFlag, enNewAccFocusState)) &&
                   (true == bGetMDVideoFocusState(m_enCurMDFocusState, enNewAccFocusState, enNewMDFocusState)))
          {
             switch (m_enCurMDFocusState)
             {
                case e8MSPIN_VIDEOFOCUSSTATE_GAIN:
                case e8MSPIN_VIDEOFOCUSSTATE_GAIN_TRANSIENT:
                {
                   //! HMI does TAKE or BORROW when MD has the screen, pause MD video playback
                   if ((e8MSPIN_VIDEOFOCUSSTATE_GAIN == enNewAccFocusState) ||
                    (e8MSPIN_VIDEOFOCUSSTATE_GAIN_TRANSIENT == enNewAccFocusState))
                   {
                      vSetVideoFocus(cou32DevId,e8MSPIN_VIDEOFOCUS_NATIVE);
                   }
                }
                   break;
                case e8MSPIN_VIDEOFOCUSSTATE_LOSS_TRANSIENT:
                {
                   //! HMI does an UNBORROW, start MD video playback
                   if (e8MSPIN_VIDEOFOCUSSTATE_LOSS_TRANSIENT == enNewAccFocusState)
                   {
                      vSetVideoFocus(cou32DevId,e8MSPIN_VIDEOFOCUS_PROJECTED);
                   }
                }
                   break;
                default:
                   ETG_TRACE_ERR(("[ERR]:vSetAccessoryDisplayContext: default state"));
                   break;
             }//switch (enNewAccFocusState)
          
             m_enCurMDFocusState = enNewMDFocusState;
          
          }//if (true == bGetAccVideoFocusState...)
          else
          {
             ETG_TRACE_ERR(("[ERR]:vSetAccessoryDisplayContext: Element not found"));
          }
          m_oCurMDFocusStateLock.vUnlock();
      }//if(0 != m_u32SelectedDeviceID)

      //! Populate TAKE Constraint, even when device is not active.
      m_oCurMDFocusStateLock.s16Lock();
      tenMspinVideoResponse enRespType = e8MSPIN_RESP_DENY;
      if (true == bGetAccRespType(m_enCurAccDispCntxt, m_enCurMDFocusState, enRespType))
      {
         m_enGrantConstraint = (e8MSPIN_RESP_ACCEPT == enRespType)? e8DIPO_CONSTRAINT_ANYTIME : e8DIPO_CONSTRAINT_NEVER ;
      }
      m_oCurMDFocusStateLock.vUnlock();
   }//if ((true == bDisplayFlag) || ...)
   else
   {
      ETG_TRACE_USR4(("spi_tclMySPINVideoResourceMngr::vSetAccessoryDisplayContext: Current active Context is %d ",
         ETG_ENUM(DISPLAY_CONTEXT, m_enCurAccDispCntxt)));
   }
}

/***************************************************************************
** FUNCTION: t_Void spi_tclMySPINVideoResourceMngr::vSelectDevice()
***************************************************************************/
t_Void spi_tclMySPINVideoResourceMngr::vSelectDevice(const trSelectDeviceRequest& corfrSelectReq)
{
    if((e8_SESSION_ERROR == m_enSessionStatus) && (m_u32SessionErrDevID == corfrSelectReq.m_u32DeviceHandle))
    {
        m_enSessionStatus = e8_SESSION_INACTIVE;
        m_u32SessionErrDevID = 0;
        vSendSessionStatusUpdate(m_enSessionStatus);
    }
}

/***************************************************************************
 ** FUNCTION:  spi_tclMySPINVideoResourceMngr::vDeviceDisconStatusCb
 ***************************************************************************/
t_Void spi_tclMySPINVideoResourceMngr::vDeviceDisconStatusCb(const t_U32 cou32DeviceHandle)
{
    if((e8_SESSION_ERROR == m_enSessionStatus) && (m_u32SessionErrDevID == cou32DeviceHandle))
    {
        m_enSessionStatus = e8_SESSION_INACTIVE;
        m_u32SessionErrDevID = 0;
        vSendSessionStatusUpdate(m_enSessionStatus);
    }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMySPINVideoResourceMngr::vOnSPISelectDeviceResult()
 ***************************************************************************/
t_Void spi_tclMySPINVideoResourceMngr::vOnSPISelectDeviceResult(t_U32 u32DevID, tenDeviceConnectionReq enDevConnReq,
         tenResponseCode enRespCode, tenErrorCode enErrorCode)
{
   ETG_TRACE_USR2(("[DESC]::vOnSPISelectDeviceResult: Select Type-%d  Response Code-%d", ETG_ENUM(CONNECTION_REQ,
            enDevConnReq), ETG_ENUM(RESPONSE_CODE, enRespCode)));

   tenSessionStatus enSessionStatus = e8_SESSION_UNKNOWN;

   if(e8DEVCONNREQ_SELECT == enDevConnReq)
   {
       if(e8SUCCESS == enRespCode)
       {
           m_u32SelectedDeviceID = u32DevID;
           enSessionStatus = e8_SESSION_ACTIVE;
       }
       else if((e8FAILURE == enRespCode) && ((e8FATAL_ERROR == enErrorCode) || (e8DEVICE_SWITCH_FAILED == enErrorCode)))
       {
            //This change is made in order for HMI to show pop up to User when selection fails due to below two reasons
            enSessionStatus = e8_SESSION_ERROR;
            m_enSessionStatus = e8_SESSION_ERROR;
            m_u32SessionErrDevID = u32DevID;
       }
   }
   else
   {
      enSessionStatus = e8_SESSION_INACTIVE;
      m_enSessionStatus = e8_SESSION_INACTIVE;
      m_u32SelectedDeviceID = 0;
      m_u32SessionErrDevID = 0;
      
      m_fBUpateLock.s16Lock();
      m_bIsFirstFrameRendered = false;
      m_fBUpateLock.vUnlock();
      
      m_oCurMDFocusStateLock.s16Lock();
      m_enCurMDFocusState = e8MSPIN_VIDEOFOCUSSTATE_LOSS;
      m_oCurMDFocusStateLock.vUnlock();      

      m_bIsPlaybackRequested = false;
      spi_tclMySPINManager::vSetSessionStatus(enSessionStatus);
      vSendSessionStatusUpdate(enSessionStatus);
      vSendDeviceDispCntxtUpdate(false, tenDisplayContextReason::e8DISPLAY_CONTEXT_REASON_UNKNOWN);
   }

   if (e8_SESSION_UNKNOWN != enSessionStatus)
   {
      //Set internal state as active
      spi_tclMySPINManager::vSetSessionStatus(e8_SESSION_ACTIVE);

      vSendSessionStatusUpdate(enSessionStatus);
      m_bMySPINSessionActive = (e8_SESSION_ACTIVE == enSessionStatus);
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINVideoResourceMngr::vSetAccessoryDisplayMode(t_U32...
 ***************************************************************************/
t_Void spi_tclMySPINVideoResourceMngr::vSetAccessoryDisplayMode(const t_U32 cu32DeviceHandle,
         const trDisplayContext corDisplayContext, const trDisplayConstraint corDisplayConstraint,
         const tenDisplayInfo coenDisplayInfo)
{
    SPI_INTENTIONALLY_UNUSED(corDisplayConstraint)
    ETG_TRACE_USR1(("spi_tclMySPINVideoResourceMngr::vSetAccessoryDisplayMode"));

    if (e8_DISPLAY_CONTEXT == coenDisplayInfo)
    {
       trUserContext rfrcUsrCntxt; // Dummy user context to have the same interface for GM
       vSetAccessoryDisplayContext(cu32DeviceHandle, corDisplayContext.bDisplayFlag, corDisplayContext.enDisplayContext);
    }
    if(e8_DISPLAY_CONSTRAINT == coenDisplayInfo)
    {
       if ((e8DIPO_CONSTRAINT_NEVER == corDisplayConstraint.enTakeConstraint) ||
                (e8DIPO_CONSTRAINT_NEVER == corDisplayConstraint.enBorrowConstraint))
       {
          //Considered Borrow constraint also as in the cases for Emergency e.g RVC,
          //Borrow can be set to Never where take can still be userinitiated/Anytime
          m_enGrantConstraint = e8DIPO_CONSTRAINT_NEVER;
       }
       else
       {
          m_enGrantConstraint = e8DIPO_CONSTRAINT_ANYTIME;
       }
           
      //Updates can come even when session is not active. Check is added to prevent
      //forwarding the context when session is not active           
        if(0 != m_u32SelectedDeviceID)
        {
            switch (corDisplayConstraint.enTransferType)
            {
               case e8DIPO_TRANSFERTYPE_TAKE:
               {
                  //Take has higher priority, process take even after borrow
                  m_oCurMDFocusStateLock.s16Lock();
                  m_enCurMDFocusState = e8MSPIN_VIDEOFOCUSSTATE_LOSS;
                  m_oCurMDFocusStateLock.vUnlock();
                  m_enPrevTransferType = corDisplayConstraint.enTransferType;
                  vSetVideoFocus(cu32DeviceHandle,e8MSPIN_VIDEOFOCUS_NATIVE);
               }
                  break;
               case e8DIPO_TRANSFERTYPE_BORROW:
                  //after take, if borrow comes - continue with take only
                  //else proceed with borrow
                  if (e8DIPO_TRANSFERTYPE_TAKE != m_enPrevTransferType)
                  {
                     m_oCurMDFocusStateLock.s16Lock();
                     m_enCurMDFocusState = e8MSPIN_VIDEOFOCUSSTATE_LOSS_TRANSIENT;
                     m_oCurMDFocusStateLock.vUnlock();
                     m_enPrevTransferType = corDisplayConstraint.enTransferType;
                     vSetVideoFocus(cu32DeviceHandle,e8MSPIN_VIDEOFOCUS_NATIVE);
                  }
                  break;
               case e8DIPO_TRANSFERTYPE_UNTAKE:
               {
                  //Nothing to do. HMI has to do Launch App
               }
                  break;
               case e8DIPO_TRANSFERTYPE_UNBORROW:
                  // check for previous state - If the last state was Borrow then only process
                  if (e8DIPO_TRANSFERTYPE_BORROW == m_enPrevTransferType)
                  {
                     m_oCurMDFocusStateLock.s16Lock();
                     m_enCurMDFocusState = e8MSPIN_VIDEOFOCUSSTATE_GAIN_TRANSIENT;
                     m_oCurMDFocusStateLock.vUnlock();
                     m_enPrevTransferType = corDisplayConstraint.enTransferType;
                     vSetVideoFocus(cu32DeviceHandle,e8MSPIN_VIDEOFOCUS_PROJECTED);
                  }
                  break;
               default:
                  ETG_TRACE_ERR(("[ERR]::Accessory Display Context:default state"));
                  break;
            }//switch(corDisplayConstraint.enTransferType)
        }//if(0 != m_u32SelectedDeviceID)   
    }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclMySPINVideoResourceMngr::vSetVideoFocus()
***************************************************************************/
t_Void spi_tclMySPINVideoResourceMngr::vSetVideoFocus(const t_U32 cou32DevId,
                                                      tenMspinVideoFocus enVideoFocus)
{
   ETG_TRACE_USR1(("spi_tclMySPINVideoResourceMngr::vSetVideoFocus:VideoFocusMode-%d ",
      ETG_ENUM(VIDEOFOCUS_MODE,enVideoFocus)));

   if((NULL != spoCmdVideo) && (0 != m_u32SelectedDeviceID))
   {
      switch(enVideoFocus)
      {
         case e8MSPIN_VIDEOFOCUS_NATIVE:
         {
            if(e8NO_ERRORS == spoCmdVideo->enStopVideoPlayback(cou32DevId))
            {
               vSendDeviceDispCntxtUpdate(false, tenDisplayContextReason::e8DISPLAY_CONTEXT_REASON_UNKNOWN);
               m_bIsFirstFrameRendered = false;
            }
            else
            {
               ETG_TRACE_ERR(("Error in starting mySPIN Video Playback"));
            }
         }
            break;
         case e8MSPIN_VIDEOFOCUS_PROJECTED:
         {
            //HMI has given the Video Focus. Start Video Playback and update with DeviceDisplayContext(TRUE)
            if (e8NO_ERRORS == spoCmdVideo->enStartVideoPlayback(cou32DevId))
            {
               vSendDeviceDispCntxtUpdate(true, tenDisplayContextReason::e8DISPLAY_CONTEXT_REASON_UNKNOWN);
            }
            else
            {
               ETG_TRACE_ERR(("[ERR]:Error in stopping mySPIN Video Playback"));
            }
         }
         break;
         default:
            ETG_TRACE_ERR(("[ERR]:Invalid video focus"));
            break;
      }//switch(enVideoFocus)
   }//if ((spoCmdVideo)...
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINVideoResourceMngr::vSendSessionStatusUpdate()
 ***************************************************************************/
t_Void spi_tclMySPINVideoResourceMngr::vSendSessionStatusUpdate(tenSessionStatus enSessionStatus)
{
   /*lint -esym(40,fvSessionStatus)fvSessionStatus Undeclared identifier */
   if (NULL != m_rMySPINRsrcMngrCb.fvSessionStatus)
   {
      (m_rMySPINRsrcMngrCb.fvSessionStatus)(enSessionStatus);
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINVideoResourceMngr::vSendDeviceDispCntxtUpdate()
 ***************************************************************************/
t_Void spi_tclMySPINVideoResourceMngr::vSendDeviceDispCntxtUpdate(t_Bool bDisplayFlag, tenDisplayContextReason enDisplayContextReason)
{
   /*lint -esym(40,fvDeviceDispCntxt)fvDeviceDispCntxt Undeclared identifier */
   if (NULL != m_rMySPINRsrcMngrCb.fvDeviceDispCntxt)
   {
      (m_rMySPINRsrcMngrCb.fvDeviceDispCntxt)(bDisplayFlag, enDisplayContextReason);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMySPINVideoResourceMngr::vFirstFBRenderedCallback()
 ***************************************************************************/
t_Void spi_tclMySPINVideoResourceMngr::vFirstFBRenderedCallback(const t_U32 u32DeviceID)
{
   ETG_TRACE_USR1(("spi_tclMySPINVideoResourceMngr- First Frame after display resumed for [%d]",u32DeviceID));

   m_fBUpateLock.s16Lock();

   m_bIsFirstFrameRendered = true;

   ETG_TRACE_USR2(("Play back requested[%d]", m_bIsPlaybackRequested));
   
   m_oCurMDFocusStateLock.s16Lock();
   m_enCurMDFocusState = e8MSPIN_VIDEOFOCUSSTATE_GAIN;
   m_oCurMDFocusStateLock.vUnlock();   

   if(true == m_bIsPlaybackRequested)
   {
       vSendDeviceDispCntxtUpdate(true, tenDisplayContextReason::e8DISPLAY_CONTEXT_REASON_UNKNOWN);
   }

   m_fBUpateLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclMySPINVideoResourceMngr::vPlaybackStartOnLaunchCallback()
 ***************************************************************************/
t_Void spi_tclMySPINVideoResourceMngr::vPlaybackStartOnLaunchCallback(const t_U32 u32DeviceID)
{
   ETG_TRACE_USR1(("spi_tclMySPINVideoResourceMngr::vPlaybackStartedOnLaunch. Request to enable projection for [%d]",u32DeviceID));

   m_fBUpateLock.s16Lock();

   m_bIsPlaybackRequested = true;

   ETG_TRACE_USR2(("First Frame Buffer Rendered already ? [%d]", m_bIsFirstFrameRendered));
   
   // m_oCurMDFocusStateLock.s16Lock();
   // tenMspinVideoFocusState enCurMDFocusState = m_enCurMDFocusState;
   // m_oCurMDFocusStateLock.vUnlock();

   // if ((e8MSPIN_VIDEOFOCUSSTATE_LOSS == enCurMDFocusState) || (e8MSPIN_VIDEOFOCUSSTATE_LOSS_TRANSIENT == enCurMDFocusState))
   // {
      // m_oVideoSetupLock.s16Lock();
      // //MD was in GAIN state and video playback was requested.
      // //Due to some user interaction, MD's state changed to LOSS. So rquest phone to stop video playback.
      // ETG_TRACE_USR2(("[DESC] Accessory has taken Video Focus due to user interactions. Stopping video"));
      // vSetVideoFocus(u32DeviceID,e8MSPIN_VIDEOFOCUS_NATIVE);

      // m_oVideoSetupLock.vUnlock();
   // }  

   m_oCurMDFocusStateLock.s16Lock();
   m_enCurMDFocusState = e8MSPIN_VIDEOFOCUSSTATE_GAIN;
   m_oCurMDFocusStateLock.vUnlock();   
   
   if(true == m_bIsFirstFrameRendered)
   {
      vSendDeviceDispCntxtUpdate(true, tenDisplayContextReason::e8DISPLAY_CONTEXT_REASON_UNKNOWN);
   }
   m_fBUpateLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclMySPINVideoResourceMngr::vLauncherAppInactiveCb()
 ***************************************************************************/
t_Void spi_tclMySPINVideoResourceMngr::vLauncherAppInactiveCb(t_Bool bIsInActive)
{
   ETG_TRACE_USR2(("[DESC]::vLauncherAppInactiveCb with [%d]", ETG_ENUM(BOOL,bIsInActive)));
//Code below is commented as we need to ignore the AppInactive callback. Retained for future use.
//It can be removed if not needed.
   // Send session status
/*   if (true == m_bMySPINSessionActive)
   {
      ETG_TRACE_USR1(("mySPIN Session is Active"));
      tenSessionStatus enSessionStatus = (bIsInActive) ? e8_SESSION_SUSPENDED : e8_SESSION_ACTIVE;
      spi_tclMySPINManager::vSetSessionStatus(enSessionStatus);
      vSendSessionStatusUpdate(enSessionStatus);

      m_fBUpateLock.s16Lock();
      if (enSessionStatus == e8_SESSION_SUSPENDED)
      {
         m_bIsFirstFrameRendered = false;
      }
      m_fBUpateLock.vUnlock();
   }*/
   if(bIsInActive == false)
   {
        m_oCurMDFocusStateLock.s16Lock();
        m_enCurMDFocusState = e8MSPIN_VIDEOFOCUSSTATE_GAIN;
        m_oCurMDFocusStateLock.vUnlock();
   }
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclMySPINVideoResourceMngr::vAppTransitionCb()
 ***************************************************************************/
t_Void spi_tclMySPINVideoResourceMngr::vAppTransitionCb(t_Bool bIsActive)
{
   ETG_TRACE_USR2(("[DESC]::vAppTransitionCb with [%d]",ETG_ENUM(BOOL,bIsActive)));
   // Send session status

   if (true == m_bMySPINSessionActive)
   {
      ETG_TRACE_USR1(("mySPIN Session is Active"));
      tenSessionStatus enSessionStatus = (bIsActive) ? e8_SESSION_SESPENDED_APP_TRANSITION : e8_SESSION_ACTIVE;
      spi_tclMySPINManager::vSetSessionStatus(enSessionStatus);
      vSendSessionStatusUpdate(enSessionStatus);

      m_fBUpateLock.s16Lock();
      if (enSessionStatus == e8_SESSION_SESPENDED_APP_TRANSITION)
      {
         m_bIsFirstFrameRendered = false;
         // m_oCurMDFocusStateLock.s16Lock();
         // m_enCurMDFocusState = e8MSPIN_VIDEOFOCUSSTATE_LOSS;
         // m_oCurMDFocusStateLock.vUnlock();
      } 
      else
      {
         m_oCurMDFocusStateLock.s16Lock();
         m_enCurMDFocusState = e8MSPIN_VIDEOFOCUSSTATE_GAIN;
         m_oCurMDFocusStateLock.vUnlock();
      }
      m_fBUpateLock.vUnlock();
   }
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclMySPINVideoResourceMngr::vIncomingCallCb()
 ***************************************************************************/
t_Void spi_tclMySPINVideoResourceMngr::vInitiateCallCb(t_String szTelePhoneNumber, t_String szDisplayText)
{
   ETG_TRACE_USR2(("[DESC]:Tel [%s]", szTelePhoneNumber.c_str()));
   ETG_TRACE_USR2(("[DESC]:Display Text [%s]", szDisplayText.c_str()));
   // Send session status

   if ((true == m_bMySPINSessionActive) && (NULL != m_rMySPINRsrcMngrCb.fvMySPINInitiateCall))
   {
      m_rMySPINRsrcMngrCb.fvMySPINInitiateCall(szTelePhoneNumber, szDisplayText);
   }
}
//lint \96restore
///////////////////////////////////////////////////////////////////////////////
// <EOF>
