/***********************************************************************/
/*!
 * \file    spi_tclMySPINAudioResourceMngr.cpp
 * \brief   MySPIN Audio Resource Manager
 *************************************************************************
 \verbatim

 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    MySPIN Audio resource manager
 AUTHOR:         tch5kor
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date         | Author                | Modification
 25.07.2016   | tch5kor               | Initial Version


 \endverbatim
 *************************************************************************/

/******************************************************************************
 | includes:
 | 1)system- and project- includes
 | 2)needed interfaces from external components
 | 3)internal and external interfaces from this component
 |----------------------------------------------------------------------------*/
#include "spi_tclResorceMngrDefines.h"
#include "spi_tclMySPINManager.h"
#include "spi_tclMySPINResourceMngr.h"
#include "spi_tclMySPINAudioResourceMngr.h"
#include "mySPINTypes.h"
//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_RSRCMNGR
#include "trcGenProj/Header/spi_tclMySPINAudioResourceMngr.cpp.trc.h"
#endif
#endif

struct trAudioMap
{
   tenAudioContext enAudioContext;
   tenIVIAudioType enIVIAudioType;
};
struct trResponseLookup
{
   tenAudioCtrlCommand enCtrlCommand;
   tenMySPINAudioType enMySPINAudioType;
   tenIVIAudioType enIVIAudioType;
   tenAudioResponse enAudioResponse;
   tenAudioDir enAudioDir;
};

#define AUDIO_TYPE_MAP_LOOKUP
static trAudioMap arAudioMap[] =
#include "mySPINAudioCfg.h"
#undef AUDIO_TYPE_MAP_LOOKUP

#define AUDIO_RESPONSE_LOOKUP
static trResponseLookup arResponseLookup[]
#include "mySPINAudioCfg.h"
#undef AUDIO_RESPONSE_LOOKUP

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | defines and macros (scope: global)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: global)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/
static spi_tclMySPINManager* spoMySPINMngr = NULL;
static t_U32 su32CurSelectedDevId = 0;

/***************************************************************************
 ** FUNCTION:  spi_tclMySPINAudioResourceMngr::spi_tclMySPINAudioResourceMngr()
 ***************************************************************************/
spi_tclMySPINAudioResourceMngr::spi_tclMySPINAudioResourceMngr(spi_tclMySPINResourceMngr* poMySPINRsrcMngr) :
   m_poMySPINRsrcMngr(poMySPINRsrcMngr),m_curIVIAudioType(e8_AudioType_TRANSIENT), 
   m_enPhoneAppState(e8SPI_PHONE_UNKNOWN), m_enIsVRActive(e8MSPIN_VR_UNKNOWN)
{
   ETG_TRACE_USR1((" spi_tclMySPINAudioResourceMngr() entered "));

   for (t_U32 u32KeyIndex = 0; u32KeyIndex < ((sizeof(arAudioMap)) / sizeof(trAudioMap)); ++u32KeyIndex)
   {
      m_audioMap.insert(std::pair<tenAudioContext,tenIVIAudioType >(
               arAudioMap[u32KeyIndex].enAudioContext, arAudioMap[u32KeyIndex].enIVIAudioType));
   }

   for (t_U32 u32KeyIndex = 0; u32KeyIndex < ((sizeof(arResponseLookup)) / sizeof(trResponseLookup)); ++u32KeyIndex)
   {
      //m_AudioResponseMap -- <<<tenCtrlCommand,tenMySPINAudioType>,tenIVIAudioType>,<tenAudioResponse,tenAudioDir>>

      m_AudioResponseMap.insert(std::pair<tMySPINAudioReqTuple, tMySPINResponseAudDirpair>(std::pair<
               tMySPINcmdAudioTypePair, tenIVIAudioType>(std::pair<tenAudioCtrlCommand, tenMySPINAudioType>(arResponseLookup[u32KeyIndex].enCtrlCommand,
               arResponseLookup[u32KeyIndex].enMySPINAudioType),
               arResponseLookup[u32KeyIndex].enIVIAudioType),
               std::pair<tenAudioResponse, tenAudioDir>(arResponseLookup[u32KeyIndex].enAudioResponse,
                        arResponseLookup[u32KeyIndex].enAudioDir)));

   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclMySPINAudioResourceMngr::~spi_tclMySPINAudioResourceMngr()
 ***************************************************************************/
spi_tclMySPINAudioResourceMngr::~spi_tclMySPINAudioResourceMngr()
{
   ETG_TRACE_USR1(("~spi_tclMySPINAudioResourceMngr() entered"));
   m_poMySPINRsrcMngr = NULL;
   spoMySPINMngr = NULL;
   m_mapAudioType.clear();
   m_AudioResponseMap.clear();
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclMySPINAudioResourceMngr::bInitialize()
 ***************************************************************************/
t_Bool spi_tclMySPINAudioResourceMngr::bInitialize()
{
   ETG_TRACE_USR1(("spi_tclMySPINAudioResourceMngr::bInitialize() entered"));

   t_Bool bRet = false;

   spoMySPINMngr = spi_tclMySPINManager::getInstance();
   SPI_NORMAL_ASSERT(NULL == spoMySPINMngr);

   if (NULL != spoMySPINMngr)
   {
      bRet = spoMySPINMngr->bRegisterObject((spi_tclMySPINRespAudio*) this);
   }
   return bRet;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMySPINAudioResourceMngr::vUninitialize()
 ***************************************************************************/
t_Void spi_tclMySPINAudioResourceMngr::vUnInitialize()
{

}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMySPINAudioResourceMngr::vOnSPISelectDeviceResult()
 ***************************************************************************/
t_Void spi_tclMySPINAudioResourceMngr::vOnSPISelectDeviceResult(t_U32 u32DevID, tenDeviceConnectionReq enDevConnReq,
         tenResponseCode enRespCode, tenErrorCode enErrorCode)
{
   ETG_TRACE_USR2(("[DESC]::vOnSPISelectDeviceResult: Select Type-%d  Response Code-%d", ETG_ENUM(CONNECTION_REQ,
            enDevConnReq), ETG_ENUM(RESPONSE_CODE, enRespCode)));

   SPI_INTENTIONALLY_UNUSED(enErrorCode);
   
   if((e8DEVCONNREQ_SELECT == enDevConnReq) && (e8SUCCESS == enRespCode))
   {
       su32CurSelectedDevId = u32DevID;
   }
   else
   {
        m_enIsVRActive = e8MSPIN_VR_UNKNOWN;
        su32CurSelectedDevId = 0;
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclMySPINAudioResourceMngr::vSetAccessoryAudioContext()
***************************************************************************/
t_Void spi_tclMySPINAudioResourceMngr::vSetAccessoryAudioContext(const t_U32 cou32DevId,
         const tenAudioContext coenAudioCntxt, t_Bool bReqFlag)
{
   SPI_INTENTIONALLY_UNUSED(cou32DevId);
   ETG_TRACE_USR1(("spi_tclMySPINAudioResourceMngr::vSetAccessoryAudioContext() entered coenAudioCntxt = %d, bReqFlag = %d",
                    ETG_ENUM(AUDIO_CONTEXT,coenAudioCntxt),ETG_ENUM(BOOL,bReqFlag)));
   m_curIVIAudioType = e8_AudioType_MAIN;
   tenAudioResponse enAudioResponse = e8_REQUEST_INVALID;
   tenBTVoiceSessionStatus enBTVoiceSessionStatus = e8BTVOICESESSION_STATUS_NONE;
   if (true == bReqFlag)
   {
      // Check who holds the audio, IVI or mySPIN.
      std::map<tenAudioContext, tenIVIAudioType>::const_iterator itrAudType;
      itrAudType = m_audioMap.find(coenAudioCntxt);

      if (m_audioMap.end() != itrAudType )
      {
         m_curIVIAudioType = itrAudType->second;
      }
   }

   switch (coenAudioCntxt)
   {
      case e8SPI_AUDIO_MAIN:
      case e8SPI_AUDIO_INTERNET_APP:
      case e8SPI_AUDIO_MAIN_DEFAULT:
      case e8SPI_AUDIO_PROJECTION:
      {
          enAudioResponse = (true == bReqFlag) ? e8_REQUEST_CLOSE : e8_REQUEST_INVALID;
      }
      break;
      case e8SPI_AUDIO_SPEECH_REC:
      case e8SPI_AUDIO_ADVISOR_PHONE:
      case e8SPI_AUDIO_EMER_PHONE:
      case e8SPI_AUDIO_PHONE:
      case e8SPI_AUDIO_INCOM_TONE:
      case e8SPI_AUDIO_SYNC_MSG:
      case e8SPI_AUDIO_ASYNC_MSG:
      case s8SPI_AUDIO_TRAFFIC:
      case e8SPI_AUDIO_LVM:
      case e8SPI_AUDIO_CUE:
      case e8SPI_AUDIO_SLEEP_MODE:
      {
          enAudioResponse = (true == bReqFlag) ? e8_REQUEST_SUSPEND : e8_REQUEST_INVALID;
          enBTVoiceSessionStatus = (true == bReqFlag) ? e8BTVOICESESSION_STATUS_UNAVAILABLE : e8BTVOICESESSION_STATUS_IDLE;
          vUpdateVRState(cou32DevId,enBTVoiceSessionStatus);
      }
      break;
      case e8SPI_AUDIO_STANDBY_MODE:
      {
          enAudioResponse = (true == bReqFlag) ? e8_REQUEST_SUSPEND : e8_REQUEST_OPEN;
          enBTVoiceSessionStatus = (true == bReqFlag) ? e8BTVOICESESSION_STATUS_UNAVAILABLE : e8BTVOICESESSION_STATUS_IDLE;
          vUpdateVRState(cou32DevId,enBTVoiceSessionStatus);
      }
      break;
      case e8SPI_AUDIO_MUTE:
      {
          enAudioResponse = (true == bReqFlag) ? e8_REQUEST_SUSPEND : e8_REQUEST_OPEN;
      }
      break;
      default:
          //do nothing
      break;
   }//switch (coenAudioCntxt)

   spi_tclMySPINManager* poMySPINMngr = spi_tclMySPINManager::getInstance();
   if ((NULL != poMySPINMngr) && (0 != su32CurSelectedDevId))
   {
       spi_tclMySPINCmdAudio * poCmdAudio = poMySPINMngr->poGetAudioInstance();
       spi_tclMySPINCmdSession* poMySPINCmdSession = poMySPINMngr->poGetSessionInstance();

       if ((NULL != poCmdAudio) && (NULL != poMySPINCmdSession))
       {
          t_Bool bRet = poCmdAudio->bSendAudioCmdResponse(e8AUD_INVALID,false,enAudioResponse,coenAudioCntxt);
       }
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINAudio::vAudioRequestCb(..
 ***************************************************************************/
t_Void spi_tclMySPINAudioResourceMngr::vAudioRequestCb(t_U32 u32DeviceId, tenAudioCtrlCommand enCommand, t_U32 u32RequestID,
         tenMySPINAudioType enAudioType)
{
   ETG_TRACE_USR1(("[DESC] Received Audio Request call-back for deviceID [%d],"
                   "command type [%d] , requestID [%d], Audio Type [%d]", u32DeviceId, enCommand, u32RequestID, enAudioType));
   trMySPINAudioCmd rMySPINAudioCmd;
   rMySPINAudioCmd.u32DeviceHandle = u32DeviceId;
   rMySPINAudioCmd.enCommand = enCommand;
   rMySPINAudioCmd.u32RequestID = u32RequestID;
   rMySPINAudioCmd.enAudioType = enAudioType;
   
   m_oAudioType.s16Lock();
   if(e8COMMAND_REQUEST == enCommand)
   {
       m_mapAudioType.insert(std::pair<t_U32,tenMySPINAudioType>(u32RequestID, enAudioType));
   }
   m_oAudioType.vUnlock();

   vHandleAudioCmd(rMySPINAudioCmd);

}
/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINAudio::vHandleAudioCmd(..
 ***************************************************************************/
t_Void spi_tclMySPINAudioResourceMngr::vHandleAudioCmd(trMySPINAudioCmd& rMySPINAudioCmd)
{
   //Obtain the responses to control requests. Releases are to be sent to mySPIN Audio directly.
    ETG_TRACE_USR2(("spi_tclMySPINAudioResourceMngr::vHandleAudioCmd m_curIVIAudioType = %d",m_curIVIAudioType));
   if (e8COMMAND_REQUEST == rMySPINAudioCmd.enCommand)
   {
      //Process the Control request only. Release will be directly handled by mySPIN Audio
      tMySPINcmdAudioTypePair MySPINcmdAudioTypePair;
      MySPINcmdAudioTypePair = std::make_pair(rMySPINAudioCmd.enCommand,rMySPINAudioCmd.enAudioType);

      tMySPINAudioReqTuple MySPINAudioReqTuple;
      MySPINAudioReqTuple = std::make_pair(MySPINcmdAudioTypePair,m_curIVIAudioType);

      std::map<tMySPINAudioReqTuple,tMySPINResponseAudDirpair>::iterator itrResponseLookUpPair;
      itrResponseLookUpPair = m_AudioResponseMap.find(MySPINAudioReqTuple);

      if (m_AudioResponseMap.end() != itrResponseLookUpPair)
      {
         //Found a response to the request from phone.
         tMySPINResponseAudDirpair MySPINResponseAudDir = itrResponseLookUpPair->second;
         rMySPINAudioCmd.enAudioResponse = MySPINResponseAudDir.first;
         rMySPINAudioCmd.enAudioDir = MySPINResponseAudDir.second;
      }

      ETG_TRACE_USR2(("[DESC] Handle Control Request with Response [%d], Audio Direction [%d]", rMySPINAudioCmd.enAudioResponse, rMySPINAudioCmd.enAudioDir));
   }
   else
   {
      m_oAudioType.s16Lock();
      std::map<t_U32,tenMySPINAudioType>::iterator itAudioType;
      itAudioType = m_mapAudioType.find(rMySPINAudioCmd.u32RequestID);
      ETG_TRACE_USR1(("spi_tclMySPINAudioResourceMngr::vHandleAudioCmd for command type [%d] , requestID [%d], Audio Type [%d]", 
                           rMySPINAudioCmd.enCommand, rMySPINAudioCmd.u32RequestID, rMySPINAudioCmd.enAudioType));      
      if(m_mapAudioType.end() != itAudioType)
      {
          rMySPINAudioCmd.enAudioType = itAudioType->second;
          ETG_TRACE_USR1(("spi_tclMySPINAudioResourceMngr::vHandleAudioCmd for Audio Type [%d]", rMySPINAudioCmd.enAudioType));  
          switch(rMySPINAudioCmd.enAudioType)
          {
              case e8_AUDIOTYPE_CHIME:
              case e8_AUDIOTYPE_ANNOUNCEMENT:
              case e8_AUDIOTYPE_CRITICALANNOUNCEMENT:
              {
                 rMySPINAudioCmd.enAudioDir = e8AUD_STEREO_MIX_OUT;
              }
              break;
              case e8_AUDIOTYPE_MAIN:
              {
                 rMySPINAudioCmd.enAudioDir = e8AUD_MAIN_OUT; 
              }
              break;
              default :
                //do nothing
                break;
          }
          m_mapAudioType.erase(itAudioType);
      }      
      m_oAudioType.vUnlock();
   }

   spi_tclMySPINManager* poMySPINMngr = spi_tclMySPINManager::getInstance();
   if (NULL != poMySPINMngr)
   {
     spi_tclMySPINCmdAudio * poCmdAudio = poMySPINMngr->poGetAudioInstance();
     if (NULL != poCmdAudio)
     {
        if(((e8SPI_PHONE_ACTIVE == m_enPhoneAppState) || 
        ((e8MSPIN_VR_ACTIVATING == m_enIsVRActive) || (e8MSPIN_VR_ACTIVATED == m_enIsVRActive))) 
        && (e8AUD_MAIN_OUT != rMySPINAudioCmd.enAudioDir))
        {
            tenDeviceSubCategory enDeviceSubCategory = e8DEVTYPE_UNKNWON;
            spi_tclMySPINManager::vGetDeviceSubCategory(enDeviceSubCategory);

            if(e8COMMAND_REQUEST == rMySPINAudioCmd.enCommand)
            {
               poCmdAudio->vSendAudioRespose(rMySPINAudioCmd.u32DeviceHandle, 
                    rMySPINAudioCmd.u32RequestID, e8_REQUEST_OPEN);
            }
            else
            {
               t_Bool bIsAudReqAvailable = poCmdAudio->bIsAudioRequestAvailable(rMySPINAudioCmd);
               if(true == bIsAudReqAvailable)
               {
                  poCmdAudio->vHandleAudioCmd(rMySPINAudioCmd);
               }
               else
               {
                  poCmdAudio->vSendAudioRespose(rMySPINAudioCmd.u32DeviceHandle,
                            rMySPINAudioCmd.u32RequestID, e8_REQUEST_CLOSE);
               }
            }
        }
        else if(e8COMMAND_RELEASE == rMySPINAudioCmd.enCommand)
        {
            t_Bool bIsAudReqAvailable = poCmdAudio->bIsAudioRequestAvailable(rMySPINAudioCmd);
            if(true == bIsAudReqAvailable)
            {
               poCmdAudio->vHandleAudioCmd(rMySPINAudioCmd);
            }
            else
            {
               poCmdAudio->vSendAudioRespose(rMySPINAudioCmd.u32DeviceHandle, 
                         rMySPINAudioCmd.u32RequestID, e8_REQUEST_CLOSE);
            }
        }
        else
        {
            poCmdAudio->vHandleAudioCmd(rMySPINAudioCmd);
        }
     }
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINAudioResourceMngr::vSetAccessoryAppState(...
 ***************************************************************************/
t_Void spi_tclMySPINAudioResourceMngr::vSetAccessoryAppState(tenPhoneAppState enPhoneAppState)
{
   ETG_TRACE_USR1(("spi_tclMySPINAudioResourceMngr::vSetAccessoryAppState : PHONE = %d",
               ETG_ENUM(PHONE_APP_STATE,enPhoneAppState))); 
               
    m_enPhoneAppState = enPhoneAppState;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINAudioResourceMngr::vSetMspinVRStatus(...
 ***************************************************************************/
t_Void spi_tclMySPINAudioResourceMngr::vSetMspinVRStatus(tenMspinVRStatus enIsVRActive)
{
    m_enIsVRActive = enIsVRActive;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINAudioResourceMngr::vSetMspinVRStatus(...
 ***************************************************************************/
t_Void spi_tclMySPINAudioResourceMngr::vUpdateVRState(const t_U32 cou32DevId,
                                                      tenBTVoiceSessionStatus enBTVoiceSessionStatus)
{
    if(su32CurSelectedDevId != 0)
    {
       t_Bool bIsHFP = false;
       t_String szBTAddress;
       spi_tclMySPINManager::vGetBTConnInfo(su32CurSelectedDevId, bIsHFP, szBTAddress);
       tenMspinVRStatus enVRState = m_poMySPINRsrcMngr->enGetVRState();
    
       if(false == bIsHFP)
       {
           enBTVoiceSessionStatus = e8BTVOICESESSION_STATUS_UNAVAILABLE;
       }
    
       spi_tclMySPINManager* poMySPINMngr = spi_tclMySPINManager::getInstance();
       if (NULL != poMySPINMngr)
       {
          spi_tclMySPINCmdSession* poMySPINCmdSession = poMySPINMngr->poGetSessionInstance();
          if (NULL != poMySPINCmdSession)
          {
              poMySPINCmdSession->vSendVREndResponse(cou32DevId,enBTVoiceSessionStatus,enVRState);
          }
       }
       if(e8BTVOICESESSION_STATUS_UNAVAILABLE == enBTVoiceSessionStatus)
       {
           m_poMySPINRsrcMngr->vSetVRState(e8MSPIN_VR_ENDED);
           m_poMySPINRsrcMngr->vUpdateSpeechAppState(e8SPI_SPEECH_END);
       }
    }
}
///////////////////////////////////////////////////////////////////////////////
// <EOF>

