/*!
 *******************************************************************************
 * \file              spi_tclMySPINInputHandler.h
 * \brief             SPI input handler for MySPIN devices
 *******************************************************************************
 \verbatim
 PROJECT:       Gen3
 SW-COMPONENT:  Smart Phone Integration
 DESCRIPTION:   Input handler class to send input events from Headunit to
                MySPIN Device
 AUTHOR:         Pruthvi Thej Nagaraju
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date        | Author                | Modification
 02.11.2015  | Pruthvi Thej Nagaraju | Initial Version
 12.04.2016  | Chaitra Srinivasa     | Trace message clean up
 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/

#define VNC_USE_STDINT_H

#include "spi_tclMySPINInputHandler.h"
#include "spi_tclMySPINManager.h"
#include <algorithm>
#include "spi_tclExtCompManager.h"
#include "spi_tclExtCmdNativeTransport.h"

#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_INPUTHANDLER
#include "trcGenProj/Header/spi_tclMySPINInputHandler.cpp.trc.h"
#endif
#endif

//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e19 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e58 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e48 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e808 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e64 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e746 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e515 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e516 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported	


/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

static tenKeyCode aSoftKeys[] =
{
   e32DEV_HOME
};
static trSoftKeyCode aSoftKeyCode [] =
{
   {e32DEV_HOME, 0x0001}
};


static trAVPKeyCode aAVPKeyCode [] =
{
   {e32MULTIMEDIA_PLAY, e8_CMD_PLAY},
   {e32MULTIMEDIA_PAUSE,e8_CMD_PAUSE},
   {e32MULTIMEDIA_NEXT, e8_CMD_NEXTTRACK},
   {e32MULTIMEDIA_PREVIOUS,e8_CMD_PREVTRACK},
   {e32MULTIMEDIA_FORWARD,e8_CMD_FORWARD},
   {e32MULTIMEDIA_REWIND,e8_CMD_REWIND}
};

static tenKeyCode aCustomKey [] =
{
   e32DEV_PTT,
   e32DEV_MENU,
   e32DEV_BACKWARD,
   e32TWODKNOB0_SHIFT_UP,
   e32TWODKNOB0_SHIFT_DOWN,
   e32TWODKNOB0_SHIFT_LEFT,
   e32TWODKNOB0_SHIFT_RIGHT
};

static trCustomKey aCustomKeyCode [] = 
{
   {e32DEV_PTT,MSPIN_CUSTOMDATA_KEYCODE_PUSHTOTALK},
   {e32DEV_MENU, MSPIN_CUSTOMDATA_KEYCODE_OK},
   {e32DEV_BACKWARD, MSPIN_CUSTOMDATA_KEYCODE_BACK},
   {e32TWODKNOB0_SHIFT_UP, MSPIN_CUSTOMDATA_KEYCODE_DPADUP},
   {e32TWODKNOB0_SHIFT_DOWN, MSPIN_CUSTOMDATA_KEYCODE_DPADDOWN},
   {e32TWODKNOB0_SHIFT_LEFT, MSPIN_CUSTOMDATA_KEYCODE_DPADLEFT},
   {e32TWODKNOB0_SHIFT_RIGHT, MSPIN_CUSTOMDATA_KEYCODE_DPADRIGHT}
};

/***************************************************************************
** FUNCTION:  spi_tclMySPINInputHandler::spi_tclMySPINInputHandler()
***************************************************************************/
spi_tclMySPINInputHandler::spi_tclMySPINInputHandler()
{
   ETG_TRACE_USR1(("spi_tclMySPINInputHandler::spi_tclMySPINInputHandler entered \n"));

   for (t_U32 u32KeyIndex = 0; u32KeyIndex < ((sizeof(aSoftKeyCode)) / sizeof(trSoftKeyCode)); ++u32KeyIndex)
   {
      m_keyBitMaskMap.insert(std::pair<tenKeyCode,t_U32 >(
             aSoftKeyCode[u32KeyIndex].enKeyCode, aSoftKeyCode[u32KeyIndex].u32BitMask));
   }

   for (t_U32 u32KeyIndex = 0; u32KeyIndex < ((sizeof(aAVPKeyCode)) / sizeof(trAVPKeyCode)); ++u32KeyIndex)
   {
      m_keyAVPMap.insert(std::pair<tenKeyCode,t_U8 >(
            aAVPKeyCode[u32KeyIndex].enKeyCode, aAVPKeyCode[u32KeyIndex].avpCtrlCode));
   }
   
   for (t_U32 u32KeyIndex = 0; u32KeyIndex < ((sizeof(aCustomKeyCode)) / sizeof(trCustomKey)); ++u32KeyIndex)
   {
      m_CustomKeyMap.insert(std::pair<tenKeyCode,t_U8 >(
            aCustomKeyCode[u32KeyIndex].enKeyCode, aCustomKeyCode[u32KeyIndex].u8CustomKeyCode));
   }
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclMySPINInputHandler::~spi_tclMySPINInputHandler()
***************************************************************************/
spi_tclMySPINInputHandler::~spi_tclMySPINInputHandler()
{
   ETG_TRACE_USR1(("spi_tclMySPINInputHandler::~spi_tclMySPINInputHandler entered \n"));
}

/***************************************************************************
** FUNCTION:  t_Void  spi_tclMySPINInputHandler::vSelectDevice
***************************************************************************/
t_Void  spi_tclMySPINInputHandler::vSelectDevice(const t_U32 cou32DevId,
   const tenDeviceConnectionReq coenConnReq)
{
	/*lint -esym(40,fvSelectDeviceResp)fvSelectDeviceResp Undeclared identifier */
  ETG_TRACE_USR2(("[DESC]::vSelectDevice:Device-0x%x \n",cou32DevId));
  SPI_INTENTIONALLY_UNUSED(coenConnReq);
  if(NULL != m_rInputCallbacks.fvSelectDeviceResp)
  {
     (m_rInputCallbacks.fvSelectDeviceResp)(cou32DevId,e8NO_ERRORS);
  }//if(NULL != m_rVideoCallbacks.fpvSelectDeviceCb
}

/***************************************************************************
** FUNCTION:  t_Void  spi_tclMySPINInputHandler::vRegisterVideoCallbacks
***************************************************************************/
t_Void  spi_tclMySPINInputHandler::vRegisterInputCallbacks(const trInputCallbacks& corfrInputCallbacks)
{
  ETG_TRACE_USR1(("spi_tclMySPINInputHandler:vRegisterInputCallbacks() \n"));
   //Copy
   m_rInputCallbacks = corfrInputCallbacks;
}

/***************************************************************************
** FUNCTION:  spi_tclMySPINInputHandler::vOnSelectDeviceResult()
***************************************************************************/
t_Void spi_tclMySPINInputHandler::vOnSelectDeviceResult(const t_U32 cou32DevId,
                                                        const tenDeviceConnectionReq coenConnReq,
                                                        const tenResponseCode coenRespCode)
{
   /*lint -esym(40,fvSrvKeyCapabilitiesInfo)fvSrvKeyCapabilitiesInfo Undeclared identifier */
   ETG_TRACE_USR2(("[DESC]::vOnSelectDeviceResult:Dev-0x%x ConnReq-%d, RespCode-%d",
                   cou32DevId,ETG_ENUM(CONNECTION_REQ,coenConnReq),coenRespCode));
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclMySPINInputHandler::vProcessKeyEvents
***************************************************************************/
t_Void spi_tclMySPINInputHandler::vProcessKeyEvents(t_U32 u32DeviceHandle,
                                                tenKeyMode enKeyMode, tenKeyCode enKeyCode)const
{
   ETG_TRACE_USR1(("spi_tclMySPINInputHandler:vProcessKeyEvents()"));

   spi_tclMySPINManager* poMySPINMngr = spi_tclMySPINManager::getInstance();

   if( NULL != poMySPINMngr )
   {
      /*
       * Note Forward the Key events to cmdInput in case they are soft keys.
       * Else the key should be sent to mediaPlayer / Audio for AVP
       */
      tenMySPINKeyType  enMySPINKeyType = enGetKeyType(enKeyCode);

      switch (enMySPINKeyType)
      {
         case e8SOFT_KEY:
            {
               //Send to mySPIN TA via cmdInput
               ETG_TRACE_USR1(("spi_tclMySPINInputHandler::vProcessKeyEvents:Send Soft Key"));
               spi_tclMySPINCmdInput* poCmdInput = poMySPINMngr->poGetInputInstance();
               if (NULL != poCmdInput)
               {
                  //Check if the soft key is supported by the phone
                  t_U32 u32SupportedKeys = poCmdInput->vGetSupportedKeys(u32DeviceHandle);

                  std::map<tenKeyCode, t_U32>::const_iterator itrKeyCode;
                  itrKeyCode = m_keyBitMaskMap.find(enKeyCode);

                  if( (m_keyBitMaskMap.end() != itrKeyCode) &&
                       (itrKeyCode->second & u32SupportedKeys) == itrKeyCode->second )
                  {
                     ETG_TRACE_USR2(("[DESC]::Valid Soft key found with bit mask [0x000%d]",itrKeyCode->second ));
                     poCmdInput->vSendKeyEvent(u32DeviceHandle,enKeyMode,enKeyCode);
                  }
               }
            }
            break;

         case e8AVPCTRL_KEY:
            {
               ETG_TRACE_USR1(("spi_tclMySPINInputHandler::vProcessKeyEvents:Send AVRC CMd"));
               spi_tclMySPINCmdAudio* poCmdAudio = poMySPINMngr->poGetAudioInstance();
               if (NULL != poCmdAudio)
               {
                  std::map<tenKeyCode, t_U8>::const_iterator itrKeyCode;
                  itrKeyCode = m_keyAVPMap.find(enKeyCode);

                  //Check if the Android device is connected over BT
                  t_Bool bIsBTConnected = poCmdAudio->bGetBTConnectionStatus(u32DeviceHandle);

                  if((m_keyAVPMap.end() != itrKeyCode) && (true == bIsBTConnected))
                  {
                     ETG_TRACE_USR4(("[PARAM]:vProcessKeyEvents-Send AVRC CMd [%d]",itrKeyCode->second));
                     poCmdAudio->bSendAVPCtrlCmd(itrKeyCode->second,
                                                 static_cast<t_U8>(enKeyMode));
                  }
               }
            }
            break;

         case e8IAP_KEY:
            {
                spi_tclExtCompManager *poExtCompMgr = spi_tclExtCompManager::getInstance();
                spi_tclExtCmdNativeTransportIntf *poExtCmdNativeTransportIntf = NULL;
                if(NULL != poExtCompMgr)
                {
                    poExtCmdNativeTransportIntf = poExtCompMgr->poGetCmdNativeTransportIntfInst();
                }
                if(NULL != poExtCmdNativeTransportIntf)
                {
                    t_Bool bResult = poExtCmdNativeTransportIntf->bSendPlayBackCtrlEvent(u32DeviceHandle,enKeyCode,enKeyMode);
                    ETG_TRACE_USR4(("[PARAM]:vProcessKeyEvents-Key Code key sent [%d]", ETG_ENUM(BOOL,bResult)));
                }
            }
            break;

         case e8CUSTOM_KEY:
            {         
               //Send to mySPIN TA via cmdInput
               ETG_TRACE_USR1(("spi_tclMySPINInputHandler::vProcessKeyEvents:Send Custom Key"));
               spi_tclMySPINCmdInput* poCmdInput = poMySPINMngr->poGetInputInstance();
               if (NULL != poCmdInput)
               {
                  std::map<tenKeyCode, t_U8>::const_iterator itrKeyCode;
                  itrKeyCode = m_CustomKeyMap.find(enKeyCode);
                  
                  if(m_CustomKeyMap.end() != itrKeyCode)
                  {
                     ETG_TRACE_USR4(("[PARAM]:vProcessKeyEvents-Send Custom key event [%d]",
                           ETG_ENUM(SPI_KEY_CODE,enKeyCode)));
                     if(e32DEV_PTT == enKeyCode)
                     {
                         if((e8KEY_PRESS == enKeyMode) || (e8KEY_LONGPRESS == enKeyMode))
                         {
                             poCmdInput->vSendCustomKeyEvent(u32DeviceHandle,e8MSPIN_CUSTOMDATA_KEYTYPE_CLICK,enKeyCode);
                         }
                     }
                     else
                     {
                         if(e8KEY_PRESS == enKeyMode)
                         {
                             poCmdInput->vSendCustomKeyEvent(u32DeviceHandle,e8MSPIN_CUSTOMDATA_KEYTYPE_PRESS,enKeyCode);
                         }
                         else if(e8KEY_RELEASE == enKeyMode)
                         {
                             poCmdInput->vSendCustomKeyEvent(u32DeviceHandle,e8MSPIN_CUSTOMDATA_KEYTYPE_RELEASE,enKeyCode);
                         }
                     }
                  } 
               }            
            }
            break;
            
         default:
            {
               //Invalid Key type
            }
      }
   }

}

/***************************************************************************
** FUNCTION:  tenMySPINKeyType spi_tclMySPINInputHandler::vGetKeyType
***************************************************************************/
tenMySPINKeyType spi_tclMySPINInputHandler::enGetKeyType(tenKeyCode enKeyCode)
{
   // This routine determines the type of key.

   tenMySPINKeyType enMySPINKeyType = e8INVALID_KEY;

   //Get the number of soft keys
   t_U8 u8NumberOfSoftKeys = sizeof(aSoftKeys)/sizeof(t_U32);
   tenKeyCode* poIndex =  std::find(aSoftKeys,aSoftKeys+u8NumberOfSoftKeys,enKeyCode);

   t_U8 u8NumberOfCustomKeys = sizeof(aCustomKey)/sizeof(t_U32);
   tenKeyCode* poCKIndex = std::find(aCustomKey, aCustomKey+u8NumberOfCustomKeys, enKeyCode);

   if( (aSoftKeys + u8NumberOfSoftKeys) != poIndex )
   {
      enMySPINKeyType = e8SOFT_KEY;
   }
   else if( (aCustomKey+u8NumberOfCustomKeys) != poCKIndex)
   {
       enMySPINKeyType = e8CUSTOM_KEY;
   }
   else
   {
      tenDeviceSubCategory enDeviceSubCategory;
      spi_tclMySPINManager::vGetDeviceSubCategory(enDeviceSubCategory);
      enMySPINKeyType = static_cast<tenMySPINKeyType>(enDeviceSubCategory);
   }

   return enMySPINKeyType;
}

/***************************************************************************
** FUNCTION: t_Void spi_tclMySPINInputHandler::vProcessKnobKeyEvents()
***************************************************************************/
t_Void spi_tclMySPINInputHandler::vProcessKnobKeyEvents(t_U32 u32DeviceHandle,t_S8 s8EncoderDeltaCnt) const
{
   ETG_TRACE_USR1(("spi_tclMySPINInputHandler::vProcessKnobKeyEvents entered "));
   ETG_TRACE_USR2(("[DESC]:vProcessKnobKeyEvents: Device Id-0x%x Encoder Delta Count-%d",
             u32DeviceHandle,s8EncoderDeltaCnt));
   spi_tclMySPINManager* poMySPINMngr = spi_tclMySPINManager::getInstance();
   if(NULL != poMySPINMngr)
   {
       spi_tclMySPINCmdInput* poCmdInput = poMySPINMngr->poGetInputInstance();
       if (NULL != poCmdInput)
       {
           poCmdInput->vSendKnobKeyEvent(u32DeviceHandle, s8EncoderDeltaCnt);
       }
   }
}

//lint \96restore
