/*!
 *******************************************************************************
 * \file             spi_tclMySPINDataService.cpp
 * \brief            MySPIN Data Service class
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    MySPIN Data Service class implements Data Service Info Management for
 MySPIN capable devices. This class must be derived from base Data Service class.
 AUTHOR:         tch5kor
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date        | Author                | Modification
 24.03.2016  | tch5kor               | Initial Version

 \endverbatim
 ******************************************************************************/

#include "spi_tclMySPINDataService.h"
#include "spi_tclMySPINManager.h"
#include "NmeaEncoder.h"
#include "spi_tclExtCompManager.h"
#include "spi_tclExtCmdNavData.h"
#include <algorithm>
//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
   #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_DATASERVICE
      #include "trcGenProj/Header/spi_tclMySPINDataService.cpp.trc.h"
   #endif
#endif

//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e19 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e58 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e48 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e808 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e64 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e746 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e515 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e516 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported	
/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/
#define MEDIAPLAYER_REQUEST 2
#define TARGETADAPTER_REQUEST 4

#define MYSPIN_LOCATION_DATA_REQUEST 2
#define MYSPIN_STOP_REQUEST 0

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/
/*  GPRMCDataStatusValues have been declared as t_U8 to avoid
 *  Static code analyzer complaint for not matching data type */
static const t_U8 scou8GPRMCDataStatusValueA = 1;
static const t_U8 scou8GPRMCDataStatusValueV = 2;
static const t_U8 scou8GPRMCDataStatusValueX = 4;
/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

/****************************************************************************
 *********************************PUBLIC*************************************
 ***************************************************************************/

/***************************************************************************
 ** FUNCTION:  spi_tclMySPINDataService::spi_tclMySPINDataService
 ***************************************************************************/
spi_tclMySPINDataService::spi_tclMySPINDataService(const trDataServiceCb& rfcorDataServiceCb)
                        :m_rDataServiceCb(rfcorDataServiceCb),
                         m_u8RequestedBy(0),
                         m_enGeoCoordinateSystemType(e8GEO_COORDINATE_SYSTEM_WGS),
                         m_bReqLocDataSubscription(false),
                         m_bIsNativeNavigationEnabled(false)
{
   ETG_TRACE_USR1(("spi_tclMySPINDataService() entered "));
}

/***************************************************************************
 ** FUNCTION:  spi_tclMySPINDataService::spi_tclMySPINDataService
 ***************************************************************************/
spi_tclMySPINDataService::~spi_tclMySPINDataService()
{
   ETG_TRACE_USR1(("~spi_tclMySPINDataService() entered "));
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclMySPINDataService::bInitialise()
 ***************************************************************************/
t_Bool spi_tclMySPINDataService::bInitialise()
{
   ETG_TRACE_USR1(("spi_tclMySPINDataService::bInitialise() entered "));
   //! Add code
   return true;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclMySPINDataService::bUninitialise()
 ***************************************************************************/
t_Bool spi_tclMySPINDataService::bUninitialise()
{
   ETG_TRACE_USR1(("spi_tclMySPINDataService::bUninitialise() entered "));
   //! Add code
   return true;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclMySPINDataService::bUninitialise()
 ***************************************************************************/
t_Void spi_tclMySPINDataService::vSelectDevice(const t_U32 cou32DevId,
         tenDeviceConnectionReq enConnReq)
{
   spi_tclMySPINManager* poMySPINMngr = spi_tclMySPINManager::getInstance();
   if ((NULL != poMySPINMngr) && (e8DEVCONNREQ_SELECT == enConnReq))
   {
      su32CurSelectedDevId = cou32DevId;

   if(NULL != m_poDataServiceSettings)
   {
       m_bIsNativeNavigationEnabled = m_poDataServiceSettings->bGetNativeNavigationEnabled();
   }
      ETG_TRACE_USR1(("spi_tclMySPINDataService::vSelectDevice() : Native Navigation setting is = %d ",m_bIsNativeNavigationEnabled ));
      spi_tclMySPINCmdVehicleData * poCmdVehData = poMySPINMngr->poGetVehDataInstance();
      if (NULL != poCmdVehData )
      {
         m_vVehicleDataRequestCb = std::bind(&spi_tclMySPINDataService::vOnVehDataCB,
                                   this,
                                   SPI_FUNC_PLACEHOLDERS_3);

         poCmdVehData->vRegisterCallbacks(cou32DevId,m_vVehicleDataRequestCb,m_bIsNativeNavigationEnabled);
      }
   }
   else if(e8DEVCONNREQ_DESELECT == enConnReq)
   {
       vPostSetLocDataSubscription(false);
   }
}


/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMySPINDataService::vOnSelectDeviceResult(t_U32...)
 ***************************************************************************/
t_Void spi_tclMySPINDataService::vOnSelectDeviceResult(t_U32 u32DeviceHandle,
        const tenDeviceConnectionReq coenConnReq,
        const tenResponseCode coenRespCode)
{
   ETG_TRACE_USR2(("[DESC]::vOnSelectDeviceResult() entered: u32DeviceHandle = %u ", u32DeviceHandle));
   SPI_INTENTIONALLY_UNUSED(coenConnReq);
   SPI_INTENTIONALLY_UNUSED(coenRespCode);
   if(e8FAILURE == coenRespCode)
   {
        su32CurSelectedDevId = 0;
        if(true == m_bReqLocDataSubscription)
        {
            vPostSetLocDataSubscription(false);
        }
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMySPINDataService::vOnDeselectDeviceResult(t_U32...)
 ***************************************************************************/
t_Void spi_tclMySPINDataService::vOnDeselectDeviceResult(t_U32 u32DeviceHandle)
{
   ETG_TRACE_USR2(("[DESC]::vOnDeselectDeviceResult() entered: u32DeviceHandle = %u ", u32DeviceHandle));
   su32CurSelectedDevId = 0;
   // Resetting the coordinate system to default
   m_enGeoCoordinateSystemType = m_rMySPINDataServiceConfigData.enGeoCoordinateSystemType;
}
/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMySPINDataService::vOnData(t_U32...)
 ***************************************************************************/
t_Void spi_tclMySPINDataService::vOnData(const trVehicleData& rfcoVehicleData, t_Bool bSolicited)
{
   SPI_INTENTIONALLY_UNUSED(bSolicited)
   spi_tclMySPINManager* poMySPINMngr = spi_tclMySPINManager::getInstance();
   m_rVehicleData = rfcoVehicleData;

   if (NULL != poMySPINMngr)
   {
      spi_tclMySPINCmdVehicleData * poCmdVehData = poMySPINMngr->poGetVehDataInstance();
      if (NULL != poCmdVehData )
      {
         poCmdVehData->vSetMovingStatus(su32CurSelectedDevId,rfcoVehicleData.enVehMovState);
      }
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclMySPINDataService::vOnData(const trSensorData&...)
***************************************************************************/
t_Void spi_tclMySPINDataService::vOnData(const trSensorData& rfcorSensorData)
{
   ETG_TRACE_USR1(("Sensor Data Received"));

   //Get the type of navigation supported

   //DONOT CHECK for null, but check anyways
   if(NULL != m_poDataServiceSettings)
   {
      t_Bool bIsBoschNavi = m_poDataServiceSettings->bIsBochNavigation();
      if (false == bIsBoschNavi)
      {
         NmeaEncoder oNmeaEnc(rfcorSensorData,m_rVehicleData);

         t_String szNmeaGGASentence, szNmeaRMCSentence;
         tenNmeaDataSource enNmeaDataSource = e8NMEA_SOURCE_SENSOR;

         //! Check and get GPGGA sentence if required.
         if (m_NmeaSentencesList.end()
                  != std::find(m_NmeaSentencesList.begin(), m_NmeaSentencesList.end(), e8NMEA_GPGGA))
         {
            szNmeaGGASentence = oNmeaEnc.szGetNmeaGGASentence(enNmeaDataSource);

         }
         //! Check and get GPRMC sentence if required.
         if (m_NmeaSentencesList.end()
                  != std::find(m_NmeaSentencesList.begin(), m_NmeaSentencesList.end(), e8NMEA_GPRMC))
         {
            szNmeaRMCSentence = oNmeaEnc.szGetNmeaRMCSentence(enNmeaDataSource);
         }

         if (TARGETADAPTER_REQUEST == (m_u8RequestedBy & TARGETADAPTER_REQUEST))
         {
            spi_tclMySPINManager* poMySPINMngr = spi_tclMySPINManager::getInstance();

            if (NULL != poMySPINMngr)
            {
               spi_tclMySPINCmdVehicleData * poCmdVehData = poMySPINMngr->poGetVehDataInstance();
               if (NULL != poCmdVehData)
               {

                  poCmdVehData->vSendPositionData(su32CurSelectedDevId, szNmeaGGASentence);
                  poCmdVehData->vSendPositionData(su32CurSelectedDevId, szNmeaRMCSentence);
               }
            }
         }
         if (MEDIAPLAYER_REQUEST == (m_u8RequestedBy & MEDIAPLAYER_REQUEST))
         {
            spi_tclExtCompManager *poExtCompMgr = spi_tclExtCompManager::getInstance();
            spi_tclExtCmdNavDataIntf *poExtCmdNavDataIntf = NULL;
            if(NULL != poExtCompMgr)
            {
                poExtCmdNavDataIntf = poExtCompMgr->poGetCmdNavDataIntfInst();
            }
            if(NULL != poExtCmdNavDataIntf)
            {
                t_String szGPGSVData, szGPHDTData;
                t_Bool bResult = poExtCmdNavDataIntf->bTransferGPSData(su32CurSelectedDevId, "", szNmeaGGASentence,
                                                                        szNmeaRMCSentence,szGPGSVData,szGPHDTData);
                ETG_TRACE_USR2(("GPS Data sent [%d]", ETG_ENUM(BOOL,bResult)));
            }
         }
      }
      else
      {
         m_rSensorData = rfcorSensorData;
      }
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclMySPINDataService::vStartLocationData(..)
***************************************************************************/
t_Void spi_tclMySPINDataService::vStartLocationData(const std::vector<tenNmeaSentenceType>& rfcoNmeaSentencesList,
                                                    t_U8 u8RequestedBy)
{
   ETG_TRACE_USR1(("Request to start Location Data by [%d]",u8RequestedBy));

   //What types of sentences are requested to start
   m_NmeaSentencesList = rfcoNmeaSentencesList;

   //Who asked for location data
   m_u8RequestedBy |= u8RequestedBy;

   //Start receiving location data
   if (MYSPIN_STOP_REQUEST < m_u8RequestedBy)
   {
      vPostSetLocDataSubscription(true);
   }

}

/***************************************************************************
** FUNCTION:  t_Void spi_tclMySPINDataService::vStopLocationData(..)
***************************************************************************/
t_Void spi_tclMySPINDataService::vStopLocationData(const std::vector<tenNmeaSentenceType>& rfcoNmeaSentencesList,
                                                   t_U8 u8RequestedBy)
{
   ETG_TRACE_USR1(("Request to stop Location Data by [%d]",u8RequestedBy));

   if (m_NmeaSentencesList.size() == rfcoNmeaSentencesList.size())
   {
      // If all the
      m_u8RequestedBy &= ~u8RequestedBy;
   }
   else
   {
      //What types of sentences are requested to stop, remove them
      for (std::vector<tenNmeaSentenceType>::const_iterator itrSentenceType = rfcoNmeaSentencesList.begin();
            rfcoNmeaSentencesList.end()!=itrSentenceType; itrSentenceType++)
      {
        // m_NmeaSentencesList.pop_back(itrSentenceType);
      }
   }

   //Check which component has requested a stop
   if (MYSPIN_STOP_REQUEST == m_u8RequestedBy)
   {
      vPostSetLocDataSubscription(false);
   }

}

/***************************************************************************
** FUNCTION:  t_Void spi_tclMySPINDataService::vPostSetLocDataSubscription(..)
 ***************************************************************************/
t_Void spi_tclMySPINDataService::vPostSetLocDataSubscription(t_Bool bSubscriptionOn)
{

   /*lint -esym(40,fvSubscribeForLocationData)fvSubscribeForLocationData Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclMySPINDataService::vPostSetLocDataSubscription() entered: bSubscriptionOn = %u ",
         ETG_ENUM(BOOL, bSubscriptionOn)));

   m_bReqLocDataSubscription = bSubscriptionOn;
   
   //Ask Data Service to register for property updates from Pos FI and Sensor FI
   if ((NULL != m_rDataServiceCb.fvSubscribeForData) && (0 != su32CurSelectedDevId) && (true == m_bIsNativeNavigationEnabled))
   {
      if(NULL != m_poDataServiceSettings)
      {
         t_Bool bIsBoschNavi = m_poDataServiceSettings->bIsBochNavigation();
         if(bIsBoschNavi == true)
         {
            m_rDataServiceCb.fvSubscribeForData(bSubscriptionOn, e8GPS_DATA);
         }
      }
      m_rDataServiceCb.fvSubscribeForData(bSubscriptionOn, e8GNSS_DATA);
   }
}
/***************************************************************************
** FUNCTION:  t_Void spi_tclMySPINDataService::vOnVehDataCB(..)
 ***************************************************************************/
t_Void spi_tclMySPINDataService::vOnVehDataCB(t_U32 u32DeviceHandle, t_Bool bStart, t_U8 u8VehDataType)
{
   ETG_TRACE_USR1(("Vehicle Data Callback Recieved for [%d] ",u32DeviceHandle));

   // For position data   : Start/Stop location data

   if (MYSPIN_LOCATION_DATA_REQUEST == (u8VehDataType & MYSPIN_LOCATION_DATA_REQUEST))
   {
      std::vector<tenNmeaSentenceType> NmeaSentencesList;
      NmeaSentencesList.push_back(e8NMEA_GPGGA);
      NmeaSentencesList.push_back(e8NMEA_GPRMC);

     (bStart)? vStartLocationData(NmeaSentencesList,TARGETADAPTER_REQUEST):
               vStopLocationData(NmeaSentencesList,TARGETADAPTER_REQUEST);
   }

}
/***************************************************************************
** FUNCTION:  t_Void spi_tclMySPINDataService::vOnData(...)
***************************************************************************/
t_Void spi_tclMySPINDataService::vOnData(const trGPSData& rfcorGpsData)
{

   //DONOT CHECK for null, but check anyways
   if(NULL != m_poDataServiceSettings)
   {
      t_Bool bIsBoschNavi = m_poDataServiceSettings->bIsBochNavigation();

      if (true == bIsBoschNavi)
      {
         //Store the GPS Data
         trGPSData rGPSData =  rfcorGpsData;
         rGPSData.enGeoCoordinateSystemType = m_enGeoCoordinateSystemType;
         NmeaEncoder oNmeaEnc(rGPSData, m_rSensorData, m_rVehicleData);

         t_String szNmeaGGASentence, szNmeaRMCSentence;
         tenNmeaDataSource enNmeaDataSource = e8NMEA_SOURCE_GPS;

         //! Check and get GPGGA sentence if required.
         if (m_NmeaSentencesList.end()
                  != std::find(m_NmeaSentencesList.begin(), m_NmeaSentencesList.end(), e8NMEA_GPGGA))
         {
            szNmeaGGASentence = oNmeaEnc.szGetNmeaGGASentence(enNmeaDataSource);

         }
         //! Check and get GPRMC sentence if required.
         if (m_NmeaSentencesList.end()
                  != std::find(m_NmeaSentencesList.begin(), m_NmeaSentencesList.end(), e8NMEA_GPRMC))
         {
            szNmeaRMCSentence = oNmeaEnc.szGetNmeaRMCSentence(enNmeaDataSource);
         }

         if (TARGETADAPTER_REQUEST == (m_u8RequestedBy & TARGETADAPTER_REQUEST))
         {
            spi_tclMySPINManager* poMySPINMngr = spi_tclMySPINManager::getInstance();

            if (NULL != poMySPINMngr)
            {
               spi_tclMySPINCmdVehicleData * poCmdVehData = poMySPINMngr->poGetVehDataInstance();
               if (NULL != poCmdVehData)
               {

                  poCmdVehData->vSendPositionData(su32CurSelectedDevId, szNmeaGGASentence);
                  poCmdVehData->vSendPositionData(su32CurSelectedDevId, szNmeaRMCSentence);
               }
            }
         }
         if (MEDIAPLAYER_REQUEST == (m_u8RequestedBy & MEDIAPLAYER_REQUEST))
         {
            spi_tclExtCompManager *poExtCompMgr = spi_tclExtCompManager::getInstance();
            spi_tclExtCmdNavDataIntf *poExtCmdNavDataIntf = NULL;
            if(NULL != poExtCompMgr)
            {
                poExtCmdNavDataIntf = poExtCompMgr->poGetCmdNavDataIntfInst();
            }
            if(NULL != poExtCmdNavDataIntf)
            {
                t_String szGPGSVData, szGPHDTData;
                t_Bool bResult = poExtCmdNavDataIntf->bTransferGPSData(su32CurSelectedDevId, "", szNmeaGGASentence,
                                                                        szNmeaRMCSentence,szGPGSVData,szGPHDTData);
                ETG_TRACE_USR2(("GPS Data sent [%d]", ETG_ENUM(BOOL,bResult)));
            }
         }
      }
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclMySPINDataService::vSetSensorDataAvailablility(...)
***************************************************************************/
t_Void spi_tclMySPINDataService::vSetSensorDataAvailablility(
      const trDataServiceConfigData& rfrDataServiceConfigInfo)
{
   ETG_TRACE_USR1(("spi_tclMySPINDataService::vSetSensorDataAvailablility() entered "));

   //! Store config data
   m_rMySPINDataServiceConfigData = rfrDataServiceConfigInfo;
   m_enGeoCoordinateSystemType = m_rMySPINDataServiceConfigData.enGeoCoordinateSystemType;
}

/******************************************************************************************************
 ** FUNCTION:  t_Void spi_tclMySPINDataService::vPostGPRMCDataStatusValues(...)
 ******************************************************************************************************/
t_Void spi_tclMySPINDataService::vPostGPRMCDataStatusValues(t_U32 u32DeviceId,t_U8 u8GPRMCDataStatusValues)
{
   ETG_TRACE_USR1(("[DESC]::spi_tclMySPINDataService::vPostGPRMCDataStatusValues() Entered with Device ID=%d, GPRMC Data Status Values=%d",
            u32DeviceId, u8GPRMCDataStatusValues));

   /**********************************************************************************************************
    * MP_FI <bit name="GPRMCDataStatusValueA" value="0"/> i.e. Decimal Value - 1 : scou8GPRMCDataStatusValueA
    * MP_FI <bit name="GPRMCDataStatusValueV" value="1"/> i.e. Decimal Value - 2 : scou8GPRMCDataStatusValueV
    * MP_FI <bit name="GPRMCDataStatusValueX" value="2"/> i.e. Decimal Value - 4 : scou8GPRMCDataStatusValueX
    *********************************************************************************************************/
    //! This change is made in order to resolve issue 264064([Chery M31T IHU High] Can't get current vehicle 
    //! position with sougou map in cherylink APP). As per Location Information R9.pdf , we need to populate “X” for 
    //! data status field of GPRMC, which is specific for a special case of carplay and not applicable for other  
    //! technologies In general, GPRMC data status field has 2 values only: “A” – Valid and “V” – Not Valid. 
    //! In case of MySPIN, we have to populate data status field of GPRMC as "A" in NMEA sentence in order to provide
    //! valid navigation data and precise location. 

   if((m_rMySPINDataServiceConfigData.enGeoCoordinateSystemType == e8GEO_COORDINATE_SYSTEM_WGS) ||
       (m_rMySPINDataServiceConfigData.enGeoCoordinateSystemType == e8GEO_COORDINATE_SYSTEM_GCJ))
   {
      m_enGeoCoordinateSystemType = ((scou8GPRMCDataStatusValueA & u8GPRMCDataStatusValues) == scou8GPRMCDataStatusValueA) ?
                                e8GEO_COORDINATE_SYSTEM_WGS : e8GEO_COORDINATE_SYSTEM_UNKNOWN;
   }
   else
   {
      m_enGeoCoordinateSystemType = e8GEO_COORDINATE_SYSTEM_UNKNOWN;
   }
}
//lint -restore
///////////////////////////////////////////////////////////////////////////////
// <EOF>
