/*!
 *******************************************************************************
 * \file         spi_tclMySPINIAPConnection.cpp
 * \brief        mySPIN IAP Connection class
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    
 AUTHOR:         tch5kor
 COPYRIGHT:      &copy; 2015 Robert Bosch Car Multimedia GmbH
 HISTORY:
 Date        | Author                | Modification
 12.04.2016  | Chaitra Srinivasa     | Trace message clean up
 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "mySPINTypes.h"
#include "spi_tclExtCompManager.h"
#include "spi_tclExtCmdNativeTransport.h"
#include "spi_tclMySPINCmdSession.h"
#include "spi_tclMySPINManager.h"
#include "spi_tclMySPINIAPConnection.h"
#include "RespRegister.h"
#include "Timer.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_CONNECTIVITY
#include "trcGenProj/Header/spi_tclMySPINIAPConnection.cpp.trc.h"
#endif
#endif
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e515 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e516 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e746 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported

const static t_U32 INVALID_DEVICE_HANDLE = 0xFFFF;
static spi_tclMySPINCmdSession* sapoMySPINCmdSession = NULL;

static timer_t srNativeTransportStartTimerID;
static t_Bool sbNativeTransportStartTimerRunning = false;
static const t_U32 cou32NativeTransportStartTimerVal = 8000;
static t_Bool sbIsSelectionResultPending = false;

/***************************************************************************
 ** FUNCTION:  spi_tclMySPINIAPConnection::spi_tclMySPINIAPConnection
 ***************************************************************************/
spi_tclMySPINIAPConnection::spi_tclMySPINIAPConnection() :
   m_u32SelectedDevice(INVALID_DEVICE_HANDLE),
   m_bStartTransportAfterSwitch(false),
   m_enDeviceSelectionState(e8_SELECTION_STATE_DEVICE_NOT_SELECTED),
   m_enMySPINError(e8MSPIN_ERROR_UNKNOWN),
   m_u16VehicleType(0xffff)
{
   ETG_TRACE_USR1(("spi_tclMySPINIAPConnection::spi_tclMySPINIAPConnection"));
}

/***************************************************************************
 ** FUNCTION:  spi_tclMySPINIAPConnection::~spi_tclMySPINIAPConnection
 ***************************************************************************/
spi_tclMySPINIAPConnection::~spi_tclMySPINIAPConnection()
{
   ETG_TRACE_USR1(("spi_tclMySPINIAPConnection::~spi_tclMySPINIAPConnection"));
   sapoMySPINCmdSession = NULL;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMySPINIAPConnection::vRegisterCallbacks
 ***************************************************************************/
t_Void spi_tclMySPINIAPConnection::vRegisterCallbacks(const trMySPINConnCbs &corfrMySPINConnCbs)
{
   m_rMySPINConnCbs = corfrMySPINConnCbs;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMySPINIAPConnection::bInitializeConnection
 ***************************************************************************/
t_Bool spi_tclMySPINIAPConnection::bInitializeConnection()
{
   spi_tclMySPINManager* poMySPINMngr = spi_tclMySPINManager::getInstance();

   if (NULL != poMySPINMngr)
   {
      sapoMySPINCmdSession = poMySPINMngr->poGetSessionInstance();
   }//if(NULL != poMySPINMngr)

   RespRegister *pRespRegister = RespRegister::getInstance();
   if(NULL!= pRespRegister)
   {
      pRespRegister->bRegisterObject((spi_tclExtRespNativeTransport*)this);
   }
   return true;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMySPINIAPConnection::vUnInitializeConnection
 ***************************************************************************/
t_Void spi_tclMySPINIAPConnection::vUnInitializeConnection()
{
}

/***************************************************************************
 ** FUNCTION:  spi_tclMySPINIAPConnection::vOnLoadSettings
 ***************************************************************************/
t_Void spi_tclMySPINIAPConnection::vOnLoadSettings(const trHeadUnitInfo &corfrheadUnitInfo,
                                                   const trVehicleInfo& corfrVehicleInfo,
                                                   t_U16 u16VehicleType)
{
   SPI_INTENTIONALLY_UNUSED(corfrheadUnitInfo);
   m_szVehicleManufacturerName = corfrVehicleInfo.szManufacturer;
   m_szVehicleManufacturerScope = corfrVehicleInfo.szVehicleVariant;
   m_u16VehicleType = u16VehicleType;
   
   tfvGetSelectedDevSerialCallback fvGetSelectedDevSerialCallback;
   fvGetSelectedDevSerialCallback = std::bind(&spi_tclMySPINIAPConnection::vGetSelectedDeviceSerial,
                                              this,
                                              SPI_FUNC_PLACEHOLDERS_2);
   spi_tclExtCompManager *poExtCompMgr = spi_tclExtCompManager::getInstance();
   spi_tclExtCmdNativeTransportIntf *poExtCmdNativeTransportIntf = NULL;
   if(NULL != poExtCompMgr)
   {
       poExtCmdNativeTransportIntf = poExtCompMgr->poGetCmdNativeTransportIntfInst();
   }
   if(NULL != poExtCmdNativeTransportIntf)
   {
       poExtCmdNativeTransportIntf->vRegisterGetSelectedDevSerialCb(fvGetSelectedDevSerialCallback);
   }   
   
   // TODO add settings if required, move to policy
   m_rAccInfo.szReadDevice = "/dev/ffs/ep4";
   m_rAccInfo.szWriteDevice = "/dev/ffs/ep3";
   trMySPINIapAppInfo rMySPINIapAppInfo;
   if (NULL != m_poConnSettings)
   {
       m_poConnSettings->vGetMySPINIapAppInfo(rMySPINIapAppInfo);
       m_rAccInfo.bAudioSupport = rMySPINIapAppInfo.bAudioSupport;
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclMySPINIAPConnection::vOnSelectDevice
 ***************************************************************************/
t_Void spi_tclMySPINIAPConnection::vOnSelectDevice(const t_U32 cou32DevID, tenDeviceConnectionType enDevConnType,
         tenDeviceConnectionReq enDevSelectType, const trUserContext corUsrCntxt,
         const trMySPINDevInfo& corfrMySPINDevInfo)
{
   SPI_INTENTIONALLY_UNUSED(enDevConnType);
   SPI_INTENTIONALLY_UNUSED(corUsrCntxt);

   ETG_TRACE_USR1(("spi_tclMySPINIAPConnection::vOnSelectDevice()"));

   tenErrorCode enErrorCode = e8UNKNOWN_ERROR;
   if (NULL != sapoMySPINCmdSession)
   {
      if (enDevSelectType == e8DEVCONNREQ_SELECT)
      {
         t_Bool bAudioSupport = m_rAccInfo.bAudioSupport;
         // Since role switch Succeeded with no errors, initiate mySPIN session
         enErrorCode = sapoMySPINCmdSession->enInitMySPINSession(cou32DevID,
                  e8DEVTYPE_IAP,
                  corfrMySPINDevInfo,
                  m_rAccInfo,
                  bAudioSupport);

         if(e8NO_ERRORS == enErrorCode)
         {
            ETG_TRACE_USR2(("[DESC]::Save the selected device Sub Category [%d]", ETG_ENUM(MSPIN_DEV_TYPE,
                     e8DEVTYPE_IAP)));
            spi_tclMySPINManager::vSetDeviceSubCategory(e8DEVTYPE_IAP);
            m_mapDeviceInfo[cou32DevID]=corfrMySPINDevInfo;
            vSetSelectedDevice(cou32DevID);
            ETG_TRACE_USR2(("m_bStartTransportAfterSwitch = %d",m_bStartTransportAfterSwitch));
            
            sbIsSelectionResultPending = true;
            
            if (m_bStartTransportAfterSwitch)
            {
               vOnNativeTransportStart(u32GetSelectedDevice());
               m_bStartTransportAfterSwitch = false;
            }
            else
            {
                spi_tclExtCompManager *poExtCompMgr = spi_tclExtCompManager::getInstance();
                spi_tclExtCmdNativeTransportIntf *poExtCmdNativeTransportIntf = NULL;
                if(NULL != poExtCompMgr)
                {
                     poExtCmdNativeTransportIntf = poExtCompMgr->poGetCmdNativeTransportIntfInst();
                }
                if((NULL != poExtCmdNativeTransportIntf) && (NULL != m_poConnSettings))
                {
                    trEAPAppInfo rEAPAppInfo;
                    trMySPINIapAppInfo rMySPINIapAppInfo;
                    m_poConnSettings->vGetMySPINIapAppInfo(rMySPINIapAppInfo);
                    rEAPAppInfo.bLaunchType = rMySPINIapAppInfo.enIapLaunchType;
                    rEAPAppInfo.szAppName = rMySPINIapAppInfo.szAppName;
                    rEAPAppInfo.szBundleID = rMySPINIapAppInfo.szBundleSeedId;
                    rEAPAppInfo.szProtocol = rMySPINIapAppInfo.szProtocol;
                    t_Bool bLaunchApp = poExtCmdNativeTransportIntf->bLaunchApp(cou32DevID,e8DEV_TYPE_MYSPIN,rEAPAppInfo);
                    vStartTimer();
                }
            }
         }
      }
      else
      {
         m_mapDeviceInfo.erase(cou32DevID);
         vOnNativeTransportStop(u32GetSelectedDevice());
         vSetSelectedDevice(INVALID_DEVICE_HANDLE);
         m_enMySPINError = e8MSPIN_ERROR_UNKNOWN;
         
         enErrorCode = sapoMySPINCmdSession->enUninitMySPINSession(cou32DevID, e8DEVTYPE_IAP);
         spi_tclMySPINManager::vSetDeviceSubCategory(e8DEVTYPE_UNKNWON);

         if (NULL != m_rMySPINConnCbs.fvSelectDeviceResult)
         {
            (m_rMySPINConnCbs.fvSelectDeviceResult)(cou32DevID, e8DEVTYPE_IAP, enErrorCode);
         }
         
         m_bStartTransportAfterSwitch = false;
      }
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclMySPINIAPConnection::vOnSelectDeviceResult()
 ***************************************************************************/
t_Void spi_tclMySPINIAPConnection::vOnSelectDeviceResult(const t_U32 cou32DevID,
         const tenDeviceConnectionReq coenConnReq, const tenResponseCode coenRespCode)
{
   SPI_INTENTIONALLY_UNUSED(cou32DevID)

   if ((NULL != sapoMySPINCmdSession) && (e8DEVCONNREQ_SELECT == coenConnReq) && (e8FAILURE == coenRespCode))
   {
      vStopTimer();
      m_enMySPINError = e8MSPIN_ERROR_UNKNOWN;
      ETG_TRACE_USR1(("spi_tclMySPINIAPConnection::vOnSelectDeviceResult:Failure on Select Device- Perform Clean up"));
      spi_tclMySPINManager::vSetDeviceSubCategory(e8DEVTYPE_UNKNWON);
      vSetSelectedDevice(INVALID_DEVICE_HANDLE);
   }
}
/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINIAPConnection::vOnNativeTransportStart()
 ***************************************************************************/
t_Void spi_tclMySPINIAPConnection::vOnNativeTransportStart(const t_U32 cou32DevID)
{
   ETG_TRACE_USR1(("spi_tclMySPINIAPConnection::vOnNativeTransportStart from MediaPlayer entered cou32DevID = %d",cou32DevID));

   vStopTimer();

   if ((NULL != sapoMySPINCmdSession) && (cou32DevID == u32GetSelectedDevice()))
   {
         tenErrorCode enError = sapoMySPINCmdSession->enSessionStart(cou32DevID, e8DEVTYPE_IAP);
         ETG_TRACE_USR4(("[PARAM]::vOnNativeTransportStart-EAP Transport Start Status : [%d]", ETG_ENUM(ERROR_CODE,
                  enError)));
   }
   else if((0 != cou32DevID) && ((m_enDeviceSelectionState == e8_SELECTION_STATE_DEVICE_SELECTION_INPROGRESS)))
   {
        // Since selection is in progress, if native transport has started,
        // start native trsnsport immediately after switch is complete.
        m_bStartTransportAfterSwitch = true;
   }
   
   if ((e8MSPIN_ERROR_PROTOCOL_STOP == m_enMySPINError) || (e8MSPIN_ERROR_INITIALIZATION_ABORT == m_enMySPINError))
   {
      spi_tclMySPINManager* poMySPINMngr = spi_tclMySPINManager::getInstance();
      if(NULL != poMySPINMngr)
      {
          spi_tclMySPINCmdVehicleData * poCmdVehData = poMySPINMngr->poGetVehDataInstance();
          if (NULL != poCmdVehData )
          {
              poCmdVehData->vSetVehicleManufactureInfo(cou32DevID, m_u16VehicleType, m_szVehicleManufacturerName, m_szVehicleManufacturerScope);
          }
      }
   }    
   
   if ((true == sbIsSelectionResultPending) && (NULL != m_rMySPINConnCbs.fvSelectDeviceResult))
   {
      sbIsSelectionResultPending = false;
      (m_rMySPINConnCbs.fvSelectDeviceResult)(cou32DevID, e8DEVTYPE_IAP, e8NO_ERRORS);
   }   
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINIAPConnection::vOnNativeTransportStop()
 ***************************************************************************/
t_Void spi_tclMySPINIAPConnection::vOnNativeTransportStop(const t_U32 cou32DevID)
{
   ETG_TRACE_USR1(("spi_tclMySPINIAPConnection::vOnNativeTransportStop from MediaPlayer"));

   if ((NULL != sapoMySPINCmdSession) && (cou32DevID == u32GetSelectedDevice()))
   {
      m_oNativeTransportStop.s16Lock();
      sapoMySPINCmdSession->vSessionStop(cou32DevID, e8DEVTYPE_IAP);
      m_oNativeTransportStop.vUnlock();
      ETG_TRACE_USR1(("spi_tclMySPINIAPConnection::vOnNativeTransportStop:EAP Transport Stopped"));
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINIAPConnection::vSetSelectedDevice(
 ***************************************************************************/
t_Void spi_tclMySPINIAPConnection::vSetSelectedDevice(const t_U32 cou32DevID)
{
   m_u32SelectedDevice = cou32DevID;
   ETG_TRACE_USR1(("spi_tclMySPINIAPConnection::vSetSelectedDevice m_u32SelectedDevice = %d",m_u32SelectedDevice));
}

/***************************************************************************
 ** FUNCTION: t_U32 spi_tclMySPINIAPConnection::u32GetSelectedDevice()
 ***************************************************************************/
t_U32 spi_tclMySPINIAPConnection::u32GetSelectedDevice()
{
   ETG_TRACE_USR1(("spi_tclMySPINIAPConnection::u32GetSelectedDevice m_u32SelectedDevice = %d",m_u32SelectedDevice));
   return m_u32SelectedDevice;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclMySPINIAPConnection::vGetSelectedDeviceSerial()
 ***************************************************************************/
t_Void spi_tclMySPINIAPConnection::vGetSelectedDeviceSerial(const t_U32 cou32DevID, t_String& szSerial)
{
   if (m_mapDeviceInfo.end() != m_mapDeviceInfo.find(cou32DevID))
   {
      size_t posUnderscore = m_mapDeviceInfo[cou32DevID].szSerial.find_last_of("_");
      szSerial = m_mapDeviceInfo[cou32DevID].szSerial.substr(posUnderscore + 1);
      ETG_TRACE_USR2(("[DESC]:Serial for device [%d]-[%s]", cou32DevID, szSerial.c_str()));
   }
}

/***************************************************************************
 ** FUNCTION: spi_tclMySPINIAPConnection::vPostNativeTransportStartResult()
 ***************************************************************************/
t_Void spi_tclMySPINIAPConnection::vPostNativeTransportStartResult(t_U32 u32DeviceId)
{
   vOnNativeTransportStart(u32DeviceId);
}
/***************************************************************************
 ** FUNCTION: spi_tclMySPINIAPConnection::vPostNativeTransportStopResult()
 ***************************************************************************/
t_Void spi_tclMySPINIAPConnection::vPostNativeTransportStopResult(t_U32 u32DeviceId)
{
   vOnNativeTransportStop(u32DeviceId);
}

//!Static
/***************************************************************************
** FUNCTION:  t_Bool spi_tclMySPINIAPConnection::bFirstFBRenderTimerCb
***************************************************************************/
t_Bool spi_tclMySPINIAPConnection::bNativeTransportStartTimerCb(timer_t rTimerID, t_Void *pvObject,
                                                 const t_Void *pvUserData)
{
   ETG_TRACE_USR4(("spi_tclMySPINIAPConnection:bNativeTransportStartTimerCb()"));

   SPI_INTENTIONALLY_UNUSED(pvUserData);
   SPI_INTENTIONALLY_UNUSED(rTimerID);

   spi_tclMySPINIAPConnection* poMySPINIAPConn = static_cast<spi_tclMySPINIAPConnection*> (pvObject);

   poMySPINIAPConn->vStopTimer();
   
   if (NULL != poMySPINIAPConn->m_rMySPINConnCbs.fvSelectDeviceResult)
   {
      sbIsSelectionResultPending = false;
      (poMySPINIAPConn->m_rMySPINConnCbs.fvSelectDeviceResult)(poMySPINIAPConn->m_u32SelectedDevice, e8DEVTYPE_IAP, e8FATAL_ERROR);
   }
   
   return true;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclMySPINIAPConnection::vStartTimer()
***************************************************************************/
t_Void spi_tclMySPINIAPConnection::vStartTimer()
{
   m_oVideoSetupLock.s16Lock();
   Timer* poTimer = Timer::getInstance();

   if((NULL != poTimer))
   {
      poTimer->StartTimer(srNativeTransportStartTimerID,
         cou32NativeTransportStartTimerVal, 0, this,
         &spi_tclMySPINIAPConnection::bNativeTransportStartTimerCb, NULL);

      ETG_TRACE_USR4(("[DESC]:First frame render callback Timer started"));

      sbNativeTransportStartTimerRunning = true;
   }//End of if(NULL != poTimer)
   m_oVideoSetupLock.vUnlock();
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclMySPINIAPConnection::vStopTimer()
***************************************************************************/
t_Void spi_tclMySPINIAPConnection::vStopTimer()
{
   m_oVideoSetupLock.s16Lock();
   Timer* poTimer = Timer::getInstance();

   if ((NULL != poTimer) && (true == sbNativeTransportStartTimerRunning))
   {
      poTimer->CancelTimer(srNativeTransportStartTimerID);
      ETG_TRACE_USR4(("[DESC]:First frame render callback Timer stopped"));

      sbNativeTransportStartTimerRunning = false;
      srNativeTransportStartTimerID = 0;
   }//End of if (NULL != poTimer)...
   m_oVideoSetupLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  spi_tclMySPINIAPConnection::vOnSetSelectionProgressState(
 ***************************************************************************/
t_Void spi_tclMySPINIAPConnection::vOnSetSelectionProgressState(tenDeviceSelectionState enDeviceSelectionState)
{
   ETG_TRACE_USR2(("spi_tclMySPINIAPConnection::vOnSetSelectionProgressState entered with result = %d", ETG_ENUM(tenDeviceSelectionState, enDeviceSelectionState)));
   m_enDeviceSelectionState = enDeviceSelectionState;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclMySPINIAPConnection::vMySPINSessionErrorCb()
***************************************************************************/
t_Void spi_tclMySPINIAPConnection::vMySPINSessionErrorCb(tenMySPINError enErrorCode)
{
   if ((e8MSPIN_ERROR_PROTOCOL_STOP == enErrorCode) || (e8MSPIN_ERROR_INITIALIZATION_ABORT == enErrorCode))
   {
       m_enMySPINError = enErrorCode;
   }       
}
//lint restore
///////////////////////////////////////////////////////////////////////////////
// <EOF>
