/*!
 *******************************************************************************
 * \file         spi_tclMySPINBluetooth.h
 * \brief        MySpin Bluetooth class
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    MySpin Bluetooth handler class for SPI
 AUTHOR:         tch5kor
 COPYRIGHT:      &copy; 2015 Robert Bosch Car Multimedia GmbH

 HISTORY:
 Date        | Author                | Modification
 02.11.2015  | tch5kor               | Initial Version
 05.04.2016  | Chaitra Srinivasa     | Adding comments
 
 \endverbatim
 ******************************************************************************/

#ifndef _SPI_TCLMYSPINBLUETOOTH_H
#define _SPI_TCLMYSPINBLUETOOTH_H

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "SPITypes.h"
#include "spi_BluetoothTypedefs.h"
#include "spi_tclBluetoothDevBase.h"
#include "spi_tclMySPINRespSession.h"
#include "spi_tclExtRespNativeTransport.h"
#include "spi_tclExtRespBluetooth.h"

class spi_tclExtCmdBluetoothIntf;

/******************************************************************************
 | defines and macros and constants(scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: global)
 |----------------------------------------------------------------------------*/
static t_U32 su32CurSelectedDevId = 0;
/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/

/* Forward Declarations. */
class spi_tclBluetoothIntf;

/*!
 * \class spi_tclMySPINBluetooth
 * \brief This is the MySpin Bluetooth class that handles the Bluetooth
 *        connection logic during a MySpin device session
 */
class spi_tclMySPINBluetooth: public spi_tclMySPINRespSession, public spi_tclBluetoothDevBase,public spi_tclExtRespNativeTransport,
                              public spi_tclExtRespBluetooth
{
   public:
      /***************************************************************************
       *********************************PUBLIC*************************************
       ***************************************************************************/

      /***************************************************************************
       ** FUNCTION:  spi_tclMySPINBluetooth::spi_tclMySPINBluetooth(...)
       ***************************************************************************/
      /*!
       * \fn      spi_tclMySPINBluetooth(spi_tclBluetoothIntf* poBTInterface)
       * \brief   Parameterized Constructor
       * \param   [IN] poBTInterface: Pointer to Bluetooth manager interface
       ***************************************************************************/
      spi_tclMySPINBluetooth(spi_tclBluetoothIntf* poBTInterface);

      /***************************************************************************
       ** FUNCTION:  spi_tclMySPINBluetooth::~spi_tclMySPINBluetooth();
       ***************************************************************************/
      /*!
       * \fn      ~spi_tclMySPINBluetooth()
       * \brief   Virtual Destructor
       ***************************************************************************/
      virtual ~spi_tclMySPINBluetooth();

      /*********Start of functions overridden from spi_tclBluetoothDevBase*******/

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclMySPINBluetooth::bInitialise();
       ***************************************************************************/
      /*!
       * \fn      bInitialise()
       * \brief   Method to initialise the service handler. (Performs initialisations which
       *          are not device specific.)
       *          Optional interface to be implemented.
       * \retval  t_Bool: TRUE - If ServiceHandler is initialised successfully, else FALSE.
       * \sa      bUninitialise()
       ***************************************************************************/
      t_Bool bInitialise();

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMySPINBluetooth::vOnSPISelectDeviceRequest(
      ***************************************************************************/
     /*!
      * \fn      vOnSPISelectDeviceRequest(const trSelectDeviceRequest& corfrSelectReq)
 
      * \brief   Called when SelectDevice request is received by SPI.
      *          Mandatory interface to be implemented.
      * \param   [IN] corfrSelectReq: Device Selection Info
      * \retval  None
      **************************************************************************/
      t_Void vOnSPISelectDeviceRequest(const trSelectDeviceRequest& corfrSelectReq);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMySPINBluetooth::vOnSPISelectDeviceResponse(
      ***************************************************************************/
      /*!
       * \fn      vOnSPISelectDeviceResponse(const trSelectDeviceRequest& corfrSelectReq,
       *             tenResponseCode enRespCode)
       * \brief   Called when SelectDevice operation is complete, with the result
       *          of the operation. 
       *          Mandatory interface to be implemented.
       * \param   [IN] corfrSelectReq: Device Selection Info
       * \param   [IN] enRespCode: Response code enumeration
       * \retval  None
       **************************************************************************/
      t_Void vOnSPISelectDeviceResponse(const trSelectDeviceRequest& corfrSelectReq, tenResponseCode enRespCode);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMySPINBluetooth::vOnSPIDeselectDeviceRequest(
       ***************************************************************************/
      /*!
       * \fn      vOnSPIDeselectDeviceRequest(const trSelectDeviceRequest& corfrSelectReq)
       * \brief   Called when DeselectDevice request is received by SPI.
       *          Mandatory interface to be implemented.
       * \param   [IN] corfrSelectReq: Device Selection Info
       * \retval  None
       **************************************************************************/
      t_Void vOnSPIDeselectDeviceRequest(const trSelectDeviceRequest& corfrSelectReq);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMySPINBluetooth::vOnSPIDeselectDeviceResponse(
      ***************************************************************************/
      /*!
       * \fn      vOnSPIDeselectDeviceResponse(const trSelectDeviceRequest& corfrSelectReq,
       *             tenResponseCode enRespCode)
       * \brief   Called when DeselectDevice operation is complete, with the result
       *          of the operation.
       *          Mandatory interface to be implemented.
       * \param   [IN] corfrSelectReq: Device Selection Info
       * \param   [IN] enRespCode: Response code enumeration
       * \retval  None
       **************************************************************************/
       t_Void vOnSPIDeselectDeviceResponse(const trSelectDeviceRequest& corfrSelectReq,
         tenResponseCode enRespCode);

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclMySPINBluetooth::vBTProfileInfoCb()
       ***************************************************************************/
      /*!
       * \fn      vBTProfileInfoCb
       * \brief   Informs the current state of App Transition
       * \param   bIsHFP :
       * \param   bIsA2DP :
       * \param   szFriendlyName:
       * \retval  None
       **************************************************************************/
      t_Void vBTProfileInfoCb(const t_U32 cou32DeviceHandle, t_Bool bIsHFP, t_Bool bIsA2DP, t_String szFriendlyName="");

      /*********End of functions overridden from spi_tclBluetoothDevBase*********/

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMySPINBluetooth::vOnBTConnectionChanged()
       ***************************************************************************/
      /*!
       * \fn      vOnBTConnectionChanged
       * \brief   Inform on BT connection change
       * \retval  t_Void
       **************************************************************************/
      t_Void vOnBTConnectionChanged(t_String, tenBTConnectionResult)
      {
      }

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMySPINBluetooth::vFetchBTDetails()
       ***************************************************************************/
      /*!
       * \fn      vFetchBTDetails(t_U32 &rfu32ConnBTDevHandle, t_String& rfszMacAddress)
       * \brief   Fetch BT details(BT handle and mac address) using BT device name
       * \param   rfu32ConnBTDevHandle : Bt handle
       * \param   rfszMacAddress : Bt Mac address
       * \retval  t_Void
       **************************************************************************/
      t_Void vFetchBTDetails(t_U32 &rfu32ConnBTDevHandle, t_String& rfszMacAddress);
      
      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMySPINBluetooth::vUpdateCmdAudio()
       ***************************************************************************/
      /*!
       * \fn      vUpdateCmdAudio(t_U32 u32ConnBTDevHandle,t_String szMacAddress)
       * \brief   Update A2DP status to cmd audio class
       * \param   u32ConnBTDevHandle : Bt handle
       * \param   szMacAddress : Bt Mac address
       * \retval  t_Void
       **************************************************************************/
      t_Void vUpdateCmdAudio(t_U32 u32ConnBTDevHandle,t_String szMacAddress);
      
      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMySPINBluetooth::vPostBTProfileStatusUpdateMsg()
       ***************************************************************************/
      /*!
       * \fn      vPostBTProfileStatusUpdateMsg()
       * \brief   Update from BT for HFP and A2DP connection status
       * \retval  t_Void
       **************************************************************************/
      t_Void vPostBTProfileStatusUpdateMsg();
      
      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMySPINBluetooth::vSendDeviceSwitchEvent()
       ***************************************************************************/
      /*!
       * \fn      vSendDeviceSwitchEvent(t_U32 u32ConnBTDevHandle,tenDeviceSubCategory enDeviceSubCategory)
       * \brief   Send device switch event update to HMI
       * \param   u32ConnBTDevHandle : Bt handle       
       * \param   enDeviceSubCategory : Device category    
       * \retval  t_Void
       **************************************************************************/
      t_Void vSendDeviceSwitchEvent(t_U32 u32ConnBTDevHandle,tenDeviceSubCategory enDeviceSubCategory);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMySPINBluetooth::vSendVoiceSessionStatus()
       ***************************************************************************/
      /*!
       * \fn      vSendVoiceSessionStatus()
       * \brief   Send voice session status
       * \retval  t_Void
       **************************************************************************/	  
       t_Void vSendVoiceSessionStatus();
      
      /**************************************************************************
       ****************************END OF PUBLIC**********************************
       **************************************************************************/
      
   protected:

      /***************************************************************************
       *********************************PROTECTED**********************************
       ***************************************************************************/
      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMySPINBluetooth::vPostBTProfileInfo()
       ***************************************************************************/
      /*!
       * \fn      t_Void vPostBTProfileInfo()
       * \brief   Informs the current state of App Transition
       * \param   bIsHFP : HFP connection status
       * \param   bIsA2DP : A2DP connection status
       * \param   szFriendlyName: BT friendly name
       * \sa      None
       **************************************************************************/
       t_Void vPostBTProfileInfo(const t_U32 cou32DeviceHandle, t_Bool bIsHFP, t_Bool bIsA2DP, t_String szFriendlyName);
       
      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMySPINBluetooth::vPostNativeTransportStartResult()
       ***************************************************************************/
      /*!
       * \fn      t_Void vPostNativeTransportStartResult()
       * \brief   Called by MPlay client handler to inform native transport has started
       * \param   [IN] u32DeviceId: Device ID
       * \sa      None
       **************************************************************************/
      t_Void vPostNativeTransportStartResult(t_U32 u32DeviceId);

      /***************************************************************************
       ******************************END OF PROTECTED******************************
       ***************************************************************************/

   private:

      /***************************************************************************
       *********************************PRIVATE************************************
       ***************************************************************************/

      /***************************************************************************
       ** FUNCTION: spi_tclMySPINBluetooth(const spi_tclMySPINBluetooth &rfcoBluetooth)
       ***************************************************************************/
      /*!
       * \fn      spi_tclMySPINBluetooth(const spi_tclMySPINBluetooth &rfcoBluetooth)
       * \brief   Copy constructor not implemented hence made private
       **************************************************************************/
      spi_tclMySPINBluetooth(const spi_tclMySPINBluetooth &rfcoBluetooth);

      /***************************************************************************
       ** FUNCTION: const spi_tclMySPINBluetooth & operator=(
       **                                 const spi_tclMySPINBluetooth &rfcoBluetooth);
       ***************************************************************************/
      /*!
       * \fn      const spi_tclMySPINBluetooth & operator=(const spi_tclMySPINBluetooth &rfcoBluetooth);
       * \brief   assignment operator not implemented hence made private
       **************************************************************************/
      const spi_tclMySPINBluetooth & operator=(const spi_tclMySPINBluetooth &rfcoBluetooth);
      
      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclMySPINConnMngr::vPostBTServiceAvailabilityMsg()
       ***************************************************************************/
      /*!
       * \fn      t_Void vPostBTConnectionChangedMsg()
       * \brief   Called when BT On Service Availability is  sent by BT Client Handler.
       * \sa      None
       **************************************************************************/
       virtual t_Void vPostBTServiceAvailabilityMsg(t_Bool bServiceAvailable);

      /***************************************************************************
       ** FUNCTION: spi_tclMySPINBluetooth::vMySPINSessionErrorCb()
       ***************************************************************************/
      /*!
       * \fn      vMySPINSessionErrorCb
       * \brief   Informs the mySPIN Error from Phone
       * \param   enErrorCode : Voice Session Request
       * \sa
       **************************************************************************/
      virtual t_Void vMySPINSessionErrorCb(const t_U32 cou32DeviceHandle,tenMySPINError enErrorCode);

      /***************************************************************************
       ** BT Manager interface pointer
       ***************************************************************************/
      spi_tclBluetoothIntf* const m_cpoBTInterface;

      std::map<t_U32,trBTDeviceInfo> m_rBTDeviceInfo;

      t_U8 m_u8CurrentProfiles;
      
      t_String m_szFriendlyName;
      
      t_Bool m_bIsHFPConn;
      
      t_Bool m_bIsA2DPConn; 
      
      tenHFPStatus m_enHFPStatus;
      
      tenA2DPStatus m_enA2DPStatus;

      //! Action of pending SetBTLimitationMode request
      tenBTLimitationAction m_enPendingBTLimAction;

      tenMySPINError m_enMySPINError;

      /***************************************************************************
       ****************************END OF PRIVATE**********************************
       ***************************************************************************/

};

#endif // _SPI_TCLMYSPINBLUETOOTH_H
