/***********************************************************************/
/*!
* \file  spi_tclOnCarVideo.cpp
* \brief  Implementation of the Class spi_tclOnCarVideo
*************************************************************************
\verbatim

PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    class for OnCar Video
AUTHOR:         Dhiraj Asopa
COPYRIGHT:      &copy; 2015 Robert Bosch Car Multimedia GmbH
HISTORY:
Date        | Author                 | Modification
02.11.2018  | Dhiraj Asopa           | Initial Version

\endverbatim
*************************************************************************/

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/

#include "spi_tclOnCarVideo.h"
#include "spi_tclOnCarManager.h"
#include "spi_tclVideoTypedefs.h"
#include "StringHandler.h"
#include "OnCarVideoEndpoint.h"

#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_VIDEO
#include "trcGenProj/Header/spi_tclOnCarVideo.cpp.trc.h"
#endif
#endif
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | defines and macros (scope: global)
 |----------------------------------------------------------------------------*/
//wl_pump_events method should be called fro every 25 msec
//#define TIMER_PUMP_WL_EVENTS_MSEC (tCU32)40

/******************************************************************************
 | variable definition (scope: global)
 |----------------------------------------------------------------------------*/
//Member variable to maintain Timer Index of Wayland touch events timer
//static timer_t sWLTimerIndex = 0;

/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/
static const t_U16 scou16Fps = 30;

/***************************************************************************
 ** FUNCTION:  spi_tclOnCarVideo::spi_tclOnCarVideo()
 ***************************************************************************/
spi_tclOnCarVideo::spi_tclOnCarVideo()
: m_poCmdVideo(NULL),m_u32SelectedDeviceID(0)
{
   ETG_TRACE_USR1(("spi_tclOnCarVideo() entered"));
}

/***************************************************************************
 ** FUNCTION:  spi_tclOnCarVideo::~spi_tclOnCarVideo()
 ***************************************************************************/
spi_tclOnCarVideo::~spi_tclOnCarVideo()
{
   ETG_TRACE_USR1(("~spi_tclOnCarVideo() entered"));
   m_u32SelectedDeviceID = 0;
   m_poCmdVideo = NULL;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclOnCarVideo::bInitialize()
 ***************************************************************************/
t_Bool spi_tclOnCarVideo::bInitialize()
{
   ETG_TRACE_USR1(("spi_tclOnCarVideo::bInitialize() entered"));
   t_Bool bInit = true;
   spi_tclOnCarManager *poOnCarManager = spi_tclOnCarManager::getInstance();
   SPI_NORMAL_ASSERT(NULL == poOnCarManager);
   if (NULL != poOnCarManager)
   {
	   m_poCmdVideo = poOnCarManager->poGetVideoInstance();
       //bInit = poOnCarManager->bRegisterObject((spi_tclOnCarRespVideo*) this);
   }
    return bInit;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarVideo::vUninitialize()
 ***************************************************************************/
t_Void spi_tclOnCarVideo::vUninitialize()
{
   ETG_TRACE_USR1(("spi_tclOnCarVideo::vUninitialize() entered"));
   m_u32SelectedDeviceID = 0;
   m_poCmdVideo = NULL;
}

/***************************************************************************
 ** FUNCTION:  t_Void  spi_tclOnCarVideo::vRegisterCallbacks()
 ***************************************************************************/
t_Void spi_tclOnCarVideo::vRegisterCallbacks(const trVideoCallbacks& corfrVideoCallbacks)
{
   ETG_TRACE_USR1(("spi_tclOnCarVideo::vRegisterCallbacks() entered"));
   m_rVideoCallbacks = corfrVideoCallbacks;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarVideo::vSelectDevice()
 ***************************************************************************/
t_Void spi_tclOnCarVideo::vSelectDevice(const t_U32 cou32DevId, const tenDeviceConnectionReq coenConnReq)
{
   ETG_TRACE_USR1(("spi_tclOnCarVideo:vSelectDevice: Device ID-0x%x Selection Type - %d ", cou32DevId, ETG_ENUM(CONNECTION_REQ,
            coenConnReq)));

   tenErrorCode enErrCode = e8NO_ERRORS;
   if (e8DEVCONNREQ_SELECT == coenConnReq)
   {
      //Initialize Video session
      enErrCode = (bInitVideoSession(cou32DevId)) ? e8NO_ERRORS : e8UNKNOWN_ERROR;
   }//if( coenConnReq == e8DEVCONNREQ_SELECT )
   else
   {
      vUnInitVideoSession(cou32DevId);
   }//else

   if (NULL != m_rVideoCallbacks.fpvSelectDeviceCb)
   {
      (m_rVideoCallbacks.fpvSelectDeviceCb)(cou32DevId, enErrCode);
   }//if(NULL != m_rVideoCallbacks.fpvSelectDeviceCb)
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclOnCarVideo::bInitVideoSession()
***************************************************************************/
t_Bool spi_tclOnCarVideo::bInitVideoSession(t_U32 u32DevID)
{
   ETG_TRACE_USR1(("spi_tclOnCarVideo::bInitVideoSession:Device ID-0x%x", u32DevID));

   t_Bool bRet = false;

   if( (m_u32SelectedDeviceID != u32DevID) && (0 != u32DevID) && (NULL != m_poCmdVideo) )
   {
      trOnCarVideoConfig rOnCarVideoConfig;

      if( (NULL != m_poCmdVideo) && (NULL != m_poVideoSettings) )
      {
         tvecVideoConfigList vecVideoConfigList;
         trVideoConfigData rVideoConfigData;
         m_poVideoSettings->vGetVideoConfigData(e8DEV_TYPE_ONCAR,vecVideoConfigList);
         m_poVideoSettings->vGetPrimaryDisplayConfiguration(vecVideoConfigList,rVideoConfigData);
         rOnCarVideoConfig.u32ProjectionScreenHeight = rVideoConfigData.u32ProjScreen_Height;
         rOnCarVideoConfig.u16Density = rVideoConfigData.u16dpi;
         rOnCarVideoConfig.u16Fps = scou16Fps;
         rOnCarVideoConfig.u32ProjectionScreenWidth= rVideoConfigData.u32ProjScreen_Width;
         StringHandler oStringUtil;
         rOnCarVideoConfig.u16PixelAspectRatio = oStringUtil.u32ConvertStrToInt(rVideoConfigData.szPixelAspectRatio);
         t_Char szVideoPipeline[MAX_KEYSIZE] = { 0 };
         snprintf(szVideoPipeline,
                  MAX_KEYSIZE,
                  "vpudec low-latency=true frame-plus=2 framedrop=false framerate-nu=30 dis-reorder=true ! "
                  "gst_apx_sink display-width=%d display-height=%d sync=false qos=false "
                  "max-lateness=3000000000 layer-id=%d surface-id=%d force-aspect-ratio=true",
                  rVideoConfigData.u32ProjScreen_Width,
                  rVideoConfigData.u32ProjScreen_Height,
                  rVideoConfigData.u32LayerId,
                  rVideoConfigData.u32SurfaceId);
         rOnCarVideoConfig.szVideoPipeline = szVideoPipeline;

         bRet = m_poCmdVideo->bInitialize(rOnCarVideoConfig);
         m_u32SelectedDeviceID = u32DevID;

      }//if((NULL != m_poVideoSettings) && (NULL != spoCmdVideo) )

   }//if( (m_u32SelectedDeviceID != u32DevID) && (0 != u32DevID) )
   else
   {
      //return true if the currently selected device is same as already selected device.
      //return an error, if the device ID is zero(invalid device).
      bRet = ((0 != u32DevID)&&(m_u32SelectedDeviceID == u32DevID));
   }

   ETG_TRACE_USR4(("spi_tclOnCarVideo::bInitVideoSession left with %d", ETG_ENUM(
      BOOL, bRet)));

   return bRet;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarVideo::vUnInitVideoSession()
***************************************************************************/
t_Void spi_tclOnCarVideo::vUnInitVideoSession(t_U32 u32DevID)
{
   ETG_TRACE_USR1(("spi_tclOnCarVideo::vUnInitVideoSession:Device ID-0x%x", u32DevID));
   if ( (0 != m_u32SelectedDeviceID) && (NULL != m_poCmdVideo) )
   {
      m_poCmdVideo->vUninitialize();
      m_u32SelectedDeviceID = 0;
   }//if (NULL != spoCmdVideo)
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarVideo::bLaunchVideo()
 ***************************************************************************/
t_Bool spi_tclOnCarVideo::bLaunchVideo(const t_U32 cou32DevId, const t_U32 cou32AppId,
         const tenEnabledInfo coenSelection)
{
   ETG_TRACE_USR1(("spi_tclOnCarVideo::bLaunchVideo:Device ID-0x%x AppID-0x%x",
      cou32DevId,cou32AppId));

   SPI_INTENTIONALLY_UNUSED(coenSelection);

   return true;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarVideo::vStartVideoRendering()
 ***************************************************************************/
t_Void spi_tclOnCarVideo::vStartVideoRendering(t_Bool bStartVideoRendering)
{
   ETG_TRACE_USR1(("spi_tclOnCarVideo:vStartVideoRendering entered "));

   if (NULL != m_rVideoCallbacks.fpvVideoRenderStatusCb)
   {
      (m_rVideoCallbacks.fpvVideoRenderStatusCb)(bStartVideoRendering, e8DEV_TYPE_ONCAR);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarVideo::vGetVideoSettings()
 ***************************************************************************/
t_Void spi_tclOnCarVideo::vGetVideoSettings(const t_U32 cou32DevId, trVideoAttributes& rfrVideoAttributes)
{
   SPI_INTENTIONALLY_UNUSED(rfrVideoAttributes)
   SPI_INTENTIONALLY_UNUSED(cou32DevId)
   ETG_TRACE_USR1(("spi_tclOnCarVideo:vGetVideoSettings entered "));
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarVideo::vSetOrientationMode()
 ***************************************************************************/
t_Void spi_tclOnCarVideo::vSetOrientationMode(const t_U32 cou32DevId, const tenOrientationMode coenOrientationMode,
         const trUserContext& corfrUsrCntxt)
{
   ETG_TRACE_USR1(("spi_tclOnCarVideo:vSetOrientationMode entered "));

   //send response
   SPI_INTENTIONALLY_UNUSED(coenOrientationMode);
   if (NULL != m_rVideoCallbacks.fpvSetOrientationModeCb)
   {
      (m_rVideoCallbacks.fpvSetOrientationModeCb)(cou32DevId, e8UNSUPPORTED_OPERATION, corfrUsrCntxt, e8DEV_TYPE_ONCAR);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarVideo::vOnSelectDeviceResult()
 ***************************************************************************/
t_Void spi_tclOnCarVideo::vOnSelectDeviceResult(const t_U32 cou32DevId, const tenDeviceConnectionReq coenConnReq,
         const tenResponseCode coenRespCode)
{
   ETG_TRACE_USR1(("spi_tclOnCarVideo:vOnSelectDeviceResult entered for [%d]",cou32DevId));

   if ((e8DEVCONNREQ_SELECT == coenConnReq) && (e8FAILURE == coenRespCode) && (NULL != m_poCmdVideo))
   {
      ETG_TRACE_USR1(("Failure in Select Device- Perform Clean up"));
 
      vUnInitVideoSession(cou32DevId);
      m_poCmdVideo->vDestroyVideoEndpointInstance();
   }
   if((e8DEVCONNREQ_DESELECT == coenConnReq) && (NULL != m_poCmdVideo))
   {
      m_poCmdVideo->vDestroyVideoEndpointInstance();
   }
}

//lint restore
///////////////////////////////////////////////////////////////////////////////
// <EOF>
