/***********************************************************************/
/*!
* \file   SPI_OnCarTypes.h
* \brief  to define the types required for OnCar
*************************************************************************
\verbatim

PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:
AUTHOR:         jun5kor
COPYRIGHT:      &copy; 2015 Robert Bosch Car Multimedia GmbH
HISTORY:
Date        | Author                | Modification
01.02.2018  | Unmukt Jain           | Initial version
29.08.2018  | Ashwini Savadi        | Added enums and structure for Oncar Audio resource manager
\endverbatim
06.12.2018  | Ashwini Savadi        | Added enum for tenOnCarSinkError
*************************************************************************/


#ifndef SPI_ONCARTYPES_H
#define SPI_ONCARTYPES_H

#include "SPITypes.h"

enum tenOnCarObjectFactory
{
    //! ID for OnCarAPI class
    e8_ONCAR_OBJECTFACTORY_ONCARAPI           = 0,
    //! ID for OnCarAudioSinkEndpoint class
    e8_ONCAR_OBJECTFACTORY_AUDIO_ENDPOINT     = 1,
    //! ID for OnCarVideoEndpoint class
    e8_ONCAR_OBJECTFACTORY_VIDEO_ENDPOINT     = 2,
    //! ID for OnCarBluetoothEndpoint class
    e8_ONCAR_OBJECTFACTORY_BLUETOOTH_ENDPOINT = 3,
    //! ID for OnCarMetadataEndpoint class
    e8_ONCAR_OBJECTFACTORY_METADATA_ENDPOINT  = 4,
    //! ID for OnCarInputEndpoint class
    e8_ONCAR_OBJECTFACTORY_INPUT_ENDPOINT     = 5,
    //! ID for OnCarSensorEndpoint class
    e8_ONCAR_OBJECTFACTORY_SENSOR_ENDPOINT    = 6
};

static const t_String scoszOnCarLibPath = "/opt/bosch/base/lib/liboncar_sink_linux.so";
typedef void* (*fvCreateInstanceOf)(tenOnCarObjectFactory objectID);
typedef bool (*fvDestroyInstance)(void *ptrToObject, tenOnCarObjectFactory objectID);
#define STR_ONCAR_METADATA_LENGTH 200

typedef enum
{
   e32MODULEID_ONCARVIDEO     = 1,
   e32MODULEID_ONCARBLUETOOTH = 2,
   e32MODULEID_ONCARAUDIO     = 3,
   e32MODULEID_ONCARSESSION   = 4,
   e32MODULEID_ONCARINPUT     = 5,
   e32MODULEID_ONCARMETADATA  = 6
} tenOnCarModuleID;

enum tenOnCarSessionState
{
    //! OnCar session started.
    e8_ONCAR_SESSION_STATE_START   = 0,
    //! OnCar session stoped.
    e8_ONCAR_SESSION_STATE_STOP    = 1,
    //! OnCar user interaction needed
    e8_ONCAR_SESSION_STATE_USR_INTERACT_NEEDED = 2,
    //! OnCar user interaction not needed
    e8_ONCAR_SESSION_STATE_USR_INTERACT_NOT_NEEDED = 3,
    //! OnCar session state unknown.
    e8_ONCAR_SESSION_STATE_UNKNOWN = 4
};

enum tenOnCarBTPairingMethod
{
    e8_ONCAR_BT_PAIRING_OOB                = 1,
    e8_ONCAR_BT_PAIRING_NUMERIC_COMPARISON = 2,
    e8_ONCAR_BT_PAIRING_PASSKEY_ENTRY      = 3,
    e8_ONCAR_BT_PAIRING_PIN                = 4
};

enum tenOnCarBTPairingState
{
   e8_ONCAR_BTPAIRING_NOT_PAIRED        = 1,
   e8_ONCAR_BTPAIRING_ALREADY_PAIRED    = 2,
   e8_ONCAR_BTPAIRING_DELAYED_PAIRING   = 3
};

static const tenOnCarBTPairingMethod scenPreferredOnCarBTPairingMethod = e8_ONCAR_BT_PAIRING_NUMERIC_COMPARISON;

enum tenOnCarStreamState
{
    e8_ONCAR_AUD_STREAM_OPEN = 0,   // Playback start received. Blocked until Audio Activation.
    e8_ONCAR_AUD_STREAMING = 1,     // Audio Activation completed. Device configured. Audio streaming in progress.
    e8_ONCAR_AUD_STREAM_CLOSED = 2, // Playback stop received. Audio streaming ended.
    e8_ONCAR_AUD_STREAM_UNKNOWN = 3
};

enum tenOnCarDayNightMode
{
    //! Day mode
    e8_ONCAR_DAYNIGHT_MODE_DAY   = 0,
    //! Night mode
    e8_ONCAR_DAYNIGHT_MODE_NIGHT = 1
};

enum tenOnCarDrivingState
{
    //! Vehicle is parked
    e8_ONCAR_DRIVING_STATE_PARKED   = 0,
    //! Vehicle is moving
    e8_ONCAR_DRIVING_STATE_MOVING   = 1
};

enum tenOnCarTouchScreenType
{
  //! Capacitive Touch Screen
  e8_ONCAR_TOUCH_SCREEN_TYPE_CAPACITIVE = 1,
  //! Resistive Touch Screen
  e8_ONCAR_TOUCH_SCREEN_TYPE_RESISTIVE  = 2
};

enum tenOnCarTouchEventType
{
    //! Unknown Touch Event
    e8_ONCAR_TOUCH_EVENT_TYPE_UKNOWN       = 0,
    //! Single Touch Event
    e8_ONCAR_TOUCH_EVENT_TYPE_SINGLE_TOUCH = 1,
    //! Multi Touch Event
    e8_ONCAR_TOUCH_EVENT_TYPE_MULTI_TOUCH  = 2
};

typedef enum
{
     //!Indicates selected key is invalid.
     e8_ONCAR_INPUT_KEYCODE_INVALID           = 1,
     //!Handled by the application.
     //!SendingKEYCODE_BACK SHOULD NOT cause OnCar to lose video focus.
     e8_ONCAR_INPUT_KEYCODE_BACK              = 2,
     //!Indicates selected media SHOULD return to next track or station.
     e8_ONCAR_INPUT_KEYCODE_MEDIA_NEXT        = 6,
     //!Indicates selected media SHOULD return to previous track or station.
     e8_ONCAR_INPUT_KEYCODE_MEDIA_PREVIOUS    = 7,
     //!Indicates selected media SHOULD move forward
     e8_ONCAR_INPUT_KEYCODE_MEDIA_FORWARD     = 8,
     //!Indicates selected media SHOULD move back
     e8_ONCAR_INPUT_KEYCODE_MEDIA_REWIND      = 9,
     //!Indicates selected media SHOULD stop.
     e8_ONCAR_INPUT_KEYCODE_MEDIA_STOP        = 10,
     //!Indicates selected media SHOULD play.
     e8_ONCAR_INPUT_KEYCODE_MEDIA_PLAY        = 11,
     //!Indicates selected media SHOULD pause.
     e8_ONCAR_INPUT_KEYCODE_MEDIA_PAUSE       = 12,
     //!Indicates Rotary controller events are sent using relativeEvent api calls.
     //!The delta value SHOULD be positive for clockwise rotation and SHOULD be negative
     //!for counterclockwise rotation.
     e8_ONCAR_INPUT_KEYCODE_ROTARY_CONTROLLER = 13,
     //! Causes projected UI to switch to media screen.
     e8_ONCAR_INPUT_KEYCODE_MEDIA             = 14,
     //! Causes projected UI to switch to navigation screen.
     e8_ONCAR_INPUT_KEYCODE_NAVIGATION        = 15,
     //! Causes projected UI to switch to telephone screen.
     e8_ONCAR_INPUT_KEYCODE_TEL               = 16,
     //! Indicates Answering a phone call.
     e8_ONCAR_INPUT_KEYCODE_PHONE_CALL        = 17,
     //! Indicates ending a phone call.
     e8_ONCAR_INPUT_KEYCODE_PHONE_END         = 18,
     //! Indicates Joystick UP
     e8_ONCAR_INPUT_KEYCODE_SHIFT_UP         = 19,
     //! Indicates Joystick DOWN
     e8_ONCAR_INPUT_KEYCODE_SHIFT_DOWN         = 20,
     //! Indicates Joystick LEFT
     e8_ONCAR_INPUT_KEYCODE_SHIFT_LEFT         = 21,
     //! Indicates Joystick RIGHT
     e8_ONCAR_INPUT_KEYCODE_SHIFT_RIGHT         = 22,
     //! Indicates HOME screen
     e8_ONCAR_INPUT_KEYCODE_HOME                = 23,
     //! Indicates OK/ENTER
     e8_ONCAR_INPUT_KEYCODE_ENTER                 = 24
}tenOnCarInputKeyCodes;

struct trOnCarInputConfig
{
    t_U16 u16TouchLayerID;
    t_U16 u16TouchSurfaceID;
    t_U32 u32DisplayWidth;
    t_U32 u32DisplayHeight;
    t_U16 u16TouchWidth;
    t_U16 u16TouchHeight;
    tenOnCarTouchScreenType enOnCarTouchScreenType;
    tenOnCarTouchEventType enOnCarTouchEventType;
};

enum tenOnCarSinkError
{
    //! No error occurred
    e32_ONCAR_SINK_NO_ERROR                        = 0,
    /** Errors in case if the order of method invocation doesnot match the specified sequence.
    *   Eg :createConnection() called before calling init() method.
    */
    e32_ONCAR_SINK_INTERNAL_ERROR                  = 1,
    /** Errors in case where some messages received from phone were
    *   missed or not processed by oncar library.
    */
    e32_ONCAR_SINK_PROTOCOL_ERROR                  = 2,
    //! Errors in case when usb device connected doesnot exist on USB bus.
    e32_ONCAR_SINK_ERROR_NO_DEVICE                 = 3,
    /** Errors in case when time out has occured for any operation.
    *   Eg:Not acknowledging a pop up on phone or HU.
    */
    e32_ONCAR_SINK_ERROR_TIMEOUT                   = 4,
    /** Errors in case while performing anz IO operation.
    *   Eg:Error during USB endpoint IO operation.
    */
    e32_ONCAR_SINK_IO_ERROR                        = 5,
    /** Errors in case where oncar app is not present on the MD.
    */
    e32_ONCAR_SINK_APP_NOT_INSTALLED_ERROR         = 6,
    /** Errors in case oncar is not supported on the MD.
    */
    e32_ONCAR_SINK_PHONE_UNSUPPORTED_ERROR         = 7,
    /** Errors in case oncar authentication failed.
    */
    e32_ONCAR_SINK_AUTHENTICATION_FAILED_ERROR     = 8,
    //! Audio Sink config error
    e32_ONCAR_SINK_AUDIO_SINK_CONFIGURATION_ERROR  = 101,
    //! Audio Sink start error
    e32_ONCAR_SINK_AUDIO_SINK_START_ERROR          = 102,
    //! Audio sink write error
    e32_ONCAR_SINK_AUDIO_SINK_WRITE_ERROR          = 103,
    //! Audio Source read error
    e32_ONCAR_SINK_AUDIO_SOURCE_READ_ERROR         = 104,

    //! Video Sink config error
    e32_ONCAR_SINK_VIDEO_SINK_CONFIGURATION_ERROR  = 201,
    //! Video sink start error
    e32_ONCAR_SINK_VIDEO_SINK_START_ERROR          = 202,
    //! Video sink stop error
    e32_ONCAR_SINK_VIDEO_SINK_STOP_ERROR           = 203,
    //! Video Sink render error
    e32_ONCAR_SINK_VIDEO_SINK_RENDER_ERROR         = 204,

    //! Errors in case when BT pairing could not completed.Eg :Incorrect BT pairing pin.
    e32_ONCAR_SINK_BT_PAIRING_ERROR                = 301,
    //! Errors in case when BT connection could not established between HU and MD.
    e32_ONCAR_SINK_BT_CONNECTION_ERROR             = 302,

    //! Input config error
    e32_ONCAR_SINK_INPUT_TOUCH_INIT_ERROR          = 401,
    //! Errors in case of Touch Inputs Not Processed.
    e32_ONCAR_SINK_INPUT_EVENT_DISPATCH_ERROR      = 402

};

//! Video configuration required for OnCar Video
struct trOnCarVideoConfig
{
   //! Frame rate per second of video. Either 60 or 30
   t_U16 u16Fps;
    //! The width of projection screen in pixels
   t_U32 u32ProjectionScreenWidth;
   //! The height of projection screen in pixels
   t_U32 u32ProjectionScreenHeight;
   //! The density in dpi(dots per inch)
   t_U16 u16Density;
   //! The ratio of the physical pixel width over height.
   t_U16 u16PixelAspectRatio;
    //! Gstreamer Video Pipeline.
   t_String szVideoPipeline;

   trOnCarVideoConfig():u16Fps(30),u32ProjectionScreenWidth(800),u32ProjectionScreenHeight(480),u16Density(160),u16PixelAspectRatio(1){}
};

 enum tenOnCarVideoFocus
{
   e8_ONCAR_VIDEOFOCUS_UNKNOWN = 0,
   e8_ONCAR_VIDEOFOCUS_PROJECTED = 1,
   e8_ONCAR_VIDEOFOCUS_NATIVE = 2
} ;

/**
 *This enum represents the reason for video focus request callback.
 */
enum tenOnCarVideoFocusReason
{
   //! When launcher app is in background
   e8_ONCAR_VIDEOFOCUS_REASON_BACKGROUND           = 0,
    //! When launcher app is in foreground
   e8_ONCAR_VIDEOFOCUS_REASON_FOREGROUND           = 1,
    //! When user launches some other app from OnCar launcher app
   e8_ONCAR_VIDEOFOCUS_REASON_TRANSITION_START     = 2,
   //! When some other app started from OnCar launcher app has finished loading and now is rendering on phone
   e8_ONCAR_VIDEOFOCUS_REASON_TRANSITION_END       = 3,
   //! When Mobile device's screen is locked.
    e8_ONCAR_VIDEOFOCUS_REASON_SCREEN_OFF           = 4,
    //! When Mobile device's screen is unlocked.
    e8_ONCAR_VIDEOFOCUS_REASON_SCREEN_ON            = 5,
   //! When OEM icon is pressed by user on OnCar Projection screen.
   e8_ONCAR_VIDEOFOCUS_REASON_OEM_ICON_PRESSED     = 6,
   //! When Phone tile is pressed by user on OnCar Projection Screen.
   e8_ONCAR_VIDEOFOCUS_REASON_SHOW_NATIVE_BT_SCREEN = 7

};

enum tenOnCarVideoFocusState
{
   e8_ONCAR_VIDEOFOCUSSTATE_UNKNOWN = 0,
   e8_ONCAR_VIDEOFOCUSSTATE_GAIN = 1,
   e8_ONCAR_VIDEOFOCUSSTATE_GAIN_TRANSIENT = 2,
   e8_ONCAR_VIDEOFOCUSSTATE_LOSS = 3,
   e8_ONCAR_VIDEOFOCUSSTATE_LOSS_TRANSIENT = 4
};

//! response type to MD Video Focus requests
enum tenOnCarMDVideoFocusReqResp
{
   e8_ONCAR_VIDEOFOCUS_GRANT = 0,
   e8_ONCAR_VIDEOFOCUS_DENY = 1,
   e8_ONCAR_VIDEOFOCUS_DELAY = 2
};

//It is used to hold the data mentioned in the config file,
//to derive the current HU Video focus state based on the context & display flag
struct trOnCarAccVideoFocusNoti
{
   tenDisplayContext enAccDispCntxt;
   //This state is set, when HMI requests for Video Focus
   tenOnCarVideoFocusState enAccFocusReqType;
   //This state is set, when HMI releases for Video Focus
   tenOnCarVideoFocusState enAccFocusRelType;

   trOnCarAccVideoFocusNoti& operator=(const trOnCarAccVideoFocusNoti& corfrOnCarAccVideoFocusNoti)
   {
      if(&corfrOnCarAccVideoFocusNoti != this)
      {
         enAccDispCntxt = corfrOnCarAccVideoFocusNoti.enAccDispCntxt;
         enAccFocusReqType = corfrOnCarAccVideoFocusNoti.enAccFocusReqType;
         enAccFocusRelType = corfrOnCarAccVideoFocusNoti.enAccFocusRelType;
      } //if(&trOnCarAccVideoFocusNoti != this)
      return *this;
   }
};

struct trOnCarVideoFocusState
{
   //Current MD Focus State
    tenOnCarVideoFocusState enCurMDFocusState;
   //Accessory or MD Requested Focus State
    tenOnCarVideoFocusState enReqFocusState;
   //New MD Focus State
    tenOnCarVideoFocusState enUpdatedMDFocusState;

   trOnCarVideoFocusState& operator=(const trOnCarVideoFocusState& corfrOnCarVideoFocusState)
   {
      if( &corfrOnCarVideoFocusState != this )
      {
         enCurMDFocusState = corfrOnCarVideoFocusState.enCurMDFocusState;
         enReqFocusState = corfrOnCarVideoFocusState.enReqFocusState;
         enUpdatedMDFocusState = corfrOnCarVideoFocusState.enUpdatedMDFocusState;
      }   //if( &trOnCarVideoFocusState != this )
      return *this;
   }
};

struct trOnCarMDVideoFocusReq
{
   //Current Accessory display context
   tenDisplayContext enAccDispCntxt;
   tenOnCarMDVideoFocusReqResp enMDFocusReqResp_StateLoss;
   tenOnCarMDVideoFocusReqResp enMDFocusReqResp_StateLossTransient;

   trOnCarMDVideoFocusReq& operator=(const trOnCarMDVideoFocusReq& corfrOnCarMDVideoFocusReq)
   {
      if( &corfrOnCarMDVideoFocusReq != this )
      {
         enAccDispCntxt = corfrOnCarMDVideoFocusReq.enAccDispCntxt;
         enMDFocusReqResp_StateLoss = corfrOnCarMDVideoFocusReq.enMDFocusReqResp_StateLoss;
         enMDFocusReqResp_StateLossTransient = corfrOnCarMDVideoFocusReq.enMDFocusReqResp_StateLossTransient;
      }   //if( &corfrOnCarMDVideoFocusReq != this )
      return *this;
   }
};

enum tenOnCarMediaPlaybackState
{
   //! Current track is stopped playing
   e8_ONCAR_MEDIA_PLAYBACK_STATE_STOPPED      = 0,
   //! Current track is in playing state
   e8_ONCAR_MEDIA_PLAYBACK_STATE_PLAYING      = 1,
   //! Current track is in paused state
   e8_ONCAR_MEDIA_PLAYBACK_STATE_PAUSED       = 2,
   //! Identifies Default value(Startup case) or status is not known
   e8_ONCAR_MEDIA_PLAYBACK_STATE_NOT_KNOWN    = 0xFF
};

enum tenOnCarAudioStreamType
{
    //! Audio type media
    e8_ONCAR_AUDIOSTREAM_TYPE_MEDIA      = 0,
    //! Audio type guidance
    e8_ONCAR_AUDIOSTREAM_TYPE_GUIDANCE   = 1,
    //! Audio type unknown
    e8_ONCAR_AUDIOSTREAM_TYPE_UNKNOWN    = 2
};

//! Indicates audio context type
enum tenOnCarAudioContextType
{
   e8_ONCAR_AUDIO_CTXT_MAIN = 0,
   e8_ONCAR_AUDIO_CTXT_TRANSIENT = 1,
   e8_ONCAR_AUDIO_CTXT_TRANSIENT_MIX = 2
};

//! Indicates audio resource owner
enum tenOnCarAudioResourceOwner
{
   e8_ONCAR_AUDIO_OWNER_CAR = 0,
   e8_ONCAR_AUDIO_OWNER_DEVICE = 1
};

enum tenOnCarDeviceAudioFocusState
{
   e8_ON_CAR_FOCUS_STATE_GAIN = 0,
   e8_ON_CAR_FOCUS_STATE_LOSS = 1,
   e8_ON_CAR_FOCUS_STATE_LOSS_TRANSIENT = 2,
   e8_ON_CAR_FOCUS_STATE_MUTE = 3,
   e8_ON_CAR_FOCUS_STATE_DEMUTE = 4
};

enum tenOnCarAccessoryAudioFocusState
{
   e8_ON_CAR_ACC_FOCUS_STATE_GAINED = 1,
   e8_ON_CAR_ACC_FOCUS_STATE_GAINED_TRANSIENT = 2,
   e8_ON_CAR_ACC_FOCUS_STATE_LOST = 3,
   e8_ON_CAR_ACC_FOCUS_STATE_LOST_TRANSIENT = 4
};

enum tenOnCarAudioFocusRequest
{
    //! When MD wants to request for audio channel
    e8_ONCAR_AUDIOFOCUS_REQUEST_GAIN      = 0,
    //! When MD wants to release audio channel
    e8_ONCAR_AUDIOFOCUS_REQUEST_RELEASE   = 1,
    e8_ONCAR_AUDIOFOCUS_REQUEST_UNKNOWN     = 255
};

enum tenOnCarDeviceAudioFocusRequest
{
    //! When MD wants to request for MEDIA audio channel
    e8_ONCAR_AUDIOFOCUS_REQUEST_MEDIA_GAIN      = 0,
    //! When MD wants to release MEDIA audio channel
    e8_ONCAR_AUDIOFOCUS_REQUEST_MEDIA_RELEASE   = 1,
    //! When MD wants to request for guidance audio channel
    e8_ONCAR_AUDIOFOCUS_REQUEST_GUIDANCE_GAIN      = 2,
    //! When MD wants to release for GUIDANCE audio channel
    e8_ONCAR_AUDIOFOCUS_REQUEST_GUIDANCE_RELEASE   = 3,
    e8_ONCAR_DEVICE_AUDIOFOCUS_REQUEST_UNKNOWN   = 255
};

/*!
* \struct trOnCarudioResponseContext
* \brief structure to hold audio context data
*/
struct trOnCarAudioResponseContext
{
   //! Audio focus request from Device
    tenOnCarDeviceAudioFocusRequest    enDevAudFocusRequest;
   //! Current Audio focus state of Accessory
    tenOnCarAccessoryAudioFocusState      enCurAccAudFocusState;
   //! Current Audio context of Accessory
    tenOnCarAudioContextType enCurAccAudContext;
   //! New audio focus state of Device
    tenOnCarDeviceAudioFocusState      eNewDevAudFocusState;
   //! New audio focus state of Accessory
    tenOnCarAccessoryAudioFocusState      enNewAccAudFocusState;
    trOnCarAudioResponseContext& operator=(trOnCarAudioResponseContext &rfrOnCarAudioContext)
   {
      enDevAudFocusRequest = rfrOnCarAudioContext.enDevAudFocusRequest;
      enCurAccAudFocusState = rfrOnCarAudioContext.enCurAccAudFocusState;
      enCurAccAudContext = rfrOnCarAudioContext.enCurAccAudContext;
      eNewDevAudFocusState = rfrOnCarAudioContext.eNewDevAudFocusState;
      enNewAccAudFocusState = rfrOnCarAudioContext.enNewAccAudFocusState;
      return *this;
   }
};

/*!
* \struct trOnCarAudioNotifContext
* \brief structure to hold audio context data
*/
struct trOnCarAudioNotifContext
{
   //! Current playing Audio resource owner
    tenOnCarAudioResourceOwner    enAudResourceOwner;
   //! Current playing audio context type
    tenOnCarAudioContextType      enAudContextType;
   //! Current Audio focus state of Accessory
    tenOnCarAccessoryAudioFocusState  enCurAccAudFocusState;
   //! New audio focus state of Device
    tenOnCarDeviceAudioFocusState      eNewDevAudFocusState;
   //! New audio focus state of Accessory
    tenOnCarAccessoryAudioFocusState      enNewAccAudFocusState;
    trOnCarAudioNotifContext& operator=(trOnCarAudioNotifContext &rfrOnCarAudioNotifContext)
   {
      enAudResourceOwner = rfrOnCarAudioNotifContext.enAudResourceOwner;
      enAudContextType = rfrOnCarAudioNotifContext.enAudContextType;
      enCurAccAudFocusState = rfrOnCarAudioNotifContext.enCurAccAudFocusState;
      eNewDevAudFocusState = rfrOnCarAudioNotifContext.eNewDevAudFocusState;
      enNewAccAudFocusState = rfrOnCarAudioNotifContext.enNewAccAudFocusState;
      return *this;
   }
};

//! Structure representing OnCar media Metadata for current playing song
struct trOnCarMediaMetadata
{
    //! Name of current song
    t_Char cSong[STR_ONCAR_METADATA_LENGTH];
    //! Name of album
    t_Char cAlbum[STR_ONCAR_METADATA_LENGTH];
    //! Name of artist
    t_Char cArtist[STR_ONCAR_METADATA_LENGTH];
    //! Album art data in png format
    std::vector<t_U8> u8AlbumArt;
    //! Name of Playlist
    t_Char cPlaylist[STR_ONCAR_METADATA_LENGTH];
    //! Song length in seconds
    t_U32 u32DurationSeconds;
    //! Rating of current song
    t_U32 u32Rating;

    trOnCarMediaMetadata():u32DurationSeconds(0),u32Rating(0)
    {
        memset(cSong, 0, STR_ONCAR_METADATA_LENGTH);
        memset(cArtist, 0, STR_ONCAR_METADATA_LENGTH);
        memset(cAlbum, 0, STR_ONCAR_METADATA_LENGTH);
        memset(cPlaylist, 0, STR_ONCAR_METADATA_LENGTH);
    }
};

//! Structure representing OnCar media Metadata for playback status
struct trOnCarMediaPlaybackStatus
{
    //! Name of application playing the music on phone
    t_Char cMediaSource[STR_ONCAR_METADATA_LENGTH];
    //! State of current song
    tenOnCarMediaPlaybackState enState;
    //! Current playback of song in seconds
    t_U32 u32PlaybackSeconds;
    //! True if shuffle is on
    t_Bool bShuffle;
    //! True if repeat all
    t_Bool bRepeatAll;
    //! True if repeating one song
    t_Bool bRepeatOne;

    trOnCarMediaPlaybackStatus():enState(e8_ONCAR_MEDIA_PLAYBACK_STATE_NOT_KNOWN),
                                 u32PlaybackSeconds(0),bShuffle(false),bRepeatAll(false),bRepeatOne(false)
    {
        memset(cMediaSource, 0, STR_ONCAR_METADATA_LENGTH);
    }
};

/**
 *This structure represents audio configurations parameters that needs to be set
 *before starting OnCar session.
*/
struct trOnCarAudioConfig
{
   //! Indicates the priority with which audio threads must run.
   t_U16 u16threadPriority;
   //! Can be one of e8_AUDIOTYPE_MEDIA,e8_AUDIOTYPE_GUIDANCE
   tenOnCarAudioStreamType enOnCarAudStreamtype;
   //! Gstreamer Audio Pipeline.
   t_String szAudioPipeline;
};

//Vector to store the Audio configurations for OnCar
typedef std::vector<trOnCarAudioConfig> tvectrOnCarAudioConfig;

#endif //SPI_ONCARTYPES_H
//ifndef
