
/***********************************************************************/
/*!
* \file   spi_tclOnCarVideoResourceMngr.h
* \brief  OnCar Video resource manager
*************************************************************************
\verbatim

PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    OnCar Video resource manager
AUTHOR:         Unmukt Jain
COPYRIGHT:      &copy; RBEI

HISTORY:
Date        | Author                | Modification
20.06.2018  | Unmukt Jain           | Initial Version

\endverbatim
*************************************************************************/

#ifndef _SPI_TCLONCARVIDEORESOURCEMNGR_H_
#define _SPI_TCLONCARVIDEORESOURCEMNGR_H_


/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/
#include "SPI_OnCarTypes.h"
#include "Lock.h"


/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/
class spi_tclOnCarResourceMngr;

enum tenVideoPlaybackState
{
   e8VID_PB_STATE_START_RQSTD =0,
   e8VID_PB_STATE_STARTED =1,
   e8VID_PB_STATE_STOP_RQSTD =2,
   e8VID_PB_STATE_STOPPED =3
};

/****************************************************************************/
/*!
* \class   spi_tclOnCarVideoResourceMngr
* \brief   OnCar Video resource manager
*
*           This class implements the logic to when to request for Video Focus
*           and how to respond to the VideoFocus request from the Phone.
*
****************************************************************************/
class spi_tclOnCarVideoResourceMngr
{
public:

   /***************************************************************************
   *********************************PUBLIC*************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  spi_tclOnCarVideoResourceMngr::spi_tclAAPVideoResourceMngr()
   ***************************************************************************/
   /*!
   * \fn      spi_tclOnCarVideoResourceMngr(spi_tclOnCarResourceMngr* poAAPRsrcMngr)
   * \brief   parameterized  Constructor
   * \param   poAAPRsrcMngr : [IN] AAP resource manager
   * \sa      ~spi_tclOnCarVideoResourceMngr()
   **************************************************************************/
    spi_tclOnCarVideoResourceMngr(spi_tclOnCarResourceMngr* poOnCarRsrcMngr);

   /***************************************************************************
   ** FUNCTION:  spi_tclOnCarVideoResourceMngr::~spi_tclOnCarVideoResourceMngr()
   ***************************************************************************/
   /*!
   * \fn      ~spi_tclOnCarVideoResourceMngr()
   * \brief   Destructor
   * \sa      spi_tclOnCarVideoResourceMngr()
   **************************************************************************/
   ~spi_tclOnCarVideoResourceMngr();

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclOnCarVideoResourceMngr::bInitialize()
   ***************************************************************************/
   /*!
   * \fn      t_Bool bInitialize()
   * \brief   To Initialize all the Video resources
   * \retval  t_Bool
   * \sa      vUninitialize()
   **************************************************************************/
   t_Bool bInitialize();

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclOnCarVideoResourceMngr::vUninitialize()
   ***************************************************************************/
   /*!
   * \fn      t_Void vUninitialize()
   * \brief    To Uninitialize  the Video resources
   * \retval  t_Void
   * \sa      bInitialize()
   **************************************************************************/
   t_Void vUnInitialize();

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclOnCarVideoResourceMngr::vSetAccessoryDisplayContext()
   ***************************************************************************/
   /*!
   * \fn      t_Void vSetAccessoryDisplayContext(const t_U32 cou32DevId,
   *             t_Bool bDisplayFlag, tenDisplayContext enDisplayContext)
   * \brief   To send accessory display context related info .
   * \pram    cou32DevId    : [IN] Uniquely identifies the target Device.
   * \param   bDisplayFlag  : [IN] Display flag
   * \pram    enDisplayContext : [IN] display context
   * \retval  t_Void
   **************************************************************************/
   t_Void vSetAccessoryDisplayContext(const t_U32 cou32DevId,
      t_Bool bDisplayFlag, tenDisplayContext enDisplayContext);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclOnCarVideoResourceMngr::vOnSPISelectDeviceResult()
   ***************************************************************************/
   /*!
   * \fn      t_Void vOnSPISelectDeviceResult(t_U32 u32DevID,
   *             tenDeviceConnectionReq enDevConnReq,
   *             tenResponseCode enRespCode,
   *             tenErrorCode enErrorCode)
   * \brief   Called when SelectDevice operation is complete & with the result
   *           of the operation.
   * \param   u32DevID : [IN] Unique handle of selected device
   * \param   enDevConnReq: [IN]  Connection request type for the device
   * \param   enRespCode: [IN]  Response code enumeration
   * \param   enErrorCode: [IN]  Error code enumeration
   * \retval  None
   **************************************************************************/
   t_Void vOnSPISelectDeviceResult(t_U32 u32DevID,
      tenDeviceConnectionReq enDevConnReq,
      tenResponseCode enRespCode,
      tenErrorCode enErrorCode);

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclOnCarVideoResourceMngr::vSetAccessoryDisplayMode(t_U32...
   ***************************************************************************/
   /*!
   * \fn     vSetAccessoryDisplayMode()
   * \brief  Accessory display mode update request.
   * \param  [IN] cou32DeviceHandle      : Uniquely identifies the target Device.
   * \param  [IN] corDisplayContext     : Display context info
   * \param  [IN] corDisplayConstraint  : DiDisplay constraint info
   * \param  [IN] coenDisplayInfo       : Display info flag
   * \sa
   **************************************************************************/
   t_Void vSetAccessoryDisplayMode(const t_U32 cou32DeviceHandle,
      const trDisplayContext corDisplayContext,
      const trDisplayConstraint corDisplayConstraint,
      const tenDisplayInfo coenDisplayInfo);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclOnCarVideoResourceMngr::vRequestProjUI()
   ***************************************************************************/
   /*!
   * \fn      t_Void vRequestProjUI(t_U32 u32DevId,
   *                 ct_U32 u32AppId)
   * \brief   To Launch the Video for the requested app
   * \pram    u32DevId  : [IN] Uniquely identifies the target Device.
   * \pram    u32AppId  : [IN] Application Id
   * \retval  t_Void
   **************************************************************************/
   t_Void vRequestProjUI(t_U32 u32DevId,t_U32 u32AppId);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclOnCarVideoResourceMngr::vPostOnCarRequestVideoFocusMsg()
   ***************************************************************************/
   /*!
   * \fn      t_Void vPostOnCarRequestVideoFocusMsg(tenVideoFocus enVideoFocus,
   *                 tenVideoFocusReason enVideoFocusReason)
   * \brief   Method to update that the Video Focus is requested
   *          or rejected from the MD
   * \param   enVideoFocus         : [IN] Video Focus Mode
   * \param   enVideoFocusReason   : [IN] Reason for the video focus request.
   * \retval  t_Void
   **************************************************************************/
   t_Void vPostOnCarRequestVideoFocusMsg(const tenOnCarVideoFocus enOnCarVideoFocus,
                                         const tenOnCarVideoFocusReason enOnCarVideoFocusReason);

   /***************************************************************************
   ** FUNCTION:  spi_tclOnCarVideoResourceMngr::vPostOnCarStartVideoPlaybackMsg()
   ***************************************************************************/
   /*!
   * \fn      virtual vPostOnCarStartVideoPlaybackMsg()
   * \brief   Invoked to provide information about OnCar Start Video Playback
   **************************************************************************/
   t_Void vPostOnCarStartVideoPlaybackMsg();

   /***************************************************************************
   ** FUNCTION:  spi_tclOnCarVideoResourceMngr::vPostOnCarStopVideoPlaybackMsg()
   ***************************************************************************/
   /*!
   * \fn      virtual vPostOnCarStopVideoPlaybackMsg()
   * \brief   Invoked to provide information about OnCar Stop Video Playback
   **************************************************************************/
   t_Void vPostOnCarStopVideoPlaybackMsg();

   /***************************************************************************
   ****************************END OF PUBLIC***********************************
   ***************************************************************************/

protected:

   /***************************************************************************
   *********************************PROTECTED**********************************
   ***************************************************************************/


   /***************************************************************************
   ****************************END OF PROTECTED********************************
   ***************************************************************************/

private:

   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/

   /***************************************************************************
   ** FUNCTION:  spi_tclOnCarVideoResourceMngr::spi_tclOnCarVideoResourceMngr()
   ***************************************************************************/
   /*!
   * \fn      spi_tclOnCarVideoResourceMngr()
   * \brief   Constructor
   * \sa      ~spi_tclOnCarVideoResourceMngr()
   **************************************************************************/
   spi_tclOnCarVideoResourceMngr();


   /***************************************************************************
   ** FUNCTION:  spi_tclOnCarVideoResourceMngr(const spi_tclOnCarVideoResourceMngr...
   ***************************************************************************/
   /*!
   * \fn      spi_tclOnCarVideoResourceMngr(
   *                             const spi_tclOnCarVideoResourceMngr& corfoSrc)
   * \brief   Copy constructor - Do not allow the creation of copy constructor
   * \param   corfoSrc : [IN] reference to source data interface object
   * \retval
   * \sa      spi_tclOnCarVideoResourceMngr()
   ***************************************************************************/
   spi_tclOnCarVideoResourceMngr(const spi_tclOnCarVideoResourceMngr& corfoSrc);


   /***************************************************************************
   ** FUNCTION:  spi_tclOnCarVideoResourceMngr& operator=( const spi_tclOnCar...
   ***************************************************************************/
   /*!
   * \fn      spi_tclOnCarVideoResourceMngr& operator=(
   *                          const spi_tclOnCarVideoResourceMngr& corfoSrc))
   * \brief   Assignment operator
   * \param   corfoSrc : [IN] reference to source data interface object
   * \retval
   * \sa      spi_tclOnCarVideoResourceMngr(const spi_tclOnCarVideoResourceMngr& otrSrc)
   ***************************************************************************/
   spi_tclOnCarVideoResourceMngr& operator=(const spi_tclOnCarVideoResourceMngr& corfoSrc);

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclOnCarVideoResourceMngr::bGetAccVideoFocusState()
   ***************************************************************************/
   /*!
   * \fn      t_Bool bGetAccVideoFocusState(tenDisplayContext enAccDispCntxt,
   *                 t_Bool bDisplayFlag,tenOnCarVideoFocusState& rfenVideoFocusState)
   * \brief   Method to get the current HMI requesting Video Focus state from the
   *          configuration file. Video Focus states for various contexts is defined
   *          based on the Carplay experience
   * \param   enAccDispCntxt : [IN] Accessory Display Context
   * \param   bDisplayFlag   : [IN] TRUE - HMI is requesting for the Video Focus
   *                          FALSE - HMI screen with the received context has lost focus
   * \param   rfenVideoFocusState : [OUT] HMI requested Video focus state
   * \retval  t_Bool - TRUE - if the received display context is available in the config
   *                   FALSE - if the received display context is not available in the config
   **************************************************************************/
   t_Bool bGetAccVideoFocusState(tenDisplayContext enAccDispCntxt,
      t_Bool bDisplayFlag, tenOnCarVideoFocusState& rfenVideoFocusState);

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclOnCarVideoResourceMngr::bGetUpdatedMDFocusState()
   ***************************************************************************/
   /*!
   * \fn      t_Bool bGetUpdatedMDFocusState(tenOnCarVideoFocusState enCurMDFocusState,
   *                 tenOnCarVideoFocusState enCurAccFocusState,
   *                 tenOnCarVideoFocusState& rfenUpdatedMDFocusState)
   * \brief   Method to get the New MD Focus state based on the Current Hu & MD focus states
   * \param   enCurMDFocusState       : [IN] MD Current Focus State
   * \param   enCurAccFocusState      : [IN] Accessory Current Focus State
   * \param   rfenUpdatedMDFocusState : [OUT] MD New Focus State
   * \retval  t_Bool - TRUE - if the received focus states are available in the config
   *                   FALSE - if the received ocus states are not available in the config
   **************************************************************************/
   t_Bool bGetUpdatedMDFocusState(tenOnCarVideoFocusState enCurMDFocusState,
           tenOnCarVideoFocusState enCurAccFocusState,
           tenOnCarVideoFocusState& rfenUpdatedMDFocusState);

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclOnCarVideoResourceMngr::bGetAccRespType()
   ***************************************************************************/
   /*!
   * \fn      t_Bool bGetAccRespType(tenAAPMDVideoFocusReqResp& rfenRespType)
   * \brief   Method to get the Accessory Response for the MD Focus request
   *          for PROJECTED_MODE
   * \param   rfenRespType    : [IN] Accessory Response for the Focus request
   * \retval  t_Bool - TRUE - if the current display context is available in the config
   **************************************************************************/
   t_Bool bGetAccRespType(tenOnCarMDVideoFocusReqResp& rfenRespType);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclOnCarVideoResourceMngr::vSetVideoFocus()
   ***************************************************************************/
   /*!
   * \fn      t_Void vSetVideoFocus(tenVideoFocus enVideoFocus,
   *                 t_Bool bUnsolicited)
   * \brief   Method to send the Video Focus request to the wrapper
   * \param   enVideoFocus : [IN] Video Focus Mode
   * \retval  t_Void
   **************************************************************************/
   t_Void vSetVideoFocus(tenOnCarVideoFocus enVideoFocus);

   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclOnCarVideoResourceMngr::bProcessRequestProjUI()
    ***************************************************************************/
   /*!
    * \fn     t_Bool bProcessRequestProjUI();
    * \brief  Method to check whether the launch app can be processed,
    *         based on the current display context.
    * \param  None
    * \retval  t_Bool
    **************************************************************************/
   t_Bool bProcessRequestProjUI();

   //OnCar resource manager pointer
   spi_tclOnCarResourceMngr* const m_cpoOnCarRsrcMngr;

   //Current Video Focus State of MD
   tenOnCarVideoFocusState m_enCurMDFocusState;

   //To contain constraint behavior
   tenDiPOConstraint m_enGrantConstraint;

   //to hold last state
   tenDiPOTransferType m_enPrevTransferType;

   //Current Display context of HMI
   tenDisplayContext m_enCurAccDispCntxt;

   //Current Video Playback State
   tenVideoPlaybackState m_enPlaybackState;

   Lock m_oVideoPlaybackStateLock;

   Lock m_oCurMDFocusStateLock;

   //! Currently selected device
   t_U32 m_u32SelectedDeviceID;

   //! Initial Video Focus to MD.
   t_Bool m_bInitialMDVideoFocus;

   //! Lock for Video Setup
   Lock  m_oVideoSetupLock;

   /***************************************************************************
   ****************************END OF PRIVATE *********************************
   ***************************************************************************/

}; //class spi_tclOnCarVideoResourceMngr

#endif //_SPI_TCLONCARVIDEORESOURCEMNGR_H_

///////////////////////////////////////////////////////////////////////////////
// <EOF>
