/*!
 *******************************************************************************
 * \file              spi_tclOnCarCmdInput.h
 * \brief             Input wrapper for OnCar
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Input wrapper for OnCar
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 20.02.2018 |  Unmukt Jain                 | Initial Version

 \endverbatim
 ******************************************************************************/

#ifndef SPI_TCLONCARCMDINPUT_H_
#define SPI_TCLONCARCMDINPUT_H_

#include "SPI_OnCarTypes.h"
#include "RespBase.h"
#include "spi_tclOnCarInputCbs.h"
#include "Lock.h"
#include <set>

/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/

/*!
 * \class spi_tclOnCarCmdInput
 * \brief
 */

struct trOnCarKeyCode
{
   //! Corresponding Switch Code
   tenKeyCode enKeyCode;
   //!Corresponding OnCar Switch Code
   tenOnCarInputKeyCodes enOnCarInputKeyCode;

};


class spi_tclOnCarCmdInput
{
public:

   /***************************************************************************
    *********************************PUBLIC*************************************
    ***************************************************************************/

   /***************************************************************************
    ** FUNCTION:  spi_tclOnCarCmdInput::spi_tclOnCarCmdInput();
    ***************************************************************************/
   /*!
    * \fn     spi_tclOnCarCmdInput()
    * \brief  Default Constructor
    * \sa     spi_tclOnCarCmdInput()
    **************************************************************************/
    spi_tclOnCarCmdInput();

   /***************************************************************************
    ** FUNCTION:  spi_tclOnCarCmdInput::~spi_tclOnCarCmdInput()
    ***************************************************************************/
   /*!
    * \fn      virtual ~spi_tclOnCarCmdInput()
    * \brief   Virtual Destructor
    * \sa      spi_tclOnCarCmdInput()
    **************************************************************************/
   ~spi_tclOnCarCmdInput();

   /***************************************************************************
    ** FUNCTION:  virtual spi_tclOnCarCmdInput::bInitializeInputEndpoint()
    ***************************************************************************/
   /*!
    * \fn      bInitializeInputEndpoint()
    * \brief   Initializes the InputSource Endpoint, registers keycodes and touch
    *         screen.
    * \param   rAAPInputConfig  : [IN] Display Configuration 
    * \sa      bUnInitializeInputEndpoint()
    **************************************************************************/
   t_Bool bInitializeInputEndpoint(const trOnCarInputConfig &rOnCarInputConfig);

   /***************************************************************************
    ** FUNCTION:  virtual spi_tclOnCarCmdInput::vUnInitializeInputEndpoint()
    ***************************************************************************/
   /*!
    * \fn      vUnInitializeInputEndpoint()
    * \brief   Uninitializes the InputSource Endpoint.
    * \sa      bInitializeInputEndpoint()
    **************************************************************************/
   t_Void vUnInitializeInputEndpoint();

   /***************************************************************************
    ** FUNCTION:  virtual spi_tclOnCarCmdInput::vDestroyInputEnpointInstance()
    ***************************************************************************/
   /*!
    * \fn      vDestroyInputEnpointInstance()
    * \brief   Destroy the InputSource Endpoint.
    * \sa      None
    **************************************************************************/
   t_Void vDestroyInputEnpointInstance();

   /***************************************************************************
    ** FUNCTION: t_Void spi_tclOnCarCmdInput::vReportTouch
    ***************************************************************************/
   /*!
    * \fn      vReportTouch(t_U32 u32DeviceHandle,trTouchData &rfrTouchData)
    * \brief   Receives the Touch events and forwards it to Project Specific Endpoint
    *         wrapper.
    * \param   u32DeviceHandle  : [IN] unique identifier to AAP device
    * \param   rfrTouchData     : [IN] reference to touch data structure which contains
    *          touch details received /ref trTouchData
    * \param   rScalingAttributes : [IN] Scaling attributes to scale the touch coordinates
    * \retval  NONE
    **************************************************************************/
   t_Void vReportTouch(t_U32 u32DeviceHandle, trTouchData &rfrTouchData,
                       trScalingAttributes rScalingAttributes);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclOnCarCmdInput::vReportkey
    ***************************************************************************/
   /*!
    * \fn      vReportkey(t_U32 u32DeviceHandle, tenKeyMode enKeyMode,
    tenKeyCode enKeyCode)
    * \brief   Receives hard key events and forwards it to Project Specific Endpoint.
    * \param   u32DeviceHandle : [IN] unique identifier to AAP device
    * \param   enKeyMode       : [IN] indicates keypress or keyrelease
    * \param   enKeyCode       : [IN] unique key code identifier
    * \retval  NONE
    **************************************************************************/
   t_Void vReportkey(t_U32 u32DeviceHandle, tenKeyMode enKeyMode,
               tenKeyCode enKeyCode);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclOnCarCmdInput::vReportKnobkey
    ***************************************************************************/
   /*!
    * \fn      vReportKnobkey(t_U32 u32DeviceHandle, t_S32 s32DeltaCnts)
    * \brief   Receives hard key events and forwards it to Project Specific Endpoint.
    * \param   u32DeviceHandle : [IN] unique identifier to AAP device
    * \param   s32DeltaCnts    : [IN] Knob Encoder delta count change
    * \retval  NONE
    **************************************************************************/
   t_Void vReportKnobkey(t_U32 u32DeviceHandle, t_S32 s32DeltaCnts);

protected:

   /***************************************************************************
    *********************************PROTECTED**********************************
    ***************************************************************************/

   /***************************************************************************
    ** FUNCTION:  spi_tclOnCarCmdInput(const spi_tclOnCarCmdInput...
    ***************************************************************************/
   /*!
    * \fn      spi_tclOnCarCmdInput(const spi_tclOnCarCmdInput& corfoSrc)
    * \brief   Copy constructor - Do not allow the creation of copy constructor
    * \param   corfoSrc : [IN] reference to source data interface object
    * \retval
    * \sa      spi_tclOnCarCmdInput()
    ***************************************************************************/
   spi_tclOnCarCmdInput(const spi_tclOnCarCmdInput& corfoSrc);

   /***************************************************************************
    ** FUNCTION:  spi_tclOnCarCmdInput& operator=( const spi_tclOnCarCmdInput...
    ***************************************************************************/
   /*!
    * \fn      spi_tclOnCarCmdInput& operator=(const spi_tclOnCarCmdInput& corfoSrc))
    * \brief   Assignment operator
    * \param   corfoSrc : [IN] reference to source data interface object
    * \retval
    * \sa      spi_tclOnCarCmdInput(const spi_tclAAPCmdInput& otrSrc)
    ***************************************************************************/
   spi_tclOnCarCmdInput& operator=(const spi_tclOnCarCmdInput& corfoSrc);

   /***************************************************************************
    ****************************END OF PROTECTED********************************
    ***************************************************************************/
private:

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclOnCarCmdInput::vSetOnCarCodes
    ***************************************************************************/
   /*!
    * \fn      vSetOnCarCodes()
    * \brief   Maps SPI and OnCar key codes.
    * \param   NONE
    * \retval  NONE
    **************************************************************************/
   t_Void vSetOnCarCodes();

   /****************************************************************************
    ** FUNCTION:  t_Void spi_tclOnCarCmdInput::enGetKeyCode
    ***************************************************************************/
   /*!
    * \fn      enGetKeyCode()
    * \brief   Retrieves OnCar key code for corresponding SPI key code.
    * \param   NONE
    * \retval  NONE
    **************************************************************************/
   tenOnCarInputKeyCodes enGetKeyCode(tenKeyCode enSpiKeyCode);


   //! Input Endpoint pointer
   OnCarInputEndpoint* m_poOnCarInputEndpoint;

   //! shared pointer for OnCar Input callbacks
   std::shared_ptr<IOnCarInputSourceCallbacks> m_spoInputCbs;

   //! Lock object for Input endpoint
   Lock  m_oInputEndpointLock;

   //! Mapping between SPI and OnCar keycodes
   std::map<tenKeyCode, tenOnCarInputKeyCodes> m_keyCodeMap;
};
#endif /* SPI_TCLONCARCMDINPUT_H_ */
