/*!
 *******************************************************************************
 * \file         spi_tclOnCarBluetooth.cpp
 * \brief        OnCar Bluetooth class
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    OnCar Bluetooth handler class for SPI
 AUTHOR:         irh1kor
 COPYRIGHT:      &copy; 2015 Robert Bosch Car Multimedia GmbH

 HISTORY:
 Date        | Author                | Modification
 12.03.2018  | Rishav Sardar         | Initial Version
 04.12.2018 |  Dundamma S B          | Registering BT callbacks to ExtCmdBluetooth class directly without the interface of common Bluetooth class

 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "BaseTypes.h"
#include "spi_tclOnCarCmdBluetooth.h"
#include "spi_tclBluetoothIntf.h"
#include "spi_tclOnCarBluetooth.h"
#include "spi_tclOnCarCmdBluetooth.h"
#include "spi_tclOnCarManager.h"
#include "spi_tclBluetoothPolicyBase.h"
#include "spi_tclExtCmdBluetooth.h"
#include "spi_tclExtCompManager.h"
#include "spi_tclBluetoothIntf.h"

#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_BLUETOOTH
#include "trcGenProj/Header/spi_tclOnCarBluetooth.cpp.trc.h"
#endif
#endif
//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e746 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported

static const tenBTSetPairingMethod scenBTSetPreferredOnCarPairingMethod =
      e8PAIRING_TYPE_SSP_NUMERIC_COMPARISON;

static t_Bool sbBTLimModeRequested = false;

/******************************************************************************
 | defines and macros and constants(scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: global)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/

/***************************************************************************
 *********************************PUBLIC*************************************
 ***************************************************************************/

/***************************************************************************
 ** FUNCTION:  spi_tclOnCarBluetooth::spi_tclOnCarBluetooth();
 ***************************************************************************/
spi_tclOnCarBluetooth::spi_tclOnCarBluetooth(spi_tclBluetoothIntf* poBTInterface) :
   m_cpoBTInterface(poBTInterface),
   m_enPairingStatus(e8DEVICE_PAIRING_NOT_STARTED),
   m_enPendingBTLimAction(e8BT_LIM_ACTION_UNKNOWN)
{
   ETG_TRACE_USR1(("[CONSTRUCTOR]:spi_tclOnCarBluetooth::spi_tclOnCarBluetooth() entered "));
   SPI_NORMAL_ASSERT(NULL == m_cpoBTInterface);
   ETG_TRACE_USR1(("[CONSTRUCTOR]:spi_tclOnCarBluetooth::spi_tclOnCarBluetooth() left "));
}//!end of spi_tclOnCarBluetooth()

/***************************************************************************
 ** FUNCTION:  spi_tclOnCarBluetooth::~spi_tclMySPINBluetooth();
 ***************************************************************************/
spi_tclOnCarBluetooth::~spi_tclOnCarBluetooth()
{
   ETG_TRACE_USR1(("[DESTRUCTOR]:spi_tclOnCarBluetooth::~spi_tclOnCarBluetooth() entered "));
   ETG_TRACE_USR1(("[DESTRUCTOR]:spi_tclOnCarBluetooth::~spi_tclOnCarBluetooth() left "));
}//!end of ~spi_tclOnCarBluetooth()

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclOnCarBluetooth::bInitialise();
 ***************************************************************************/
t_Bool spi_tclOnCarBluetooth::bInitialise()
{
    ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarBluetooth::bInitialise() entered "));
   //! Register to BT client
    t_Bool bInit = false;
    spi_tclOnCarManager *poOnCarManager = spi_tclOnCarManager::getInstance();
    SPI_NORMAL_ASSERT(NULL == poOnCarManager);
    if (poOnCarManager)
    {
        //! Register with OnCar manager for BT callbacks
        bInit = poOnCarManager->bRegisterObject((spi_tclOnCarRespBluetooth*) this);
    }//if (poOnCarManager)

    RespRegister *pRespRegister = RespRegister::getInstance();
    if(NULL!= pRespRegister)
    {
        t_Bool bRet = pRespRegister->bRegisterObject((spi_tclExtRespBluetooth*)this);
        bInit = bInit && bRet;
    }
    return bInit;
    ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarBluetooth::bInitialise() left "));
}//!end of bInitialise()

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarBluetooth::vOnSPISelectDeviceRequest(
 ***************************************************************************/
t_Void spi_tclOnCarBluetooth::vOnSPISelectDeviceRequest(const trSelectDeviceRequest& corfrSelectReq)
{
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarBluetooth::vOnSPISelectDeviceRequest() entered "));
   SPI_INTENTIONALLY_UNUSED(corfrSelectReq);
   
   if (m_cpoBTInterface)
   {
      //! Set Device status - this is in order to prevent SwitchDevice during ongoing selection
      m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_SELECTION_IN_PROGRESS);
      //! Initialize BT Endpoint
      spi_tclOnCarManager* poOnCarManager = spi_tclOnCarManager::getInstance();
      spi_tclOnCarCmdBluetooth* poCmdBluetooth = (NULL != poOnCarManager)?
            (poOnCarManager->poGetBluetoothInstance()) : (NULL);
      t_Bool bInitSuccess = false;

      if ((poCmdBluetooth) && (NULL != m_poBluetoothSettings))
      {
         t_String szVehicleBTAddress;
         m_poBluetoothSettings->vGetVehicleBTAddress(szVehicleBTAddress);
         bInitSuccess = poCmdBluetooth->bInitialiseBTEndpoint(
               szVehicleBTAddress, scenPreferredOnCarBTPairingMethod);
      }//if ((poCmdBluetooth) && (poExtCmdBluetoothIntf))

      //! send success result for SelectDevice
      m_cpoBTInterface->vSendSelectDeviceResult(bInitSuccess);
   }//if (m_cpoBTInterface)
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarBluetooth::vOnSPISelectDeviceRequest() left "));
} //!end of vOnSPISelectDeviceRequest()

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarBluetooth::vOnSPISelectDeviceResponse(
***************************************************************************/
t_Void spi_tclOnCarBluetooth::vOnSPISelectDeviceResponse(const trSelectDeviceRequest& corfrSelectReq,
      tenResponseCode enRespCode)
{
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarBluetooth::vOnSPISelectDeviceResponse() entered : ResponseCode = %d",enRespCode));
   SPI_INTENTIONALLY_UNUSED(corfrSelectReq);
   
   //! szBTDeviceName is an optional parameter and should be used when BT address is not available.
   //! In cases (name not known, BT address is available) this parameter should be an empty string
   t_String szBTDeviceName;
     
   if ((e8FAILURE == enRespCode) && (sbBTLimModeRequested) && (m_cpoBTInterface))
   {
      //! Clear device status
      m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_COMPLETE);
      spi_tclExtCmdBluetoothIntf *poExtCmdBluetoothIntf = m_cpoBTInterface->poFetchExtCmdBluetoothIntf();
      if(poExtCmdBluetoothIntf)
      {        
         sbBTLimModeRequested = poExtCmdBluetoothIntf->bSetBTLimitationMode(
                        m_cpoBTInterface->szGetSelectedDevBTAddress(),
                        szBTDeviceName,
                        e8BT_TECH_ONCAR,
                        e8BT_COMM_CHN_USB,
                        e8BT_LIM_ACTION_DEACTIVATE);
         if(false == sbBTLimModeRequested)
         {
             m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_DELAYED); 
             m_enPendingBTLimAction = e8BT_LIM_ACTION_DEACTIVATE;
         }
      }//if (m_cpoBTInterface)
   }////if ((sbBTLimModeRequested) && (m_cpoBTInterface))

   //! On selection failure, perform cleanup.
   if (e8FAILURE == enRespCode)
   {
      spi_tclOnCarManager* poOnCarManager = spi_tclOnCarManager::getInstance();
      spi_tclOnCarCmdBluetooth* poCmdBluetooth = (NULL != poOnCarManager)?
            (poOnCarManager->poGetBluetoothInstance()) : (NULL);
      if (poCmdBluetooth)
      {
         poCmdBluetooth->vUninitialiseBTEndpoint();
         poCmdBluetooth->vDestroyBtEndpointInstance();
      }//if (poCmdBluetooth)
   }//else if (e8FAILURE == enRespCode)
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarBluetooth::vOnSPISelectDeviceResponse() left "));
} //!end of vOnSPISelectDeviceResponse()


/***************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarBluetooth::vOnSPIDeselectDeviceRequest(
 ***************************************************************************/
t_Void spi_tclOnCarBluetooth::vOnSPIDeselectDeviceRequest(const trSelectDeviceRequest& corfrSelectReq)
{
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarBluetooth::vOnSPIDeselectDeviceRequest() entered"));
   SPI_INTENTIONALLY_UNUSED(corfrSelectReq);
   if (m_cpoBTInterface)
   {
      // Set Device status - this is in order to prevent SwitchDevice during ongoing deselection
      m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_DESELECTION_IN_PROGRESS);
      spi_tclOnCarManager* poOnCarManager = spi_tclOnCarManager::getInstance();
      spi_tclOnCarCmdBluetooth* poCmdBluetooth = (NULL != poOnCarManager)?
                   (poOnCarManager->poGetBluetoothInstance()) : (NULL);
      if (poCmdBluetooth)
      {
          poCmdBluetooth->vUninitialiseBTEndpoint();
      }//if (poCmdBluetooth)
      // Nothing else to be done. Simply send success result.
      m_cpoBTInterface->vSendSelectDeviceResult(true);
   }//if (m_cpoBTInterface)
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarBluetooth::vOnSPIDeselectDeviceRequest() left"));
} //!end of vOnSPIDeselectDeviceRequest()

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarBluetooth::vOnSPIDeselectDeviceResponse(
 ***************************************************************************/
t_Void spi_tclOnCarBluetooth::vOnSPIDeselectDeviceResponse(const trSelectDeviceRequest& corfrSelectReq,
         tenResponseCode enRespCode)
{
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarBluetooth::vOnSPIDeselectDeviceResponse() entered : ResponseCode = %d",enRespCode));
   SPI_INTENTIONALLY_UNUSED(corfrSelectReq);
   SPI_INTENTIONALLY_UNUSED(enRespCode);
   
   //! szBTDeviceName is an optional parameter and should be used when BT address is not available.
   //! In cases (name not known, BT address is available) this parameter should be an empty string
   t_String szBTDeviceName;

    // Clear device status
   if (m_cpoBTInterface)
   {
      m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_COMPLETE);
   }//if (m_cpoBTInterface)

   //! Perform cleanup
   spi_tclOnCarManager* poOnCarManager = spi_tclOnCarManager::getInstance();
   spi_tclOnCarCmdBluetooth* poCmdBluetooth = (NULL != poOnCarManager)?
         (poOnCarManager->poGetBluetoothInstance()) : (NULL);
   if (poCmdBluetooth)
   {
      poCmdBluetooth->vDestroyBtEndpointInstance();
   }//if (poCmdBluetooth)

   vSetPairingStatus(e8DEVICE_PAIRING_NOT_STARTED);

   if ((sbBTLimModeRequested) && (m_cpoBTInterface))
   {
      spi_tclExtCmdBluetoothIntf *poExtCmdBluetoothIntf = m_cpoBTInterface->poFetchExtCmdBluetoothIntf();
      if(poExtCmdBluetoothIntf)
      {
          sbBTLimModeRequested = poExtCmdBluetoothIntf->bSetBTLimitationMode(
                m_cpoBTInterface->szGetSelectedDevBTAddress(),
                szBTDeviceName,
                e8BT_TECH_ONCAR,
                e8BT_COMM_CHN_USB,
                e8BT_LIM_ACTION_DEACTIVATE);
          if(false == sbBTLimModeRequested)
          {
              m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_DELAYED);
              m_enPendingBTLimAction = e8BT_LIM_ACTION_DEACTIVATE;
          }
      }//if(poExtCmdBluetoothIntf)
   }//if ((sbBTLimModeRequested) && (m_cpoBTInterface))
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarBluetooth::vOnSPIDeselectDeviceResponse() left "));
} //!end of vOnSPIDeselectDeviceResponse()

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarBluetooth::vPostBTPairingRequest(...)
***************************************************************************/
t_Void spi_tclOnCarBluetooth::vPostBTPairingRequest(t_String poszOnCarBTAddress,
        tenOnCarBTPairingMethod enPairingMethod)
{
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarBluetooth::vPostBTPairingRequest() entered: "
         "PairingMethod = %d, OnCarBTAddress = %s ",
         ETG_ENUM(ONCAR_BTPAIRING_METHOD, enPairingMethod), poszOnCarBTAddress.c_str()));

   //! Store BT address of OnCar device
   //! IMPORTANT! This must be done first, since in following logic, other functions will use the stored BT address.
   t_Bool bIsBTAddrAlreadyReceived = false;
   
   //! szBTDeviceName is an optional parameter and should be used when BT address is not available.
   //! In cases (name not known, BT address is available) this parameter should be an empty string
   t_String szBTDeviceName;
   
   spi_tclExtCmdBluetoothIntf *poExtCmdBluetoothIntf = NULL;
   if (m_cpoBTInterface)
   {
      bIsBTAddrAlreadyReceived = (poszOnCarBTAddress == m_cpoBTInterface->szGetSelectedDevBTAddress());
      m_cpoBTInterface->vSetSelectedDevBTAddress(poszOnCarBTAddress);
      poExtCmdBluetoothIntf = m_cpoBTInterface->poFetchExtCmdBluetoothIntf();
   }//if (m_cpoBTInterface)

   //! Send initial response as not ready (response will be sent later based on success of pairing initiation)
   //! @Note: Pairing state of device is not relevant when sending delayed status.
   
   vSetPairingStatus(e8DEVICE_PAIRING_REQUESTED);   
   if((poExtCmdBluetoothIntf))
   {
       ETG_TRACE_USR2(("BT service is available"));
       sbBTLimModeRequested = poExtCmdBluetoothIntf->bSetBTLimitationMode(
            poszOnCarBTAddress,
            szBTDeviceName,
            e8BT_TECH_ONCAR,
            e8BT_COMM_CHN_USB,
            e8BT_LIM_ACTION_ACTIVATE);

       if(NULL != m_cpoBTInterface) 
       {
          if(true == sbBTLimModeRequested)
          {
             m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_COMPLETE);
          }
          else
          {
             m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_DELAYED);
             m_enPendingBTLimAction = e8BT_LIM_ACTION_ACTIVATE;
          }
       }
   }//if (poExtCmdBluetoothIntf)
   else
   {
       ETG_TRACE_USR2(("vPostBTPairingRequest: BT service unavailable. Will request BT Lim mode later"));
       if(m_cpoBTInterface)
       {
           m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_DELAYED);
       }
       vSendDevicePairingResponse(false, e8_ONCAR_BTPAIRING_DELAYED_PAIRING);
   }
   if ((false == bIsBTAddrAlreadyReceived) && (m_cpoBTInterface))
   {
      //! Notify OnCar BT device info
      t_String szBTDeviceName = (NULL != poExtCmdBluetoothIntf) ?
            (poExtCmdBluetoothIntf->szGetBTDeviceName(poszOnCarBTAddress)) : ("");

      m_cpoBTInterface->vSendBTDeviceInfo(
            m_cpoBTInterface->u32GetSelectedDevHandle(), szBTDeviceName, poszOnCarBTAddress);
   }//if ((false == bIsBTAddrAlreadyReceived) && (m_cpoBTInterface))
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarBluetooth::vPostBTPairingRequest() left "));
} //!end of vPostBTPairingRequest()

/***************************************************************************
*********************************PRIVATE************************************
***************************************************************************/

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarBluetooth::enGetPairingStatus()
***************************************************************************/
tenDevicePairingStatus spi_tclOnCarBluetooth::enGetPairingStatus()
{
    ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarBluetooth::enGetPairingStatus() entered "));
   m_oPairingStateLock.s16Lock();
   tenDevicePairingStatus enPairingStatus = m_enPairingStatus;
   m_oPairingStateLock.vUnlock();
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarBluetooth::enGetPairingStatus() left with : %d",
          ETG_ENUM(SPI_BT_PAIRING_STATE, enPairingStatus)));
   return enPairingStatus;
} //!end of enGetPairingStatus()

/***************************************************************************
** FUNCTION:  t_Void spi_tclOnCarBluetooth::vSetPairingStatus()
***************************************************************************/
t_Void spi_tclOnCarBluetooth::vSetPairingStatus(tenDevicePairingStatus enPairingStatus)
{
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarBluetooth::vSetPairingStatus() entered: %d ",
         ETG_ENUM(SPI_BT_PAIRING_STATE, enPairingStatus)));

   m_oPairingStateLock.s16Lock();
   m_enPairingStatus = enPairingStatus;
   m_oPairingStateLock.vUnlock();
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarBluetooth::vSetPairingStatus() left "));
} //!end of vSetPairingStatus()

/***************************************************************************
** FUNCTION:   t_Void spi_tclOnCarBluetooth::vDeselectOnCarDevice()
***************************************************************************/
t_Void spi_tclOnCarBluetooth::vDeselectOnCarDevice()
{
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarBluetooth::vDeselectOnCarDevice() entered "));
   if (m_cpoBTInterface)
   {
      //Clear device status
      m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_COMPLETE);
      m_cpoBTInterface->vSendDeselectDeviceRequest();
   }//if (m_cpoBTInterface)
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarBluetooth::vDeselectOnCarDevice() left "));
} //!end of vDeselectOnCarDevice()

/***************************************************************************
** FUNCTION:   t_Void spi_tclOnCarBluetooth::vSendDevicePairingResponse()
***************************************************************************/
t_Void spi_tclOnCarBluetooth::vSendDevicePairingResponse(t_Bool bReadyToPair,
                                          tenOnCarBTPairingState enOnCarBTPairingState)
{
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarBluetooth::vSendDevicePairingResponse() entered : ReadyToPair = %d",bReadyToPair));
   if (m_cpoBTInterface)
   {
      spi_tclExtCmdBluetoothIntf *poExtCmdBluetoothIntf = m_cpoBTInterface->poFetchExtCmdBluetoothIntf();
      t_String szOnCarBTAddress = m_cpoBTInterface->szGetSelectedDevBTAddress();

      spi_tclOnCarManager* poOnCarManager = spi_tclOnCarManager::getInstance();
      spi_tclOnCarCmdBluetooth* poCmdBluetooth = (NULL != poOnCarManager)?
            (poOnCarManager->poGetBluetoothInstance()) : (NULL);

      //! Send BTDevicePairingResponse to device
      if ((true == IS_VALID_BT_ADDRESS(szOnCarBTAddress)) &&
            (poExtCmdBluetoothIntf) && (poCmdBluetooth))
      {
         //@Note: If HU is not ready to pair, send OnCar device paired info as unpaired.
         t_Bool bOnCarDevicePaired = (true == bReadyToPair) ?
               (poExtCmdBluetoothIntf->bGetPairingStatus(szOnCarBTAddress)) :
               (false);
         poCmdBluetooth->vSendBTPairingResponse(bReadyToPair, bOnCarDevicePaired, enOnCarBTPairingState);
         ETG_TRACE_USR2(("[DESC]spi_tclOnCarBluetooth::vSendDevicePairingResponse: Sending BTPairingResponse "));
      }//if ((true == IS_VALID_BT_ADDRESS(szOnCarBTAddress)) && (poExtCmdBluetoothIntf) && (poCmdBluetooth))
   }//if (m_cpoBTInterface)
   ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarBluetooth::vSendDevicePairingResponse() left" ));
} //!end of vSendDevicePairingResponse()

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarBluetooth::vPostBTPairingInfoMsg()
 ***************************************************************************/
t_Void spi_tclOnCarBluetooth::vPostBTPairingInfoMsg(trBTPairingRequestInfo rBTPairingReqInfo)
{
    ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarBluetooth::vPostBTPairingInfoMsg() entered: BTAddress = %s ",
            rBTPairingReqInfo.szRemoteDevBTAddr.c_str()));
    ETG_TRACE_USR4(("[PARAM]:vPostBTPairingInfoMsg: PairingMethod = %d, PairingPin = %s ",
          ETG_ENUM(BTSET_PAIRING_METHOD, rBTPairingReqInfo.enPairingMethod),
          rBTPairingReqInfo.szPairingPin.c_str()));
    if ( (m_cpoBTInterface)
         &&
         (e8DEV_TYPE_ONCAR == m_cpoBTInterface->enGetSelectedDeviceCategory()) /*If OnCar device is active*/
       )
    {
        //! If pairing is started for OnCar device, accept the pairing and send PIN number to MD.
        if ((rBTPairingReqInfo.szRemoteDevBTAddr == m_cpoBTInterface->szGetSelectedDevBTAddress())
                &&
            (scenBTSetPreferredOnCarPairingMethod == rBTPairingReqInfo.enPairingMethod)
           )
        {
            ETG_TRACE_USR2(("[DESC]:spi_tclOnCarBluetooth::vPostBTPairingInfoMsg : Pairing request from OnCar device received. "));

            spi_tclOnCarManager* poOnCarManager = spi_tclOnCarManager::getInstance();
            spi_tclOnCarCmdBluetooth* poCmdBluetooth = (NULL != poOnCarManager)?
                    (poOnCarManager->poGetBluetoothInstance()) : (NULL);
            if (poCmdBluetooth)
            {
                poCmdBluetooth->vSendBTPairingPIN(rBTPairingReqInfo.szPairingPin);
            }//if (poCmdBluetooth)
        }//if ((rBTPairingReqInfo.szRemoteDevBTAddr == m_cpoBTInterface->szGetSelectedDevBTAddress()) && ...)
    }//if ((m_cpoBTInterface) && (e8DEV_TYPE_ONCAR == m_cpoBTInterface->enGetSelectedDeviceCategory()))
    ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarBluetooth::vPostBTPairingInfoMsg() left "));
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarBluetooth::vPostBTLimitationModeMsg()
 ***************************************************************************/
t_Void spi_tclOnCarBluetooth::vPostBTLimitationModeMsg(trBTLimitationModeInfo rBTLimitationModeInfo)
{
    ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarBluetooth::vPostBTLimitationModeMsg() entered "));
    ETG_TRACE_USR4(("[PARAM]::vPostBTLimitationModeMsg: "
          "Technology = %d, CommChannel = %d, ActionState = %d, BTAddress = %s ",
          ETG_ENUM(BTSET_TECHNOLOGY, rBTLimitationModeInfo.enTechnology),
          ETG_ENUM(BTSET_COMM_CHN, rBTLimitationModeInfo.enCommChannel),
          ETG_ENUM(BTSET_LIMIT_ACTION_STATE, rBTLimitationModeInfo.enActionState),
          rBTLimitationModeInfo.szBTAddress.c_str()));

    if (
       (m_cpoBTInterface)
       &&
       (e8DEV_TYPE_ONCAR == m_cpoBTInterface->enGetSelectedDeviceCategory()) /*If OnCar device is active*/
       &&
       (e8BT_TECH_ONCAR == rBTLimitationModeInfo.enTechnology)
       &&
       (e8BT_COMM_CHN_USB == rBTLimitationModeInfo.enCommChannel)
       &&
       (rBTLimitationModeInfo.szBTAddress == m_cpoBTInterface->szGetSelectedDevBTAddress())
       )
    {
        switch (rBTLimitationModeInfo.enActionState)
        {
        case e8BT_LIM_ACTION_STATE_ACTIVE:
        {
            ETG_TRACE_USR2(("[DESC]::vPostBTLimitationModeMsg: OnCar BT mode activation is successful. "));
            if (e8DEVICE_PAIRING_REQUESTED == enGetPairingStatus())
            {
                ETG_TRACE_USR2(("[DESC]::vPostBTLimitationModeMsg: Sending pairing response to MD. "));
                vSetPairingStatus(e8DEVICE_PAIRING_RESPONSE_SENT);
                vSendDevicePairingResponse(true, e8_ONCAR_BTPAIRING_ALREADY_PAIRED);
            }//if (e8DEVICE_PAIRING_REQUESTED == enGetPairingStatus())
        }
        break;
        case e8BT_LIM_ACTION_STATE_ERROR:
        {
            //! TODO - Deselection is currently disabled. To be enabled again after BT limitation
            //! mode interface is extended to get more information on the error.
            //ETG_TRACE_USR2(("[DESC]::vOnBTLimitationMode: "
            //"OnCar BT mode activation is unsuccessful. Deactivating OnCar device. "));
            //vDeselectOnCarDevice();
        }
        break;
        case e8BT_LIM_ACTION_STATE_ERROR_USER_DENIED:
        {
             ETG_TRACE_USR2(("[DESC]::vPostBTLimitationModeMsg: Sending BT pairing delayed msg to MD. "));
             vSendDevicePairingResponse(false, e8_ONCAR_BTPAIRING_DELAYED_PAIRING);
        }
        break;
        default:
            //! Add code
            break;
        }//switch (rBTLimitationMode.enActionState)
    }//if ((m_cpoBTInterface)&&(e8DEV_TYPE_ONCAR == m_cpoBTInterface->enGetSelectedDeviceCategory()) &&...)
       ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarBluetooth::vPostBTLimitationModeMsg() left "));
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclOnCarBluetooth::vPostBTDeviceNameUpdateMsg()
 ***************************************************************************/
t_Void spi_tclOnCarBluetooth::vPostBTDeviceNameUpdateMsg(t_String szBTDeviceAddress,t_String szBTDeviceName)
{
    ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarBluetooth::vPostBTDeviceNameUpdateMsg() entered: BTDeviceAddress = %s",
            szBTDeviceAddress.c_str()));
    ETG_TRACE_USR1(("[PARAM]:spi_tclOnCarBluetooth::vPostBTDeviceNameUpdateMsg() entered: BTDeviceName = %s",
            szBTDeviceName.c_str()));

    if ((false == szBTDeviceName.empty()) && (m_cpoBTInterface) &&
        (e8DEV_TYPE_ONCAR == m_cpoBTInterface->enGetSelectedDeviceCategory()))
    {
        t_String szSelDevBTAddr = m_cpoBTInterface->szGetSelectedDevBTAddress();

        if ((IS_VALID_BT_ADDRESS(szBTDeviceAddress))
            &&
            (IS_VALID_BT_ADDRESS(szSelDevBTAddr)) /*If BT Address of active projection device is available*/
            &&
            (szBTDeviceAddress == szSelDevBTAddr) /*If BT device is active projection device*/
            )
        {
            t_U32 u32SelProjDevHandle = m_cpoBTInterface->u32GetSelectedDevHandle();
            m_cpoBTInterface->vSendBTDeviceInfo(u32SelProjDevHandle, szBTDeviceName, szSelDevBTAddr);
        }//if ((IS_VALID_BT_ADDRESS(szBTDeviceAddress))&&(IS_VALID_BT_ADDRESS(szSelDevBTAddr))&&(szBTDeviceAddress == szSelDevBTAddr))
    }//if ((false == szBTDeviceName.empty()) && (m_cpoBTInterface) && ....)
    ETG_TRACE_USR1(("[FUNC]:spi_tclOnCarBluetooth::vPostBTDeviceNameUpdateMsg() left"));
}

 /***************************************************************************
  ** FUNCTION:  t_Void spi_tclOnCarBluetooth::vPostBTServiceAvailabilityMsg()
  ***************************************************************************/
 t_Void spi_tclOnCarBluetooth::vPostBTServiceAvailabilityMsg(t_Bool bServiceAvailable)
 {
    ETG_TRACE_USR1(("spi_tclOnCarBluetooth::vPostBTServiceAvailabilityMsg entered bServiceAvailable = %d",ETG_ENUM(BOOL,bServiceAvailable)));
    //! szBTDeviceName is an optional parameter and should be used when BT address is not available.
    //! In cases (name not known, BT address is available) this parameter should be an empty string
    t_String szBTDeviceName;

    if( (true == bServiceAvailable) &&
        (NULL != m_cpoBTInterface) &&
        (e8DEVICE_CHANGE_DELAYED == m_cpoBTInterface->enGetSelectedDevStatus()) &&
        (e8DEV_TYPE_ONCAR == m_cpoBTInterface->enGetSelectedDeviceCategory()) &&
        (e8BT_LIM_ACTION_UNKNOWN != m_enPendingBTLimAction))
    {
        spi_tclExtCmdBluetoothIntf *poExtCmdBluetoothIntf = m_cpoBTInterface->poFetchExtCmdBluetoothIntf();
        if(NULL != poExtCmdBluetoothIntf)
        {
            sbBTLimModeRequested = poExtCmdBluetoothIntf->bSetBTLimitationMode(
            m_cpoBTInterface->szGetSelectedDevBTAddress(), 
            szBTDeviceName,
            e8BT_TECH_ONCAR,
            e8BT_COMM_CHN_USB, 
            m_enPendingBTLimAction);
            if(true == sbBTLimModeRequested)
            {
               m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_COMPLETE);
            }
            else
            {
               m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_DELAYED);
            }
        }
    }
}
// <EOF>
