 /**
 ***********************************************************************************
 * \file              OnCarTypes.h
 * \brief             File describing different types for each OnCar related classes.
 ************************************************************************************/

#ifndef ONCAR_TYPES_HPP
#define ONCAR_TYPES_HPP 
 
#include <stdint.h>
#include <string>
#include <vector>
#include <memory>
using namespace std;

/*********************************ONCAROBJECTFACTORY RELATED TYPES************************************************/
/**
 * Enum defining ID's for all objects for OnCar Object Factory
 */
enum ONCAR_OBJECTFACTORY
{
    //! ID for OnCarAPI class
    eONCAR_OBJECTFACTORY_ONCARAPI           = 0,
    //! ID for OnCarAudioSinkEndpoint class
    eONCAR_OBJECTFACTORY_AUDIO_ENDPOINT     = 1,
    //! ID for OnCarVideoEndpoint class
    eONCAR_OBJECTFACTORY_VIDEO_ENDPOINT     = 2,
    //! ID for OnCarBluetoothEndpoint class
    eONCAR_OBJECTFACTORY_BLUETOOTH_ENDPOINT = 3,
    //! ID for OnCarMetadataEndpoint class
    eONCAR_OBJECTFACTORY_METADATA_ENDPOINT  = 4,
    //! ID for OnCarInputEndpoint class
    eONCAR_OBJECTFACTORY_INPUT_ENDPOINT     = 5,
    //! ID for OnCarSensorEndpoint class
    eONCAR_OBJECTFACTORY_SENSOR_ENDPOINT    = 6
};

/*********************************ONCARENDPOINTBASE RELATED TYPES************************************************/
/**
 *
 * Enum defining all the ID's for the endpoint classes which will be 
 * registered with OnCarAPI object.
 */
enum ONCARENDPOINTID
{
   eONCARENDPOINTID_AUDIO         = 10,
   eONCARENDPOINTID_VIDEO         = 20,
   eONCARENDPOINTID_BLUETOOTH     = 30,
   eONCARENDPOINTID_METADATA      = 40,
   eONCARENDPOINTID_INPUT         = 50,
   eONCARENDPOINTID_SENSOR        = 60
};

 /*********************************SESSION AND TRANSPORT RELATED TYPES************************************************/
 
 /**
  *Structure holding information about AOAP device.
 */
struct AoapAccessory
{
   //! Serial number of the device
   string szSerialNumber;
};

/**
 * Enum defining possible errors.
 */
enum ONCARSINKERROR
{
    //! No error occurred
    eONCARSINK_NO_ERROR                        = 0, 
    /** Errors in case if the order of method invocation doesnot match the specified sequence.
    *   Eg :createConnection() called before calling init() method.
    */
    eONCARSINK_INTERNAL_ERROR                  = 1, 
    /** Errors in case where some messages received from phone were 
    *   missed or not processed by oncar library.
    */
    eONCARSINK_PROTOCOL_ERROR                  = 2,
    //! Errors in case when usb device connected doesnot exist on USB bus.
    eONCARSINK_ERROR_NO_DEVICE                 = 3,
    /** Errors in case when time out has occured for any operation.
    *   Eg:Not acknowledging a pop up on phone or HU.
    */
    eONCARSINK_ERROR_TIMEOUT                   = 4,
   /** Errors in case while performing anz IO operation.
    *   Eg:Error during USB endpoint IO operation.
    */
    eONCARSINK_IO_ERROR                        = 5,
   /** Errors in case where oncar app is not present on the MD.
    */
    eONCARSINK_APP_NOT_INSTALLED_ERROR         = 6,
   /** Errors in case oncar is not supported on the MD.
    */
    eONCARSINK_PHONE_UNSUPPORTED_ERROR         = 7,
   /** Errors in case oncar authentication failed.
    */
    eONCARSINK_AUTHENTICATION_FAILED_ERROR     = 8,     
    //! Audio Sink config error
    eONCARSINK_AUDIO_SINK_CONFIGURATION_ERROR  = 101,
    //! Audio Sink start error
    eONCARSINK_AUDIO_SINK_START_ERROR          = 102,
    //! Audio sink write error
    eONCARSINK_AUDIO_SINK_WRITE_ERROR          = 103,
    //! Audio Source read error
    eONCARSINK_AUDIO_SOURCE_READ_ERROR         = 104,
    //! Video Sink config error
    eONCARSINK_VIDEO_SINK_CONFIGURATION_ERROR  = 201,
    //! Video sink start error
    eONCARSINK_VIDEO_SINK_START_ERROR          = 202,
    //! Video sink stop error
    eONCARSINK_VIDEO_SINK_STOP_ERROR           = 203,
    //! Video Sink render error
    eONCARSINK_VIDEO_SINK_RENDER_ERROR         = 204,
    
    //! Errors in case when BT pairing could not completed.Eg :Incorrect BT pairing pin.
    eONCARSINK_BT_PAIRING_ERROR                = 301,
    //! Errors in case when BT connection could not established between HU and MD.
    eONCARSINK_BT_CONNECTION_ERROR             = 302,
    
    //! Input config error
    eONCARSINK_INPUT_TOUCH_INIT_ERROR          = 401,
    //! Errors in case of Touch Inputs Not Processed.
    eONCARSINK_INPUT_EVENT_DISPATCH_ERROR      = 402

};

/**
 *This enum represents app states.
 */
enum ONCARSINK_APP_STATE
{
    //! When during OnCar session application state (Navigation,speech etc.) is projected.
    eONCARSINK_APP_STATE_PROJECTION = 0,
    //! When during OnCar session application state (Navigation,speech etc.) is native .
    eONCARSINK_APP_STATE_NATIVE     = 1,
    //! When during OnCar session application state (Navigation,speech etc.) is unknown.
    eONCARSINK_APP_STATE_UNKNOWN    = 2
};  

/**
 *Structure holding initial setup configuration for OnCarAPI. 
 */
struct OnCarConfig 
{
    //! vendor name
    string HUvendor;
    //! vendor code
    uint8_t HUvendor_code;
    //! manufacturer name
    string HUmanufacturer;
    //! Physical screen width in pixels
    uint32_t width;
    //! Physical screen height in pixels
    uint32_t height;
    //! Vehicle code
    uint16_t VehicleCode;
    //! Vehicle region code
    uint8_t RegionCode;
};

/**
 *This enum represents OnCar Session states
 */
enum ONCARSESSION_STATE
{ 
    //! OnCar session started.
    eONCARSESSION_STATE_START = 0,
    //! OnCar session stoped.
    eONCARSESSION_STATE_STOP = 1,
    //! OnCar user interaction needed.
    eONCARSESSION_USER_INTERACTION_NEEDED = 2,
    //! OnCar user interaction not needed.
    eONCARSESSION_USER_INTERACTION_NOT_NEEDED = 3
};

/**
 *This enum represents vehicle movement state
 */
enum ORIENTATION_MODE
{ 
    //! Only Landscape mode is allowed
    eORIENTATION_MODE_LANDSCAPE  = 0,
    //! Only Portrait mode is allowed
    eORIENTATION_MODE_PORTRAIT   = 1,
    //! Both Landscape and Portrait are allowed
    eORIENTATION_MODE_BOTH       = 2
};

/**
 * This enum represents navigation focus types
 */
enum NAVIGATION_FOCUS
{
    //! MD has the navigation focus. Also means that RG is active in MD.
    eNAVIGATION_FOCUS_PROJECTED = 1,
    //! HU has navigation focus.
    eNAVIGATION_FOCUS_NATIVE = 2
};

/*********************************AUDIO SINK TYPES************************************************/

/**
 *This enum represents different audio streams.
 */
enum AUDIOSTREAM_TYPE
{
   //! Audio type media
   eAUDIOSTREAM_TYPE_MEDIA      = 0,
   //! Audio type guidance
   eAUDIOSTREAM_TYPE_GUIDANCE   = 1
};

/**
 *This enum represents different audio focus request types.
 */
enum AUDIOFOCUS_REQUEST
{
    //! When MD wants to request for audio channel
    eAUDIOFOCUS_REQUEST_GAIN      = 0,
    //! When MD wants to release audio channel
    eAUDIOFOCUS_REQUEST_RELEASE   = 1   
};

/**
 *This enum represents different audio focus state.
 */
enum AUDIOFOCUS_STATE
{
    //! Audio channel is with MD and it can stream over it.
    eAUDIOFOCUS_STATE_GAIN    = 0,
    //! Audio channel is not with MD and it cannot stream over it.
    eAUDIOFOCUS_STATE_LOSS    = 1,
    //! Audio channel is with HU to stream transient source
    eAUDIOFOCUS_STATE_LOSS_TRANSIENT    = 2,
    //! Mute is enabled and hence media audio cannot be streamed
    eAUDIOFOCUS_STATE_MUTE    = 3,
    //! Mute is disabled and media audio can be resumed/played
    eAUDIOFOCUS_STATE_DEMUTE    = 4
};

/**
 *This structure represents audio configurations parameters that needs to be set 
 *before starting OnCar session.
*/
struct AudioConfig
{
    //! Indicates the priority with which audio threads must run.
    uint16_t threadPriority;
    //! Can be one of eAUDIOSTREAM_TYPE_GUIDANCE,eAUDIOSTREAM_TYPE_MEDIA
    AUDIOSTREAM_TYPE type;
    //! Gstreamer Audio Pipeline.
    string audioPipeline;
};

/*********************************VIDEO SINK TYPES************************************************/

/**
 * Enum defining possible video focus.
 */
enum VIDEOFOCUS
{
   //! Video Focus unknown
   eVIDEOFOCUS_UNKNOWN = 0,
   //! When OnCar wants to display Video
   eVIDEOFOCUS_PROJECTED = 1,
   //! When HU should display video
   eVIDEOFOCUS_NATIVE = 2
};

/**
 *This enum represents the reason for video focus request callback.
 */
enum VIDEOFOCUS_REASON
{
    //! When launcher app is in background
    eVIDEOFOCUS_REASON_BACKGROUND           = 0,
    //! When launcher app is in foreground  
    eVIDEOFOCUS_REASON_FOREGROUND           = 1,
    //! When user launches some other app from OnCar launcher app   
    eVIDEOFOCUS_REASON_TRANSITION_START     = 2,
    //! When some other app started from OnCar launcher app has finished loading and now is rendering on phone
    eVIDEOFOCUS_REASON_TRANSITION_END       = 3,
    //! When Mobile device's screen is locked.
    eVIDEOFOCUS_REASON_SCREEN_OFF           = 4,
    //! When Mobile device's screen is unlocked.        
    eVIDEOFOCUS_REASON_SCREEN_ON            = 5,
    //! When OEM icon is pressed by user on OnCar Projection screen.    
    eVIDEOFOCUS_REASON_OEM_ICON_PRESSED      = 6,
    //! When Native BT screen needs to be shown to the user.
    eVIDEOFOCUS_REASON_SHOW_NATIVE_BT_SCREEN = 7
};

/**
 *This structure represents display configurations parameters that needs to be set 
 *before starting OnCar session.
 */
struct VideoConfig
{
    //! Frame rate per second of video. Either 60 or 30
    uint16_t fps;
    //! The width of projection screen in pixels
    uint32_t projectionScreenWidth;
    //! The height of projection screen in pixels
    uint32_t projectionScreenHeight;
    //! The density in dpi(dots per inch)
    uint16_t density;
    //! The ratio of the physical pixel width over height.
    uint16_t pixelAspectRatio;
    //! Gstreamer Video Pipeline.
    string videoPipeline;
        
};

/*********************************BLUETOOTH SINK TYPES************************************************/

/**
 *This enum represents different BT pairing types.
 */
enum BTPAIRING_METHOD
{
   //! BT PAIRING METHOD OOB
   eBTPAIRING_METHOD_OOB                = 1,
   //! BT PAIRING METHOD NUMERIC COMAPRISION
   eBTPAIRING_METHOD_NUMERIC_COMPARISON = 2,
   //! BT PAIRING METHOD PASS KEY ENTRY
   eBTPAIRING_METHOD_PASSKEY_ENTRY      = 3,
   //! BT PAIRING METHOD PAIRING PIN
   eBTPAIRING_METHOD_PIN                = 4
};

/**
 *This enum represents different BT pairing types.
 */
enum BTPAIRING_STATE
{
   //! BT PAIRING STATE NOT PAIRED
   eBTPAIRING_STATE_NOT_PAIRED       = 1,
   //! BT PAIRING STATE ALREADY PAIRED
   eBTPAIRING_STATE_ALREADY_PAIRED   = 2,
   //! BT PAIRING STATE DELAYED Note: Retained for future use
   eBTPAIRING_STATE_DELAYED_PAIRING  = 3
};

/*********************************METADATA SINK TYPES************************************************/

/**
 * This enumeration identifies State of the current playing song
 */
enum MEDIA_PLAYBACK_STATE
{
   //! Current track is stopped playing
   eMEDIA_PLAYBACK_STATE_STOPPED      = 0,
   //! Current track is in playing state
   eMEDIA_PLAYBACK_STATE_PLAYING      = 1,
   //! Current track is in paused state
   eMEDIA_PLAYBACK_STATE_PAUSED       = 2,
   //! Identifies Default value(Startup case) or status is not known
   eMEDIA_PLAYBACK_STATE_NOT_KNOWN    = 0xFF
};

/**
 *This structure represents media Metadata that OnCar can send to SPI regarding currently playing song.
 */
 struct OnCarMediaMetadata
 {
    //! Name of current song
    string song;
    //! Name of album
    string album;
    //! Name of artist
    string artist;
    //! Album art data in png format
    vector<uint8_t> album_art;
    //! Name of Playlist
    string playlist;
    //! Song length in seconds
    uint32_t duration_seconds;
    //! Rating of current song
    uint32_t rating;
};

/**
 *This structure represents media Metadata that OnCar can send to SPI regarding playback status.
 */
struct OnCarMediaPlaybackStatus
{
    //! Name of application playing the music on phone
    string media_source;
    //! State of current song
    MEDIA_PLAYBACK_STATE state;
    //! Current playback of song in seconds
    uint32_t playback_seconds;
    //! True if shuffle is on
    bool shuffle;
    //! True if repeat all
    bool repeat_all;
    //! True if repeating one song
    bool repeat_one;
};

/**
 * This enumeration identifies State of the phone call
 * \note
 *  This enum is kept for future use. It is not used now.
 */
enum PHONE_CALL_STATE
{
    //! Incoming call
    ePHONE_CALL_STATE_INCOMING = 0, 
    //! Outgoing call
    ePHONE_CALL_STATE_OUTGOING = 1  
};

/**
 * This enumeration identifies type of number
 * \note
 *  This enum is kept for future use. It is not used now.
 */
enum PHONE_NUMBER_TYPE
{
    //! Number type is mobile
    ePHONE_NUMBER_TYPE_MOBILE  = 0,  
    //! Number type is home
    ePHONE_NUMBER_TYPE_HOME    = 1, 
    //! Number type is work
    ePHONE_NUMBER_TYPE_WORK    = 2, 
    //! Number type is other
    ePHONE_NUMBER_TYPE_OTHER   = 3,
    //! Number type is private
    ePHONE_NUMBER_TYPE_PRIVATE = 4
};

/**
 * This enumeration identifies signal strength of phone
 * \note
 *  This enum is kept for future use. It is not used now.
 */
enum PHONE_SIGNAL_STRENGTH
{
    //! very poor quality signal
    ePHONE_SIGNAL_STRENGTH_VERYLOW   = 0,
    //! poor quality signal
    ePHONE_SIGNAL_STRENGTH_LOW       = 1, 
    //! medium quality signal   
    ePHONE_SIGNAL_STRENGTH_MEDIUM    = 2,
    //! good quality signal
    ePHONE_SIGNAL_STRENGTH_GOOD      = 3,
    //! excellent quality signal    
    ePHONE_SIGNAL_STRENGTH_EXCELLENT = 4 
};

/**
 * This enumeration identifies Battery strength of phone
 * \note
 *  This enum is kept for future use. It is not used now.
 */
enum PHONE_BATTERY_STRENGTH
{
    //! very low battery strength
    ePHONE_BATTERY_STRENGTH_VERYLOW   = 0,
    //! poor battery strength
    ePHONE_BATTERY_STRENGTH_LOW       = 1,
    //! medium battery strength 
    ePHONE_BATTERY_STRENGTH_MEDIUM    = 2,
    //! good battery strength   
    ePHONE_BATTERY_STRENGTH_GOOD      = 3,
    //! excellent battery strength
    ePHONE_BATTERY_STRENGTH_EXCELLENT = 4
};

/**
 *This structure represents PhoneCallMetaData that OnCar can send to SPI regarding current ongoing phone call.
 * \note
 *  This structure is kept for future use. It is not used now.
 */
struct PhoneCallMetaData
{
    //! State of phone call
    PHONE_CALL_STATE state;
    //! caller number
    string caller_number;
    //! caller ID
    string caller_id;
    //! Caller number type i.e. work, home, other etc.
    PHONE_NUMBER_TYPE caller_number_type;
    //! Call time elapsed in seconds
    uint32_t call_duration_seconds;
    //! Image of caller in png format
    vector<uint8_t> caller_thumbnail;
};

/**
 *This structure represents PhoneStatusMetaData that OnCar can send to SPI regarding current state of phone.
 * \note
 *  This structure is kept for future use. It is not used now.
 */
struct PhoneStatusMetaData
{
    //! Signal state of phone
    PHONE_SIGNAL_STRENGTH signal_strength;
    //! Battery state of phone
    PHONE_BATTERY_STRENGTH battery_strength;
    //! Name of network operator
    string network_operator_name;
};

/*********************************INPUT SINK TYPES************************************************/

/**
 * Enum defining touch screen type.
 */
enum TOUCH_SCREEN_TYPE
{
  //! Capacitive Touch Screen
  eTOUCH_SCREEN_TYPE_CAPACITIVE = 1,
  //! Resistive Touch Screen
  eTOUCH_SCREEN_TYPE_RESISTIVE  = 2
};

/**
 * Enum defining touch event type.
 */
enum TOUCH_EVENT_TYPE
{
    //! Unknown Touch Event
    eTOUCH_EVENT_TYPE_UKNOWN       = 0,
    //! Single Touch Event
    eTOUCH_EVENT_TYPE_SINGLE_TOUCH = 1,
    //! Multi Touch Event
    eTOUCH_EVENT_TYPE_MULTI_TOUCH  = 2
};

/**
 * Enum defining touch event type.
 */
enum INPUT_KEYCODE
{
  //!Indicates selected key is invalid.
  eINPUT_KEYCODE_INVALID           = 1,
  //!Handled by the application. 
  //!SendingKEYCODE_BACK SHOULD NOT cause OnCar to lose video focus.
  eINPUT_KEYCODE_BACK              = 2,
  //!Indicates selected media SHOULD return to next track or station.
  eINPUT_KEYCODE_MEDIA_NEXT        = 6,
  //!Indicates selected media SHOULD return to previous track or station.
  eINPUT_KEYCODE_MEDIA_PREVIOUS    = 7,
  //!Indicates selected media SHOULD move forward 
  eINPUT_KEYCODE_MEDIA_FORWARD     = 8,
  //!Indicates selected media SHOULD move back
  eINPUT_KEYCODE_MEDIA_REWIND      = 9,
  //!Indicates selected media SHOULD stop.
  eINPUT_KEYCODE_MEDIA_STOP        = 10,
  //!Indicates selected media SHOULD play.
  eINPUT_KEYCODE_MEDIA_PLAY        = 11,
  //!Indicates selected media SHOULD pause.
  eINPUT_KEYCODE_MEDIA_PAUSE       = 12,
  //!Indicates Rotary controller events are sent using relativeEvent api calls.
  //!The delta value SHOULD be positive for clockwise rotation and SHOULD be negative
  //!for counterclockwise rotation.
  eINPUT_KEYCODE_ROTARY_CONTROLLER = 13,
  //! Causes projected UI to switch to media screen.
  eINPUT_KEYCODE_MEDIA             = 14,
  //! Causes projected UI to switch to navigation screen.
  eINPUT_KEYCODE_NAVIGATION        = 15,
  //! Launches MD's VR (eg. Google Voice)
  eINPUT_KEYCODE_PTT               = 16,
  //!Indicates Accept Phone call
  eINPUT_KEYCODE_PHONE_CALL_ANSWER = 17,
  //!Indicates Reject Phone call
  eINPUT_KEYCODE_PHONE_CALL_END    = 18,
  //! Indicates Joystick UP
  eINPUT_KEYCODE_SHIFT_UP          = 19,
  //! Indicates Joystick DOWN
  eINPUT_KEYCODE_SHIFT_DOWN        = 20,
  //! Indicates Joystick LEFT
  eINPUT_KEYCODE_SHIFT_LEFT        = 21,
  //! Indicates Joystick RIGHT
  eINPUT_KEYCODE_SHIFT_RIGHT       = 22,
  //! Indicates HOME screen
  eINPUT_KEYCODE_HOME              = 23,
  //! Indicates OK/ENTER
  eINPUT_KEYCODE_ENTER             = 24
};

/**
 *This structure represents input configurations parameters that needs to be set 
 *before starting OnCar session.
 */
struct InputConfig
{
    //! Indicates the Touch Layer that needs to be used for OnCar.
    int TouchLayerID;
    //! Indicates the Touch Surface that needs to be used for OnCar.
    int TouchSurfaceID;
    //! Width of the Touch screen
    int DisplayWidth;
    //! Height of Touch screen
    int DisplayHeight;
    //! Projection screen width
    int TouchWidth;
    //! Projection screen height
    int TouchHeight;
    //! Touch Screen type
    TOUCH_SCREEN_TYPE screenType;
    //! Touch Event type
    TOUCH_EVENT_TYPE eventType; 
};

/*********************************SENSOR RELATED TYPES************************************************/

/**
 *This enum represents DayNightMode
 */
enum DAYNIGHT_MODE
{ 
    //! Day mode
    eDAYNIGHT_MODE_DAY   = 0,
    //! Night mode
    eDAYNIGHT_MODE_NIGHT = 1
};

/**
 *This enum represents vehicle movement state
 */
enum DRIVING_STATE
{ 
    //! Vehicle is parked
    eDRIVING_STATE_PARKED   = 0,
    //! Vehicle is moving
    eDRIVING_STATE_MOVING   = 1
};

/**
 * Enum defining Gear Positions
 */
enum ONCAR_GEARINFO
{
    //! Gear Neutral
    eONCAR_GEAR_NEUTRAL   = 0,
    //! Gear 1
    eONCAR_GEAR_1         = 1,
    //! Gear 2
    eONCAR_GEAR_2         = 2,
    //! Gear 3
    eONCAR_GEAR_3         = 3,
    //! Gear 4
    eONCAR_GEAR_4         = 4,
    //! Gear 5
    eONCAR_GEAR_5         = 5,
    //! Gear 6
    eONCAR_GEAR_6         = 6,
    //! Gear Drive 
    eONCAR_GEAR_DRIVE     = 7,
    //! Gear Park
    eONCAR_GEAR_PARK      = 8,
    //! Gear Reverse
    eONCAR_GEAR_REVERSE   = 9
}; 

#endif //ONCAR_TYPES_HPP