/**
 *******************************************************************************
 * \file              OnCarInputEndpoint.h
 * \brief             This sink handles Touch and key Input between HU and OnCar.
 *******************************************************************************/

#ifndef ONCAR_INPUT_ENDPOINT_HPP
#define ONCAR_INPUT_ENDPOINT_HPP

#include "OnCarTypes.h"
#include "OnCarEndPointBase.h"
 
/**
 * This class represents an interface that every sink implementation 
 * must subclass if they wish to implement a OnCarInputEndpoint.  
 */
class IOnCarInputSourceCallbacks
{
public:
    virtual ~IOnCarInputSourceCallbacks() {}

   /**
	* @brief Called when an error happens in the underlying layer.
    * The callback is introduced to inform the upper layer about error cases,
    * for which another notification, such as return values, is not possible.
    * @param error Error code
    * */
    virtual void notifyErrorCallback(ONCARSINKERROR error) = 0;
};

 /**
 * OnCarInputEndpoint: This sink handles touch,key events between HU and OnCar.
 * \image html Input.jpg
 * \image latex Input.eps "OnCar InputEndpoint Sequence" width=\textwidth
 */
class OnCarInputEndpoint : public OnCarEndPointBase
{
	public:
	
	/**
    * @brief Constructor
	* @param endpointID End point ID
	*/
	   OnCarInputEndpoint(ONCARENDPOINTID endpointID):OnCarEndPointBase(endpointID)
	   {}
	   
	/**
	 * Called by HU just after creating the Input endpoint to enable/disable the touch share surface.
	 * @param touchShareSurface  This parameter indicates that video and touch, shares the same surface.
	 *                           If true then shared touch surface is enabled, if set to false then shared touch surface is disabled
     *                           If value is true, Oncar to take surface ID,projectionwidth and projectionheight from videoconfig. 
	 *                           And if it is FALSE,Oncar to take surface ID,displaywidth,displayheight from InputConfig.
	 */
	   virtual void setTouchShareSurface(bool touchShareSurface) = 0;

    /**
     * @brief destructor
     */
	 virtual ~OnCarInputEndpoint(){}
	 
	 /**
     * Register callback interface to receive touch,key related notifications.
     * @param callbacks object implementing the IOnCarInputSourceCallbacks interface
	 */
	 
     virtual void registerCallbacks(const shared_ptr <IOnCarInputSourceCallbacks> &callbacks) = 0;
	  
	 /**
     * Use this method to report a keypress or release. This is the preferred interface for
     * signaling key presses.
     * @param keycode The keycode of the key that was pressed or released.
     * @param isDown true if it is a press, false if it is a release.
     * @return STATUS_SUCCESS on success, an appropriate return code otherwise.
     */
      virtual void reportKey(INPUT_KEYCODE keycode, bool isDown) = 0;
	 
	/**
     * Use this method to report a value from a device that can output a relative value.
     * @param keycode While these devices aren't keys per se, they use a keycode as a
     *        device identifier, which should be registered using registerKeycodes.
     * @param delta The value associated with this event. You can indicate a magnitude and
     *        direction using the value and its sign. For instance, if a jog dial were
     *        turned two steps to the left, delta would be -2. If it were turned 5 steps
     *        to the right, delta would be +5.
     * @return STATUS_SUCCESS on success, an appropriate return code otherwise.
     */
     virtual void reportRelative(unsigned int keycode,int delta) = 0;
	 
	 	   
	/**
    * @brief  Set the configuration for the platform and Galreceiver. Must be called before init(). 
    * @param  config reference to object of InputConfig type struct
    *	
    * \note This method will be used only when Wayland is not used.
	*/
     virtual void setInputConfig(const InputConfig &config) = 0;
	 
   /**
    * Call this method during the initialization phase to register a touchscreen.
    * For now, the touchscreen is assumed to be related to the primary display.
    * @param width The width of the touch screen.
    * @param height The height of the touch screen.
	* @param type enumeration of TOUCH_SCREEN_TYPE
	*
	* \note This method will be used only when Wayland is not used.
    */
     virtual void registerTouchScreen(uint32_t width, uint32_t height, TOUCH_SCREEN_TYPE type) = 0;
	 
	 /**
     * Use this method to report a touch event on the touch screen that is associcated with
     * the primary display. Coordinates reported must be translated and/or scaled as appropriate
     * such that they are within the coordinate space of the display and so if the display
     * is running a scaled resolution, the x & y values should be scaled appropriately too.
     * @param numPointers The number of pointers being described.
     * @param pointerIds The pointer ids of the pointers.
     * @param x The x values of the pointers.
     * @param y The y values of the pointers.
     * @param action The action that took place in this event.
     * @param actionIndex The index of the pointer to which action corresponds.
     * @return STATUS_SUCCESS on success, an appropriate return code otherwise.
	 *
     * \note This method will be used only when Wayland is not used.
     */
     virtual int reportTouch(unsigned int numPointers, const unsigned int* pointerIds,
            const unsigned int* x, const unsigned int* y, int action, int actionIndex) = 0;
};

#endif //ONCAR_INPUT_ENDPOINT_HPP