/**
 *******************************************************************************
 * \file              OnCarBluetoothEndpoint.h
 * \brief             This sink handles Bluetooth Connectivity between HU and OnCar.
 *******************************************************************************/

#ifndef ONCAR_BLUETOOTH_ENDPOINT_HPP
#define ONCAR_BLUETOOTH_ENDPOINT_HPP
 
#include "OnCarTypes.h"
#include "OnCarEndPointBase.h"

/**
 * This class represents an interface that every sink implementation 
 * must subclass if they wish to implement a OnCarBluetoothEndpoint.  
 */
class IOnCarBluetoothCallbacks 
{
	public:
	  virtual ~IOnCarBluetoothCallbacks() { }
	  
	 /**
     * Called when BT connection is needed to established between HU and OnCar device.
	 * Once this function is called, the HU must make itself prepared for Bluetooth pairing which will be initiated by the phone.
	 * @param btAddress BT address of the phone
	 * @param btPairingMethod Pairing method used by the phone.
     */
	  virtual void onBTPairingRequestCallback(const string &btAddress, BTPAIRING_METHOD btPairingMethod) = 0;
};

/**
 * OnCarBluetoothEndpoint: This sink handles Bluetooth Connectivity between HU and OnCar.
 * \image html Bluetooth.jpg
 * \image latex Bluetooth.eps "OnCar BluetoothEndPoint Sequence" width=\textwidth
 */
class OnCarBluetoothEndpoint : public OnCarEndPointBase
{
	public:
	
	/**
     * @brief Constructor
     * @param endpointID End point ID
     */
	   OnCarBluetoothEndpoint(ONCARENDPOINTID endpointID):OnCarEndPointBase(endpointID)
	   {}

    /**
     * @brief destructor
     */
	   virtual ~OnCarBluetoothEndpoint(){}
	   
	/**
     * Set the HU's BT address with the OnCarAPI. BT address is in aa:bb:cc::dd::ee::ff format
     * @param btAddress: BT address of the HU
     */ 
	   virtual void setHUBluetoothAddress(const string &btAddress) = 0;
	   
	/**
     * Set the BT pairing method supported by HU. Must be called during initialization.
	 * @param btPairingMethod Pairing method supported by HU.
     */ 
	   virtual void setHUPairingMethod(BTPAIRING_METHOD btPairingMethod) = 0;
	   
	/**
     * Called to share the BT Pairing pin with the phone. Phone will use this information to pair with the HU.
     * @param pairingPin: Pairing pin as received from HU BT stack.
     */
	   virtual void sendBTPairingPIN(const string &pairingPin) = 0;
	   
	/**
     * Called When HU becomes ready for BT Pairing after receiving onBTPairingRequestCallback from phone.
     * @param btPairingState eBTPAIRING_STATE_NOT_PAIRED if link key does not exist between HU and Phone.
	                         eBTPAIRING_STATE_ALREADY_PAIRED if link key exists between HU and Phone.
     */
	   virtual void btPairingResponse(const BTPAIRING_STATE btPairingState) = 0;
	
    /**
     * Register callback interface to receive Bluetooth-related notifications.
     * @param callbacks object implementing the IOnCarBluetoothCallbacks interface
	 */
	   virtual void registerCallbacks(const shared_ptr <IOnCarBluetoothCallbacks> &callbacks) = 0;
};

#endif //ONCAR_BLUETOOTH_ENDPOINT_HPP