/**
 *******************************************************************************
 * \file              OnCarAudioSinkEndpoint.h
 * \brief             This sink handles audio data that must be played by the car.
 *******************************************************************************/

#ifndef ONCAR_AUDIO_ENDPOINT_HPP
#define ONCAR_AUDIO_ENDPOINT_HPP
 
#include "OnCarTypes.h"
#include "OnCarEndPointBase.h"
 
 /**
 *This class represents an interface that every sink implementation 
 *must subclass if they wish to implement a OnCarAudioSinkEndpoint.  
 */
class IOnCarAudioSinkCallbacks 
{
    public:
      virtual ~IOnCarAudioSinkCallbacks() { }
      
    /**
     * Called when MD wants to start audio streaming. This callback should be called after
     * HU has invoked setAudioFocus for the requested stream type. On receipt of this callback,
     * HU will allocate audio channel and MD must not send audio packets untill HU has acknowledged
     * this call back with acknowledgeStartAudioRequest() for the requested stream type.
     * @param type Enumeration of AUDIOSTREAM_TYPE
     */
      virtual void startAudioPlaybackRequestCallback(const AUDIOSTREAM_TYPE type) = 0;
      
    /**
     * Called when MD wants to stop audio streaming. On receipt of this callback,
     * HU will deallocate audio channel and will acknowledge this request by invoking acknowledgeStopAudioRequest().
     * After that MD should send audioFocusRequestCallback(AUDIOSTREAM_TYPE,eAUDIOFOCUS_REQUEST_RELEASE) and HU will
     * acknowledge it by invoking setAudioFocus(AUDIOSTREAM_TYPE,eAUDIOFOCUS_STATE_LOSS).
     * @param type Enumeration of AUDIOSTREAM_TYPE
     * \note
     *  HU is not expecting any callback from MD when user has paused the audio playback on OnCar APP.
     */
      virtual void stopAudioPlaybackRequestCallback(const AUDIOSTREAM_TYPE type) = 0;
      
     /**
     * Called when the OnCar source wishes to change the audio focus.
     * @param streamtype Enumeration of AUDIOSTREAM_TYPE
     * @param request Enumeration of AUDIOFOCUS_REQUEST
     */
      virtual void audioFocusRequestCallback(const AUDIOSTREAM_TYPE streamtype,const AUDIOFOCUS_REQUEST request) = 0;
      
     /**
     * Called when error has occured during audio playback.
     * @param enError: enum containing error code.
     */
      virtual void audioErrorCallback(const ONCARSINKERROR enError) = 0;
};

/**
* OnCarAudioSinkEndpoint: This sink handles audio data that must be played by the car..
* \image html Audio.jpg
* \image latex Audio.eps "OnCar AudioEndPoint Sequence" height=23cm
*/
class OnCarAudioSinkEndpoint : public OnCarEndPointBase
{
    public:
    /**
     * @brief Constructor
     * @param endpointID End point ID
     */
       OnCarAudioSinkEndpoint(ONCARENDPOINTID endpointID):OnCarEndPointBase(endpointID)
       {}
       
    /**
     * @brief destructor
     */
       virtual ~OnCarAudioSinkEndpoint(){}
       
     /**
     * Set the Audio Configuration like threadPriority, AUDIOSTREAM_TYPE, audioPipeline etc
     * @param audioConfig: reference to object of AudioConfig type struct
     */ 
      virtual void setAudioConfig(const AudioConfig& audioConfig) = 0;
       
     /**
     * When this API is invoked by HU, OnCar sink app should start sending audio packets to HU.
     * This is the acknowledgement from HU side for requested startAudioPlaybackRequestCallback.
     * @param type Enumeration of AUDIOSTREAM_TYPE
     */
       virtual void acknowledgeStartAudioRequest(AUDIOSTREAM_TYPE type) = 0;
       
    /**
     * When this API is invoked by HU, OnCar sink app should stop sending audio packets to HU.
     * This is the acknowledgement from HU side for requested vStopAudioPlaybackRequestCallback.
     * @param type Enumeration of AUDIOSTREAM_TYPE
     */
       virtual void acknowledgeStopAudioRequest(AUDIOSTREAM_TYPE type) = 0;
       
    /**
     * Called by HU to set MD's audio focus.
     * This API basically informs MD that whether MD is allowed to stream audio or not.
     * If MD wants to stream audio, MD should send vAudioFocusRequestCallback() and should wait
     * for HU to invoke this API. HU can also invoke this API if HU wants to inform MD about
     * change in ownership of audio resource and then MD should take proper action.
     * @param streamType type of audio for which audio focus changes
     * @param state audio focus state
     */
       virtual void setAudioFocus(const AUDIOSTREAM_TYPE streamType,const AUDIOFOCUS_STATE state) = 0;
    
     /**
      *Register callback interface to receive AudioSink-related notifications.
       @param callbacks object implementing the IOnCarAudioSink interface
     */
       virtual void registerCallbacks(const shared_ptr <IOnCarAudioSinkCallbacks> &callbacks) = 0;
       
    /**
     * This API basically asks MD to stop the streaming on ALSA device based on the audio stream type
     * pararmeter. This is followed by setAudioFocus which informs MD why MD was asked to stop audio streaming
     * @param streamType type of audio for which audio focus changes
     */
       virtual void stopALSAStreaming(const AUDIOSTREAM_TYPE streamType) = 0;
};

#endif //ONCAR_AUDIO_ENDPOINT_HPP