/**
 *******************************************************************************
 * \file              OnCarAPI.h
 * \brief             Interface for OnCar
 *******************************************************************************/

#ifndef ONCAR_API_HPP
#define ONCAR_API_HPP

#include "OnCarTypes.h"
#include "OnCarEndPointBase.h"

 /**
 *This class represents an interface that can be used to receive
 *notifications regarding OnCar session.  
 */
class  IOnCarSessionCallbacks
{
	public :
	   virtual ~IOnCarSessionCallbacks(){}
	 
	 /**
     * Called by MD during OnCar Session to notify status of different application states like Navigation,speech etc.
	 * @param NavigationAppState eONCARSINKAPPSTATE_PROJECTION if OnCar route guidance is active.
	 * @param SpeechAppState eONCARSINKAPPSTATE_PROJECTION if OnCar voice sessoin is active. Note this is retained for future use. For now eONCARSINKAPPSTATE_NATIVE is *                       valid.
	 * @param PhoneAppState eONCARSINKAPPSTATE_PROJECTION if OnCar Phone call is active.
     */
	   virtual void onCarSinkAppStateCallbacks (const ONCARSINK_APP_STATE NavigationAppState,
	                                             const ONCARSINK_APP_STATE SpeechAppState,
												 const ONCARSINK_APP_STATE PhoneAppState) = 0;
	 /**
     * Called when error occurs during OnCar session.
	 * @param error Enumeration of ONCARSINKERROR
     */
	   virtual void onCarSessionErrorCallbacks(const ONCARSINKERROR error) = 0;
	   
	 /**
     * Called to notify the OnCar session status.
	 * @param state Enumeration of ONCARSESSION_STATE
     */
	   virtual void onCarSessionStatusCallback(const ONCARSESSION_STATE state) = 0;

};

 /**
 *This class is the fundamental building block of the OnCar protocol.
 * \image html Connection_Disconnection.jpg
 * \image latex Connection_Disconnection.eps "OnCar Connection/Disconnection Sequence" width=\textwidth
 */
class OnCarAPI
{
public:

	/**
     * Constructor
     * Creates an OnCarAPI Instance.
     */
	OnCarAPI(){}
	
    /**
     * Destructor
     */
	virtual ~OnCarAPI(){}
	
    /**
     * Set the OnCarAPI Configuration
	 * @param config struct of OnCarConfig type
     */ 
	virtual void setOnCarAPIConfig(const OnCarConfig &config) = 0;
	
	/**
	* The first called function to intitialization of API, when Server authentication service is used.     
    * After this function call only, all endpoints should be initialized 
	* and registration of callbacks and necessary allocation of resources will be done.
	* @param auth  A parameter for authentication of Server. 
	*              The value should be the same as the registered value in Server.
	*              If auth is empty, No authentication is required.
	* @return enumeration of ONCARSINKERROR type.
	*/
	virtual ONCARSINKERROR init(string auth) = 0;
	
	/** 
	* The final called function to exit CarLink API.
	* @return enumeration of ONCARSINKERROR type.
	*/
	virtual ONCARSINKERROR deinit() = 0;

	/**
	* Called to establish OnCar session with OnCar source app.
	* OnCar Protocol communication will be established once this function is called.
	* @param aoap structure of AoapAccessory type.
	*/
	virtual ONCARSINKERROR createConnection (const AoapAccessory &aoap) = 0;
	
   /**
	* Called to end OnCar session with OnCar source app.
	* OnCar Protocol communication must be stopped when this function is called.
	* All the end points will be destroyed when this function has returned.
    * @param aoap  structure of AoapAccessory type.
	* @return enumeration of ONCARSINKERROR type.
	*/
	virtual ONCARSINKERROR destroyConnection(const AoapAccessory &aoap) = 0;

   /**
    * Called by HU to lock the orientation mode of the phone so that video frames are received in the
	* spcified orientation mode only.
    * @param mode if set to eORIENTATION_MODE_LANDSCAPE then Only Landscape mode is allowed
	*             if set to eORIENTATION_MODE_PORTRAIT then Only Portrait mode is allowed
    *             if set to eORIENTATION_MODE_BOTH then Both Landscape and Portrait are allowed
    */
	virtual void setOrientationMode(const ORIENTATION_MODE mode) = 0;

   /**
    * Register callback interface to receive IOnCarSessionCallbacks-related notifications.
    * @param sessionCbs object implementing the IOnCarSessionCallbacks interface.
	*/	
	virtual void registerSessionCallbacks(const shared_ptr <IOnCarSessionCallbacks> &sessionCbs) = 0;
	
   /** 
	* Call this function to register a new 'Service' with the receiver.
	*/
	virtual bool registerService(OnCarEndPointBase * endpoint) = 0;
	
   /**
    * Called by HU to set MD's Navigation focus.
	* HU can invoke this API if HU wants to inform MD about change in ownership of navigation resource 
	* and then MD should take proper action.
	* @param focus navigation focus type.
    */
	virtual void setNavigationFocus(NAVIGATION_FOCUS focus) = 0;
	
	/**
    * Called by HU to set PrivateMode value.
	* HU can invoke this API if HU wants to inform MD about change in PrivateMode value. 
	* and then MD should take proper action.
	* @param bPrivateModeVal private mode value.
    */
	virtual void setPrivateMode(bool bPrivateModeVal) = 0;
};
#endif //ONCAR_API_HPP
