/*!
*******************************************************************************
* \file              spi_tclExtCmdWiFi.cpp
* \brief             WiFi Adapter Implementation Class
*******************************************************************************
\verbatim
PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    WiFi Adapter Implementation Class
COPYRIGHT:      &copy; RBEI

HISTORY:
Date        |  Author                       | Modifications
11.01.2017  |  Unmukt Jain                  | Initial Version

\endverbatim
******************************************************************************/

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/

#include "spi_tclExtCmdWiFi.h"
#include "Trace.h"
#include "spi_tclWBLClientHandler.h"
#include "spi_tclExtClientFactory.h"
#include "spi_tclExtWiFiDispatcher.h"
#include "spi_tclExtCompMsgQInterface.h"

#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_EXTINTERFACE
#include "trcGenProj/Header/spi_tclExtCmdWiFi.cpp.trc.h"
#endif
//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e19 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e58 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e48 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e808 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e64 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e746 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e515 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e516 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported

/*******************************************************************************
* FUNCTION: spi_tclExtCmdWiFi:: spi_tclExtCmdWiFi(t_Void)
*******************************************************************************/

spi_tclExtCmdWiFi::spi_tclExtCmdWiFi() : m_poWBLClientHandler(NULL),m_poGMainLoopThread(NULL)
{
   m_mapPreferredFrequencies[e8DEV_TYPE_ANDROIDAUTO] = e8_2_dot_4_GHz;
   m_mapPreferredFrequencies[e8DEV_TYPE_DIPO] = e8_2_dot_4_GHz;
}

/*******************************************************************************
* FUNCTION: spi_tclExtCmdWiFi::~spi_tclExtCmdWiFi(t_Void)
*******************************************************************************/
spi_tclExtCmdWiFi::~spi_tclExtCmdWiFi()
{
   ETG_TRACE_USR1(("[FUNC]spi_tclExtCmdWiFi::~spi_tclExtCmdWiFi destructor entered "));
   m_poWBLClientHandler  = NULL;
   m_poGMainLoopThread = NULL;
   m_mapPreferredFrequencies.clear();
}

/*******************************************************************************
* FUNCTION: spi_tclExtCmdWiFi::bInitialize()
*******************************************************************************/
t_Bool spi_tclExtCmdWiFi::bInitialize()
{
   ETG_TRACE_USR1(("[FUNC]spi_tclExtCmdWiFi::bInitialize entered "));
   t_Bool bInit = false;
   spi_tclExtClientFactory *poExtClientFactory = spi_tclExtClientFactory::getInstance();
   if (NULL != poExtClientFactory)
   {
      m_poWBLClientHandler = poExtClientFactory->poGetWBLClientHandlerInstance();
      if (NULL != m_poWBLClientHandler)
      {
         vRegisterWiFiCallbacks();
         m_poGMainLoopThread = new shl::thread::FunctionThreader(m_poWBLClientHandler);
         SPI_NORMAL_ASSERT(NULL == m_poGMainLoopThread);
      }
      if (NULL != m_poGMainLoopThread)
      {
         m_poGMainLoopThread->bRunThread();
      }

      bInit = (NULL != m_poGMainLoopThread);
   }

   ETG_TRACE_USR1(("[DESC]spi_tclExtCmdWiFi::bInitialize() status = %d", ETG_ENUM(BOOL, bInit)));
   return bInit;
}

/*******************************************************************************
* FUNCTION: spi_tclExtCmdWiFi::bUnInitialize()
*******************************************************************************/
t_Bool spi_tclExtCmdWiFi::bUnInitialize()
{
   ETG_TRACE_USR1(("[FUNC]spi_tclExtCmdWiFi::bUnInitialize entered "));
   t_Bool bUninitSuccess = true;

   RELEASE_MEM(m_poGMainLoopThread);

   ETG_TRACE_USR2(("[DESC]spi_tclExtCmdWiFi::bUnInitialize() status = %d ", ETG_ENUM(BOOL,bUninitSuccess)));
   return bUninitSuccess;
}

/*******************************************************************************
* FUNCTION: spi_tclExtCmdWiFi::vInitiateWirelessDiscovery(tenDeviceCategory enDevCategory)
*******************************************************************************/
t_Void spi_tclExtCmdWiFi::vInitiateWirelessDiscovery(tenDeviceCategory enDevCategory)
{
   ETG_TRACE_USR1(("[FUNC]spi_tclExtCmdWiFi::vInitiateWirelessDiscovery"));

   tenWifiFrequency enFrequency = enGetPreferredFrequency(enDevCategory);

   t_String szFrequency;
   t_String szTechnology;

   switch(enDevCategory)
   {
      case e8DEV_TYPE_ANDROIDAUTO:
         szTechnology= scoszTechnology_AAW;
         break;

      case e8DEV_TYPE_DIPO:
         szTechnology = scoszTechnology_CPW;
      break;

      default:
         szTechnology = scoszTechnology_Default;
      break;
   }
   switch(enFrequency)
   {
      case e8_5_dot_0_GHz:
         szFrequency = scoszFrequencyBand_5Ghz;
      break;

      default:
         szFrequency = scoszFrequencyBand_2_4Ghz;
      break;

   }

   if(NULL != m_poWBLClientHandler)
   {
      m_poWBLClientHandler->vConfigureWiFi(szTechnology,szFrequency);
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclExtCmdWiFi::vRegisterWiFiCallbacks()
***************************************************************************/
t_Void spi_tclExtCmdWiFi::vRegisterWiFiCallbacks()
{
   ETG_TRACE_USR1(("[FUNC]spi_tclExtCmdWiFi::vRegisterWiFiCallbacks() entered "));
   if(NULL != m_poWBLClientHandler)
   {
      trWiFiCallbacks rWiFiCb;
      rWiFiCb.fvOnConfigureWiFiResult =
               std::bind(&spi_tclExtCmdWiFi::vOnConfigureWiFiResult,
                        this, SPI_FUNC_PLACEHOLDERS_1);

      rWiFiCb.fvOnWiFiConfig =
               std::bind(&spi_tclExtCmdWiFi::vOnWiFiConfig,
                        this, SPI_FUNC_PLACEHOLDERS_2);

      rWiFiCb.fvOnSupportedFrequencies =
               std::bind(&spi_tclExtCmdWiFi::vOnSupportedFrequencyUpdate,
                        this,SPI_FUNC_PLACEHOLDERS_1);

      m_poWBLClientHandler->vRegisterCallbacks(rWiFiCb);
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclExtCmdWiFi::vOnConfigureWiFiResult()
***************************************************************************/
t_Void spi_tclExtCmdWiFi::vOnConfigureWiFiResult(t_Bool bResult)
{
   ETG_TRACE_USR1(("[FUNC]spi_tclExtCmdWiFi::vOnConfigureWiFiResult() entered "));
   ConfigureWiFiResultMsg oConfigureWiFiResultMsg;
   oConfigureWiFiResultMsg.m_bConfigureWiFiResult = bResult;
   spi_tclExtCompMsgQInterface *poExtMsgQinterface = spi_tclExtCompMsgQInterface::getInstance();
   if (NULL != poExtMsgQinterface)
   {
	   poExtMsgQinterface->bWriteMsgToQ(&oConfigureWiFiResultMsg,sizeof(oConfigureWiFiResultMsg));
   }//if (NULL != poExtMsgQinterface)
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclExtCmdWiFi::vOnWiFiConfig()
***************************************************************************/
t_Void spi_tclExtCmdWiFi::vOnWiFiConfig(const trWiFiConfig &corfrWiFiConfig,const std::vector<trStationInfo>& corfvecStationInfo)
{
   ETG_TRACE_USR1(("[FUNC]spi_tclExtCmdWiFi::vOnWiFiConfig() entered "));
   WiFiConfigMsg oWiFiConfigMsg(corfrWiFiConfig,corfvecStationInfo);
   spi_tclExtCompMsgQInterface *poExtMsgQinterface = spi_tclExtCompMsgQInterface::getInstance();
   if (NULL != poExtMsgQinterface)
   {
      poExtMsgQinterface->bWriteMsgToQ(&oWiFiConfigMsg,sizeof(oWiFiConfigMsg));
   }//if (NULL != poExtMsgQinterface)
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclExtCmdWiFi::vSetWiFiLimitationMode()
 ***************************************************************************/
t_Void spi_tclExtCmdWiFi::vSetWiFiLimitationMode(tenDeviceCategory enDevCategory,t_Bool bActivate)
{
   ETG_TRACE_USR1(("[FUNC]spi_tclExtCmdWiFi::vSetWiFiLimitationMode:DeviceCategory:%d, Activate:%d",
            ETG_ENUM(DEVICE_CATEGORY,enDevCategory),ETG_ENUM(BOOL,bActivate)));
   if(NULL != m_poWBLClientHandler)
   {
      m_poWBLClientHandler->vSetWiFiLimitationMode(enDevCategory,bActivate);
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclExtCmdWiFi::enGetPreferredFrequency()
 ***************************************************************************/
tenWifiFrequency spi_tclExtCmdWiFi::enGetPreferredFrequency(tenDeviceCategory enCategory)
{
   ETG_TRACE_USR1(("[FUNC]spi_tclExtCmdWiFi::enGetPreferredFrequency() entered "));
   tenWifiFrequency enFrequency = e8_Unknown;

   m_LockPreferredFrequencies.s16Lock();
   auto iter_PreferredFreq = m_mapPreferredFrequencies.find(enCategory);
   if(iter_PreferredFreq != m_mapPreferredFrequencies.end())
   {
      enFrequency = iter_PreferredFreq->second;
   }
   m_LockPreferredFrequencies.vUnlock();

   ETG_TRACE_USR4(("enGetPreferredFrequency()::Frequency:%d",
            ETG_CENUM(tenWifiFrequency,enFrequency)));
   return enFrequency;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclExtCmdWiFi::vStorePreferredFrequency()
 ***************************************************************************/
t_Void spi_tclExtCmdWiFi::vStorePreferredFrequency(const tMapSupportedFrequencies& corfmapSuppFrequencies)
{
   ETG_TRACE_USR1(("[FUNC]spi_tclExtCmdWiFi::vStorePreferredFrequency"));

   auto iter_Frequency_5GHz = corfmapSuppFrequencies.find(e8_5_dot_0_GHz);
   if(iter_Frequency_5GHz != corfmapSuppFrequencies.end())
   {
      for(auto suppTechIter : iter_Frequency_5GHz->second.mapSuppTechnologies)
      {
         if(e8_TECHNOLOGY_ALLOWED == suppTechIter.second)
         {
            m_LockPreferredFrequencies.s16Lock();
            auto iter_PreferredFreq = m_mapPreferredFrequencies.find(suppTechIter.first);
            if(iter_PreferredFreq != m_mapPreferredFrequencies.end())
            {
               iter_PreferredFreq->second = e8_5_dot_0_GHz;
            }
            m_LockPreferredFrequencies.vUnlock();
         }
      }
   }
   else
   {
      m_LockPreferredFrequencies.s16Lock();
      m_mapPreferredFrequencies[e8DEV_TYPE_ANDROIDAUTO] = e8_2_dot_4_GHz;
      m_mapPreferredFrequencies[e8DEV_TYPE_DIPO] = e8_2_dot_4_GHz;
      m_LockPreferredFrequencies.vUnlock();
   }
   vPrintPreferredFrequencies();
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclExtCmdWiFi::vPrintPreferredFrequencies()
 ***************************************************************************/
t_Void spi_tclExtCmdWiFi::vPrintPreferredFrequencies()
{
   for(auto preferredFreqElem:m_mapPreferredFrequencies)
   {
      ETG_TRACE_USR4(("[DEBUG:]vPrintPreferredFrequencies::DeviceCategory:%d",
               ETG_CENUM(tenDeviceCategory,preferredFreqElem.first)));
      ETG_TRACE_USR4(("[DEBUG:]vPrintPreferredFrequencies::PreferredFrequency:%d",
               ETG_CENUM(tenWifiFrequency,preferredFreqElem.second)));

   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclExtCmdWiFi::OnSupportedFrequencyUpdate()
 ***************************************************************************/
t_Void spi_tclExtCmdWiFi::vOnSupportedFrequencyUpdate(const tMapSupportedFrequencies& corfmapSuppFrequencies)
{
   ETG_TRACE_USR1(("[FUNC]spi_tclExtCmdWiFi::OnSupportedFrequencyUpdate"));

   vStorePreferredFrequency(corfmapSuppFrequencies);
   SupportedFrequenciesMsg oSupportedFrequenciesMsg(corfmapSuppFrequencies);
   spi_tclExtCompMsgQInterface *poExtMsgQinterface = spi_tclExtCompMsgQInterface::getInstance();
   if (NULL != poExtMsgQinterface)
   {
      poExtMsgQinterface->bWriteMsgToQ(&oSupportedFrequenciesMsg,sizeof(oSupportedFrequenciesMsg));
   }//if (NULL != poExtMsgQinterface)
}

//lint -restore
