/***********************************************************************/
/*!
* \file  spi_tclDiPoAudioResourceMngr.h
* \brief Manages DiPo Audio resource
*************************************************************************
\verbatim

PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    Manages DiPo Audio resource
AUTHOR:         Pruthvi Thej Nagaraju
COPYRIGHT:      &copy; RBEI

HISTORY:
Date        | Author                | Modification
20.08.2014  | Pruthvi Thej Nagaraju | Initial Version
22.08.2018  | Chaitra Srinivasa     | Added audio related functionality from DipoResourceMngr   

\endverbatim
*************************************************************************/

#ifndef SPI_TCLDIPOAUDIORESOURCEMNGR_H_
#define SPI_TCLDIPOAUDIORESOURCEMNGR_H_

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/

#include "DiPOTypes.h"
#include "Timer.h"
#include "Lock.h"
#include "spi_tclDiPORespAudio.h"
#include "spi_tclExtRespMediaDevInfo.h"
#include "spi_tclResourceMngrSettingsIntf.h"


struct trAudioContextInfo
{
    tenAudioContext enAudioContext;
    t_Bool bAudioFlag;

    trAudioContextInfo(): enAudioContext(e8SPI_AUDIO_MAIN), bAudioFlag(false)
    {
    }
};

class spi_tclDipoResourceMngrIntf;
class spi_tclDiPOCmdRsrcMngr;

/*!
 * \class spi_tclDiPoAudioResourceMngr
 * \brief Manages DiPo Audio resource
 */
class spi_tclDiPoAudioResourceMngr: public spi_tclDiPORespAudio, public spi_tclExtRespMediaDevInfo
{
   public:
      /***************************************************************************
       *********************************PUBLIC*************************************
       ***************************************************************************/

      /***************************************************************************
       ** FUNCTION:  spi_tclDiPoAudioResourceMngr::spi_tclDiPoAudioResourceMngr
       ***************************************************************************/
      /*!
       * \fn     spi_tclDiPoAudioResourceMngr(spi_tclDipoResourceMngrIntf* poDipoResrcMngrIntf)
       * \brief  Default Constructor
       * \sa      ~spi_tclDiPoAudioResourceMngr()
       **************************************************************************/
      spi_tclDiPoAudioResourceMngr(spi_tclDipoResourceMngrIntf* poDipoResrcMngrIntf);

      /***************************************************************************
       ** FUNCTION:  spi_tclDiPoAudioResourceMngr::~spi_tclDiPoAudioResourceMngr
       ***************************************************************************/
      /*!
       * \fn     ~spi_tclDiPoAudioResourceMngr()
       * \brief  destructor
       * \sa     spi_tclDiPoAudioResourceMngr()
       **************************************************************************/
      ~spi_tclDiPoAudioResourceMngr();

      /***************************************************************************
       ** FUNCTION:  spi_tclDiPoAudioResourceMngr::vOnSPISelectDeviceResult()
       ***************************************************************************/
       /*!
       * \fn     vOnSPISelectDeviceResult()
        * \brief   Interface to receive result of SPI device selection/deselection
       * \param   u32DevID : [IN] Resource Manager callbacks structure.
       * \param   enDeviceConnReq : [IN] Select/ deselect.
       * \param   enRespCode : [IN] Response code (success/failure)
       * \param   enErrorCode : [IN] Error
       * \retval  t_Void
       **************************************************************************/
       t_Void vOnSPISelectDeviceResult(t_U32 u32DevID,
             tenDeviceConnectionReq enDeviceConnReq,
             tenResponseCode enRespCode,
             tenErrorCode enErrorCode);

       /***************************************************************************
       ** FUNCTION:  spi_tclDiPoAudioResourceMngr::vUpdateInitialAudioSettings()
       ***************************************************************************/
       /*!
       * \fn     vUpdateInitialAudioSettings()
       * \brief  Function to set the initial audio settings
       * \param  bIsLastModeSupported : Indicates last mode feature supported or not
       * \retval  t_Void
       **************************************************************************/
       t_Void vUpdateInitialAudioSettings(t_Bool bIsLastModeSupported);

       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclDiPoAudioResourceMngr::vSetAccessoryAudioContext()
       ***************************************************************************/
       /*!
       * \fn     vSetAccessoryAudioContext(const tenAudioContext coenAudioCntxt,
       *                                   t_Bool bReqFlag)
       * \brief   To set accessory audio context related info .
       * \param   cu8AudioCntxt: [IN] Audio Context
       * \param   bReqFlag     : [IN] Request/ Release flag      
       * \retval  t_Void
       **************************************************************************/
       t_Void vSetAccessoryAudioContext(const tenAudioContext coenAudioCntxt,
                                        t_Bool bReqFlag);

       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclDiPoAudioResourceMngr::vUpdateAudioContext()
       ***************************************************************************/
       /*!
       * \fn      t_Void vUpdateAudioContext()
       * \brief   Used to update the audio context      
       * \param   corfoDiPOModeState : [IN] CarPlay mode state
       * \retval  t_Void
       **************************************************************************/
       t_Void vUpdateAudioContext(const trDiPOModeState &corfoDiPOModeState);           

       /***************************************************************************
        ** FUNCTION:  t_Void spi_tclDiPoAudioResourceMngr::vOnAudioAllocateRequest(...
        ***************************************************************************/
       /*!
        * \fn      t_Void vOnAudioAllocateRequest(AudioChannelType enAudioChannelType,
        *                         tenDiPOMainAudioType enAudioType,tenAudioReqType enAudioReqType)
        * \brief   To Post the Audio allocate/de-allocate request
        * \param   enAudioChannelType : [IN] Audio channel type
        * \param   enAudioType : [IN] Alternate audio/ default audio
        * \param   enAudioReqType : [IN] Audio allocate/deallocate request
        * \param   corfrDiPOAudioFormat : [IN] Audio format which includes sample rate, bits per channel.         
        * \retval  t_Void
        ***************************************************************************/
       t_Void vOnAudioAllocateRequest(AudioChannelType enAudioChannelType,
                tenDiPOMainAudioType enAudioType,
                tenAudioReqType enAudioReqType,
                const trDiPOAudioFormat& corfrDiPOAudioFormat); 

       /***************************************************************************
        ** FUNCTION:  t_Void spi_tclDiPoAudioResourceMngr::vOnPluginLoaded(...
        ***************************************************************************/
       /*!
        * \fn      t_Void vOnPluginLoaded()
        * \brief   To handle audio context
        * \retval  t_Void
        ***************************************************************************/                 
        t_Void vOnPluginLoaded();      

        /***************************************************************************
        ** FUNCTION: t_Void spi_tclDiPoAudioResourceMngr::vPostBTMediaDeviceInfo()
        ***************************************************************************/
        /*!
        * \fn     t_Void vPostBTMediaDeviceInfo(const trMediaDeviceInfo& corfrMediaDeviceInfo)
        * \brief  It gets triggered when MediaPlayer notifies changes in BT devices
        * \param  corfrMediaDeviceInfo  : [IN] BT Media deviec info
        * \retVal None
        **************************************************************************/
        t_Void vPostBTMediaDeviceInfo(const trMediaDeviceInfo& corfrMediaDeviceInfo) override;

        /***************************************************************************
        ** FUNCTION:  t_Void spi_tclDiPoAudioResourceMngr::vOnEvaluateInitalModes()
        ***************************************************************************/
        /*!
        * \fn      t_Void vOnEvaluateInitalModes
        * \brief   Method to check and set inital modes before session is started
        * \param   [IN] corfszBTAddress: Unique Device ID
        * \retval  t_Void
        **************************************************************************/
        t_Void vOnEvaluateInitalModes(const t_String& corfszBTAddress);

        /***************************************************************************
         ** FUNCTION:  t_Void spi_tclDiPoAudioResourceMngr::vSetResourceMngrSettingsInstance(...)
         ***************************************************************************/
         /*!
         * \fn      t_Void vSetResourceMngrSettingsInstance()
         * \brief   Method to set Resource manager settings instance.
         * \param   poResourceMngrSettingsIntf : pointer to InputHandlerSettingsIntf.
         * \retval  t_Void
         **************************************************************************/
        t_Void vSetResourceMngrSettingsInstance(spi_tclResourceMngrSettingsIntf* poResourceMngrSettingsIntf);
        
   private:
       /***************************************************************************
       ** FUNCTION:  t_Void spi_tclDiPoAudioResourceMngr::vHandleAudioLastMode(...)
       ***************************************************************************/
       /*!
       * \fn      t_Void vHandleAudioLastMode(trDiPOAudioContext& rfrDiPOAudioContext)
       * \brief   Sets the audio context to be set based on the last mode
       * \param   rfrDiPOAudioContext: [OUT] audio context that should be set based on last mode
       * \retval  t_Void
       ***************************************************************************/
       t_Void vHandleAudioLastMode(trDiPOAudioContext& rfrDiPOAudioContext);
       
       /***************************************************************************
       ** FUNCTION: t_Void spi_tclDiPoAudioResourceMngr::vStoreLastAudioTakeContext()
       ***************************************************************************/
       /*!
        * \fn      t_Void vStoreLastAudioTakeContext()
        * \brief   Stores latest audio context details for an audio Take request
        * \retval  t_Bool
        ***************************************************************************/
       t_Void vStoreLastAudioTakeContext(const trDiPOAudioContext& corfrDiPOAudioContext);
       
       /***************************************************************************
        ** FUNCTION:  t_Void spi_tclDiPoAudioResourceMngr::vUpdateLastAudioMode(...)
        ***************************************************************************/
       /*!
        * \fn      t_Void vUpdateLastAudioMode(t_Bool bIsResetLastMode)
        * \brief   To update the last audio mode
        * \param   bIsResetLastMode: [IN] Indicates whether last audio mode should be reset
        * \retval  t_Void
        ***************************************************************************/
       t_Void vUpdateLastAudioMode(t_Bool bIsResetLastMode);

       //! Stores the last take context for audio resource
       trDiPOAudioContext m_rLastAudioTakeContext;       

       //! current audio source for resource arbitration
       tenAudioContext m_enCurAudioSource;
       
       //! Current audio flag
       t_Bool m_bCurAudioFlag;             
       
       //! queue of audio context updates failed
       std::queue<trAudioContextInfo> m_rAudioContextInfo;       
       
       //! Indicates if last mode settings is enabled
       t_Bool m_bIsLastModeSupported;      

       //! Pointer to Dipo cmd resource manager
       spi_tclDiPOCmdRsrcMngr* m_poDiPOCmdRsrcMngr;

       //! Pointer to Dipo resrc mngr interface
       spi_tclDipoResourceMngrIntf* m_poDipoRsrcMngrIntf;
       
       //! To store current mode state
       trDiPOModeState m_rCurrModeState;
       
       //! To store audio last mode
       tenLastMode m_enLastAudioMode;

       //! Map to maintain the list of BT media devices info
       std::map<t_U32,trMediaDeviceInfo> m_mapBTMediaDevInfo;

       //! Lock to perform atomic operations on m_mapBTMediaDevInfo
       Lock m_mapBTMediaDevInfoLock;

       // sets resource manager settings pointer
       spi_tclResourceMngrSettingsIntf* m_poResourceMngrSettings;
};


#endif /* SPI_TCLDIPOAUDIORESOURCEMNGR_H_ */
