/*!
*******************************************************************************
* \file              spi_tclDiPOResourceMngr.h
* \brief             DiPO Resource manager
*******************************************************************************
\verbatim
PROJECT:        G3G
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    DiPO Resource manager
COPYRIGHT:      &copy; RBEI

HISTORY:
Date       |  Author                      | Modifications
24.03.2014 |  Shihabudheen P M            | Initial Version
05.04.2014 |  Priju K Padiyath            | Modified to handle display & audio context
19.06.2014 |  Shihabudheen P M            | Modified foe App state resource arbitration,
                                            session end handling
25.06.2014 |  Shihabudheen P M            | Adapted to the CarPlay design changes
16.12.2014 |  Shihabudheen P M            | Changed resource transfer requests.
11.02.2015 |  Shihabudheen P M            | Added timer implementation
29.11.2016 |  Noopur R K                  | HMI Synchronization
22.08.2018 |  Chaitra Srinivasa           | Removed audio/video related functionality and
                                            added functions to set and get generic member variables

\endverbatim
******************************************************************************/
#ifndef SPI_TCLDIPORESOURCEMNGR_H
#define SPI_TCLDIPORESOURCEMNGR_H

/******************************************************************************
| includes:
|----------------------------------------------------------------------------*/
#include <queue>
#include "BaseTypes.h"
#include "DiPOTypes.h"
#include "spi_tclResourceMngrBase.h"
#include "spi_tclDiPORespSession.h"
#include "spi_tclDiPORespAudio.h"
#include "spi_tclDiPORespRsrcMngr.h"
#include "spi_tclDiPODeviceMsgRcvr.h"
#include "spi_tclResourceArbitrator.h"
#include "spi_tclDipoResourceMngrIntf.h"

class spi_tclDiPoVideoResourceMngr;
class spi_tclDiPoAudioResourceMngr;

/****************************************************************************/
/*!
* \class spi_tclDiPOResourceMngr
* \brief DiPO Resource manager class
*
* spi_tclDiPOResourceMngr is the resource manager class which consists of the 
* the logics to handle  DiPO resource arbitration related messages between HMI and device.
****************************************************************************/
class spi_tclDiPOResourceMngr: public spi_tclResourceMngrBase, public spi_tclDiPORespSession,
public spi_tclDiPORespRsrcMngr, public spi_tclDipoResourceMngrIntf
{
public:
  /***************************************************************************
   *********************************PUBLIC************************************
   ***************************************************************************/
   
   /***************************************************************************
   ** FUNCTION:  spi_tclDiPOResourceMngr::spi_tclDiPOResourceMngr()
   ***************************************************************************/
   /*!
   * \fn     spi_tclDiPOResourceMngr()
   * \brief  Constructor
   * \sa     ~spi_tclDiPOResourceMngr()
   **************************************************************************/
   spi_tclDiPOResourceMngr();

  /***************************************************************************
   ** FUNCTION:  spi_tclDiPOResourceMngr::~spi_tclDiPOResourceMngr()
   ***************************************************************************/
   /*!
   * \fn     ~spi_tclDiPOResourceMngr()
   * \brief  Destructor
   * \sa     spi_tclDiPOResourceMngr()
   **************************************************************************/
   ~spi_tclDiPOResourceMngr();

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclResourceMngrBase::vSelectDevice()
   ***************************************************************************/
   /*!
   * \fn     t_Void vSelectDevice(const t_U32 cou32DevId, tenDeviceConnectionReq enConnReq,
   *              tenDeviceCategory enDevCat)
   * \brief  method to send the select device request to resouce manager
   * \param  cou32DevId : [IN] Uniquely identifies the target Device.
   * \param  enConnReq  : [IN] Device selection type.
   * \param  enDevCat   : [IN] Device category.
   * \retval t_Void
   **************************************************************************/
   t_Void vSelectDevice(const t_U32 cou32DevId, tenDeviceConnectionReq enConnReq,
            tenDeviceCategory enDevCat) override;

  /***************************************************************************
   ** FUNCTION:  spi_tclDiPOResourceMngr::vOnSPISelectDeviceResult()
   ***************************************************************************/
   /*!
   * \fn     vOnSPISelectDeviceResult()
    * \brief   Interface to receive result of SPI device selection/deselection
   * \param   u32DevID : [IN] Resource Manager callbacks structure.
   * \param   enDeviceConnReq : [IN] Select/ deselect.
   * \param   enRespCode : [IN] Response code (success/failure)
   * \param   enErrorCode : [IN] Error
   * \retval  t_Void 
   **************************************************************************/
   t_Void vOnSPISelectDeviceResult(t_U32 u32DevID, tenDeviceConnectionReq enDeviceConnReq,
         tenResponseCode enRespCode, tenErrorCode enErrorCode) override;

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDiPOResourceMngr::vRegRsrcMngrCallBack()
   ***************************************************************************/
   /*!
   * \fn      t_Void vRegRsrcMngrCallBack()
   * \brief   To register call back to the resource manager
   * \param   rRsrcMngrCallback : [IN] Callback functions holder
   * \retval  t_Void
   * \sa      bInitialize()
   **************************************************************************/
   t_Void vRegRsrcMngrCallBack(trRsrcMngrCallback rRsrcMngrCallback) override;

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDiPOResourceMngr::vSetAccessoryDisplayContext()
   ***************************************************************************/
   /*!
   * \fn      virtual t_Void vSetAccessoryDisplayContext(const t_U32 cou32DevId,
   *        t_Bool bDisplayFlag, tenDisplayContext enDisplayContext)
   * \brief   To send accessory display context related info .
   * \pram    cou32DevId  : [IN] Uniquely identifies the target Device.
   * \param   bDisplayFlag : [IN] Display flag
   * \pram    enDisplayContext : [IN] Display context
   * \pram    rfrcUsrCntxt: [IN] User Context Details.
   * \retval  t_Void
   **************************************************************************/
   t_Void vSetAccessoryDisplayContext(const t_U32 cou32DevId,
      t_Bool bDisplayFlag, tenDisplayContext enDisplayContext,
      const trUserContext& rfrcUsrCntxt) override;

      /***************************************************************************
   ** FUNCTION: t_Void spi_tclDiPOResourceMngr::vSetAccessoryDisplayMode(t_U32...
   ***************************************************************************/
   /*!
   * \fn     vSetAccessoryDisplayMode()
   * \brief  Accessory display mode update request.
   * \param  [IN] cu32DeviceHandle      : Uniquely identifies the Device.
   * \param  [IN] corDisplayContext : Display context info
   * \param  [IN] corDisplayConstraint : DiDisplay constraint info
   * \param  [IN] coenDisplayInfo       : Display info flag
   * \sa
   **************************************************************************/
   t_Void vSetAccessoryDisplayMode(const t_U32 cu32DeviceHandle,
      const trDisplayContext corDisplayContext,
      const trDisplayConstraint corDisplayConstraint,
      const tenDisplayInfo coenDisplayInfo) override;

  /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDiPOResourceMngr::vSetAccessoryAudioContext()
   ***************************************************************************/
   /*!
   * \fn     vSetAccessoryAudioContext(const t_U32 cou32DevId, const t_U8 cu8AudioCntxt
   *       t_Bool bDisplayFlag, const trUserContext& rfrcUsrCntxt)
   * \brief   To send accessory audio context related info .
   * \pram    cou32DevId   : [IN] Uniquely identifies the target Device.
   * \param   cu8AudioCntxt: [IN] Audio Context
   * \pram    bReqFlag     : [IN] Request/ Release flag
   * \pram    rfrcUsrCntxt : [IN] User Context Details.
   * \retval  t_Void
   **************************************************************************/
   t_Void vSetAccessoryAudioContext(const t_U32 cou32DevId, 
      const tenAudioContext coenAudioCntxt,
      t_Bool bReqFlag, 
      const trUserContext& rfrcUsrCntxt) override;

  /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDiPOResourceMngr::vSetAccessoryAppState()
   ***************************************************************************/
   /*!
   * \fn     t_Void vSetAccessoryAppState(tenSpeechAppState enSpeechAppState, tenPhoneAppState enPhoneAppState,
   *         tenNavAppState enNavAppState
   *
   * \brief   To set accessory app state related info.
   * \pram    enSpeechAppState: [IN] Accessory speech state.
   * \param   enPhoneAppState : [IN] Accessory phone state
   * \pram    enNavAppState   : [IN] Accessory navigation app state
   * \pram    rfrcUsrCntxt    : [IN] User Context Details.
   * \retval  t_Void
   **************************************************************************/
   t_Void vSetAccessoryAppState(const tenSpeechAppState enSpeechAppState, 
      const tenPhoneAppState enPhoneAppState,
      const tenNavAppState enNavAppState, 
      const trUserContext& rfrcUsrCntxt) override;

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDiPOResourceMngr::vSetVehicleBTAddress()
   ***************************************************************************/
   /*!
   * \fn      vSetVehicleBTAddress(t_Bool bLocDataAvailable)
   * \brief   Interface to update the vehicle BT address info update.
   * \param   szBtAddress: [IN] BT address.
   * \retval  None
   **************************************************************************/
   t_Void vSetVehicleBTAddress(t_String szBtAddress) override;

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclDiPOResourceMngr::vSetFeatureRestrictions()
    ***************************************************************************/
   /*!
    * \fn      t_Void vSetFeatureRestrictions(const t_U8 cou8ParkModeRestrictionInfo,
    *          const t_U8 cou8DriveModeRestrictionInfo)
    * \brief   Method to set Vehicle Park/Drive Mode Restriction
    *          cou8ParkModeRestrictionInfo : Park mode restriction
    *          cou8DriveModeRestrictionInfo : Drive mode restriction
    * \retval  t_Void
    **************************************************************************/
   t_Void vSetFeatureRestrictions(const t_U8 cou8ParkModeRestrictionInfo,
         const t_U8 cou8DriveModeRestrictionInfo) override;

   /***************************************************************************
    ** FUNCTION: t_Void spi_tclDiPOResourceMngr::vSetGeneralRestrictions
    ***************************************************************************/
   /*!
    * \fn     vSetGeneralRestriction
    * \brief  Interface to set the general restrictions for various technologies.
    * \param  u16GeneralRestrInfo : General Restriction Info for the technology.
    * \param  [OUT] void.
    **************************************************************************/
   t_Void vSetGeneralRestrictions(const t_U16 u16GeneralRestrInfo) override;

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclDiPOResourceMngr::vOnSetNativeVideoRenderingStatus()
   ***************************************************************************/
   /*!
   * \fn     t_Void vOnSetNativeVideoRenderingStatus(tenNativeVideoRenderingStatus enNativeVideoRenderingStatus)
   * \brief  Method to inform the native video rendering status update.
   * \param  enNativeVideoRenderingStatus : Whether the video rendering has
   *                                        started or stopped.
   * \retval t_Void
   **************************************************************************/
   t_Void vOnSetNativeVideoRenderingStatus(tenNativeVideoRenderingStatus enNativeVideoRenderingStatus) override;
   
   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclDiPOResourceMngr::vSetETCStatus()
    ***************************************************************************/
   /*!
    * \fn      t_Void vSetETCStatus(t_Bool bETCStatus)
    * \brief   Method to set ETC Status
    * \param   bETCStatus: [IN] ETC Status (Enabled or Disabled)
    * \retval  t_Void
    **************************************************************************/
   t_Void vSetETCStatus(t_Bool bETCStatus) override;

   /***************************************************************************
   ** FUNCTION:  tVoid spi_tclDiPOResourceMngr::vOnTelephoneCallStatus()
   ***************************************************************************/
   /*!
   * \fn      vOnTelephoneCallStatus(
   *             const std::vector<trTelCallStatusInfo>& rfcoTelCallStatusList)
   * \brief   Interface to recieve info on each Telephone call
   * \param   [IN] rfcoTelCallStatusList : Telephone call info list
   **************************************************************************/
   tVoid vOnTelephoneCallStatus(
          const std::vector<trTelCallStatusInfo>& rfcoTelCallStatusList) override;

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDiPOResourceMngr::vOnDisableBluetoothTrigger(...
   ***************************************************************************/
   /*!
   * \fn      virtual t_Void vOnDisableBluetoothTrigger(t_String szBTMACAddr)
   * \brief   To Post the disableBluetooth update to registered clients
   * \param   szBTMACAddr : [IN] Unique ID of the device.
   * \retval  t_Void
   ***************************************************************************/
   t_Void vOnDisableBluetoothTrigger(t_String szBTMACAddr) override;

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDiPOResourceMngr::vOnEvaluateInitalModes()
   ***************************************************************************/
   /*!
   * \fn      t_Void vOnEvaluateInitalModes
   * \brief   Method to check and set inital modes before session is started
   * \param   [IN] corfszBTAddress: Unique Device ID
   * \retval  t_Void
   **************************************************************************/
   t_Void vOnEvaluateInitalModes(const t_String& corfszBTAddress) override;

   /***************************************************************************
   ** FUNCTION: t_Bool spi_tclDiPOResourceMngr::bIsSessionActive()
   ***************************************************************************/
   /*!
   * \fn     t_Bool bIsSessionActive()
   * \brief  To check whether session is active or not
   * \retVal t_Bool
   **************************************************************************/   
   t_Bool bIsSessionActive();   
   
   /***************************************************************************
   ** FUNCTION: t_Void spi_tclDiPOResourceMngr::vSetDisplayLastMode()
   ***************************************************************************/
   /*!
   * \fn     t_Void vSetDisplayLastMode(tenLastMode enLastDisplayMode)
   * \brief  To set the last mode of display
   * \param  enLastDisplayMode : To set last modes of video
   * \retVal None
   **************************************************************************/   
   t_Void vSetDisplayLastMode(tenLastMode enLastDisplayMode);
   
   /***************************************************************************
   ** FUNCTION: tenLastMode spi_tclDiPOResourceMngr::enGetDisplayLastMode()
   ***************************************************************************/
   /*!
   * \fn     tenLastMode enGetDisplayLastMode()
   * \brief  To get the last mode of display
   * \retVal tenLastMode
   **************************************************************************/   
   tenLastMode enGetDisplayLastMode();
   
   /***************************************************************************
   ** FUNCTION: t_Void spi_tclDiPOResourceMngr::vSetAudioLastMode()
   ***************************************************************************/
   /*!
   * \fn     t_Void vSetAudioLastMode(tenLastMode enLastDisplayMode)
   * \brief  To set the last mode of audio
   * \param  enLastDisplayMode : To set last modes of video
   * \retVal None
   **************************************************************************/   
   t_Void vSetAudioLastMode(tenLastMode enLastDisplayMode);
   
   /***************************************************************************
   ** FUNCTION: tenLastMode spi_tclDiPOResourceMngr::enGetAudioLastMode()
   ***************************************************************************/
   /*!
   * \fn     tenLastMode enGetAudioLastMode()
   * \brief  To get the last mode of audio
   * \retVal tenLastMode
   **************************************************************************/   
   tenLastMode enGetAudioLastMode();   
   
   /***************************************************************************
   ** FUNCTION: t_Void spi_tclDiPOResourceMngr::vSetCallStatus()
   ***************************************************************************/
   /*!
   * \fn     t_Void vSetCallStatus(t_Bool bIsCallActiveBeforeDisableBT)
   * \brief  To set the call status
   * \param  bIsCallActiveBeforeDisableBT : To set the call status
   * \retVal None
   **************************************************************************/    
   t_Void vSetCallStatus(t_Bool bIsCallActiveBeforeDisableBT);   
   
   /***************************************************************************
   ** FUNCTION: t_Bool spi_tclDiPOResourceMngr::bGetCallStatus()
   ***************************************************************************/
   /*!
   * \fn     t_Bool bGetCallStatus()
   * \brief  To get the last mode of audio
   * \retVal t_Bool
   **************************************************************************/   
   t_Bool bGetCallStatus();      

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclDiPOResourceMngr::vPostAudioContext()
   ***************************************************************************/
   /*!
   * \fn     t_Void vPostAudioContext(t_Bool bPlayStatus, t_U8 u8AudioCntxt,
   *                                  tenDiPOEntity enPermMainAudio)
   * \brief  To update the audio context
   * \param  bPlayStatus : Audio playing status
   * \param  u8AudioCntxt : Audio context
   * \param  enPermMainAudio : Permanent owner of main audio resource
   * \retVal None
   **************************************************************************/      
   t_Void vPostAudioContext(t_Bool bPlayStatus, t_U8 u8AudioCntxt,
                            tenDiPOEntity enPermMainAudio);
   
   /***************************************************************************
   ** FUNCTION: t_Void spi_tclDiPOResourceMngr::vPostDisplayContext()
   ***************************************************************************/
   /*!
   * \fn     t_Void vPostDisplayContext(t_Bool bRenderStatus, 
   *          tenDisplayContextReason enDispCntxtReason, tenDiPOEntity enPermScreen)
   * \brief  To update the display context
   * \param  bRenderStatus : Video rendering status
   * \param  enDispCntxtReason : Display context reason
   * \param  enPermScreen : Permanent owner of display resource
   * \retVal None
   **************************************************************************/      
   t_Void vPostDisplayContext(t_Bool bRenderStatus, tenDisplayContextReason enDispCntxtReason,
                              tenDiPOEntity enPermScreen);

  /***************************************************************************
   ** FUNCTION:  spi_tclDiPOResourceMngr::bGetVideoModeChangeMsg()
   ***************************************************************************/
   /*!
   * \fn     bGetVideoModeChangeMsg()
   * \brief  Request for video mode change. This will trigger a 
   * \       ModeChangeRequest to the device, because of the change in accessory
   * \       display context.
   * \param  enDisplayContext    : [IN] Accessory display context 
   * \param  bDisplayFlag        : [IN] Display flag
   * \param  rfrDiPOVideoContext : [OUT] Video context update
   * \retVal  Bool, True if successfully create the context change request,
   *         False otherwise.
   * \sa    
   **************************************************************************/    
   t_Bool bGetVideoModeChangeMsg(const tenDisplayContext enDisplayContext, 
        t_Bool bDisplayFlag, trDiPOVideoContext &rfrDiPOVideoContext);   
        
  /***************************************************************************
   ** FUNCTION:  spi_tclDiPOResourceMngr::bGetAudioModechangeMsg()
   ***************************************************************************/
   /*!
   * \fn     bGetAudioModechangeMsg()
   * \brief  function to get a resource mode change message for audio context change 
   * \param  coenAudioCntxt        : [IN] Accessory audio context   
   * \param  bReqStatus            : [IN] Request status, True for request and False for release
   * \param  rfrDiPOAudioContext : [OUT] Audio context
   * \retVal t_Bool : True if successful, false otherwise
   * \sa    
   **************************************************************************/ 
   t_Bool bGetAudioModechangeMsg(const tenAudioContext coenAudioCntxt,
      t_Bool bReqStatus,
      trDiPOAudioContext &rfrDiPOAudioContext);     
      
   /***************************************************************************
   ** FUNCTION:  spi_tclDiPOResourceMngr::vUpdateInitialSettings()
   ***************************************************************************/
   /*!
   * \fn     vUpdateInitialSettings()
   * \brief  Function to set the initial settings
   * \param  trRsrcSettings : [IN] Settings
   * \
   **************************************************************************/
   t_Void vUpdateInitialSettings(trRsrcSettings rRsrcSettings) override;
   
   /***************************************************************************
   ** FUNCTION: t_Void spi_tclDiPOResourceMngr::vProjVideoRenderingStatus(...
   ***************************************************************************/
   /*!
   * \fn      t_Void vProjVideoRenderingStatus(tenProjVideoRenderingStatus enProjRenderingStatus)
   * \brief   Method to update the projection video rendering status update
   * *        Projection stopped update from Phone.stop in IVideoAdapter.
   * \param   enProjRenderingStatus : [IN] CarPlay projection status
   * \retval  t_Void
   ***************************************************************************/
   t_Void vProjVideoRenderingStatus(tenProjVideoRenderingStatus enProjRenderingStatus);   

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclResourceMngr::vPostDeviceSpmSubState()
   ***************************************************************************/
   /*!
   * \fn     vPostDeviceSpmSubState()
   * \brief  This function used to update LCM about the SPM substate changes
   * \param  enSpmSubState  : [IN] Spm SubState type
   * \param  m_bSubStateData  : [IN] Spm SubState data
   **************************************************************************/
   t_Void vPostDeviceSpmSubState(tenSpmSubState enSpmSubState, t_Bool m_bSubStateData);
    
   
   /***************************************************************************
   **************************END OF PUBLIC************************************
   ***************************************************************************/

private:

  /***************************************************************************
   ***************************PRIVATE*****************************************
   ***************************************************************************/

  /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDiPOResourceMngr::vHandleSessionEnd()
   ***************************************************************************/
   /*!
   * \fn      t_Void vHandleSessionEnd()
   * \brief   Function to handle the session end update from CarPlay. This is
   *          used to trigger an automatic device deselection and thereby 
   *          deallocate all the allocated resources.
   * \param   rRsrcMngrCallback : [IN] Callback functions holder
   * \retval  t_Void
   * \sa      bInitialize()
   **************************************************************************/
   t_Void vHandleSessionEnd();

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDiPOResourceMngr::vOnSessionMsg(...
   ***************************************************************************/
   /*!
   * \fn      t_Void vOnSessionMsg(tenDiPOSessionState enDiPOSessionState,
   *                                       tenDiPOSessionTransport enSessionTransport,
   *                                       t_String szSessionIPAddress)
   * \brief   To Post the session message to SPI, when the session state changes.
   * \param   enDiPOSessionState : [IN] Session State : Session Started,Session_Ended,
   *                                                    CarPlayd_PlugIn_Loaded or UnLoaded.
   * \param   enSessionTransport : [IN] Session is started over USB Transport or WiFi
   *                                    Applicable, only when SessionState is Session Started/ended.
   * \param   szSessionIPAddress : [IN] IP Address of the active carplay device.
   *                                    used, only if the Session Transport is WiFi
   * \retval  t_Void
   ***************************************************************************/
   t_Void vOnSessionMsg(tenDiPOSessionState enDiPOSessionState,
                        tenDiPOSessionTransport enSessionTransport,
                        t_String szSessionIPAddress);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDiPOResourceMngr::vUpdateAppState()
   ***************************************************************************/
   /*!
   * \fn      t_Void vUpdateAppState()
   * \brief   Function to determine the CarPlay device app state changes and,
   * \        update it to the HMI if required 
   * \param   corfoDiPOModeState : [IN] mode state 
   * \retval  t_Void
   **************************************************************************/
   t_Void vUpdateAppState(const trDiPOModeState &corfoDiPOModeState);

   /***************************************************************************
   ** FUNCTION:  spi_tclDiPOResourceMngr::vFormatBTMacAddress
   ***************************************************************************/
   /*!
   * \fn     vFormatBTMacAddress()
   * \brief  This function is used to format the BT address to MAC address
   * \param  szMACAddress: [IN] MAC address
   * \retVal Bool : True if successfully transferred.
   **************************************************************************/
   t_Bool vFormatBTMacAddress(t_String &szMACAddress);
      
   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDiPOResourceMngr::vOnModesChangedUpdate(...
   ***************************************************************************/
   /*!
   * \fn      t_Void vOnModesChangedUpdate(const trDiPOModeState& corfrDiPOModeState)
   * \brief   To Post the ModesChanged update from Phone
   * \param   corfrDiPOModeState : [IN] Mode states
   * \retval  t_Void
   ***************************************************************************/
   t_Void vOnModesChangedUpdate(const trDiPOModeState& corfrDiPOModeState);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDiPOResourceMngr::vUpdateCallStatus(...
   ***************************************************************************/
   /*!
   * \fn      t_Void vUpdateCallStatus()
   * \brief   To update call status based on last mode
   * \retval  t_Void
   ***************************************************************************/   
   t_Void vUpdateCallStatus();

   //! Resourec arbitrator.
   spi_tclResourceArbitrator *m_poResArb;

   //! Callback functions to Resource manager
   trRsrcMngrCallback m_rRsrcMngrCallback;

   //! Current mode state of the CarPlay
   trDiPOModeState m_rCurrModeState;

   //! Selected device ID
   t_U32 m_u32DevId;

   //! Latest app state updated.
   trDiPOAppState m_rDiPOAppState;

   //! Latest Drive mode restriction
   t_U8 m_u8DriveModeRestrictionInfo;
  
   //! Enum that indicates if ETC feature is enabled
   tenETCInfo m_enETCSupport;

   //! Indicates if display last mode was projection or native.
   tenLastMode m_enLastDisplayMode;    
   
   //! To store audio last mode
   tenLastMode m_enLastAudioMode;   
   
   //! Indicates if session is active or not
   t_Bool m_bIsSessionActive;
   
   //! Indicates if last mode settings is enabled
   t_Bool m_bIsLastModeSupported;

   //! Used to store the Telephone call status info
   std::vector<trTelCallStatusInfo> mvecTelCallStatusInfo;

   //! Lock to protect Call status
   Lock m_CallStatusLock;
   
   //!Pointer to Dipo audio resource manager
   spi_tclDiPoAudioResourceMngr* m_poAudioResourceMngr;
   
   //!Pointer to Dipo video resource manager
   spi_tclDiPoVideoResourceMngr* m_poVideoResourceMngr;
   
   //! Bt call status
   t_Bool m_bIsCallActiveBeforeDisableBT;
   
   
   /***************************************************************************
   **************************END OF PRIVATE************************************
   ***************************************************************************/
};


#endif
