/*!
*******************************************************************************
* \file              spi_tclDiPODeviceMsgRcvr.cpp
* \brief             DiPO Msg handler for RM
*******************************************************************************
\verbatim
PROJECT:        G3G
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    DiPO IPC message handler for resource management
COPYRIGHT:      &copy; RBEI

HISTORY:
Date       |  Author                      | Modifications
22.03.2014 |  Shihabudheen P M            | Initial Version
05.04.2014 |  Priju K Padiyath            | Updating mesage receive functionality
17.04.2014 |  Shihabudheen P M            | Update the message handling and triggering
                                            as per the resource manager design.
23.06.2014 |  Shihabudheen P M            | Adapted to the latest CarPlay design
21.07.2014 |  Shihabudheen P M            | Added vInitAudioMap() 
                                            Modified bHandleAudioMsg() for handling sampling rates 
08.10.2014 |  Ramya Murthy                | Implementation for DisableBluetooth msg.
28.11.2014 |  Hari Priya E R               | Added a separate trace to measure audio source switching timing
06.05.2015 |  Tejaswini HB                  | Lint Fix
17.07.2015 |  Sameer Chandra               | Memory leak fix
29.11.2016 |  Noopur R K                  | HMI Synchronization
01.03.2017 |  Shiva Kumar G               | R12N Plugin Adaptations

                                    
\endverbatim
******************************************************************************/
#include <algorithm>

#include "DiPOTypes.h"
#include "spi_tclDiPOMsgQInterface.h"
#include "spi_tclDiPODiscovererDispatcher.h"
#include "spi_tclDiPOVideoDispatcher.h"
#include "spi_tclDiPOSessionDispatcher.h"
#include "spi_tclDiPOBluetoothDispatcher.h"
#include "spi_tclDiPORsrcMngrDispatcher.h"
#include "spi_tclDiPOAudioDispatcher.h"
#include "spi_tclDiPODeviceMsgRcvr.h"

#include "Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_RSRCMNGR
#include "trcGenProj/Header/spi_tclDiPODeviceMsgRcvr.cpp.trc.h"
#endif


/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/
//typedef std::map<t_U16, tenAudioSamplingRate>::iterator tAudSampleRateMapItr;   //Commented to suppress Lint Warning

static const t_U8 scu8TraceArraySize = 10;

/***************************************************************************
 ** FUNCTION: spi_tclDiPODeviceMsgRcvr::spi_tclDiPODeviceMsgRcvr()
 ***************************************************************************/
spi_tclDiPODeviceMsgRcvr::spi_tclDiPODeviceMsgRcvr()
   :m_enAudioDir(e8AUD_INVALID)
{
   ETG_TRACE_USR1((" %s entered \n", __PRETTY_FUNCTION__));
   m_mapAudioSamplingRate[0]     = e8AUD_SAMPLERATE_DEFAULT;
   m_mapAudioSamplingRate[8000]  = e8AUD_SAMPLERATE_8KHZ;
   m_mapAudioSamplingRate[16000] = e8AUD_SAMPLERATE_16KHZ;
   m_mapAudioSamplingRate[24000] = e8AUD_SAMPLERATE_24KHZ;
}

/***************************************************************************
 ** FUNCTION: spi_tclDiPODeviceMsgRcvr::~spi_tclDiPODeviceMsgRcvr()
 ***************************************************************************/
spi_tclDiPODeviceMsgRcvr::~spi_tclDiPODeviceMsgRcvr()
{
   ETG_TRACE_USR1((" %s entered \n", __PRETTY_FUNCTION__));

   m_mapAudioSamplingRate.clear();
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclDiPODeviceMsgRcvr::vExecute()
 ***************************************************************************/
t_Void spi_tclDiPODeviceMsgRcvr::vExecute(tShlMessage *poMessage)
{
   ETG_TRACE_USR1((" %s entered \n", __PRETTY_FUNCTION__));
   trMsgQBase *poBaseMsg =  NULL;
   t_Bool bStatus = false;
   if ((NULL != poMessage) && (NULL != poMessage->pvBuffer))
   {
      poBaseMsg = (trMsgQBase*)poMessage->pvBuffer;
      switch((t_U8)poBaseMsg->enMsgType)
      {
      case e8RM_RESP_MESSAGE:
         {
            ETG_TRACE_USR1(("ModeChanged message received from [CarPlay] "));
            bStatus = bHandleModeChangeMsg(poMessage);
         }
         break;
      case e8AUDIO_ALLOC_MESSAGE:
         {
            ETG_TRACE_USR1(("AudioAllocation message received from [CarPlay] "));
            bStatus = bHandleAudioMsg(poMessage);
         }
         break;
      case e8ON_REQUEST_UI_MESSAGE:
         {
            ETG_TRACE_USR1(("RequestUI message received from [CarPlay] "));
            bStatus = bHandleRequestUIMsg(poMessage);
         }
         break;
	  case e8AUDIO_DUCK_MESSAGE:
         {
            ETG_TRACE_USR1(("Audio Duck message received from [CarPlay] "));
            bStatus = bHandleAudioDuckMsg(poMessage);
         }
         break;
     case e8SESSION_MESSAGE:
        {
           ETG_TRACE_USR1(("Session message received from [CarPlay]"));
           bStatus = bHandleSessionMsg(poMessage);
        }
        break;
     case e8DISABLE_BLUETOOTH_MESSAGE:
        {
           ETG_TRACE_USR1(("DisableBluetooth message received from [CarPlay]"));
           bStatus = bHandleDisableBluetoothMsg(poMessage);
        }
        break;
     case e8VIDEOPLAYBACK_STATUS_MESSAGE:
        {
           ETG_TRACE_USR1(("video Playback status message received from [CarPlay]"));
           bStatus = bHandleVideoPlaybackStatusMsg(poMessage);
        }
        break;
	 case e8DEVICE_INFO_MESSAGE:
	    {
            ETG_TRACE_USR1(("Device Info message received from [CarPlay]"));
	        bStatus = bHandleDeviceInfoMsg(poMessage);		   
	    }
	    break;
     case e8DEVICE_CONNECT_RESP_MESSAGE:
	    {
		    ETG_TRACE_USR1(("ConnectDevice Response message received from [CarPlay]"));
		    bStatus = bHandleConnectDeviceRespMsg(poMessage);
	    }
	    break;
     case e8DEVICE_DISCONNECT_RESP_MESSAGE:
        {
	        ETG_TRACE_USR1(("DisconnectDevice Response  message received from [CarPlay]"));
		    bStatus = bHandleDisconnectDeviceRespMsg(poMessage);
        }
        break;
     case e8PROJECTIONVIDEORENDERING_STATUS_MESSAGE:
        {
        	ETG_TRACE_USR1(("Projection Video Rendering status message received from [CarPlay]"));
            bStatus = bHandleProjectionVideoRenderingStatusMsg(poMessage);
        }
        break;
     default:
        {
            ETG_TRACE_USR1(("Invalid message "));
        }
      } // switch((t_U8)poBaseMsg->enMsgType)
      // Release the allocated memory
      t_Char* pczBuffer = static_cast<t_Char*> (poMessage->pvBuffer);
      RELEASE_ARRAY_MEM(pczBuffer);
   }// if ((NULL != poMessage) && (NULL != poMessage->pvBuffer))
   RELEASE_MEM(poMessage); // Move out if condition to make Lint happy, NULL check is part of RELEASE_MEM.
   ETG_TRACE_USR1(("Message recieved from CarPlay Adapter plugin processesd with status: %d ", bStatus));
}

/***************************************************************************
 ** FUNCTION: tShlMessage spi_tclDiPODeviceMsgRcvr::poGetMsgBuffer()
 ***************************************************************************/
tShlMessage* spi_tclDiPODeviceMsgRcvr::poGetMsgBuffer(size_t siBuffer)
{
   ETG_TRACE_USR1((" %s entered \n", __PRETTY_FUNCTION__));

   tShlMessage* poMessage = new tShlMessage;

   if (NULL != poMessage)
   {
      if(0 < siBuffer)
      {
         //! Allocate the requested memory
         poMessage->pvBuffer = new t_Char[siBuffer];
      }
      else
      {
         poMessage->pvBuffer = NULL;
      } // if(0 < siBuffer)

      if (NULL != poMessage->pvBuffer)
      {
         poMessage->size = (siBuffer);
      }
      else
      {
         //! Free the message as internal allocation failed.
         delete poMessage;
         poMessage = NULL;
      } //   if (NULL != poMessage->pvBuffer)
   } // if (NULL != poMessage)

   return poMessage;
}


/****Private methods******/
/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiPODeviceMsgRcvr::bHandleModeChangeMsg()
 ***************************************************************************/
t_Bool spi_tclDiPODeviceMsgRcvr::bHandleModeChangeMsg(tShlMessage *poMessage)
{
   ETG_TRACE_USR1(("spi_tclDiPODeviceMsgRcvr::bHandleModeChangeMsg"));
   
   t_Bool bRetVal= false;
   
   trDiPORMMsgResp *poRmMsgResp = static_cast<trDiPORMMsgResp*>(poMessage->pvBuffer);
   if(NULL != poRmMsgResp)
   {
	   ModesChangedMsg oModesChangedMsg;
	   *(oModesChangedMsg.prDiPOModeState) = poRmMsgResp->rDiPOModeState;
	   
	   spi_tclDiPOMsgQInterface* poDiPOMsgQIntf = spi_tclDiPOMsgQInterface::getInstance();
       bRetVal = poDiPOMsgQIntf->bWriteMsgToQ(&oModesChangedMsg, sizeof(oModesChangedMsg));
   }
   
   ETG_TRACE_USR1(("spi_tclDiPODeviceMsgRcvr::bHandleModeChangeMsg left with %d",
            ETG_ENUM(BOOL,bRetVal)));
			
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiPODeviceMsgRcvr::bHandleRequestUIMsg()
 ***************************************************************************/
t_Bool spi_tclDiPODeviceMsgRcvr::bHandleRequestUIMsg(tShlMessage *poMessage)
{
   ETG_TRACE_USR1(("spi_tclDiPODeviceMsgRcvr::bHandleRequestUIMsg"));

   t_Bool bRetVal = false;
   
   if(NULL != poMessage)
   {
      trOnRequestUIMsg *poOnRequestUIMsg = static_cast<trOnRequestUIMsg*>(poMessage->pvBuffer);
      if(NULL != poOnRequestUIMsg)
      {
         OnRequestUIMsg oOnRequestUIMsg;
         strncpy(oOnRequestUIMsg.cRequestUIURL,poOnRequestUIMsg->cRequestUIURL,MAX_STR_LEN);
         spi_tclDiPOMsgQInterface* poDiPOMsgQIntf = spi_tclDiPOMsgQInterface::getInstance();
         bRetVal = poDiPOMsgQIntf->bWriteMsgToQ(&oOnRequestUIMsg, sizeof(oOnRequestUIMsg));
      }
   }

   ETG_TRACE_USR1(("spi_tclDiPODeviceMsgRcvr::bHandleRequestUIMsg left with %d",
            ETG_ENUM(BOOL,bRetVal)));

   return bRetVal;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiPODeviceMsgRcvr::bHandleSessionMsg()
 ***************************************************************************/
t_Bool spi_tclDiPODeviceMsgRcvr::bHandleSessionMsg(tShlMessage *poMessage)
{
   ETG_TRACE_USR1(("spi_tclDiPODeviceMsgRcvr::bHandleSessionMsg()"));
   
   t_Bool bRetVal = false;
   
   if (NULL != poMessage)
   {
      trDiPOSessionMsg *poDiPOSessionMsg = static_cast<trDiPOSessionMsg*>(poMessage->pvBuffer);
      if(NULL != poDiPOSessionMsg)
      {
         DiPOSessionMsg oDiPOSessionMsg;

         ETG_TRACE_USR2(("bHandleSessionMsg: Session State:%d",poDiPOSessionMsg->enDiPOSessionState));
         oDiPOSessionMsg.enDiPOSessionState = poDiPOSessionMsg->enDiPOSessionState;

         ETG_TRACE_USR2(("bHandleSessionMsg: Session Transport:%d",poDiPOSessionMsg->enSessionTransport));
         oDiPOSessionMsg.enSessionTransport = static_cast<tenDiPOSessionTransport>(poDiPOSessionMsg->enSessionTransport);

         ETG_TRACE_USR1(("spi_tclDiPODeviceMsgRcvr::bHandleSessionMsg: Session IP Address:%s",
                  poDiPOSessionMsg->szSessionIPAddress));
         strncpy(oDiPOSessionMsg.szSessionIPAddress, poDiPOSessionMsg->szSessionIPAddress, MAX_STR_LEN);

         spi_tclDiPOMsgQInterface* poDiPOMsgQIntf = spi_tclDiPOMsgQInterface::getInstance();
         bRetVal = poDiPOMsgQIntf->bWriteMsgToQ(&oDiPOSessionMsg, sizeof(oDiPOSessionMsg));
      }
   }
   
   ETG_TRACE_USR1(("spi_tclDiPODeviceMsgRcvr::bHandleSessionMsg left with %d",
      ETG_ENUM(BOOL,bRetVal)));

   return bRetVal;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiPODeviceMsgRcvr::bHandleAudioMsg()
 ***************************************************************************/
t_Bool spi_tclDiPODeviceMsgRcvr::bHandleAudioMsg(tShlMessage *poMessage)
{
   ETG_TRACE_USR1(("spi_tclDiPODeviceMsgRcvr::bHandleAudioMsg(tShlMessage *poMessage)"));
   t_Bool bRetVal = false;
   trAudioAllocMsg *poAudioAllocMsg = static_cast<trAudioAllocMsg*>(poMessage->pvBuffer);
   if(NULL != poAudioAllocMsg)
   {
       /* Separate binary trace added,which will be manually decoded for every allocation/deallocation request.
       This trace will be used along with a trace in Audio component to determine the Source Switch timing */
       tU8 u8Data[scu8TraceArraySize] = {
       enModule_SPI,
       enFile_spi_tclDiPODeviceMsgRcvr,
       enFkt_bHandleAudioMsg,(poAudioAllocMsg->enAudioChannelType),
       (poAudioAllocMsg->enAudioType),(poAudioAllocMsg->enAudioReqType),0,0,0,0};
       tU32 u32Line = __LINE__;
       u8Data[6] = (tU8)(((u32Line+6) & 0xFF000000)>>24);
       u8Data[7] = (tU8)(((u32Line+6) & 0x00FF0000)>>16);
       u8Data[8] = (tU8)(((u32Line+6) & 0x0000FF00)>>8);
       u8Data[9] = (tU8)((u32Line+6) & 0x000000FF);
       et_vTraceBinary(TR_CLASS_AUDIO_SRC_CHANGE_MEAS, TR_LEVEL_ERRORS, ET_EN_T8LIST, scu8TraceArraySize, u8Data, ET_EN_DONE);
		 
	   AudioAllocReq oAudioAllocReq;
	   oAudioAllocReq.enAudioChannelType = poAudioAllocMsg->enAudioChannelType;
	   oAudioAllocReq.enAudioType = poAudioAllocMsg->enAudioType ;
	   oAudioAllocReq.enAudioReqType = poAudioAllocMsg->enAudioReqType;
	   *(oAudioAllocReq.prDiPOAudioFormat) = poAudioAllocMsg->rDiPOAudioFormat;
	   
	   spi_tclDiPOMsgQInterface* poDiPOMsgQIntf = spi_tclDiPOMsgQInterface::getInstance();
       bRetVal = poDiPOMsgQIntf->bWriteMsgToQ(&oAudioAllocReq, sizeof(oAudioAllocReq));	   
   }
 
   ETG_TRACE_USR1(("spi_tclDiPODeviceMsgRcvr::bHandleAudioMsg left with %d",
      ETG_ENUM(BOOL,bRetVal)));
 
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiPODeviceMsgRcvr::bHandleAudioMsg()
 ***************************************************************************/
t_Bool spi_tclDiPODeviceMsgRcvr::bHandleAudioDuckMsg(tShlMessage *poMessage)
{
   ETG_TRACE_USR1(("spi_tclDiPODeviceMsgRcvr::bHandleAudioDuckMsg"));
   
   t_Bool bRetVal = false;
   trAudioDuckMsg *poAudioDuckMsg = static_cast<trAudioDuckMsg*>(poMessage->pvBuffer);
   if (NULL != poAudioDuckMsg)
   {
	  DuckAudioMsg oDuckAudioMsg;
	  oDuckAudioMsg.dFinalVolume = poAudioDuckMsg->dFinalVolume;
	  oDuckAudioMsg.dDurationInMs = poAudioDuckMsg->dDurationInMs;

	  spi_tclDiPOMsgQInterface* poDiPOMsgQIntf = spi_tclDiPOMsgQInterface::getInstance();
      bRetVal = poDiPOMsgQIntf->bWriteMsgToQ(&oDuckAudioMsg, sizeof(oDuckAudioMsg));	 	   
   }

   ETG_TRACE_USR1(("spi_tclDiPODeviceMsgRcvr::bHandleAudioDuckMsg left with %d",
      ETG_ENUM(BOOL,bRetVal)));

   return bRetVal;
}


/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiPODeviceMsgRcvr::bHandleDisableBluetoothMsg()
 ***************************************************************************/
t_Bool spi_tclDiPODeviceMsgRcvr::bHandleDisableBluetoothMsg(tShlMessage *poMessage)
{
   ETG_TRACE_USR1((" spi_tclDiPODeviceMsgRcvr::bHandleDisableBluetoothMsg"));
   t_Bool bRetVal = false;

   if (NULL != poMessage)
   {
      trOnDisBluetoothMsg *poOnDisBluetoothMsg = static_cast<trOnDisBluetoothMsg*>(poMessage->pvBuffer);

      if (NULL != poOnDisBluetoothMsg)
      {
         ETG_TRACE_USR1(("spi_tclDiPODeviceMsgRcvr::bHandleDisableBluetoothMsg: Device BT MAC Address:%s",
                  poOnDisBluetoothMsg->cBluetoothID));
		 ETG_TRACE_USR1(("spi_tclDiPODeviceMsgRcvr::bHandleDisableBluetoothMsg: Session Transport:%d",
                  poOnDisBluetoothMsg->enSessionTransport));

         DisableBluetoothMsg oDisableBluetoothMsg;
		 t_String szBTAddress = szGetBTAddress(poOnDisBluetoothMsg->cBluetoothID);
		 if(MAX_STR_LEN > szBTAddress.length())
		 {
	        strncpy(oDisableBluetoothMsg.szBMACAddress, szBTAddress.c_str(), MAX_STR_LEN);
		 }
		 else
		 {
			ETG_TRACE_ERR(("[ERR]:spi_tclDiPODeviceMsgRcvr::bHandleDisableBluetoothMsg:BtMacAddress length is more than defined"));
		 }  

         oDisableBluetoothMsg.enSessionTransport = poOnDisBluetoothMsg->enSessionTransport;
        
         spi_tclDiPOMsgQInterface* poDiPOMsgQIntf = spi_tclDiPOMsgQInterface::getInstance();
         bRetVal = poDiPOMsgQIntf->bWriteMsgToQ(&oDisableBluetoothMsg, sizeof(oDisableBluetoothMsg));
      }//if (NULL != poOnDisBluetoothMsg)
   }//if (NULL != poMessage)
   
   ETG_TRACE_USR1(("spi_tclDiPODeviceMsgRcvr::bHandleDisableBluetoothMsg left with %d",
      ETG_ENUM(BOOL,bRetVal)));

   return bRetVal;
}


/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiPODeviceMsgRcvr::bHandleVideoPlaybackStatusMsg()
 ***************************************************************************/
t_Bool spi_tclDiPODeviceMsgRcvr::bHandleVideoPlaybackStatusMsg(tShlMessage *poMessage)
{
   ETG_TRACE_USR1(("spi_tclDiPODeviceMsgRcvr::bHandleVideoPlaybackStatusMsg entered"));

   t_Bool bRetVal = false;
   if(NULL != poMessage)
   {
      trVideoPlayBackStatusMsg *poVideoPlaybackMsg = static_cast<trVideoPlayBackStatusMsg*>(poMessage->pvBuffer);
      if(NULL != poVideoPlaybackMsg)
      {
          VideoPlaybackStatusMsg oVideoPlaybackStatusMsg;
          oVideoPlaybackStatusMsg.bIsPlaybackStarted = poVideoPlaybackMsg->bIsPlaybackStarted;
		  
          spi_tclDiPOMsgQInterface* poDiPOMsgQIntf = spi_tclDiPOMsgQInterface::getInstance();
          bRetVal = poDiPOMsgQIntf->bWriteMsgToQ(&oVideoPlaybackStatusMsg, sizeof(oVideoPlaybackStatusMsg));

      }
      ETG_TRACE_USR1(("spi_tclDiPODeviceMsgRcvr::bHandleVideoPlaybackStatusMsg left with %d",ETG_ENUM(BOOL,bRetVal)));
   }//if(NULL != poMessage)

   ETG_TRACE_USR1(("spi_tclDiPODeviceMsgRcvr::bHandleVideoPlaybackStatusMsg left with %d",
      ETG_ENUM(BOOL,bRetVal)));
   
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION: t_String spi_tclDiPODeviceMsgRcvr::szGetBTAddress(const t_String...)
 ***************************************************************************/
t_String spi_tclDiPODeviceMsgRcvr::szGetBTAddress(const t_String& rfcoszBTMACAddress) const
{
   //! Initialize string with the BT MAC address
   t_String szBTAddress(rfcoszBTMACAddress);

   //! Remove all instances of ':' character from string and convert to uppercase.
   //! (Example: If szBTMACAddress is "28:e1:4c:df:30:72", format string as "28E14CDF3072"
   if (false == szBTAddress.empty())
   {
      szBTAddress.erase(std::remove(szBTAddress.begin(), szBTAddress.end(), ':'), szBTAddress.end());
      std::transform(szBTAddress.begin(), szBTAddress.end(), szBTAddress.begin(), ::toupper);
   }

   ETG_TRACE_USR4(("spi_tclDiPODeviceMsgRcvr::szGetBTAddress() left with %s ", szBTAddress.c_str()));
   return szBTAddress;
}

/***************************************************************************
 ** FUNCTION: t_String spi_tclDiPODeviceMsgRcvr::bHandleProjectionVideoRenderingStatusMsg(tShlMessage *poMessage)
 ***************************************************************************/
t_Bool spi_tclDiPODeviceMsgRcvr::bHandleProjectionVideoRenderingStatusMsg(tShlMessage *poMessage)
{
   ETG_TRACE_USR1(("spi_tclDiPODeviceMsgRcvr::bHandleProjectionVideoRenderingStatusMsg entered"));
   t_Bool bRetVal = false;
   if (NULL != poMessage)
   {
      trProjectionVideoRenderingStatusMsg *poProjectionVideoRenderingMsg = static_cast<trProjectionVideoRenderingStatusMsg*> (poMessage->pvBuffer);
      if(NULL != poProjectionVideoRenderingMsg)
      {
          ProjVideoRenderingStatusMsg oProjVideoRenderingStatusMsg;
          oProjVideoRenderingStatusMsg.enProjRenderingStatus = poProjectionVideoRenderingMsg->enProjectionVideoRenderingStatus;

          spi_tclDiPOMsgQInterface* poDiPOMsgQIntf = spi_tclDiPOMsgQInterface::getInstance();
          bRetVal = poDiPOMsgQIntf->bWriteMsgToQ(&oProjVideoRenderingStatusMsg, sizeof(oProjVideoRenderingStatusMsg));
      }
   }//if(NULL != poMessage)
   ETG_TRACE_USR1(("spi_tclDiPODeviceMsgRcvr::bHandleProjectionVideoRenderingStatusMsg left with %d",ETG_ENUM(BOOL,bRetVal)));
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDiPODeviceMsgRcvr::bHandleDeviceInfoMsg
***************************************************************************/
t_Bool spi_tclDiPODeviceMsgRcvr::bHandleDeviceInfoMsg(tShlMessage *poMessage)
{
   ETG_TRACE_USR1(("spi_tclDiPODeviceMsgRcvr::bHandleDeviceInfoMsg entered"));

   t_Bool bRetVal = false;
   if(NULL != poMessage)
   {
      DiPODeviceInfoMsg oDiPODeviceInfoMsg;
      trDiPODeviceInfo rDiPODeviceInfo;

      trDeviceInfoMsg *poDeviceInfoMsg = static_cast<trDeviceInfoMsg*>(poMessage->pvBuffer);
      if(NULL != poDeviceInfoMsg)
      {
         ETG_TRACE_USR1(("spi_tclDiPODeviceMsgRcvr::bHandleDeviceInfoMsg: Device Name:%s",poDeviceInfoMsg->szDeviceName));
            rDiPODeviceInfo.szDeviceName = poDeviceInfoMsg->szDeviceName;

         ETG_TRACE_USR1(("spi_tclDiPODeviceMsgRcvr::bHandleDeviceInfoMsg: Device Version:%s",
            poDeviceInfoMsg->szDeviceVersion));
         rDiPODeviceInfo.szDeviceVersion = poDeviceInfoMsg->szDeviceVersion;

         ETG_TRACE_USR1(("spi_tclDiPODeviceMsgRcvr::bHandleDeviceInfoMsg: Device BT MAC Address:%s",
            poDeviceInfoMsg->szBTMACAddress));
         rDiPODeviceInfo.szBTMACAddress = poDeviceInfoMsg->szBTMACAddress;

         ETG_TRACE_USR1(("spi_tclDiPODeviceMsgRcvr::bHandleDeviceInfoMsg: is Device Connected:%d",
            ETG_ENUM(BOOL,poDeviceInfoMsg->bDeviceConnStatus)));
         oDiPODeviceInfoMsg.bIsDeviceConnected = poDeviceInfoMsg->bDeviceConnStatus;
      }
      *(oDiPODeviceInfoMsg.prDeviceInfo) = rDiPODeviceInfo;

      spi_tclDiPOMsgQInterface* poDiPOMsgQIntf = spi_tclDiPOMsgQInterface::getInstance();
      bRetVal = poDiPOMsgQIntf->bWriteMsgToQ(&oDiPODeviceInfoMsg, sizeof(oDiPODeviceInfoMsg));
   }//if(NULL != poMessage)
   
   ETG_TRACE_USR1(("spi_tclDiPODeviceMsgRcvr::bHandleDeviceInfoMsg left with %d",ETG_ENUM(BOOL,bRetVal)));
   
   return bRetVal;	
}

/***************************************************************************
** FUNCTION:  spi_tclDiPODeviceMsgRcvr::bHandleConnectDeviceRespMsg
***************************************************************************/
t_Bool spi_tclDiPODeviceMsgRcvr::bHandleConnectDeviceRespMsg(tShlMessage *poMessage)
{
   ETG_TRACE_USR1(("spi_tclDiPODeviceMsgRcvr::bHandleConnectDeviceRespMsg entered"));

   t_Bool bRetVal = false;
   if (NULL != poMessage)
   {
      trConnectDeviceRespMsg *poConnectDeviceRespMsg = static_cast<trConnectDeviceRespMsg*>(poMessage->pvBuffer);
      ConnectDeviceRespMsg oConnectDeviceRespMsg;
      if (NULL != poConnectDeviceRespMsg)
      {
         ETG_TRACE_USR1(("spi_tclDiPODeviceMsgRcvr::bHandleConnectDeviceRespMsg: Device BT MAC Address:%s",
            poConnectDeviceRespMsg->szBTMACAddress));
         strncpy(oConnectDeviceRespMsg.szBTMACAddress, poConnectDeviceRespMsg->szBTMACAddress, MAX_STR_LEN);

         ETG_TRACE_USR1(("spi_tclDiPODeviceMsgRcvr::bHandleConnectDeviceRespMsg: ConnectDevice succeeded:%d",
            ETG_ENUM(BOOL, poConnectDeviceRespMsg->bConnectResult)));
         oConnectDeviceRespMsg.bConnectSucceeded = poConnectDeviceRespMsg->bConnectResult;
      }

      spi_tclDiPOMsgQInterface* poDiPOMsgQIntf = spi_tclDiPOMsgQInterface::getInstance();
      bRetVal = poDiPOMsgQIntf->bWriteMsgToQ(&oConnectDeviceRespMsg, sizeof(oConnectDeviceRespMsg));
   }//if(NULL != poMessage)
  
     
   ETG_TRACE_USR1(("spi_tclDiPODeviceMsgRcvr::bHandleConnectDeviceRespMsg left with %d",ETG_ENUM(BOOL,bRetVal)));

   
   return bRetVal;
}


/***************************************************************************
** FUNCTION:  t_Bool spi_tclDiPODeviceMsgRcvr::bHandleDisconnectDeviceRespMsg
***************************************************************************/
t_Bool spi_tclDiPODeviceMsgRcvr::bHandleDisconnectDeviceRespMsg(tShlMessage *poMessage)
{
   ETG_TRACE_USR1(("spi_tclDiPODeviceMsgRcvr::bHandleDisconnectDeviceRespMsg entered"));

   t_Bool bRetVal = false;
   if (NULL != poMessage)
   {
      trDisconnectDeviceRespMsg *poDisconnectDeviceRespMsg = static_cast<trDisconnectDeviceRespMsg*>(poMessage->pvBuffer);
      DisconnectDeviceRespMsg oDisconnectDeviceRespMsg;
      if (NULL != poDisconnectDeviceRespMsg)
      {
         ETG_TRACE_USR1(("spi_tclDiPODeviceMsgRcvr::bHandleDisconnectDeviceRespMsg: Device BT MAC Address:%s",
            poDisconnectDeviceRespMsg->szBTMACAddress));
         strncpy(oDisconnectDeviceRespMsg.szBTMACAddress, poDisconnectDeviceRespMsg->szBTMACAddress, MAX_STR_LEN);

         ETG_TRACE_USR1(("spi_tclDiPODeviceMsgRcvr::bHandleDisconnectDeviceRespMsg: DisconnectDevice succeeded:%d",
            ETG_ENUM(BOOL, poDisconnectDeviceRespMsg->bDisconnectResult)));
         oDisconnectDeviceRespMsg.bDisconnectSucceeded = poDisconnectDeviceRespMsg->bDisconnectResult;
      }

      spi_tclDiPOMsgQInterface* poDiPOMsgQIntf = spi_tclDiPOMsgQInterface::getInstance();
      bRetVal = poDiPOMsgQIntf->bWriteMsgToQ(&oDisconnectDeviceRespMsg, sizeof(oDisconnectDeviceRespMsg));
   }//if(NULL != poMessage)
   
   ETG_TRACE_USR1(("spi_tclDiPODeviceMsgRcvr::bHandleDisconnectDeviceRespMsg left with %d",ETG_ENUM(BOOL,bRetVal)));

   
   return bRetVal;
}


/////////////////////////////////////<EOF>//////////////////////////////////////////
