  /*!
 *******************************************************************************
 * \file         spi_tclDiPoBluetooth.cpp
 * \brief        DiPo Bluetooth class
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    DiPo Bluetooth handler class for SPI
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                           | Modifications
 10.10.2014 |  Ramya Murthy (RBEI/ECP2)         | Initial Version
 24.11.2014 |  Ramya Murthy (RBEI/ECP2)         | Implemented BT block/unblock for GM
 26.05.2015 |  Tejaswini H B(RBEI/ECP2)     | Added Lint comments to suppress C++11 Errors
 31.03.2016 |  Ramya Murthy (RBEI/ECP2)         | Implementation to delay blocking BT until availability 
                                                  of BT service (NCG3D-10797)
 13.05.2016 |  Ramya Murthy (RBEI/ECP2)         | Fix to prevent DeviceSwitch if BT connection 
                                                  is ongoing when CP device is activated (GMMY17-14606).
 23.05.2016 |  Ramya Murthy (RBEI/ECP2)         | Implementation of project specific BT unblocking strategy (NCG3D-11953)
 04.12.2018 |  Dundamma S B (RBEI/ECO2)         | Registering BT callbacks to ExtCmdBluetooth directly without the interface of common Bluetooth class

 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/

#include "RespRegister.h"
#include "spi_tclBluetoothIntf.h"
#include "spi_tclDeviceIDDataIntf.h"
#include "spi_tclDiPOManager.h"
#include "spi_tclDiPoBluetooth.h"


#include "Trace.h"
#ifdef TARGET_BUILD
   #ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
      #define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_BLUETOOTH
      #include "trcGenProj/Header/spi_tclDiPoBluetooth.cpp.trc.h"
   #endif
#endif

//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e19 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e58 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e48 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e808 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e64 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e746 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e515 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e516 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported  



/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/

/******************************************************************************
| variable definition (scope: global)
|----------------------------------------------------------------------------*/
static const t_U32 scou32BTLimModeTimeSec = 45000;

/******************************************************************************
| variable definition (scope: module-local)
|----------------------------------------------------------------------------*/

/***************************************************************************
*********************************PUBLIC*************************************
***************************************************************************/

/***************************************************************************
** FUNCTION:  spi_tclDiPoBluetooth::spi_tclDiPoBluetooth();
***************************************************************************/
spi_tclDiPoBluetooth::spi_tclDiPoBluetooth(spi_tclBluetoothIntf* poBTInterface)
      : m_cpoBTInterface(poBTInterface),
        m_rBTLimModeTimerID(0),
        m_enDeviceConnectionType(e8UNKNOWN_CONNECTION),
        m_enPendingBTLimAction(e8BT_LIM_ACTION_UNKNOWN)
{
   ETG_TRACE_USR1(("spi_tclDiPoBluetooth() entered "));
   

   SPI_NORMAL_ASSERT(NULL == m_cpoBTInterface);
   
   spi_tclDiPOManager* spoDiPOManager = spi_tclDiPOManager::getInstance();
   SPI_NORMAL_ASSERT( NULL == spoDiPOManager);
   
   if (NULL != spoDiPOManager)
   {
       spoDiPOManager->bRegisterObject((spi_tclDiPORespBluetooth*) this);
   }//if(NULL != spoDiPOManager )
       
   RespRegister *poRespRegister = RespRegister::getInstance();
   if(NULL != poRespRegister)
   {
       poRespRegister->bRegisterObject((spi_tclExtRespWiFi*)this);
       poRespRegister->bRegisterObject((spi_tclExtRespAppleDiscoverer*)this);
       poRespRegister->bRegisterObject((spi_tclExtRespBluetooth*)this);
   }
} //!end of spi_tclDiPoBluetooth()

/***************************************************************************
** FUNCTION:  spi_tclDiPoBluetooth::~spi_tclDiPoBluetooth();
***************************************************************************/
spi_tclDiPoBluetooth::~spi_tclDiPoBluetooth()
{
   ETG_TRACE_USR1(("~spi_tclDiPoBluetooth() entered "));
   m_rBTLimModeTimerID = 0;
   m_enDeviceConnectionType = e8UNKNOWN_CONNECTION;
} //!end of ~spi_tclDiPoBluetooth()

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoBluetooth::vOnSPISelectDeviceRequest(
***************************************************************************/
t_Void spi_tclDiPoBluetooth::vOnSPISelectDeviceRequest(const trSelectDeviceRequest& corfrSelectReq)
{
   ETG_TRACE_USR1(("spi_tclDiPoBluetooth::vOnSPISelectDeviceRequest:DevID:0x%x,DeviceConnectionType:%d,SelectionType:%d,DeviceCategory:%d,DeviceType:%d,SelectReason:%d",
   corfrSelectReq.m_u32DeviceHandle,ETG_ENUM(CONNECTION_TYPE,corfrSelectReq.m_enDevConnType),ETG_ENUM(CONNECTION_REQ,corfrSelectReq.m_enDevConnReq),
   ETG_ENUM(DEVICE_CATEGORY,corfrSelectReq.m_enDevCategory),ETG_ENUM(DEVICE_TYPE,corfrSelectReq.m_enDeviceType),
   ETG_ENUM(SELECT_REASON,corfrSelectReq.m_enSelectionReason)));


   ETG_TRACE_USR4(("[PARAM]::vOnSPISelectDeviceRequest: "
            "Technology = %d, CommChannel = %d, ActionState = %d, BTAddress = %s ",
            ETG_ENUM(BTSET_TECHNOLOGY, m_rBTLimitationModeInfo.enTechnology),
            ETG_ENUM(BTSET_COMM_CHN, m_rBTLimitationModeInfo.enCommChannel),
            ETG_ENUM(BTSET_LIMIT_ACTION_STATE, m_rBTLimitationModeInfo.enActionState),
            m_rBTLimitationModeInfo.szBTAddress.c_str()));


   m_enDeviceConnectionType = e8UNKNOWN_CONNECTION; //clear this variable on select request
   //! szBTDeviceName is an optional parameter and should be used when BT address is not available.
   //! In cases (name not known, BT address is available) this parameter should be an empty string
   t_String szBTDeviceName;

   if (NULL != m_cpoBTInterface)
   {
      //! Set Device status - this is in order to prevent SwitchDevice during ongoing selection
      m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_SELECTION_IN_PROGRESS);
      if(e8USB_CONNECTED == corfrSelectReq.m_enDevConnType)
      {
         //! Proceed with DiPO selection (wait for DisableBluetooth msg to disconnect BT if block is not triggered).
         m_cpoBTInterface->vSendSelectDeviceResult(true);        
      }
      else
      {
         if(0 != m_rBTLimModeTimerID) //Cancel the timer if it was started, when Selection has started
         {
            ETG_TRACE_USR4(("[DESC] spi_tclDiPoBluetooth::vOnSPISelectDeviceRequest(): Cancel BT Lim Mode Timer as Selection has started"));
            Timer* poTimer = Timer::getInstance();
            poTimer->CancelTimer(m_rBTLimModeTimerID);
            m_rBTLimModeTimerID = 0;
         }

         //BTLimitationMode status after the request in case Wireless selection.
         spi_tclExtCmdBluetoothIntf *poExtCmdBluetoothIntf = m_cpoBTInterface->poFetchExtCmdBluetoothIntf();
         if(NULL != poExtCmdBluetoothIntf)
         {
            //Get the BTMACAddress of the device and then trigger BTLimitationMode.
            t_String szBTMACAddress;
            spi_tclDeviceIDDataIntf oDataIntf;
            oDataIntf.vGetBTAddress(szBTMACAddress,corfrSelectReq.m_u32DeviceHandle);

            m_szSelectionInProgress = szBTMACAddress;
            if (false == poExtCmdBluetoothIntf->bSetBTLimitationMode(szBTMACAddress,szBTDeviceName,e8BT_TECH_CARPLAY,enGetBTCommnChannel(corfrSelectReq.m_enDevConnType),e8BT_LIM_ACTION_PREACTIVATE))
            {
                m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_DELAYED);
                m_enPendingBTLimAction = e8BT_LIM_ACTION_PREACTIVATE;
            }
         }
      }
   }
} //!end of vOnSPISelectDeviceRequest()

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoBluetooth::vOnSPISelectDeviceResponse(
***************************************************************************/
t_Void spi_tclDiPoBluetooth::vOnSPISelectDeviceResponse(const trSelectDeviceRequest& corfrSelectReq,
      tenResponseCode enRespCode)
{
   ETG_TRACE_USR1(("spi_tclDiPoBluetooth::vOnSPISelectDeviceResponse() entered "));
   SPI_INTENTIONALLY_UNUSED(corfrSelectReq);
   //! szBTDeviceName is an optional parameter and should be used when BT address is not available.
   //! In cases (name not known, BT address is available) this parameter should be an empty string
   t_String szBTDeviceName;
   //! Clear device status
   if (NULL != m_cpoBTInterface)
   {
      spi_tclExtCmdBluetoothIntf *poExtCmdBluetoothIntf = m_cpoBTInterface->poFetchExtCmdBluetoothIntf();
      m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_COMPLETE);

      t_String szSelDevBTAddress = m_cpoBTInterface->szGetSelectedDevBTAddress();
      if(IS_VALID_BT_ADDRESS(szSelDevBTAddress))
      {
         if(e8SUCCESS == enRespCode)
         {
            ETG_TRACE_ERR((" vOnSPISelectDeviceResponse: Processing DisableBT command now.. "));
            tenDiPOSessionTransport enSessionTransport = (e8USB_CONNECTED == corfrSelectReq.m_enDevConnType) ?
              e8_DIPO_SESSION_TRANSPORT_USB:e8_DIPO_SESSION_TRANSPORT_WIFI;
            vOnDisableBluetooth(szSelDevBTAddress,enSessionTransport);
         }
         else
         {
            if(m_szSelectionInProgress == szSelDevBTAddress)
            {
               m_szSelectionInProgress.clear();
            }
            if( (e8WIRELESS_CONNECTED == corfrSelectReq.m_enDevConnType) && (NULL != poExtCmdBluetoothIntf) )
            {
               //Incase of wireless connection, trigger BT Limitation Mode Deactivate request to BT to switch to IDLE mode.
               //@todo - check if we need to do this only when BTLimitationMode action is PREPARE.
               if(false == poExtCmdBluetoothIntf->bSetBTLimitationMode(
                        szSelDevBTAddress,szBTDeviceName,
                        e8BT_TECH_CARPLAY,
                        enGetBTCommnChannel(corfrSelectReq.m_enDevConnType),
                        e8BT_LIM_ACTION_DEACTIVATE))
               {
                   m_enPendingBTLimAction = e8BT_LIM_ACTION_DEACTIVATE;
                   m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_DELAYED);
               }
            }
         }
      }
      else
      {
         if( (e8WIRELESS_CONNECTED == corfrSelectReq.m_enDevConnType) && (e8FAILURE == enRespCode) && (NULL != poExtCmdBluetoothIntf) )
         {
            //@TODO - to be evaluated again. SelectDeviceResponse failure without receiving SelectDevice request in BT Submodule to be handled.
            // please also look at retreiving the Device BT MAC Address approach
            t_String szBTMACAddress;
            spi_tclDeviceIDDataIntf oDataIntf;
            oDataIntf.vGetBTAddress(szBTMACAddress,corfrSelectReq.m_u32DeviceHandle);

            if(false == poExtCmdBluetoothIntf->bSetBTLimitationMode(
                     szBTMACAddress,szBTDeviceName,
                     e8BT_TECH_CARPLAY,
                     enGetBTCommnChannel(corfrSelectReq.m_enDevConnType),
                     e8BT_LIM_ACTION_DEACTIVATE))
            {
                m_enPendingBTLimAction = e8BT_LIM_ACTION_DEACTIVATE;
                m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_DELAYED);
            }
         }
      }
   }
} //!end of vOnSPISelectDeviceResponse()

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoBluetooth::vOnSPIDeselectDeviceRequest(
***************************************************************************/
t_Void spi_tclDiPoBluetooth::vOnSPIDeselectDeviceRequest(const trSelectDeviceRequest& corfrSelectReq)
{
   ETG_TRACE_USR1(("spi_tclDiPoBluetooth::vOnSPIDeselectDeviceRequest() entered"));
   SPI_INTENTIONALLY_UNUSED(corfrSelectReq);
   if (NULL != m_cpoBTInterface)
   {
      // Set Device status - this is in order to prevent SwitchDevice during ongoing deselection
      m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_DESELECTION_IN_PROGRESS);
      // Nothing else to be done. Simply send success result.
      m_cpoBTInterface->vSendSelectDeviceResult(true);
   }
} //!end of vOnSPIDeselectDeviceRequest()

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoBluetooth::vOnSPIDeselectDeviceResponse(
***************************************************************************/
t_Void spi_tclDiPoBluetooth::vOnSPIDeselectDeviceResponse(const trSelectDeviceRequest& corfrSelectReq,
         tenResponseCode enRespCode)
{
   ETG_TRACE_USR1(("spi_tclDiPoBluetooth::vOnSPIDeselectDeviceResponse() entered"));

   SPI_INTENTIONALLY_UNUSED(enRespCode);
   //! szBTDeviceName is an optional parameter and should be used when BT address is not available.
   //! In cases (name not known, BT address is available) this parameter should be an empty string
   t_String szBTDeviceName;

   //! Clear device status
   if (NULL != m_cpoBTInterface)
   {
      spi_tclExtCmdBluetoothIntf *poExtCmdBluetoothIntf = m_cpoBTInterface->poFetchExtCmdBluetoothIntf();
      m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_COMPLETE);

      if(NULL != poExtCmdBluetoothIntf)
      {
         /*
         In case of WirelessConnection, BTLimitationMode action should be sent based on the trigger for de-selection.
         Device switch -> DEACTIVATE_WAIT
         User Trigger and HMI Trigger -> DEACTIVATE
         Unknown -> PREPARE
         SPM State Change -> SPM_STATE_OFF.

         In case of USB sConnection, BTLimitationMode action should be De-Activate to not to reconnect to the device after sessionEnd.       
          */

         tenBTLimitationAction enBTLimAction = e8BT_LIM_ACTION_DEACTIVATE;

         if(true == corfrSelectReq.m_bisSwitchRequired)
         {
            enBTLimAction =  e8BT_LIM_ACTION_DEACTIVATE_WAIT;
         }
         else if(e8_REASON_SPM_STATE_CHANGE == corfrSelectReq.m_enSelectionReason)
         {
            enBTLimAction = e8BT_LIM_ACTION_SPM_STATE_OFF;
         }
         else if(e8USB_CONNECTED != corfrSelectReq.m_enDevConnType)
         {
            switch(corfrSelectReq.m_enSelectionReason)
            {
               case e8_RETRIAL_FAILED:
               {
                  enBTLimAction = e8BT_LIM_ACTION_PREPARE;
               }
               break;
               case e8_REASON_SPM_STATE_CHANGE:
               {
                  enBTLimAction = e8BT_LIM_ACTION_SPM_STATE_OFF;
               }
               break;

               case e8_REASON_HMI_SELECT_SPI_AUTO_DESELECT_TRIGGER:
                  //! Do nothing
                  break;

               case e8_REASON_HMI_TRIGGER:
               case e8_REASON_USER_TRIGGER:
               case e8_HU_WIFI_AP_OFF:
               case e8_FACTORY_RESET:
               case e8_REASON_UNKNOWN:
               case e8_STATION_DISCONNECTED:
               default:
               {
                  enBTLimAction = e8BT_LIM_ACTION_DEACTIVATE;
               }

               break;
            }
         }

         //! for the select reason - e8_REASON_HMI_SELECT_SPI_AUTO_DESELECT_TRIGGER, spi_tclWifi will trigger the PREPARE as it has the Device Details
         //! that needs to SELECTED further
         if(corfrSelectReq.m_enSelectionReason != e8_REASON_HMI_SELECT_SPI_AUTO_DESELECT_TRIGGER)
         {
            if(false == poExtCmdBluetoothIntf->bSetBTLimitationMode(
                     m_cpoBTInterface->szGetSelectedDevBTAddress(),szBTDeviceName,
                     e8BT_TECH_CARPLAY,
                     enGetBTCommnChannel(corfrSelectReq.m_enDevConnType),
                     enBTLimAction))
            {
                m_enPendingBTLimAction = enBTLimAction;
                m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_DELAYED);
            }
         }

      }
   }//if (NULL != m_cpoBTInterface)
} //!end of vOnSPIDeselectDeviceResponse()

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoBluetooth::vOnDisableBluetooth(t_String...
***************************************************************************/
t_Void spi_tclDiPoBluetooth::vOnDisableBluetooth(t_String szDiPODevBTAddress,tenDiPOSessionTransport enSessionTransport)
{
   ETG_TRACE_USR1(("spi_tclDiPoBluetooth::vOnDisableBluetooth() entered: DiPO device BTAddress = %s ",
         szDiPODevBTAddress.c_str()));

   if ((NULL != m_cpoBTInterface) && (IS_VALID_BT_ADDRESS(szDiPODevBTAddress)))
   {
      spi_tclExtCmdBluetoothIntf *poExtCmdBluetoothIntf = m_cpoBTInterface->poFetchExtCmdBluetoothIntf();
      //! Store BT address of DiPO device (since BTAddress is only received with DisableBT msg)
      t_Bool bIsBTAddrAlreadyReceived = (szDiPODevBTAddress == m_cpoBTInterface->szGetSelectedDevBTAddress());
      m_cpoBTInterface->vSetSelectedDevBTAddress(szDiPODevBTAddress);
      m_cpoBTInterface->vOnDisableBluetoothTrigger(szDiPODevBTAddress);

      //! szBTDeviceName is an optional parameter and should be used when BT address is not available.
      //! In cases (name not known, BT address is available) this parameter should be an empty string
      t_String szBTDeviceName;

      //! Notify the BT Address of CP device
      //! Note: BT Device name is not sent since it is unused.
      if (false == bIsBTAddrAlreadyReceived)
      {
         t_U32 u32DiPODevHandle = m_cpoBTInterface->u32GetSelectedDevHandle();
         m_cpoBTInterface->vSendBTDeviceInfo(u32DiPODevHandle, "", szDiPODevBTAddress);
      }

      if (
         (e8DEVICE_CHANGE_COMPLETE == m_cpoBTInterface->enGetSelectedDevStatus())
         &&
         (NULL != poExtCmdBluetoothIntf)
         )
      {
         tenDeviceConnectionType enDeviceConnectionType = e8USB_CONNECTED ;

         if(enSessionTransport == e8_DIPO_SESSION_TRANSPORT_WIFI)
         {
             enDeviceConnectionType = e8WIRELESS_CONNECTED ;
         }
         if(0 != m_rBTLimModeTimerID)//Fix for NCG3D-224591
         {
            ETG_TRACE_USR2(("vOnDisableBluetooth::cancelling BTLimitation timer as session is already active"));
            Timer* poTimer = Timer::getInstance();
            poTimer->CancelTimer(m_rBTLimModeTimerID);
            m_rBTLimModeTimerID = 0;
         }
         if(false == poExtCmdBluetoothIntf->bSetBTLimitationMode(
                   szDiPODevBTAddress, szBTDeviceName,
                   e8BT_TECH_CARPLAY,
                   enGetBTCommnChannel(enDeviceConnectionType),
                   e8BT_LIM_ACTION_ACTIVATE))
         {
             ETG_TRACE_USR2(("vOnDisableBluetooth: BT service unavailable. Will request BT Lim mode later"));
             m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_DELAYED);
             m_enDeviceConnectionType = enDeviceConnectionType;
             m_enPendingBTLimAction = e8BT_LIM_ACTION_ACTIVATE;
         }
      }//if (e8DEVICE_CHANGE_COMPLETE ==...)
      else
      {
         ETG_TRACE_USR3((" vOnDisableBluetooth: No action taken since there is an ongoing device change. "));
      }
   }//if ((NULL != m_cpoBTInterface)&&...)
}//!end of vOnDisableBluetooth()


/***************************************************************************
*********************************PRIVATE************************************
***************************************************************************/


/***************************************************************************
** FUNCTION:   t_Void spi_tclDiPoBluetooth::vDeselectDipoDevice()
***************************************************************************/
t_Void spi_tclDiPoBluetooth::vDeselectDipoDevice()
{
   ETG_TRACE_USR1(("spi_tclDiPoBluetooth::vDeselectDipoDevice() entered "));
   if (NULL != m_cpoBTInterface)
   {
      //Clear device status
      m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_COMPLETE);
      m_cpoBTInterface->vSendDeselectDeviceRequest();
   }
} //!end of vDeselectDipoDevice()

/***************************************************************************
** FUNCTION:  t_Void spi_tclDiPoBluetooth::vPostWiFiConfig()
***************************************************************************/
t_Void spi_tclDiPoBluetooth::vPostWiFiConfig(const trWiFiConfig &rfcorWiFiConfig,const std::vector<trStationInfo>& corvecStationsInfo)
{
   SPI_INTENTIONALLY_UNUSED(corvecStationsInfo);
   ETG_TRACE_USR1(("spi_tclWiFi::vPostWiFiConfig() entered"));
   m_rWiFiConfig = rfcorWiFiConfig;

   ETG_TRACE_USR1(("spi_tclDiPoBluetooth::vPostWiFiConfig szSSID = %s",m_rWiFiConfig.szSSID.c_str()));
   ETG_TRACE_USR1(("spi_tclDiPoBluetooth::vPostWiFiConfig szPassphrase = %s",m_rWiFiConfig.szPassphrase.c_str()));
   ETG_TRACE_USR1(("spi_tclDiPoBluetooth::vPostWiFiConfig enFrequency = %d",ETG_ENUM(WIFI_FREQUENCY,m_rWiFiConfig.enFrequency)));

}

/***************************************************************************
** FUNCTION:  tenBTCommunicationChannel spi_tclDiPoBluetooth::enGetBTCommnChannel(...
***************************************************************************/
tenBTCommunicationChannel spi_tclDiPoBluetooth::enGetBTCommnChannel(tenDeviceConnectionType enDeviceConnectionType)
{
   ETG_TRACE_USR1(("spi_tclDiPoBluetooth::enGetBTCommnChannel():Device Transport:%d ",ETG_ENUM(CONNECTION_TYPE,enDeviceConnectionType)));

   tenBTCommunicationChannel enChannel = e8BT_COMM_CHN_USB;

    if(e8USB_CONNECTED != enDeviceConnectionType)
    {
        enChannel = e8BT_COMM_CHN_WIFI;
    }
   return enChannel;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoBluetooth::vPostDeviceListMsg()
 ***************************************************************************/
 t_Void spi_tclDiPoBluetooth::vPostDeviceListMsg(const trDeviceInfo &rfcoDeviceInfo)
 {
    ETG_TRACE_USR4(("[PARAM] spi_tclDiPoBluetooth::vPostDeviceListMsg():Device ID:%d ",rfcoDeviceInfo.u32DeviceHandle));
    ETG_TRACE_USR4(("[PARAM] spi_tclDiPoBluetooth::vPostDeviceListMsg():Connection Status for BT :%d ",
             ETG_ENUM(CONNECTION_STATUS, rfcoDeviceInfo.rConnectionStatus.enBTConnectionStatus)));
    ETG_TRACE_USR4(("[PARAM] spi_tclDiPoBluetooth::vPostDeviceListMsg():Carplay Wireless Setting :%d ",
             ETG_ENUM(SPI_SUPPORT, rfcoDeviceInfo.rProjectionCapability.enCarplayWirelessSupport)));
    ETG_TRACE_USR4(("[PARAM] spi_tclDiPoBluetooth::vPostDeviceListMsg():Device Type :%d ",
             ETG_ENUM(DEVICE_TYPE, rfcoDeviceInfo.rProjectionCapability.enDeviceType)));
    ETG_TRACE_USR4(("[PARAM] spi_tclDiPoBluetooth::vPostDeviceListMsg(): BTAddress :%s ",
            rfcoDeviceInfo.szBTAddress.c_str()));
    ETG_TRACE_USR4(("[PARAM] spi_tclDiPoBluetooth::vPostDeviceListMsg():BT limitaiton mode action state :%d ",
             ETG_ENUM(BTSET_LIMIT_ACTION_STATE, m_rBTLimitationModeInfo.enActionState)));
    ETG_TRACE_USR4(("[PARAM] spi_tclDiPoBluetooth::vPostDeviceListMsg():BT limitaiton mode is for device with BT address %s ",
            m_rBTLimitationModeInfo.szBTAddress.c_str()));
    ETG_TRACE_USR4(("[PARAM] spi_tclDiPoBluetooth::vPostDeviceListMsg():serial number %s ",
            rfcoDeviceInfo.szSerialNumber.c_str()));

    //! szBTDeviceName is an optional parameter and should be used when BT address is not available.
    //! In cases (name not known, BT address is available) this parameter should be an empty string
    t_String szBTDeviceName;

    //! check if the device is connected over BT for CPW but carplay setting is turned OFF on phone.
    //! If so trigger deactivate to BT sothat BT can connect to classic profiles
    if((e8DEV_CONNECTED == rfcoDeviceInfo.rConnectionStatus.enBTConnectionStatus) &&
             (e8SPI_NOTSUPPORTED == rfcoDeviceInfo.rProjectionCapability.enCarplayWirelessSupport)
             && (e8_APPLE_DEVICE == rfcoDeviceInfo.rProjectionCapability.enDeviceType))
    {
       //! Check if the BT limitation mode is in prepared state for this device
       if((e8BT_LIM_ACTION_STATE_PREPARED == m_rBTLimitationModeInfo.enActionState) && (rfcoDeviceInfo.szBTAddress == m_rBTLimitationModeInfo.szBTAddress))
       {
          if(0 != m_rBTLimModeTimerID) //Cancel the timer if it was started
          {
              ETG_TRACE_USR4(("[DESC] spi_tclDiPoBluetooth::vPostDeviceListMsg(): Cancel BT Lim Mode Timer as CPW is not supported"));
              Timer* poTimer = Timer::getInstance();
              poTimer->CancelTimer(m_rBTLimModeTimerID);
              m_rBTLimModeTimerID = 0;
          }
          if (NULL != m_cpoBTInterface)
          {
              spi_tclExtCmdBluetoothIntf *poExtCmdBluetoothIntf = m_cpoBTInterface->poFetchExtCmdBluetoothIntf();
              if(false == poExtCmdBluetoothIntf->bSetBTLimitationMode(
                   rfcoDeviceInfo.szBTAddress, szBTDeviceName,
                e8BT_TECH_CARPLAY,
                enGetBTCommnChannel(e8WIRELESS_CONNECTED),
                e8BT_LIM_ACTION_DEACTIVATE))
              {
                  m_enPendingBTLimAction = e8BT_LIM_ACTION_DEACTIVATE;
                  m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_DELAYED);
              }
          }

       }
    }
    else if((e8DEV_CONNECTED == rfcoDeviceInfo.rConnectionStatus.enBTConnectionStatus) &&
            (e8SPI_SUPPORTED == rfcoDeviceInfo.rProjectionCapability.enCarplayWirelessSupport) &&
            (e8_APPLE_DEVICE == rfcoDeviceInfo.rProjectionCapability.enDeviceType) &&
            ("invalid" != rfcoDeviceInfo.szSerialNumber) &&
            (rfcoDeviceInfo.szBTAddress == m_rBTLimitationModeInfo.szBTAddress))
    {
        if(0 != m_rBTLimModeTimerID) //Cancel the timer if it was started
        {
            ETG_TRACE_USR4(("[DESC] spi_tclDiPoBluetooth::vPostDeviceListMsg(): Cancel BT Lim Mode Timer as CPW is supported"));
            Timer* poTimer = Timer::getInstance();
            poTimer->CancelTimer(m_rBTLimModeTimerID);
            m_rBTLimModeTimerID = 0;
        }
    }
 }

 //!Static
 /***************************************************************************
  ** FUNCTION:  spi_tclDiPoBluetooth::bBTLimModeTimerCb
  ***************************************************************************/
 t_Bool spi_tclDiPoBluetooth::bBTLimModeTimerCb(timer_t rTimerID, t_Void *pvObject,
          const t_Void *pvUserData)
 {
    SPI_INTENTIONALLY_UNUSED(pvUserData);
    ETG_TRACE_USR1(("spi_tclDiPoBluetooth::bBTLimModeTimerCb timer expired!!"));
    spi_tclDiPoBluetooth *poDiPoBluetooth = static_cast<spi_tclDiPoBluetooth*>(pvObject);
    Timer* poTimer = Timer::getInstance();
    if(NULL != poDiPoBluetooth && NULL != poTimer)
    {
        poTimer->CancelTimer(rTimerID);
        poDiPoBluetooth->m_rBTLimModeTimerID = 0;
        if (NULL != poDiPoBluetooth->m_cpoBTInterface)
        {
           //! szBTDeviceName is an optional parameter and should be used when BT address is not available.
           //! In cases (name not known, BT address is available) this parameter should be an empty string
           t_String szBTDeviceName;
            spi_tclExtCmdBluetoothIntf *poExtCmdBluetoothIntf = poDiPoBluetooth->m_cpoBTInterface->poFetchExtCmdBluetoothIntf();

            if(false == poExtCmdBluetoothIntf->bSetBTLimitationMode(
               poDiPoBluetooth->m_rBTLimitationModeInfo.szBTAddress,
               szBTDeviceName,
               e8BT_TECH_CARPLAY,
               poDiPoBluetooth->enGetBTCommnChannel(e8WIRELESS_CONNECTED),
               e8BT_LIM_ACTION_DEACTIVATE))
            {
                poDiPoBluetooth->m_enPendingBTLimAction = e8BT_LIM_ACTION_DEACTIVATE;
                poDiPoBluetooth->m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_DELAYED);
            }
        }
    }
    return true;
 }

 /***************************************************************************
  ** FUNCTION:  t_Void spi_tclDiPoBluetooth::vPostBTServiceAvailabilityMsg()
  ***************************************************************************/
 t_Void spi_tclDiPoBluetooth::vPostBTServiceAvailabilityMsg(t_Bool bServiceAvailable)
 {
    ETG_TRACE_USR1(("spi_tclDiPoBluetooth::vPostBTServiceAvailabilityMsg entered bServiceAvailable = %d",ETG_ENUM(BOOL,bServiceAvailable)));
    if( (true == bServiceAvailable) &&
        (NULL != m_cpoBTInterface) &&
        (e8DEVICE_CHANGE_DELAYED == m_cpoBTInterface->enGetSelectedDevStatus()) &&
        (e8DEV_TYPE_DIPO == m_cpoBTInterface->enGetSelectedDeviceCategory()) &&
        (e8BT_LIM_ACTION_UNKNOWN != m_enPendingBTLimAction))
    {
       //! szBTDeviceName is an optional parameter and should be used when BT address is not available.
       //! In cases (name not known, BT address is available) this parameter should be an empty string
       t_String szBTDeviceName;

        spi_tclExtCmdBluetoothIntf *poExtCmdBluetoothIntf = m_cpoBTInterface->poFetchExtCmdBluetoothIntf();
        if(NULL != poExtCmdBluetoothIntf)
        {
           t_Bool sbBTLimModeResult = poExtCmdBluetoothIntf->bSetBTLimitationMode(
               m_cpoBTInterface->szGetSelectedDevBTAddress(),szBTDeviceName,
               e8BT_TECH_CARPLAY,
               enGetBTCommnChannel(m_enDeviceConnectionType),
               m_enPendingBTLimAction);
           if(true == sbBTLimModeResult)
           {
               m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_COMPLETE);
           }
           else
           {
               m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_DELAYED);
           }
           m_enDeviceConnectionType = e8UNKNOWN_CONNECTION;
        }
    }
 }

 /***************************************************************************
  ** FUNCTION:  t_Void spi_tclDiPoBluetooth::vPostBTLimitationModeMsg()
  ***************************************************************************/
 t_Void spi_tclDiPoBluetooth::vPostBTLimitationModeMsg(trBTLimitationModeInfo rBTLimitationModeInfo)
 {
    ETG_TRACE_USR4(("[PARAM]::spi_tclDiPoBluetooth::vPostBTLimitationModeMsg: entered "
             "Technology = %d, CommChannel = %d, ActionState = %d, BTAddress = %s ",
             ETG_ENUM(BTSET_TECHNOLOGY, rBTLimitationModeInfo.enTechnology),
             ETG_ENUM(BTSET_COMM_CHN, rBTLimitationModeInfo.enCommChannel),
             ETG_ENUM(BTSET_LIMIT_ACTION_STATE, rBTLimitationModeInfo.enActionState),
             rBTLimitationModeInfo.szBTAddress.c_str()));

    m_rBTLimitationModeInfo = rBTLimitationModeInfo;

    if (
       (NULL != m_cpoBTInterface)
       &&
       (e8BT_TECH_CARPLAY == rBTLimitationModeInfo.enTechnology)
       &&
       (e8BT_COMM_CHN_BLUETOOTH != rBTLimitationModeInfo.enCommChannel)
       )
    {

       //! Update the BT Limitation Mode to the Common BT
       m_cpoBTInterface->vSetBTLimitationModeResult(rBTLimitationModeInfo);

       switch (rBTLimitationModeInfo.enActionState)
       {
          case e8BT_LIM_ACTION_STATE_ACTIVE:
          {
             ETG_TRACE_USR2(("[DESC]::vPostBTLimitationModeMsg: CarPlay BT mode activation is successful."));
          }
          break;
          case e8BT_LIM_ACTION_STATE_ERROR:
          case e8BT_LIM_ACTION_STATE_ERROR_USER_DENIED:
          {
             ETG_TRACE_USR2(("[DESC]::vOnBTLimitationMode:"
                      "CarPlay BT mode activation is unsuccessful. Deactivating CarPlay device. If the Selecti in Progress or Selected"));

             //! BT Limitation Action ERROR is reported, before Carplay Wireless Feasibility is reported by MediaPlayer
             //! Cancle the BT Limitation Timer - Otherwise the timer will be continued untill it elapses
             if(0 != m_rBTLimModeTimerID) //Cancel the timer if it was started
             {
                ETG_TRACE_USR4(("[DESC] spi_tclDiPoBluetooth::vOnBTLimitationMode(): Cancel BT Lim Mode Timer as"
                         "BT Limitation ERROR is reported before Mediaplayer reported Feasibility"));
                Timer* poTimer = Timer::getInstance();
                poTimer->CancelTimer(m_rBTLimModeTimerID);
                m_rBTLimModeTimerID = 0;
             }
             else if( (m_szSelectionInProgress == rBTLimitationModeInfo.szBTAddress) &&
                      (e8DEV_TYPE_DIPO == m_cpoBTInterface->enGetSelectedDeviceCategory()) )
             {
                m_szSelectionInProgress.clear();
                m_cpoBTInterface->vSendSelectDeviceResult(false);
             }
             else if( (rBTLimitationModeInfo.szBTAddress == m_cpoBTInterface->szGetSelectedDevBTAddress()) &&
                      (e8DEV_TYPE_DIPO == m_cpoBTInterface->enGetSelectedDeviceCategory()) )
             {
                vDeselectDipoDevice();
             }
          }
          break;
          case e8BT_LIM_ACTION_STATE_PREACTIVATING:
          {
             //Nothing to do
          }
          break;
          case e8BT_LIM_ACTION_STATE_PREACTIVATED:
          {
             if( (m_szSelectionInProgress == rBTLimitationModeInfo.szBTAddress) &&
                      (e8DEV_TYPE_DIPO == m_cpoBTInterface->enGetSelectedDeviceCategory()))
             {
                spi_tclDeviceIDDataIntf oDataIntf;
                m_cpoBTInterface->vTriggerWiFiCredentialExchange(oDataIntf.u32GetDeviceIDFromBTAddress(m_szSelectionInProgress));
                m_szSelectionInProgress.clear();
                m_cpoBTInterface->vSendSelectDeviceResult(true);
             }
          }
          break;
          case e8BT_LIM_ACTION_STATE_PREPARED:
          {
               if(0 != m_rBTLimModeTimerID)
               {
                  vCanclePreviousBTLimTimer();
               }

               ETG_TRACE_USR2(("[DESC]::vOnBTLimitationMode: Starting BT Lim Mode timer"));
               Timer* poTimer = Timer::getInstance();
               if(NULL != poTimer)
               {
                  timer_t rTimerID;
                  //! Start BT Lim Mode timer
                  poTimer->StartTimer(rTimerID, scou32BTLimModeTimeSec, 0, this,
                           &spi_tclDiPoBluetooth::bBTLimModeTimerCb, NULL);
                  m_rBTLimModeTimerID = rTimerID;
               }
            
          }
          break;
          default:
             //! Add code
             break;
       }//switch (corfrBTLimitationMode.enActionState)
    }
 }

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclDiPoBluetooth::vSetBTLimitationMode()
 ***************************************************************************/
t_Void spi_tclDiPoBluetooth::vSetBTLimitationMode(const t_String& rfcszBTAddress, 
         const t_String& rfcszBTDeviceName, tenBTTechnology enTechnology,
         tenBTCommunicationChannel enCommChannel, tenBTLimitationAction enAction)
{
   ETG_TRACE_USR1(("spi_tclDiPoBluetooth::vSetBTLimitationMode() entered"));
   t_Bool bRetval = false;

   if(NULL != m_cpoBTInterface)
   {
      spi_tclExtCmdBluetoothIntf *poExtCmdBluetoothIntf = m_cpoBTInterface->poFetchExtCmdBluetoothIntf();
      if(NULL != poExtCmdBluetoothIntf)
      {
          bRetval = poExtCmdBluetoothIntf->bSetBTLimitationMode(rfcszBTAddress,rfcszBTDeviceName,enTechnology,enCommChannel,enAction);
         if(false == bRetval)
         {
             m_enPendingBTLimAction = enAction;
             m_cpoBTInterface->vSetSelectedDevStatus(e8DEVICE_CHANGE_DELAYED);
         }
      }
   }
   ETG_TRACE_USR2(("[DESC]::vSetBTLimitationMode: left with %d",ETG_ENUM(BOOL, bRetval)));
}
 

/***************************************************************************
** FUNCTION:  t_Bool spi_tclDiPoBluetooth::vOnAutoSelectionFailureForWireless()
***************************************************************************/
t_Void spi_tclDiPoBluetooth::vOnAutoSelectionFailureForWireless(const t_String& rfcszBTAddress, tenBTTechnology enTechnology,
                                      tenBTCommunicationChannel enCommChannel, tenBTLimitationAction enAction)
{
   ETG_TRACE_USR1(("spi_tclDiPoBluetooth::vOnAutoSelectionFailureForWireless() entered"));
   //! szBTDeviceName is an optional parameter and should be used when BT address is not available.
   //! In cases (name not known, BT address is available) this parameter should be an empty string
   t_String szBTDeviceName;
   if((m_rBTLimitationModeInfo.szBTAddress == rfcszBTAddress) && (m_rBTLimitationModeInfo.enActionState == e8BT_LIM_ACTION_STATE_PREPARED))
   {
      vSetBTLimitationMode(rfcszBTAddress,szBTDeviceName,enTechnology,enCommChannel,enAction);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoBluetooth::vCanclePreviousBTLimTimer(...)
 ***************************************************************************/
t_Void spi_tclDiPoBluetooth::vCanclePreviousBTLimTimer()
{
   ETG_TRACE_USR3(("spi_tclDiPoBluetooth::vCanclePreviousBTLimTimer Entered"));
   Timer* poTimer = Timer::getInstance();
   poTimer->CancelTimer(m_rBTLimModeTimerID);
   m_rBTLimModeTimerID = 0;
}

//lint -restore
///////////////////////////////////////////////////////////////////////////////
// <EOF>
