/*!
 *******************************************************************************
 * \file             spi_tclDiPoAudio.h
 * \brief            Implements the Audio functionality for Digital IPOd Out using
 interface to Real VNC SDK through VNC Wrapper.
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3 Projects
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Audio Implementation for Digital IPOd Out
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                          | Modifications
 29.10.2013 |  Hari Priya E R(RBEI/ECP2)       | Initial Version
 04.04.2014 |  Shihabudheen P M(RBEI/ECP2)     | Added bStartAudio()
 26.05.2015 |  Tejaswini H B(RBEI/ECP2)        | Added Lint comments to suppress C++11 Errors
 17.07.2015 |  Sameer Chandra                   | Memory leak fix
 23.05.2016 |  Ramya Murthy                    | Included ECNR handling

 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "Trace.h"
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_AUDIO
#include "trcGenProj/Header/spi_tclDiPoAudio.cpp.trc.h"
#endif

#include "SPITypes.h"
#include "DiPOTypes.h"
#include "DiPOWrapperTypes.h"
#include "spi_tclDiPOCmdAudio.h"
#include "spi_tclDiPOManager.h"
#include "spi_tclDiPoAudio.h"
#include "spi_tclDiPOCmdRsrcMngr.h"

//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e19 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e58 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e48 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e808 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e64 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e746 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e515 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e516 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported	

static const t_Double scodAudioUnduck = 1.0;
//static const t_Double scodAudioDuckMaxdB   = 120.0;
//static const t_Double scodAudioDuckDefdB   = 41.0;
static const t_Double scodAudioDuckMindB = 0.0;
static const t_U16 scou16AudioRampDefDuration = 1000;  //In msec

static const t_U8 scou8AudioDuckMaxdB = 120;
static const t_U8 scou8AudioDuckDefdB = 41;

//! Timer ID and flag of timer for restoration of last main audio source after Phone/Siri
static timer_t srAudRestoreTimerID;
static t_Bool bAudRestoreTimerRunning = false;

//! Timer ID and flag of timer for dellocation of Phone/Siri/Alert sources when resource is with CarPlay
static timer_t srAudReleaseTimerID;
static t_Bool bAudRelTimerRunning = false;
static spi_tclDiPOCmdAudio* spoCmdAudio = NULL;

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/
typedef std::map<t_U16, tenAudioSamplingRate>::iterator tAudSampleRateMapItr;

/***************************************************************************
 ** FUNCTION:  spi_tclDiPoAudio::spi_tclDiPoAudio()
 ***************************************************************************/
spi_tclDiPoAudio::spi_tclDiPoAudio() :
                  spi_tclAudioDevBase(),
                  m_enCurrentAudioMode(e8DIPO_ENTITY_NA),
                  m_enCurrAudioMainSrc(e8AUD_INVALID),
                  m_enCurrAudioAltSrc(e8AUD_INVALID),
                  m_enTransferType(e8DIPO_TRANSFERTYPE_NA),
                  m_u32DeviceHandle(0),
                  m_bIsSessionActive(false),
                  m_enAudioSamplingRate(e8AUD_SAMPLERATE_DEFAULT),
                  m_bAudDuckEnabled(false),
                  m_enMediaStreamState(e8CP_AUD_STREAM_CLOSED),
                  m_poDiPOCmdRsrcMngr(NULL)
{
   ETG_TRACE_USR1(("spi_tclDiPoAudio() entered "));
   //! populate the sample rates map
   m_mapAudioSamplingRate[0] = e8AUD_SAMPLERATE_DEFAULT;
   m_mapAudioSamplingRate[8000] = e8AUD_SAMPLERATE_8KHZ;
   m_mapAudioSamplingRate[16000] = e8AUD_SAMPLERATE_16KHZ;
   m_mapAudioSamplingRate[24000] = e8AUD_SAMPLERATE_24KHZ;

   spi_tclDiPOManager* spoDiPOManager = spi_tclDiPOManager::getInstance();
   if (NULL != spoDiPOManager)
   {
      spoCmdAudio = spoDiPOManager->poGetDiPOAudioInstance();
      m_poDiPOCmdRsrcMngr = spoDiPOManager->poGetDiPORsrcMngrInstance();
      spoDiPOManager->bRegisterObject((spi_tclDiPORespAudio*) this);
      spoDiPOManager->bRegisterObject((spi_tclDiPORespSession*) this);
      spoDiPOManager->bRegisterObject((spi_tclDiPORespRsrcMngr*) this);
   }  //if (NULL != spoDiPOManager)

      // Initializations to be done which are not specific to device connected/selected
}  //spi_tclDiPoAudio::spi_tclDiPoAudio()

/***************************************************************************
 ** FUNCTION:  spi_tclDiPoAudio::~spi_tclDiPoAudio()
 ***************************************************************************/
spi_tclDiPoAudio::~spi_tclDiPoAudio()
{
   ETG_TRACE_USR1(("~spi_tclDiPoAudio() entered "));

   m_u32DeviceHandle = 0;
   m_bIsSessionActive = false;
   m_bAudDuckEnabled = false;
   m_enCurrAudioMainSrc = e8AUD_INVALID;
   m_enCurrAudioAltSrc = e8AUD_INVALID;
   m_enCurrentAudioMode = e8DIPO_ENTITY_NA;
   m_enTransferType = e8DIPO_TRANSFERTYPE_NA;
}  //spi_tclDiPoAudio::~spi_tclDiPoAudio()

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAudio::vRegisterCallbacks()
 ***************************************************************************/
t_Void spi_tclDiPoAudio::vRegisterCallbacks(trAudioCallbacks &rfrAudCallbacks)
{
   ETG_TRACE_USR1(("spi_tclDiPoAudio::vRegisterCallbacks entered "));
   m_rAudCallbacks = rfrAudCallbacks;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAudio::bStartAudio()
 ***************************************************************************/
t_Bool spi_tclDiPoAudio::bStartAudio(t_U32 u32DeviceId, t_String szAudioDev, tenAudioDir enAudDir)
{
   /*lint -esym(40,fvStartAudioResp)fvStartAudioResp Undeclared identifier */

   ETG_TRACE_USR1(("spi_tclDiPoAudio::bStartAudio entered for Audio Source %d with Audio Device - %s ", ETG_ENUM(SPI_AUDIO_DIRECTION,
            enAudDir), szAudioDev.c_str()));
   SPI_INTENTIONALLY_UNUSED(u32DeviceId);
   t_Bool bRetVal = true;

   if (0 != m_u32DeviceHandle)
   {
      m_oStreamStateLock.s16Lock();
      t_Bool bSendAllocResponse =
               (e8AUD_MAIN_OUT == enAudDir) ? (e8CP_AUD_STREAM_CLOSED != m_enMediaStreamState) : (true);
      m_enMediaStreamState =
               ((e8AUD_MAIN_OUT == enAudDir) && (e8CP_AUD_STREAM_CLOSED != m_enMediaStreamState)) ?
                        (e8CP_AUD_STREAMING) : (m_enMediaStreamState);
      ETG_TRACE_USR4(("[PARAM]::bStartAudio: New MediaStreamState = %d", m_enMediaStreamState));
      m_oStreamStateLock.vUnlock();

      if ((e8AUD_DUCK != enAudDir) && (NULL != spoCmdAudio) && (bSendAllocResponse))
      {
         tenAudioStreamType enAudioStreamType =
                  (e8AUD_MAIN_OUT == enAudDir || e8AUD_ALERT_OUT == enAudDir) ? e8MAIN_AUDIO : e8ALTERNATE_AUDIO;
         bRetVal = spoCmdAudio->bSendAudioAllocResponse(e8_DIPO_AUDIO_ALLOC_RESP_AUDIO,
                  enAudioStreamType,
                  szAudioDev,
                  "");
      }
   }  //if (0 != m_u32DeviceHandle)

   //@Note: Respond to SA_ON before sending new request to audio manager
   if (NULL != m_rAudCallbacks.fvStartAudioResp)
   {
      (m_rAudCallbacks.fvStartAudioResp)(enAudDir, bRetVal);
   }

   if (0 == m_u32DeviceHandle)
   {
      if (m_bIsLastModeSupported)
      {
         (e8AUD_MAIN_OUT == enAudDir) ? (vMuteAudio(true)) : (vDeactivateChannel(enAudDir));
      }
      else
      {
         vDeactivateChannel(enAudDir);
      }
   }
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAudio::bStartAudio()
 ***************************************************************************/
t_Bool spi_tclDiPoAudio::bStartAudio(t_U32 u32DeviceId, t_String szOutputAudioDev, t_String szInputAudioDev,
         tenAudioDir enAudDir)
{
   SPI_INTENTIONALLY_UNUSED(u32DeviceId);
   ETG_TRACE_USR1(("spi_tclDiPoAudio::bStartAudio entered for Audio Source %d ", ETG_ENUM(SPI_AUDIO_DIRECTION, enAudDir)));
   ETG_TRACE_USR4(("Audio Output Device szOutputAudioDev =%s ", szOutputAudioDev.c_str()));
   ETG_TRACE_USR4(("Audio Input Device szInputAudioDev =%s ", szInputAudioDev.c_str()));
   t_Bool bRetVal = false;

   if ((0 != m_u32DeviceHandle) && (NULL != spoCmdAudio))
   {
      tenAudioStreamType enAudioStreamType = e8MAIN_AUDIO;
      bRetVal = spoCmdAudio->bSendAudioAllocResponse(e8_DIPO_AUDIO_ALLOC_RESP_AUDIO_IN,
               enAudioStreamType,
               szOutputAudioDev,
               szInputAudioDev);
   }

   //@Note: Respond to SA_ON before sending new request to audio manager
   if (NULL != m_rAudCallbacks.fvStartAudioResp)
   {
      (m_rAudCallbacks.fvStartAudioResp)(enAudDir, bRetVal);
   }
   if (NULL != m_rAudCallbacks.fvStartAudioIn)
   {
      (m_rAudCallbacks.fvStartAudioIn)(enAudDir);
   }

   if (0 == m_u32DeviceHandle)
   {
      vDeactivateChannel(enAudDir);
   }

   return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAudio::bStopAudio()
 ***************************************************************************/
t_Void spi_tclDiPoAudio::vStopAudio(t_U32 u32DeviceId, tenAudioDir enAudDir, t_Bool bIsPaused)
{
   /*lint -esym(40,fvStopAudioResp)fvStopAudioResp Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclDiPoAudio::vStopAudio entered: DeviceId = 0x%x, AudDir = %d ", u32DeviceId, ETG_ENUM(SPI_AUDIO_DIRECTION,
            enAudDir)));

   t_Bool bRetVal = false; //TODO

   if ((NULL != spoCmdAudio)  && (0 != m_u32DeviceHandle))
   {
      if ((true == bIsPaused) && (e8AUD_PHONE_IN == enAudDir))
      {
         spoCmdAudio->vSendStopAudioRequest(e8DEALLOCATION_REQUEST, true);
      }
      else if ((e8AUD_MAIN_OUT == enAudDir))
      {
         spoCmdAudio->vSendStopAudioRequest(e8DEALLOCATION_REQUEST, false);
      }
   }
   else
   {
      bRetVal = true;
   }

   if (((e8AUD_PHONE_IN == enAudDir) || (e8AUD_VR_IN == enAudDir) || (e8AUD_DEFAULT == enAudDir))
            && (NULL != m_rAudCallbacks.fvStopAudioIn))
   {
      (m_rAudCallbacks.fvStopAudioIn)(enAudDir);
   }
   if (NULL != m_rAudCallbacks.fvStopAudioResp)
   {
      (m_rAudCallbacks.fvStopAudioResp)(enAudDir, bRetVal);
   }

   if ((0 == m_u32DeviceHandle) && (false == bIsPaused) && (e8AUD_MAIN_OUT == enAudDir))
   {
      vDeactivateChannel(enAudDir);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAudio::bSelectAudioDevice()
 ***************************************************************************/
t_Bool spi_tclDiPoAudio::bSelectAudioDevice(t_U32 u32DeviceId)
{
   /*lint -esym(40,fvSelectDeviceResp)fvSelectDeviceResp Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclDiPoAudio::bSelectAudioDevice entered: DeviceId = 0x%x ", u32DeviceId));

   if (NULL != (m_rAudCallbacks.fvSelectDeviceResp))
   {
      (m_rAudCallbacks.fvSelectDeviceResp)(u32DeviceId, e8DEVCONNREQ_SELECT, true);
   }
   return true;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAudio::vDeselectAudioDevice()
 ***************************************************************************/
t_Void spi_tclDiPoAudio::vDeselectAudioDevice(t_U32 u32DeviceId)
{
   /*lint -esym(40,fvSelectDeviceResp)fvSelectDeviceResp Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclDiPoAudio::vDeselectAudioDevice entered: DeviceId = 0x%x ", u32DeviceId));

   //! @Note: Clear device handle at start of deselection so that if any
   //! audio channel gets allocated during deselection will be handled correctly
   m_u32DeviceHandle = 0;

   if (NULL != (m_rAudCallbacks.fvSelectDeviceResp))
   {
      (m_rAudCallbacks.fvSelectDeviceResp)(u32DeviceId, e8DEVCONNREQ_DESELECT, true);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void  spi_tclDiPoAudio::vUpdateDeviceSelection()
 ***************************************************************************/
t_Void spi_tclDiPoAudio::vUpdateDeviceSelection(t_U32 u32DevID, tenDeviceCategory enDevCat,
         tenDeviceConnectionReq enDeviceConnReq, tenResponseCode enRespCode, tenErrorCode enErrorCode)
{
   m_oSessionStateLock.s16Lock();

   ETG_TRACE_USR4(("spi_tclDiPoAudio::vUpdateDeviceSelection entered"));
   SPI_INTENTIONALLY_UNUSED(enDevCat);
   SPI_INTENTIONALLY_UNUSED(enErrorCode);

   if ((e8DEVCONNREQ_SELECT == enDeviceConnReq) && (e8SUCCESS == enRespCode))
   {
      m_u32DeviceHandle = u32DevID;
   }
   else if (e8DEVCONNREQ_DESELECT == enDeviceConnReq)
   {
      vHandleSessionEnd();
   }

   m_oSessionStateLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAudio::bIsAudioLinkSupported()
 ***************************************************************************/
t_Bool spi_tclDiPoAudio::bIsAudioLinkSupported(t_U32 u32DeviceId, tenAudioLink enLink)
{
   ETG_TRACE_USR1(("spi_tclDiPoAudio::bIsAudioLinkSupported entered "));
   SPI_INTENTIONALLY_UNUSED(u32DeviceId);
   SPI_INTENTIONALLY_UNUSED(enLink);
   return true;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiPoAudio::bSetAudioBlockingMode()
 ***************************************************************************/
t_Bool spi_tclDiPoAudio::bSetAudioBlockingMode(const t_U32 cou32DevId, const tenBlockingMode coenBlockingMode)
{
   //always return an error, if the audio blocking request comes for DiPo device.
   SPI_INTENTIONALLY_UNUSED(cou32DevId);
   SPI_INTENTIONALLY_UNUSED(coenBlockingMode);
   return false;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiPoAudio::vOnAudioError()
 ***************************************************************************/
t_Void spi_tclDiPoAudio::vOnAudioError(tenAudioDir enAudDir, tenAudioError enAudioError)
{
   ETG_TRACE_USR1((" spi_tclDiPoAudio::vOnAudioError: enAudDir = %d enAudioError =%d ", ETG_ENUM(SPI_AUDIO_DIRECTION,
            enAudDir), ETG_ENUM(AUDIO_ERROR, enAudioError)));
   switch (enAudioError)
   {
      case e8_AUDIOERROR_AVACTIVATION:
      case e8_AUDIOERROR_ALLOCATE:
      case e8_AUDIOERROR_STARTSOURCEACT:
      {
         if (NULL != spoCmdAudio)
         {
            tenAudioStreamType enAudioStreamType =
                     (e8AUD_MAIN_OUT == enAudDir || e8AUD_ALERT_OUT == enAudDir) ? e8MAIN_AUDIO : e8ALTERNATE_AUDIO;
            spoCmdAudio->bSendAudioAllocResponse(e8_DIPO_AUDIO_ALLOC_RESP_ERROR, enAudioStreamType, "AdevSpiDummy", "");
         }
         break;
      }
      default:
      {
         ETG_TRACE_USR4(("Audio Error Not Handled"));
      }
   }   //switch(enAudioError)
}

/***************************************************************************
 ** FUNCTION:  spi_tclDiPoAudio::bInitializeAudioPlayback(...)
 ***************************************************************************/
t_Bool spi_tclDiPoAudio::bInitializeAudioPlayback(t_U32 u32DeviceId,tenAudioDir enAudDir, tenAudioSamplingRate enSamplingRate,tenAudioSamplingRate enNativeSamplingRate)
{
   /*lint -esym(40,fvInitializeAudioIn)fvInitializeAudioIn Undeclared identifier */
   /*lint -esym(40,fvSetAlsaDevice)fvSetAlsaDevice Undeclared identifier */
   SPI_INTENTIONALLY_UNUSED(u32DeviceId);

   ETG_TRACE_USR1(("spi_tclDiPoAudio::bInitializeAudioPlayback entered: AudDir %d, SamplingRate %d ", ETG_ENUM(SPI_AUDIO_DIRECTION,
            enAudDir), ETG_ENUM(SAMPLING_RATE, enSamplingRate)));

   //! Initialize ECNR
   if (((e8AUD_PHONE_IN == enAudDir) || (e8AUD_VR_IN == enAudDir) || (e8AUD_DEFAULT == enAudDir))
            && (NULL != m_rAudCallbacks.fvInitializeAudioIn) && (NULL != m_rAudCallbacks.fvSetAlsaDevice))
   {
      //! Initialise ECNR
      t_Bool bCPWStatus = (NULL != m_poAudioSettings)? m_poAudioSettings->bReadCPWStatus():false;
      ETG_TRACE_USR4(("[DEBUG:]spi_tclDiPoAudio::bCPWStatus:%d",bCPWStatus));

      tenAudioInDataSet enDataSet = ((e8AUD_PHONE_IN == enAudDir) && (true == bCPWStatus))?
            (enGetAudioInDataSet(enAudDir,enNativeSamplingRate)) :
            (enGetAudioInDataSet(enAudDir,enSamplingRate));

      m_rAudCallbacks.fvInitializeAudioIn(enAudDir,enDataSet);
      m_rAudCallbacks.fvSetAlsaDevice(enAudDir);
   }

   if ((e8AUD_MAIN_OUT == enAudDir) && (NULL != m_poAudioSettings))
   {
      // Set the audio last mode as projection/native in datapool when projection media is active based on last mode configuration
      t_Bool bRetValue = false;
      bRetValue = m_poAudioSettings->bWriteAudioLastMode(m_bIsLastModeSupported);
      ETG_TRACE_USR2(("[DESC]::bInitializeAudioPlayback - Audio last mode is written to datapool: %d", ETG_ENUM(BOOL,
               bRetValue)));
   }

   return true;

}   //t_Bool spi_tclDiPoAudio::bInitializeAudioPlayback(t_U32 u32DeviceId)

/***************************************************************************
 ** FUNCTION:  spi_tclDiPoAudio::bFinalizeAudioPlayback(...)
 ***************************************************************************/
t_Bool spi_tclDiPoAudio::bFinalizeAudioPlayback(t_U32 u32DeviceId, tenAudioDir enAudDir)
{
   /*lint -esym(40,fvUninitializeAudioIn)fvUninitializeAudioIn Undeclared identifier */
   SPI_INTENTIONALLY_UNUSED(u32DeviceId);

   ETG_TRACE_USR1(("spi_tclDiPoAudio::bFinalizeAudioPlayback entered "));

   //! Uninitialize ECNR
   if (((e8AUD_PHONE_IN == enAudDir) || (e8AUD_VR_IN == enAudDir) || (e8AUD_DEFAULT == enAudDir))
            && (NULL != m_rAudCallbacks.fvUninitializeAudioIn))
   {
      m_rAudCallbacks.fvUninitializeAudioIn(enAudDir);
   }

   return true;

}   //t_Bool spi_tclDiPoAudio::bFinalizeAudioPlayback(t_U32 u32DeviceId)

/***************************************************************************
 ** FUNCTION:  spi_tclDiPoAudio::enGetAudioDataSet(...)
 ***************************************************************************/
tenAudioInDataSet spi_tclDiPoAudio::enGetAudioInDataSet(tenAudioDir enAudioDir,
         tenAudioSamplingRate enAudioSamplingRate)
{
   tenAudioInDataSet enAudioDataSet = e32_AUDIOIN_DATASET_UNKNOWN;
   switch (enAudioDir)
   {
      case e8AUD_PHONE_IN:
         enAudioDataSet =
                  (e8AUD_SAMPLERATE_8KHZ == enAudioSamplingRate) ?
                           (e32_AUDIOIN_DATASET_PHONE_NB) : (e32_AUDIOIN_DATASET_PHONE_WB);
         break;
      case e8AUD_VR_IN:
         enAudioDataSet =
                  (e8AUD_SAMPLERATE_8KHZ == enAudioSamplingRate) ?
                           (e32_AUDIOIN_DATASET_VR_NB) : (e32_AUDIOIN_DATASET_VR_WB);
         break;
      case e8AUD_DEFAULT:
         //! Default audio type is without ECNR
         enAudioDataSet = e32_AUDIOIN_DATASET_VR_MIC_ONLY;
         break;
      default:
         ETG_TRACE_ERR(("Invalid audio direction type %d ", enAudioDir));
         break;
   }   //switch (enAudioDir)

   return enAudioDataSet;
}   //! end of spi_tclDiPoAudio::enGetAudioInDataSet()

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAudio::vActivateChannel(tenAudioDir enAudDir)
 ***************************************************************************/
t_Void spi_tclDiPoAudio::vActivateChannel(tenAudioDir enAudDir, tenAudioSamplingRate enSamplingRate,
      tenAudioSamplingRate enNativeSamplingRate)
{
   /*lint -esym(40,fvLaunchAudioReq)fvLaunchAudioReq Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclDiPoAudio::vActivateChannel() entered for Audio direction %d ", ETG_ENUM(SPI_AUDIO_DIRECTION,
            enAudDir)));

   if (NULL != m_rAudCallbacks.fvLaunchAudioReq)
   {
      trAudSampleRate rAudioSampleRate;
      rAudioSampleRate.enSampleRate = enSamplingRate;
      rAudioSampleRate.enNativeSampleRate = enNativeSamplingRate;
      (m_rAudCallbacks.fvLaunchAudioReq)(m_u32DeviceHandle, e8DEV_TYPE_DIPO, enAudDir,rAudioSampleRate);
   }
}   //t_Void spi_tclDiPoAudio::vDeactivateChannel()

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAudio::vDeactivateChannel(tenAudioDir enAudDir, tenSrcAvailabilityReason enSrcAvlReason)
 ***************************************************************************/
t_Void spi_tclDiPoAudio::vDeactivateChannel(tenAudioDir enAudDir, tenSrcAvailabilityReason enSrcAvlReason)
{
   /*lint -esym(40,fvTerminateAudioReq)fvTerminateAudioReq Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclDiPoAudio::vDeactivateChannel() entered for Audio direction %d ", ETG_ENUM(SPI_AUDIO_DIRECTION,
            enAudDir)));

   if (NULL != m_rAudCallbacks.fvTerminateAudioReq)
   {
      (m_rAudCallbacks.fvTerminateAudioReq)(m_u32DeviceHandle, enAudDir, enSrcAvlReason);
   }
}   //t_Void spi_tclDiPoAudio::vDeactivateChannel()

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAudio::vMuteAudio(...)
 ***************************************************************************/
t_Void spi_tclDiPoAudio::vMuteAudio(t_Bool bMute)
{
   /*lint -esym(40,fbSetSourceMute)fbSetSourceMute Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclDiPoAudio::vMuteAudio: %d ", ETG_ENUM(BOOL, bMute)));

   if (NULL != m_rAudCallbacks.fbSetSourceMute)
   {
      (m_rAudCallbacks.fbSetSourceMute)(e8AUD_MAIN_OUT, bMute);
   }
}   //t_Void spi_tclDiPoAudio::vMuteAudio()

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAudio::vSendAudioStatusChange(...)
 ***************************************************************************/
t_Void spi_tclDiPoAudio::vSendAudioStatusChange(tenAudioStatus enAudStatus)
{
   /*lint -esym(40,fvSendAudStatusChange)fvSendAudStatusChange Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclDiPoAudio::vSendAudioStatusChange: %d ", ETG_ENUM(AUDIO_STATUS, enAudStatus)));

   if (NULL != m_rAudCallbacks.fvSendAudStatusChange)
   {
      (m_rAudCallbacks.fvSendAudStatusChange)(enAudStatus);
   }
}   //t_Void spi_tclDiPoAudio::vMuteAudio()

/***************************************************************************
 ** FUNCTION:  spi_tclDiPoAudio::vOnModesChangedUpdate
 ***************************************************************************/
t_Void spi_tclDiPoAudio::vOnModesChangedUpdate(const trDiPOModeState& corfrDiPOModeState)
{
   tenDiPOEntity enRequestedAudioMode = corfrDiPOModeState.enAudio;
   ETG_TRACE_USR4(("spi_tclDiPoAudio::vOnModesChangedUpdate: DeviceHandle = %d "
            "CurrentAudioMode = %d, RequestedAudioMode = %d, CurrAudioMainSrc = %d ", m_u32DeviceHandle, ETG_ENUM(DIPO_ENTITY_MODE,
            m_enCurrentAudioMode), ETG_ENUM(DIPO_ENTITY_MODE, enRequestedAudioMode), ETG_ENUM(SPI_AUDIO_DIRECTION,
            m_enCurrAudioMainSrc)));

   //! check if the audio mode has changed
   if (m_enCurrentAudioMode != enRequestedAudioMode)
   {
      Timer* poTimer = Timer::getInstance();
      if ((true == bAudRelTimerRunning) && (NULL != poTimer))
      {
         poTimer->CancelTimer(srAudReleaseTimerID);
         ETG_TRACE_USR4(("Aud Release Timer Stopped "));
         bAudRelTimerRunning = false;
      }   //End of if ((true == bAudRelTimerRunning) && (NULL != poTimer))

      //! Requested Mode is MOBILE and current mode is NA/CAR.
      if (e8DIPO_ENTITY_MOBILE == enRequestedAudioMode)
      {
         m_enTransferType = e8DIPO_TRANSFERTYPE_BORROW;
         ETG_TRACE_USR4(("Mode Change: TransferType changed to %d ", ETG_ENUM(DIPO_TRANSFERTYPE, m_enTransferType)));
      }   //End of if(e8DIPO_ENTITY_MOBILE == enRequestedAudioMode)

      //! Requested Mode is NA/CAR and current mode is MOBILE.
      else
      {
         m_enTransferType = e8DIPO_TRANSFERTYPE_NA;
         ETG_TRACE_USR4(("Mode Change: TransferType changed to %d ", ETG_ENUM(DIPO_TRANSFERTYPE, m_enTransferType)));

         if ((true == bAudRestoreTimerRunning) && (NULL != poTimer))
         {
            poTimer->CancelTimer(srAudRestoreTimerID);
            ETG_TRACE_USR4(("Aud Restore Timer Stopped "));
            bAudRestoreTimerRunning = false;

            if (NULL != m_rAudCallbacks.fvRestoreLastMediaAudSrc)
            {
               m_rAudCallbacks.fvRestoreLastMediaAudSrc();
            }
         }   //End of if ((true == bAudRestoreTimerRunning) && (NULL != poTimer))

         //! If the audio mode has changed from NA/CAR to MOBILE
         //! Do nothing
         //! Terminate audio  for current audio source(other than e8AUD_MAIN_OUT) if audio mode has changed from MOBILE to NA/CAR
         if (e8AUD_MAIN_OUT != m_enCurrAudioMainSrc)
         {
            vDeactivateChannel(m_enCurrAudioMainSrc);
         }         //End of if ((e8DIPO_ENTITY_MOBILE == m_enCurrentAudioMode) && ...)
         else
         {
            ETG_TRACE_USR1(("spi_tclDiPoAudio::vOnModesChangedUpdate - enRequestedAudioMode = %d, bIsAudioCtxtRequestedbyCar = %d", ETG_ENUM(DIPO_ENTITY_MODE,
                     enRequestedAudioMode), ETG_ENUM(BOOL, m_poDiPOCmdRsrcMngr->bIsAudioCtxtRequestedbyCar())));

            //Deactivate audio source if, Audio resource owner changes from Mobile -> Car,
            //Audio channel allocated is SPI_MEDIA and head unit has not requested for audio resource
            if ((NULL != m_poDiPOCmdRsrcMngr) && (false == m_poDiPOCmdRsrcMngr->bIsAudioCtxtRequestedbyCar()))
            {
               //The audio channel is deactivated by sending reason as e8REASON_SAMEMEDIA so that AM does not remove the channel
               ETG_TRACE_USR1(("spi_tclDiPoAudio::vOnModesChangedUpdate - dipo audio source channel terminated by controller"));
               vDeactivateChannel(e8AUD_MAIN_OUT, e8REASON_SAMEMEDIA);
            }
            vSendAudioStatusChange(e8AUDIO_STATUS_RELEASE_AUDIO_DEVICE);
         }

         if ((NULL != m_poDiPOCmdRsrcMngr) && (true == m_poDiPOCmdRsrcMngr->bIsAudioCtxtRequestedbyCar()))
         {
            m_poDiPOCmdRsrcMngr->vSetAudioCtxtRequestedbyCar(false);
         }

         m_enCurrAudioMainSrc = e8AUD_INVALID;

         vSendAudioStatusChange(e8AUDIO_STATUS_MEDIA_RELEASE);
      }             //End of Else

      m_enCurrentAudioMode = enRequestedAudioMode;
   }             //End of if ((m_enCurrentAudioMode != enRequestedAudioMode) ..)

   ETG_TRACE_USR4(("spi_tclDiPoAudio::vOnModesChangedUpdate left with CurrentAudioMode = %d, CurrAudioMainSrc = %d ", ETG_ENUM(DIPO_ENTITY_MODE,
            m_enCurrentAudioMode), ETG_ENUM(SPI_AUDIO_DIRECTION, m_enCurrAudioMainSrc)));
}

/***************************************************************************
 ** FUNCTION:  spi_tclDiPoAudio::vOnAudioAllocateRequest
 ***************************************************************************/
t_Void spi_tclDiPoAudio::vOnAudioAllocateRequest(AudioChannelType enAudioChannelType,
         tenDiPOMainAudioType enAudioType,
         tenAudioReqType enAudioReqType,
         const trDiPOAudioFormat& corfrDiPOAudioFormat)
{
   ETG_TRACE_USR4(("spi_tclDiPoAudio::vOnAudioAllocateRequest: "
            "Received request %d with Audio channel %d and type %d  Current source: m_enCurrAudioMainSrc = %d", ETG_ENUM(AUDIO_REQUEST_TYPE,
            enAudioReqType), ETG_ENUM(DIPO_AUDIO_STREAMTYPE, enAudioChannelType), ETG_ENUM(DIPO_AUDIO_TYPE,
            enAudioType), ETG_ENUM(SPI_AUDIO_DIRECTION, m_enCurrAudioMainSrc)));

   if (e8ALLOCATION_REQUEST == enAudioReqType)
   {
      vHandleAllocRequest(enAudioChannelType, enAudioType, corfrDiPOAudioFormat);
   }
   else if ((e8DEALLOCATION_REQUEST == enAudioReqType))
   {
      vHandleDeallocRequest(enAudioChannelType);
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDiPoAudio::vOnDuckAudioMsg
 ***************************************************************************/
t_Void spi_tclDiPoAudio::vOnDuckAudioMsg(t_Double dFinalVolume, t_Double dDurationInMs)
{
   /*lint -esym(40,fbSetAudioDucking)fbSetAudioDucking Undeclared identifier */
   //To avoid Gen4Compiler warnings converting double to float. Reason: ETG trace does not support for double values
   ETG_TRACE_USR1(("spi_tclDiPoAudio::vOnDuckAudioMsg: "
            "Received request with Final volume: %f, Duration = %f ", static_cast<t_Float>(dFinalVolume), static_cast<t_Float>(dDurationInMs)));

   t_U8 u8RampVolIndB = 0;
   t_Bool bRetVal = false;

   if (NULL != m_rAudCallbacks.fbSetAudioDucking)
   {
      if (scodAudioUnduck != dFinalVolume)
      {
         //! Convert Relative value to Decibels.Info from ADIT
         //! rfrAudioDuckMsg.dFinalVolume = 1.0 -> Unduck
         //! <= -144 translates to 0.0
         //! > -144 .. < 0 translates to 0.2
         //!   >= 0 translates to 1.0 */
         u8RampVolIndB =
                  (t_U8) ((dFinalVolume > scodAudioDuckMindB) && (dFinalVolume < scodAudioUnduck)) ?
                           (scou8AudioDuckDefdB) : (scou8AudioDuckMaxdB);

         bRetVal = m_rAudCallbacks.fbSetAudioDucking((t_U16) dDurationInMs,
                  u8RampVolIndB,
                  e8_DUCKINGTYPE_DUCK,
                  e8DEV_TYPE_DIPO);

         m_bAudDuckEnabled = true;
      }         //End of if (scodAudioUnduck !=  rfrAudioDuckMsg.dFinalVolume)
      else
      {
         bRetVal = m_rAudCallbacks.fbSetAudioDucking((t_U16) dDurationInMs,
                  u8RampVolIndB,
                  e8_DUCKINGTYPE_UNDUCK,
                  e8DEV_TYPE_DIPO);

         m_bAudDuckEnabled = false;
      }         //End of else
   } //if (NULL != m_rAudCallbacks.fbSetAudioDucking)
   SPI_INTENTIONALLY_UNUSED(bRetVal);
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPoAudio::vOnSessionMsg(...
 ***************************************************************************/
t_Void spi_tclDiPoAudio::vOnSessionMsg(tenDiPOSessionState enDiPOSessionState,
         tenDiPOSessionTransport enSessionTransport, t_String szSessionIPAddress)
{
   m_oSessionStateLock.s16Lock();
   ETG_TRACE_USR1(("spi_tclDiPoAudio::vOnSessionMsg entered with state: %d, "
            "Session Transport:%d, Session IP Address:%s", ETG_ENUM(DIPO_SESSION_STATE, enDiPOSessionState), ETG_ENUM(DIPO_SESSION_TRANSPORT,
            enSessionTransport), szSessionIPAddress.c_str()));

   if (e8DIPO_SESSION_START == enDiPOSessionState)
   {
      m_bIsSessionActive = true;
   }
   else if (e8DIPO_SESSION_END == enDiPOSessionState)
   {
      vHandleSessionEnd();
   }
   m_oSessionStateLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  spi_tclDiPoAudio::vHandleAllocRequest()
 ***************************************************************************/
t_Void spi_tclDiPoAudio::vHandleAllocRequest(AudioChannelType enAudioChannelType, tenDiPOMainAudioType enAudioType,
         const trDiPOAudioFormat& corfrDiPOAudioFormat)
{
   ETG_TRACE_USR1(("spi_tclDiPoAudio::vHandleAllocRequest"));

   tenAudioSamplingRate enAudioSamplingRate = e8AUD_SAMPLERATE_DEFAULT;
   tenAudioSamplingRate enAudioNativeSampleRate = e8AUD_SAMPLERATE_DEFAULT;
   tenAudioDir enRequestedAudioSrc = e8AUD_INVALID;

   //!If ALTERNATE channel is requested from the device
   if (AudioChannelType_Alternate == (t_U8) enAudioChannelType)
   {
      vActivateChannel(e8AUD_MIX_OUT, enAudioSamplingRate);
      m_enCurrAudioAltSrc = e8AUD_MIX_OUT;
   }
   else if (AudioChannelType_Main == (t_U8) enAudioChannelType)
   {
      //!If MAIN channel is requested from the device
      switch (enAudioType)
      {
         case e8AUDIO_MEDIA:
         case e8AUDIO_ALERT:
         {
            enRequestedAudioSrc = (e8AUDIO_ALERT == enAudioType) ? (e8AUD_ALERT_OUT) : (e8AUD_MAIN_OUT);
         }
            break;
         case e8AUDIO_DEFAULT:
         case e8AUDIO_TELEPHONY:
         case e8AUDIO_SPEECHREC:
         case e8AUDIO_SPOKEN:
         {
            if (e8AUDIO_DEFAULT == enAudioType)
            {
               enRequestedAudioSrc = e8AUD_DEFAULT;
            }
            else
            {
               enRequestedAudioSrc = (e8AUDIO_TELEPHONY == enAudioType) ? (e8AUD_PHONE_IN) : (e8AUD_VR_IN);
            }

            tAudSampleRateMapItr itSampleRate = m_mapAudioSamplingRate.find(corfrDiPOAudioFormat.u16SampleRate);
            enAudioSamplingRate = (m_mapAudioSamplingRate.end()!= itSampleRate)
                                      ? (itSampleRate->second) : (e8AUD_SAMPLERATE_DEFAULT);

            if(e8AUD_PHONE_IN == enRequestedAudioSrc)
            {
               t_U16 u16VocoderSampleRate = u16VocoderSampleRatefromDouble(corfrDiPOAudioFormat.dVocoderSampleRate);

               tAudSampleRateMapItr itNativeSampleRate = m_mapAudioSamplingRate.find(u16VocoderSampleRate);
               enAudioNativeSampleRate = (m_mapAudioSamplingRate.end()!= itNativeSampleRate)
                                              ? (itNativeSampleRate->second) : (e8AUD_SAMPLERATE_DEFAULT);
            }
         }
         break;

         default:
         {
            enRequestedAudioSrc = e8AUD_INVALID;
         }
            break;
      } //  End of switch((t_U8)rfrAudioAllocMsg.enAudioType)

      m_oStreamStateLock.s16Lock();
      m_enMediaStreamState = (e8AUD_MAIN_OUT == enRequestedAudioSrc) ? (e8CP_AUD_STREAM_OPEN) : (m_enMediaStreamState);
      ETG_TRACE_USR4(("[PARAM]::vHandleAllocRequest: New MediaStreamState = %d", m_enMediaStreamState));
      m_oStreamStateLock.vUnlock();

      Timer* poTimer = Timer::getInstance();
      if ((true == bAudRelTimerRunning) && (NULL != poTimer))
      {
         poTimer->CancelTimer(srAudReleaseTimerID);
         ETG_TRACE_USR4(("Aud Release Timer Stopped "));
         bAudRelTimerRunning = false;
      } //End of if ((true == bAudRelTimerRunning) && (NULL != poTimer))

      if (enRequestedAudioSrc != m_enCurrAudioMainSrc)
      {
         //!If requested audio source is differs with the current active source.
         vProcessChangedMainAudSrc(enRequestedAudioSrc, enAudioSamplingRate,enAudioNativeSampleRate);
      } //End of if (enRequestedAudioSrc != m_enCurrAudioMainSrc)
      else
      {
         //!If requested audio source is same as the current active source.
         vProcessUnchangedMainAudSrc(enAudioSamplingRate,enAudioNativeSampleRate);
      } //End of else

      if (e8AUD_MAIN_OUT == enRequestedAudioSrc)
      {
         vSendAudioStatusChange(e8AUDIO_STATUS_MEDIA_SETUP);
      }
      else if ((e8AUD_ALERT_OUT == enRequestedAudioSrc) || (e8AUD_PHONE_IN == enRequestedAudioSrc)
               || (e8AUD_VR_IN == enRequestedAudioSrc) || (e8AUD_DEFAULT == enRequestedAudioSrc))
      {
         //@Note: This is to ensure cleanup is done for main audio if Siri/Phone/Alert prepare comes
         //after a dummy media prepare
         vSendAudioStatusChange(e8AUDIO_STATUS_MEDIA_RELEASE);
      }
   } // End of else if (AudioChannelType_Main ==... )
}

t_U16 spi_tclDiPoAudio::u16VocoderSampleRatefromDouble(t_Double dVocoderSampleRate)
{
   t_U16 u16SampleRate = 0;
   if(dVocoderSampleRate<4000)
   {
      u16SampleRate = 0;
   }

   else if(dVocoderSampleRate>=4000 && dVocoderSampleRate<12000)
   {
     u16SampleRate = 8000;
   }
   else if(dVocoderSampleRate>=12000 && dVocoderSampleRate<20000)
   {
     u16SampleRate = 16000;
   }
   else
   {
      u16SampleRate = 24000;
   }

   return u16SampleRate;
}


/***************************************************************************
 ** FUNCTION:  spi_tclDiPoAudio::vHandleDeallocRequest()
 ***************************************************************************/
t_Void spi_tclDiPoAudio::vHandleDeallocRequest(const AudioChannelType coenAudioChannelType)
{
   /*lint -esym(40,fvStopAudioIn)fvStopAudioIn Undeclared identifier */
   ETG_TRACE_USR1(("spi_tclDiPoAudio::vHandleDeallocRequest "));

   if (AudioChannelType_Main == (t_U8) coenAudioChannelType)
   {
      m_oStreamStateLock.s16Lock();
      m_enMediaStreamState =
               (e8AUD_MAIN_OUT == m_enCurrAudioMainSrc) ? (e8CP_AUD_STREAM_CLOSED) : (m_enMediaStreamState);
      ETG_TRACE_USR4(("[PARAM]::vHandleDeallocRequest: New MediaStreamState = %d", m_enMediaStreamState));
      m_oStreamStateLock.vUnlock();

      if (NULL != m_rAudCallbacks.fvStopAudioIn)
      {
         m_rAudCallbacks.fvStopAudioIn(m_enCurrAudioMainSrc);
      }

      if ((e8AUD_MAIN_OUT == m_enCurrAudioMainSrc) || (e8AUD_INVALID == m_enCurrAudioMainSrc))
      {
         vSendAudioStatusChange(e8AUDIO_STATUS_MEDIA_TEARDOWN);
      }
      else if (false == bAudRelTimerRunning)
      {
         Timer* poTimer = Timer::getInstance();
         if (NULL != poTimer)
         {
            poTimer->StartTimer(srAudReleaseTimerID, 2000, 0, this, &spi_tclDiPoAudio::bAudReleaseTimerCb, NULL);
            ETG_TRACE_USR4(("Aud Release Timer started "));
            bAudRelTimerRunning = true;
         } //End of if (NULL != poTimer)
      } //End of else if ((e8AUD_MAIN_OUT != m_enCurrAudioMainSrc) && (false == bAudRelTimerRunning))
   } //End of if (AudioChannelType_Main == (t_U8) coenAudioChannelType)

   //! If request is received to deallocate alternate channel,
   //! send terminate audio request for e8AUD_MIX_OUT
   if (AudioChannelType_Alternate == (t_U8) coenAudioChannelType)
   {
      vDeactivateChannel(e8AUD_MIX_OUT);
      m_enCurrAudioAltSrc = e8AUD_INVALID;
   }   //End of if(AudioChannelType_Alternate == (t_U8)coenAudioChannelType)
}

//!Static
/***************************************************************************
 ** FUNCTION:  spi_tclDiPoAudio::bAudRestoreTimerCb
 ***************************************************************************/
t_Bool spi_tclDiPoAudio::bAudRestoreTimerCb(timer_t rTimerID, t_Void *pvObject, const t_Void *pvUserData)
{
   SPI_INTENTIONALLY_UNUSED(pvUserData);
   SPI_INTENTIONALLY_UNUSED(rTimerID);
   ETG_TRACE_USR1((" spi_tclDiPoAudio::bAudRestoreTimerCb entered  "));

   Timer* poTimer = Timer::getInstance();
   if (NULL != poTimer)
   {
      poTimer->CancelTimer(srAudRestoreTimerID);
      ETG_TRACE_USR4(("Aud Restore Timer stopped "));

      bAudRestoreTimerRunning = false;
   }   //End of if (NULL != poTimer)

   spi_tclDiPoAudio* poAudResrcMngr = static_cast<spi_tclDiPoAudio*>(pvObject);

   if (NULL != poAudResrcMngr)
   {
      poAudResrcMngr->m_enTransferType = e8DIPO_TRANSFERTYPE_TAKE;

      /*t_U32 u32DevHandle = poAudResrcMngr->m_u32DeviceHandle;
       tenAudioDir enAudSrc = poAudResrcMngr->m_enCurrAudioMainSrc;

       //! If the current active source is not Entertainment source, then
       //! de-activate the current active source and activate the new requested source.
       if(e8AUD_MAIN_OUT != enAudSrc)
       {
       poAudResrcMngr->vDeactivateChannel(enAudSrc);
       }//End of if(e8AUD_MAIN_OUT != poAudResrcMngr->enAudSrc)

       poAudResrcMngr->vActivateChannel(e8AUD_MAIN_OUT, e8AUD_SAMPLERATE_DEFAULT);

       poAudResrcMngr->m_enCurrAudioMainSrc = e8AUD_MAIN_OUT;
       poAudResrcMngr->m_enAudioSamplingRate = e8AUD_SAMPLERATE_DEFAULT;*/
   }   //End of if (NULL != poAudResrcMngr)

   return true;
}

//!Static
/***************************************************************************
 ** FUNCTION:  spi_tclDiPoAudio::bAudReleaseTimerCb
 ***************************************************************************/
t_Bool spi_tclDiPoAudio::bAudReleaseTimerCb(timer_t rTimerID, t_Void *pvObject, const t_Void *pvUserData)
{
   SPI_INTENTIONALLY_UNUSED(pvUserData);
   SPI_INTENTIONALLY_UNUSED(rTimerID);
   ETG_TRACE_USR1((" spi_tclDiPoAudio::bAudReleaseTimerCb entered  "));

   Timer* poTimer = Timer::getInstance();
   if (NULL != poTimer)
   {
      poTimer->CancelTimer(srAudReleaseTimerID);
      ETG_TRACE_USR4(("Aud Release Timer expired "));

      bAudRelTimerRunning = false;
   }   //End of if (NULL != poTimer)

   spi_tclDiPoAudio* poDipoAudio = static_cast<spi_tclDiPoAudio*> (pvObject);
   if (NULL != poDipoAudio)
   {
      poDipoAudio->vDeactivateChannel(poDipoAudio->m_enCurrAudioMainSrc);
      poDipoAudio->m_enCurrAudioMainSrc = e8AUD_INVALID;
   }

   return true;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDiPoAudio::vProcessChangedMainAudSrc()
 ***************************************************************************/
t_Void spi_tclDiPoAudio::vProcessChangedMainAudSrc(const tenAudioDir coenRequestedAudSrc,
         const tenAudioSamplingRate coenRequestedSamplingRate,
         const tenAudioSamplingRate coenRequestedNativeSamplingRate)
{
   ETG_TRACE_USR1(("spi_tclDiPoAudio::vProcessChangedMainAudSrc TransferType %d ", ETG_ENUM(DIPO_TRANSFERTYPE,
            m_enTransferType)));

   //! No Carplay Source was playing. Media Prepare received after Audio Transfer to Mobile
   //! INVALID -> MEDIA
   if ((e8AUD_MAIN_OUT == coenRequestedAudSrc) && (e8AUD_INVALID == m_enCurrAudioMainSrc)
            && (e8DIPO_TRANSFERTYPE_BORROW == m_enTransferType))
   {
      m_enTransferType = e8DIPO_TRANSFERTYPE_TAKE;
      ETG_TRACE_USR4(("TransferType changed to %d ", ETG_ENUM(DIPO_TRANSFERTYPE, m_enTransferType)));
      vActivateChannel(coenRequestedAudSrc, coenRequestedSamplingRate, coenRequestedNativeSamplingRate);
   }   //End of ((e8AUD_MAIN_OUT == coenRequestedAudSrc) && (e8AUD_INVALID == m_enCurrAudioMainSrc)..

       //! PHONE/SIRI/ALERT -> MEDIA
   else if ((e8AUD_MAIN_OUT == coenRequestedAudSrc) && (e8AUD_INVALID != m_enCurrAudioMainSrc)
            && (e8DIPO_TRANSFERTYPE_BORROW == m_enTransferType))
   {
      vDeactivateChannel(m_enCurrAudioMainSrc);

      vActivateChannel(coenRequestedAudSrc, coenRequestedSamplingRate, coenRequestedNativeSamplingRate);

      Timer* poTimer = Timer::getInstance();
      if (NULL != poTimer)
      {
         poTimer->StartTimer(srAudRestoreTimerID, 3000, 0, this, &spi_tclDiPoAudio::bAudRestoreTimerCb, NULL);

         ETG_TRACE_USR4(("Aud Restore Timer started "));

         bAudRestoreTimerRunning = true;
      }   //End of if (NULL != poTimer)
   }   //End of else if ((e8AUD_MAIN_OUT == coenRequestedAudSrc) && (e8AUD_INVALID != m_enCurrAudioMainSrc)...

       //! No Carplay Source was playing. Alert/Phone/Siri Audio Prepare received
       //! INVALID -> ALERT/PHONE/SIRI
       //! SIRI->PHONE, ALERT->PHONE
   else
   {
      ETG_TRACE_USR4(("Transfer Type Unchanged. Aud Restore Timer not started "));

      if (e8AUD_MAIN_OUT != m_enCurrAudioMainSrc)
      {
         vDeactivateChannel(m_enCurrAudioMainSrc);
      }

      vActivateChannel(coenRequestedAudSrc, coenRequestedSamplingRate, coenRequestedNativeSamplingRate);
   }   //End of Else

   m_enCurrAudioMainSrc = coenRequestedAudSrc;
   m_enAudioSamplingRate = coenRequestedSamplingRate;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDiPoAudio::vProcessUnchangedMainAudSrc()
 ***************************************************************************/
t_Void spi_tclDiPoAudio::vProcessUnchangedMainAudSrc( const tenAudioSamplingRate coenRequestedSamplingRate,
      const tenAudioSamplingRate coenRequestedNativeSamplingRate)
{
   ETG_TRACE_USR1(("spi_tclDiPoAudio::vProcessUnchangedMainAudSrc "));

   //!If current source is Phone/Siri.
   if ((e8AUD_PHONE_IN == m_enCurrAudioMainSrc) || (e8AUD_VR_IN == m_enCurrAudioMainSrc)
            || (e8AUD_DEFAULT == m_enCurrAudioMainSrc))
   {
      m_enAudioSamplingRate = coenRequestedSamplingRate;

      //!Reset the dataset
      if (NULL != m_rAudCallbacks.fvSetAudioInConfig)
      {
         m_rAudCallbacks.fvSetAudioInConfig(m_enCurrAudioMainSrc,
                  enGetAudioInDataSet(m_enCurrAudioMainSrc, coenRequestedSamplingRate));
      }

      //! Reset the audio information
      vActivateChannel(m_enCurrAudioMainSrc, coenRequestedSamplingRate, coenRequestedNativeSamplingRate);

      //! Ecnr will be restarted when source activation will be mocked from
      //! common Audio class on triggering LaunchAudio.
      //! Restart ecnr service with new sampling rate
      //m_rAudCallbacks.fvSetAlsaDevice(m_enCurrAudioMainSrc);
      //m_rAudCallbacks.fvStartAudioIn(m_enCurrAudioMainSrc);
   }      //End of if (e8AUD_PHONE_IN == m_enCurrAudioMainSrc) ...)
   /*else if (e8AUD_MAIN_OUT == m_enCurrAudioMainSrc)
    {
    vMuteAudio(false);
    }*/

   //! For all sources signal Audio Out adapter to unlock from 2 sec signaled wait
   if (NULL != spoCmdAudio)
   {
      tenAudioStreamType enAudioStreamType = e8MAIN_AUDIO;
      spoCmdAudio->bSendAudioAllocResponse(e8_DIPO_AUDIO_ALLOC_RESP_AUDIO_ALREADY_ACTIVE, enAudioStreamType, "", "");
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDiPoAudio::vHandleSessionEnd
 ***************************************************************************/
t_Void spi_tclDiPoAudio::vHandleSessionEnd()
{
   /*lint -esym(40,fbSetAudioDucking)fbSetAudioDucking Undeclared identifier */
   ETG_TRACE_USR4(("spi_tclDiPoAudio::vHandleSessionEnd entered"));

   //@Note: SessionActive flag check ensures vHandleSessionEnd() is executed only once -
   //On Device deselection or Session end, whichever arrives first
   if (true == m_bIsSessionActive)
   {
      Timer* poTimer = Timer::getInstance();
      if (NULL != poTimer)
      {
         if (true == bAudRestoreTimerRunning)
         {
            poTimer->CancelTimer(srAudRestoreTimerID);
            bAudRestoreTimerRunning = false;
            ETG_TRACE_USR4(("Session Ended. Aud Restore Timer stopped "));
         }
         if (true == bAudRelTimerRunning)
         {
            poTimer->CancelTimer(srAudReleaseTimerID);
            ETG_TRACE_USR4(("Session Ended. Aud Release Timer stopped "));
            bAudRelTimerRunning = false;
         }
      }   //End of if (NULL != poTimer)

      //! Set audio status to STATUS_MEDIA_RELEASE, on session termination.
      vSendAudioStatusChange(e8AUDIO_STATUS_MEDIA_RELEASE);

      //TODO - set audio mute at this stage even if main audio channel is in paused state?
      if (e8AUD_MAIN_OUT == m_enCurrAudioMainSrc)
      {
         vSendAudioStatusChange(e8AUDIO_STATUS_RELEASE_AUDIO_DEVICE);

         //@Note: Release audio channel only after CP device is deselected/disconnected(SUZUKI-22901),
         //since ALSA device can still be in use until session is completely ended
         if (m_bIsLastModeSupported)
         {
            vMuteAudio(true);
         }
         else
         {
            vDeactivateChannel(m_enCurrAudioMainSrc);
         }
      }
      else if (e8AUD_INVALID != m_enCurrAudioMainSrc)
      {
         //! Terminate the currently active Main Audio(other than CP media) source on session end
         //! CarPlay media audio should not be terminated to handle last mode feature
         vDeactivateChannel(m_enCurrAudioMainSrc);
      }

      //@Note: Releasing main audio channel on device disconnection if native audio is active is taken care in common audio class
      /*else if ((e8AUD_INVALID == m_enCurrAudioMainSrc) && (m_bIsLastModeSupported))
       {
       vDeactivateChannel(e8AUD_MAIN_OUT);
       }*/

      if (e8AUD_INVALID != m_enCurrAudioAltSrc)
      {
         //! Terminate the currently active Alternate Audio source on session end
         vDeactivateChannel(m_enCurrAudioAltSrc);
      }

      if ((true == m_bAudDuckEnabled) && (NULL != m_rAudCallbacks.fbSetAudioDucking))
      {
         m_rAudCallbacks.fbSetAudioDucking(scou16AudioRampDefDuration,
                  scou8AudioDuckDefdB,
                  e8_DUCKINGTYPE_UNDUCK,
                  e8DEV_TYPE_DIPO);
         m_bAudDuckEnabled = false;
      }         //End of if(true = m_bAudDuckEnabled)

      m_oStreamStateLock.s16Lock();
      m_enMediaStreamState = e8CP_AUD_STREAM_CLOSED;
      ETG_TRACE_USR4(("[PARAM]::vHandleSessionEnd: New MediaStreamState = %d", m_enMediaStreamState));
      m_oStreamStateLock.vUnlock();

      m_enCurrAudioMainSrc = e8AUD_INVALID;
      m_enCurrAudioAltSrc = e8AUD_INVALID;
      m_enCurrentAudioMode = e8DIPO_ENTITY_NA;
      m_enAudioSamplingRate = e8AUD_SAMPLERATE_DEFAULT;
      m_enTransferType = e8DIPO_TRANSFERTYPE_NA;

      m_bIsSessionActive = false;
   }         //if (0 != m_u32DeviceHandle)

   ETG_TRACE_USR4(("spi_tclDiPoAudio::vHandleSessionEnd left"));
}

//lint -restore
///////////////////////////////////////////////////////////////////////////////
// <EOF>
