/*!
 *******************************************************************************
 * \file              spi_tclDiPOVideoOutAdapterImpl.cpp
 * \brief             CarPlay video out adapter extended implementation
 *******************************************************************************
 \verbatim
 PROJECT:        G3G
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    CarPlay audio out adapter extended implementation
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 17.11.2016 |  Noopur R K                  | Initial Version

 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include <time.h>
#include <errno.h>
#include "Lock.h"
#include "Event.h"
#include "IPCMessageQueue.h"
#include "spi_tclDiPOVideoOutAdapterImpl.h"
#include "GstreamerVideoOut.h"

#include "spi_tclDiPOAdapterMsgQInterface.h"

#define SPI_ENABLE_DLT //enable DLT
#define SPI_LOG_CLASS Spi_CarPlay
#define SEM_TIMEOUT_IN_SEC 1
#define SEM_ERROR -1

#include "Trace.h"
//Declare DiPO context
LOG_IMPORT_CONTEXT( Spi_CarPlay);
static Lock rVideoAdapterSyncLock;
spi_tclDiPOVideoOutAdapterImpl* spi_tclDiPOVideoOutAdapterImpl::m_poVideoOutAdapter = NULL;

/***************************************************************************
 ** FUNCTION: t_Void spi_tclDiPOVideoOutAdapterImpl:: spi_tclDiPOVideoOutAdapterImpl()
 ***************************************************************************/
spi_tclDiPOVideoOutAdapterImpl::spi_tclDiPOVideoOutAdapterImpl() :
   m_bVideoRenderingLockStatus(false)
{
   ETG_TRACE_USR1(("spi_tclDiPOVideoOutAdapterImpl::spi_tclDiPOVideoOutAdapterImpl entered"));
   rVideoAdapterSyncLock.s16Lock();
   m_poVideoOutAdapter = this;
   rVideoAdapterSyncLock.vUnlock();
   if (SEM_ERROR == sem_init(&m_DiPOVideoRenderingSem, 0, 0))
   {
      ETG_TRACE_ERR(("[ERR]:spi_tclDiPOVideoOutAdapterImpl - Creation of semaphore failed : Error Number %d ", errno));
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclDiPOVideoOutAdapterImpl::~spi_tclDiPOVideoOutAdapterImpl()
 ***************************************************************************/
spi_tclDiPOVideoOutAdapterImpl::~spi_tclDiPOVideoOutAdapterImpl()
{
   ETG_TRACE_USR1(("spi_tclDiPOVideoOutAdapterImpl::~spi_tclDiPOVideoOutAdapterImpl entered"));
   if (SEM_ERROR == sem_destroy(&m_DiPOVideoRenderingSem))
   {
      ETG_TRACE_ERR(("[ERR]:spi_tclDiPOVideoOutAdapterImpl::Destruction of semaphore failed : Error Number %d ", errno));
   }
   rVideoAdapterSyncLock.s16Lock();
   m_poVideoOutAdapter = NULL;
   rVideoAdapterSyncLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclDiPOVideoOutAdapterImpl:: vSetOEMIconVisibility(..)
 ***************************************************************************/
spi_tclDiPOVideoOutAdapterImpl* spi_tclDiPOVideoOutAdapterImpl::poGetDiPOVideoAdapterInstance()
{
   ETG_TRACE_USR1(("spi_tclDiPOAudioOutAdapterImpl::poGetInstance entered"));
   rVideoAdapterSyncLock.s16Lock();
   spi_tclDiPOVideoOutAdapterImpl *poVideoAdapter = m_poVideoOutAdapter;
   rVideoAdapterSyncLock.vUnlock();
   return poVideoAdapter;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclDiPOVideoOutAdapterImpl:: vStop()
 ***************************************************************************/
t_Void spi_tclDiPOVideoOutAdapterImpl::Stop()
{
   ETG_TRACE_USR1(("spi_tclDiPOVideoAdpaterImpl::Stop entered"));
   //acquire lock here and send the message till Service
   trProjectionVideoRenderingStatusMsg rProjectionVideoRenderingStatusMsg;
   rProjectionVideoRenderingStatusMsg.enProjectionVideoRenderingStatus = e8_PROJECTION_VIDEO_RENDERING_STOPPED;

   t_Bool bStatus = bSendMessage<trProjectionVideoRenderingStatusMsg> (rProjectionVideoRenderingStatusMsg);

   ETG_TRACE_USR2(("[DESC]:spi_tclDiPOVideoAdpaterImpl::Stop : Projection Video Rendering Stopped message sent to SPI with status = %d", ETG_ENUM(STATUS,
            bStatus)));

   /*Here we are setting a semaphore for Projection video rendering status update,
    * note that the timed wait is going to be set for 2 seconds the semaphore will
    * be incremented by vNativeVideoRenderingStatus()*/
   struct timespec rTimeSpec;
   if (SEM_ERROR == clock_gettime(CLOCK_REALTIME, &rTimeSpec))
   {
      ETG_TRACE_USR2(("spi_tclDiPOVideoAdpaterImpl::vStop - Error in fetching real time"));
   }
   else
   {
      ETG_TRACE_USR1(("[DESC]:spi_tclDiPOVideoAdpaterImpl::vStop Timer value Left_01 %ld", (long int) rTimeSpec.tv_sec));
      rTimeSpec.tv_sec += SEM_TIMEOUT_IN_SEC;
      m_bVideoRenderingLockStatus = true;
      ETG_TRACE_USR2(("spi_tclDiPOVideoAdpaterImpl::vStop : Acquiring the semaphore"));
      ETG_TRACE_USR1(("[DESC]:spi_tclDiPOVideoAdpaterImpl::vStop Timer value Left_02 %ld", (long int) rTimeSpec.tv_sec));
      if (SEM_ERROR == sem_timedwait(&m_DiPOVideoRenderingSem, &rTimeSpec))
      {
         if (errno == ETIMEDOUT)
         {
            ETG_TRACE_USR1(("[DESC]:spi_tclDiPOVideoAdpaterImpl::vStop Timer value Left_03 %ld", (long int) rTimeSpec.tv_sec));
            ETG_TRACE_USR2(("spi_tclDiPOVideoAdpaterImpl::vStop - sem_timedwait timer expired; Unblocking"
               "from Stop()"));
            m_bVideoRenderingLockStatus = false;
         }
      }
      else
      {
         ETG_TRACE_USR2(("spi_tclDiPOVideoAdpaterImpl::vStop - sem_timedwait is posted"));
      }
   }
   ETG_TRACE_USR1(("spi_tclDiPOVideoAdpaterImpl::vStop left"));
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclDiPOVideoOutAdapterImpl:: vNativeVideoRenderingStatus(..)
 ***************************************************************************/
t_Void spi_tclDiPOVideoOutAdapterImpl::vNativeVideoRenderingStatus(
         tenNativeVideoRenderingStatus enNativeVideoRenderingStatus)
{
   ETG_TRACE_USR1(("spi_tclDiPOVideoAdpaterImpl::vNativeVideoRenderingStatus entered"));
   ETG_TRACE_USR1(("spi_tclDiPOVideoAdpaterImpl::vNativeVideoRenderingStatus m_bVideoRenderingLockStatus %d", ETG_ENUM(BOOL, m_bVideoRenderingLockStatus)));
   if ((enNativeVideoRenderingStatus == e8_NATIVE_VIDEO_RENDERING_STARTED) && (m_bVideoRenderingLockStatus == true))
   {
      ETG_TRACE_USR1(("spi_tclDiPOVideoAdpaterImpl::vNativeVideoRenderingStatus - I am here to unlock"));
      if (SEM_ERROR == sem_post(&m_DiPOVideoRenderingSem))
      {
         ETG_TRACE_USR1(("spi_tclDiPOVideoAdpaterImpl::vNativeVideoRenderingStatus error in sem_post : %d", errno));
      }
      else
      {
         ETG_TRACE_USR1(("spi_tclDiPOVideoAdpaterImpl::vNativeVideoRenderingStatus - Posting semaphore!"));
      }
   }
   m_bVideoRenderingLockStatus = false;
   ETG_TRACE_USR1(("spi_tclDiPOVideoAdpaterImpl::vNativeVideoRenderingStatus left"));
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPOAudioOutAdapterImpl::bSendMessage()
 ***************************************************************************/
template<typename trMessage>
t_Bool spi_tclDiPOVideoOutAdapterImpl::bSendMessage(trMessage rMessage)
{
   ETG_TRACE_USR1(("spi_tclDiPOVideoOutAdapterImpl::bSendMessage entered"));
   t_Bool bRetVal = false;
   spi_tclDiPOAdapterMsgQInterface* poDiPOAdapterMsgQInterface = spi_tclDiPOAdapterMsgQInterface::getInstance();
   bRetVal = (poDiPOAdapterMsgQInterface) ? poDiPOAdapterMsgQInterface->bWriteMsgToQ(&rMessage, sizeof(trMessage)) : false;
   ETG_TRACE_USR1(("spi_tclDiPOVideoOutAdapterImpl::bSendMessage send (bRetVal = %d)", ETG_ENUM(BOOL,bRetVal)));
   return bRetVal;
}

