/*!
 *******************************************************************************
 * \file              spi_tclDiPOKeyInputAdapterImpl.cpp
 * \brief             DiPO Key Input Adapter implementation
 *******************************************************************************
 \verbatim
 PROJECT:        G3G
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    DiPO key input adapter implementation.
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                       | Modifications
 03.04.2014 |  Hari Priya E R               | Initial Version
 26.05.2015 |  Tejaswini H B(RBEI/ECP2)     | Added Lint comments to suppress C++11 Errors
 15.07.2015 |  Sameer Chandra               | knob encoder implementation
 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 | 1)system- and project- includes
 | 2)needed interfaces from external components
 | 3)internal and external interfaces from this component
 |----------------------------------------------------------------------------*/
#include "spi_tclDiPOKeyInputAdapterImpl.h"
#include "Lock.h" 

#define SPI_ENABLE_DLT //enable DLT
#define SPI_LOG_CLASS Spi_CarPlay

#include "Trace.h"
//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e19 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e58 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e48 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e808 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e64 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//import the DLT context
LOG_IMPORT_CONTEXT(Spi_CarPlay);

#define DIPO_CFG_CONSUMER_KEY_BITMAP_SIZE  7    //max supported consumer key from apple spec
#define DIPO_CFG_PHONE_KEY_BITMAP_SIZE     16   //max supported phone key from apple spec
#define DIPO_CFG_KNOB_KEY_BITMAP_SIZE      1    //max supported knob key from apple spec

static t_U8 coConsumerKeyDescriptor[MAX_STR_LEN];
static size_t coConsumerKeyDescriptorSize = 0x00;

static t_U8 coTelephonyKeyDescriptor[MAX_STR_LEN];
static size_t coTelephonyKeyDescriptorSize = 0x00;

static t_U8 coKnobKeyDescriptor[MAX_STR_LEN];
static size_t coKnobKeyDescriptorSize = 0x00;

static t_U8 au8KeyMapBitValues[DIPO_CFG_CONSUMER_KEY_BITMAP_SIZE];
static t_U8 au8TelKeyMapBitValues[DIPO_CFG_PHONE_KEY_BITMAP_SIZE];
static t_U8 au8KnobKeyMapBitValues[DIPO_CFG_KNOB_KEY_BITMAP_SIZE];

#define KNOB_PRODUCT_ID  832;
#define KNOB_VENDOR_ID  52;

spi_tclDiPOKeyInputAdapterImpl* spi_tclDiPOKeyInputAdapterImpl::m_poDiPOKeyInputAdapter = NULL;
t_Bool spi_tclDiPOKeyInputAdapterImpl::m_bKnobKeySupported = true;
t_Bool spi_tclDiPOKeyInputAdapterImpl::m_bKnobKeyDirectionalPadSupported = true;

static Lock rKeyAdapterSyncLock;

/***************************************************************************
 ** FUNCTION:  spi_tclDiPOKeyInputAdapterImpl::spi_tclDiPOKeyInputAdapterImpl()
 ***************************************************************************/
spi_tclDiPOKeyInputAdapterImpl::spi_tclDiPOKeyInputAdapterImpl() :
                  m_poIInputReceiver(NULL),
                  m_u8ConsumerKeydata(0),
                  m_u8TelephoneKeydata(0),
                  m_u8KnobKeyDirectionalPadMax(0x7F),
                  m_u8KnobKeyDirectionalPadMin(static_cast<t_S8>(0x81))
{
   ETG_TRACE_USR1(("spi_tclDiPOKeyInputAdapterImpl::spi_tclDiPOKeyInputAdapterImpl entered"));

   rKeyAdapterSyncLock.s16Lock();
   m_poDiPOKeyInputAdapter = this;
   rKeyAdapterSyncLock.vUnlock();

   memset(&m_rHIDConsumerKeyDeviceInfo, 0, sizeof(trHIDDeviceInfo));
   memset(&m_rHIDTelKeyDeviceInfo, 0, sizeof(trHIDDeviceInfo));
   memset(&m_rHIDKnobKeyDeviceInfo, 0, sizeof(trHIDDeviceInfo));
   memset(m_s8KnobKeydata, 0, KNOB_DATA_SIZE);
}

/***************************************************************************
 ** FUNCTION:  spi_tclDiPOKeyInputAdapterImpl::~spi_tclDiPOKeyInputAdapterImpl()
 ***************************************************************************/
spi_tclDiPOKeyInputAdapterImpl::~spi_tclDiPOKeyInputAdapterImpl()
{
   ETG_TRACE_USR1(("spi_tclDiPOKeyInputAdapterImpl::~spi_tclDiPOKeyInputAdapterImpl entered"));

   rKeyAdapterSyncLock.s16Lock();
   m_poIInputReceiver = NULL;
   m_poDiPOKeyInputAdapter = NULL;
   m_u8ConsumerKeydata = 0;
   m_u8TelephoneKeydata = 0;
   rKeyAdapterSyncLock.vUnlock();

   RELEASE_ARRAY_MEM(m_rHIDConsumerKeyDeviceInfo.Name);
   RELEASE_ARRAY_MEM(m_rHIDConsumerKeyDeviceInfo.UUID);
   RELEASE_ARRAY_MEM(m_rHIDTelKeyDeviceInfo.Name);
   RELEASE_ARRAY_MEM(m_rHIDTelKeyDeviceInfo.UUID);
   RELEASE_ARRAY_MEM(m_rHIDKnobKeyDeviceInfo.Name);
   RELEASE_ARRAY_MEM(m_rHIDKnobKeyDeviceInfo.UUID);
   memset(&m_rHIDConsumerKeyDeviceInfo, 0, sizeof(trHIDDeviceInfo));
   memset(&m_rHIDTelKeyDeviceInfo, 0, sizeof(trHIDDeviceInfo));
   memset(&m_rHIDKnobKeyDeviceInfo, 0, sizeof(trHIDDeviceInfo));
   memset(m_s8KnobKeydata, 0, KNOB_DATA_SIZE);

}

/***************************************************************************
 ** FUNCTION:  spi_tclDiPOKeyInputAdapterImpl* spi_tclDiPOKeyInputAdapterImpl::
 **            poGetDiPOKeyInputAdapterInstance()
 ***************************************************************************/
spi_tclDiPOKeyInputAdapterImpl* spi_tclDiPOKeyInputAdapterImpl::poGetDiPOKeyInputAdapterInstance()
{
   ETG_TRACE_USR1(("spi_tclDiPOKeyInputAdapterImpl::poGetDiPOKeyInputAdapterInstance entered"));
   rKeyAdapterSyncLock.s16Lock();
   spi_tclDiPOKeyInputAdapterImpl *poKeyInputAdapter = m_poDiPOKeyInputAdapter;
   rKeyAdapterSyncLock.vUnlock();
   return poKeyInputAdapter;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclDiPOKeyInputAdapterImpl::vSetConsumerKey
 **
 ***************************************************************************/
t_Void spi_tclDiPOKeyInputAdapterImpl::vSetConsumerKey(std::vector<t_U8> u8ConsumerKey)
{
   ETG_TRACE_USR1(("spi_tclDiPOKeyInputAdapterImpl::vSetConsumerKey entered"));
   coConsumerKeyDescriptorSize = u8ConsumerKey.size();
   ETG_TRACE_USR1(("spi_tclDiPOKeyInputAdapterImpl::vSetConsumerKey - coConsumerKeyDescriptorSize (%lu)", coConsumerKeyDescriptorSize));

   for (size_t siarrCtr = 0; siarrCtr < u8ConsumerKey.size(); siarrCtr++)
   {
      coConsumerKeyDescriptor[siarrCtr] = u8ConsumerKey[siarrCtr];
      ETG_TRACE_USR1(("spi_tclDiPOKeyInputAdapterImpl::vSetKeySupported coConsumerKeyDescriptor[%lu] = %x",siarrCtr ,coConsumerKeyDescriptor[siarrCtr]));
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPOKeyInputAdapterImpl::vSetPhoneKey
 **
 ***************************************************************************/
t_Void spi_tclDiPOKeyInputAdapterImpl::vSetPhoneKey(std::vector<t_U8> u8PhoneKey)
{
   ETG_TRACE_USR1(("spi_tclDiPOKeyInputAdapterImpl::vSetPhoneKey entered"));
   coTelephonyKeyDescriptorSize = u8PhoneKey.size();
   ETG_TRACE_USR1(("spi_tclDiPOKeyInputAdapterImpl::vSetPhoneKey - coTelephonyKeyDescriptorSize (%lu)", coTelephonyKeyDescriptorSize));
   for (size_t siarrCtr = 0; siarrCtr < u8PhoneKey.size(); siarrCtr++)
   {
      coTelephonyKeyDescriptor[siarrCtr] = u8PhoneKey[siarrCtr];
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPOKeyInputAdapterImpl::vSetKnobKey
 **
 ***************************************************************************/
t_Void spi_tclDiPOKeyInputAdapterImpl::vSetKnobKey(std::vector<t_U8> u8KnobKey)
{
   ETG_TRACE_USR1(("spi_tclDiPOKeyInputAdapterImpl::vSetKnobKey entered"));
   coKnobKeyDescriptorSize = u8KnobKey.size();
   ETG_TRACE_USR1(("spi_tclDiPOKeyInputAdapterImpl::vSetKnobKey - coKnobKeyDescriptorSize (%lu)", coKnobKeyDescriptorSize));
   for (size_t siarrCtr = 0; siarrCtr < u8KnobKey.size(); siarrCtr++)
   {
      coKnobKeyDescriptor[siarrCtr] = u8KnobKey[siarrCtr];
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPOKeyInputAdapterImpl::vSetConsKeybitmapKey
 **
 ***************************************************************************/
t_Void spi_tclDiPOKeyInputAdapterImpl::vSetConsKeybitmapKey(std::vector<t_U8> u8ConsKeyMapBitValues)
{
   ETG_TRACE_USR1(("spi_tclDiPOKeyInputAdapterImpl::vSetConsKeybitmapKey entered"));
   ETG_TRACE_USR1(("spi_tclDiPOKeyInputAdapterImpl::vSetKnobKey - u8ConsKeyMapBitValues.size (%lu)", u8ConsKeyMapBitValues.size()));
   for (size_t siarrCtr = 0; siarrCtr < u8ConsKeyMapBitValues.size(); siarrCtr++)
   {
      au8KeyMapBitValues[siarrCtr] = u8ConsKeyMapBitValues[siarrCtr];
      ETG_TRACE_USR1(("spi_tclDiPOKeyInputAdapterImpl::vSetKeySupported au8KeyMapBitValues[%lu] = %d",siarrCtr ,au8KeyMapBitValues[siarrCtr]));
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPOKeyInputAdapterImpl::vSetTelKeybitmapKey
 **
 ***************************************************************************/
t_Void spi_tclDiPOKeyInputAdapterImpl::vSetTelKeybitmapKey(std::vector<t_U8> u8TelKeyMapBitValues)
{
   ETG_TRACE_USR1(("spi_tclDiPOKeyInputAdapterImpl::vSetTelKeybitmapKey entered"));
   ETG_TRACE_USR1(("spi_tclDiPOKeyInputAdapterImpl::vSetKnobKey - u8TelKeyMapBitValues.size (%lu)", u8TelKeyMapBitValues.size()));
   for (size_t siarrCtr = 0; siarrCtr < u8TelKeyMapBitValues.size(); siarrCtr++)
   {
      au8TelKeyMapBitValues[siarrCtr] = u8TelKeyMapBitValues[siarrCtr];
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPOKeyInputAdapterImpl::vSetKnobKeybitmapKey
 **
 ***************************************************************************/
t_Void spi_tclDiPOKeyInputAdapterImpl::vSetKnobKeybitmapKey(std::vector<t_U8> u8KnobKeyMapBitValues)
{
   ETG_TRACE_USR1(("spi_tclDiPOKeyInputAdapterImpl::vSetKnobKeybitmapKey entered"));
   ETG_TRACE_USR1(("spi_tclDiPOKeyInputAdapterImpl::vSetKnobKey - u8KnobKeyMapBitValues.size (%lu)", u8KnobKeyMapBitValues.size()));
   for (size_t siarrCtr = 0; siarrCtr < u8KnobKeyMapBitValues.size(); siarrCtr++)
   {
      au8KnobKeyMapBitValues[siarrCtr] = u8KnobKeyMapBitValues[siarrCtr];
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDiPOKeyInputAdapterImpl::Initialize()
 ***************************************************************************/
/* ADIT Interface Implementation.*/
t_Bool spi_tclDiPOKeyInputAdapterImpl::Initialize(const IConfiguration& rfConfig, IInputReceiver& rfReceiver,
         SessionId inSessionId)
{
   SPI_INTENTIONALLY_UNUSED(inSessionId);
   ETG_TRACE_USR1(("spi_tclDiPOKeyInputAdapterImpl::Initialize entered"));
   t_Bool bRetVal = false;

   m_poIInputReceiver = &rfReceiver;
   SPI_NORMAL_ASSERT(NULL == m_poIInputReceiver);
   if (coConsumerKeyDescriptorSize)
   {
      // populate the Consumer Key device info.
      m_rHIDConsumerKeyDeviceInfo.HIDProductId =
               static_cast<t_S32>(rfConfig.GetNumber("key-input-hid-consumer-product-id", 0LL));
      m_rHIDConsumerKeyDeviceInfo.HIDVendorId =
               static_cast<t_S32>(rfConfig.GetNumber("key-input-hid-consumer-vendor-id", 0LL));

      t_String szTempStr;
      size_t sTempStrLen;
      szTempStr = rfConfig.GetItem("key-input-consumer-device-name", "consumer-keys");
      sTempStrLen = szTempStr.length();
      t_Char* pczConsumerDevName = new t_Char[sTempStrLen + 1];
      if (NULL != pczConsumerDevName)
      {
         memset(pczConsumerDevName, '\0', (sTempStrLen + 1));
         strncpy(pczConsumerDevName, szTempStr.c_str(), sTempStrLen);
         m_rHIDConsumerKeyDeviceInfo.Name = pczConsumerDevName;
      }

      szTempStr = rfConfig.GetItem("key-input-hid-consumer-uuid", "");
      sTempStrLen = szTempStr.length();
      t_Char* pczConsumerUUID = new t_Char[sTempStrLen + 1];
      if (NULL != pczConsumerUUID)
      {
         memset(pczConsumerUUID, '\0', (sTempStrLen + 1));
         strncpy(pczConsumerUUID, szTempStr.c_str(), sTempStrLen);
         m_rHIDConsumerKeyDeviceInfo.UUID = pczConsumerUUID;
      }

      szTempStr = rfConfig.GetItem("display-uuid", "");
      sTempStrLen = szTempStr.length();
      t_Char* pczDisplayUUID = new t_Char[sTempStrLen + 1];
      if (NULL != pczDisplayUUID)
      {
         memset(pczDisplayUUID, '\0', (sTempStrLen + 1));
         strncpy(pczDisplayUUID, szTempStr.c_str(), sTempStrLen);
         m_rHIDConsumerKeyDeviceInfo.DisplayUUID = pczDisplayUUID;
      }
      m_rHIDConsumerKeyDeviceInfo.HIDCountryCode = static_cast<t_S32>(rfConfig.GetNumber("key-input-hid-country-code",
               0LL));
      m_rHIDConsumerKeyDeviceInfo.HIDDescriptor = coConsumerKeyDescriptor;
      m_rHIDConsumerKeyDeviceInfo.HIDDescriptorLen = coConsumerKeyDescriptorSize;
      rKeyAdapterSyncLock.s16Lock();
      // Attach the Consumer Input Device Info
      bRetVal = m_poIInputReceiver->AttachInput(m_rHIDConsumerKeyDeviceInfo);
      rKeyAdapterSyncLock.vUnlock();
      ETG_TRACE_USR2(("[DESC]: HID consumer device attach status  = %d", ETG_ENUM(STATUS, bRetVal)));

   }

   if (coTelephonyKeyDescriptorSize)
   {
      // populate the Telephone Key device info.
      m_rHIDTelKeyDeviceInfo.HIDProductId = static_cast<t_S32>(rfConfig.GetNumber("key-input-hid-telephony-product-id",
               0LL));
      m_rHIDTelKeyDeviceInfo.HIDVendorId = static_cast<t_S32>(rfConfig.GetNumber("key-input-hid-telephony-vendor-id",
               0LL));

      t_String szTempStr;
      size_t sTempStrLen;
      szTempStr = rfConfig.GetItem("key-input-telephony-device-name", "telephony-keys");
      sTempStrLen = szTempStr.length();
      t_Char* pczTelephonyDevName = new t_Char[sTempStrLen + 1];
      if (NULL != pczTelephonyDevName)
      {
         memset(pczTelephonyDevName, '\0', (sTempStrLen + 1));
         strncpy(pczTelephonyDevName, szTempStr.c_str(), sTempStrLen);
         m_rHIDTelKeyDeviceInfo.Name = pczTelephonyDevName;
      }

      //Since the UUID value for telephony key input in dipo.cfg is invalidly formatted,use another UUID
      szTempStr = rfConfig.GetItem("key-input-hid-telephony-uuid ", "e3a8f0b8-d7c1-4d94-bdde-40bbfdcf6c05");
      sTempStrLen = szTempStr.length();
      t_Char* pczTelephonyUUID = new t_Char[sTempStrLen + 1];
      if (NULL != pczTelephonyUUID)
      {
         memset(pczTelephonyUUID, '\0', (sTempStrLen + 1));
         strncpy(pczTelephonyUUID, szTempStr.c_str(), sTempStrLen);
         m_rHIDTelKeyDeviceInfo.UUID = pczTelephonyUUID;
      }

      szTempStr = rfConfig.GetItem("display-uuid", "");
      sTempStrLen = szTempStr.length();
      t_Char* pczDisplayUUID = new t_Char[sTempStrLen + 1];
      if (NULL != pczDisplayUUID)
      {
         memset(pczDisplayUUID, '\0', (sTempStrLen + 1));
         strncpy(pczDisplayUUID, szTempStr.c_str(), sTempStrLen);
         m_rHIDTelKeyDeviceInfo.DisplayUUID = pczDisplayUUID;
      }
      m_rHIDTelKeyDeviceInfo.HIDCountryCode = static_cast<t_S32>(rfConfig.GetNumber("key-input-hid-country-code", 0LL));

      m_rHIDTelKeyDeviceInfo.HIDDescriptor = coTelephonyKeyDescriptor;
      m_rHIDTelKeyDeviceInfo.HIDDescriptorLen = coTelephonyKeyDescriptorSize;
      rKeyAdapterSyncLock.s16Lock();
      bRetVal = m_poIInputReceiver->AttachInput(m_rHIDTelKeyDeviceInfo);
      rKeyAdapterSyncLock.vUnlock();
      ETG_TRACE_USR2(("[DESC]: HID telephony device attach status  = %d", ETG_ENUM(STATUS, bRetVal)));

   }
   if (coKnobKeyDescriptorSize /*&& true == m_bKnobKeySupported*/)
   {
      //populate the Knob Key device info.
      m_rHIDKnobKeyDeviceInfo.HIDProductId = KNOB_PRODUCT_ID
      ;

      m_rHIDKnobKeyDeviceInfo.HIDVendorId = KNOB_VENDOR_ID
      ;

      t_String szTempStr;
      size_t sTempStrLen;
      szTempStr = rfConfig.GetItem("key-input-knob-device-name", "knob-key");
      sTempStrLen = szTempStr.length();
      t_Char* pczKnobDevName = new t_Char[sTempStrLen + 1];
      if (NULL != pczKnobDevName)
      {
         memset(pczKnobDevName, '\0', (sTempStrLen + 1));
         strncpy(pczKnobDevName, szTempStr.c_str(), sTempStrLen);
         m_rHIDKnobKeyDeviceInfo.Name = pczKnobDevName;
      }

      //UUID for Knob key has been randomly generated.
      szTempStr = rfConfig.GetItem("key-input-hid-knob-uuid", "dc68c8c0-5d31-4c81-b184-a1529959ce8d");
      sTempStrLen = szTempStr.length();
      t_Char* pczKnobUUID = new t_Char[sTempStrLen + 1];
      if (NULL != pczKnobUUID)
      {
         memset(pczKnobUUID, '\0', (sTempStrLen + 1));
         strncpy(pczKnobUUID, szTempStr.c_str(), sTempStrLen);
         m_rHIDKnobKeyDeviceInfo.UUID = pczKnobUUID;
      }

      szTempStr = rfConfig.GetItem("display-uuid", "");
      sTempStrLen = szTempStr.length();
      t_Char* pczDisplayUUID = new t_Char[sTempStrLen + 1];
      if (NULL != pczDisplayUUID)
      {
         memset(pczDisplayUUID, '\0', (sTempStrLen + 1));
         strncpy(pczDisplayUUID, szTempStr.c_str(), sTempStrLen);
         m_rHIDKnobKeyDeviceInfo.DisplayUUID = pczDisplayUUID;
      }
      m_rHIDKnobKeyDeviceInfo.HIDCountryCode =
               static_cast<t_S32>(rfConfig.GetNumber("key-input-hid-country-code", 0LL));

      m_rHIDKnobKeyDeviceInfo.HIDDescriptor = coKnobKeyDescriptor;

      m_rHIDKnobKeyDeviceInfo.HIDDescriptorLen = coKnobKeyDescriptorSize;
      rKeyAdapterSyncLock.s16Lock();
      bRetVal = m_poIInputReceiver->AttachInput(m_rHIDKnobKeyDeviceInfo);
      rKeyAdapterSyncLock.vUnlock();
      ETG_TRACE_USR2(("[DESC]: Knob encoder attach status  = %d", ETG_ENUM(STATUS, bRetVal)));

   }

   return bRetVal;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiPOKeyInputAdapterImpl::bSendKeyEvent()
 ***************************************************************************/
t_Bool spi_tclDiPOKeyInputAdapterImpl::bSendKeyEvent(tenKeyMode enKeyMode, tenKeyCode enKeyCode)
{
   ETG_TRACE_USR1(("spi_tclDiPOKeyInputAdapterImpl::bSendKeyEvent entered"));
   ETG_TRACE_USR4(("[PARAM]:bSendKeyEvent - Key Mode  = %d", enKeyMode));
   ETG_TRACE_USR4(("[PARAM]:bSendKeyEvent - Key code  = %d", enKeyCode));
   // HID device consumer Key Input report
   trHIDInputReport rHIDConsumerKeyInputReport;

   // HID device Telephone Key Input report
   trHIDInputReport rHIDTelKeyInputReport;

   t_Bool bIsValidKey = false;
   t_Bool bRetVal = false;

   //!Flag that checks if the key received is a consumer key
   t_Bool bIsConsumerKey = true;

   /*While populating the key data descriptor,set the bit corresponding to the key to 1,if the key is pressed.
    If the key is released,set the corresponding bit to 0 */
   switch (enKeyCode)
   {

      case e32DEV_BACKWARD:
      {
         if (0 != au8KeyMapBitValues[5])
         {
            m_u8ConsumerKeydata =
                     (e8KEY_PRESS == enKeyMode) ?
                              (m_u8ConsumerKeydata | au8KeyMapBitValues[5]) :
                              (m_u8ConsumerKeydata ^ au8KeyMapBitValues[5]);
            bIsValidKey = true;
         }

      }
         break;
      case e32MULTIMEDIA_NEXT:
      {
         if (0 != au8KeyMapBitValues[1])
         {
            m_u8ConsumerKeydata =
                     (e8KEY_PRESS == enKeyMode) ?
                              (m_u8ConsumerKeydata | au8KeyMapBitValues[1]) :
                              (m_u8ConsumerKeydata ^ au8KeyMapBitValues[1]);
            bIsValidKey = true;
         }

      }
         break;

      case e32MULTIMEDIA_PREVIOUS:
      {
         if (0 != au8KeyMapBitValues[2])
         {
            m_u8ConsumerKeydata =
                     (e8KEY_PRESS == enKeyMode) ?
                              (m_u8ConsumerKeydata | au8KeyMapBitValues[2]) :
                              (m_u8ConsumerKeydata ^ au8KeyMapBitValues[2]);
            bIsValidKey = true;
         }

      }
         break;

      case e32MULTIMEDIA_PLAY_PAUSE:
      {
         if (0 != au8KeyMapBitValues[0])
         {
            m_u8ConsumerKeydata = 
                     (e8KEY_PRESS == enKeyMode) ?
                              (m_u8ConsumerKeydata | au8KeyMapBitValues[0]) :
                              (m_u8ConsumerKeydata ^ au8KeyMapBitValues[0]);
            bIsValidKey = true;
         }
      }
      break;


      case e32MULTIMEDIA_PLAY:
      {
         if (0 != au8KeyMapBitValues[3])
         {
            m_u8ConsumerKeydata =
                     (e8KEY_PRESS == enKeyMode) ?
                              (m_u8ConsumerKeydata | au8KeyMapBitValues[3]) :
                              (m_u8ConsumerKeydata ^ au8KeyMapBitValues[3]);
            bIsValidKey = true;
         }
      }
         break;

      case e32MULTIMEDIA_PAUSE:
      {
         if (0 != au8KeyMapBitValues[4])
         {
            m_u8ConsumerKeydata =
                     (e8KEY_PRESS == enKeyMode) ?
                              (m_u8ConsumerKeydata | au8KeyMapBitValues[4]) :
                              (m_u8ConsumerKeydata ^ au8KeyMapBitValues[4]);
            bIsValidKey = true;
         }
      }
         break;

   case e32DEV_HOME:
      {
         if(0 != au8KeyMapBitValues[6])
         {
                 m_u8ConsumerKeydata = (e8KEY_PRESS==enKeyMode)?(m_u8ConsumerKeydata|au8KeyMapBitValues[6]):(m_u8ConsumerKeydata^au8KeyMapBitValues[6]);
                 bIsValidKey = true;
         }
      }
      break;

   case e32DEV_PHONE_CALL:
      {
         if (0 != au8TelKeyMapBitValues[0])
         {
            m_u8TelephoneKeydata =
                     (e8KEY_PRESS == enKeyMode) ?
                              (m_u8TelephoneKeydata | au8TelKeyMapBitValues[0]) :
                              (m_u8TelephoneKeydata ^ au8TelKeyMapBitValues[0]);
            bIsValidKey = true;
         }
         bIsConsumerKey = false;
      }
         break;

      case e32DEV_PHONE_END:
      {
         if (0 != au8TelKeyMapBitValues[2])
         {
            m_u8TelephoneKeydata =
                     (e8KEY_PRESS == enKeyMode) ?
                              (m_u8TelephoneKeydata | au8TelKeyMapBitValues[2]) :
                              (m_u8TelephoneKeydata ^ au8TelKeyMapBitValues[2]);
            bIsValidKey = true;
         }

         bIsConsumerKey = false;
      }
         break;
      case e32DEV_PHONE_FLASH:
      {
         if (0 != au8TelKeyMapBitValues[1])
         {
            m_u8TelephoneKeydata =
                     (e8KEY_PRESS == enKeyMode) ?
                              (m_u8TelephoneKeydata | au8TelKeyMapBitValues[1]) :
                              (m_u8TelephoneKeydata ^ au8TelKeyMapBitValues[1]);
            bIsValidKey = true;
         }
         bIsConsumerKey = false;
      }
      break;

   case e32DEV_PHONE_MUTE:
      {
         if(0 != au8TelKeyMapBitValues[3])
         {
            m_u8TelephoneKeydata = (e8KEY_PRESS == enKeyMode) ? (m_u8TelephoneKeydata|au8TelKeyMapBitValues[3]) : (m_u8TelephoneKeydata^ au8TelKeyMapBitValues[3]);
            bIsValidKey = true;
         }
         bIsConsumerKey = false;
      }
         break;

   case e32DEV_MENU:
      {
         if (0 != au8KnobKeyMapBitValues[0])
         {
            m_s8KnobKeydata[0] = static_cast<t_S8>(
                     (e8KEY_PRESS == enKeyMode) ?
                              (m_s8KnobKeydata[0] | au8KnobKeyMapBitValues[0]) :
                              (m_s8KnobKeydata[0] ^ au8KnobKeyMapBitValues[0]));
            //Send the selection action via knob key HID descriptor only.
            bSendKnobKeyEvent();
         }
      }
         break;

      case e32TWODKNOB0_SHIFT_RIGHT:
      {
         if (true == m_bKnobKeyDirectionalPadSupported)
         {
            m_s8KnobKeydata[2] = (e8KEY_PRESS == enKeyMode) ? m_u8KnobKeyDirectionalPadMax : 0x00;
            bSendKnobKeyEvent();
         }
      }
         break;

      case e32TWODKNOB0_SHIFT_LEFT:
      {
         if (true == m_bKnobKeyDirectionalPadSupported)
         {
            m_s8KnobKeydata[2] = (e8KEY_PRESS == enKeyMode) ? m_u8KnobKeyDirectionalPadMin : 0x00;
            bSendKnobKeyEvent();
         }
      }
         break;

      case e32TWODKNOB0_SHIFT_UP:
      {
         if (true == m_bKnobKeyDirectionalPadSupported)
         {
            m_s8KnobKeydata[3] = (e8KEY_PRESS == enKeyMode) ? m_u8KnobKeyDirectionalPadMin : 0x00;
            bSendKnobKeyEvent();
         }
      }
         break;

      case e32TWODKNOB0_SHIFT_DOWN:
      {
         if (true == m_bKnobKeyDirectionalPadSupported)
         {
            m_s8KnobKeydata[3] = (e8KEY_PRESS == enKeyMode) ? m_u8KnobKeyDirectionalPadMax : 0x00;
            bSendKnobKeyEvent();
         }
      }
         break;

      default:
      {
         ETG_TRACE_ERR(("[ERR]: No valid key found to Send to iPhone"));
      }
         break;
   }

   rKeyAdapterSyncLock.s16Lock();
   if ((NULL != m_poIInputReceiver) && (true == bIsValidKey))
   {
      if (true == bIsConsumerKey)
      {
         //Populate the HID input report for Consumer Key data
         rHIDConsumerKeyInputReport.HIDReportData = &m_u8ConsumerKeydata;
         rHIDConsumerKeyInputReport.HIDReportLen = sizeof(m_u8ConsumerKeydata);
         rHIDConsumerKeyInputReport.UUID = m_rHIDConsumerKeyDeviceInfo.UUID;

         bRetVal = m_poIInputReceiver->SendInput(rHIDConsumerKeyInputReport);
      }
      else
      {
         //Populate the HID input report for Telephone Key data
         rHIDTelKeyInputReport.HIDReportData = &m_u8TelephoneKeydata;
         rHIDTelKeyInputReport.HIDReportLen = sizeof(m_u8TelephoneKeydata);
         rHIDTelKeyInputReport.UUID = m_rHIDTelKeyDeviceInfo.UUID;

         bRetVal = m_poIInputReceiver->SendInput(rHIDTelKeyInputReport);
      }
   }
   rKeyAdapterSyncLock.vUnlock();
   return bRetVal;
}

/***************************************************************************
** FUNCTION: t_Bool spi_tclDiPOKeyInputAdapterImpl::bSendKnobKeyEvent()
***************************************************************************/
t_Bool spi_tclDiPOKeyInputAdapterImpl::bSendKnobKeyEvent(t_S8 s8EncoderDeltaCounts)
{
   ETG_TRACE_USR1(("spi_tclDiPOKeyInputAdapterImpl::bSendKnobKeyEvent entered"));

   ETG_TRACE_USR4(("[PARAM]:bSendKeyEvent - Delta counts  = %d", s8EncoderDeltaCounts));
   /**
    * @Note : The HID descriptor for Knob key actually looks like
    *
    *      Byte 1           Byte 2             Byte 3             Byte 4
    *  _ _ _ _ _ _ _ _   _ _ _ _ _ _ _ _   _ _ _ _ _ _ _ _   _ _ _ _ _ _ _ _
    * |_|_|_|_|_|_|_|x| |x|x|x|x|x|x|x|x| |x|x|x|x|x|x|x|x| |x|x|x|x|x|x|x|x|
    *
    * $Byte 1
    *  0th position - Button 1 (Used for selection)
    * $Byte 2
    *  8 bits for encoder change in delta counts(signed).
    * $Byte 3
    *  press/depress value for X
    * $Byte 4
    *  press/depress value for Y
    *
    * @Note
    * To denote the same in HID descriptor and maintain bit alignment
    * added Inverse Report Count and Report Size.
    */

   t_Bool bRetVal = false;
   // HID device Telephone Key Input report
   trHIDInputReport rHIDKnobKeyInputReport;

   rKeyAdapterSyncLock.s16Lock();
   //Assign the deltaCounts to the second byte of the HID array.
   if (((true == m_bKnobKeySupported) || (true == m_bKnobKeyDirectionalPadSupported)) && (0 != coKnobKeyDescriptorSize)
            && (NULL != m_poIInputReceiver))
   {
      t_U8 u8EncoderDeltaCtr = 0;
      do{
         if(0 != s8EncoderDeltaCounts)
         {
            m_s8KnobKeydata[1] = (s8EncoderDeltaCounts > 0) ? 1 : -1;
         }
         else
         {
            m_s8KnobKeydata[1] = 0;
         }

         rHIDKnobKeyInputReport.HIDReportData = (uint8_t*) (m_s8KnobKeydata);
         rHIDKnobKeyInputReport.HIDReportLen = sizeof(m_s8KnobKeydata);
         rHIDKnobKeyInputReport.UUID = m_rHIDKnobKeyDeviceInfo.UUID;

         bRetVal = m_poIInputReceiver->SendInput(rHIDKnobKeyInputReport);
         ++u8EncoderDeltaCtr;

         ETG_TRACE_USR2(("[DESC]: Knob key event (Keydata = %d) sent status = %d", m_s8KnobKeydata[1], ETG_ENUM(STATUS, bRetVal)));
      }while(u8EncoderDeltaCtr < std::abs(s8EncoderDeltaCounts));
   }
   rKeyAdapterSyncLock.vUnlock();

   return bRetVal;
}

// Need to check whether to remove the function or not
/***************************************************************************
** FUNCTION: t_Void spi_tclDiPOKeyInputAdapterImpl::vSetDisplayInputParam()
***************************************************************************/
t_Void spi_tclDiPOKeyInputAdapterImpl::vSetDisplayInputParam(t_U8 u8DisplayInput)
{
   ETG_TRACE_USR1(("spi_tclDiPOKeyInputAdapterImpl::vSetDisplayInputParam entered"));
   //@Note: expand the switch case for any other input capability handling
   switch (u8DisplayInput)
   {
      case LOW_FIDELITY_TOUCH_WITH_KNOB:
      case HIGH_FIDELITY_TOUCH_WITH_KNOB:
      {
//         m_bKnobKeySupported = true;
      }
         break;
      case LOW_FIDELITY_TOUCH:
      case HIGH_FIDELITY_TOUCH:
      {
//         m_bKnobKeySupported = false;
      }
         break;
      default:
      {
         ETG_TRACE_ERR(("[ERR]:Invalid display input entry received"));
      }
         break;
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclDiPOKeyInputAdapterImpl::vSetKnobKeyDirectionalPadSupport()
 ***************************************************************************/
t_Void spi_tclDiPOKeyInputAdapterImpl::vSetKnobKeyDirectionalPadSupport(t_Bool bKnobKeyDirectionalPadSupport,t_Bool bRotaryKnobSupport)
{
   ETG_TRACE_USR1(("spi_tclDiPOKeyInputAdapterImpl::vSetKnobKeyDirectionalPadSupport entered - %d", ETG_ENUM(STATUS, bKnobKeyDirectionalPadSupport)));
   m_bKnobKeyDirectionalPadSupported = bKnobKeyDirectionalPadSupport;
   m_bKnobKeySupported               = bRotaryKnobSupport;
}
