/*!
*******************************************************************************
* \file              spi_tclDiPOControlAdapterImpl.cpp
* \brief             DiPO IControlAdapter impmentation
*******************************************************************************
\verbatim
PROJECT:        G3G
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    DiPO IControlAdapter implementation. This class implement all the 
                functions of the IControlAdapter interface of ADIT. This class
                is used to handle the callbacks from iOS device.
COPYRIGHT:      &copy; RBEI

HISTORY:
Date      |  Author                      | Modifications
21.1.2014 |  Shihabudheen P M            | Initial Version
18.4.2014 |  Shihabudheen P M            | Modified for resource arbitration
07.05.2014|  Shihabudheen P M            | Modified for resource arbitration 
13.10.2014| Vinoop U					 | Changes to set OEM icon Image And OEM icon label
25.10.2014 |  Shihabudheen P M           | updated video pipeline with dis-reorder=true.
15.12.2014 | Tejaswini HB                | Lint fix
27.05.2015 |  Tejaswini H B(RBEI/ECP2)   | Added Lint comments to suppress C++11 Errors
20.08.2015 |  Shihabudheen P M           | Added methods to dynamically set the limited UI elements.
08.01.2015 |  Shihabudheen P M           | Code improvements.
28.03.2016 |  Rachana L Achar            | Fix for PSARCCB-4588

\endverbatim
******************************************************************************/

/******************************************************************************
| includes:
| 1)system- and project- includes
| 2)needed interfaces from external components
| 3)internal and external interfaces from this component
|----------------------------------------------------------------------------*/
#include <netinet/ip.h>
#include <arpa/inet.h>
#include "IPCMessageQueue.h"
#include "spi_tclDiPOControlAdapterImpl.h"
#include "Timer.h"


#include "spi_tclDiPOAdapterMsgQInterface.h"

using namespace std;

#define SPI_ENABLE_DLT //enable DLT
#define SPI_LOG_CLASS Spi_CarPlay

#include "Trace.h"
//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e19 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e58 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e48 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e808 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e64 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e746 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//! import the DLT context 
LOG_IMPORT_CONTEXT(Spi_CarPlay);

#define MAX_BUF_SIZE   256
spi_tclDiPOControlAdapterImpl* spi_tclDiPOControlAdapterImpl::m_poDiPOControlAdapter = NULL;
trModeChange spi_tclDiPOControlAdapterImpl::m_AccessoryMode;

t_String spi_tclDiPOControlAdapterImpl::m_szBlutoothMac = "";
t_String spi_tclDiPOControlAdapterImpl::m_szFirmwareVersion = "";
t_String spi_tclDiPOControlAdapterImpl::m_szHardwareVersion = "";
t_Bool spi_tclDiPOControlAdapterImpl::m_bCurNightModeInfo = false;
t_Bool spi_tclDiPOControlAdapterImpl::m_bCurDriveModeInfo = false;
trDiPODisplayAttributes spi_tclDiPOControlAdapterImpl::m_rDiPODisplayAttr;
trDiPOTouchInputAttributes spi_tclDiPOControlAdapterImpl::m_rDiPOTouchInputAttr;
trDiPOInfoRespParam spi_tclDiPOControlAdapterImpl::m_rDiPOInfoRespParam;
std::map<t_String, tenDiPOMainAudioType> spi_tclDiPOControlAdapterImpl::m_mAudioTypeMap =
		spi_tclDiPOControlAdapterImpl::vInitializeAudioTypeMap();
t_U8 spi_tclDiPOControlAdapterImpl::m_u8LimitedUIElementCount = 0;
trVehicleInformation spi_tclDiPOControlAdapterImpl::m_rVehicleInfo;
trVehicleInformation spi_tclDiPOControlAdapterImpl::m_rETCStatusInfo;
t_Bool spi_tclDiPOControlAdapterImpl::m_bAudioLastMode = false;
t_Bool spi_tclDiPOControlAdapterImpl::m_bDisplayLastMode = false;
trExtendedFeaturesSupport spi_tclDiPOControlAdapterImpl::m_rExtendedFeaturesSupport;

static t_U32 su32TimerInterval=1000;
static timer_t srTimerID  = 0;
static Lock rAdaperSyncLock;

/***************************************************************************
 ** FUNCTION:  spi_tclDiPOControlAdapterImpl::spi_tclDiPOControlAdapterImpl()
 ***************************************************************************/
spi_tclDiPOControlAdapterImpl::spi_tclDiPOControlAdapterImpl():
 m_poIDynamicConfig(NULL), 
 m_poIControlReceiver(NULL), 
 m_CurrAudioMessage(e8AUDIO_MESSAGE),
 m_bIsSessionStarted(false),
 m_enSessionTransport(CarPlay_Over_USB),
 u32PrevAudioBorrowCnt(0),
 u32PrevScreenBorrowCnt(0),
 m_bIsdropAudio(false)
{
   ETG_TRACE_USR1((" spi_tclDiPOControlAdapterImpl::spi_tclDiPOControlAdapterImpl entered"));
   rAdaperSyncLock.s16Lock();
   m_poDiPOControlAdapter = this;
   rAdaperSyncLock.vUnlock();
   
   ETG_TRACE_USR1((" spi_tclDiPOControlAdapterImpl::m_AccessoryMode.screen.type %d",m_AccessoryMode.screen.type));
   ETG_TRACE_USR1((" spi_tclDiPOControlAdapterImpl::m_AccessoryMode.screen.priority %d",m_AccessoryMode.screen.priority));
   ETG_TRACE_USR1((" spi_tclDiPOControlAdapterImpl::m_AccessoryMode.screen.takeConstraint %d",m_AccessoryMode.screen.takeConstraint));
   ETG_TRACE_USR1((" spi_tclDiPOControlAdapterImpl::m_AccessoryMode.screen.borrowOrUnborrowConstraint %d",m_AccessoryMode.screen.borrowOrUnborrowConstraint));

   ETG_TRACE_USR1((" spi_tclDiPOControlAdapterImpl::m_AccessoryMode.audio.type %d",m_AccessoryMode.audio.type));
   ETG_TRACE_USR1((" spi_tclDiPOControlAdapterImpl::m_AccessoryMode.audio.priority %d",m_AccessoryMode.audio.priority));
   ETG_TRACE_USR1((" spi_tclDiPOControlAdapterImpl::m_AccessoryMode.audio.takeConstraint %d",m_AccessoryMode.audio.takeConstraint));
   ETG_TRACE_USR1((" spi_tclDiPOControlAdapterImpl::m_AccessoryMode.audio.borrowOrUnborrowConstraint %d",m_AccessoryMode.audio.borrowOrUnborrowConstraint));

   ETG_TRACE_USR1((" spi_tclDiPOControlAdapterImpl::m_AccessoryMode.speech %d",m_AccessoryMode.speech));
   ETG_TRACE_USR1((" spi_tclDiPOControlAdapterImpl::m_AccessoryMode.phone %d",m_AccessoryMode.phone));
   ETG_TRACE_USR1((" spi_tclDiPOControlAdapterImpl::m_AccessoryMode.navigation %d",m_AccessoryMode.navigation));
}

/***************************************************************************
 ** FUNCTION:  spi_tclDiPOControlAdapterImpl::~spi_tclDiPOControlAdapterImpl()
 ***************************************************************************/
spi_tclDiPOControlAdapterImpl::~spi_tclDiPOControlAdapterImpl()
{
   ETG_TRACE_USR1((" spi_tclDiPOControlAdapterImpl::~spi_tclDiPOControlAdapterImpl entered"));
   rAdaperSyncLock.s16Lock();
   m_bIsSessionStarted = false;
   m_poIDynamicConfig = NULL;
   m_poIControlReceiver = NULL;
   m_poDiPOControlAdapter = NULL;
   m_enSessionTransport = CarPlay_Over_USB;
   u32PrevAudioBorrowCnt = 0;
   u32PrevScreenBorrowCnt = 0;
   m_bIsdropAudio = false;
   rAdaperSyncLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclDiPOControlAdapterImpl::Initialize(IDynamicConfiguration ..)
 ***************************************************************************/
/* ADIT Interface Implementation.*/
t_Bool spi_tclDiPOControlAdapterImpl::Initialize(IDynamicConfiguration& rfoConfig, 
                                                 IControlReceiver& rfoControl)
{ 
   ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::Initialize entered"));

   rAdaperSyncLock.s16Lock();
   m_poIDynamicConfig   = &rfoConfig;
   SPI_NORMAL_ASSERT(NULL == m_poIDynamicConfig);
   m_poIControlReceiver = &rfoControl;
   SPI_NORMAL_ASSERT(NULL == m_poIControlReceiver);
   rAdaperSyncLock.vUnlock();

   vSetVideoConfiguration();
   vSetBluetoothIds();
   vSetNightMode(m_bCurNightModeInfo);
   vConfigLimitedUIElements();
   vSetLimitedUI(m_bCurDriveModeInfo);
   vSetOEMIconVisible();
   vSetSoftwareVersion();
   vSetHardwareVersion();
   vUpdateVehicleInformation();
   vSetExtendedFeatures();

   return ((NULL != m_poIDynamicConfig) && (NULL != m_poIControlReceiver));
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclDiPOControlAdapterImpl::Initialize(IDynamicConfiguration ..)
 ***************************************************************************/
/* ADIT Interface Implementation.*/
t_Void spi_tclDiPOControlAdapterImpl::OnSessionStart(CarPlayTransportType enTransPortType)
{
   ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::OnSessionStart: Session started over the Transport:%d",enTransPortType));

   //Get the active session transport & IP Address
   m_enSessionTransport = enTransPortType;
   vGetSessionIPAddress(m_szSessionIPAddress);


   Timer* poTimer = Timer::getInstance();
   if (NULL != poTimer)
   {
      poTimer->StartTimer( srTimerID, su32TimerInterval,
         0, this,bSessionStartTimerCb,NULL );
   }//if (NULL != poTimer) 

   u32PrevAudioBorrowCnt = 0;
   u32PrevScreenBorrowCnt = 0;
   m_bIsdropAudio = false;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclDiPOControlAdapterImpl::OnSessionEnd()
 ***************************************************************************/
/* ADIT Interface Implementation.*/
t_Void spi_tclDiPOControlAdapterImpl::OnSessionEnd()
{
   ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::OnSessionEnd entered"));

   trDiPOSessionMsg rDiPOSessionMsg;
   rDiPOSessionMsg.enMsgType = e8SESSION_MESSAGE;
   rDiPOSessionMsg.enDiPOSessionState = e8DIPO_SESSION_END;
   rDiPOSessionMsg.enSessionTransport = m_enSessionTransport;
   strncpy(rDiPOSessionMsg.szSessionIPAddress, m_szSessionIPAddress.c_str(), MAX_STR_LEN);

   rAdaperSyncLock.s16Lock();
   m_poIDynamicConfig = NULL;
   m_poIControlReceiver = NULL;
   u32PrevAudioBorrowCnt = 0;
   u32PrevScreenBorrowCnt = 0;
   m_bIsdropAudio = false;
   rAdaperSyncLock.vUnlock();

   t_Bool bStatus = bSendMessage<trDiPOSessionMsg>(rDiPOSessionMsg);
   ETG_TRACE_USR2(("[DESC]: Session termination message sent to SPI with status = %d", ETG_ENUM( STATUS, bStatus)));

}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiPOControlAdapterImpl::DropAudio(..
 ***************************************************************************/
t_Void spi_tclDiPOControlAdapterImpl::DropAudio(t_Bool bIsdropAudio)
{
    if (NULL != m_poIControlReceiver)
    {
       rAdaperSyncLock.s16Lock();
       if(m_bIsdropAudio != bIsdropAudio)
       {
          ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::DropAudio entered with bIsdropAudio = %d", ETG_ENUM( BOOL, bIsdropAudio)));
          m_bIsdropAudio = bIsdropAudio;
          m_poIControlReceiver->DropAudio(bIsdropAudio);
       }
       rAdaperSyncLock.vUnlock();
    }
}

/***************************************************************************
 ** FUNCTION:  spi_tclDiPOControlAdapterImpl* spi_tclDiPOControlAdapterImpl::poGetDiPOControlAdapterInstance()
 ***************************************************************************/
spi_tclDiPOControlAdapterImpl* spi_tclDiPOControlAdapterImpl::poGetDiPOControlAdapterInstance()
{
   ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::poGetDiPOControlAdapterInstance entered"));
   rAdaperSyncLock.s16Lock();
   spi_tclDiPOControlAdapterImpl *poAdapterHandler = m_poDiPOControlAdapter;
   rAdaperSyncLock.vUnlock();
   return poAdapterHandler;
}

/***************************************************************************
 ** FUNCTION:  bool spi_tclDiPOControlAdapterImpl::OnModesChanged()...
 ***************************************************************************/
/* ADIT Interface Implementation.*/
t_Void spi_tclDiPOControlAdapterImpl::OnModesChanged(trModeState rState)
{
   ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::OnModesChanged entered"));

   if(false == m_bIsSessionStarted)
   {
      Timer* poTimer = Timer::getInstance();
      if ((NULL != poTimer) && (0 != srTimerID))
      {
         poTimer->CancelTimer(srTimerID);
         srTimerID  =0;
      }

      //! Inform SPI about the session startup during the first mode change message
      trDiPOSessionMsg rDiPOSessionMsg;
      rDiPOSessionMsg.enMsgType = e8SESSION_MESSAGE;
      rDiPOSessionMsg.enDiPOSessionState = e8DIPO_SESSION_START;
      rDiPOSessionMsg.enSessionTransport = m_enSessionTransport;
      strncpy(rDiPOSessionMsg.szSessionIPAddress, m_szSessionIPAddress.c_str(), MAX_STR_LEN);

      t_Bool bStatus = bSendMessage<trDiPOSessionMsg>(rDiPOSessionMsg);
      ETG_TRACE_USR2(("[DESC]: Session start message sent to SPI with status = %d", ETG_ENUM( STATUS, bStatus)));
      m_bIsSessionStarted = true;
      vUpdateVehicleInformation();
   }

   // Populate message data
   trDiPORMMsgResp rDiPORMMsgResp;
   rDiPORMMsgResp.enMsgType = e8RM_RESP_MESSAGE;
   vConvertModeState(rState, rDiPORMMsgResp.rDiPOModeState);

   //In case borrow was send to iPhone earlier and IHU send the unborrow request later,
   //this unborrow should not be send to iPhone. Here, iPhone has borrowed/taken the resource (i.e; CAR->MOBILE)
   //Ticket reference - NCG3D-163914
   if((0 != u32PrevAudioBorrowCnt) &&
      (e8DIPO_ENTITY_MOBILE == rDiPORMMsgResp.rDiPOModeState.enAudio) && (m_rCurrentModeState.enAudio != rDiPORMMsgResp.rDiPOModeState.enAudio))
   {
      --u32PrevAudioBorrowCnt;
      ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::OnModesChanged - u32PrevAudioBorrowCnt = %d", u32PrevAudioBorrowCnt));
   }
   if((0 != u32PrevScreenBorrowCnt) &&
      (e8DIPO_ENTITY_MOBILE == rDiPORMMsgResp.rDiPOModeState.enScreen) && (m_rCurrentModeState.enScreen != rDiPORMMsgResp.rDiPOModeState.enScreen))
   {
      --u32PrevScreenBorrowCnt;
      ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::OnModesChanged - u32PrevScreenBorrowCnt = %d", u32PrevScreenBorrowCnt));
   } 
   // Keep track of the latest mode changed message from iPhone.
   m_rCurrentModeState = rDiPORMMsgResp.rDiPOModeState;

   ETG_TRACE_USR4(("[DESC]: modesChanged message received from iPhone with following States :: "
            "Display State = %d, Audio State = %d, Phone State = %d, Navigation state = %d, Speech State = %d"
            "Display permanent entity = %d, Main audio permanent entity = %d",
            ETG_ENUM( CARPLAY_RESOURCE_STATE, rState.screen),
            ETG_ENUM( CARPLAY_RESOURCE_STATE, rState.audio),
            ETG_ENUM( CARPLAY_APP_STATE, rState.phone),
            ETG_ENUM( CARPLAY_APP_STATE, rState.navigation),
            ETG_ENUM( CARPLAY_APP_STATE, rState.speech.entity),
            ETG_ENUM( CARPLAY_RESOURCE_STATE, rState.permScreen),
            ETG_ENUM( CARPLAY_RESOURCE_STATE, rState.permMainAudio)));

   t_Bool bStatus = bSendMessage<trDiPORMMsgResp>(rDiPORMMsgResp);
   ETG_TRACE_USR2(("[DESC]: modesChanged message with new modes sent to SPI with status = %d", ETG_ENUM( STATUS, bStatus)));
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPOControlAdapterImpl::OnRequestUI()
 ***************************************************************************/
/* ADIT Interface Implementation.*/
t_Void spi_tclDiPOControlAdapterImpl::OnRequestUI(const t_String& corfszRequestUIUrl)
{
   // switch to the native HMI
	ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::OnRequestUI entered"));

   trOnRequestUIMsg rOnRequestUIMsg;
   rOnRequestUIMsg.enMsgType =e8ON_REQUEST_UI_MESSAGE;
   if(corfszRequestUIUrl.length()< MAX_STR_LEN)
   {
      strncpy(rOnRequestUIMsg.cRequestUIURL, corfszRequestUIUrl.c_str(), MAX_STR_LEN);
   }
   t_Bool bStatus = bSendMessage<trOnRequestUIMsg>(rOnRequestUIMsg);
   ETG_TRACE_USR2(("[DESC]: onRequestUI call received from iPhone and the same is passed to SPI with status: %d", ETG_ENUM(STATUS, bStatus)));
}

/***************************************************************************
 ** FUNCTION:  bool spi_tclDiPOControlAdapterImpl::InitializeInternal()...
 ***************************************************************************/
/* ADIT Interface Implementation.*/
t_Void spi_tclDiPOControlAdapterImpl::OnAudioPrepare(AudioChannelType channel, 
                                                     const t_String& audioType)
{
   ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::OnAudioPrepare entered"));
   SPI_INTENTIONALLY_UNUSED(channel);
   SPI_INTENTIONALLY_UNUSED(audioType);
   //@Note : Nothing to do here. AudioPrepare is handled from AudioOutAdapterImpl.
}


/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPOControlAdapterImpl::OnAudioStop()
 ***************************************************************************/
/* ADIT Interface Implementation.*/
t_Void spi_tclDiPOControlAdapterImpl::OnAudioStop(AudioChannelType enChannel)
{
   ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::OnAudioStop entered"));
   SPI_INTENTIONALLY_UNUSED(enChannel);
  //@Note : Nothing to do here. AudioStop is handled from AudioOutAdapterImpl.
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPOControlAdapterImpl::OnGetBluetoothIDs()
 ***************************************************************************/
/* ADIT Interface Implementation.*/
t_Void spi_tclDiPOControlAdapterImpl::OnGetBluetoothIDs(std::list<t_String>& deviceIDs)
{
   ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::OnGetBluetoothIDs entered"));
   deviceIDs.push_back(m_szBlutoothMac);
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPOControlAdapterImpl::OnGetNightMode()
 ***************************************************************************/
/* ADIT Interface Implementation.*/
t_Bool spi_tclDiPOControlAdapterImpl::OnGetNightMode()
{
   ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::OnGetNightMode entered"));
   return true;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPOControlAdapterImpl::OnVideoPlaybackStarted()
 ***************************************************************************/
/* ADIT Interface Implementation.*/
void spi_tclDiPOControlAdapterImpl::OnVideoPlaybackStarted()
{
   ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::OnVideoPlaybackStarted entered"));
   trVideoPlayBackStatusMsg rVideoPlayBackStatusMsg;
   rVideoPlayBackStatusMsg.bIsPlaybackStarted = true;

   t_Bool bStatus = bSendMessage<trVideoPlayBackStatusMsg>(rVideoPlayBackStatusMsg);
   ETG_TRACE_USR2(("[DESC]: Start video playback request is received from iPhone and the same is passed to SPI with status: %d",
                   ETG_ENUM(STATUS, bStatus)));
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPOControlAdapterImpl::OnRampVolume()
 ***************************************************************************/
/* ADIT Interface Implementation.*/
t_Void spi_tclDiPOControlAdapterImpl::OnRampVolume
(
 t_Double d64FinalVolume, 
 t_Double d64Duration
)
{
   ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::OnRampVolume entered"));
   trAudioDuckMsg rDuckMsg;

   rDuckMsg.enMsgType = e8AUDIO_DUCK_MESSAGE;
   rDuckMsg.dFinalVolume = d64FinalVolume;
   rDuckMsg.dDurationInMs = d64Duration;

   t_Bool bStatus = bSendMessage<trAudioDuckMsg>(rDuckMsg);
   ETG_TRACE_USR2(("[DESC]: Audio duck request received from iPhone with finalVolume = %f ,duration = %f and request forwarded to SPI with status = %d",
		   d64FinalVolume, d64Duration,  ETG_ENUM( STATUS, bStatus)));

}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPOControlAdapterImpl::OnDisableBluetooth()
 ***************************************************************************/
/* ADIT Interface Implementation.*/
t_Void spi_tclDiPOControlAdapterImpl::OnDisableBluetooth(const t_String& szDeviceId)
{
   ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::OnDisableBluetooth entered"));
   t_Bool bStatus = false;
   ETG_TRACE_USR2(("[DESC]: Disable Bluetooth command received from iPhone with MAC address : %s", szDeviceId.c_str()));
   trOnDisBluetoothMsg rOnDisBluetoothMsg;
   rOnDisBluetoothMsg.enMsgType = e8DISABLE_BLUETOOTH_MESSAGE;
   strncpy(rOnDisBluetoothMsg.cBluetoothID, szDeviceId.c_str(), MAX_STR_LEN);
   rOnDisBluetoothMsg.enSessionTransport= static_cast<tenDiPOSessionTransport>(m_enSessionTransport);
   bStatus = bSendMessage<trOnDisBluetoothMsg>(rOnDisBluetoothMsg);
   ETG_TRACE_USR2(("[DESC]: Disable bluetooth command sent to SPI with status = %d", ETG_ENUM( STATUS, bStatus)));
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPOControlAdapterImpl::OnGetCurrentResourceMode()
 ***************************************************************************/
/* ADIT Interface Implementation.*/
t_Void spi_tclDiPOControlAdapterImpl::OnGetCurrentResourceMode(trModeChange& rfoModeChange)
{
   ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::OnGetCurrentResourceMode entered"));
   rfoModeChange = m_AccessoryMode;
   //The Checking below is required to avoid sending empty valuse to iPhone
   // while the current values are not yet available. Need for initial handshaking.
   if (TransferType_Borrow != rfoModeChange.screen.type)
   {
      t_Char* pczBorrowId = new (std::nothrow) t_Char[MAX_STR_LEN];
      if (NULL != pczBorrowId)
      {
         memset(pczBorrowId, 0, MAX_STR_LEN);
         rfoModeChange.screen.borrowID = pczBorrowId;
      }
      if (true == m_bDisplayLastMode)
      {
         ETG_TRACE_USR2(("spi_tclDiPOControlAdapterImpl::OnGetCurrentResourceMode for CarPlay last mode display entered "));
         rfoModeChange.screen.type = TransferType_Take;
         rfoModeChange.screen.priority = TransferPriority_NiceToHave;
         rfoModeChange.screen.takeConstraint = Constraint_Anytime;
         rfoModeChange.screen.borrowOrUnborrowConstraint = Constraint_Anytime;
      }
      else
      {
         ETG_TRACE_USR2(("spi_tclDiPOControlAdapterImpl::OnGetCurrentResourceMode for native last mode display entered "));
         rfoModeChange.screen.type = TransferType_Take;
         rfoModeChange.screen.priority = TransferPriority_UserInitiated;
         rfoModeChange.screen.takeConstraint = Constraint_UserInitiated;
         rfoModeChange.screen.borrowOrUnborrowConstraint = Constraint_Anytime;
      }
   }

   if (TransferType_Borrow != rfoModeChange.audio.type)
   {
      ETG_TRACE_USR2(("spi_tclDiPOControlAdapterImpl::OnGetCurrentResourceMode for default values entered"));
      t_Char* pczBorrowId = new (std::nothrow) t_Char[MAX_STR_LEN];
      if (NULL != pczBorrowId)
      {
         memset(pczBorrowId, 0, MAX_STR_LEN);
         rfoModeChange.audio.borrowID = pczBorrowId;
      }
      if(true == m_bAudioLastMode)
      {
         ETG_TRACE_USR2(("spi_tclDiPOControlAdapterImpl::OnGetCurrentResourceMode for CarPlay last mode audio entered "));
         rfoModeChange.audio.type = TransferType_Take;
         rfoModeChange.audio.priority = TransferPriority_NiceToHave;
         rfoModeChange.audio.takeConstraint = Constraint_Anytime;
         rfoModeChange.audio.borrowOrUnborrowConstraint = Constraint_Anytime;
      }
      else
      {
         ETG_TRACE_USR2(("spi_tclDiPOControlAdapterImpl::OnGetCurrentResourceMode for native last mode audio entered "));
         rfoModeChange.audio.type = TransferType_Take;
         rfoModeChange.audio.priority = TransferPriority_UserInitiated;
         rfoModeChange.audio.takeConstraint = Constraint_UserInitiated;
         rfoModeChange.audio.borrowOrUnborrowConstraint = Constraint_Anytime;
      }
   }

   if(SpeechMode_NA  == rfoModeChange.speech)
   {
      rfoModeChange.speech = SpeechMode_None;
   }
   if(AppState_NA == rfoModeChange.phone)
   {
      rfoModeChange.phone = AppState_False;
   }
   if(AppState_NA == rfoModeChange.navigation)
   {
      rfoModeChange.navigation = AppState_False;
   }
}


/***************************************************************************
 ** FUNCTION: t_Void spi_tclDiPOControlAdapterImpl::vSetVideoConfiguration()
 ***************************************************************************/
t_Void spi_tclDiPOControlAdapterImpl::vSetVideoConfiguration()
{
   /*lint -esym(746,to_string) function to_string not made in the presence of a prototype */

   ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::vSetVideoConfiguration entered"));
   if(NULL != m_poIDynamicConfig)
   {

	  rAdaperSyncLock.s16Lock();

      t_Char szVideoPipeline[MAX_BUF_SIZE * 2] = {0};

      m_poIDynamicConfig->SetItem("display-width", to_string(m_rDiPODisplayAttr.u32Width));

      m_poIDynamicConfig->SetItem("display-height", to_string(m_rDiPODisplayAttr.u32Height));

      m_poIDynamicConfig->SetItem("wl-touch-layer-id", to_string(m_rDiPOTouchInputAttr.u32TouchLayerId));

      m_poIDynamicConfig->SetItem("wl-touch-surface-id", to_string(m_rDiPOTouchInputAttr.u32TouchSurfaceId));

      t_String szDriveSideInfo = (e8RIGHT_HAND_DRIVE == m_rDiPOInfoRespParam.enDriveSideInfo) ? "1" : "0";
      m_poIDynamicConfig->SetItem("core-rightHandDrive", szDriveSideInfo);

      m_poIDynamicConfig->SetItem("display-width-millimeter", to_string(m_rDiPODisplayAttr.u32Width_Millimeter));
      m_poIDynamicConfig->SetItem("display-height-millimeter", to_string(m_rDiPODisplayAttr.u32Height_Millimeter));

      // set up the video pipeline
#ifdef VARIANT_S_FTR_ENABLE_G4G
      snprintf(szVideoPipeline,MAX_BUF_SIZE,
               "h264parse ! omxh264dec ! vspfilter input-color-range=full ! video/x-raw, format=BGRA ! waylandsink sync=false  qos=false surface-id=59");
#else
      if(m_rDiPODisplayAttr.szDisplayDev.empty())
      {
         sprintf(szVideoPipeline, "vpudec low-latency=true frame-plus=2 framedrop=false framerate-nu=60 dis-reorder=true ! "
            "gst_apx_sink display-width=%d display-height=%d "
            "layer-id=%d surface-id=%d sync=false qos=false max-lateness=3000000000",
            m_rDiPODisplayAttr.u32Width,
            m_rDiPODisplayAttr.u32Height,
            m_rDiPODisplayAttr.u32LayerId,
            m_rDiPODisplayAttr.u32SurfaceId);
      }
      else
      {
       //H264 video strem pipeline
         sprintf(szVideoPipeline, "vpudec low-latency=true frame-plus=2 framedrop=false framerate-nu=60 dis-reorder=true ! "
            "v4l_csc devicename=%s csc-matrix-type=5 ! video/x-raw-rgb,bpp=32,depth=24 ! "
            "gst_apx_sink display-width=%d display-height=%d "
            "layer-id=%d surface-id=%d sync=false qos=false max-lateness=3000000000",
            m_rDiPODisplayAttr.szDisplayDev.c_str(),
            m_rDiPODisplayAttr.u32Width,
            m_rDiPODisplayAttr.u32Height,
            m_rDiPODisplayAttr.u32LayerId,
            m_rDiPODisplayAttr.u32SurfaceId);

      }
#endif

      ETG_TRACE_USR2(("[DESC]: CarPlay video pipeline configured = %s", szVideoPipeline));

      // set the video pipeline to the config file
      m_poIDynamicConfig->SetItem("gstreamer-video-pipeline", szVideoPipeline);
      m_poIDynamicConfig->SetItem("core-oemIconPath", m_rDiPOInfoRespParam.szOemIconPath);
      m_poIDynamicConfig->SetItem("core-oemIconLabel", m_rDiPOInfoRespParam.szOemIcon);
      
      if(!m_rDiPOInfoRespParam.szOemIconsPath.empty())
      {
          ETG_TRACE_USR2(("spi_tclDiPOControlAdapterImpl::vSetVideoConfiguration vSetOemIcons - enabled"));
          vSetOemIcons();
      }
      else
      {
          ETG_TRACE_USR2(("spi_tclDiPOControlAdapterImplvSetVideoConfiguration vSetOemIcons - not enabled"));
      }
      // Core model is same as oem icon name.
      m_poIDynamicConfig->SetItem("core-model",m_rDiPOInfoRespParam.szModelName);
      m_poIDynamicConfig->SetItem("core-manufacturer",m_rDiPOInfoRespParam.szManufacturer);
	  rAdaperSyncLock.vUnlock();
   }//End of (NULL != m_poIDynamicConfig))
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclDiPOControlAdapterImpl::vSetVideoConfiguration()
 ***************************************************************************/
t_Void spi_tclDiPOControlAdapterImpl::vSetOemIcons()
{
      ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::vSetOemIcons "));
      t_String szOemIcon0, szOemIcon1, szOemIcon2;
      //The parameters would be like:
      //oemIcons-0    path=/var/opt/bosch/static/spi/CarPlay/icons/Nissan/Renault_120x120.png full-bleed=false
      //oemIcons-1    path=/var/opt/bosch/static/spi/CarPlay/icons/Nissan/Renault_180x180.png full-bleed=false
      //oemIcons-2    path=/var/opt/bosch/static/spi/CarPlay/icons/Nissan/Renault_256x256.png full-bleed=false
      //
      //The resolution name appended with the respective resolution
      const t_String sOemIconRes120 = "_120x120.png full-bleed=true";
      const t_String sOemIconRes180 = "_180x180.png full-bleed=true";
      const t_String sOemIconRes256 = "_256x256.png full-bleed=true";
      const t_String sOemIconPreArg = "path=" + m_rDiPOInfoRespParam.szOemIconsPath;

      //Icon name string created by prepending token "path=" + "<previously created token string>"
      szOemIcon0 = sOemIconPreArg + sOemIconRes120;
      szOemIcon1 = sOemIconPreArg + sOemIconRes180;
      szOemIcon2 = sOemIconPreArg + sOemIconRes256;

      ETG_TRACE_USR2(("spi_tclDiPOControlAdapterImpl::vSetOemIcons - szOemIcon0(%s)", szOemIcon0.c_str()));
      ETG_TRACE_USR2(("spi_tclDiPOControlAdapterImpl::vSetOemIcons - szOemIcon1(%s)", szOemIcon1.c_str()));
      ETG_TRACE_USR2(("spi_tclDiPOControlAdapterImpl::vSetOemIcons - szOemIcon2(%s)", szOemIcon2.c_str()));

      //The above created tokens are passed to the ADIT interface for dynamically setting the oemicons
      m_poIDynamicConfig->SetItem("oemIcons-0", szOemIcon0);
      m_poIDynamicConfig->SetItem("oemIcons-1", szOemIcon1);
      m_poIDynamicConfig->SetItem("oemIcons-2", szOemIcon2);
}


/***************************************************************************
 ** FUNCTION: t_Void spi_tclDiPOControlAdapterImpl::vConfigLimitedUIElements()
 ***************************************************************************/
t_Void spi_tclDiPOControlAdapterImpl::vConfigLimitedUIElements()
{
   ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::vConfigLimitedUIElements entered"));
   t_U8 u8BitMask = 1;
   t_U8 u8itemCount = 0;

   for (t_U8 u8Iterator = 0; (u8Iterator < 8) && (u8BitMask != 0); u8Iterator++)
   {
      t_U8 u8TempValue = m_rDiPOInfoRespParam.u8DriveRestrictionInfo & u8BitMask;
      t_Char szKeyValueName[100];
      sprintf(szKeyValueName, "limited-ui-elements-%d", u8itemCount);
      switch (u8TempValue)
      {
         case e8SOFT_KEYBOARD:
         {
            u8itemCount++;
            if (NULL != m_poIDynamicConfig)
            {
               rAdaperSyncLock.s16Lock();
               m_poIDynamicConfig->SetItem(szKeyValueName, "softKeyboard");
               rAdaperSyncLock.vUnlock();
            }//if(NULL != m_poIDynamicConfig)
         }
            break;
         case e8SOFT_PHONEKEYPAD:
         {
            u8itemCount++;
            if (NULL != m_poIDynamicConfig)
            {
               rAdaperSyncLock.s16Lock();
               m_poIDynamicConfig->SetItem(szKeyValueName, "softPhoneKeypad");
               rAdaperSyncLock.vUnlock();
            }//if(NULL != m_poIDynamicConfig)
         }
            break;
         case e8NON_MUSIC_LIST:
         {
            u8itemCount++;
            if (NULL != m_poIDynamicConfig)
            {
               rAdaperSyncLock.s16Lock();
               m_poIDynamicConfig->SetItem(szKeyValueName, "nonMusicLists");
               rAdaperSyncLock.vUnlock();
            }//if(NULL != m_poIDynamicConfig)

         }
            break;
         case e8MUSIC_LIST:
         {
            u8itemCount++;
            if (NULL != m_poIDynamicConfig)
            {
               rAdaperSyncLock.s16Lock();
               m_poIDynamicConfig->SetItem(szKeyValueName, "musicLists");
               rAdaperSyncLock.vUnlock();
            }//if(NULL != m_poIDynamicConfig)
         }
            break;
         case e8JAPAN_MAPS:
         {
            u8itemCount++;
            if (NULL != m_poIDynamicConfig)
            {
               rAdaperSyncLock.s16Lock();
               m_poIDynamicConfig->SetItem(szKeyValueName, "japanMaps");
               rAdaperSyncLock.vUnlock();
            }//if(NULL != m_poIDynamicConfig)
         }
            break;
         default:
         {
            //do nothing
         }
            break;
      } //switch(u8TemoValue)
      u8BitMask = static_cast<t_U8>(u8BitMask << 1);
   }//for(t_U8 u8Iterator = 0; u8Iterator <8; u8Iterator++)

   m_u8LimitedUIElementCount = u8itemCount;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiPOControlAdapterImpl::bSetAudioConfiguration()
 ***************************************************************************/
t_Bool spi_tclDiPOControlAdapterImpl::bSetAudioOutConfig(const t_String szAudioDevice,
		const tenAudioStreamType enStreamType,
		const tenMsgTypes enMsgTypes)
{
   ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::bSetAudioOutConfig entered"));
   /*ETG_TRACE_USR4(("[PARAM]:bSetAudioOutConfig received with streamType = %d, deviceName = %s, messageType = %d",
		   ETG_ENUM(DIPO_AUDIO_STREAMTYPE , enStreamType), szAudioDevice.c_str(), enMsgTypes));*/
   t_Bool bRetVal = false;
   t_Char szAudPipeline[MAX_BUF_SIZE] = {0};
   SPI_INTENTIONALLY_UNUSED(szAudPipeline);
    rAdaperSyncLock.s16Lock();
   if((NULL != m_poIDynamicConfig) && (false == szAudioDevice.empty()))
   {
      if((e8ALTERNATE_AUDIO == enStreamType) && (e8AUDIO_MESSAGE == enMsgTypes))
      {
         if(e8AUDIO_MESSAGE != m_CurrAudioMessage)
         {
            t_String szDummyAudioPipe = m_poIDynamicConfig->GetItem("alsa-alternate-audio-dummy","");
            t_String szActualAudioPipe = m_poIDynamicConfig->GetItem("alsa-alternate-audio-0","");
            m_poIDynamicConfig->SetItem("alsa-alternate-audio-0", szDummyAudioPipe);
            m_poIDynamicConfig->SetItem("alsa-alternate-audio-dummy", szActualAudioPipe);
           
         }//if(e8AUDIO_MESSAGE!= m_CurrAudioMessage)
         m_CurrAudioMessage = e8AUDIO_MESSAGE;  
      } //  if(enStreamType == e8MAIN_AUDIO)
      else if((e8ALTERNATE_AUDIO == enStreamType) && (e8AUDIO_ERROR_MESSAGE == enMsgTypes))
      {
         if( e8AUDIO_ERROR_MESSAGE!= m_CurrAudioMessage)
         {
            t_String szDummyAudioPipe = m_poIDynamicConfig->GetItem("alsa-alternate-audio-dummy","");
            t_String szActualAudioPipe = m_poIDynamicConfig->GetItem("alsa-alternate-audio-0","");
            m_poIDynamicConfig->SetItem("alsa-alternate-audio-0", szDummyAudioPipe);
            m_poIDynamicConfig->SetItem("alsa-alternate-audio-dummy", szActualAudioPipe);
         }
         m_CurrAudioMessage = e8AUDIO_ERROR_MESSAGE;
      }//else if((e8ALTERNATE_AUDIO == enStreamType) && (e8AUDIO_ERROR_MESSAGE == enMsgTypes))
	   bRetVal = true;
   }
   rAdaperSyncLock.vUnlock();
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiPOControlAdapterImpl::bSetAudioInConfig()
 ***************************************************************************/
 //This function is not used currently. As per the new audio design, all the configurations are statically
 //set in the config file. The function kept for future use.
t_Bool spi_tclDiPOControlAdapterImpl::bSetAudioInConfig(const t_String szAudioOutDev, const t_String szAudioInDev)
{
   ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::bSetAudioInConfig entered"));
   SPI_INTENTIONALLY_UNUSED(szAudioOutDev);
   SPI_INTENTIONALLY_UNUSED(szAudioInDev);
   return true;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiPOControlAdapterImpl::bLaunchApp(t_String szAppUrl)
 ***************************************************************************/
t_Bool spi_tclDiPOControlAdapterImpl::bLaunchApp(const t_String szAppUrl)
{
   ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::bLaunchApp entered"));
   ETG_TRACE_USR4(("[PARAM]:: bLaunchApp - url = %s\n", szAppUrl.c_str()));
   t_Bool bRetVal = false;
   
   rAdaperSyncLock.s16Lock();
   if(NULL != m_poIControlReceiver)
   {
      // Invoke the IControlReciever function to request device UI.
      m_poIControlReceiver->RequestUI(szAppUrl);
      bRetVal = true;
   }
   rAdaperSyncLock.vUnlock();
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiPOControlAdapterImpl::bAccessoryModeChange(..)
 ***************************************************************************/
t_Bool spi_tclDiPOControlAdapterImpl::bAccessoryModeChange(const trModeChange& rfcorModeChange,
		const tenAccModeChangeType enAccModeChangeType)
{
   ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::bAccessoryModeChange entered"));

   ETG_TRACE_USR4(("[PARAM]:: bAccessoryModeChange - Display transfer type  = %d", ETG_ENUM( DIPO_TRANSFERTYPE, rfcorModeChange.screen.type)));
   ETG_TRACE_USR4(("[PARAM]:: bAccessoryModeChange - Display transfer priority  = %d", ETG_ENUM( DIPO_TRANSFERPRIO, rfcorModeChange.screen.priority)));
   ETG_TRACE_USR4(("[PARAM]:: bAccessoryModeChange - Display take constraint  = %d", ETG_ENUM( DIPO_CONSTRAINT, rfcorModeChange.screen.takeConstraint)));
   ETG_TRACE_USR4(("[PARAM]:: bAccessoryModeChange - Display borrow constraint  = %d", ETG_ENUM( DIPO_CONSTRAINT, rfcorModeChange.screen.borrowOrUnborrowConstraint)));
   ETG_TRACE_USR4(("[PARAM]:: bAccessoryModeChange - Display borrow Id  = %s", rfcorModeChange.screen.borrowID));

   ETG_TRACE_USR4(("[PARAM]:: bAccessoryModeChange - Audio transfer type  = %d", ETG_ENUM( DIPO_TRANSFERTYPE, rfcorModeChange.audio.type)));
   ETG_TRACE_USR4(("[PARAM]:: bAccessoryModeChange - Audio transfer priority  = %d", ETG_ENUM( DIPO_TRANSFERPRIO, rfcorModeChange.audio.priority)));
   ETG_TRACE_USR4(("[PARAM]:: bAccessoryModeChange - Audio take constraint = %d", ETG_ENUM( DIPO_CONSTRAINT, rfcorModeChange.audio.takeConstraint)));
   ETG_TRACE_USR4(("[PARAM]:: bAccessoryModeChange - Audio borrow constraint  = %d", ETG_ENUM( DIPO_CONSTRAINT, rfcorModeChange.audio.borrowOrUnborrowConstraint)));
   ETG_TRACE_USR4(("[PARAM]:: bAccessoryModeChange - Audio borrow Id  = %s", rfcorModeChange.audio.borrowID));

   ETG_TRACE_USR4(("[PARAM]:: bAccessoryModeChange - Speech state  = %d", ETG_ENUM(SPEECH_APP_STATE, rfcorModeChange.speech)));
   ETG_TRACE_USR4(("[PARAM]:: bAccessoryModeChange - Phone state  = %d", ETG_ENUM(PHONE_APP_STATE, rfcorModeChange.phone)));
   ETG_TRACE_USR4(("[PARAM]:: bAccessoryModeChange - Navigation state = %d", ETG_ENUM(NAV_APP_STATE, rfcorModeChange.navigation)));

   t_Bool bUpdateRequired = false;
   rAdaperSyncLock.s16Lock();

   if(NULL != m_poIControlReceiver)
   {
      bUpdateRequired = true;

      // Apple expectation is that we should not send unborrow for specific use case:
      // When the accessory has borrowed the resource and phone took the resource (Eg: for phone call/Siri) before sending unborrow.
      // For details check: SUZUKI-30463 and PSARCC30-2930.
      // However, unborrow request should be sent in below case:
      // Accessory has borrowed the resource and phone has not taken the resource in between.
      if (((TransferType_Unborrow == rfcorModeChange.screen.type) && (e8DIPO_ENTITY_MOBILE == m_rCurrentModeState.enScreen))
            || ((TransferType_Unborrow == rfcorModeChange.audio.type) && (e8DIPO_ENTITY_MOBILE == m_rCurrentModeState.enAudio)))
      {
          if(0 != u32PrevAudioBorrowCnt)
          {
              //If previously borrow was send successfully to iPhone (still resource with iPhone),
              //following unborrow should still be send to iPhone
              //Reference: NCG3D-163383, NCG3D-163384, NCG3D-161021
              bUpdateRequired = true;
              --u32PrevAudioBorrowCnt;
              ETG_TRACE_USR4(("[PARAM]:: bAccessoryModeChange - Unborrow audio request is sent to phone as audio resource was with controller borrowed earlier"));
          }
          else if(0 != u32PrevScreenBorrowCnt)
          {
              //Reference: NCG3D-163383, NCG3D-163384, NCG3D-161021
              bUpdateRequired = true;
              --u32PrevScreenBorrowCnt;
              ETG_TRACE_USR4(("[PARAM]:: bAccessoryModeChange - Unborrow screen request is sent to phone as screen resource was with controller borrowed earlier"));
          }
          else
          {
              bUpdateRequired = false;
              ETG_TRACE_USR4(("[PARAM]:: bAccessoryModeChange - Unborrow request is not sent to phone as the resource is no more with the head unit"));
          }
      } // if (((TransferType_Unborrow == rfcorModeChange.screen.type)

      //if valid audio context is send from HMI
      //Reference: NCG3D-163383, NCG3D-163384, NCG3D-161021
      if(e8DIPO_TRANSFERTYPE_NA != (tenDiPOTransferType)rfcorModeChange.audio.type)
      {
          //check if HMI has send a single borrow request or multiple borrow request consecutively. Ignore other transfer types
          if(e8DIPO_TRANSFERTYPE_BORROW == static_cast<tenDiPOTransferType>(rfcorModeChange.audio.type))
          {
              ++u32PrevAudioBorrowCnt;
              ETG_TRACE_USR4(("[PARAM]:: bAccessoryModeChange - borrow saved AudioTransferType count = %d", u32PrevAudioBorrowCnt));
          }
          else
          {
              //The variable 'u32PrevAudioBorrowCnt' is used to track the number of consecutive borrow,
              //to send matching unborrow in case respective resource is not released by iPhone.
              //When HMI send setAccessoryAudioContext(), wherein consecutive borrow is not send or borrow was converted to take,
              //the borrow track counter should be reset
              u32PrevAudioBorrowCnt = 0;
              ETG_TRACE_USR4(("[PARAM]:: bAccessoryModeChange - reset audio borrow count"));
          }
      }

      //if valid screen context is send from HMI
      //Reference: NCG3D-163383, NCG3D-163384, NCG3D-161021
      if(e8DIPO_TRANSFERTYPE_NA != (tenDiPOTransferType)rfcorModeChange.screen.type)
      {
          //check if HMI has send a single borrow request or multiple borrow request. Ignore other transfer types
          if(e8DIPO_TRANSFERTYPE_BORROW == static_cast<tenDiPOTransferType>(rfcorModeChange.screen.type))
          {
              ++u32PrevScreenBorrowCnt;
              ETG_TRACE_USR4(("[PARAM]:: bAccessoryModeChange - borrow saved ScreenTransferType count = %d", u32PrevScreenBorrowCnt));
          }
          else
          {
              //The variable 'u32PrevAudioBorrowCnt' is used to track the number of consecutive borrow,
              //to send matching unborrow in case respective resource is not released by iPhone.
              //When HMI send setAccessoryDisplayContext(), wherein consecutive borrow is not send or borrow was converted to take,
              //the borrow track counter should be reset
              u32PrevScreenBorrowCnt = 0;
              ETG_TRACE_USR4(("[PARAM]:: bAccessoryModeChange - reset screen borrow count"));
          }
      }

      if (true == bUpdateRequired)
      {
         m_poIControlReceiver->ChangeResourceMode(rfcorModeChange);
         vSetCurrentAccessoryMode(rfcorModeChange, enAccModeChangeType);
      } // if(true == bUpdateRequired)
   }
   rAdaperSyncLock.vUnlock();

   ETG_TRACE_USR4(("[PARAM]:: bAccessoryModeChange - request sent to phone: %d", ETG_ENUM(BOOL, bUpdateRequired)));
   return bUpdateRequired;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiPOControlAdapterImpl::bSiriAction(..
 ***************************************************************************/
t_Bool spi_tclDiPOControlAdapterImpl::bSiriAction(const SiriAction enSiriAction)
{
   ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::bSiriAction entered"));
   t_Bool bRetVal = false;

   ETG_TRACE_USR4(("[PARAM]:: bSiriAction - Siri Action  = %d", ETG_ENUM( CARPLAY_SIRI_EVENT, enSiriAction)));
   rAdaperSyncLock.s16Lock();
   if(NULL != m_poIControlReceiver)
   {
      // Invoke the IControlReciever function to initiate Siri Action.
      m_poIControlReceiver->RequestSiriAction(enSiriAction);
      bRetVal = true;
   }
   rAdaperSyncLock.vUnlock();
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclDiPOControlAdapterImpl::bSetNightMode(..
 ***************************************************************************/
t_Void spi_tclDiPOControlAdapterImpl::vSetNightMode(const t_Bool bIsNightMode)
{
   ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::vSetNightMode entered"));
   ETG_TRACE_USR4(("[PARAM]:: vSetNightMode - NightMode  = %d", ETG_ENUM( BOOL, bIsNightMode)));
   rAdaperSyncLock.s16Lock();
   if(NULL != m_poIControlReceiver)
   {
      m_poIControlReceiver->SetNightMode(bIsNightMode);
   }
   rAdaperSyncLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclDiPOControlAdapterImpl::vSetLimitedUI(..
 ***************************************************************************/
t_Void spi_tclDiPOControlAdapterImpl::vSetLimitedUI(const t_Bool bLimitedUIStatus)
{
   ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::vSetLimitedUI entered"));
   ETG_TRACE_USR4(("[PARAM]:: vSetLimitedUI - Limited UI = %d", ETG_ENUM( BOOL, bLimitedUIStatus)));
   rAdaperSyncLock.s16Lock();
   if((NULL != m_poIControlReceiver) && (0 != m_u8LimitedUIElementCount))
   {
      m_poIControlReceiver->SetLimitedUI(bLimitedUIStatus);
   }
   rAdaperSyncLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclDiPOControlAdapterImpl::vSetBluetoothIds(..
 ***************************************************************************/
t_Void spi_tclDiPOControlAdapterImpl::vSetBluetoothIds()
{
   ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::vSetBluetoothIds entered"));
   ETG_TRACE_USR4(("[PARAM]:: vSetBluetoothIds - Bluetooth Id = %s", m_szBlutoothMac.c_str()));
   
   rAdaperSyncLock.s16Lock();
   if(NULL != m_poIControlReceiver)
   {
      std::list<t_String> deviceIDs;
      deviceIDs.push_back(m_szBlutoothMac);
      m_poIControlReceiver->SetBluetoothIDs(deviceIDs);
   }
   rAdaperSyncLock.vUnlock();
}

//!static
/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiPOControlAdapterImpl::vSetCurrentAccessoryMode(..
 ***************************************************************************/
t_Void spi_tclDiPOControlAdapterImpl::vSetCurrentAccessoryMode(const trModeChange &corModeChange, 
      const tenAccModeChangeType enAccModeChangeType)
{
   ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::vSetCurrentAccessoryMode entered"));
   ETG_TRACE_USR4(("[PARAM]:: vSetCurrentAccessoryMode - Mode Change Type Id = %d", ETG_ENUM(MODE_UPDATE, enAccModeChangeType)));
   switch(enAccModeChangeType)
   {
   case e8MODE_CHANGE_DISPLAY:
      {
    	 m_AccessoryMode.screen = corModeChange.screen;
      }
      break;
   case e8MODE_CHANGE_AUDIO:
      {
    	 m_AccessoryMode.audio = corModeChange.audio;
      }
      break;
   case e8MODE_CHANGE_APP:
      {
         m_AccessoryMode.speech = corModeChange.speech;
         m_AccessoryMode.phone = corModeChange.phone;
         m_AccessoryMode.navigation = corModeChange.navigation;
      }
      break;
   default:
      {
      }
      break;
   }
}

//!static
/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiPOControlAdapterImpl::vSetBluetoothMacAddr(..
 ***************************************************************************/
t_Void spi_tclDiPOControlAdapterImpl::vSetBluetoothMacAddr(const t_String szMacAddress)
{
   ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::vSetBluetoothMacAddr entered"));
   m_szBlutoothMac = szMacAddress;
}

//!static
/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiPOControlAdapterImpl::vSetOemIconData(..
 ***************************************************************************/
t_Void spi_tclDiPOControlAdapterImpl::vSetVehicleBrandInfo(const trVehicleBrandInfo &rfoVehicleBrandInfo)
{
   ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::vSetVehicleBrandInfo entered"));
   m_rDiPOInfoRespParam.szOemIconPath = rfoVehicleBrandInfo.szOemIconPath;
   ETG_TRACE_USR4(("[PARAM]::vSetVehicleBrandInfo - Icon Path = %s", m_rDiPOInfoRespParam.szOemIconPath.c_str()));
   m_rDiPOInfoRespParam.szOemIconsPath = rfoVehicleBrandInfo.szOemIconsPath;
   ETG_TRACE_USR4(("[PARAM]::vSetVehicleBrandInfo - Icons Path = %s", m_rDiPOInfoRespParam.szOemIconsPath.c_str()));
   m_rDiPOInfoRespParam.szOemIcon = rfoVehicleBrandInfo.szOemName;
   ETG_TRACE_USR4(("[PARAM]::vSetVehicleBrandInfo - Icon Label = %s", m_rDiPOInfoRespParam.szOemIcon.c_str()));
   m_rDiPOInfoRespParam.szModelName = rfoVehicleBrandInfo.szModel;
   ETG_TRACE_USR4(("[PARAM]::vSetVehicleBrandInfo - Model Name = %s", m_rDiPOInfoRespParam.szModelName.c_str()));
   m_rDiPOInfoRespParam.szManufacturer = rfoVehicleBrandInfo.szManufacturer;
   ETG_TRACE_USR4(("[PARAM]::vSetVehicleBrandInfo - Manufacturer = %s", m_rDiPOInfoRespParam.szManufacturer.c_str()));
}

//!static
/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiPOControlAdapterImpl::vSetDriveSideInfo(..
 ***************************************************************************/
t_Void spi_tclDiPOControlAdapterImpl::vSetDriveSideInfo(const tenDriveSideInfo enDriveSideInfo)
{
	ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::vSetDriveSideInfo entered"));
	m_rDiPOInfoRespParam.enDriveSideInfo = enDriveSideInfo;
	ETG_TRACE_USR4(("[PARAM]::vSetDriveSideInfo - Drive Side = %d", ETG_ENUM(DRIVE_SIDE_TYPE, enDriveSideInfo)));
}

/***************************************************************************
** FUNCTION:  virtual t_Void spi_tclDiPOControlAdapterImpl::vSetVideoConfigInfo()
***************************************************************************/
t_Void spi_tclDiPOControlAdapterImpl::vSetVideoConfigInfo(trVideoConfigData rVideoConfigData)
{
   ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::vSetVideoConfigInfo: Screen Width:%d,Screen Height:%d,LayerID:%d,SurfaceID:%d,ScreenWidthMM:%d,ScreenHeightMM:%d,TouchLayerID:%d,TouchSurfaceID:%d, szDisplayDev:%s",
   rVideoConfigData.u32ProjScreen_Height,rVideoConfigData.u32ProjScreen_Width,rVideoConfigData.u32LayerId,rVideoConfigData.u32SurfaceId,
   rVideoConfigData.u32ProjScreen_Width_Mm,rVideoConfigData.u32ProjScreen_Height_Mm,rVideoConfigData.u32TouchLayerId,rVideoConfigData.u32TouchSurfaceId, rVideoConfigData.szDisplayDev));
   
   m_rDiPODisplayAttr.u32Height = rVideoConfigData.u32ProjScreen_Height;
   m_rDiPODisplayAttr.u32Width = rVideoConfigData.u32ProjScreen_Width;
   m_rDiPODisplayAttr.u32LayerId = rVideoConfigData.u32LayerId;
   m_rDiPODisplayAttr.u32SurfaceId = rVideoConfigData.u32SurfaceId;
   m_rDiPODisplayAttr.u32Width_Millimeter = rVideoConfigData.u32ProjScreen_Width_Mm;
   m_rDiPODisplayAttr.u32Height_Millimeter = rVideoConfigData.u32ProjScreen_Height_Mm;
   m_rDiPODisplayAttr.szDisplayDev = rVideoConfigData.szDisplayDev;
   m_rDiPOTouchInputAttr.u32TouchLayerId = rVideoConfigData.u32TouchLayerId;
   m_rDiPOTouchInputAttr.u32TouchSurfaceId = rVideoConfigData.u32TouchSurfaceId;

}

//!static
/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiPOControlAdapterImpl::bSessionStartTimerCb(..
 ***************************************************************************/
t_Bool spi_tclDiPOControlAdapterImpl::bSessionStartTimerCb(timer_t timerID , tVoid *pObject, 
                         const tVoid *pcoUserData)
{
   ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::bSessionStartTimerCb entered"));
   SPI_INTENTIONALLY_UNUSED(pObject);
   SPI_INTENTIONALLY_UNUSED(pcoUserData);
   SPI_INTENTIONALLY_UNUSED(timerID);

   //!Cancel the start up timer if it still running. 
   // Note  that the cancel  will destroy the timer from the context.
   Timer* poTimer = Timer::getInstance();
   if ((NULL != poTimer) && (0 != srTimerID ))
   {
      poTimer->CancelTimer(srTimerID );
      srTimerID  =0;
   }//if ((NULL != poTimer) && (0 != srTimerID))

   trDiPOSessionMsg oBuffer;
   oBuffer.enMsgType = e8SESSION_MESSAGE;
   oBuffer.enDiPOSessionState = e8DIPO_SESSION_START;
   if (NULL != m_poDiPOControlAdapter)
   {
       oBuffer.enSessionTransport = m_poDiPOControlAdapter->m_enSessionTransport;
       strncpy(oBuffer.szSessionIPAddress, m_poDiPOControlAdapter->m_szSessionIPAddress.c_str(), MAX_STR_LEN);
   }
   spi_tclDiPOAdapterMsgQInterface* poDiPOAdapterMsgQInterface = spi_tclDiPOAdapterMsgQInterface::getInstance();
   t_Bool bStatus = (poDiPOAdapterMsgQInterface) ? poDiPOAdapterMsgQInterface->bWriteMsgToQ(&oBuffer, sizeof(trDiPOSessionMsg)) : false;

   ETG_TRACE_USR2(("[DESC]: Session start message sent to SPI with status = %d", ETG_ENUM(STATUS, bStatus)));

   return true; // Lint fix.
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiPOControlAdapterImpl::vSetNightModeInfo(..
 ***************************************************************************/
t_Void spi_tclDiPOControlAdapterImpl::vSetNightModeInfo(const t_Bool bIsNightMode)
{
	ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::vSetNightModeInfo entered"));
	m_bCurNightModeInfo = bIsNightMode;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiPOControlAdapterImpl::vSetDriveModeInfo(..
 ***************************************************************************/
t_Void spi_tclDiPOControlAdapterImpl::vSetDriveModeInfo(const t_Bool bIsDriveMode)
{
	ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::vSetDriveModeInfo entered"));
	m_bCurDriveModeInfo = bIsDriveMode;
}

//static
/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiPOControlAdapterImpl::vSetDriveRestrictionInfo(..
 ***************************************************************************/
t_Void spi_tclDiPOControlAdapterImpl::vSetDriveRestrictionInfo(const t_U8 u8DriveRest)
{
	ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::vSetDriveRestrictionInfo entered"));
	m_rDiPOInfoRespParam.u8DriveRestrictionInfo = u8DriveRest;
	ETG_TRACE_USR4(("[PARAM]::vSetDriveRestrictionInfo - Driver Restriction Bitmap = %d", m_rDiPOInfoRespParam.u8DriveRestrictionInfo));
}

//static
/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiPOControlAdapterImpl::vInitializeAudioTypeMap
 ***************************************************************************/
std::map<t_String, tenDiPOMainAudioType> spi_tclDiPOControlAdapterImpl::vInitializeAudioTypeMap()
{
	ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::vInitializeAudioTypeMap entered"));
	std::map<t_String, tenDiPOMainAudioType> mTempAudioMap;
	mTempAudioMap["media"] = e8AUDIO_MEDIA;
	mTempAudioMap["speechRecognition"] = e8AUDIO_SPEECHREC;
	mTempAudioMap["spokenAudio"] = e8AUDIO_SPOKEN;
	mTempAudioMap["telephony"] = e8AUDIO_TELEPHONY;
	mTempAudioMap["alert"] = e8AUDIO_ALERT;
	mTempAudioMap["default"] = e8AUDIO_DEFAULT;
	return mTempAudioMap;
}

//static
/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiPOControlAdapterImpl::vStoreSoftwareVersion(..
 ***************************************************************************/
t_Void spi_tclDiPOControlAdapterImpl::vStoreSoftwareVersion(const t_String& corfszFirmwareVersion)
{
    ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::vStoreSoftwareVersion entered"));
    m_szFirmwareVersion = corfszFirmwareVersion;
    ETG_TRACE_USR2(("[DESC]:vStoreSoftwareVersion - Firmware version = %s", corfszFirmwareVersion.c_str()));
}

//static
/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiPOControlAdapterImpl::vStoreHardwareVersion(..
 ***************************************************************************/
t_Void spi_tclDiPOControlAdapterImpl::vStoreHardwareVersion(const t_String& corfszHardwareVersion)
{
    ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::vStoreHardwareVersion entered"));
    m_szHardwareVersion = corfszHardwareVersion;
    ETG_TRACE_USR2(("[DESC]:vStoreHardwareVersion - Hardware version = %s", corfszHardwareVersion.c_str()));
}

//static
/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiPOControlAdapterImpl::vStoreVehicleInfo(..
 ***************************************************************************/
t_Void spi_tclDiPOControlAdapterImpl::vStoreVehicleInfo(const trVehicleInformation& corfrVehicleInfo)
{
    ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::vStoreVehicleInfo entered"));
    m_rVehicleInfo = corfrVehicleInfo;
    ETG_TRACE_USR2(("[DESC]:vStoreVehicleInfo - ETC enabled = %d", ETG_ENUM(ETC_SUPPORT, corfrVehicleInfo.electronicTollSupport)));
}

/* Private methods*/

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPOControlAdapterImpl::vGetAudioType()
 ***************************************************************************/
t_Void spi_tclDiPOControlAdapterImpl::vGetAudioType(const t_String szAudioType, 
                                                    tenDiPOMainAudioType &enAudioType)
 {
	ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::vGetAudioType entered"));
    //Mapping the Apple specific audio types to the SPI internal types
	std::map<t_String, tenDiPOMainAudioType>::iterator it = m_mAudioTypeMap.find(szAudioType.c_str());
	if(it != m_mAudioTypeMap.end())
	{
		enAudioType = it->second;
	}
	ETG_TRACE_USR2(("[DESC]: vGetAudioType returned with audio type = %d", ETG_ENUM(DIPO_AUDIO_TYPE, enAudioType)));
 }

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPOControlAdapterImpl::vConvertModeState()
 ***************************************************************************/
t_Void spi_tclDiPOControlAdapterImpl::vConvertModeState(const trModeState& rfcorModestate,
                                                    trDiPOModeState &rfoDiPOModestate)
{
   ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::vConvertModeState entered"));
   rfoDiPOModestate.enScreen = (tenDiPOEntity)rfcorModestate.screen;
   rfoDiPOModestate.enAudio = (tenDiPOEntity)rfcorModestate.audio;
   rfoDiPOModestate.rSpeechState.enEntity = (tenDiPOEntity)rfcorModestate.speech.entity;
   rfoDiPOModestate.rSpeechState.enSpeechMode = (tenDiPOSpeechMode)rfcorModestate.speech.mode;
   rfoDiPOModestate.enPhone = (tenDiPOEntity)rfcorModestate.phone;
   rfoDiPOModestate.enNavigation = (tenDiPOEntity)rfcorModestate.navigation;
   rfoDiPOModestate.enPermScreen = (tenDiPOEntity)rfcorModestate.permScreen;
   rfoDiPOModestate.enPermMainAudio = (tenDiPOEntity)rfcorModestate.permMainAudio;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclDiPOControlAdapterImpl::bSendMessage()
 ***************************************************************************/
template<typename trMessage>
t_Bool spi_tclDiPOControlAdapterImpl::bSendMessage(trMessage rMessage)
{
   ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::bSendMessage entered"));
   t_Bool bRetVal = false;
   spi_tclDiPOAdapterMsgQInterface* poDiPOAdapterMsgQInterface = spi_tclDiPOAdapterMsgQInterface::getInstance();
   bRetVal = (poDiPOAdapterMsgQInterface) ? poDiPOAdapterMsgQInterface->bWriteMsgToQ(&rMessage, sizeof(trMessage)) : false;

   ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::bSendMessage send"));
   return bRetVal;
}

//!static
/***************************************************************************
 ** FUNCTION: t_Void spi_tclDiPOControlAdapterImpl:: vSetOEMIconVisibility(..)
 ***************************************************************************/
t_Void spi_tclDiPOControlAdapterImpl::vSetOEMIconVisibility(t_Bool bOEMiconVisibility)
{
   ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::bSetOEMIconVisibility entered"));
   m_rDiPOInfoRespParam.bOEMIconVisibility=bOEMiconVisibility;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclDiPOControlAdapterImpl::vSetOEMIconVisible()
 ***************************************************************************/
t_Void spi_tclDiPOControlAdapterImpl::vSetOEMIconVisible()
{
   /*lint -esym(746,to_string) function to_string not made in the presence of a prototype */

   ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::vSetOEMIconVisible entered"));
   if(NULL != m_poIDynamicConfig)
   {
      rAdaperSyncLock.s16Lock();
      m_poIDynamicConfig->SetItem("core-oemIconVisible", to_string(static_cast<unsigned>(m_rDiPOInfoRespParam.bOEMIconVisibility)));
	  rAdaperSyncLock.vUnlock();
   }
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiPOControlAdapterImpl::vSetSoftwareVersion(..
 ***************************************************************************/
t_Void spi_tclDiPOControlAdapterImpl::vSetSoftwareVersion()
{
    ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::vSetSoftwareVersion entered"));
	if (NULL != m_poIDynamicConfig)
    {
       rAdaperSyncLock.s16Lock();
       m_poIDynamicConfig->SetItem("core-firmwareRevision", m_szFirmwareVersion);
       rAdaperSyncLock.vUnlock();
    }
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiPOControlAdapterImpl::vSetHardwareVersion(..
 ***************************************************************************/
t_Void spi_tclDiPOControlAdapterImpl::vSetHardwareVersion()
{
    ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::vSetHardwareVersion entered"));
	if (NULL != m_poIDynamicConfig)
    {
       rAdaperSyncLock.s16Lock();
       m_poIDynamicConfig->SetItem("core-hardwareRevision", m_szHardwareVersion);
       rAdaperSyncLock.vUnlock();
    }
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiPOControlAdapterImpl::vUpdateVehicleInformation(..
 ***************************************************************************/
t_Void spi_tclDiPOControlAdapterImpl::vUpdateVehicleInformation()
{
    ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::vUpdateVehicleInformation entered with ETC Feature enabled = %d", ETG_ENUM(ETC_SUPPORT, m_rVehicleInfo.electronicTollSupport)));
    ETG_TRACE_USR2(("[PARAM]:spi_tclDiPOControlAdapterImpl::vUpdateVehicleInformation -ETC  Status : %d",ETG_ENUM(ETC_SUPPORT,m_rETCStatusInfo.electronicTollSupport)));
	if ((NULL != m_poIControlReceiver) && (ETCInfo_NotSupported != m_rVehicleInfo.electronicTollSupport))
    {
       rAdaperSyncLock.s16Lock();
       m_poIControlReceiver->UpdateVehicleInformation(m_rETCStatusInfo);
       rAdaperSyncLock.vUnlock();
    }
}

/***************************************************************************
** FUNCTION: t_Void spi_tclDiPOControlAdapterImpl::vGetSessionIPAddress()
***************************************************************************/
t_Void spi_tclDiPOControlAdapterImpl::vGetSessionIPAddress(t_String& rfSessionIPAddress)
{
   ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::vGetSessionIPAddress():Session Transport:%d",m_enSessionTransport));

   if(NULL != m_poIControlReceiver)
   {
      struct sockaddr const *sockaddr = m_poIControlReceiver->getIPAddress();

      if(NULL != sockaddr)
      {
      
         switch(sockaddr->sa_family)
         {
            case AF_INET:
            {
               ETG_TRACE_USR2(("spi_tclDiPOControlAdapterImpl::vGetSessionIPAddress():IPV4 IP Address"));
               char czIPAddr[INET_ADDRSTRLEN]={0};
               inet_ntop(AF_INET, &(((struct sockaddr_in *)sockaddr)->sin_addr),czIPAddr, INET_ADDRSTRLEN);
               rfSessionIPAddress = czIPAddr ;
            }
            break;
         
            case AF_INET6:
            {
               ETG_TRACE_USR2(("spi_tclDiPOControlAdapterImpl::vGetSessionIPAddress():IPV6 IP Address"));
               char czIPAddr[INET6_ADDRSTRLEN]={0};
               inet_ntop(AF_INET6, &(((struct sockaddr_in6 *)sockaddr)->sin6_addr), czIPAddr, INET6_ADDRSTRLEN);
               rfSessionIPAddress = czIPAddr ;
            }
            break;
         
            default:
            {
               ETG_TRACE_USR2(("spi_tclDiPOControlAdapterImpl::vGetSessionIPAddress():Unknown"));
            }
            break;
         }
      }
      
   }

   ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::vGetSessionIPAddress():Session IP:%s",rfSessionIPAddress.c_str()));

}

/***************************************************************************
** FUNCTION: t_Void spi_tclDiPOControlAdapterImpl::bETCStatus()
***************************************************************************/
t_Void spi_tclDiPOControlAdapterImpl::bSetETCStatus()
{
	//This method to greyedIN or greyedOUT ETC Icon
	ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::bETCStatus() entered"));
	vUpdateVehicleInformation();
	ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::bETCStatus() left"));
}

/***************************************************************************
** FUNCTION: t_Void spi_tclDiPOControlAdapterImpl::vStoreETCStatus()
***************************************************************************/
t_Void spi_tclDiPOControlAdapterImpl::vStoreETCStatus(const tenETCInfo enETCStatus)
{
	ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::vStoreETCStatus() entered"));
	ETG_TRACE_USR2(("[PARAM]::spi_tclDiPOControlAdapterImpl::vStoreETCStatus -ETC Status : %d",ETG_ENUM(ETC_SUPPORT,enETCStatus)));
	m_rETCStatusInfo.electronicTollSupport = static_cast<ETCInfo>(enETCStatus);
	ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::vStoreETCStatus() left"));
}

/***************************************************************************
** FUNCTION: t_Void spi_tclDiPOControlAdapterImpl::vStoreLastModeInfo()
***************************************************************************/
t_Void spi_tclDiPOControlAdapterImpl::vStoreLastModeInfo(t_Bool bAudioLastMode, t_Bool bDisplayLastMode)
{
   ETG_TRACE_USR2(("[DESC]:spi_tclDiPOControlAdapterImpl::vStoreLastModeInfo - Audio last mode: %d"
            " Display last mode: %d",ETG_ENUM(BOOL,bAudioLastMode), ETG_ENUM(BOOL,bDisplayLastMode)));
   m_bAudioLastMode = bAudioLastMode;
   m_bDisplayLastMode = bDisplayLastMode;
}

/***************************************************************************
** FUNCTION: t_Void spi_tclDiPOControlAdapterImpl::vSetExtendedFeatures()
***************************************************************************/
t_Void spi_tclDiPOControlAdapterImpl::vStoreExtendedFeatures(trExtendedFeaturesSupport rExtendedFeaturesSupport)
{
   ETG_TRACE_USR2(("[DESC]:spi_tclDiPOControlAdapterImpl::vStoreExtendedFeatures - Enhanced Request UI Support: %d",
       ETG_ENUM(BOOL, rExtendedFeaturesSupport.bEnhancedRequestUISupport)));
   ETG_TRACE_USR2(("[DESC]:spi_tclDiPOControlAdapterImpl::vStoreExtendedFeatures - Vocoder info Support: %d",
       ETG_ENUM(BOOL, rExtendedFeaturesSupport.bVocoderInfoSupport)));
   m_rExtendedFeaturesSupport.bEnhancedRequestUISupport = rExtendedFeaturesSupport.bEnhancedRequestUISupport;
   m_rExtendedFeaturesSupport.bVocoderInfoSupport = rExtendedFeaturesSupport.bVocoderInfoSupport;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiPOControlAdapterImpl::vSetExtendedFeatures(..
 ***************************************************************************/
t_Void spi_tclDiPOControlAdapterImpl::vSetExtendedFeatures()
{
    ETG_TRACE_USR1(("spi_tclDiPOControlAdapterImpl::vSetExtendedFeatures entered"));
    ETG_TRACE_USR4(("[PARAM]: spi_tclDiPOControlAdapterImpl::vSetExtendedFeatures - Enhanced Request UI Support: %d",
        ETG_ENUM(BOOL, m_rExtendedFeaturesSupport.bEnhancedRequestUISupport)));
    ETG_TRACE_USR4(("[PARAM]: spi_tclDiPOControlAdapterImpl::vSetExtendedFeatures - Vocoder info Support: %d",
        ETG_ENUM(BOOL, m_rExtendedFeaturesSupport.bVocoderInfoSupport)));
    if (NULL != m_poIDynamicConfig)
    {
       rAdaperSyncLock.s16Lock();
       if (true == m_rExtendedFeaturesSupport.bEnhancedRequestUISupport)
       {
          m_poIDynamicConfig->SetItem("extended-features-0", "enhancedRequestCarUI");

          if (true == m_rExtendedFeaturesSupport.bVocoderInfoSupport)
          {
             m_poIDynamicConfig->SetItem("extended-features-1", "vocoderInfo");
          }
       }
       else if (true == m_rExtendedFeaturesSupport.bVocoderInfoSupport)
       {
          m_poIDynamicConfig->SetItem("extended-features-0", "vocoderInfo");
       }
       rAdaperSyncLock.vUnlock();
    }
}
//lint -restore
