/*!
 *******************************************************************************
 * \file              spi_tclConfigurationManager.cpp
 * \brief             Configuration manager for SPI
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Configuration manager for SPI
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 07.06.2018 |  Kavya Mogeri                | Initial Version

 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/

#include "spi_tclConfigurationManager.h"

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_CONFIGREADER
#include "trcGenProj/Header/spi_tclConfigurationManager.cpp.trc.h"
#endif
#endif

static const t_U8 su8ETCEnabledMask = 0x01;

/***************************************************************************
 ** FUNCTION:  spi_tclConfigurationManager::spi_tclConfigurationManager()
 ***************************************************************************/
spi_tclConfigurationManager::spi_tclConfigurationManager(spi_tclStaticConfigurationBaseIntf* poStaticConfiguration,
         spi_tclDynamicConfigurationBase* poDynamicConfiguration) :
         m_poStaticConfigurationBase(poStaticConfiguration), m_poDynamicConfigurationBase(poDynamicConfiguration)
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::spi_tclConfigurationManager entered "));

}

/***************************************************************************
 ** FUNCTION:  spi_tclConfigurationManager::~spi_tclConfigurationManager()
 ***************************************************************************/
spi_tclConfigurationManager::~spi_tclConfigurationManager()
{
   ETG_TRACE_USR1(("~spi_tclConfigurationManager::spi_tclConfigurationManager entered "));
   m_poStaticConfigurationBase = NULL;
   m_poDynamicConfigurationBase = NULL;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclConfigurationManager::vInitialize()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bInitialize()
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::bInitialize entered "));
   return true;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConfigurationManager::vUninitialize()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vUninitialize()
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::vUninitialize entered "));
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclConfigurationManager::bNonCertAppsReq()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetNonCertAppsReq() const
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::bGetNonCertAppsReq"));
   t_Bool bRet = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bRet = m_poStaticConfigurationBase->bGetNonCertAppsReq();
   }
   return bRet;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConfigurationManager::vReadSPIPolicySettings()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vReadSPIPolicySettings()
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::vReadSPIPolicySettings"));
   if (NULL != m_poStaticConfigurationBase)
   {
      vGetDeviceTechnologyPreferenceFromDatapool();
      vGetSelectionModeStoredInDatapool();
      m_poStaticConfigurationBase->vGetSPIConfigurationData();
      m_poStaticConfigurationBase->vReadSPISettings();
   }
   vReadConfigurationData();
   trVehicleManufacturerInformation rVehicleManufacturerInfoAttr;
   t_Bool RetVal = false;
   if ((NULL != m_poStaticConfigurationBase) && (NULL != m_poDynamicConfigurationBase))
   {
      tenDriveSideInfo enDriveSideInfo = m_poStaticConfigurationBase->enGetDriveSideInfofromKDS();
      m_poDynamicConfigurationBase->vSetDriveSideInfoToDatapool(enDriveSideInfo);

      RetVal = m_poDynamicConfigurationBase->bReadVehicleManufacturerInfofromDatapool(rVehicleManufacturerInfoAttr);
      m_poStaticConfigurationBase->vSetVehicleManufacturerInformationtoKDS(RetVal, rVehicleManufacturerInfoAttr);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclConfigurationManager::bGetAAPNavigationEnabledInfo()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetAAPNavigationEnabledInfo() const
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::bGetAAPNavigationEnabledInfo"));
   t_Bool bAAPNavigationEnabledInfo = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bAAPNavigationEnabledInfo = m_poStaticConfigurationBase->bGetAAPNavigationEnabledInfo();
   }
   return bAAPNavigationEnabledInfo;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclConfigurationManager::bGetAAPMediaPlaybackEnabledInfo()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetAAPMediaPlaybackEnabledInfo() const
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::bGetAAPMediaPlaybackEnabledInfo"));
   t_Bool bAAPMediaPlaybackEnabledInfo = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bAAPMediaPlaybackEnabledInfo = m_poStaticConfigurationBase->bGetAAPMediaPlaybackEnabledInfo();
   }
   return bAAPMediaPlaybackEnabledInfo;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclConfigurationManager::bGetAAPNotificationUpdatesEnabledInfo()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetAAPNotificationUpdatesEnabledInfo() const
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::bGetAAPNotificationUpdatesEnabledInfo"));
   t_Bool bAAPNotificationUpdatesEnabledInfo = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bAAPNotificationUpdatesEnabledInfo = m_poStaticConfigurationBase->bGetAAPNotificationUpdatesEnabledInfo();
   }
   return bAAPNotificationUpdatesEnabledInfo;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclConfigurationManager::bGetTBTUpdatesEnabledInfo()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetTBTUpdatesEnabledInfo() const
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::bGetTBTUpdatesEnabledInfo"));
   t_Bool bTBTUpdatesEnabledInfo = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bTBTUpdatesEnabledInfo = m_poStaticConfigurationBase->bGetTBTUpdatesEnabledInfo();
   }
   return bTBTUpdatesEnabledInfo;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclConfigurationManager::bGetAssisstantGuideUpdatesEnabledInfo()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetAssisstantGuideUpdatesEnabledInfo() const
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::bGetAssisstantGuideUpdatesEnabledInfo"));
   t_Bool bAssisstantGuideUpdatesEnabledInfo = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bAssisstantGuideUpdatesEnabledInfo = m_poStaticConfigurationBase->bGetAssisstantGuideUpdatesEnabledInfo();
   }
   return bAssisstantGuideUpdatesEnabledInfo;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclConfigurationManager::vSetMLNotificationOnOff()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vSetMLNotificationOnOff(t_Bool bSetNotificationsOn)
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::vSetMLNotificationOnOff:bSetNotificationsOn-%d", ETG_ENUM(BOOL,
            bSetNotificationsOn)));
   if (NULL != m_poDynamicConfigurationBase)
   {
      m_poDynamicConfigurationBase->vSetMLNotificationOnOff(bSetNotificationsOn);
   }
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetMLNotificationEnabledInfo()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetMLNotificationEnabledInfo() const
{
   t_Bool bNotificationsOn = false;
   if (NULL != m_poDynamicConfigurationBase)
   {
      bNotificationsOn = m_poDynamicConfigurationBase->bGetMLNotificationEnabledInfo();
      ETG_TRACE_USR2(("spi_tclConfigurationManager::bGetMLNotificationEnabledInfo: bNotificationsOn-%d", ETG_ENUM(BOOL,
               bNotificationsOn)));
   }
   return bNotificationsOn;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclConfigurationManager::vSetMLNotificationOnOff()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vSetRegion(tenRegion enRegion)
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::vSetRegionentered with Region = %d ", ETG_ENUM(REGION_TYPE, enRegion)));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vSetRegion(enRegion);
   }
}

/***************************************************************************
 ** FUNCTION: t_String spi_tclConfigurationManager::szGetRegion()
 ***************************************************************************/
t_String spi_tclConfigurationManager::szGetRegion() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::szGetRegion entered"));
   t_String szRegion;
   if (NULL != m_poStaticConfigurationBase)
   {
      szRegion = m_poStaticConfigurationBase->szGetRegion();
   }
   return szRegion;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetNonCertAppAllowedInfo()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetNonCertAppAllowedInfo() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::bGetNonCertAppAllowedInfo entered"));
   t_Bool bNonCertAppAllowedInfo = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bNonCertAppAllowedInfo = m_poStaticConfigurationBase->bGetNonCertAppAllowedInfo();
   }
   return bNonCertAppAllowedInfo;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bEnableCTSTest()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bEnableCTSTest()
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::bEnableCTSTest entered"));
   t_Bool bCTSTest = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bCTSTest = m_poStaticConfigurationBase->bEnableCTSTest();
   }
   return bCTSTest;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclConfigurationManager::vGetRTPPayloadType()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vGetRTPPayloadType(t_U32& rfu32RTPPayload) const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::vGetRTPPayloadType entered"));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vGetRTPPayloadType(rfu32RTPPayload);
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclConfigurationManager::vGetRTPPayloadType()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vGetAudioIPL(t_U32& rfu32AudioIPL) const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::vGetAudioIPL entered"));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vGetAudioIPL(rfu32AudioIPL);
   }
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetAudBlockTriggerBasedOnCat()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetAudBlockTriggerBasedOnCat() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::bGetAudBlockTriggerBasedOnCat entered"));
   t_Bool bAudBlockTriggerBasedOnCat = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bAudBlockTriggerBasedOnCat = m_poStaticConfigurationBase->bGetAudBlockTriggerBasedOnCat();
   }
   return bAudBlockTriggerBasedOnCat;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::vSetAudBlockTriggerBasedOnCat()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vSetAudBlockTriggerBasedOnCat(t_Bool bIsML11OrAboveDevice)
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::vSetAudBlockTriggerBasedOnCat entered"));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vSetAudBlockTriggerBasedOnCat(bIsML11OrAboveDevice);
   }
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetAudBlockTriggerBasedOnGlobaleMute()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetAudBlockTriggerBasedOnGlobaleMute() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::bGetAudBlockTriggerBasedOnGlobaleMute entered"));
   t_Bool m_bAudBlocking_GlobalMute = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      m_bAudBlocking_GlobalMute = m_poStaticConfigurationBase->bGetAudBlockTriggerBasedOnGlobaleMute();
   }
   return m_bAudBlocking_GlobalMute;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclConfigurationManager::vGetAudioPipeConfig()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vGetAudioPipeConfig(tmapAudioPipeConfig &rfmapAudioPipeConfig)
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::vGetAudioPipeConfig entered"));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vGetAudioPipeConfig(rfmapAudioPipeConfig);
   }
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetMLDynAudioSupport()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetMLDynAudioSupport() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::bGetMLDynAudioSupport entered"));
   t_Bool bMLDynAudioSupport = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bMLDynAudioSupport = m_poStaticConfigurationBase->bGetMLDynAudioSupport();
   }
   return bMLDynAudioSupport;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclConfigurationManager::vSetMLDynAudioSupport()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vSetMLDynAudioSupport(t_Bool bIsML11OrAboveDevice)
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::vSetMLDynAudioSupport entered"));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vSetMLDynAudioSupport(bIsML11OrAboveDevice);
   }
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetAAPMediaStreamRecEnabled()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetAAPMediaStreamRecEnabled() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::bGetAAPMediaStreamRecEnabled entered"));
   t_Bool bAAPMediaStreamRecEnabled = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bAAPMediaStreamRecEnabled = m_poStaticConfigurationBase->bGetAAPMediaStreamRecEnabled();
   }
   return bAAPMediaStreamRecEnabled;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclConfigurationManager::bGetAAPGuidanceStreamRecEnabled()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetAAPGuidanceStreamRecEnabled() const
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::bGetAAPGuidanceStreamRecEnabled"));
   t_Bool bAAPGuidanceStreamRec = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bAAPGuidanceStreamRec = m_poStaticConfigurationBase->bGetAAPGuidanceStreamRecEnabled();
   }
   return bAAPGuidanceStreamRec;
}

/***************************************************************************
 ** FUNCTION: t_String spi_tclConfigurationManager::szGetAudSinkMaxUnackedFrames()
 ***************************************************************************/
t_String spi_tclConfigurationManager::szGetAudSinkMaxUnackedFrames() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::szGetAudSinkMaxUnackedFrames entered"));
   t_String szAudSinkMaxUnackedFrames;
   if (NULL != m_poStaticConfigurationBase)
   {
      szAudSinkMaxUnackedFrames = m_poStaticConfigurationBase->szGetAudSinkMaxUnackedFrames();
   }
   return szAudSinkMaxUnackedFrames;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetRTPInSupport()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetRTPInSupport() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::bGetRTPInSupport entered"));
   t_Bool bRTPInSupport = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bRTPInSupport = m_poStaticConfigurationBase->bGetRTPInSupport();
   }
   return bRTPInSupport;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetMySIPNAdvAudioSupport()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetMySIPNAdvAudioSupport() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::bGetMySIPNAdvAudioSupport entered"));
   t_Bool bMySIPNAdvAudioSupport = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bMySIPNAdvAudioSupport = m_poStaticConfigurationBase->bGetMySIPNAdvAudioSupport();
   }
   return bMySIPNAdvAudioSupport;
}

/***************************************************************************
 ** FUNCTION: t_String spi_tclConfigurationManager::szGetDisableRealTimePriorityAudio()
 ***************************************************************************/
t_String spi_tclConfigurationManager::szGetDisableRealTimePriorityAudio() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::szGetDisableRealTimePriorityAudio entered"));
   t_String szDisableRealTimePriorityAudio;
   if (NULL != m_poStaticConfigurationBase)
   {
      szDisableRealTimePriorityAudio = m_poStaticConfigurationBase->szGetDisableRealTimePriorityAudio();
   }
   return szDisableRealTimePriorityAudio;
}

/***************************************************************************
 ** FUNCTION: t_String spi_tclConfigurationManager::szGetAudThreadsRealTimePriority()
 ***************************************************************************/
t_String spi_tclConfigurationManager::szGetAudThreadsRealTimePriority() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::szGetAudThreadsRealTimePriority entered"));
   t_String szAudThreadsRealTimePriority;
   if (NULL != m_poStaticConfigurationBase)
   {
      szAudThreadsRealTimePriority = m_poStaticConfigurationBase->szGetAudThreadsRealTimePriority();
   }
   return szAudThreadsRealTimePriority;
}

/***************************************************************************
 ** FUNCTION: t_String spi_tclConfigurationManager::szGetAudSinkThresholdMs()
 ***************************************************************************/
t_String spi_tclConfigurationManager::szGetAudSinkThresholdMs() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::szGetAudSinkThresholdMs entered"));
   t_String szAudSinkThresholdMs;
   if (NULL != m_poStaticConfigurationBase)
   {
      szAudSinkThresholdMs = m_poStaticConfigurationBase->szGetAudSinkThresholdMs();
   }
   return szAudSinkThresholdMs;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclConfigurationManager::vGetBdclMediaSinkConfig()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vGetBdclMediaSinkConfig(std::map<t_String, t_String>& mapAudioConfig) const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::vGetBdclMediaSinkConfig entered"));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vGetBdclMediaSinkConfig(mapAudioConfig);
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclConfigurationManager::vGetBdclTTSSinkConfig()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vGetBdclTTSSinkConfig(std::map<t_String, t_String>& mapAudioConfig) const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::vGetBdclTTSSinkConfig entered"));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vGetBdclTTSSinkConfig(mapAudioConfig);
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclConfigurationManager::vGetBdclVRSinkConfig()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vGetBdclVRSinkConfig(std::map<t_String, t_String>& mapAudioConfig) const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::vGetBdclVRSinkConfig entered"));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vGetBdclVRSinkConfig(mapAudioConfig);
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclConfigurationManager::vGetBdclVRSourceConfig()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vGetBdclVRSourceConfig(std::map<t_String, t_String>& mapAudioConfig) const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::vGetBdclVRSourceConfig entered"));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vGetBdclVRSourceConfig(mapAudioConfig);
   }
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetLastModeSupport()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetLastModeSupport(tenDeviceCategory enDevCat) const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::bGetLastModeSupport entered"));
   t_Bool bLastModeSupport = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bLastModeSupport = m_poStaticConfigurationBase->bGetLastModeSupport(enDevCat);
   }
   return bLastModeSupport;
}

/***************************************************************************
 ** FUNCTION: tenLastModeSupport spi_tclConfigurationManager::enGetLastModeSupport()
 ***************************************************************************/
tenLastModeSupport spi_tclConfigurationManager::enGetLastModeSupport() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::enGetLastModeSupport entered"));
   tenLastModeSupport enLastModeSupport = e8LASTMODE_DISABLED;
   if (NULL != m_poStaticConfigurationBase)
   {
      enLastModeSupport = m_poStaticConfigurationBase->enGetLastModeSupport();
   }
   return enLastModeSupport;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetAudioMuteEnabled()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetAudioMuteEnabled() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::bGetAudioMuteEnabled entered"));
   t_Bool bAudioMuteEnabled = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bAudioMuteEnabled = m_poStaticConfigurationBase->bGetAudioMuteEnabled();
   }
   return bAudioMuteEnabled;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclConfigurationManager::bGetVolumeLockEnabled()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetVolumeLockEnabled() const
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::bGetVolumeLockEnabled"));
   t_Bool bVolumeLockEnabled = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bVolumeLockEnabled = m_poStaticConfigurationBase->bGetVolumeLockEnabled();
   }
   return bVolumeLockEnabled;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetMLDynAudioApplicability()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetMLDynAudioApplicability() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::bGetMLDynAudioApplicability entered"));
   t_Bool bMLDynAudioApplicability = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bMLDynAudioApplicability = m_poStaticConfigurationBase->bGetMLDynAudioApplicability();
   }
   return bMLDynAudioApplicability;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclConfigurationManager::vClearPrivateData()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vClearPrivateData()
{
   ETG_TRACE_USR1((" [DESC]  spi_tclConfigurationManager::vClearPrivateData(): Clearing private data "));
   if (NULL != m_poDynamicConfigurationBase)
   {
      m_rHeadUnitInfo.szVehicleID = m_poDynamicConfigurationBase->szGetVehicleId();
   }
}

/***************************************************************************
 ** FUNCTION: tenDeviceSelectionMode spi_tclConfigurationManager::enGetDeviceSelectionMode()
 ***************************************************************************/
tenDeviceSelectionMode spi_tclConfigurationManager::enGetDeviceSelectionMode(tenDeviceCategory enTechnologyType) const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::enGetDeviceSelectionMode entered"));
   tenDeviceSelectionMode enDeviceSelectionMode = e16DEVICESEL_UNKNOWN;
   if (NULL != m_poStaticConfigurationBase)
   {
      enDeviceSelectionMode = m_poStaticConfigurationBase->enGetDeviceSelectionMode(enTechnologyType);
   }
   return enDeviceSelectionMode;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclConfigurationManager::vSetDeviceSelectionMode()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vSetDeviceSelectionMode(tenDeviceSelectionMode enSelectionMode,
         tenDeviceCategory enTechnologyType)
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::vSetDeviceSelectionMode entered"));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vSetDeviceSelectionMode(enSelectionMode, enTechnologyType);
   }

   if ((e8DEV_TYPE_DEFAULT == enTechnologyType) && (NULL != m_poDynamicConfigurationBase))
   {
      m_poDynamicConfigurationBase->vSetDeviceSelectionMode(enSelectionMode, enTechnologyType);
   }
}

/***************************************************************************
 ** FUNCTION: tenDeviceCategory spi_tclConfigurationManager::enGetTechnologyPreference()
 ***************************************************************************/
tenDeviceCategory spi_tclConfigurationManager::enGetTechnologyPreference(tenDeviceType enDeviceType) const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::enGetTechnologyPreference entered"));
   tenDeviceCategory enTechnologyPreference = e8DEV_TYPE_UNKNOWN;
   if (NULL != m_poStaticConfigurationBase)
   {
      enTechnologyPreference = m_poStaticConfigurationBase->enGetTechnologyPreference(enDeviceType);
   }
   return enTechnologyPreference;
}

/***************************************************************************
 ** FUNCTION: tenCertificateType spi_tclConfigurationManager::enGetCertificateType()
 ***************************************************************************/
tenCertificateType spi_tclConfigurationManager::enGetCertificateType() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::enGetCertificateType entered"));
   tenCertificateType enCertificateType = e8_CERTIFICATETYPE_DEVELOPER;
   if (NULL != m_poStaticConfigurationBase)
   {
      enCertificateType = m_poStaticConfigurationBase->enGetCertificateType();
   }
   return enCertificateType;
}

/***************************************************************************
 ** FUNCTION: tenSelModePriority spi_tclConfigurationManager::enGetSelectionModePriority()
 ***************************************************************************/
tenSelModePriority spi_tclConfigurationManager::enGetSelectionModePriority() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::enGetSelectionModePriority entered"));
   tenSelModePriority enSelectionModePriority = e8PRIORITY_DEVICELIST_HISTORY;
   if (NULL != m_poStaticConfigurationBase)
   {
      enSelectionModePriority = m_poStaticConfigurationBase->enGetSelectionModePriority();
   }
   return enSelectionModePriority;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclConfigurationManager::vGetPersistentStoragePath()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vGetPersistentStoragePath(t_String &rfszPersStoragePath)
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::vGetPersistentStoragePath entered"));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vGetPersistentStoragePath(rfszPersStoragePath);
   }
}

/***************************************************************************
 ** FUNCTION: t_U32 spi_tclConfigurationManager::u32GetDeviceHistorySize()
 ***************************************************************************/
t_U32 spi_tclConfigurationManager::u32GetDeviceHistorySize() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::u32GetDeviceHistorySize entered"));
   t_U32 u32DeviceHistorySize = 0;
   if (NULL != m_poStaticConfigurationBase)
   {
      u32DeviceHistorySize = m_poStaticConfigurationBase->u32GetDeviceHistorySize();
   }
   return u32DeviceHistorySize;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bIsXMLValidationEnabled()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bIsXMLValidationEnabled() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::bIsXMLValidationEnabled entered"));
   t_Bool bXMLValidation = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bXMLValidation = m_poStaticConfigurationBase->bIsXMLValidationEnabled();
   }
   return bXMLValidation;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclConfigurationManager::vGetHeadUnitInfo()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vGetHeadUnitInfo(trHeadUnitInfo &rfrHeadUnitInfo)
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::vGetHeadUnitInfo entered"));
   rfrHeadUnitInfo = m_rHeadUnitInfo;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclConfigurationManager::vGetDefaultProjUsageSettings()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vGetDefaultProjUsageSettings(tenEnabledInfo &enEnabledInfo,
         tenDeviceCategory enSPIType)
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::vGetDefaultProjUsageSettings entered"));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vGetDefaultProjUsageSettings(enEnabledInfo, enSPIType);
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclConfigurationManager::vGetMySPINAoapAppInfo()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vGetMySPINAoapAppInfo(trMySPINAoapAppInfo& rfrMySPINAoapAppInfo)
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::vGetMySPINAoapAppInfo entered"));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vGetMySPINAoapAppInfo(rfrMySPINAoapAppInfo);
   }
}

/***************************************************************************
 ** FUNCTION:  t_U16 spi_tclConfigurationManager::u16GetGeneralRestrictionsInfo()
 ***************************************************************************/
t_U16 spi_tclConfigurationManager::u16GetGeneralRestrictionsInfo(tenDeviceCategory enDeviceCategory) const
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::u16GetGeneralRestrictionsInfo"));
   t_U16 u16GeneralRestrictions = 0;
   if (NULL != m_poDynamicConfigurationBase)
   {
      u16GeneralRestrictions = m_poDynamicConfigurationBase->u16GetGeneralRestrictions(enDeviceCategory);
   }
   return u16GeneralRestrictions;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetCdbGPSSeviceEnabled()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetCdbGPSSeviceEnabled() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::bGetCdbGPSSeviceEnabled entered"));
   t_Bool bCdbGPSSeviceEnabled = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bCdbGPSSeviceEnabled = m_poStaticConfigurationBase->bGetCdbGPSSeviceEnabled();
   }
   return bCdbGPSSeviceEnabled;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetCdbLocationSeviceEnabled()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetCdbLocationSeviceEnabled() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::bGetCdbLocationSeviceEnabled entered"));
   t_Bool bCdbLocationSeviceEnabled = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bCdbLocationSeviceEnabled = m_poStaticConfigurationBase->bGetCdbLocationSeviceEnabled();
   }
   return bCdbLocationSeviceEnabled;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetCdbPrivateSeviceEnabled()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetCdbPrivateSeviceEnabled() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::bGetCdbPrivateSeviceEnabled entered"));
   t_Bool bCdbPrivateSeviceEnabled = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bCdbPrivateSeviceEnabled = m_poStaticConfigurationBase->bGetCdbPrivateSeviceEnabled();
   }
   return bCdbPrivateSeviceEnabled;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetCdbMediaMetadataSinkEnabled()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetCdbMediaMetadataSinkEnabled() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::bGetCdbMediaMetadataSinkEnabled entered"));
   t_Bool bCdbMediaMetadataSinkEnabled = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bCdbMediaMetadataSinkEnabled = m_poStaticConfigurationBase->bGetCdbMediaMetadataSinkEnabled();
   }
   return bCdbMediaMetadataSinkEnabled;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetCdbNaviMetadataSinkEnabled()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetCdbNaviMetadataSinkEnabled() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::bGetCdbNaviMetadataSinkEnabled entered"));
   t_Bool bCdbNaviMetadataSinkEnabled = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bCdbNaviMetadataSinkEnabled = m_poStaticConfigurationBase->bGetCdbNaviMetadataSinkEnabled();
   }
   return bCdbNaviMetadataSinkEnabled;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetEnvDataSubEnabled()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetEnvDataSubEnabled() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::bGetEnvDataSubEnabled entered"));
   t_Bool bEnvDataSubEnabled = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bEnvDataSubEnabled = m_poStaticConfigurationBase->bGetEnvDataSubEnabled();
   }
   return bEnvDataSubEnabled;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetGearStatusEnabled()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetGearStatusEnabled() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::bGetGearStatusEnabled entered"));
   t_Bool bGearStatusEnabled = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bGearStatusEnabled = m_poStaticConfigurationBase->bGetGearStatusEnabled();
   }
   return bGearStatusEnabled;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetAccelerometerDataEnabled()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetAccelerometerDataEnabled() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::bGetAccelerometerDataEnabled entered"));
   t_Bool bAccelerometerDataEnabled = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bAccelerometerDataEnabled = m_poStaticConfigurationBase->bGetAccelerometerDataEnabled();
   }
   return bAccelerometerDataEnabled;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetGyroDataEnabled()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetGyroDataEnabled() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::bGetGyroDataEnabled entered"));
   t_Bool bGyroDataEnabled = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bGyroDataEnabled = m_poStaticConfigurationBase->bGetGyroDataEnabled();
   }
   return bGyroDataEnabled;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetParkBrakeDataEnabled()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetParkBrakeDataEnabled() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::bGetParkBrakeDataEnabled entered"));
   t_Bool bParkBrakeDataEnabled = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bParkBrakeDataEnabled = m_poStaticConfigurationBase->bGetParkBrakeDataEnabled();
   }
   return bParkBrakeDataEnabled;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bIsBochNavigation()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bIsBochNavigation() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::bIsBochNavigation entered"));
   t_Bool bBochNavi = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bBochNavi = m_poStaticConfigurationBase->bIsBochNavigation();
   }
   return bBochNavi;
}

/***************************************************************************
 ** FUNCTION: t_U32 spi_tclConfigurationManager::u32GetSpeedTimerInterval()
 ***************************************************************************/
t_U32 spi_tclConfigurationManager::u32GetSpeedTimerInterval() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::u32GetSpeedTimerInterval entered"));
   t_U32 u32SpeedTimerInterval = 0;
   if (NULL != m_poStaticConfigurationBase)
   {
      u32SpeedTimerInterval = m_poStaticConfigurationBase->u32GetSpeedTimerInterval();
   }
   return u32SpeedTimerInterval;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetLocDataEnabled()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetLocDataEnabled() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::bGetLocDataEnabled entered"));
   t_Bool bLocDataEnabled = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bLocDataEnabled = m_poStaticConfigurationBase->bGetLocDataEnabled();
   }
   return bLocDataEnabled;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetMLNotiSettingVal()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetMLNotiSettingVal() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::bGetMLNotiSettingVal entered"));
   t_Bool bMLNotiSettingVal = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bMLNotiSettingVal = m_poStaticConfigurationBase->bGetMLNotiSettingVal();
   }
   return bMLNotiSettingVal;
}

/***************************************************************************
 ** FUNCTION: tenEnabledInfo spi_tclConfigurationManager::bGetLocDataEnabled()
 ***************************************************************************/
tenEnabledInfo spi_tclConfigurationManager::enGetMLLinkEnableSetVal() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::enGetMLLinkEnableSetVal entered"));
   tenEnabledInfo enMLLinkEnableSetVal = e8USAGE_UNKNOWN;
   if (NULL != m_poStaticConfigurationBase)
   {
      enMLLinkEnableSetVal = m_poStaticConfigurationBase->enGetMLLinkEnableSetVal();
   }
   return enMLLinkEnableSetVal;
}

/***************************************************************************
 ** FUNCTION: tenEnabledInfo spi_tclConfigurationManager::bGetLocDataEnabled()
 ***************************************************************************/
tenEnabledInfo spi_tclConfigurationManager::enGetDipoEnableSetVal() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::enGetDipoEnableSetVal entered"));
   tenEnabledInfo enDipoEnableSetVal = e8USAGE_UNKNOWN;
   if (NULL != m_poStaticConfigurationBase)
   {
      enDipoEnableSetVal = m_poStaticConfigurationBase->enGetDipoEnableSetVal();
   }
   return enDipoEnableSetVal;
}

/***************************************************************************
 ** FUNCTION: tenEnabledInfo spi_tclConfigurationManager::enGetAAPEnableSetVal()
 ***************************************************************************/
tenEnabledInfo spi_tclConfigurationManager::enGetAAPEnableSetVal() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::enGetAAPEnableSetVal entered"));
   tenEnabledInfo enAAPEnableSetVal = e8USAGE_UNKNOWN;
   if (NULL != m_poStaticConfigurationBase)
   {
      enAAPEnableSetVal = m_poStaticConfigurationBase->enGetAAPEnableSetVal();
   }
   return enAAPEnableSetVal;
}

/***************************************************************************
 ** FUNCTION: tenEnabledInfo spi_tclConfigurationManager::enGetCarlifeEnableSetVal()
 ***************************************************************************/
tenEnabledInfo spi_tclConfigurationManager::enGetCarlifeEnableSetVal() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::enGetCarlifeEnableSetVal entered"));
   tenEnabledInfo enCarlifeEnableSetVal = e8USAGE_UNKNOWN;
   if (NULL != m_poStaticConfigurationBase)
   {
      enCarlifeEnableSetVal = m_poStaticConfigurationBase->enGetCarlifeEnableSetVal();
   }
   return enCarlifeEnableSetVal;
}

/***************************************************************************
 ** FUNCTION: tenEnabledInfo spi_tclConfigurationManager::enGetMySPINEnableSetVal()
 ***************************************************************************/
tenEnabledInfo spi_tclConfigurationManager::enGetMySPINEnableSetVal() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::enGetMySPINEnableSetVal entered"));
   tenEnabledInfo enMySPINEnableSetVal = e8USAGE_UNKNOWN;
   if (NULL != m_poStaticConfigurationBase)
   {
      enMySPINEnableSetVal = m_poStaticConfigurationBase->enGetMySPINEnableSetVal();
   }
   return enMySPINEnableSetVal;
}

/***************************************************************************
 ** FUNCTION: tenDriveSideInfo spi_tclConfigurationManager::enGetSteeringWheelPos()
 ***************************************************************************/
tenDriveSideInfo spi_tclConfigurationManager::enGetSteeringWheelPos() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::enGetSteeringWheelPos entered"));
   tenDriveSideInfo enSteeringWheelPos = e8UNKNOWN_DRIVE_SIDE;
   if (NULL != m_poStaticConfigurationBase)
   {
      enSteeringWheelPos = m_poStaticConfigurationBase->enGetSteeringWheelPos();
   }
   return enSteeringWheelPos;
}

/***************************************************************************
 ** FUNCTION: tenDeviceSelectionMode spi_tclConfigurationManager::enGetAAPEnableSetVal()
 ***************************************************************************/
tenDeviceSelectionMode spi_tclConfigurationManager::enGetSelectMode() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::enGetSelectMode entered"));
   tenDeviceSelectionMode enSelectMode = e16DEVICESEL_UNKNOWN;
   if (NULL != m_poStaticConfigurationBase)
   {
      enSelectMode = m_poStaticConfigurationBase->enGetSelectMode();
   }
   return enSelectMode;
}

/***************************************************************************
 ** FUNCTION: tenDeviceCategory spi_tclConfigurationManager::enGetAndroidDeviceTechnologyPreference()
 ***************************************************************************/
tenDeviceCategory spi_tclConfigurationManager::enGetAndroidDeviceTechnologyPreference() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::enGetAndroidDeviceTechnologyPreference entered"));
   tenDeviceCategory enAndroidDevTechPref = e8DEV_TYPE_UNKNOWN;
   if (NULL != m_poStaticConfigurationBase)
   {
      enAndroidDevTechPref = m_poStaticConfigurationBase->enGetAndroidDeviceTechnologyPreference();
   }
   return enAndroidDevTechPref;
}

/***************************************************************************
 ** FUNCTION: tenDeviceCategory spi_tclConfigurationManager::enGetAppleDeviceTechnologyPreference()
 ***************************************************************************/
tenDeviceCategory spi_tclConfigurationManager::enGetAppleDeviceTechnologyPreference() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::enGetAppleDeviceTechnologyPreference entered"));
   tenDeviceCategory enAppleDevTechPref = e8DEV_TYPE_UNKNOWN;
   if (NULL != m_poStaticConfigurationBase)
   {
      enAppleDevTechPref = m_poStaticConfigurationBase->enGetAppleDeviceTechnologyPreference();
   }
   return enAppleDevTechPref;
}

/***************************************************************************
 ** FUNCTION: t_U32 spi_tclConfigurationManager::u32GetStartUpTimeInterval()
 ***************************************************************************/
t_U32 spi_tclConfigurationManager::u32GetStartUpTimeInterval() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::u32GetStartUpTimeInterval entered"));
   t_U32 u32StartUpTimeInterval = 0;
   if (NULL != m_poStaticConfigurationBase)
   {
      u32StartUpTimeInterval = m_poStaticConfigurationBase->u32GetStartUpTimeInterval();
   }
   return u32StartUpTimeInterval;
}

/***************************************************************************
 ** FUNCTION: tenETCInfo spi_tclConfigurationManager::enGetAppleDeviceTechnologyPreference()
 ***************************************************************************/
tenETCInfo spi_tclConfigurationManager::enGetCarPlayETCEnabledFlag(tenDeviceCategory enDevCat) const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::enGetCarPlayETCEnabledFlag entered"));
   tenETCInfo m_enETCSupport = e8ETCInfo_NotSupported;
   if (NULL != m_poDynamicConfigurationBase)
   {
      m_enETCSupport =
               ((m_poDynamicConfigurationBase->u16GetGeneralRestrictions(enDevCat)) & su8ETCEnabledMask) ?
                        e8ETCInfo_InActive : e8ETCInfo_NotSupported;
      ETG_TRACE_USR2(("[PARAM]:spi_tclConfigurationManager::enGetCarPlayETCEnabledFlag -ETC Status : %d", ETG_ENUM(ETC_SUPPORT,
               m_enETCSupport)));
   }
   return m_enETCSupport;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclConfigurationManager::vGetScreenOffset()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vGetScreenOffset(trScreenOffset &rfoScreenOffset) const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::vGetScreenOffset entered"));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vGetScreenOffset(rfoScreenOffset);
   }
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bIsServerSideScalingRequired()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bIsServerSideScalingRequired()
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::bIsServerSideScalingRequired entered"));
   t_Bool bScalingRequired = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bScalingRequired = m_poStaticConfigurationBase->bIsServerSideScalingRequired();
   }
   return bScalingRequired;
}

/***************************************************************************
 ** FUNCTION: t_U8 spi_tclConfigurationManager::u8EnableTouchInputEvents()
 ***************************************************************************/
t_U8 spi_tclConfigurationManager::u8EnableTouchInputEvents()
{
   t_U8 u8TouchInputEvents = 0;
   if (NULL != m_poStaticConfigurationBase)
   {
      u8TouchInputEvents = m_poStaticConfigurationBase->u8EnableTouchInputEvents();
   }
   ETG_TRACE_USR2(("spi_tclConfigurationManager::u8EnableTouchInputEvents = %d ", u8TouchInputEvents));
   return u8TouchInputEvents;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclConfigurationManager::vSetScreenSize()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vSetScreenSize(const trScreenAttributes& corfrScreenAttributes)
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::vSetScreenSize entered"));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vSetScreenSize(corfrScreenAttributes);
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclConfigurationManager::vSetContAttestFlag()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vSetContAttestFlag(t_U8 u8ContAttestFlag)
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::vSetContAttestFlag entered"));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vSetContAttestFlag(u8ContAttestFlag);
   }
}

/***************************************************************************
 ** FUNCTION: t_String spi_tclConfigurationManager::szGetVideoCodecType()
 ***************************************************************************/
t_String spi_tclConfigurationManager::szGetVideoCodecType() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::szGetVideoCodecType entered"));
   t_String szVideoCodecType;
   if (NULL != m_poStaticConfigurationBase)
   {
      szVideoCodecType = m_poStaticConfigurationBase->szGetVideoCodecType();
   }
   ETG_TRACE_USR1(("spi_tclConfigurationManager:szGetVideoCodecType()-%s ", szVideoCodecType.c_str()));
   return szVideoCodecType.c_str();
}

/***************************************************************************
 ** FUNCTION: t_U8 spi_tclConfigurationManager::u8GetMaxUnackedFrames()
 ***************************************************************************/
t_U8 spi_tclConfigurationManager::u8GetMaxUnackedFrames() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::u8GetMaxUnackedFrames entered"));
   t_U8 u8MaxUnackedFrames = 0;
   if (NULL != m_poStaticConfigurationBase)
   {
      u8MaxUnackedFrames = m_poStaticConfigurationBase->u8GetMaxUnackedFrames();
   }
   return u8MaxUnackedFrames;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetAutoStartProjection()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetAutoStartProjection() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::bGetAutoStartProjection entered"));
   t_Bool bAutoStartProjection = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bAutoStartProjection = m_poStaticConfigurationBase->bGetAutoStartProjection();
   }
   return bAutoStartProjection;
}

/***************************************************************************
 ** FUNCTION: t_U8 spi_tclConfigurationManager::u8GetFramesPerSec()
 ***************************************************************************/
t_U8 spi_tclConfigurationManager::u8GetFramesPerSec() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::u8GetFramesPerSec entered"));
   t_U8 u8FramesPerSec = 0;
   if (NULL != m_poStaticConfigurationBase)
   {
      u8FramesPerSec = m_poStaticConfigurationBase->u8GetFramesPerSec();
   }
   return u8FramesPerSec;
}

/***************************************************************************
 ** FUNCTION: t_U8 spi_tclConfigurationManager::u8GetContAttestSignFlag()
 ***************************************************************************/
t_U8 spi_tclConfigurationManager::u8GetContAttestSignFlag() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::u8GetContAttestSignFlag entered"));
   t_U8 u8ContAttestSignFlag = 0;
   if (NULL != m_poStaticConfigurationBase)
   {
      u8ContAttestSignFlag = m_poStaticConfigurationBase->u8GetContAttestSignFlag();
   }
   return u8ContAttestSignFlag;
}

/***************************************************************************
 ** FUNCTION: tenPixelFormat spi_tclConfigurationManager::enGetPixelFormat()
 ***************************************************************************/
tenPixelFormat spi_tclConfigurationManager::enGetPixelFormat() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::enGetPixelFormat entered"));
   tenPixelFormat enPixelFormat = e8_PIXELFORMAT_UNKNOWN;
   if (NULL != m_poStaticConfigurationBase)
   {
      enPixelFormat = m_poStaticConfigurationBase->enGetPixelFormat();
   }
   return enPixelFormat;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclConfigurationManager::vGetBdclVideoSinkConfig()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vGetBdclVideoSinkConfig(std::map<t_String, t_String>& mapVideoConfig) const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::vGetBdclVideoSinkConfig entered"));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vGetBdclVideoSinkConfig(mapVideoConfig);
   }
}

/***************************************************************************
 ** FUNCTION: tenGeoCoordinateSystemType spi_tclConfigurationManager::vGetBdclVideoSinkConfig()
 ***************************************************************************/
tenGeoCoordinateSystemType spi_tclConfigurationManager::enGetGeoCoordinateSystem() const
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::enGetGeoCoordinateSystem entered"));
   tenGeoCoordinateSystemType enGeoCoordinateSystemType = e8GEO_COORDINATE_SYSTEM_UNKNOWN;
   if (NULL != m_poStaticConfigurationBase)
   {
      enGeoCoordinateSystemType = m_poStaticConfigurationBase->enGetGeoCoordinateSystem();
   }
   return enGeoCoordinateSystemType;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConfigurationManager::vReadConfigurationData
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vReadConfigurationData()
{
   ETG_TRACE_USR2(("spi_tclConfigurationManager::vReadConfigurationData()"));

   if ((NULL != m_poStaticConfigurationBase) && (NULL != m_poDynamicConfigurationBase))
   {
      m_rHeadUnitInfo.enDrivePosition = m_poDynamicConfigurationBase->enGetDriveSideInfo();
      trVehicleInfo rfrVehicleInfo;
      m_poStaticConfigurationBase->vGetVehicleInfoDataAAP(rfrVehicleInfo);
      m_rHeadUnitInfo.szVehicleManufacturer = rfrVehicleInfo.szManufacturer;
      m_rHeadUnitInfo.szVehicleModelName = rfrVehicleInfo.szModel;
      m_rHeadUnitInfo.szModelYear = m_poStaticConfigurationBase->szGetModelYear();
      m_rHeadUnitInfo.szSoftwareVersion = m_poDynamicConfigurationBase->szGetSoftwareVersion();
      m_rHeadUnitInfo.szSoftwareBuild = m_poStaticConfigurationBase->szGetSoftwareBuild();
      m_rHeadUnitInfo.szHardwareVersion = m_poStaticConfigurationBase->szGetHardwareVersion();
      m_rHeadUnitInfo.szHUModelName = m_poStaticConfigurationBase->szGetHeadUnitModelName();
      m_rHeadUnitInfo.szHUManufacturer = m_poStaticConfigurationBase->szGetHeadUnitManufacturerName();
      m_rHeadUnitInfo.szVehicleID = m_poDynamicConfigurationBase->szGetVehicleId();
      m_rHeadUnitInfo.enDrivePosition = m_poDynamicConfigurationBase->enGetDriveSideInfo();

      //Android Auto Certificates
      trAAPCertConfigData rfrCertsConfigData;
      m_poStaticConfigurationBase->vGetAAPCertificatesConfigData(rfrCertsConfigData);
      m_rHeadUnitInfo.oAAAutoCertConfigData.szKeyId = rfrCertsConfigData.szKeyId;
      m_rHeadUnitInfo.oAAAutoCertConfigData.szCertificateFolderPath = rfrCertsConfigData.szCertificateFolderPath;
      vSetDefaultKeyInfo();
   }

   ETG_TRACE_USR2((" Drive position  = %d", ETG_ENUM(DRIVE_SIDE_TYPE, m_rHeadUnitInfo.enDrivePosition)));
   ETG_TRACE_USR2((" Head unit Manufacturer name  = %s", m_rHeadUnitInfo.szHUManufacturer.c_str()));
   ETG_TRACE_USR2((" Head unit Model name  = %s", m_rHeadUnitInfo.szHUModelName.c_str()));
   ETG_TRACE_USR2((" Serial number  = %s", m_rHeadUnitInfo.szVehicleID.c_str()));
   ETG_TRACE_USR2((" Software Version  = %s", m_rHeadUnitInfo.szSoftwareVersion.c_str()));
   ETG_TRACE_USR2((" Hardware Version  = %s", m_rHeadUnitInfo.szHardwareVersion.c_str()));
   ETG_TRACE_USR2((" Vehicle Manufacturer  = %s", m_rHeadUnitInfo.szVehicleManufacturer.c_str()));
   ETG_TRACE_USR2((" Vehicle Model  = %s", m_rHeadUnitInfo.szVehicleModelName.c_str()));
   ETG_TRACE_USR2((" Vehicle Model Year = %s", m_rHeadUnitInfo.szModelYear.c_str()));
   ETG_TRACE_USR2((" Software Build = %s", m_rHeadUnitInfo.szSoftwareBuild.c_str()));
   //!Certificates Configuration Data
   ETG_TRACE_USR1((" Key Id is %s", m_rHeadUnitInfo.oAAAutoCertConfigData.szKeyId.c_str()));
   ETG_TRACE_USR1((" Certificate Folder Path is %s", m_rHeadUnitInfo.oAAAutoCertConfigData.szCertificateFolderPath.c_str()));
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConfigurationManager::vSetMLNotiSettingVal()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vSetMLNotiSettingVal(t_Bool bMLNotiSettingVal)
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::bSetMLNotiSettingVal() entered "));
   if (NULL != m_poDynamicConfigurationBase)
   {
      m_poDynamicConfigurationBase->vSetMLNotiSettingVal(bMLNotiSettingVal);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclConfigurationManager::vSetMLLinkEnableSetVal()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bSetMLLinkEnableSetVal(tenEnabledInfo enMLLinkEnableSetVal)
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::vSetMLLinkEnableSetVal() entered "));
   t_Bool bRetVal = false;
   if (NULL != m_poDynamicConfigurationBase)
   {
      bRetVal = m_poDynamicConfigurationBase->bSetMLLinkEnableSetVal(enMLLinkEnableSetVal);
   }
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclConfigurationManager::bSetDipoEnableSetVal()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bSetDipoEnableSetVal(tenEnabledInfo enDipoEnableSetVal)
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::vSetDipoEnableSetVal() entered "));
   t_Bool bDipoEnableSetVal = false;
   if (NULL != m_poDynamicConfigurationBase)
   {
      bDipoEnableSetVal = m_poDynamicConfigurationBase->bSetDipoEnableSetVal(enDipoEnableSetVal);
   }
   return bDipoEnableSetVal;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclConfigurationManager::bSetAAPEnableSetVal()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bSetAAPEnableSetVal(tenEnabledInfo enAAPEnableSetVal)
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::bSetAAPEnableSetVal() entered "));
   t_Bool bRetVal = false;
   if (NULL != m_poDynamicConfigurationBase)
   {
      bRetVal = m_poDynamicConfigurationBase->bSetAAPEnableSetVal(enAAPEnableSetVal);
   }
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclConfigurationManager::bSetCarlifeEnableSetVal()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bSetCarlifeEnableSetVal(tenEnabledInfo enCarlifeEnableSetVal)
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::bSetCarlifeEnableSetVal() entered "));
   t_Bool bRetVal = false;
   if (NULL != m_poDynamicConfigurationBase)
   {
      bRetVal = m_poDynamicConfigurationBase->bSetCarlifeEnableSetVal(enCarlifeEnableSetVal);
   }
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConfigurationManager::vSetSteeringWheelPos()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vSetSteeringWheelPos(tenDriveSideInfo enSteeringWheelPos)
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::vSetSteeringWheelPos() entered "));
   if (NULL != m_poDynamicConfigurationBase)
   {
      m_poDynamicConfigurationBase->vSetSteeringWheelPos(enSteeringWheelPos);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConfigurationManager::vSetSelectMode()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vSetSelectMode(tenDeviceSelectionMode enSelectMode)
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::vSetSelectMode() entered "));
   if (NULL != m_poDynamicConfigurationBase)
   {
      m_poDynamicConfigurationBase->vSetSelectMode(enSelectMode);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclConfigurationManager::bSetAndroidDeviceTechnologyPreference()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bSetAndroidDeviceTechnologyPreference(tenDeviceCategory enAndroidDeviceTechPrefVal)
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::vSetAndroidDeviceTechnologyPreference() entered "));
   t_Bool bRetVal = false;
   if (NULL != m_poDynamicConfigurationBase)
   {
      bRetVal = m_poDynamicConfigurationBase->bSetAndroidDeviceTechnologyPreference(enAndroidDeviceTechPrefVal);
   }
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclConfigurationManager::bSetAppleDeviceTechnologyPreference()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bSetAppleDeviceTechnologyPreference(tenDeviceCategory enAppleDeviceTechPrefVal)
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::bSetAppleDeviceTechnologyPreference() entered "));
   t_Bool bRetVal = false;
   if (NULL != m_poDynamicConfigurationBase)
   {
      bRetVal = m_poDynamicConfigurationBase->bSetAppleDeviceTechnologyPreference(enAppleDeviceTechPrefVal);
   }
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  t_U8 spi_tclConfigurationManager::u8GetSourceNumber()
 ***************************************************************************/
t_U8 spi_tclConfigurationManager::u8GetSourceNumber(tenAudioDir enAudDir) const
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::u8GetSourceNumber"));
   t_U8 u8SourceNumber = 0;
   if (NULL != m_poStaticConfigurationBase)
   {
      u8SourceNumber = m_poStaticConfigurationBase->u8GetSourceNumber(enAudDir);
   }
   return u8SourceNumber;
}

/***************************************************************************
 ** FUNCTION:  t_U8 spi_tclConfigurationManager::u8ReadVirginStartSetting()
 ***************************************************************************/
t_U8 spi_tclConfigurationManager::u8ReadVirginStartSetting()
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::u8ReadVirginStartSetting entered"));
   t_U8 u8VirginStart = 0;
   if (NULL != m_poDynamicConfigurationBase)
   {
      u8VirginStart = m_poDynamicConfigurationBase->u8ReadVirginStartSetting();
   }

   return u8VirginStart;

}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConfigurationManager::vWriteVirginStartSetting()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vWriteVirginStartSetting(t_U8 u8DefaultVgStrt)
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::vWriteVirginStartSetting entered"));
   if (NULL != m_poDynamicConfigurationBase)
   {
      m_poDynamicConfigurationBase->vWriteVirginStartSetting(u8DefaultVgStrt);
   }
}
/***************************************************************************
 ** FUNCTION: t_U8 spi_tclConfigurationManager::u8GetRestrictionsInfo(tenDeviceCategory enDevCat, t_Bool bDriveRestrictions)
 ***************************************************************************/
t_U8 spi_tclConfigurationManager::u8GetRestrictionsInfo(tenDeviceCategory enDevCat, t_Bool bDriveRestrictions) const
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::u8GetRestrictionsInfo:DeviceCatgeory:%d, Is request sent for drive restrictions:%d ", ETG_ENUM(DEVICE_CATEGORY,
            enDevCat), ETG_ENUM(BOOL, bDriveRestrictions)));

   t_U8 u8RestrictionsInfo = 0;
   switch (enDevCat)
   {
      case e8DEV_TYPE_DIPO:
      {
         if (NULL != m_poDynamicConfigurationBase)
         {
            u8RestrictionsInfo = m_poDynamicConfigurationBase->u8GetDipoRestrictionInfo(bDriveRestrictions);
         }
         ETG_TRACE_USR2(("spi_tclConfigurationManager::u8GetRestrictionsInfo : DiPO Restrictions:%d ", u8RestrictionsInfo));

      }
         break;
      case e8DEV_TYPE_ANDROIDAUTO:
      {
         if (NULL != m_poStaticConfigurationBase)
         {
            u8RestrictionsInfo = m_poStaticConfigurationBase->u8GetAAPRestrictionInfo(bDriveRestrictions);
         }
         ETG_TRACE_USR2(("spi_tclConfigurationManager::u8GetRestrictionsInfo : AAP Restrictions:%d ", u8RestrictionsInfo));
      }
         break;
      default:
      {

      }
   }

   return u8RestrictionsInfo;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConfigurationManager::vSetFeatureRestrictions(...)
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vSetFeatureRestrictions(tenDeviceCategory enDevCategory,
         const t_U8 cou8ParkModeRestrictionInfo, const t_U8 cou8DriveModeRestrictionInfo)
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::vSetFeatureRestrictions() Entered with Device Category = %d", ETG_ENUM(DEVICE_CATEGORY,
            enDevCategory)));

   if (enDevCategory == e8DEV_TYPE_ANDROIDAUTO)
   {
      if (NULL != m_poStaticConfigurationBase)
      {
         m_poStaticConfigurationBase->vSetAAPFeatureRestrictions(cou8ParkModeRestrictionInfo,
                  cou8DriveModeRestrictionInfo);
      }

   }
   else if (enDevCategory == e8DEV_TYPE_DIPO)
   {
      if (NULL != m_poDynamicConfigurationBase)
      {
         m_poDynamicConfigurationBase->vSetDipoFeatureRestrictions(cou8ParkModeRestrictionInfo,
                  cou8DriveModeRestrictionInfo);
      }
   }
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bWriteAudioLastMode()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bWriteAudioLastMode(t_Bool bAudioLastMode)
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::bWriteAudioLastMode() entered - Audio last mode: %d", ETG_ENUM(BOOL,
            bAudioLastMode)));
   t_Bool bRetVal = false;
   if (NULL != m_poDynamicConfigurationBase)
   {
      bRetVal = m_poDynamicConfigurationBase->bWriteAudioLastMode(bAudioLastMode);
   }
   ETG_TRACE_USR2(("[DESC]: spi_tclConfigurationManager::bWriteAudioLastMode - Value is written successfully = %d", ETG_ENUM(BOOL,
            bRetVal)));
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION   :t_Void spi_tclConfigurationManager::vGetVideoConfigData()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vGetVideoConfigData(tenDeviceCategory enDevCat,
         tvecVideoConfigList& rfvecVideoConfigList)
{

   ETG_TRACE_USR1(("spi_tclConfigurationManager::vGetVideoConfigData - %d", ETG_ENUM(DEVICE_CATEGORY, enDevCat)));
   if (NULL != m_poDynamicConfigurationBase)
   {
      m_poDynamicConfigurationBase->vGetVideoConfigData(enDevCat, rfvecVideoConfigList);
   }
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bWriteDisplayLastMode()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bWriteDisplayLastMode(t_Bool bDisplayLastMode)
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::bWriteDisplayLastMode() entered - Display last mode: %d", ETG_ENUM(BOOL,
            bDisplayLastMode)));
   t_Bool bRetVal = false;
   if (NULL != m_poDynamicConfigurationBase)
   {
      bRetVal = m_poDynamicConfigurationBase->bWriteDisplayLastMode(bDisplayLastMode);
   }
   ETG_TRACE_USR2(("[DESC]: spi_tclConfigurationManager::bWriteDisplayLastMode - Value is written successfully = %d", ETG_ENUM(BOOL,
            bRetVal)));
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bIsDisplayLastModeProjection()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bIsDisplayLastModeProjection()
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::bIsDisplayLastModeProjection() entered "));
   t_Bool bDisplayLastMode = false;
   if (NULL != m_poDynamicConfigurationBase)
   {
      bDisplayLastMode = m_poDynamicConfigurationBase->bIsDisplayLastModeProjection();
   }
   ETG_TRACE_USR2(("[DESC]: spi_tclConfigurationManager::bIsDisplayLastModeProjection - Display last mode is projection = %d", ETG_ENUM(BOOL,
            bDisplayLastMode)));
   return bDisplayLastMode;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclConfigurationManager::enGetDriveSideInfo()
 ***************************************************************************/
tenDriveSideInfo spi_tclConfigurationManager::enGetDriveSideInfo()
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::enGetDriveSideInfo"));
   tenDriveSideInfo enDriveSideInfo = e8UNKNOWN_DRIVE_SIDE;
   if (NULL != m_poDynamicConfigurationBase)
   {
      enDriveSideInfo = m_poDynamicConfigurationBase->enGetDriveSideInfo();
   }
   return enDriveSideInfo;
}

/***************************************************************************
 ** FUNCTION:  t_String spi_tclConfigurationManager::szGetSoftwareVersion()
 ***************************************************************************/
t_String spi_tclConfigurationManager::szGetSoftwareVersion() const
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::szGetSoftwareVersion"));
   t_String szSoftwareVersion;
   if (NULL != m_poDynamicConfigurationBase)
   {
      szSoftwareVersion = m_poDynamicConfigurationBase->szGetSoftwareVersion();
   }
   return szSoftwareVersion;
}

/***************************************************************************
 ** FUNCTION:  t_String spi_tclConfigurationManager::szGetHardwareVersion()
 ***************************************************************************/
t_String spi_tclConfigurationManager::szGetHardwareVersion()
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::szGetHardwareVersion"));
   t_String szHardwareVersion;
   if (NULL != m_poStaticConfigurationBase)
   {
      szHardwareVersion = m_poStaticConfigurationBase->szGetHardwareVersion();
   }
   return szHardwareVersion;
}

/***************************************************************************
 *  ** FUNCTION: t_Void spi_tclConfigurationManager::bIsUSBPort1Enabled
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vGetUSBPortsEnabledStatus(t_Bool &bUSBPort1, t_Bool &bUSBPort2, t_Bool &bUSBPort3)
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::vGetUSBPortsEnabledStatus"));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vCheckUSBPortEnabled(bUSBPort1, bUSBPort2, bUSBPort3);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConfigurationManager::vIntializeSPISettings()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vIntializeSPISettings()
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::vIntializeSPISettings() entered "));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vIntializeSPISettings();
   }
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclConfigurationManager::bIsDeviceBlacklisted()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bIsDeviceBlacklisted(t_S32 s32ProductID, t_S32 s32VendorID,
         tenDeviceCategory enDeviceCtegory)
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::bIsDeviceBlacklisted() entered "));
   t_Bool bRetVal = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bRetVal = m_poStaticConfigurationBase->bIsDeviceBlacklisted(s32ProductID, s32VendorID, enDeviceCtegory);
   }
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConfigurationManager::vIsDeviceWhitelisted()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vIsDeviceWhitelisted(t_S32 s32ProductID, t_S32 s32VendorID,
         tenDeviceCategory &rfenDevCat)
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::vIsDeviceWhitelisted() entered "));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vIsDeviceWhitelisted(s32ProductID, s32VendorID, rfenDevCat);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclConfigurationManager::bIsDeviceWhitelistingEnabled
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bIsDeviceWhitelistingEnabled()
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::bIsDeviceWhitelistingEnabled() entered "));
   t_Bool bRetVal = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bRetVal = m_poStaticConfigurationBase->bIsDeviceWhitelistingEnabled();
   }
   return bRetVal;
}

/***************************************************************************
 *  ** FUNCTION: t_Bool spi_tclDiscovererSettings::bIsWhitelistEnabledCat
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bIsWhitelistEnabledCat(tenDeviceCategory enChosenDeviceCategory)
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::bIsWhitelistEnabledCat() entered"));
   t_Bool bRetVal = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bRetVal = m_poStaticConfigurationBase->bIsWhitelistEnabledCat(enChosenDeviceCategory);
   }
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiscovererSettings::bIsAudioLastModeProjection()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bIsAudioLastModeProjection()
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::bIsAudioLastModeProjection() entered"));
   t_Bool bAudioLastMode = false;
   if (NULL != m_poDynamicConfigurationBase)
   {
      bAudioLastMode = m_poDynamicConfigurationBase->bIsAudioLastModeProjection();
   }
   return bAudioLastMode;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclConfigurationManager::vSetDriveModeInfo()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vSetDriveModeInfo(const tenVehicleConfiguration enDriveModeInfo)
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::vSetDriveModeInfo() entered"));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vSetDriveModeInfo(enDriveModeInfo);
   }
}

/***************************************************************************
 ** FUNCTION: t_String spi_tclConfigurationManager::szGetClientManufacturerName()
 ***************************************************************************/
t_String spi_tclConfigurationManager::szGetClientManufacturerName() const
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::szGetClientManufacturerName() entered"));
   t_String szManufacturerName;
   if (NULL != m_poStaticConfigurationBase)
   {
      szManufacturerName = m_poStaticConfigurationBase->szGetClientManufacturerName();
   }
   return szManufacturerName;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclConfigurationManager::vSetNightModeInfo()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vSetNightModeInfo(const tenVehicleConfiguration enNightModeInfo)
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::vSetNightModeInfo() entered"));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vSetNightModeInfo(enNightModeInfo);
   }
}

/***************************************************************************
 ** FUNCTION: t_String spi_tclConfigurationManager::szGetChannelID()
 ***************************************************************************/
t_String spi_tclConfigurationManager::szGetChannelID() const
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::szGetChannelID() entered"));
   t_String szChannelID;
   if (NULL != m_poStaticConfigurationBase)
   {
      szChannelID = m_poStaticConfigurationBase->szGetChannelID();
   }
   return szChannelID;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetRotaryCtrlSupport()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetRotaryCtrlSupport()
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::bGetRotaryCtrlSupport() entered"));
   t_Bool bRotaryCtrl = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bRotaryCtrl = m_poStaticConfigurationBase->bGetRotaryCtrlSupport();
   }
   return bRotaryCtrl;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclConfigurationManager::vGetOemIconData()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vGetOemIconData(trVehicleBrandInfo& rfrVehicleBrandInfo)
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::vGetOemIconData() entered"));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vGetOemIconData(rfrVehicleBrandInfo);
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclConfigurationManager::vGetDriveModeInfo()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vGetDriveModeInfo(tenVehicleConfiguration &rfoenDriveModeInfo)
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::vGetDriveModeInfo() entered"));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vGetDriveModeInfo(rfoenDriveModeInfo);
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclConfigurationManager::vGetNightModeInfo()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vGetNightModeInfo(tenVehicleConfiguration &rfoenNightModeInfo)
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::vGetNightModeInfo() entered"));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vGetNightModeInfo(rfoenNightModeInfo);
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclConfigurationManager::vGetDisplayInputParam()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vGetDisplayInputParam(t_U8 &rfu8DisplayInput)
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::vGetDisplayInputParam() entered"));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vGetDisplayInputParam(rfu8DisplayInput);
   }
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetNativeNavigationEnabled()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetNativeNavigationEnabled() const
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::bGetNativeNavigationEnabled() entered"));
   t_Bool bNativeNavigationEnabled = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bNativeNavigationEnabled = m_poStaticConfigurationBase->bGetNativeNavigationEnabled();
   }
   return bNativeNavigationEnabled;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bSetUpAPonOOBT()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bSetUpAPonOOBT()
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::bSetUpAPonOOBT() entered"));
   t_Bool bAPonOOBT = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bAPonOOBT = m_poStaticConfigurationBase->bSetUpAPonOOBT();
   }
   return bAPonOOBT;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclConfigurationManager::szGetVehicleModelNumber()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::szGetVehicleModelNumber(t_String& szVehicleModelNumber) const
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::szGetVehicleModelNumber() entered"));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->szGetVehicleModelNumber(szVehicleModelNumber);
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclConfigurationManager::vGetDiPOExtendedFeaturesSupport()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vGetDiPOExtendedFeaturesSupport(trExtendedFeaturesSupport& rfrExtendedFeaturesSupport)
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::vGetDiPOExtendedFeaturesSupport() entered"));
   if (NULL != m_poStaticConfigurationBase)
   {
      std::set<tenKeyCode> rfrsetDiPoSuppHardKeys, rfrsetDiPoSuppSoftKeys;
      m_poStaticConfigurationBase->vGetDiPOExtendedFeaturesSupport(rfrExtendedFeaturesSupport);

      m_poDynamicConfigurationBase->bGetKeyInfo(rfrsetDiPoSuppHardKeys,e8DEV_TYPE_DIPO);
      m_poDynamicConfigurationBase->bGetSoftKeyInfo(rfrsetDiPoSuppSoftKeys,e8DEV_TYPE_DIPO);
      auto itDiPoSuppHardKeys = rfrsetDiPoSuppHardKeys.find(e32DEV_BACKWARD);
      auto itDiPoSuppSoftKeys = rfrsetDiPoSuppSoftKeys.find(e32DEV_BACKWARD);
      if((( itDiPoSuppHardKeys != rfrsetDiPoSuppHardKeys.end()) || (itDiPoSuppSoftKeys != rfrsetDiPoSuppSoftKeys.end())))
      {
         rfrExtendedFeaturesSupport.bEnhancedRequestUISupport = true;
      }
      else
      {
         rfrExtendedFeaturesSupport.bEnhancedRequestUISupport = false;
      }
      ETG_TRACE_USR1((" spi_tclConfigurationManager::vGetDiPOExtendedFeaturesSupport() - bEnhancedRequestUISupport = %d", ETG_ENUM(BOOL, rfrExtendedFeaturesSupport.bEnhancedRequestUISupport)));
   }
}
/***************************************************************************
 ** FUNCTION: t_String spi_tclConfigurationManager::szGetClientID()
 ***************************************************************************/
t_String spi_tclConfigurationManager::szGetClientID() const
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::szGetClientID() entered"));
   t_String szClientID;
   if (NULL != m_poStaticConfigurationBase)
   {
      szClientID = m_poStaticConfigurationBase->szGetClientID();
   }
   return szClientID;
}

/***************************************************************************
 ** FUNCTION: t_String spi_tclConfigurationManager::szGetClientFriendlyName()
 ***************************************************************************/
t_String spi_tclConfigurationManager::szGetClientFriendlyName() const
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::szGetClientFriendlyName() entered"));
   t_String szClientName;
   if (NULL != m_poStaticConfigurationBase)
   {
      szClientName = m_poStaticConfigurationBase->szGetClientFriendlyName();
   }
   return szClientName;
}

/***************************************************************************
 ** FUNCTION: tenDAPAttestationInfo spi_tclConfigurationManager::enGetDAPAttestationInfo()
 ***************************************************************************/
tenDAPAttestationInfo spi_tclConfigurationManager::enGetDAPAttestationInfo() const
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::enGetDAPAttestationInfo() entered"));
   tenDAPAttestationInfo enDAPInfo = e8_DAP_ATTESTATION_NOT_REQ;
   if (NULL != m_poStaticConfigurationBase)
   {
      enDAPInfo = m_poStaticConfigurationBase->enGetDAPAttestationInfo();
   }
   return enDAPInfo;
}

/***************************************************************************
 ** FUNCTION:  t_String spi_tclConfigurationManager::szGetSPIVersion()
 ***************************************************************************/
t_String spi_tclConfigurationManager::szGetSPIVersion() const
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::szGetSPIVersion() entered"));
   t_String szSPIVersion;
   if (NULL != m_poDynamicConfigurationBase)
   {
      szSPIVersion = m_poDynamicConfigurationBase->szGetSPIVersion();
   }
   return szSPIVersion;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclConfigurationManager::vGetTouchScreenType()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vGetTouchScreenType(tenAAPTouchScreenType &rfenScreenType)
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::vGetTouchScreenType() entered"));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vGetTouchScreenType(rfenScreenType);
   }
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bIsLocDataEnabled()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bIsLocDataEnabled()
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::bIsLocDataEnabled() entered"));
   t_Bool bLocDataEnabled = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bLocDataEnabled = m_poStaticConfigurationBase->bIsLocDataEnabled();
   }
   return bLocDataEnabled;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclConfigurationManager::vGetEAPAppInfo()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vGetEAPAppInfo(trEAPAppInfo& rfrEAPAppInfo)
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::vGetEAPAppInfo() entered"));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vGetEAPAppInfo(rfrEAPAppInfo);
   }
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclConfigurationManager::bReadCPWStatus()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bReadCPWStatus()
{
   t_Bool bCPWStatus = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bCPWStatus = m_poStaticConfigurationBase->bReadCPWStatus();
   }
   ETG_TRACE_USR1((" spi_tclConfigurationManager::bReadCPWStatus():%d",bCPWStatus));
   return bCPWStatus;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclConfigurationManager::vGetEAPAppInfo()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vSelectCarplayConfiguration()
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::vSelectCarplayConfiguration() entered"));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vSelectCarplayConfiguration();
   }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclConfigurationManager::vGetPrimaryDisplayConfiguration()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vGetPrimaryDisplayConfiguration(const tvecVideoConfigList& rfvecVideoConfigList,
         trVideoConfigData& rVideoConfigData)
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::vGetPrimaryDisplayConfiguration() entered"));
   if (NULL != m_poDynamicConfigurationBase)
   {
      m_poDynamicConfigurationBase->vGetPrimaryDisplayConfiguration(rfvecVideoConfigList, rVideoConfigData);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConfigurationManager::vGetSpiFeatureSupport()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vGetSpiFeatureSupport(trSpiFeatureSupport& rfrSpiFeatureSupport)
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::vGetSpiFeatureSupport entered"));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vGetSpiFeatureSupport(rfrSpiFeatureSupport);
   }
}

/***************************************************************************
 ** FUNCTION:  tenATBoxSupportedState spi_tclConfigurationManager::vGetATBoxSupportedState()
 ***************************************************************************/
tenATBoxSupportedState spi_tclConfigurationManager::vGetATBoxSupportedState() const
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::vGetATBoxSupportedState entered"));
   tenATBoxSupportedState enATBoxSupportedState = ATBOX_UNKNOWN;
   if (NULL != m_poStaticConfigurationBase)
   {
      enATBoxSupportedState = m_poStaticConfigurationBase->vGetATBoxSupportedState();
   }
   return enATBoxSupportedState;
}

/***************************************************************************
 ** FUNCTION: tenRegion spi_tclConfigurationManager::enGetRegion()
 ***************************************************************************/
tenRegion spi_tclConfigurationManager::enGetRegion() const
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::enGetRegion() entered"));
   tenRegion enRegion = e8_INVALID;
   if (NULL != m_poStaticConfigurationBase)
   {
      enRegion = m_poStaticConfigurationBase->enGetRegion();
   }
   return enRegion;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConfigurationManager::vSetDriveSideInfo()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vSetDriveSideInfo(const tenVehicleConfiguration enVehicleConfig)
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::vSetDriveSideInfo() entered"));
   if (NULL != m_poDynamicConfigurationBase)
   {
      m_poDynamicConfigurationBase->vSetDriveSideInfo(enVehicleConfig);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConfigurationManager::vSetVehicleManufacturerInformation()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vSetVehicleManufacturerInformation(
         const trVehicleManufacturerInformation& corfrVehicleManufacturerInfoAttr)
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::vSetVehicleManufacturerInformation() entered"));

   if (NULL != m_poDynamicConfigurationBase)
   {
      m_poDynamicConfigurationBase->vSetVehicleManufacturerInformation(corfrVehicleManufacturerInfoAttr);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConfigurationManager::vGetMySPINIapAppInfo()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vGetMySPINIapAppInfo(trMySPINIapAppInfo& rMySPINIapAppInfo)
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::vGetMySPINIapAppInfo"));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vGetMySPINIapAppInfo(rMySPINIapAppInfo);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConfigurationManager::vGetSelectionModeStoredInDatapool()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vGetSelectionModeStoredInDatapool()
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::vGetSelectionModeStoredInDatapool"));
   tenDeviceSelectionMode enStoredSelectionMode = e16DEVICESEL_UNKNOWN;
   if (NULL != m_poStaticConfigurationBase && NULL != m_poDynamicConfigurationBase)
   {
      m_poDynamicConfigurationBase->bReadSelectionMode(enStoredSelectionMode);
      m_poStaticConfigurationBase->vSetSelectionModeStoredInDatapool(enStoredSelectionMode);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConfigurationManager::vGetDeviceTechnologyPreferenceFromDatapool()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vGetDeviceTechnologyPreferenceFromDatapool()
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::vGetDeviceTechnologyPreferenceFromDatapool"));
   tenDeviceCategory enAndroidDevicePreference = e8DEV_TYPE_UNKNOWN;
   tenDeviceCategory enAppleDevicePreference = e8DEV_TYPE_UNKNOWN;
   if (NULL != m_poStaticConfigurationBase && NULL != m_poDynamicConfigurationBase)
   {
      m_poDynamicConfigurationBase->bReadAndroidDeviceTechnologyPreference(enAndroidDevicePreference);
      m_poDynamicConfigurationBase->bReadAppleDeviceTechnologyPreference(enAppleDevicePreference);
      m_poStaticConfigurationBase->vStoreDevicePreferencefromDatapool(enAndroidDevicePreference,
               enAppleDevicePreference);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclConfigurationManager::bSetMySPINEnableSetVal()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bSetMySPINEnableSetVal(tenEnabledInfo enmySPINLinkEnable)
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::bSetMySPINEnableSetVal() entered "));
   t_Bool bRetVal = false;
   if (NULL != m_poDynamicConfigurationBase)
   {
      bRetVal = m_poDynamicConfigurationBase->bSetMySPINEnableSetVal(enmySPINLinkEnable);
   }
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION: tenEnabledInfo spi_tclConfigurationManager::enReadDipoEnableSettingVal()
 ***************************************************************************/
tenEnabledInfo spi_tclConfigurationManager::enReadDipoEnableSettingVal() const
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::enReadDipoEnableSettingVal() entered"));
   tenEnabledInfo enEnabledInfo = e8USAGE_DISABLED;
   if (NULL != m_poDynamicConfigurationBase)
   {
      enEnabledInfo = m_poDynamicConfigurationBase->enReadDipoEnableSettingVal();
   }
   return enEnabledInfo;
}

/***************************************************************************
 ** FUNCTION: tenEnabledInfo spi_tclConfigurationManager::enReadMLEnableSettingVal()
 ***************************************************************************/
tenEnabledInfo spi_tclConfigurationManager::enReadMLEnableSettingVal() const
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::enReadMLEnableSettingVal() entered"));
   tenEnabledInfo enEnabledInfo = e8USAGE_UNKNOWN;
   if (NULL != m_poDynamicConfigurationBase)
   {
      enEnabledInfo = m_poDynamicConfigurationBase->enReadMLEnableSettingVal();
   }
   return enEnabledInfo;
}

/***************************************************************************
 ** FUNCTION: tenEnabledInfo spi_tclConfigurationManager::enReadAAPEnableSettingVal()
 ***************************************************************************/
tenEnabledInfo spi_tclConfigurationManager::enReadAAPEnableSettingVal() const
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::enReadAAPEnableSettingVal() entered"));
   tenEnabledInfo enEnabledInfo = e8USAGE_UNKNOWN;
   if (NULL != m_poDynamicConfigurationBase)
   {
      enEnabledInfo = m_poDynamicConfigurationBase->enReadAAPEnableSettingVal();
   }
   return enEnabledInfo;
}

/***************************************************************************
 ** FUNCTION: tenEnabledInfo spi_tclConfigurationManager::enReadmySPINEnableSettingVal()
 ***************************************************************************/
tenEnabledInfo spi_tclConfigurationManager::enReadmySPINEnableSettingVal() const
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::enReadmySPINEnableSettingVal() entered"));
   tenEnabledInfo enEnabledInfo = e8USAGE_UNKNOWN;
   if (NULL != m_poDynamicConfigurationBase)
   {
      enEnabledInfo = m_poDynamicConfigurationBase->enReadmySPINEnableSettingVal();
   }
   return enEnabledInfo;
}

/***************************************************************************
 ** FUNCTION: tenEnabledInfo spi_tclConfigurationManager::enReadCarlifeEnableSettingVal()
 ***************************************************************************/
tenEnabledInfo spi_tclConfigurationManager::enReadCarlifeEnableSettingVal() const
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::enReadCarlifeEnableSettingVal() entered"));
   tenEnabledInfo enEnabledInfo = e8USAGE_UNKNOWN;
   if (NULL != m_poDynamicConfigurationBase)
   {
      enEnabledInfo = m_poDynamicConfigurationBase->enReadCarlifeEnableSettingVal();
   }
   return enEnabledInfo;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiscovererSettings::bReadAndroidDeviceTechnologyPreferenceVal
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bReadAndroidDeviceTechnologyPreferenceVal(tenDeviceCategory &enTechnologyPreference)
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::bReadAndroidDeviceTechnologyPreferenceVal() entered"));
   t_Bool bRetVal = false;
   if (NULL != m_poDynamicConfigurationBase)
   {
      bRetVal = m_poDynamicConfigurationBase->bReadAndroidDeviceTechnologyPreference(enTechnologyPreference);
   }
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclDiscovererSettings::bReadAppleDeviceTechnologyPreferenceVal
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bReadAppleDeviceTechnologyPreferenceVal(tenDeviceCategory &enTechnologyPreference)
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::bReadAppleDeviceTechnologyPreferenceVal() entered"));
   t_Bool bRetVal = false;
   if (NULL != m_poDynamicConfigurationBase)
   {
      bRetVal = m_poDynamicConfigurationBase->bReadAppleDeviceTechnologyPreference(enTechnologyPreference);
   }
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConfigurationManager::vReadSysConfigs()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vReadSysConfigs()
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::vReadSysConfigs()"));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vReadSysConfigs();
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConfigurationManager::vClearVehicleId()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vClearVehicleId()
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::vClearVehicleId()"));
   if (NULL != m_poDynamicConfigurationBase)
   {
      m_poDynamicConfigurationBase->vClearVehicleId();
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConfigurationManager::vSetScreenConfigs()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vSetScreenConfigs(const trDisplayAttributes& corfrDispLayerAttr)
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::vSetScreenConfigs()"));
   if (NULL != m_poDynamicConfigurationBase)
   {
      m_poDynamicConfigurationBase->vSetScreenConfigs(corfrDispLayerAttr);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConfigurationManager::vSetGeneralRestrictions()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vSetGeneralRestrictions(tenDeviceCategory enDeviceCategory,
         const t_U16 u16GeneralRestrictionInfo)
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::vSetGeneralRestrictions()"));
   if (NULL != m_poDynamicConfigurationBase)
   {
      m_poDynamicConfigurationBase->vSetGeneralRestrictions(enDeviceCategory, u16GeneralRestrictionInfo);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConfigurationManager::vDisplayAudioSettings()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vDisplayAudioSettings()
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::vDisplayAudioSettings()"));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vDisplayAudioSettings();
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConfigurationManager::vDisplayVideoSettings()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vDisplayVideoSettings()
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::vDisplayVideoSettings()"));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vDisplayVideoSettings();
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConfigurationManager::vDisplayAppSettings()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vDisplayAppSettings()
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::vDisplayAppSettings()"));
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vDisplayAppSettings();
   }
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclConfigurationManager::bWriteOnCarEnableSetting()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bWriteOnCarEnableSetting(tenEnabledInfo enOnCarLinkEnable)
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::bWriteOnCarEnableSetting()"));
   t_Bool bRet = false;
   if (NULL != m_poDynamicConfigurationBase)
   {
      bRet = m_poDynamicConfigurationBase->bWriteOnCarEnableSetting(enOnCarLinkEnable);
   }
   return bRet;
}

/***************************************************************************
 ** FUNCTION:  tenEnabledInfo spi_tclConfigurationManager::enReadOnCarEnableSetting()
 ***************************************************************************/
tenEnabledInfo spi_tclConfigurationManager::enReadOnCarEnableSetting() const
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::enReadOnCarEnableSetting()"));
   tenEnabledInfo enEnabledInfo = e8USAGE_UNKNOWN;
   if (NULL != m_poDynamicConfigurationBase)
   {
      enEnabledInfo = m_poDynamicConfigurationBase->enReadOnCarEnableSetting();
   }
   return enEnabledInfo;
}

/***************************************************************************
 ** FUNCTION:  tenEnabledInfo spi_tclConfigurationManager::enGetOnCarEnableSetVal()
 ***************************************************************************/
tenEnabledInfo spi_tclConfigurationManager::enGetOnCarEnableSetVal() const
{
   tenEnabledInfo enOnCarEnableSetVal = e8USAGE_UNKNOWN;
   if (NULL != m_poStaticConfigurationBase)
   {
      enOnCarEnableSetVal = m_poStaticConfigurationBase->enGetOnCarEnableSetVal();
   }
   ETG_TRACE_USR2(("[DESC]: spi_tclConfigurationManager :: OnCar enable settings default value %u ", ETG_ENUM(tenEnabledInfo,
            enOnCarEnableSetVal)));
   return enOnCarEnableSetVal;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclConfigurationManager::bGetITCommanderSupport()
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetITCommanderSupport()
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::bGetITCommanderSupport()"));
   t_Bool bRet = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bRet = m_poStaticConfigurationBase->bGetITCommanderSupport();
   }
   return bRet;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConfigurationManager::vGetVehicleBTAddress()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vGetVehicleBTAddress(t_String &szVehicleBTAddress)
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::vGetVehicleBTAddress()"));
   szVehicleBTAddress = "";
   if (NULL != m_poDynamicConfigurationBase)
   {
#ifdef VARIANT_S_FTR_ENABLE_G4G
      szVehicleBTAddress = "c0fbee2e7118";
#else
      m_poDynamicConfigurationBase->vGetVehicleBTAddress(szVehicleBTAddress);
#endif
   }
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConfigurationManager::vOnVehicleBTAddress()
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vOnVehicleBTAddress(t_String &szVehicleBTAddress)
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::vOnVehicleBTAddress()"));
   if (NULL != m_poDynamicConfigurationBase)
   {
      m_poDynamicConfigurationBase->vOnVehicleBTAddress(szVehicleBTAddress);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclConfigurationManager::bGetSysVehicleInfo(..
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetSysVehicleInfo(t_U8 &u8RegionCode, t_U8 &u8VendorCode, t_U16 &u16VehicleCode)
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::vOnVehicleBTAddress()"));
   t_Bool bRet = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bRet = m_poStaticConfigurationBase->bGetSysVehicleInfo(u8RegionCode, u8VendorCode, u16VehicleCode);
   }
   return bRet;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclConfigurationManager::bGetCarlifeIOSFeatureSupport
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetCarlifeIOSFeatureSupport()
{
   t_Bool bIsCarlifeIOSEnabled = false;

   if (NULL != m_poStaticConfigurationBase)
   {
      bIsCarlifeIOSEnabled = m_poStaticConfigurationBase->bGetCarlifeIOSFeatureSupport();
      ETG_TRACE_USR2(("spi_tclConfigurationManager::bGetCarlifeIOSFeatureSupport() IsCarlifeIOSEnabled = %d ", bIsCarlifeIOSEnabled));
   }
   return bIsCarlifeIOSEnabled;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclConfigurationManager::u8ReadOEMBrandFromKDS
 ***************************************************************************/
t_U8 spi_tclConfigurationManager::u8ReadOEMBrandFromKDS()
{
   t_U8 u8OEMBrand = 0xff;
   if (NULL != m_poStaticConfigurationBase)
   {
      u8OEMBrand = m_poStaticConfigurationBase->u8ReadOEMBrandFromKDS();
      ETG_TRACE_USR2(("spi_tclConfigurationManager::u8ReadOEMBrandFromKDS()8OEMBrand= %d ", u8OEMBrand));
   }
   return u8OEMBrand;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclConfigurationManager::vGetVehicleInfoDataMySPIN
 ***************************************************************************/
t_Void spi_tclConfigurationManager::vGetVehicleInfoDataMySPIN(trVehicleInfo& rfrVehicleInfo)
{
   if (NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vGetVehicleInfoDataMySPIN(rfrVehicleInfo);
      ETG_TRACE_USR2(("spi_tclConfigurationManager::vGetVehicleInfoDataMySPIN() vehicle manufacturer name is [%s]", rfrVehicleInfo.szManufacturer.c_str()));
      ETG_TRACE_USR2(("spi_tclConfigurationManager::vGetVehicleInfoDataMySPIN() vehicle manufacturer Scope is [%s]", rfrVehicleInfo.szVehicleVariant.c_str()));
   }
}

/***************************************************************************
 ** FUNCTION:  t_U16 spi_tclConfigurationManager::vGetVehicleInfoDataMySPIN
 ***************************************************************************/
t_U16 spi_tclConfigurationManager::u16ReadVehicleTypeFromKDS()
{
   t_U16 u16AllianceVehicleType = 0xffff;
   if (NULL != m_poStaticConfigurationBase)
   {
      u16AllianceVehicleType = m_poStaticConfigurationBase->u16ReadVehicleTypeFromKDS();
   }
   return u16AllianceVehicleType;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclConfigurationManager::bIsDeviceWhitelisted
 ***************************************************************************/
t_Bool spi_tclConfigurationManager::bIsDeviceWhitelisted(t_U32 u32ProductID, t_U32 u32VendorID,
         tenDeviceCategory enReqDevCat)
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::bIsDeviceWhitelisted() entered "));
   t_Bool bRet = false;
   if (NULL != m_poStaticConfigurationBase)
   {
      bRet = m_poStaticConfigurationBase->bIsDeviceWhitelisted(u32ProductID, u32VendorID, enReqDevCat);
   }
   return bRet;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclConfigurationManager::vRegisterKeyCodes
***************************************************************************/
t_Void spi_tclConfigurationManager::vRegisterKeyCodes(const std::map<tenDeviceCategory,std::set<tenKeyCode>>& corfrmapTechSuppHardkeys,
                                                      const std::map<tenDeviceCategory,std::set<tenKeyCode>>& corfrmapTechSuppSoftkeys)
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::vRegisterKeyCodes()"));
   if(NULL != m_poDynamicConfigurationBase )
   {
           m_poDynamicConfigurationBase->vRegisterKeyCodes(corfrmapTechSuppHardkeys,corfrmapTechSuppSoftkeys);
   }
}
/***************************************************************************
** FUNCTION:  t_Bool spi_tclConfigurationManager::bGetKeyInfo
***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetKeyInfo(std::set<tenKeyCode>& rfrsetKeyCodes,tenDeviceCategory enCategory)
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::bGetKeyInfo()"));
   t_Bool bretVal = false;
   if(NULL != m_poDynamicConfigurationBase )
   {
       bretVal = m_poDynamicConfigurationBase->bGetKeyInfo(rfrsetKeyCodes,enCategory);
       // On First time booting or on factory reset default values to be set
       if(false == bretVal || (rfrsetKeyCodes.empty()))
       {
          if (m_mapTechSuppHardKeys.find(enCategory) != m_mapTechSuppHardKeys.end())
          {
             rfrsetKeyCodes = m_mapTechSuppHardKeys[enCategory];
             bretVal = true;
          }
       }
   }
   return bretVal;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclConfigurationManager::vGetSoftKeyInfo()
***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetSoftKeyInfo(std::set<tenKeyCode>& rfrsetKeyCodes,tenDeviceCategory enCategory)
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::bGetSoftKeyInfo()"));
   t_Bool bretVal = false;
   if(NULL != m_poDynamicConfigurationBase )
   {
       bretVal = m_poDynamicConfigurationBase->bGetSoftKeyInfo(rfrsetKeyCodes,enCategory);
       if(false == bretVal)
       {
          if (m_mapTechSuppSoftKeys.find(enCategory) != m_mapTechSuppSoftKeys.end())
          {
             rfrsetKeyCodes = m_mapTechSuppSoftKeys[enCategory];
             bretVal = true;
          }
       }
   }
   return bretVal;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclConfigurationManager::vSetDefaultKeyInfo
***************************************************************************/
t_Void spi_tclConfigurationManager::vSetDefaultKeyInfo()
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::vSetDefaultKeyInfo()"));

   t_Bool bAAretVal = false;
   t_Bool bDiPoretVal = false;
   std::set<tenKeyCode> setAAKeyCodes,setDiPoKeyCodes;
   std::map<tenDeviceCategory, std::set<tenKeyCode>> mapTechSuppHardKeys;
   std::map<tenDeviceCategory, std::set<tenKeyCode>> mapTechSuppSoftKeys;

   if(NULL != m_poDynamicConfigurationBase)
   {
      bAAretVal   = m_poDynamicConfigurationBase->bGetKeyInfo(setAAKeyCodes,e8DEV_TYPE_ANDROIDAUTO);
      bDiPoretVal = m_poDynamicConfigurationBase->bGetKeyInfo(setDiPoKeyCodes,e8DEV_TYPE_DIPO);
      if(m_mapTechSuppHardKeys.empty() || m_mapTechSuppSoftKeys.empty())
      {
         vInsertDefKeystoMap();
      }
      if(false == bAAretVal)
      {
         mapTechSuppHardKeys[e8DEV_TYPE_ANDROIDAUTO] = m_mapTechSuppHardKeys[e8DEV_TYPE_ANDROIDAUTO];
      }
      if(false == bDiPoretVal)
      {
         mapTechSuppHardKeys[e8DEV_TYPE_DIPO] = m_mapTechSuppHardKeys[e8DEV_TYPE_DIPO];
      }
   }
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclConfigurationManager::vInsertDefKeystoMap
***************************************************************************/
t_Void spi_tclConfigurationManager::vInsertDefKeystoMap()
{
   ETG_TRACE_USR1(("spi_tclConfigurationManager::vInsertDefKeystoMap()"));
   if(NULL != m_poStaticConfigurationBase)
   {
      m_poStaticConfigurationBase->vGetDefaultKeys(m_mapTechSuppHardKeys);
   }
}

/*********************************************************************
** FUNCTION:  t_Bool spi_tclConfigurationManager::bGetCarlifeGPSDataSupportflag
***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetCarlifeGPSDataSupportflag() const
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::bGetCarlifeGPSDataSupportflag() entered "));
   t_Bool bRet = false;
   if(NULL != m_poStaticConfigurationBase)
   {
      bRet = m_poStaticConfigurationBase->bGetCarlifeGPSDataSupportflag();
   }
   return bRet;
}

/*********************************************************************
** FUNCTION:  t_Bool spi_tclConfigurationManager::bGetCarPlayGPSDataSupportflag
***************************************************************************/
t_Bool spi_tclConfigurationManager::bGetCarPlayGPSDataSupportflag()
{
   ETG_TRACE_USR1((" spi_tclConfigurationManager::bGetCarPlayGPSDataSupportflag() entered "));
   t_Bool bRet = true;
   if(NULL != m_poStaticConfigurationBase)
   {
      bRet = m_poStaticConfigurationBase->bGetCarPlayGPSDataSupportflag();
   }
   return bRet;
}
