/*!
 *******************************************************************************
 * \file              spi_tclConfigurationManager.h
 * \brief             Configuration manager for SPI
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Configuration manager for SPI
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 07.06.2018 |  Kavya Mogeri                | Initial Version

 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/

#ifndef SPI_TCLCONFIGURATIONMANAGER_
#define SPI_TCLCONFIGURATIONMANAGER_

#include "SPITypes.h"
#include "BaseTypes.h"
#include "spi_tclConnSettingsIntf.h"
#include "spi_tclDataServiceSettingsIntf.h"
#include "spi_tclInputHandlerSettingsIntf.h"
#include "spi_tclAppSettingsIntf.h"
#include "spi_tclAudioSettingsIntf.h"
#include "spi_tclBluetoothSettingsIntf.h"
#include "spi_tclResourceMngrSettingsIntf.h"
#include "spi_tclDefaultSettingsIntf.h"
#include "spi_tclVideoSettingsIntf.h"
#include "spi_tclWiFiSettingsIntf.h"
#include "spi_tclDynamicConfigurationBase.h"
#include "spi_ConnMngrTypeDefines.h"
#include "spi_tclDiscovererSettingsIntf.h"
#include "spi_tclStaticConfigurationBaseIntf.h"

class spi_tclConfigurationManager: public spi_tclConnSettingsIntf,
         public spi_tclDataServiceSettingsIntf,
         public spi_tclInputHandlerSettingsIntf,
         public spi_tclAppSettingsIntf,
         public spi_tclAudioSettingsIntf,
         public spi_tclBluetoothSettingsIntf,
         public spi_tclResourceMngrSettingsIntf,
         public spi_tclVideoSettingsIntf,
         public spi_tclWiFiSettingsIntf,
         public spi_tclDiscovererSettingsIntf,
         public spi_tclDefaultSettingsIntf
{
   public:
      /***************************************************************************
       *********************************PUBLIC*************************************
       ***************************************************************************/

      /***************************************************************************
       ** FUNCTION:  spi_tclConfigurationManager()
       ***************************************************************************/
      /*!
       * \fn      spi_tclConfigurationManager()
       * \brief   Constructor
       * \sa      spi_tclConfigurationManager()
       **************************************************************************/
      spi_tclConfigurationManager(spi_tclStaticConfigurationBaseIntf* poStaticConfiguration,
               spi_tclDynamicConfigurationBase* poDynamicConfiguration);

      /***************************************************************************
       ** FUNCTION:  ~spi_tclConfigurationManager()
       ***************************************************************************/
      /*!
       * \fn      ~spi_tclConfigurationManager()
       * \brief   Destructor
       * \sa      spi_tclConfigurationManager()
       **************************************************************************/
      ~spi_tclConfigurationManager();

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::bInitialize()
       ***************************************************************************/
      /*!
       * \fn      bInitialize()
       * \brief   Method to Initialize
       * \sa      bUnInitialize()
       **************************************************************************/
      t_Bool bInitialize();

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::bInitialize()
       ***************************************************************************/
      /*!
       * \fn      bInitialize()
       * \brief   Method to Uninitialize
       * \sa      bUnInitialize()
       **************************************************************************/
      t_Void vUninitialize();

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::bGetNonCertAppsReq()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetNonCertAppsReq()
       * \brief  To get whether the Non certified Applications are required
       *          when the application certification is enabled for the
       *          particular version of the device.
       * \retval t_Bool
       **************************************************************************/
      t_Bool bGetNonCertAppsReq() const override;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetAAPNavigationEnabledInfo()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetAAPNavigationEnabledInfo(t_Bool bEnable)
       * \brief  method to get the navigation feature enable settings for AAP.
       * \retval t_Bool
       **************************************************************************/
      t_Bool bGetAAPNavigationEnabledInfo() const override;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetAAPMediaPlaybackEnabledInfo()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetAAPMediaPlaybackEnabledInfo(t_Bool bEnable)
       * \brief  method to get the media playback feature enable settings for AAP.
       * \retval t_Bool
       **************************************************************************/
      t_Bool bGetAAPMediaPlaybackEnabledInfo() const override;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetAAPNotificationUpdatesEnabledInfo()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetAAPNotificationUpdatesEnabledInfo(t_Bool bEnable)
       * \brief  method to get notification update enable settings for AAP.
       * \retval t_Bool
       **************************************************************************/
      t_Bool bGetAAPNotificationUpdatesEnabledInfo() const override;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclConfigurationManager::bGetTBTUpdatesEnabledInfo()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetTBTUpdatesEnabledInfo()
       * \brief  Provides information on whether TBT updates are enabled or disabled.
       * \retval t_Bool TRUE- Enabled FALSE-Disabled
       **************************************************************************/
      t_Bool bGetTBTUpdatesEnabledInfo() const override;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetAssisstantGuideUpdatesEnabledInfo()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetAssisstantGuideUpdatesEnabledInfo()
       * \brief  Provides information on whether Assisstant Guide updates are enabled
       *         or disabled.
       * \retval t_Bool TRUE- Enabled FALSE-Disabled
       **************************************************************************/
      t_Bool bGetAssisstantGuideUpdatesEnabledInfo() const override;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclConfigurationManager::vSetMLNotificationOnOff()
       ***************************************************************************/
      /*!
       * \fn     t_Void vSetMLNotificationOnOff(t_Bool bSetNotificationsOn)
       * \brief  To Set the Notifications to On/Off
       * \param  bSetNotificationsOn : [IN] True-Set Notifications to ON
       *                                    False - Set Notifications to OFF
       * \retval t_Void
       **************************************************************************/
      t_Void vSetMLNotificationOnOff(t_Bool bSetNotificationsOn) override;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetMLNotificationEnabledInfo()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetMLNotificationEnabledInfo()
       * \brief  Provides information on whether ML Notifications are enabled or
       *         disabled.
       * \retval t_Bool  TRUE- Enabled FALSE-Disabled
       **************************************************************************/
      t_Bool bGetMLNotificationEnabledInfo() const override;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclConfigurationManager::vSetRegion(.)
       ***************************************************************************/
      /*!
       * \fn     t_Void vSetRegion(tenRegion enRegion)
       * \brief  Interface to set the region for application certification.
       *         It gives the info of which region CCC Guidelines should be followed
       *         for the Application Certification Filtering
       * \param  [IN] enRegion : Region enumeration
       * \sa
       **************************************************************************/
      t_Void vSetRegion(tenRegion enRegion) override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vReadSPIPolicySettings()
       ***************************************************************************/
      /*!
       * \fn     t_Void vReadSPIPolicySettings()
       * \brief  To read the SPI settings from policy XML
       * \retval t_Void
       **************************************************************************/
      t_Void vReadSPIPolicySettings();

      /***************************************************************************
       ** FUNCTION:  t_String spi_tclConfigurationManager::szGetRegion()
       ***************************************************************************/
      /*!
       * \fn     t_String szGetRegion()
       * \brief  To Get the which regions CCC guidelines should be followed for the
       *          Application certification.
       * \retval t_String
       **************************************************************************/
      t_String szGetRegion() const override;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetNonCertAppAllowedInfo()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetNonCertAppAllowedInfo()
       * \brief  Provides information on whether non certified ML apps are allowed
       *         in all modes(drive/park)
       * \retval t_Bool TRUE- Allowed FALSE-Not allowed
       **************************************************************************/
      t_Bool bGetNonCertAppAllowedInfo() const override;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclConfigurationManager::bEnableCTSTest()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bEnableCTSTest()
       * \brief  To get whether the CTS test is enabled.
       * \retval t_Bool  TRUE- Enabled FALSE-Disabled
       **************************************************************************/
      t_Bool bEnableCTSTest() override;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclConfigurationManager::vGetRTPPayloadType
       (t_U32& rfu32RTPPayload)
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetRTPPayloadType(t_U32& u32RTPPayload)
       * \brief   Method to get the RTPPayload type
       * \param   [rfu32RTPPayload]: Reference to the RTP Payload type
       * \retval  NONE
       **************************************************************************/
      t_Void vGetRTPPayloadType(t_U32& rfu32RTPPayload) const override;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclConfigurationManager::vGetAudioIPL
       (tU32& rfu32AudioIPL)
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetAudioIPL(tU32& rfu32AudioIPL)
       * \brief   Method to get the Audio IPL
       * \param   [rfu32AudioIPL]: Reference to the Audio IPL
       * \retval  t_Bool value: TRUE if Setting is Valid, FALSE Otherwise
       **************************************************************************/
      t_Void vGetAudioIPL(t_U32& rfu32AudioIPL) const override;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetAudioBlockingTriggerBasedOnCat()
       ***************************************************************************/
      /*!
       * \fn      t_Bool bGetAudBlockTriggerBasedOnCat()
       * \brief   Method to get the Audio Blocking enable/disable trigger based on
       *          Application category or not. This feature works only in conjunction with the
       *          Dynamic Audio feature
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bGetAudBlockTriggerBasedOnCat() const override;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclConfigurationManager::vSetAudBlockTriggerBasedOnCat()
       ***************************************************************************/
      /*!
       * \fn      vSetAudBlockTriggerBasedOnCat()
       * \brief   Method to Set the Audio Blocking enable/disable trigger based on
       *          Application category or not. This feature works only in conjunction with the
       *          Dynamic Audio feature
       * \retval  t_Void
       **************************************************************************/
      t_Void vSetAudBlockTriggerBasedOnCat(t_Bool bIsML11OrAboveDevice) override;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetAudBlockTriggerBasedOnGlobaleMute()
       ***************************************************************************/
      /*!
       * \fn      t_Bool bGetAudBlockTriggerBasedOnGlobaleMute()
       * \brief   Method to get the Audio Blocking enable/disable trigger based on
       *          Global mute is enabled or not
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bGetAudBlockTriggerBasedOnGlobaleMute() const override;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclConfigurationManager::vGetAudioPipeConfig()
       ***************************************************************************/
      /*!
       * \fn     t_Void vGetAudioPipeConfig()
       * \brief  provides audio configuration for the specified audio steam
       * \param  [OUT]rfrmapAudioPipeConfig : returns audio configuration as string
       **************************************************************************/
      t_Void vGetAudioPipeConfig(tmapAudioPipeConfig &rfmapAudioPipeConfig) override;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetMLDynAudioSupport()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetMLDynAudioSupport() const
       * \brief  To Get whether ML Audio is supported or not.
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bGetMLDynAudioSupport() const override;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclConfigurationManager::vSetMLDynAudioSupport()
       ***************************************************************************/
      /*!
       * \fn     vSetMLDynAudioSupport() const
       * \brief  To Set whether ML Audio is supported or not.
       *         (For ML 1.0 = Not Supported, For Ml 1.1 onwards = supported)
       * \retval  t_Void
       **************************************************************************/
      t_Void vSetMLDynAudioSupport(t_Bool bIsML11OrAboveDevice) override;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetAAPMediaStreamRecEnabled()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetAAPMediaStreamRecEnabled() const
       * \brief  To Get whether AAP Media audio stream recording is enabled or not.
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bGetAAPMediaStreamRecEnabled() const override;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetAAPGuidanceStreamRecEnabled()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetAAPGuidanceStreamRecEnabled() const
       * \brief  To Get whether AAP Guidance audio stream recording is enabled or not.
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bGetAAPGuidanceStreamRecEnabled() const override;

      /***************************************************************************
       ** FUNCTION: t_String spi_tclConfigurationManager::szGetAudSinkMaxUnackedFrames()
       ***************************************************************************/
      /*!
       * \fn     szGetAudSinkMaxUnackedFrames() const
       * \brief  To Get "max-unacked-frames" value for Audio Sink configuration
       * \retval  t_String
       **************************************************************************/
      t_String szGetAudSinkMaxUnackedFrames() const override;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetRTPInSupport()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetRTPInSupport() const
       * \brief  To Get whether RTPIn support is enabled or not.
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bGetRTPInSupport() const override;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetMySIPNAdvAudioSupport()
       ***************************************************************************/
      /*!
       * \fn     bGetMySIPNAdvAudioSupport() const
       * \brief  To Get support for mySPIN Advanced Audio
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bGetMySIPNAdvAudioSupport() const override;

      /***************************************************************************
       ** FUNCTION: t_String spi_tclConfigurationManager::szGetDisableRealTimePriorityAudio()
       ***************************************************************************/
      /*!
       * \fn     szGetDisableRealTimePriorityAudio() const
       * \brief  To Get the value of disabling real time priority audio for Audio Sink configuration
       * \retval  t_String
       **************************************************************************/
      t_String szGetDisableRealTimePriorityAudio() const override;

      /***************************************************************************
       ** FUNCTION: t_String spi_tclConfigurationManager::szGetAudThreadsRealTimePriority()
       ***************************************************************************/
      /*!
       * \fn     szGetAudThreadsRealTimePriority() const
       * \brief  To Get Audio threads Real time priority value for Audio Sink configuration
       * \retval  t_String
       **************************************************************************/
      t_String szGetAudThreadsRealTimePriority() const override;

      /***************************************************************************
       ** FUNCTION: t_String spi_tclConfigurationManager::szGetAudSinkThresholdMs()
       ***************************************************************************/
      /*!
       * \fn     szGetAudSinkThresholdMs() const
       * \brief  To Get Audio Sink Threshold configuration
       * \retval  t_String
       **************************************************************************/
      t_String szGetAudSinkThresholdMs() const override;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclConfigurationManager::vGetBdclMediaSinkConfig()
       ***************************************************************************/
      /*!
       * \fn     vGetBdclMediaSinkConfig() const
       * \brief  To Get BDCL Media Audio Sink configuration
       * \retval  t_Void
       **************************************************************************/
      t_Void vGetBdclMediaSinkConfig(std::map<t_String, t_String>& mapAudioConfig) const override;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclConfigurationManager::vGetBdclTTSSinkConfig()
       ***************************************************************************/
      /*!
       * \fn     vGetBdclTTSSinkConfig() const
       * \brief  To Get BDCL TTS Audio Sink configuration
       * \retval  t_Void
       **************************************************************************/
      t_Void vGetBdclTTSSinkConfig(std::map<t_String, t_String>& mapAudioConfig) const override;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclConfigurationManager::vGetBdclVRSinkConfig()
       ***************************************************************************/
      /*!
       * \fn     vGetBdclVRSinkConfig() const
       * \brief  To Get BDCL VR Audio Sink configuration
       * \retval  t_Void
       **************************************************************************/
      t_Void vGetBdclVRSinkConfig(std::map<t_String, t_String>& mapAudioConfig) const override;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclConfigurationManager::vGetBdclVRSourceConfig()
       ***************************************************************************/
      /*!
       * \fn     vGetBdclVRSourceConfig() const
       * \brief  To Get BDCL VR Audio Source configuration
       * \retval  t_Void
       **************************************************************************/
      t_Void vGetBdclVRSourceConfig(std::map<t_String, t_String>& mapAudioConfig) const override;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetLastModeSupport()
       ***************************************************************************/
      /*!
       * \fn     bGetLastModeSupport() const
       * \brief  To Get last mode support configuration for a device category
       * \param  enDevCat [IN]: Device category
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bGetLastModeSupport(tenDeviceCategory enDevCat) const override;

      /***************************************************************************
       ** FUNCTION: tenLastModeSupport spi_tclConfigurationManager::enGetLastModeSupport()
       ***************************************************************************/
      /*!
       * \fn     enGetLastModeSupport() const
       * \brief  To Get last mode support configuration
       * \retval  tenLastModeSupport

       **************************************************************************/
      tenLastModeSupport enGetLastModeSupport() const override;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetAudioMuteEnabled()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetAudioMuteEnabled() const
       * \brief  To Get whether audio mute is enabled (during last mode audio)
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bGetAudioMuteEnabled() const override;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetVolumeLockEnabled()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetVolumeLockEnabled() const
       * \brief  To Get whether Volume lock is enabled (during last mode audio)
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bGetVolumeLockEnabled() const override;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetMLDynAudioApplicability()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetMLDynAudioApplicability() const
       * \brief  To Get whether ML Dnamic Audio is applicable or not from policy.
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bGetMLDynAudioApplicability() const override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vClearPrivateData
       ***************************************************************************/
      /*!
       * \fn     vClearPrivateData
       * \brief  Clears device history persistently stored on target and other private data
       **************************************************************************/
      t_Void vClearPrivateData() override;

      /***************************************************************************
       ** FUNCTION:  tenDeviceSelectionMode spi_tclConfigurationManager::enGetDeviceSelectionMode
       ***************************************************************************/
      /*!
       * \fn     tenDeviceSelectionMode enGetDeviceSelectionMode()
       * \brief  returns Device Selection Mode
       **************************************************************************/
      tenDeviceSelectionMode enGetDeviceSelectionMode(tenDeviceCategory enTechnologyType) const override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vSetDeviceSelectionMode()
       ***************************************************************************/
      /*!
       * \fn      t_Void vSetDeviceSelectionMode(
       * \brief   Method to set the device selection mode to automatic/manual. Changes
       *          will take into effect on successive connection
       * \param   enSelectionMode : Device selection mode @see tenDeviceSelectionMode
       * \retval  t_Void
       **************************************************************************/
      t_Void vSetDeviceSelectionMode(tenDeviceSelectionMode enSelectionMode, tenDeviceCategory enTechnologyType)
               override;

      /***************************************************************************
       ** FUNCTION:  tenDeviceCategory spi_tclConfigurationManager::enGetTechnologyPreference
       ***************************************************************************/
      /*!
       * \fn     tenDeviceCategory enGetTechnologyPreference()
       * \brief  returns Device type preference
       **************************************************************************/
      tenDeviceCategory enGetTechnologyPreference(tenDeviceType enDeviceType) const override;

      /***************************************************************************
       ** FUNCTION:  tenEnabledInfo spi_tclConfigurationManager::enGetCertificateType
       ***************************************************************************/
      /*!
       * \fn     tenEnabledInfo enGetCertificateType()
       * \brief  returns Certificate type
       **************************************************************************/
      tenCertificateType enGetCertificateType() const override;

      /***************************************************************************
       ** FUNCTION:  tenSelModePriority spi_tclConfigurationManager::enGetSelectionModePriority
       ***************************************************************************/
      /*!
       * \fn     tenSelModePriority enGetSelectionModePriority()
       * \brief  returns Priority of selection mode
       **************************************************************************/
      tenSelModePriority enGetSelectionModePriority() const override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vGetPersistentStoragePath
       ***************************************************************************/
      /*!
       * \fn     t_Void vGetPersistentStoragePath()
       * \brief  returns path for persistent Storage
       **************************************************************************/
      t_Void vGetPersistentStoragePath(t_String &rfszPersStoragePath) override;

      /***************************************************************************
       ** FUNCTION:  t_U32 spi_tclConfigurationManager::u32GetDeviceHistorySize
       ***************************************************************************/
      /*!
       * \fn     t_U32 u32GetDeviceHistorySize()
       * \brief  returns the Maximum size of Device History Database
       **************************************************************************/
      t_U32 u32GetDeviceHistorySize() const override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::bIsXMLValidationEnabled
       ***************************************************************************/
      /*!
       * \fn     t_Bool bIsXMLValidationEnabled()
       * \brief  Method to check whether the XML validation is enabled or not
       * \retval t_Bool
       **************************************************************************/
      t_Bool bIsXMLValidationEnabled() const override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::vGetHeadUnitInfo
       ***************************************************************************/
      /*!
       * \fn     t_Bool vGetHeadUnitInfo()
       * \brief  Method to retrieve head unit information
       * \retval t_Bool
       **************************************************************************/
      t_Void vGetHeadUnitInfo(trHeadUnitInfo &rfrHeadUnitInfo) override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vGetDefaultProjUsageSettings
       ***************************************************************************/
      /*!
       * \fn     vGetDefaultProjUsageSettings()
       * \brief  Method to retrieve default setting for projection usage
       * \param [OUT] : returns the current value of device projection enable
       * \enSPIType  : indicates the type of SPI technology. e8DEV_TYPE_UNKNOWN indicates default value for any SPI technology
       * \retval t_Void
       **************************************************************************/
      t_Void vGetDefaultProjUsageSettings(tenEnabledInfo &enEnabledInfo, tenDeviceCategory enSPIType) override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vGetHeadUnitInfo
       ***************************************************************************/
      /*!
       * \fn     vGetHeadUnitInfo()
       * \brief  Method to retrieve head unit information
       * \retval t_Void
       **************************************************************************/
      t_Void vGetMySPINIapAppInfo(trMySPINIapAppInfo& rMySPINIapAppInfo) override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vGetMySPINAoapAppInfo
       ***************************************************************************/
      /*!
       * \fn     vGetMySPINAoapAppInfo(trMySPINAoapAppInfo& rfrMySPINAoapAppInfo)
       * \brief  Method to retrieve head unit information
       * \param  rfrMySPINAoapAppInfo:[IN]Aoap application information
       * \retval t_Void
       **************************************************************************/
      t_Void vGetMySPINAoapAppInfo(trMySPINAoapAppInfo& rfrMySPINAoapAppInfo) override;

      /***************************************************************************
       ** FUNCTION:  t_U16 spi_tclConfigurationManager::u16GetGeneralRestrictionsInfo
       ***************************************************************************/
      /*!
       * \fn     t_U16 u16GetGeneralRestrictionsInfo(t_String &rfszDeviceExclusionListPath)
       * \brief  Method to get General restriction info according to the dev category
       * \param  enDeviceCategory : Device category
       * \retval t_U16
       **************************************************************************/
      t_U16 u16GetGeneralRestrictionsInfo(tenDeviceCategory enDeviceCategory) const override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::bGetCdbGPSSeviceEnabled()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetCdbGPSSeviceEnabled()
       * \brief  Interface to read whether GPS CDB service is enabled
       * \retval t_Bool
       **************************************************************************/
      t_Bool bGetCdbGPSSeviceEnabled() const override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::bGetCdbLocationSeviceEnabled()
       ***************************************************************************/
      /*!
       * \fn     bGetCdbLocationSeviceEnabled()
       * \brief  Interface to read whether Location CDB service is enabled
       * \retval t_Bool
       **************************************************************************/
      t_Bool bGetCdbLocationSeviceEnabled() const override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::bGetCdbPrivateSeviceEnabled()
       ***************************************************************************/
      /*!
       * \fn     bGetCdbPrivateSeviceEnabled()
       * \brief  Interface to read whether Private CDB service is enabled
       * \retval t_Bool
       **************************************************************************/
      t_Bool bGetCdbPrivateSeviceEnabled() const override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::bGetCdbMediaMetadataSinkEnabled()
       ***************************************************************************/
      /*!
       * \fn     bGetCdbMediaMetadataSinkEnabled()
       * \brief  Interface to read whether CDB Sink Media Metadata sink is enabled
       * \retval t_Bool
       **************************************************************************/
      t_Bool bGetCdbMediaMetadataSinkEnabled() const override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::bGetCdbMediaMetadataSeviceEnabled()
       ***************************************************************************/
      /*!
       * \fn     bGetCdbNaviMetadataSinkEnabled()
       * \brief  Interface to read whether CDB Navigation Metadata sink is enabled
       * \retval t_Bool
       **************************************************************************/
      t_Bool bGetCdbNaviMetadataSinkEnabled() const override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::bGetEnvDataSubEnabled()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetEnvDataSubEnabled() const
       * \brief  Interface to get whether to subscribe for Environment data or not
       * \retval t_Bool
       **************************************************************************/
      t_Bool bGetEnvDataSubEnabled() const override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::bGetGearStatusEnabled()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetGearStatusEnabled() const
       * \brief  Interface to get whether to subscribe for Gear Status data or not
       * \retval t_Bool
       **************************************************************************/
      t_Bool bGetGearStatusEnabled() const override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::bGetAccelerometerDataEnabled()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetAccelerometerDataDisable() const
       * \brief  Interface to get whether to subscribe for Accelerometer data or not
       * \retval t_Bool
       **************************************************************************/
      t_Bool bGetAccelerometerDataEnabled() const override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::bGetGyroDataEnabled()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetGyroDataEnabled() const
       * \brief  Interface to get whether to subscribe for Gyroscope data or not
       * \retval t_Bool
       **************************************************************************/
      t_Bool bGetGyroDataEnabled() const override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::bGetParkBrakeDataEnabled()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetParkBrakeDataEnabled() const
       * \brief  Interface to get whether to subscribe for park brake data or not
       * \retval t_Bool
       **************************************************************************/
      t_Bool bGetParkBrakeDataEnabled() const override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::bIsBochNavigation()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bIsBochNavigation() const
       * \brief  Interface to get the type of navigation used.
       * \retval t_Bool
       **************************************************************************/
      t_Bool bIsBochNavigation() const override;

      /***************************************************************************
       ** FUNCTION:  t_U32 spi_tclConfigurationManager:: u32GetSpeedTimerInterval()
       ***************************************************************************/
      /*!
       * \fn     u32GetSpeedTimerInterval() const
       * \brief  Interface to get the Speed timer interval in ms.
       * \retval t_U32
       **************************************************************************/
      t_U32 u32GetSpeedTimerInterval() const override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::bGetLocDataEnabled()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetLocDataEnabled() const
       * \brief  Interface to get location data enabled setting
       * \retval t_Bool
       **************************************************************************/
      t_Bool bGetLocDataEnabled() const override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::bGetMLNotiSettingVal()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetMLNotiSettingVal()
       * \brief  To get the default ML Notification Setting Value
       * \retval t_Bool
       *************************************************************************/
      t_Bool bGetMLNotiSettingVal() const override;

      /***************************************************************************
       ** FUNCTION:  tenEnabledInfo spi_tclConfigurationManager::enGetMLLinkEnableSetVal()
       ***************************************************************************/
      /*!
       * \fn     tenEnabledInfo enGetMLLinkEnableSetVal()
       * \brief  To get the default ML Link Enable Setting Value
       * \retval tenEnabledInfo
       **************************************************************************/
      tenEnabledInfo enGetMLLinkEnableSetVal() const override;

      /***************************************************************************
       ** FUNCTION:  tenEnabledInfo spi_tclConfigurationManager::enGetDipoEnableSetVal()
       ***************************************************************************/
      /*!
       * \fn     tenEnabledInfo enGetDipoEnableSetVal()
       * \brief  To get the default Dipo Enable Setting Value
       * \retval tenEnabledInfo
       **************************************************************************/
      tenEnabledInfo enGetDipoEnableSetVal() const override;

      /***************************************************************************
       ** FUNCTION:  tenEnabledInfo spi_tclConfigurationManager::enGetAAPEnableSetVal()
       ***************************************************************************/
      /*!
       * \fn     tenEnabledInfo enGetAAPEnableSetVal()
       * \brief  To get the default AAP Enable Setting Value
       * \retval tenEnabledInfo
       **************************************************************************/
      tenEnabledInfo enGetAAPEnableSetVal() const override;

      /***************************************************************************
       ** FUNCTION:  tenEnabledInfo spi_tclConfigurationManager::enGetCarlifeEnableSetVal()
       ***************************************************************************/
      /*!
       * \fn     tenEnabledInfo enGetCarlifeEnableSetVal()
       * \brief  To get the default Carlife Enable Setting Value
       * \retval tenEnabledInfo
       **************************************************************************/
      tenEnabledInfo enGetCarlifeEnableSetVal() const override;

      /***************************************************************************
       ** FUNCTION:  tenEnabledInfo spi_tclConfigurationManager::enGetMySPINEnableSetVal()
       ***************************************************************************/
      /*!
       * \fn     tenEnabledInfo enGetMySPINEnableSetVal()
       * \brief  To get the default MySPIN Enable Setting Value
       * \retval tenEnabledInfo
       **************************************************************************/
      tenEnabledInfo enGetMySPINEnableSetVal() const override;

      /***************************************************************************
       ** FUNCTION:  tenDriveSideInfo  spi_tclConfigurationManager::enGetSteeringWheelPos()
       ***************************************************************************/
      /*!
       * \fn     tenDriveSideInfo  enGetSteeringWheelPos()
       * \brief  To get the default Steering Wheel Position
       * \retval tenDriveSideInfo
       **************************************************************************/
      tenDriveSideInfo enGetSteeringWheelPos() const override;

      /***************************************************************************
       ** FUNCTION:  tenDeviceSelectionMode spi_tclConfigurationManager::enGetSelectMode()
       ***************************************************************************/
      /*!
       * \fn     tenDriveSideInfo  enGetSelectMode()
       * \brief  To get the default Selection Mode
       * \retval tenDeviceSelectionMode
       **************************************************************************/
      tenDeviceSelectionMode enGetSelectMode() const override;

      /***************************************************************************
       ** FUNCTION:  tenDeviceCategory spi_tclConfigurationManager::enGetAndroidDeviceTechnologyPreference()
       ***************************************************************************/
      /*!
       * \fn     tenDeviceCategory  enGetAndroidDeviceTechnologyPreference()
       * \brief  To get the default Technology Preference for Android Devices
       * \retval tenDeviceCategory
       **************************************************************************/
      tenDeviceCategory enGetAndroidDeviceTechnologyPreference() const override;

      /***************************************************************************
       ** FUNCTION:  tenDeviceCategory spi_tclConfigurationManager::enGetAppleDeviceTechnologyPreference()
       ***************************************************************************/
      /*!
       * \fn     tenDeviceCategory  enGetAppleDeviceTechnologyPreference()
       * \brief  To get the default Technology Preference for Apple Devices
       * \retval tenDeviceCategory
       **************************************************************************/
      tenDeviceCategory enGetAppleDeviceTechnologyPreference() const override;

      /***************************************************************************
       ** FUNCTION:  spi_tclConfigurationManager::u32GetStartUpTimeInterval()
       ***************************************************************************/
      /*!
       * \fn      u32GetStartUpTimeInterval()
       * \brief   Function to get the CarPlay session startup time interval
       * \param   NONE
       * \retVal  t_U32 : Time interval in milli seconds.
       * \sa
       **************************************************************************/
      t_U32 u32GetStartUpTimeInterval() const override;

      /***************************************************************************
       ** FUNCTION:  tenETCInfo spi_tclConfigurationManager::enGetCarPlayETCEnabledFlag()
       ***************************************************************************/
      /*!
       * \fn      enGetCarPlayETCEnabledFlag()
       * \brief   Function to get the CarPlay ETC feature enable flag
       * \param   enDevCat: [IN] Category of the device whose ETC status is needed
       * \retVal  tenETCInfo: ETC Status
       * \sa
       **************************************************************************/
      tenETCInfo enGetCarPlayETCEnabledFlag(tenDeviceCategory enDevCat) const override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vGetScreenOffset(t_U32& u32Scre...
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetScreenOffset(t_U32& u32Screen_X_Offset,
       *          t_U32& u32Screen_Y_Offset)
       * \brief   To Get the Screen Offsets
       * \param   u32Screen_X_Offset : [OUT] X Offset
       * \param   u32Screen_Y_Offset : [OUT] Y Offset
       * \retval  t_Void
       * \sa
       **************************************************************************/
      t_Void vGetScreenOffset(trScreenOffset &rfoScreenOffset) const override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::bIsServerSideScalingRequired()
       ***************************************************************************/
      /*!
       * \fn      t_Bool bIsServerSideScalingRequired()
       * \brief   To Check whether the server side scaling required
       * \retval  t_Bool : false if the scaling is not required, else true
       **************************************************************************/
      t_Bool bIsServerSideScalingRequired() override;

      /***************************************************************************
       ** FUNCTION:  t_U8 spi_tclConfigurationManager::u8EnableTouchInputEvents()()
       ***************************************************************************/
      /*!
       * \fn      short t_U32 u8EnableTouchInputEvents()()
       * \brief   To Check whether the touch input handling is required
       * \retval  t_U8 0x0 -   if the touch input handling is not done using wayland
       *                       0x1 - if the Touch Input Handling is done via wayland
       **************************************************************************/
      t_U8 u8EnableTouchInputEvents() override;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclConfigurationManager::vSetScreenSize()
       ***************************************************************************/
      /*!
       * \fn     t_Void vSetScreenSize(const trScreenAttributes& corfrScreenAttributes)
       * \brief  Interface to set the screen size of Head Unit.
       * \param  corScreenAttributes : [IN] Screen Setting attributes.
       * \retval t_Void
       **************************************************************************/
      t_Void vSetScreenSize(const trScreenAttributes& corfrScreenAttributes) override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vSetContAttestFlag()
       ***************************************************************************/
      /*!
       * \fn      t_Void vSetContAttestFlag(t_U8 u8ContAttestFlag)
       * \brief   To enable/disable content attestation.Enabling or disableing should be done
       *          before the device is selected for the session
       * \pram    u8ContAttestFlag  : [IN] TRUE - Enable attestation
       *                                  FALSE - disable attestation
       * \retval  t_Void
       **************************************************************************/
      t_Void vSetContAttestFlag(t_U8 u8ContAttestFlag) override;

      /***************************************************************************
       ** FUNCTION:  t_String spi_tclConfigurationManager::szGetVideoCodecType()
       ***************************************************************************/
      /*!
       * \fn      t_String szGetVideoCodecType()
       * \brief   To get the video codec type
       *          before the device is selected for the session
       * \retval  t_String
       **************************************************************************/
      t_String szGetVideoCodecType() const override;

      /***************************************************************************
       ** FUNCTION:  t_U8 spi_tclConfigurationManager::u8GetMaxUnackedFrames()
       ***************************************************************************/
      /*!
       * \fn      t_U8 u8GetMaxUnackedFrames()
       * \brief   To get the max number of frames that can be recieved with out ack
       * \retval  t_U8
       **************************************************************************/
      t_U8 u8GetMaxUnackedFrames() const override;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetAutoStartProjection()
       ***************************************************************************/
      /*!
       * \fn      t_Bool bGetAutoStartProjection()
       * \brief   To get whether the auto projection start is enabled or not
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bGetAutoStartProjection() const override;

      /***************************************************************************
       ** FUNCTION:  t_U8 spi_tclConfigurationManager::u8GetFramesPerSec()
       ***************************************************************************/
      /*!
       * \fn      t_U8 u8GetFramesPerSec()
       * \brief   To get the frame per pixel rate
       * \retval  t_U8
       **************************************************************************/
      t_U8 u8GetFramesPerSec() const override;

      /***************************************************************************
       ** FUNCTION:  t_U8 spi_tclConfigurationManager::u8GetContAttestSignFlag()
       ***************************************************************************/
      /*!
       * \fn      t_U8 u8GetContAttestSignFlag()
       * \brief   To get the Content attestation info signed flag
       * \retval  t_U8
       **************************************************************************/
      t_U8 u8GetContAttestSignFlag() const override;

      /***************************************************************************
       ** FUNCTION:  tenPixelFormat spi_tclConfigurationManager::enGetPixelFormat()
       ***************************************************************************/
      /*!
       * \fn      tenPixelFormat enGetPixelFormat()
       * \brief   To get the pixel format for display
       * \retval  tenPixelFormat
       **************************************************************************/
      tenPixelFormat enGetPixelFormat() const override;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclConfigurationManager::vGetBdclVideoSinkConfig()
       ***************************************************************************/
      /*!
       * \fn     vGetBdclVideoSinkConfig() const
       * \brief  To Get BDCL VR Video Sink configuration
       * \retval  t_Void
       **************************************************************************/
      t_Void vGetBdclVideoSinkConfig(std::map<t_String, t_String>& mapVideoConfig) const override;

      /***************************************************************************
       ** FUNCTION: tenGeoCoordinateSystemType spi_tclConfigurationManager::enGetGeoCoordinateSystem()
       ***************************************************************************/
      /*!
       * \fn      enGetGeoCoordinateSystem()
       * \brief   Method to read the supported coordinate system
       * \retval  tenGeoCoordinateSystemType : To indicate type of Geo Coordinate System
       **************************************************************************/
      tenGeoCoordinateSystemType enGetGeoCoordinateSystem() const override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vReadConfigurationData
       ***************************************************************************/
      /*!
       * \fn     t_Void vReadConfigurationData()
       * \brief  Reads configuration data from the configuration handlers specific to the projects
       **************************************************************************/
      t_Void vReadConfigurationData();

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vSetMLNotiSettingVal()
       ***************************************************************************/
      /*!
       * \fn      vSetMLNotiSettingVal(t_Bool bMLNotiSettingVal)
       * \brief   Called to set the default value to notification setting.
       * \param   [IN] bMLNotiSettingVal: default value for notification setting
       * \retval  None
       **************************************************************************/
      t_Void vSetMLNotiSettingVal(t_Bool bMLNotiSettingVal) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::vSetMLLinkEnableSetVal()
       ***************************************************************************/
      /*!
       * \fn      vSetMLLinkEnableSetVal(tenEnabledInfo enMLLinkEnableSetVal)
       * \brief   Called to set the default value to enable ML usage.
       * \param   [IN] enMLLinkEnableSetVal: default value for enabling ML link setting
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bSetMLLinkEnableSetVal(tenEnabledInfo enMLLinkEnableSetVal) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::vSetDipoEnableSetVal()
       ***************************************************************************/
      /*!
       * \fn      vSetDipoEnableSetVal(tenEnabledInfo enDipoEnableSetVal)
       * \brief   Called to set the default value to enable Dipo usage.
       * \param   [IN] enDipoEnableSetVal: default value for enabling Dipo setting
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bSetDipoEnableSetVal(tenEnabledInfo enDipoEnableSetVal) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::bSetAAPEnableSetVal()
       ***************************************************************************/
      /*!
       * \fn      bSetAAPEnableSetVal(tenEnabledInfo enAAPEnableSetVal)
       * \brief   Called to set the default value to enable AAP usage.
       * \param   [IN] enAAPEnableSetVal: default value for enabling AAP setting
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bSetAAPEnableSetVal(tenEnabledInfo enAAPEnableSetVal) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::vSetCarlifeEnableSetVal()
       ***************************************************************************/
      /*!
       * \fn      vSetCarlifeEnableSetVal(tenEnabledInfo enCarlifeEnableSetVal)
       * \brief   Called to set the default value to enable AAP usage.
       * \param   [IN] enCarlifeEnableSetVal: default value for enabling Carlife setting
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bSetCarlifeEnableSetVal(tenEnabledInfo enCarlifeEnableSetVal) override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vSetSteeringWheelPos()
       ***************************************************************************/
      /*!
       * \fn      vSetSteeringWheelPos(tenDriveSideInfo enSteeringWheelPos)
       * \brief   Called to set the default value to steering wheel position.
       * \param   [IN] enSteeringWheelPos: default value for steering wheel position
       * \retval  None
       **************************************************************************/
      t_Void vSetSteeringWheelPos(tenDriveSideInfo enSteeringWheelPos) override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vSetSelectMode()
       ***************************************************************************/
      /*!
       * \fn      vSetSelectMode(tenDeviceSelectionMode enSelectMode)
       * \brief   Called to set the default value to selection mode.
       * \param   [IN] enSelectMode: default value for selection mode
       * \retval  None
       **************************************************************************/
      t_Void vSetSelectMode(tenDeviceSelectionMode enSelectMode) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::bSetAndroidDeviceTechnologyPreference()
       ***************************************************************************/
      /*!
       * \fn      bSetAndroidDeviceTechnologyPreference(tenDeviceCategory enAndroidDeviceTechPrefVal)
       * \brief   Called to set the default value to technology preference.
       * \param   [IN] enAndroidDeviceTechPrefVal: default value -technology preference for Android Device
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bSetAndroidDeviceTechnologyPreference(tenDeviceCategory enAndroidDeviceTechPrefVal) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::bSetAppleDeviceTechnologyPreference()
       ***************************************************************************/
      /*!
       * \fn      bSetAppleDeviceTechnologyPreference(tenDeviceCategory enAppleDeviceTechPrefVal)
       * \brief   Called to set the default value to technology preference.
       * \param   [IN] enAppleDeviceTechPrefVal: default value -technology preference for Apple Device
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bSetAppleDeviceTechnologyPreference(tenDeviceCategory enAppleDeviceTechPrefVal) override;

      /***************************************************************************
       ** FUNCTION:  t_U8 spi_tclConfigurationManager::u8GetSourceNumber(tenAudioDir)
       ***************************************************************************/
      /*!
       * \fn      u8GetSourceNumber( tenAudioDir enAudDir)
       * \brief   Function to get the Source Number for Audio Link & selected device
       *          category.
       * \param   [enAudDir]: Audio Link Direction
       * \retval  t_U8 value: Source Number
       **************************************************************************/
      t_U8 u8GetSourceNumber(tenAudioDir enAudDir) const override;

      /***************************************************************************
       ** FUNCTION:  t_U8 spi_tclConfigurationManager::u8ReadVirginStartSetting()
       ***************************************************************************/
      /*!
       * \fn      u8ReadVirginStartSetting( )
       * \brief   Called to read VirginStartSetting.
       * \param   None
       * \retval  t_U8
       **************************************************************************/
      t_U8 u8ReadVirginStartSetting() override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vWriteVirginStartSetting()
       ***************************************************************************/
      /*!
       * \fn      vWriteVirginStartSetting( )
       * \brief   Called to write VirginStartSetting to datapool.
       * \param   None
       * \retval  t_Void
       **************************************************************************/
      t_Void vWriteVirginStartSetting(t_U8 u8DefaultVgStrt) override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vGetSpiFeatureSupport(...)
       ***************************************************************************/
      /*!
       * \fn      vGetSpiFeatureSupport(trSpiFeatureSupport& rfrSpiFeatureSupport)
       * \brief   Method to read supported SPI features info.
       * \param   rfrSpiFeatureSupport: [OUT] Provides supported SPI features
       * \retval  t_Void
       **************************************************************************/
      t_Void vGetSpiFeatureSupport(trSpiFeatureSupport& rfrSpiFeatureSupport) override;

      /***************************************************************************
       ** FUNCTION:  tenATBoxSupportedState spi_tclConfigurationManager::vGetATBoxSupportedState()
       ***************************************************************************/
      /*!
       * \fn      tenATBoxSupportedState vGetATBoxSupportedState()
       * \brief   Method to read supported AT Box features info.
       * \param   NONE
       * \retval  tenATBoxSupportedState
       **************************************************************************/
      tenATBoxSupportedState vGetATBoxSupportedState() const override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::bWriteAudioLastMode()
       ***************************************************************************/
      /*!
       * \fn      bWriteAudioLastMode()
       * \brief   Method to write the audio last mode value into Datapool
       * \param   bAudioLastMode: [IN] Display last mode value to be written into datapool
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bWriteAudioLastMode(t_Bool bAudioLastMode) override;

      /***************************************************************************
       ** FUNCTION: t_U8 spi_tclConfigurationManager::u8GetRestrictionsInfo(tenDeviceCategory enDevCat, t_Bool bDriveRestrictions)
       ***************************************************************************/
      /*!
       * \fn      t_U8 u8GetRestrictionsInfo(tenDeviceCategory enDevCat, t_Bool bDriveRestrictions)
       * \brief   Method to get restrictions enabled in park/drive mode
       * \param   enDevCat: [IN] Device category
       * \param   bDriveRestrictions: [IN] drive restrictions
       * \retval  t_U8
       **************************************************************************/
      t_U8 u8GetRestrictionsInfo(tenDeviceCategory enDevCat, t_Bool bDriveRestrictions) const override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vSetFeatureRestrictions(tenDeviceCategory enDevCategory,const t_U8 cou8ParkModeRestrictionInfo,const t_U8 cou8DriveModeRestrictionInfo)
       ***************************************************************************/
      /*!
       * \fn      t_Void vSetFeatureRestrictions(tenDeviceCategory enDevCategory,const t_U8 cou8ParkModeRestrictionInfo,const t_U8 cou8DriveModeRestrictionInfo)
       * \brief   Method to set Vehicle Park/Drive Mode Restriction
       * \param   NONE
       * \retval  t_Void
       **************************************************************************/
      t_Void vSetFeatureRestrictions(tenDeviceCategory enDevCategory, const t_U8 cou8ParkModeRestrictionInfo,
               const t_U8 cou8DriveModeRestrictionInfo) override;

      /***************************************************************************
       ** FUNCTION:  tenDriveSideInfo spi_tclConfigurationManager::enGetDriveSideInfo(...)
       ***************************************************************************/
      /*!
       * \fn      enGetDriveSideInfo()
       * \brief   Method to read Drive side info.
       * \param   tenDriveSideInfo
       * \retval  Drive Side enum
       **************************************************************************/
      tenDriveSideInfo enGetDriveSideInfo() override;

      /***************************************************************************
       ** FUNCTION:  t_String spi_tclConfigurationManager::szGetSoftwareVersion()
       ***************************************************************************/
      /*!
       * \fn      t_String szGetSoftwareVersion()
       * \brief   Method to get the software version
       * \param   NONE
       * \retval  t_String
       **************************************************************************/
      t_String szGetSoftwareVersion() const override;

      /***************************************************************************
       ** FUNCTION:  t_String spi_tclConfigurationManager::szGetHardwareVersion()
       ***************************************************************************/
      /*!
       * \fn      t_String szGetHardwareVersion()
       * \brief   Method to get the accessory hardware version
       * \param   NONE
       * \retval  t_String
       **************************************************************************/
      t_String szGetHardwareVersion() override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vGetUSBPortsEnabledStatus(t_Bool &bUSBPort1,t_Bool &bUSBPort2,t_Bool &bUSBPort3)
       ***************************************************************************/
      /*!
       * \fn     t_Void vGetUSBPortsEnabledStatus()
       * \brief  to check whether the PORT is enabled/disabled
       **************************************************************************/
      t_Void vGetUSBPortsEnabledStatus(t_Bool &bUSBPort1, t_Bool &bUSBPort2, t_Bool &bUSBPort3) override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vGetDeviceInclusionListPath
       ***************************************************************************/
      /*!
       * \fn     t_Void vIntializeSPISettings()
       * \brief  Reads settings from xml file and store it internally
       **************************************************************************/
      t_Void vIntializeSPISettings() override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::bIsDeviceBlacklisted
       ***************************************************************************/
      /*!
       * \fn     t_Bool bIsDeviceBlacklisted()
       * \brief  to check whether  the device is blacklisted.
       **************************************************************************/
      t_Bool bIsDeviceBlacklisted(t_S32 s32ProductID, t_S32 s32VendorID, tenDeviceCategory enDeviceCtegory =
               e8DEV_TYPE_DEFAULT) override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vIsDeviceWhitelisted(...
       ***************************************************************************/
      /*!
       * \fn     t_Void vIsDeviceWhitelisted()
       * \brief  to check whether the connected device is whitelisted or not and return category
       **************************************************************************/
      t_Void vIsDeviceWhitelisted(t_S32 s32ProductID, t_S32 s32VendorID, tenDeviceCategory &rfenDevCat) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::bIsDeviceWhitelistingEnabled(...
       ***************************************************************************/
      /*!
       * \fn     t_Bool bIsDeviceWhitelistingEnabled()
       * \brief  to check whether the whitelisting is enabled for a project or not
       **************************************************************************/
      t_Bool bIsDeviceWhitelistingEnabled() override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::bIsWhitelistEnabledCat(...
       ***************************************************************************/
      /*!
       * \fn     t_Bool bIsWhitelistEnabledCat()
       * \brief  to set the category for which whitelisting is enabled
       **************************************************************************/
      t_Bool bIsWhitelistEnabledCat(tenDeviceCategory enChosenDeviceCategory) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::bIsDisplayLastModeProjection()
       ***************************************************************************/
      /*!
       * \fn      bIsDisplayLastModeProjection()
       * \brief   Method to read the display last mode from Datapool
       * \retval  t_Bool - True if display last mode is Projection
       **************************************************************************/
      t_Bool bIsDisplayLastModeProjection() override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::bIsAudioLastModeProjection()
       ***************************************************************************/
      /*!
       * \fn      bIsAudioLastModeProjection()
       * \brief   Method to read the audio last mode from Datapool
       * \retval  t_Bool - True if audio last mode is Projection
       **************************************************************************/
      t_Bool bIsAudioLastModeProjection() override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vSetDriveModeInfo()
       ***************************************************************************/
      /*!
       * \fn      t_Void vSetDriveModeInfo()
       * \brief   Method set the drive mode information
       * \param   enDriveModeInfo : [IN] Drive mode info.
       * \retval  t_Void
       **************************************************************************/
      t_Void vSetDriveModeInfo(const tenVehicleConfiguration enDriveModeInfo) override;

      /***************************************************************************
       ** FUNCTION:  t_String spi_tclConfigurationManager::szGetClientManufacturerName()
       ***************************************************************************/
      /*!
       * \fn      t_String szGetClientManufacturerName()
       * \brief   Method to get the client manufacturer name.
       * \param   NONE
       * \retval  t_String
       **************************************************************************/
      t_String szGetClientManufacturerName() const override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vSetNightModeInfo()
       ***************************************************************************/
      /*!
       * \fn      t_Void vSetNightModeInfo()
       * \brief   Method set the night  mode information
       * \param   enNightModeInfo : [IN] Night mode info
       * \retval  t_Void
       **************************************************************************/
      t_Void vSetNightModeInfo(const tenVehicleConfiguration enNightModeInfo) override;

      /***************************************************************************
       ** FUNCTION:   t_String spi_tclConfigurationManager::szGetChannelID()
       ***************************************************************************/
      /*!
       * \fn     t_String szGetChannelID()
       * \brief   Method to get channel ID
       * \retval  szChannelID : Unique Channel ID for Baidu Carlife technology.
       **************************************************************************/
      t_String szGetChannelID() const override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::bGetRotaryCtrlSupport()
       ***************************************************************************/
      /*!
       * \fn      t_Bool bGetRotaryCtrlSupport()
       * \brief   Method to know if Rotary Controller is supported
       * \param   NONE
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bGetRotaryCtrlSupport() override;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclConfigurationManager::vGetOemIconData(
       trVehicleBrandInfo& rfrVehicleBrandInfo)
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetOemIconData(trVehicleBrandInfo& rfrVehicleBrandInfo)
       * \brief   Method to read the EOL value and get the Brand configuration data
       from the look up table
       * \param   rfrVideoConfig: [IN]Vehicle Brand Data
       * \retval  NONE
       **************************************************************************/
      t_Void vGetOemIconData(trVehicleBrandInfo& rfrVehicleBrandInfo) override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vGetDriveModeInfo()
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetDriveModeInfo()
       * \brief   Method set the drive mode information
       * \param   enDriveModeInfo : [OUT] Drive mode info.
       * \retval  t_Void
       **************************************************************************/
      t_Void vGetDriveModeInfo(tenVehicleConfiguration &rfoenDriveModeInfo) override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vGetNightModeInfo()
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetNightModeInfo()
       * \brief   Method get the night  mode information
       * \param   rfoenNightModeInfo : [OUT] Night mode info
       * \retval  t_Void
       **************************************************************************/
      t_Void vGetNightModeInfo(tenVehicleConfiguration &rfoenNightModeInfo) override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vGetDisplayInputParam()
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetDisplayInputParam()
       * \brief   Method get the display input configuration.Input methods enabled.
       * \param   rfu8DisplayInput : [OUT] Display input parameter for DiPO
       * \retval  t_Void
       **************************************************************************/
      t_Void vGetDisplayInputParam(t_U8 &rfu8DisplayInput) override;


      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetNativeNavigationEnabled()
       ***************************************************************************/
      /*!
       * \fn      t_Bool bGetNativeNavigationEnabled()
       * \brief  Method to retrieve setting for native navigation
       * \param [OUT] : none
       * \retval t_Bool
       **************************************************************************/
      t_Bool bGetNativeNavigationEnabled() const override;

      /***************************************************************************
       ** FUNCTION:  virtual t_Bool spi_tclConfigurationManager::bSetUpAPonOOBT(const t_String coszVehicleModelName)
       ***************************************************************************/
      /*!
       * \fn      virtual t_Bool bSetUpAPonOOBT(const t_String coszVehicleModelName)
       * \brief   Method to Read the OOBT Enabled.
       * \param   NONE
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bSetUpAPonOOBT() override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::szGetVehicleModelNumber()
       ***************************************************************************/
      /*!
       * \fn      szGetVehicleModelNumber()
       * \brief   Method to get Vehicle Model Number information
       * \retval  t_Void
       **************************************************************************/
      t_Void szGetVehicleModelNumber(t_String& szVehicleModelNumber) const override;

      /***************************************************************************
       ** FUNCTION:  t_String spi_tclConfigurationManager::szGetClientID()
       ***************************************************************************/
      /*!
       * \fn      t_String szGetClientID()
       * \brief   Method to get the client ID.
       * \param   NONE
       * \retval  t_String
       **************************************************************************/
      t_String szGetClientID() const override;

      /***************************************************************************
       ** FUNCTION:  t_String spi_tclConfigurationManager::szGetClientFriendlyName()
       ***************************************************************************/
      /*!
       * \fn      t_String szGetClientFriendlyName()
       * \brief   Method to get the client friendly name.
       * \param   NONE
       * \retval  t_String
       **************************************************************************/
      t_String szGetClientFriendlyName() const override;

      /***************************************************************************
       ** FUNCTION: tenDAPAttestationInfo spi_tclConfigurationManager::enGetDAPAttestationInfo(...)
       ***************************************************************************/
      /*!
       * \fn      enGetDAPAttestationInfo()
       * \brief   Method to get DAP attestation information
       * \retval  tenDAPAttestationInfo
       **************************************************************************/
      tenDAPAttestationInfo enGetDAPAttestationInfo() const override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vGetSPIVersion()
       ***************************************************************************/
      /*!
       * \fn      szGetSPIVersion()
       * \brief   Method to get SPI Version information in Datapool
       * \retval  t_String
       **************************************************************************/
      t_String szGetSPIVersion() const override;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclConfigurationManager::vGetTouchScreenType
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetTouchScreenType()
       * \brief  Method to retrieve default setting of Touch Screen Type
       * \param [OUT] : returns the value of Touch screen
       * \retval t_Void
       **************************************************************************/
      t_Void vGetTouchScreenType(tenAAPTouchScreenType &rfenScreenType) override;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclConfigurationManager::bIsLocDataEnabled()
       ***************************************************************************/
      /*!
       * \fn      t_Bool bIsLocDataEnabled()
       * \brief  Method to check if Location data is enabled.
       * This method was added to check whether location data is enabled or not.
       * This function enable sensor data irrespective of Native/External Navigation.
       * For PSA,SUZUKI,GM ->this is set to true always
       * For AIVI ->This will be true only for Navi variants.
       * \param [OUT] : none.
       * \retval t_Bool
       **************************************************************************/
      t_Bool bIsLocDataEnabled() override;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclConfigurationManager::vGetEAPAppInfo(
       trEAPAppInfo& rfrEAPAppInfo)
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetEAPAppInfo(trEAPAppInfo& rfrEAPAppInfo)
       * \brief   Method to get EAP App info
       * \param   rfrEAPAppInfo: [IN]EAP App information
       * \retval  NONE
       **************************************************************************/
      t_Void vGetEAPAppInfo(trEAPAppInfo& rfrEAPAppInfo) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::bWriteDisplayLastMode()
       ***************************************************************************/
      /*!
       * \fn      bWriteDisplayLastMode()
       * \brief   Method to write the display last mode value into Datapool
       * \param   bDisplayLastMode: [IN] Display last mode value to be written into datapool
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bWriteDisplayLastMode(t_Bool bDisplayLastMode) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::bReadCPWStatus()
       ***************************************************************************/
      /*!
       * \fn      t_Bool bReadCPWStatus()
       * \brief   Method to Read the KDS value of CPW
       * \param   NONE
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bReadCPWStatus() override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vCarplayDipoConfigurations()
       ***************************************************************************/
      /*!
       * \fn      t_Void vCarplayDipoConfigurations()
       * \brief   Method to change the carplay dipo configurations
       * \param   NONE
       * \retval  t_Void
       **************************************************************************/
      t_Void vSelectCarplayConfiguration() override;

      /***************************************************************************
       ** FUNCTION: t_String spi_tclConfigurationManager::vGetPrimaryDisplayConfiguration()
       ***************************************************************************/
      /*!
       * \fn      t_String vGetPrimaryDisplayConfiguration(t_String &rfrSoftwareBuild)
       * \brief   Method to get primary display configuration from multiple display configuration
       * \param   rfvecVideoConfigList  : Vector consisting multiple display configurations.
       *          rVideoConfigData [out]: To conssit the primary display configuration.
       * \retval  NONE
       **************************************************************************/
      t_Void vGetPrimaryDisplayConfiguration(const tvecVideoConfigList& rfvecVideoConfigList,
               trVideoConfigData& rVideoConfigData) override;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclConfigurationManager::vGetVideoConfigData()
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetVideoConfigData(tenDeviceCategory enDevCat,
       *                 trVideoConfigData& rfrVideoConfig)
       * \brief   Method to read the EOL value and get the video configuration data
       from the look up table
       * \param   enDevCat      : [IN] Provide configurations fro the requested technology
       * \param   rfrVideoConfig: [IN] Video Config Data
       * \retval  NONE
       **************************************************************************/
      t_Void vGetVideoConfigData(tenDeviceCategory enDevCat, tvecVideoConfigList& rfvecVideoConfigList) override;

      /***************************************************************************
       ** FUNCTION:  tenRegion spi_tclConfigurationManager::enGetRegion()
       ***************************************************************************/
      /*!
       * \fn      tenRegion enGetRegion()
       * \brief   Method to get the current vehicle sale region
       * \param   NONE
       * \retval  tenRegion
       **************************************************************************/
      tenRegion enGetRegion() const override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vSetDriveSideInfo()
       ***************************************************************************/
      /*!
       * \fn      t_Void vSetDriveSideInfo()
       * \brief   Method to set the night mode info
       * \param   enVehicleConfig : [IN] Drive side info.
       * \retval  t_Void
       **************************************************************************/
      t_Void vSetDriveSideInfo(const tenVehicleConfiguration enVehicleConfig) override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vSetVehicleManufacturerInformation(...)
       ***************************************************************************/
      /*!
       * \fn      t_Void vSetVehicleManufacturerInformation(const trVehicleManufacturerInformation& corfrVehicleManufacturerInfoAttr)
       * \brief   Method to set the  Vehicle Manufacturer Information
       * \param   [IN] corfrVehicleManufacturerInfoAttr:Vehicle Manufacturer Information
       * \retval  None
       **************************************************************************/
      t_Void vSetVehicleManufacturerInformation(
               const trVehicleManufacturerInformation& corfrVehicleManufacturerInfoAttr) override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vGetSelectionModeStoredInDatapool()
       ***************************************************************************/
      /*!
       * \fn      vGetSelectionModeStoredInDatapool()
       * \brief   Method to get selection mode from datapool
       * \retval  t_Void
       **************************************************************************/
      t_Void vGetSelectionModeStoredInDatapool();

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vGetDeviceTechnologyPreferenceFromDatapool()
       ***************************************************************************/
      /*!
       * \fn      vGetDeviceTechnologyPreferenceFromDatapool()
       * \brief   Method to get device technology preference from datapool
       * \retval  t_Void
       **************************************************************************/
      t_Void vGetDeviceTechnologyPreferenceFromDatapool();

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::bSetAAPEnableSetVal()
       ***************************************************************************/
      /*!
       * \fn      bSetAAPEnableSetVal(tenEnabledInfo enmySPINLinkEnable)
       * \brief   Called to set the default value to enable AAP usage.
       * \param   [IN] enmySPINLinkEnable: default value for enabling AAP setting
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bSetMySPINEnableSetVal(tenEnabledInfo enmySPINLinkEnable) override;

      /***************************************************************************
       ** FUNCTION:  tenEnabledInfo spi_tclConfigurationManager::enReadDipoEnableSettingVal()
       ***************************************************************************/
      /*!
       * \fn      enReadDipoEnableSettingVal
       * \brief   Provides DiPO On/Off setting value stored in Datapool.
       * \param   None
       * \retval  tenEnabledInfo:
       *               e8USAGE_DISABLED - to set value as OFF
       *               e8USAGE_ENABLED - to set value as ON
       ***************************************************************************/
      tenEnabledInfo enReadDipoEnableSettingVal() const override;

      /***************************************************************************
       ** FUNCTION:  tenEnabledInfo spi_tclConfigurationManager::enReadMLEnableSettingVal()
       ***************************************************************************/
      /*!
       * \fn      enReadMLEnableSettingVal
       * \brief   Provides MirrorLink On/Off setting value stored in Datapool.
       * \param   None
       * \retval  tenEnabledInfo:
       *               e8USAGE_DISABLED - to set value as OFF
       *               e8USAGE_ENABLED - to set value as ON
       ***************************************************************************/
      tenEnabledInfo enReadMLEnableSettingVal() const override;

      /***************************************************************************
       ** FUNCTION:  tenEnabledInfo spi_tclConfigurationManager::enReadAAPEnableSettingVal()
       ***************************************************************************/
      /*!
       * \fn      enReadAAPEnableSettingVal
       * \brief   Provides AAP On/Off setting value stored in Datapool.
       * \param   None
       * \retval  tenEnabledInfo:
       *               e8USAGE_DISABLED - to set value as OFF
       *               e8USAGE_ENABLED - to set value as ON
       ***************************************************************************/
      tenEnabledInfo enReadAAPEnableSettingVal() const override;

      /***************************************************************************
       ** FUNCTION:  tenEnabledInfo spi_tclConfigurationManager::enReadmySPINEnableSettingVal()
       ***************************************************************************/
      /*!
       * \fn      enReadmySPINEnableSettingVal
       * \brief   Provides mySPIN On/Off setting value stored in Datapool.
       * \param   None
       * \retval  tenEnabledInfo:
       *               e8USAGE_DISABLED - to set value as OFF
       *               e8USAGE_ENABLED - to set value as ON
       ***************************************************************************/
      tenEnabledInfo enReadmySPINEnableSettingVal() const override;

      /***************************************************************************
       ** FUNCTION:  tenEnabledInfo spi_tclConfigurationManager::enReadCarlifeEnableSettingVal()
       ***************************************************************************/
      /*!
       * \fn      enReadCarlifeEnableSettingVal
       * \brief   Provides Carlife On/Off setting value stored in Datapool.
       * \param   None
       * \retval  tenEnabledInfo:
       *               e8USAGE_DISABLED - to set value as OFF
       *               e8USAGE_ENABLED - to set value as ON
       ***************************************************************************/
      tenEnabledInfo enReadCarlifeEnableSettingVal() const override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::bReadAndroidDeviceTechnologyPreferenceVal()
       ***************************************************************************/
      /*!
       * \fn      bReadAndroidDeviceTechnologyPreferenceVal()
       * \brief   Function to read Android Device the TechnologyPreference
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bReadAndroidDeviceTechnologyPreferenceVal(tenDeviceCategory &enTechnologyPreference) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::bReadAppleDeviceTechnologyPreferenceVal()
       ***************************************************************************/
      /*!
       * \fn      bReadAppleDeviceTechnologyPreferenceVal()
       * \brief   Function to read Apple Device the TechnologyPreference
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bReadAppleDeviceTechnologyPreferenceVal(tenDeviceCategory &enTechnologyPreference) override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vReadSysConfigs()
       ***************************************************************************/
      /*!
       * \fn      t_Void vReadSysConfigs()
       * \brief   Method to read & store the system configurations from KDS.
       * \param   NONE
       * \retval  t_Void
       **************************************************************************/
      t_Void vReadSysConfigs() override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vClearVehicleId()
       ***************************************************************************/
      /*!
       * \fn      vClearVehicleId()
       * \brief   Method to set VehicleId information
       * \retval  t_Void
       **************************************************************************/
      t_Void vClearVehicleId() override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vSetScreenConfigs()
       ***************************************************************************/
      /*!
       * \fn      t_Void vSetScreenConfigs(const trDisplayAttributes& corfrDispLayerAttr)
       * \brief   Method to set screen attributes
       * \param   corfrDispLayerAttr : [IN] Display attributes
       * \retval  t_Void
       **************************************************************************/
      t_Void vSetScreenConfigs(const trDisplayAttributes& corfrDispLayerAttr) override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vSetGeneralRestrictions(const t_U8 cou8GeneralRestrictionInfo)
       ***************************************************************************/
      /*!
       * \fn      t_Void vSetGeneralRestrictions(const t_U8 cou8GeneralRestrictionInfo)
       * \brief   Method to set General Restriction
       * \param   NONE
       * \retval  t_Void
       **************************************************************************/
      t_Void vSetGeneralRestrictions(tenDeviceCategory enDeviceCategory, const t_U16 u16GeneralRestrictionInfo)
               override;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclConfigurationManager::vDisplayAudioSettings()
       ***************************************************************************/
      /*!
       * \fn     t_Void vDisplayAudioSettings()
       * \brief  To print the audio settings data
       * \retval  t_Void
       **************************************************************************/
      t_Void vDisplayAudioSettings() override;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclConfigurationManager::vDisplayVideoSettings()
       ***************************************************************************/
      /*!
       * \fn     t_Void vDisplayVideoSettings()
       * \brief  To print the video settings data
       * \retval  t_Void
       **************************************************************************/
      t_Void vDisplayVideoSettings() override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vDisplayAppSettings()
       ***************************************************************************/
      /*!
       * \fn     t_Void vDisplayAppSettings()
       * \brief  To print the settings
       * \retval t_Void
       **************************************************************************/
      t_Void vDisplayAppSettings() override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::bWriteOnCarEnableSetting
       ***************************************************************************/
      /*!
       * \fn      bWriteOnCarEnableSetting(tenEnabledInfo enOnCarLinkEnable)
       * \brief   Sets the OnCar On/Off setting value in datapool.
       *          (True - if enabled, else False)
       * \param   [IN] enOnCarLinkEnable :
       *               e8USAGE_DISABLED - to set value as OFF
       *               e8USAGE_ENABLED - to set value as ON
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bWriteOnCarEnableSetting(tenEnabledInfo enOnCarLinkEnable) override;

      /***************************************************************************
       ** FUNCTION:  tenEnabledInfo spi_tclConfigurationManager::enReadOnCarEnableSetting()
       ***************************************************************************/
      /*!
       * \fn      enReadOnCarEnableSetting
       * \brief   Provides OnCar On/Off setting value stored in Datapool.
       * \param   None
       * \retval  tenEnabledInfo:
       *               e8USAGE_DISABLED - to set value as OFF
       *               e8USAGE_ENABLED - to set value as ON
       ***************************************************************************/
      tenEnabledInfo enReadOnCarEnableSetting() const override;

      /***************************************************************************
       ** FUNCTION:  tenEnabledInfo spi_tclConfigurationManager::enGetOnCarEnableSetVal()
       ***************************************************************************/
      /*!
       * \fn     tenEnabledInfo enGetOnCarEnableSetVal()
       * \brief  To get the default OnCar Enable Setting Value
       * \retval tenEnabledInfo
       **************************************************************************/
      tenEnabledInfo enGetOnCarEnableSetVal() const override;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetITCommanderSupport())
       ***************************************************************************/
      /*!
       * \fn      t_Bool bGetITCommanderSupport()
       * \brief   Function to Get ITCommander Support
       * \retval  NONE
       **************************************************************************/
      t_Bool bGetITCommanderSupport() override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vGetVehicleBTAddress()
       ***************************************************************************/
      /*!
       * \fn     vGetVehicleBTAddress()
       * \brief  Function to get the vehicle BT address.
       * \param  szVehicleBTAddress:[OUT] Vehicle BT address
       * \retval  None
       **************************************************************************/
      t_Void vGetVehicleBTAddress(t_String &szVehicleBTAddress) override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclConfigurationManager::vOnVehicleBTAddress()
       ***************************************************************************/
      /*!
       * \fn     vOnVehicleBTAddress()
       * \brief  Function to set the vehicle BT address.
       * \param  szVehicleBTAddress:[OUT] Vehicle BT address
       * \retval  None
       **************************************************************************/
      t_Void vOnVehicleBTAddress(t_String &szVehicleBTAddress) override;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclConfigurationManager::bGetSysVehicleInfo())
       ***************************************************************************/
      /*!
       * \fn      t_Bool bGetSysVehicleInfo()
       * \brief   Function to Get Region, Vendor and Vehicle
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bGetSysVehicleInfo(t_U8 &u8RegionCode, t_U8 &u8VendorCode, t_U16 &u16VehicleCode) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetCarlifeIOSFeatureSupport
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetCarlifeIOSFeatureSupport()
       * \brief  Method to retrieve Carlife Ios enable info from policy
       * \retval t_Bool
       **************************************************************************/
      t_Bool bGetCarlifeIOSFeatureSupport() override;

      /***************************************************************************
       ** FUNCTION: t_U8 spi_tclConfigurationManager::u8ReadOEMBrandFromKDS()
       ***************************************************************************/
      /*!
       * \fn      t_U8 u8ReadOEMBrandFromKDS()
       * \brief   Method to read the KDS value and get the Brand type info
       * \retval  NONE
       **************************************************************************/
      t_U8 u8ReadOEMBrandFromKDS() override;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclConfigurationManager::vGetVehicleInfoDataMySPIN(trVehicleInfo& rfrVehicleInfo)
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetVehicleInfoDataMySPIN()
       * \brief   Method to get the vehicle manufacturer name and variant info
       * \param   rfrVehicleInfo: [IN]Vehicle Info Data
       * \retval  NONE
       **************************************************************************/
      t_Void vGetVehicleInfoDataMySPIN(trVehicleInfo& rfrVehicleInfo) override;

      /***************************************************************************
      FUNCTION: t_U16 spi_tclConfigurationManager::u16ReadVehicleTypeFromKDS())
      ***************************************************************************/
      /*!
      * \fn      t_U16 u16ReadVehicleTypeFromKDS()
      * \brief   Function to Get Vehicle Type
      * \retval  t_U16
      ***************************************************************************/
      t_U16 u16ReadVehicleTypeFromKDS() override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclConfigurationManager::bIsDeviceWhitelisted
       ***************************************************************************/
      /*!
       * \brief  check if the device is whitelisted based on VID and PID of device
       * \param  [IN] u32ProductID :Product ID
       * \param  [IN] u32VendorID : Vendor ID
       * \param  [IN] enReqDevCat : Device Category
       ***************************************************************************/
      t_Bool bIsDeviceWhitelisted(t_U32 u32ProductID, t_U32 u32VendorID, tenDeviceCategory enReqDevCat) override;


      /***************************************************************************
      ** FUNCTION:  t_Void spi_tclConfigurationManager::vRegisterKeyCodes()
      ***************************************************************************/
      /*!
      * \fn     vRegisterKeyCodes()
      * \brief  Function to Register keyinfo into datapool
      * \param  corfrmapTechSuppHardkeys:[IN] Hard Keys Supported for Technologies
      * \param  corfrmapTechSuppSoftkeys:[IN] Soft Keys Supported for Technologies
      * \retval  None
      **************************************************************************/
      t_Void vRegisterKeyCodes(const std::map<tenDeviceCategory,std::set<tenKeyCode>>& corfrmapTechSuppHardkeys,
                               const std::map<tenDeviceCategory,std::set<tenKeyCode>>& corfrmapTechSuppSoftkeys) override;

      /***************************************************************************
      ** FUNCTION:  t_Bool spi_tclConfigurationManager::bGetKeyInfo()
      ***************************************************************************/
      /*!
      * \fn     bGetKeyInfo()
      *\brief  Function to Get the KeyInfo.
      * \param  rfrsetKeyCodes:[OUT] Keys supported
      * \param  enCategory    :[IN] For Which Technology
      * \retval  t_Bool
      **************************************************************************/
      t_Bool bGetKeyInfo(std::set<tenKeyCode>& rfrsetKeyCodes,tenDeviceCategory enCategory) override;

      /***************************************************************************
      ** FUNCTION:  t_Void spi_tclConfigurationManager::bGetSoftKeyInfo()
      ***************************************************************************/
      /*!
      * \fn     bGetSoftKeyInfo()
      *\brief  Function to Get Soft KeyInfo.
      * \param  rfrsetKeyCodes:[OUT] Keys supported
      * \param  enCategory:[IN] For Which Technology
      * \retval  t_Bool
      **************************************************************************/
      t_Bool bGetSoftKeyInfo(std::set<tenKeyCode>& rfrsetKeyCodes,tenDeviceCategory enCategory) override;


      /***************************************************************************
      ** FUNCTION:  t_Bool spi_tclDataServiceSettingsIntf::bGetCarlifeGPSDataSupportflag()
      ***************************************************************************/
      /*!
      * \fn     t_Bool bGetCarlifeGPSDataSupportflag() const
      * \brief  Interface to get location data enabled setting for carlfie
      * \retval t_Bool
      **************************************************************************/
      t_Bool bGetCarlifeGPSDataSupportflag() const override;

      /***************************************************************************
      ** FUNCTION: t_Void spi_tclConfigurationManager::vGetDiPOExtendedFeaturesSupport(
                        trExtendedFeaturesSupport& rfrExtendedFeaturesSupport)
      ***************************************************************************/
      /*!
      * \fn      t_Void vGetDiPOExtendedFeaturesSupport(trExtendedFeaturesSupport& rfrExtendedFeaturesSupport)
      * \brief   Method to read DiPO Extended Features Support
      * \param   rfrExtendedFeaturesSupport: [OUT]Extended Features Support
      * \retval  NONE
      **************************************************************************/
      t_Void vGetDiPOExtendedFeaturesSupport(trExtendedFeaturesSupport& rfrExtendedFeaturesSupport) override;

      /***************************************************************************
      ** FUNCTION:  t_Bool spi_tclConfigurationManager::bGetCarPlayGPSDataSupportflag()
      ***************************************************************************/
      /*!
      * \fn     t_Bool bGetCarPlayGPSDataSupportflag()
      * \brief  Interface to get GPS data enabled setting
      * \retval t_Bool
      **************************************************************************/
      t_Bool bGetCarPlayGPSDataSupportflag() override;

       /***************************************************************************
       ****************************END OF PUBLIC *********************************
       ***************************************************************************/

   private:
      /***************************************************************************
       *********************************PRIVATE*************************************
       ***************************************************************************/

      /***************************************************************************
      ** FUNCTION:  t_Void spi_tclConfigurationManager::vSetDefaultKeyInfo()
      ***************************************************************************/
      /*!
      * \fn     vSetDefaultKeyInfo()
      *\brief   Function to Set Default KeyInfo.
      * \retval None
      **************************************************************************/
      t_Void vSetDefaultKeyInfo();

      /***************************************************************************
      ** FUNCTION:  t_Void spi_tclConfigurationManager::vInsertDefKeystoMap()
      ***************************************************************************/
      /*!
      * \fn     vInsertDefKeystoMap()
      *\brief   Function to Fill Default KeyInfo to Map.
      * \retval None
      **************************************************************************/
      t_Void vInsertDefKeystoMap();

      //! pointer object to StaticConfigurationBase
      spi_tclStaticConfigurationBaseIntf *m_poStaticConfigurationBase;

      //! pointer object to DynamicConfigurationBase
      spi_tclDynamicConfigurationBase *m_poDynamicConfigurationBase;

      //! Head unit information
      trHeadUnitInfo m_rHeadUnitInfo;

      //! Default Supported Hard Keys
      std::map<tenDeviceCategory, std::set<tenKeyCode>> m_mapTechSuppHardKeys;

      //! Default Supported Soft Keys
      std::map<tenDeviceCategory, std::set<tenKeyCode>> m_mapTechSuppSoftKeys;

      /***************************************************************************
       ****************************END OF PRIVATE *********************************
       ***************************************************************************/
};
#endif // SPI_TCLCONFIGURATIONMANAGER_
