/***********************************************************************/
/*!
* \file  spi_tclCommonSettingsIntf.h
* \brief  Implementation of the Class spi_tclCommonSettingsIntf
*************************************************************************
\verbatim


PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    Interface class for all SPI Common Settings
AUTHOR:         kmo6kor
COPYRIGHT:      &copy; 2017 Robert Bosch Car Multimedia GmbH
HISTORY:
Date        | Author                | Modification
07.06.2018  | kmo6kor			    Initial Version                                               
\endverbatim
*************************************************************************/

#include "SPITypes.h"
//#include "DiPOTypes.h"

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/
/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/
#ifndef SPI_TCLCOMMONSETTINGSINTF_
#define SPI_TCLCOMMONSETTINGSINTF_


class spi_tclCommonSettingsIntf
{

public:

   /***************************************************************************
   ****************************PUBLIC *********************************
   ***************************************************************************/

   /***************************************************************************
	** FUNCTION: spi_tclCommonSettingsIntf::spi_tclCommonSettingsIntf()
	***************************************************************************/
   /*!
	* \fn      spi_tclCommonSettingsIntf()
	* \brief   Constructor
	* \param   None
	**************************************************************************/
   spi_tclCommonSettingsIntf(){}

   /***************************************************************************
	** FUNCTION: spi_tclCommonSettingsIntf::~spi_tclCommonSettingsIntf()
	***************************************************************************/
   /*!
	* \fn      ~spi_tclCommonSettingsIntf()
	* \brief   Destructor
	* \param   None
	**************************************************************************/
   virtual ~spi_tclCommonSettingsIntf(){}

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclCommonSettingsIntf::vSetRegion(.)
   ***************************************************************************/
   /*!
   * \fn     t_Void vSetRegion(tenRegion enRegion)
   * \brief  Interface to set the region for application certification.
   *         It gives the info of which region CCC Guidelines should be followed
   *         for the Application Certification Filtering
   * \param  [IN] enRegion : Region enumeration
   * \retval  NONE
   **************************************************************************/
   virtual t_Void vSetRegion(tenRegion enRegion) = 0;

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclCommonSettingsIntf::vSetMLNotificationOnOff()
   ***************************************************************************/
   /*!
   * \fn     t_Void vSetMLNotificationOnOff(t_Bool bSetNotificationsOn)
   * \brief  To Set the Notifications to On/Off
   * \param  bSetNotificationsOn : [IN] True-Set Notifications to ON
   *                                    False - Set Notifications to OFF
   * \retval t_Void
   **************************************************************************/
   virtual t_Void vSetMLNotificationOnOff(t_Bool bSetNotificationsOn) = 0;

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclCommonSettingsIntf::vGetSpiFeatureSupport(...)
   ***************************************************************************/
   /*!
   * \fn      vGetSpiFeatureSupport(trSpiFeatureSupport& rfrSpiFeatureSupport)
   * \brief   Method to read supported SPI features info.
   * \param   rfrSpiFeatureSupport: [OUT] Provides supported SPI features
   * \retval  NONE
   **************************************************************************/
   virtual t_Void vGetSpiFeatureSupport(trSpiFeatureSupport& rfrSpiFeatureSupport) = 0;

   /***************************************************************************
    ** FUNCTION: t_Bool spi_tclCommonSettingsIntf::bGetMLDynAudioSupport()
    ***************************************************************************/
   /*!
    * \fn     t_Bool bGetMLDynAudioSupport() const
    * \brief  To Get whether ML Audio is supported or not.
    * \retval  t_Bool
    **************************************************************************/
   virtual t_Bool bGetMLDynAudioSupport() const = 0;

   /***************************************************************************
   ** FUNCTION: t_Bool spi_tclCommonSettingsIntf::bGetMySIPNAdvAudioSupport()
   ***************************************************************************/
   /*!
    * \fn     bGetMySIPNAdvAudioSupport() const
    * \brief  To Get support for mySPIN Advanced Audio
    * \retval  t_Bool
    **************************************************************************/
   virtual t_Bool bGetMySIPNAdvAudioSupport() const = 0;

   /***************************************************************************
    ** FUNCTION: t_Bool spi_tclCommonSettingsIntf::bGetLastModeSupport()
    ***************************************************************************/
   /*!
    * \fn     bGetLastModeSupport() const
    * \brief  To Get last mode support configuration for a device category
    * \param  enDevCat [IN]: Device category
    * \retval  t_Bool
    **************************************************************************/
   virtual t_Bool bGetLastModeSupport(tenDeviceCategory enDevCat) const = 0;

   /***************************************************************************
   ** FUNCTION: tenLastModeSupport spi_tclCommonSettingsIntf::enGetLastModeSupport()
   ***************************************************************************/
   /*!
   * \fn     enGetLastModeSupport() const
   * \brief  To Get last mode support configuration
   * \retval  tenLastModeSupport

   **************************************************************************/
   virtual tenLastModeSupport enGetLastModeSupport() const = 0;

   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclCommonSettingsIntf::bIsXMLValidationEnabled
    ***************************************************************************/
   /*!
    * \fn     t_Bool bIsXMLValidationEnabled()
    * \brief  Method to check whether the XML validation is enabled or not
    * \retval t_Bool
    **************************************************************************/
   virtual t_Bool bIsXMLValidationEnabled() const = 0;

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclCommonSettingsIntf::vGetHeadUnitInfo
    ***************************************************************************/
   /*!
    * \fn     t_Void vGetHeadUnitInfo()
    * \brief  Method to retrieve head unit information
    * \retval t_Void
    **************************************************************************/
   virtual t_Void vGetHeadUnitInfo(trHeadUnitInfo &rfrHeadUnitInfo) = 0;

   /***************************************************************************
   ** FUNCTION:  t_U8 spi_tclCommonSettingsIntf::u8EnableTouchInputEvents()()
   ***************************************************************************/
   /*!
   * \fn      short t_U32 u8EnableTouchInputEvents()()
   * \brief   To Check whether the touch input handling is required
   * \retval  t_U8 0x0 -   if the touch input handling is not done using wayland
   *                       0x1 - if the Touch Input Handling is done via wayland
   **************************************************************************/
   virtual t_U8 u8EnableTouchInputEvents() = 0;

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclCommonSettingsIntf::vSetScreenSize()
   ***************************************************************************/
   /*!
   * \fn     t_Void vSetScreenSize(const trScreenAttributes& corfrScreenAttributes)
   * \brief  Interface to set the screen size of Head Unit.
   * \param  corScreenAttributes : [IN] Screen Setting attributes.
   * \retval t_Void
   **************************************************************************/
   virtual t_Void vSetScreenSize(const trScreenAttributes& corfrScreenAttributes) = 0;

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclCommonSettingsIntf::vSetContAttestFlag()
   ***************************************************************************/
   /*!
   * \fn      t_Void vSetContAttestFlag(t_U8 u8ContAttestFlag)
   * \brief   To enable/disable content attestation.Enabling or disableing should be done
   *          before the device is selected for the session
   * \pram    u8ContAttestFlag  : [IN] TRUE - Enable attestation
   *                                  FALSE - disable attestation
   * \retval  t_Void
   **************************************************************************/
   virtual t_Void vSetContAttestFlag(t_U8 u8ContAttestFlag) = 0;

   /***************************************************************************
   ** FUNCTION: t_Bool spi_tclCommonSettingsIntf::bGetMLNotificationEnabledInfo()
   ***************************************************************************/
   /*!
   * \fn     t_Bool bGetMLNotificationEnabledInfo()
   * \brief  Provides information on whether ML Notifications are enabled or
   *         disabled.
   * \retval t_Bool  TRUE- Enabled FALSE-Disabled
   **************************************************************************/
   virtual t_Bool bGetMLNotificationEnabledInfo() const = 0;

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclCommonSettingsIntf::vGetDefaultProjUsageSettings
    ***************************************************************************/
   /*!
    * \fn     t_Void vGetDefaultProjUsageSettings()
    * \brief  Method to retrieve default setting for projection usage
    * \param [OUT] : returns the current value of device projection enable
    * \enSPIType  : indicates the type of SPI technology. e8DEV_TYPE_UNKNOWN indicates default value for any SPI technology
    * \retval t_Void
    **************************************************************************/
   virtual t_Void vGetDefaultProjUsageSettings(tenEnabledInfo &enEnabledInfo, tenDeviceCategory enSPIType) = 0;

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclCommonSettingsIntf::vClearPrivateData
    ***************************************************************************/
   /*!
    * \fn     vClearPrivateData
    * \brief  Clears device history persistently stored on target and other private data
	* \retval  t_Void
    **************************************************************************/
   virtual t_Void vClearPrivateData() = 0;

   /***************************************************************************
    ** FUNCTION:  spi_tclCommonSettingsIntf::vSetDeviceSelectionMode()
    ***************************************************************************/
   /*!
    * \fn      t_Void vSetDeviceSelectionMode(
    * \brief   Method to set the device selection mode to automatic/manual. Changes
    *          will take into effect on successive connection
    * \param   enSelectionMode : Device selection mode @see tenDeviceSelectionMode
    * \retval  t_Void
    **************************************************************************/
   virtual t_Void vSetDeviceSelectionMode(tenDeviceSelectionMode enSelectionMode, tenDeviceCategory enTechnologyType) = 0;

   /***************************************************************************
    ** FUNCTION:  tenDeviceCategory spi_tclCommonSettingsIntf::enGetTechnologyPreference
    ***************************************************************************/
   /*!
    * \fn     tenDeviceCategory enGetTechnologyPreference()
    * \brief  returns Device type preference
	* \retval  tenDeviceCategory
    **************************************************************************/
   virtual tenDeviceCategory enGetTechnologyPreference(tenDeviceType enDeviceType) const = 0;

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclCommonSettingsIntf::vGetMySPINIapAppInfo
   ***************************************************************************/
   /*!
   * \fn     t_Void vGetMySPINIapAppInfo()
   * \brief  Method to retrieve head unit information
   * \retval t_Void
   **************************************************************************/
   virtual t_Void vGetMySPINIapAppInfo(trMySPINIapAppInfo& rMySPINIapAppInfo) = 0;

   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclCommonSettingsIntf::bWriteAudioLastMode()
    ***************************************************************************/
   /*!
    * \fn      bWriteAudioLastMode()
    * \brief   Method to write the audio last mode value into Datapool
    * \param   bAudioLastMode: [IN] Display last mode value to be written into datapool
    * \retval  t_Bool
    **************************************************************************/
   virtual t_Bool bWriteAudioLastMode(t_Bool bAudioLastMode) = 0;

   /***************************************************************************
   ** FUNCTION: t_U8 spi_tclCommonSettingsIntf::u8GetRestrictionsInfo(tenDeviceCategory enDevCat, t_Bool bDriveRestrictions)
   ***************************************************************************/
   /*!
   * \fn      t_U8 u8GetRestrictionsInfo(tenDeviceCategory enDevCat, t_Bool bDriveRestrictions)
   * \brief   Method to get restrictions enabled in park/drive mode
   * \param   enDevCat: [IN] Device category
   * \param   bDriveRestrictions: [IN] drive restrictions
   * \retval  t_U8
   **************************************************************************/
   virtual t_U8 u8GetRestrictionsInfo(tenDeviceCategory enDevCat,t_Bool bDriveRestrictions) const =0;

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclCommonSettingsIntf::vSetFeatureRestrictions(tenDeviceCategory enDevCategory,const t_U8 cou8ParkModeRestrictionInfo,const t_U8 cou8DriveModeRestrictionInfo)
    ***************************************************************************/
   /*!
    * \fn      t_Void vSetFeatureRestrictions(tenDeviceCategory enDevCategory,const t_U8 cou8ParkModeRestrictionInfo,const t_U8 cou8DriveModeRestrictionInfo)
    * \brief   Method to set Vehicle Park/Drive Mode Restriction
    * \param   NONE
    * \retval  t_Void
    **************************************************************************/
   virtual t_Void vSetFeatureRestrictions(tenDeviceCategory enDevCategory,const t_U8 cou8ParkModeRestrictionInfo,const t_U8 cou8DriveModeRestrictionInfo) = 0;
   
   /***************************************************************************
   ** FUNCTION: t_Void spi_tclCommonSettingsIntf::vGetVideoConfigData()
   ***************************************************************************/
   /*!
   * \fn      t_Void vGetVideoConfigData(tenDeviceCategory enDevCat,
   *                 trVideoConfigData& rfrVideoConfig)
   * \brief   Method to read the EOL value and get the video configuration data
              from the look up table
   * \param   enDevCat      : [IN] Provide configurations fro the requested technology
   * \param   rfrVideoConfig: [IN] Video Config Data
   * \retval  NONE
   **************************************************************************/
   virtual t_Void vGetVideoConfigData(tenDeviceCategory enDevCat,
            tvecVideoConfigList& rfvecVideoConfigList) = 0;

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclCommonSettingsIntf::vGetPrimaryDisplayConfiguration()
   ***************************************************************************/
   /*!
   * \fn      t_Void vGetPrimaryDisplayConfiguration(t_String &rfrSoftwareBuild)
   * \brief   Method to get primary display configuration from multiple display configuration
   * \param   rfvecVideoConfigList  : Vector consisting multiple display configurations.
   *          rVideoConfigData [out]: To conssit the primary display configuration.
   * \retval  NONE
   **************************************************************************/
   virtual t_Void vGetPrimaryDisplayConfiguration(const tvecVideoConfigList& rfvecVideoConfigList,
                                          trVideoConfigData& rVideoConfigData) = 0;

   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclCommonSettingsIntf::bWriteDisplayLastMode()
    ***************************************************************************/
   /*!
    * \fn      bWriteDisplayLastMode()
    * \brief   Method to write the display last mode value into Datapool
    * \param   bDisplayLastMode: [IN] Display last mode value to be written into datapool
    * \retval  t_Bool
    **************************************************************************/
   virtual t_Bool bWriteDisplayLastMode(t_Bool bDisplayLastMode) = 0;

   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclCommonSettingsIntf::bIsDisplayLastModeProjection()
    ***************************************************************************/
   /*!
    * \fn      bIsDisplayLastModeProjection()
    * \brief   Method to read the display last mode from Datapool
    * \retval  t_Bool - True if display last mode is Projection
    **************************************************************************/
   virtual t_Bool bIsDisplayLastModeProjection() = 0;

   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclCommonSettingsIntf::bIsAudioLastModeProjection()
    ***************************************************************************/
   /*!
    * \fn      bIsAudioLastModeProjection()
    * \brief   Method to read the audio last mode from Datapool
    * \retval  t_Bool - True if audio last mode is Projection
    **************************************************************************/
   virtual t_Bool bIsAudioLastModeProjection() = 0;

   /***************************************************************************
    ** FUNCTION: tenDAPAttestationInfo spi_tclCommonSettingsIntf::enGetDAPAttestationInfo(...)
    ***************************************************************************/
   /*!
    * \fn      enGetDAPAttestationInfo()
    * \brief   Method to get DAP attestation information
    * \retval  tenDAPAttestationInfo
    **************************************************************************/
   virtual tenDAPAttestationInfo enGetDAPAttestationInfo() const = 0;

   /***************************************************************************
    ** FUNCTION:  t_String spi_tclCommonSettingsIntf::vGetSPIVersion()
    ***************************************************************************/
   /*!
    * \fn      szGetSPIVersion()
    * \brief   Method to get SPI Version information in Datapool
    * \retval  t_String
    **************************************************************************/
   virtual t_String szGetSPIVersion() const = 0;

   /***************************************************************************
    ** FUNCTION: t_Void spi_tclCommonSettingsIntf::vGetTouchScreenType
    ***************************************************************************/
   /*!
    * \fn      t_Void vGetTouchScreenType()
    * \brief  Method to retrieve default setting of Touch Screen Type
    * \param [OUT] : returns the value of Touch screen
    * \retval t_Void
    **************************************************************************/
   virtual t_Void vGetTouchScreenType(tenAAPTouchScreenType &rfenScreenType) = 0;

   /***************************************************************************
    ** FUNCTION: t_Bool spi_tclCommonSettingsIntf::bIsLocDataEnabled()
    ***************************************************************************/
   /*!
    * \fn      t_Bool bIsLocDataEnabled()
    * \brief  Method to check if Location data is enabled.
    * This method was added to check whether location data is enabled or not.
    * This function enable sensor data irrespective of Native/External Navigation.
    * For PSA,SUZUKI,GM ->this is set to true always
    * For AIVI ->This will be true only for Navi variants.
    * \param [OUT] : none.
    * \retval t_Bool
    **************************************************************************/
   virtual t_Bool bIsLocDataEnabled() = 0;

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclCommonSettingsIntf::vGetEAPAppInfo(
                     trEAPAppInfo& rfrEAPAppInfo)
   ***************************************************************************/
   /*!
   * \fn      t_Void vGetEAPAppInfo(trEAPAppInfo& rfrEAPAppInfo)
   * \brief   Method to get EAP App info
   * \param   rfrEAPAppInfo: [IN]EAP App information
   * \retval  NONE
   **************************************************************************/
   virtual t_Void vGetEAPAppInfo(trEAPAppInfo& rfrEAPAppInfo) = 0;

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclCommonSettingsIntf::bReadCPWStatus()
   ***************************************************************************/
   /*!
   * \fn      t_Bool bReadCPWStatus()
   * \brief   Method to Read the KDS value of CPW
   * \param   NONE
   * \retval  t_Bool
   **************************************************************************/
   virtual t_Bool bReadCPWStatus() = 0;

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclCommonSettingsIntf::vCarplayDipoConfigurations()
   ***************************************************************************/
   /*!
   * \fn      t_Void vCarplayDipoConfigurations()
   * \brief   Method to change the carplay dipo configurations
   * \param   NONE
   * \retval  t_Void
   **************************************************************************/
   virtual t_Void  vSelectCarplayConfiguration() = 0;

   /***************************************************************************
   ** FUNCTION:  tenRegion spi_tclCommonSettingsIntf::enGetRegion()
   ***************************************************************************/
   /*!
   * \fn      tenRegion enGetRegion()
   * \brief   Method to get the current vehicle sale region
   * \param   NONE
   * \retval  tenRegion
   **************************************************************************/
   virtual tenRegion enGetRegion() const = 0;

   /***************************************************************************
   ** FUNCTION:  tenATBoxSupportedState spi_tclCommonSettingsIntf::vGetATBoxSupportedState()
   ***************************************************************************/
   /*!
   * \fn      tenATBoxSupportedState vGetATBoxSupportedState()
   * \brief   Method to read supported AT Box features info.
   * \param   NONE
   * \retval  tenATBoxSupportedState
   **************************************************************************/
   virtual tenATBoxSupportedState vGetATBoxSupportedState() const = 0;

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclCommonSettingsIntf::vSetVehicleManufacturerInformation(...)
   ***************************************************************************/
   /*!
   * \fn      t_Void vSetVehicleManufacturerInformation(const trVehicleManufacturerInformation& corfrVehicleManufacturerInfoAttr)
   * \brief   Method to set the  Vehicle Manufacturer Information
   * \param   [IN] corfrVehicleManufacturerInfoAttr:Vehicle Manufacturer Information
   * \retval  None
   **************************************************************************/
   virtual t_Void vSetVehicleManufacturerInformation(const trVehicleManufacturerInformation& corfrVehicleManufacturerInfoAttr) = 0;

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclCommonSettingsIntf::bSetDipoEnableSetVal(...)
   ***************************************************************************/
   /*!
   * \fn      vSetDipoEnableSetVal(tenEnabledInfo enDipoEnableSetVal)
   * \brief   Called to set the default value to enable Dipo usage.
   * \param   [IN] enDipoEnableSetVal: default value for enabling Dipo setting
   * \retval  t_Bool
   **************************************************************************/
   virtual t_Bool bSetDipoEnableSetVal(tenEnabledInfo enDipoEnableSetVal) = 0;

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclCommonSettingsIntf::vSetMLLinkEnableSetVal()
   ***************************************************************************/
   /*!
   * \fn      vSetMLLinkEnableSetVal(tenEnabledInfo enMLLinkEnableSetVal)
   * \brief   Called to set the default value to enable ML usage.
   * \param   [IN] enMLLinkEnableSetVal: default value for enabling ML link setting
   * \retval  t_Bool
   **************************************************************************/
   virtual t_Bool bSetMLLinkEnableSetVal(tenEnabledInfo enMLLinkEnableSetVal) = 0;

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclCommonSettingsIntf::bSetAAPEnableSetVal()
   ***************************************************************************/
   /*!
   * \fn      bSetAAPEnableSetVal(tenEnabledInfo enAAPEnableSetVal)
   * \brief   Called to set the default value to enable AAP usage.
   * \param   [IN] enAAPEnableSetVal: default value for enabling AAP setting
   * \retval  t_Bool
   **************************************************************************/
   virtual t_Bool bSetAAPEnableSetVal(tenEnabledInfo enAAPEnableSetVal) = 0;

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclCommonSettingsIntf::bSetMySPINEnableSetVal()
   ***************************************************************************/
   /*!
   * \fn      bSetMySPINEnableSetVal(tenEnabledInfo enmySPINLinkEnable)
   * \brief   Called to set the default value to enable AAP usage.
   * \param   [IN] enmySPINLinkEnable: default value for enabling Myspin setting
   * \retval  t_Bool
   **************************************************************************/
   virtual t_Bool bSetMySPINEnableSetVal(tenEnabledInfo enmySPINLinkEnablel) = 0;

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclCommonSettingsIntf::vSetCarlifeEnableSetVal()
   ***************************************************************************/
   /*!
   * \fn      vSetCarlifeEnableSetVal(tenEnabledInfo enCarlifeEnableSetVal)
   * \brief   Called to set the default value to enable AAP usage.
   * \param   [IN] enCarlifeEnableSetVal: default value for enabling Carlife setting
   * \retval  t_Bool
   **************************************************************************/
   virtual t_Bool bSetCarlifeEnableSetVal(tenEnabledInfo enCarlifeEnableSetVal) = 0;

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclCommonSettingsIntf::vSetAndroidDeviceTechnologyPreference()
   ***************************************************************************/
   /*!
   * \fn      vSetAndroidDeviceTechnologyPreference(tenDeviceCategory enAndroidDeviceTechPrefVal)
   * \brief   Called to set the default value to technology preference.
   * \param   [IN] enAndroidDeviceTechPrefVal: default value -technology preference for Android Device
   * \retval  t_Bool
   **************************************************************************/
   virtual t_Bool bSetAndroidDeviceTechnologyPreference(tenDeviceCategory enAndroidDeviceTechPrefVal) = 0;

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclCommonSettingsIntf::bSetAppleDeviceTechnologyPreference()
   ***************************************************************************/
   /*!
   * \fn      bSetAppleDeviceTechnologyPreference(tenDeviceCategory enAppleDeviceTechPrefVal)
   * \brief   Called to set the default value to technology preference.
   * \param   [IN] enAppleDeviceTechPrefVal: default value -technology preference for Apple Device
   * \retval  t_Bool
   **************************************************************************/
   virtual t_Bool bSetAppleDeviceTechnologyPreference(tenDeviceCategory enAppleDeviceTechPrefVal) = 0;

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclCommonSettingsIntf::vClearVehicleId()
    ***************************************************************************/
   /*!
    * \fn      vClearVehicleId()
    * \brief   Method to set VehicleId information
    * \retval  t_Void
    **************************************************************************/
   virtual t_Void vClearVehicleId() = 0;

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclCommonSettingsIntf::vSetScreenConfigs()
   ***************************************************************************/
   /*!
   * \fn      t_Void vSetScreenConfigs(const trDisplayAttributes& corfrDispLayerAttr)
   * \brief   Method to set screen attributes
   * \param   corfrDispLayerAttr : [IN] Display attributes
   * \retval  t_Void
   **************************************************************************/
   virtual t_Void vSetScreenConfigs(const trDisplayAttributes& corfrDispLayerAttr) = 0;

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclCommonSettingsIntf::vSetGeneralRestrictions(const t_U8 cou8GeneralRestrictionInfo)
   ***************************************************************************/
   /*!
   * \fn      t_Void vSetGeneralRestrictions(const t_U8 cou8GeneralRestrictionInfo)
   * \brief   Method to set General Restriction
   * \param   NONE
   * \retval  t_Void
   **************************************************************************/
   virtual t_Void vSetGeneralRestrictions(tenDeviceCategory enDeviceCategory,const t_U16 u16GeneralRestrictionInfo) = 0;

   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclCommonSettingsIntf::bWriteOnCarEnableSetting
    ***************************************************************************/
   /*!
    * \fn      bWriteOnCarEnableSetting(tenEnabledInfo enOnCarLinkEnable)
    * \brief   Sets the OnCar On/Off setting value in datapool.
    *          (True - if enabled, else False)
    * \param   [IN] enOnCarLinkEnable :
    *               e8USAGE_DISABLED - to set value as OFF
    *               e8USAGE_ENABLED - to set value as ON
    * \retval  t_Bool
    ***************************************************************************/
   virtual t_Bool bWriteOnCarEnableSetting(tenEnabledInfo enOnCarLinkEnable) = 0;

   /***************************************************************************
	** FUNCTION: t_Bool spi_tclCommonSettingsIntf::bGetITCommanderSupport())
	***************************************************************************/
   /*!
	* \fn      t_Bool bGetITCommanderSupport()
	* \brief   Function to Get ITCommander Support
	* \retval  NONE
	**************************************************************************/
   virtual t_Bool bGetITCommanderSupport() = 0;
   
   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclCommonSettingsIntf::bGetCarlifeIOSFeatureSupport
    **************************************************************************/
   /*!
    * \fn     t_Bool bGetCarlifeIOSFeatureSupport()
    * \brief  Method to retrieve Carlife Ios enable info from policy
    * \retval t_Bool
    **************************************************************************/
   virtual t_Bool bGetCarlifeIOSFeatureSupport() = 0;   

	/***************************************************************************
   ** FUNCTION: t_U8 spi_tclCommonSettingsIntf::u8ReadOEMBrandFromKDS()
   ***************************************************************************/
   /*!
   * \fn      t_U8 u8ReadOEMBrandFromKDS()
   * \brief   Method to read the KDS value and get the Brand type info
   * \retval  NONE
   **************************************************************************/
   virtual t_U8 u8ReadOEMBrandFromKDS() = 0;

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclCommonSettingsIntf::vGetVehicleInfoDataMySPIN(trVehicleInfo& rfrVehicleInfo)
   ***************************************************************************/
   /*!
   * \fn      t_Void vGetVehicleInfoDataMySPIN()
   * \brief   Method to get the vehicle manufacturer name and variant info
   * \param   rfrVehicleInfo: [IN]Vehicle Info Data
   * \retval  NONE
   **************************************************************************/
   virtual t_Void vGetVehicleInfoDataMySPIN(trVehicleInfo& rfrVehicleInfo) = 0;

   /***************************************************************************
   FUNCTION: t_U16 spi_tclCommonSettingsIntf::u16ReadVehicleTypeFromKDS())
   ***************************************************************************/
   /*!
   * \fn      t_U16 u16ReadVehicleTypeFromKDS()
   * \brief   Function to Get Vehicle Type
   * \retval  t_U16
   ***************************************************************************/
   virtual t_U16 u16ReadVehicleTypeFromKDS() = 0;

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclCommonSettingsIntf::vGetVehicleBTAddress()
   ***************************************************************************/
   /*!
   * \fn     vGetVehicleBTAddress()
   * \brief  Function to get the vehicle BT address.
   * \param  szVehicleBTAddress:[OUT] Vehicle BT address
   * \retval  None
   **************************************************************************/
   virtual t_Void vGetVehicleBTAddress(t_String &szVehicleBTAddress) = 0;

   /***************************************************************************
   ** FUNCTION: t_Bool spi_tclCommonSettingsIntf::bGetNativeNavigationEnabled()
   ***************************************************************************/
   /*!
   * \fn      t_Bool bGetNativeNavigationEnabled()
   * \brief  Method to retrieve setting for native navigation
   * \param [OUT] : none
   * \retval t_Bool
   **************************************************************************/
   virtual t_Bool bGetNativeNavigationEnabled() const = 0;

    /***************************************************************************
    ****************************END OF PUBLIC *********************************
    ***************************************************************************/

};

//! Callback for setting CommonsettingInstance
typedef std::function<t_Void(spi_tclCommonSettingsIntf*)>   vSetCommonsettingInstance;
//! \brief   Structure holding the callbacks to subscriber.
//!          Used by CmdInterface to set Commonsetting instance
struct trCmdInterfaceCallback
{
	   //! Called when Sensor data is received by client handler.
	   vSetCommonsettingInstance     fvSetCommonsettingInstance ;

	   trCmdInterfaceCallback() : fvSetCommonsettingInstance (NULL)
	   {
	   }
};

#endif //SPI_TCLCOMMONSETTINGSINTF_
