/***********************************************************************/
/*!
* \file  spi_tclAudioSettingsIntf.h
* \brief  Implementation of the Class spi_tclAudioSettingsIntf
*************************************************************************
\verbatim


PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    Interface class for Audio Settings
AUTHOR:         kmo6kor
COPYRIGHT:      &copy; 2017 Robert Bosch Car Multimedia GmbH
HISTORY:
Date        | Author                | Modification
             kmo6kor			    Initial Version                                               
\endverbatim
*************************************************************************/


/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/
/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/
#ifndef SPI_TCLAUDIOSETTINGSINTF_
#define SPI_TCLAUDIOSETTINGSINTF_

#include "spi_tclCommonSettingsIntf.h"

class spi_tclAudioSettingsIntf : public virtual spi_tclCommonSettingsIntf
{

public:

   /***************************************************************************
   **************************** PUBLIC *********************************
   ***************************************************************************/

   /***************************************************************************
	** FUNCTION: spi_tclAudioSettingsIntf::spi_tclAudioSettingsIntf()
	***************************************************************************/
   /*!
	* \fn      spi_tclAudioSettingsIntf
	* \brief   Default Constructor
	**************************************************************************/
   spi_tclAudioSettingsIntf(){}

   /***************************************************************************
	** FUNCTION:  spi_tclAudioSettingsIntf::~spi_tclAudioSettingsIntf()
	***************************************************************************/
   /*!
	* \fn      virtual ~spi_tclAudioSettingsIntf()
	* \brief   Destructor
	**************************************************************************/
   virtual ~spi_tclAudioSettingsIntf(){}

   /***************************************************************************
    ** FUNCTION:  t_U8 spi_tclAudioSettingsIntf::u8GetSourceNumber(tenAudioDir)
    ***************************************************************************/
   /*!
    * \fn      u8GetSourceNumber( tenAudioDir enAudDir)
    * \brief   Function to get the Source Number for Audio Link & selected device
    *          category.
    * \param   [enAudDir]: Audio Link Direction
    * \retval  t_U8 value: Source Number
    **************************************************************************/
   virtual t_U8 u8GetSourceNumber(tenAudioDir enAudDir) const = 0;

   /***************************************************************************
    ** FUNCTION: t_Void spi_tclAudioSettingsIntf::vGetRTPPayloadType
    (t_U32& rfu32RTPPayload)
    ***************************************************************************/
   /*!
    * \fn      t_Void vGetRTPPayloadType(t_U32& u32RTPPayload)
    * \brief   Method to get the RTPPayload type
    * \param   [rfu32RTPPayload]: Reference to the RTP Payload type
    * \retval  NONE
    **************************************************************************/
   virtual t_Void vGetRTPPayloadType(t_U32& rfu32RTPPayload) const = 0;

   /***************************************************************************
    ** FUNCTION: t_Void spi_tclAudioSettingsIntf::vGetAudioIPL
    (tU32& rfu32AudioIPL)
    ***************************************************************************/
   /*!
    * \fn      t_Void vGetAudioIPL(tU32& rfu32AudioIPL)
    * \brief   Method to get the Audio IPL
    * \param   [rfu32AudioIPL]: Reference to the Audio IPL
    * \retval  t_Bool value: TRUE if Setting is Valid, FALSE Otherwise
    **************************************************************************/
   virtual t_Void vGetAudioIPL(t_U32& rfu32AudioIPL) const = 0;

   /***************************************************************************
   ** FUNCTION: t_Bool spi_tclAudioSettingsIntf::bGetAudioBlockingTriggerBasedOnCat()
   ***************************************************************************/
   /*!
   * \fn      t_Bool bGetAudBlockTriggerBasedOnCat()
   * \brief   Method to get the Audio Blocking enable/disable trigger based on
   *          Application category or not. This feature works only in conjunction with the
   *          Dynamic Audio feature
   * \retval  t_Bool
   **************************************************************************/
   virtual t_Bool bGetAudBlockTriggerBasedOnCat() const = 0;

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclAudioSettingsIntf::vSetAudBlockTriggerBasedOnCat()
   ***************************************************************************/
   /*!
   * \fn      vSetAudBlockTriggerBasedOnCat()
   * \brief   Method to Set the Audio Blocking enable/disable trigger based on
   *          Application category or not. This feature works only in conjunction with the
   *          Dynamic Audio feature
   * \retval  t_Void
   **************************************************************************/
   virtual t_Void vSetAudBlockTriggerBasedOnCat(t_Bool bIsML11OrAboveDevice) = 0;

   /***************************************************************************
   ** FUNCTION: t_Bool spi_tclAudioSettingsIntf::bGetAudBlockTriggerBasedOnGlobaleMute()
   ***************************************************************************/
   /*!
   * \fn      t_Bool bGetAudBlockTriggerBasedOnGlobaleMute()
   * \brief   Method to get the Audio Blocking enable/disable trigger based on
   *          Global mute is enabled or not
   * \retval  t_Bool
   **************************************************************************/
   virtual t_Bool bGetAudBlockTriggerBasedOnGlobaleMute() const = 0;

   /***************************************************************************
    ** FUNCTION: t_Void spi_tclAudioSettingsIntf::vGetAudioPipeConfig()
    ***************************************************************************/
   /*!
    * \fn     t_Void vGetAudioPipeConfig()
    * \brief  provides audio configuration for the specified audio steam
    * \param  [OUT]rfrmapAudioPipeConfig : returns audio configuration as string
    **************************************************************************/
   virtual t_Void vGetAudioPipeConfig(tmapAudioPipeConfig &rfmapAudioPipeConfig) = 0;

   /***************************************************************************
    ** FUNCTION: t_Void spi_tclAudioSettingsIntf::vSetMLDynAudioSupport()
    ***************************************************************************/
   /*!
    * \fn     vSetMLDynAudioSupport() const
    * \brief  To Set whether ML Audio is supported or not.
    *         (For ML 1.0 = Not Supported, For Ml 1.1 onwards = supported)
    * \retval  t_Void
    **************************************************************************/
   virtual t_Void vSetMLDynAudioSupport(t_Bool bIsML11OrAboveDevice) = 0;

   /***************************************************************************
   ** FUNCTION: t_Bool spi_tclAudioSettingsIntf::bGetAAPMediaStreamRecEnabled()
   ***************************************************************************/
   /*!
    * \fn     t_Bool bGetAAPMediaStreamRecEnabled() const
    * \brief  To Get whether AAP Media audio stream recording is enabled or not.
    * \retval  t_Bool
    **************************************************************************/
   virtual t_Bool bGetAAPMediaStreamRecEnabled() const = 0;

   /***************************************************************************
   ** FUNCTION: t_Bool spi_tclAudioSettingsIntf::bGetAAPGuidanceStreamRecEnabled()
   ***************************************************************************/
   /*!
    * \fn     t_Bool bGetAAPGuidanceStreamRecEnabled() const
    * \brief  To Get whether AAP Guidance audio stream recording is enabled or not.
    * \retval  t_Bool
    **************************************************************************/
   virtual t_Bool bGetAAPGuidanceStreamRecEnabled() const = 0;

   /***************************************************************************
   ** FUNCTION: t_String spi_tclAudioSettingsIntf::szGetAudSinkMaxUnackedFrames()
   ***************************************************************************/
   /*!
    * \fn     szGetAudSinkMaxUnackedFrames() const
    * \brief  To Get "max-unacked-frames" value for Audio Sink configuration
    * \retval  t_String
    **************************************************************************/
   virtual t_String szGetAudSinkMaxUnackedFrames() const = 0;

   /***************************************************************************
   ** FUNCTION: t_Bool spi_tclAudioSettingsIntf::bGetRTPInSupport()
   ***************************************************************************/
   /*!
    * \fn     t_Bool bGetRTPInSupport() const
    * \brief  To Get whether RTPIn support is enabled or not.
    * \retval  t_Bool
    **************************************************************************/
   virtual t_Bool bGetRTPInSupport() const = 0;

   /***************************************************************************
   ** FUNCTION: t_String spi_tclAudioSettingsIntf::szGetDisableRealTimePriorityAudio()
   ***************************************************************************/
   /*!
   * \fn     szGetDisableRealTimePriorityAudio() const
   * \brief  To Get the value of disabling real time priority audio for Audio Sink configuration
   * \retval  t_String
   **************************************************************************/
   virtual t_String szGetDisableRealTimePriorityAudio() const = 0;

   /***************************************************************************
   ** FUNCTION: t_String spi_tclAudioSettingsIntf::szGetAudThreadsRealTimePriority()
   ***************************************************************************/
   /*!
   * \fn     szGetAudThreadsRealTimePriority() const
   * \brief  To Get Audio threads Real time priority value for Audio Sink configuration
   * \retval  t_String
   **************************************************************************/
   virtual t_String szGetAudThreadsRealTimePriority() const = 0;

   /***************************************************************************
   ** FUNCTION: t_String spi_tclAudioSettingsIntf::szGetAudSinkThresholdMs()
   ***************************************************************************/
   /*!
   * \fn     szGetAudSinkThresholdMs() const
   * \brief  To Get Audio Sink Threshold configuration
   * \retval  t_String
   **************************************************************************/
   virtual t_String szGetAudSinkThresholdMs() const = 0;

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclAudioSettingsIntf::vGetBdclMediaSinkConfig()
   ***************************************************************************/
   /*!
   * \fn     vGetBdclMediaSinkConfig() const
   * \brief  To Get BDCL Media Audio Sink configuration
   * \retval  t_Void
   **************************************************************************/
   virtual t_Void vGetBdclMediaSinkConfig(std::map< t_String, t_String >& mapAudioConfig) const = 0;

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclAudioSettingsIntf::vGetBdclTTSSinkConfig()
   ***************************************************************************/
   /*!
   * \fn     vGetBdclTTSSinkConfig() const
   * \brief  To Get BDCL TTS Audio Sink configuration
   * \retval  t_Void
   **************************************************************************/
   virtual t_Void vGetBdclTTSSinkConfig(std::map< t_String, t_String >& mapAudioConfig) const = 0;

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclAudioSettingsIntf::vGetBdclVRSinkConfig()
   ***************************************************************************/
   /*!
   * \fn     vGetBdclVRSinkConfig() const
   * \brief  To Get BDCL VR Audio Sink configuration
   * \retval  t_Void
   **************************************************************************/
   virtual t_Void vGetBdclVRSinkConfig(std::map< t_String, t_String >& mapAudioConfig) const = 0;

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclAudioSettingsIntf::vGetBdclVRSourceConfig()
   ***************************************************************************/
   /*!
   * \fn     vGetBdclVRSourceConfig() const
   * \brief  To Get BDCL VR Audio Source configuration
   * \retval  t_Void
   **************************************************************************/
   virtual t_Void vGetBdclVRSourceConfig(std::map< t_String, t_String >& mapAudioConfig) const = 0;

   /***************************************************************************
   ** FUNCTION: t_Bool spi_tclAudioSettingsIntf::bGetAudioMuteEnabled()
   ***************************************************************************/
   /*!
    * \fn     t_Bool bGetAudioMuteEnabled() const
    * \brief  To Get whether audio mute is enabled (during last mode audio)
    * \retval  t_Bool
    **************************************************************************/
   virtual t_Bool bGetAudioMuteEnabled() const = 0;

   /***************************************************************************
   ** FUNCTION: t_Bool spi_tclAudioSettingsIntf::bGetVolumeLockEnabled()
   ***************************************************************************/
   /*!
    * \fn     t_Bool bGetVolumeLockEnabled() const
    * \brief  To Get whether Volume lock is enabled (during last mode audio)
    * \retval  t_Bool
    **************************************************************************/
   virtual t_Bool bGetVolumeLockEnabled() const = 0;

   /***************************************************************************
   ** FUNCTION: t_Bool spi_tclAudioSettingsIntf::bGetMLDynAudioApplicability()
   ***************************************************************************/
   /*!
    * \fn     t_Bool bGetMLDynAudioApplicability() const
    * \brief  To Get whether ML Dnamic Audio is applicable or not from policy.
    * \retval  t_Bool
    **************************************************************************/
   virtual t_Bool bGetMLDynAudioApplicability() const = 0;

   /***************************************************************************
    ** FUNCTION: t_Void spi_tclAudioSettingsIntf::vDisplayAudioSettings()
    ***************************************************************************/
   /*!
    * \fn     t_Void vDisplayAudioSettings()
    * \brief  To print the audio settings data
    * \retval  t_Void
    **************************************************************************/
   virtual t_Void vDisplayAudioSettings() = 0;
   /***************************************************************************
   ** FUNCTION:  tenATBoxSupportedState spi_tclAudioSettingsIntf::vGetATBoxSupportedState()
   ***************************************************************************/
   /*!
   * \fn      tenATBoxSupportedState vGetATBoxSupportedState()
   * \brief   Method to read supported AT Box features info.
   * \param   NONE
   * \retval  tenATBoxSupportedState
   **************************************************************************/
   virtual tenATBoxSupportedState vGetATBoxSupportedState() const = 0;
   /***************************************************************************
   ****************************END OF PUBLIC *********************************
   ***************************************************************************/

};
#endif //SPI_TCLAUDIOSETTINGSINTF_
