/*!
 *******************************************************************************
 * \file              spi_tclStaticConfigurationBase.cpp
 * \brief             Configuration manager for SPI
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    common base class for SPI Static Configuration
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 07.06.2018 |  Kavya Mogeri                | Initial Version

 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/

#include "spi_tclStaticConfigurationBase.h"

//! Includes for Trace files
#include "Trace.h"
#include "FileHandler.h"
#include "XmlDocument.h"
#include "XmlReader.h"
#include "StringHandler.h"

#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_CONFIGREADER
#include "trcGenProj/Header/spi_tclStaticConfigurationBase.cpp.trc.h"
#endif
#endif

   static const t_Char* pczConfigFilePath = "/opt/bosch/spi/xml/policy.xml";

//! Defines for Apple specific HID descriptor to readable
#define HID_USAGE_PAGE(a)                  0x05, (a)

//#ifdef USING_HID_8BIT
#define HID_USAGE(a)                       0x09, (a)
//#endif

//#ifdef USING_HID_16BIT
#define HID_USAGE_16BIT(a, b)              0x0A, (a), (b)
//#endif

#define HID_COLLECTION(a)                  0xA1, (a)
#define HID_END_COLLECTION()               0xC0
#define HID_REPORT_SIZE(size)              0x75, (size)
#define HID_REPORT_COUNT(count)            0x95, (count)
#define HID_INPUT(a)                       0x81, (a)
#define HID_LOGICAL_MINMAX(min, max)       0x15, (min), 0x25, (max)

static const t_Char sczKeyId            [] = "10200";
static const t_Char szCertificateFolderPath  [] = "/opt/bosch/spi/AAuto/certificate/default";

static const t_String sczCarplayConfigPath = "/var/opt/bosch/dynamic/spi/dipo.cfg";
static const t_String sczCarplayUSBPath = "/var/opt/bosch/dynamic/spi/dipo_usb.cfg";

//Client Manufacturer Name
static const t_String sczHeadUnitManfacturerName = "Bosch CM";

//!Android Auto Certificates Details
struct trAAutoCertConfigData
{
   t_String szCertificateModel; //KDS Value
   t_String szCertificateName; //Certificate Name for the Reference - to avoid KDS Ambiguity - as KDS Value could be Hex Value
   t_String szKeyId; // KeyId - Which was used to wrap the certificates
   t_String szCertificateFolderPath; // Certificates Path in Target
};

#define AAAUTO_CERTS_CONFIG
static trAAutoCertConfigData aAAAutoCertsConfig[]=
#include "spi_tclCertificateConfig.cfg"
#undef AAAUTO_CERTS_CONFIG

using namespace shl::xml;

/***************************************************************************
** FUNCTION:  spi_tclStaticConfigurationBase::spi_tclStaticConfigurationBase()
***************************************************************************/
spi_tclStaticConfigurationBase::spi_tclStaticConfigurationBase(spi_tclCalibrationReaderBase *poCalibrationReader):
m_poCalibrationReader(poCalibrationReader),
m_poAppSettings(NULL),
m_poConnSettings(NULL),
m_poAudioSettings(NULL),
m_poDataServiceSettings(NULL),
m_poResourceMngrSettings(NULL),
m_poVideoSettings(NULL),
m_poDefaultSettings(NULL),
m_poDiscovererSettings(NULL),
m_enStoredSelectionMode(e16DEVICESEL_UNKNOWN),
m_enAndroidDevicePreference(e8DEV_TYPE_UNKNOWN),
m_enAppleDevicePreference(e8DEV_TYPE_UNKNOWN),
m_enDriveModeInfo(e8PARK_MODE),
m_enNightModeInfo(e8_DAY_MODE),
m_u8AAPParkModeRestrictionInfo(0),
m_u8AAPDriveModeRestrictionInfo(0)
{

   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::spi_tclStaticConfigurationBase Entered"));
}
/***************************************************************************
** FUNCTION:  spi_tclStaticConfigurationBase::~spi_tclStaticConfigurationBase()
***************************************************************************/
spi_tclStaticConfigurationBase::~spi_tclStaticConfigurationBase(){

   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::~spi_tclStaticConfigurationBase Entered"));
   RELEASE_MEM(m_poCalibrationReader);
   RELEASE_MEM(m_poAppSettings);
   RELEASE_MEM(m_poConnSettings);
   RELEASE_MEM(m_poAudioSettings);
   RELEASE_MEM(m_poDataServiceSettings);
   RELEASE_MEM(m_poResourceMngrSettings);
   RELEASE_MEM(m_poVideoSettings);
   RELEASE_MEM(m_poDefaultSettings);
   RELEASE_MEM(m_poDiscovererSettings);
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vReadSPISettings()
***************************************************************************/
t_Void spi_tclStaticConfigurationBase::vReadSPISettings()
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vReadSPISettings"));
   t_String szDeviceExclusionListPath;
   t_String szDeviceInclusionListPath;
   
   //Check the validity of the xml file
   spi::io::FileHandler oPolicySettingsFile(pczConfigFilePath, spi::io::SPI_EN_RDONLY);

   if (true == oPolicySettingsFile.bIsValid())
   {

      m_poAppSettings = new spi_tclAppSettings();
      m_poConnSettings = new spi_tclConnSettings();
      m_poAudioSettings = new spi_tclAudioSettings();
      m_poDataServiceSettings = new spi_tclDataServiceSettings();
      m_poResourceMngrSettings = new spi_tclResourceMngrSettings();
      m_poVideoSettings = new spi_tclVideoSettings();
      m_poDefaultSettings = new spi_tclDefaultSettings();
      m_poDiscovererSettings = new spi_tclDiscovererSettings();

      tclXmlDocument oXmlDoc(pczConfigFilePath);

      if( NULL != m_poConnSettings)
      {
         m_poConnSettings->vStoreSelectionModefromDatapool(m_enStoredSelectionMode);
         m_poConnSettings->vStoreDevicePreferencefromDatapool(m_enAndroidDevicePreference,m_enAppleDevicePreference);
         tclXmlReader oXmlReaderConn(&oXmlDoc, m_poConnSettings);
         oXmlReaderConn.bRead("CONNMNGR");
         m_poConnSettings->vDisplayConnSettings();
         m_poConnSettings->vGetDeviceExclusionListPath(szDeviceExclusionListPath);
         m_poConnSettings->vGetDeviceInclusionListPath(szDeviceInclusionListPath);
      }
      if( NULL != m_poAppSettings)
      {
         tclXmlReader oXmlReaderApp(&oXmlDoc, m_poAppSettings);
         oXmlReaderApp.bRead("APPMNGR");
         m_poAppSettings->vDisplayAppSettings();
      }
      if( NULL != m_poVideoSettings)
      {
         tclXmlReader oXmlReaderVid(&oXmlDoc, m_poVideoSettings);
         oXmlReaderVid.bRead("VIDEO");
         m_poVideoSettings->vDisplayVideoSettings();
      }

      if( NULL != m_poAudioSettings)
      {
         tclXmlReader oXmlReaderAud(&oXmlDoc, m_poAudioSettings);
         oXmlReaderAud.bRead("AUDIO");
         m_poAudioSettings->vDisplayAudioSettings();
      }

      if( NULL != m_poResourceMngrSettings)
      {
         tclXmlReader oXmlReaderResource(&oXmlDoc, m_poResourceMngrSettings);
         oXmlReaderResource.bRead("RESOURCEMNGR");
      }

      if( NULL != m_poDataServiceSettings)
      {
         tclXmlReader oXmlReaderData(&oXmlDoc, m_poDataServiceSettings);
         oXmlReaderData.bRead("DATASERVICE");
      }

      if( NULL != m_poDefaultSettings)
      {
         tclXmlReader oXmlReaderDefSet(&oXmlDoc, m_poDefaultSettings);
         oXmlReaderDefSet.bRead("DEFSET");
      }
      if( NULL != m_poDiscovererSettings)
      {
         m_poDiscovererSettings->vSetDeviceListPath(szDeviceInclusionListPath, szDeviceExclusionListPath);
      }


   } // if (true == oPolicySettingsFile.bIsValid())
}


/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bNonCertAppsReq()
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bGetNonCertAppsReq() const
{

   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bGetNonCertAppsReq"));
   t_Bool bNonCertAppsReq  = false;
   if(NULL != m_poAppSettings )
   {
      bNonCertAppsReq = m_poAppSettings->bNonCertAppsReq();
   }
   return bNonCertAppsReq;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetAAPNavigationEnabledInfo()
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bGetAAPNavigationEnabledInfo() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bGetAAPNavigationEnabledInfo"));
   t_Bool bAAPNavigationEnabledInfo = false;
   if(NULL != m_poAppSettings )
   {
      bAAPNavigationEnabledInfo = m_poAppSettings->bEnableAAPNavigation();
   }
   return bAAPNavigationEnabledInfo;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetAAPMediaPlaybackEnabledInfo()
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bGetAAPMediaPlaybackEnabledInfo() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bGetAAPMediaPlaybackEnabledInfo"));
   t_Bool bAAPMediaPlaybackEnabledInfo = false;
   if(NULL != m_poAppSettings )
   {
      bAAPMediaPlaybackEnabledInfo = m_poAppSettings->bEnableAAPMediaPlayback();
   }
   return bAAPMediaPlaybackEnabledInfo;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetAAPNotificationUpdatesEnabledInfo()
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bGetAAPNotificationUpdatesEnabledInfo() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bGetAAPNotificationUpdatesEnabledInfo"));
   t_Bool bAAPNotificationUpdatesEnabledInfo = false;
   if(NULL != m_poAppSettings )
   {
      bAAPNotificationUpdatesEnabledInfo = m_poAppSettings->bEnableAAPNotification();
   }
   return bAAPNotificationUpdatesEnabledInfo;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetTBTUpdatesEnabledInfo()
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bGetTBTUpdatesEnabledInfo() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bGetTBTUpdatesEnabledInfo"));
   t_Bool bTBTUpdatesEnabledInfo = false;
   if(NULL != m_poAppSettings )
   {
      bTBTUpdatesEnabledInfo = m_poAppSettings->bGetTBTUpdatesEnabledInfo();
   }
   return bTBTUpdatesEnabledInfo;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetAssisstantGuideUpdatesEnabledInfo()
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bGetAssisstantGuideUpdatesEnabledInfo() const
{ 
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bGetAssisstantGuideUpdatesEnabledInfo"));
   t_Bool bAssisstantGuideUpdatesEnabledInfo = false;
   if(NULL != m_poAppSettings )
   {
      bAssisstantGuideUpdatesEnabledInfo = m_poAppSettings->bGetAssisstantGuideUpdatesEnabledInfo();
   }
   return bAssisstantGuideUpdatesEnabledInfo;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vSetRegion()
***************************************************************************/
t_Void spi_tclStaticConfigurationBase::vSetRegion(tenRegion enRegion)
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vSetRegion"));
   if(NULL != m_poAppSettings )
   {
      m_poAppSettings->vSetRegion(enRegion);
   }
}

/***************************************************************************
** FUNCTION:  t_String spi_tclStaticConfigurationBase::szGetRegion()
***************************************************************************/
t_String spi_tclStaticConfigurationBase::szGetRegion() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::szGetRegion"));
   t_String szRegion;
   if(NULL != m_poAppSettings )
   {
      szRegion = m_poAppSettings->szGetRegion();
   }
   return szRegion;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetNonCertAppAllowedInfo()
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bGetNonCertAppAllowedInfo() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bGetNonCertAppAllowedInfo"));
   t_Bool bNonCertAppInfo = false;
   if(NULL != m_poAppSettings )
   {
     bNonCertAppInfo = m_poAppSettings->bGetNonCertAppAllowedInfo();
   }
   return bNonCertAppInfo;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bEnableCTSTest()
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bEnableCTSTest()
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bEnableCTSTest"));
   t_Bool bCTSTest = false;
   if(NULL != m_poAppSettings )
   {
      bCTSTest = m_poAppSettings->bEnableCTSTest();
   }
   return bCTSTest;
}

/***************************************************************************
** FUNCTION:  t_U8 spi_tclStaticConfigurationBase::u8GetSourceNumber()
***************************************************************************/
t_U8 spi_tclStaticConfigurationBase::u8GetSourceNumber(tenAudioDir enAudDir) const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::u8GetSourceNumber"));
   t_U8 u8SourceNumber = 0;
   if(NULL != m_poAudioSettings )
   {
      u8SourceNumber = m_poAudioSettings->u8GetSourceNumber(enAudDir);
   }
   return u8SourceNumber;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vGetRTPPayloadType()
***************************************************************************/
t_Void spi_tclStaticConfigurationBase::vGetRTPPayloadType(t_U32& rfu32RTPPayload) const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vGetRTPPayloadType"));
   if(NULL != m_poAudioSettings )
   {
      m_poAudioSettings->vGetRTPPayloadType(rfu32RTPPayload);
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vGetAudioIPL()
***************************************************************************/
t_Void spi_tclStaticConfigurationBase::vGetAudioIPL(t_U32& rfu32AudioIPL) const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vGetAudioIPL"));
   if(NULL != m_poAudioSettings )
   {
      m_poAudioSettings->vGetAudioIPL(rfu32AudioIPL);
   }
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetAudBlockTriggerBasedOnCat()
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bGetAudBlockTriggerBasedOnCat() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bGetAudBlockTriggerBasedOnCat"));
   t_Bool bAudBlockTrigger = false;
   if(NULL != m_poAudioSettings )
   {
     bAudBlockTrigger = m_poAudioSettings->bGetAudBlockTriggerBasedOnCat();
   }
   return bAudBlockTrigger;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vSetAudBlockTriggerBasedOnCat()
***************************************************************************/
t_Void spi_tclStaticConfigurationBase::vSetAudBlockTriggerBasedOnCat(t_Bool bIsML11OrAboveDevice)
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vSetAudBlockTriggerBasedOnCat"));
   if(NULL != m_poAudioSettings )
   {
      m_poAudioSettings->vSetAudBlockTriggerBasedOnCat(bIsML11OrAboveDevice);
   }
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetAudBlockTriggerBasedOnGlobaleMute()
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bGetAudBlockTriggerBasedOnGlobaleMute() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bGetAudBlockTriggerBasedOnGlobaleMute"));
   t_Bool bAudBlockGlobaleMute = false;
   if(NULL != m_poAudioSettings )
   {
      bAudBlockGlobaleMute = m_poAudioSettings->bGetAudBlockTriggerBasedOnGlobaleMute();
   }
   return bAudBlockGlobaleMute;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vGetAudioPipeConfig()
***************************************************************************/
t_Void spi_tclStaticConfigurationBase::vGetAudioPipeConfig(
tmapAudioPipeConfig &rfmapAudioPipeConfig)
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vGetAudioPipeConfig"));
   if(NULL != m_poAudioSettings )
   {
      m_poAudioSettings->vGetAudioPipeConfig(rfmapAudioPipeConfig);
   }

}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetMLDynAudioSupport()
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bGetMLDynAudioSupport() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bGetMLDynAudioSupport"));
   t_Bool bMLDynAudioSupport = false;
   if(NULL != m_poAudioSettings )
   {
      bMLDynAudioSupport = m_poAudioSettings->bGetMLDynAudioSupport();
   }
   return bMLDynAudioSupport;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vSetMLDynAudioSupport()
***************************************************************************/
t_Void spi_tclStaticConfigurationBase::vSetMLDynAudioSupport(t_Bool bIsML11OrAboveDevice)
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vSetMLDynAudioSupport"));
   if(NULL != m_poAudioSettings )
   {
      m_poAudioSettings->vSetMLDynAudioSupport(bIsML11OrAboveDevice);
   }
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetAAPMediaStreamRecEnabled()
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bGetAAPMediaStreamRecEnabled() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bGetAAPMediaStreamRecEnabled"));
   t_Bool bAAPMediaStreamRec = false;
   if(NULL != m_poAudioSettings )
   {
     bAAPMediaStreamRec = m_poAudioSettings->bGetAAPMediaStreamRecEnabled();
   }
   return bAAPMediaStreamRec;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetAAPGuidanceStreamRecEnabled()
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bGetAAPGuidanceStreamRecEnabled() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bGetAAPGuidanceStreamRecEnabled"));
   t_Bool bAAPGuidanceStreamRec = false;
   if(NULL != m_poAudioSettings )
   {
      bAAPGuidanceStreamRec = m_poAudioSettings->bGetAAPGuidanceStreamRecEnabled();
   }
   return bAAPGuidanceStreamRec;
}

/***************************************************************************
** FUNCTION:  t_String spi_tclStaticConfigurationBase::szGetAudSinkMaxUnackedFrames()
***************************************************************************/
t_String spi_tclStaticConfigurationBase::szGetAudSinkMaxUnackedFrames() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::szGetAudSinkMaxUnackedFrames"));
   t_String szAudSinkMaxUnackedFrames;
   if(NULL != m_poAudioSettings )
   {
      szAudSinkMaxUnackedFrames = m_poAudioSettings->szGetAudSinkMaxUnackedFrames();
   }
   return szAudSinkMaxUnackedFrames;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetRTPInSupport()
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bGetRTPInSupport() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bGetRTPInSupport"));
   t_Bool bRTPInSupport = false;
   if(NULL != m_poAudioSettings )
   {
     bRTPInSupport = m_poAudioSettings->bGetRTPInSupport();
   }
   return bRTPInSupport;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetMySIPNAdvAudioSupport()
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bGetMySIPNAdvAudioSupport() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bGetMySIPNAdvAudioSupport"));
   t_Bool bMySIPNAdvAudioSupport = false;
   if(NULL != m_poAudioSettings )
   {
     bMySIPNAdvAudioSupport = m_poAudioSettings->bGetMySIPNAdvAudioSupport();
   }
   return bMySIPNAdvAudioSupport;
}

/***************************************************************************
** FUNCTION:  t_String spi_tclStaticConfigurationBase::szGetDisableRealTimePriorityAudio()
***************************************************************************/
t_String spi_tclStaticConfigurationBase::szGetDisableRealTimePriorityAudio() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::szGetDisableRealTimePriorityAudio"));
   t_String szDisableAudio;
   if(NULL != m_poAudioSettings )
   {
     szDisableAudio = m_poAudioSettings->szGetDisableRealTimePriorityAudio();
   }
   return szDisableAudio;
}

/***************************************************************************
** FUNCTION:  t_String spi_tclStaticConfigurationBase::szGetAudThreadsRealTimePriority()
***************************************************************************/
t_String spi_tclStaticConfigurationBase::szGetAudThreadsRealTimePriority() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::szGetAudThreadsRealTimePriority"));
   t_String szAudThreadsRealTimePriority;
   if(NULL != m_poAudioSettings )
   {
      szAudThreadsRealTimePriority = m_poAudioSettings->szGetAudThreadsRealTimePriority();
   }
   return szAudThreadsRealTimePriority;
}

/***************************************************************************
** FUNCTION:  t_String spi_tclStaticConfigurationBase::szGetAudSinkThresholdMs()
***************************************************************************/
t_String spi_tclStaticConfigurationBase::szGetAudSinkThresholdMs() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::szGetAudSinkThresholdMs"));
   t_String szAudSinkThresholdMs;
   if(NULL != m_poAudioSettings )
   {
      szAudSinkThresholdMs = m_poAudioSettings->szGetAudSinkThresholdMs();
   }
   return szAudSinkThresholdMs;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vGetBdclMediaSinkConfig()
***************************************************************************/
t_Void spi_tclStaticConfigurationBase::vGetBdclMediaSinkConfig(std::map< t_String, t_String >& mapAudioConfig) const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vGetBdclMediaSinkConfig"));
   if(NULL != m_poAudioSettings )
   {
      m_poAudioSettings->vGetBdclMediaSinkConfig(mapAudioConfig);
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vGetBdclTTSSinkConfig()
***************************************************************************/
t_Void spi_tclStaticConfigurationBase::vGetBdclTTSSinkConfig(std::map< t_String, t_String >& mapAudioConfig) const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vGetBdclTTSSinkConfig"));
   if(NULL != m_poAudioSettings )
   {
      m_poAudioSettings->vGetBdclTTSSinkConfig(mapAudioConfig);
   }
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::vGetBdclVRSinkConfig()
***************************************************************************/
 t_Void spi_tclStaticConfigurationBase::vGetBdclVRSinkConfig(std::map< t_String, t_String >& mapAudioConfig) const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vGetBdclVRSinkConfig"));
   if(NULL != m_poAudioSettings )
   {
      m_poAudioSettings->vGetBdclVRSinkConfig(mapAudioConfig);
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vGetBdclVRSourceConfig()
***************************************************************************/
t_Void spi_tclStaticConfigurationBase::vGetBdclVRSourceConfig(std::map< t_String, t_String >& mapAudioConfig) const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vGetBdclVRSourceConfig"));
   if(NULL != m_poAudioSettings )
   {
      m_poAudioSettings->vGetBdclVRSourceConfig(mapAudioConfig);
   }
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetAudioMuteEnabled()
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bGetAudioMuteEnabled() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bGetAudioMuteEnabled"));
   t_Bool bAudioMuteEnabled = false;
   if(NULL != m_poAudioSettings )
   {
      bAudioMuteEnabled = m_poAudioSettings->bGetAudioMuteEnabled();
   }
   return bAudioMuteEnabled;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetVolumeLockEnabled()
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bGetVolumeLockEnabled() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bGetVolumeLockEnabled"));
   t_Bool bVolumeLockEnabled = false;
   if(NULL != m_poAudioSettings )
   {
      bVolumeLockEnabled = m_poAudioSettings->bGetVolumeLockEnabled();
   }
   return bVolumeLockEnabled;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetMLDynAudioApplicability()
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bGetMLDynAudioApplicability() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bGetMLDynAudioApplicability"));
   t_Bool bMLDynAudio = false;
   if(NULL != m_poAudioSettings )
   {
      bMLDynAudio = m_poAudioSettings->bGetMLDynAudioApplicability();
   }
   return bMLDynAudio;
}

/***************************************************************************
** FUNCTION:  tenDeviceSelectionMode spi_tclStaticConfigurationBase::enGetDeviceSelectionMode()
***************************************************************************/
tenDeviceSelectionMode spi_tclStaticConfigurationBase::enGetDeviceSelectionMode(
tenDeviceCategory enTechnologyType) const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::enGetDeviceSelectionMode"));
   tenDeviceSelectionMode enDeviceSelectionMode = e16DEVICESEL_UNKNOWN;
   if(NULL != m_poConnSettings )
   {
      enDeviceSelectionMode = m_poConnSettings->enGetDeviceSelectionMode(enTechnologyType);
   }
   return enDeviceSelectionMode;
}

/***************************************************************************
** FUNCTION:  tenDeviceCategory spi_tclStaticConfigurationBase::enGetTechnologyPreference()
***************************************************************************/
tenDeviceCategory spi_tclStaticConfigurationBase::enGetTechnologyPreference(tenDeviceType enDeviceType) const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::enGetTechnologyPreference"));
   tenDeviceCategory enTechnologyPreference = e8DEV_TYPE_DEFAULT;
   if(NULL != m_poConnSettings )
   {
      enTechnologyPreference = m_poConnSettings->enGetTechnologyPreference(enDeviceType);
   }
   return enTechnologyPreference;
}

/***************************************************************************
** FUNCTION:  tenCertificateType spi_tclStaticConfigurationBase::enGetCertificateType()
***************************************************************************/
tenCertificateType spi_tclStaticConfigurationBase::enGetCertificateType() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::enGetCertificateType"));
   tenCertificateType enCertificateType = e8_CERTIFICATETYPE_DEVELOPER;
   if(NULL != m_poConnSettings )
   {
      enCertificateType = m_poConnSettings->enGetCertificateType();
   }
   return enCertificateType;
}

/***************************************************************************
** FUNCTION:  tenSelModePriority spi_tclStaticConfigurationBase::enGetSelectionModePriority()
***************************************************************************/
tenSelModePriority spi_tclStaticConfigurationBase::enGetSelectionModePriority() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::enGetSelectionModePriority"));
   tenSelModePriority enSelectionModePriority = e8PRIORITY_DEVICELIST_HISTORY;
   if(NULL != m_poConnSettings )
   {
      enSelectionModePriority = m_poConnSettings->enGetSelectionModePriority();
   }
   return enSelectionModePriority;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vGetPersistentStoragePath()
***************************************************************************/
t_Void spi_tclStaticConfigurationBase::vGetPersistentStoragePath(t_String &rfszPersStoragePath)
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vGetPersistentStoragePath"));
   if(NULL != m_poConnSettings )
   {
      m_poConnSettings->vGetPersistentStoragePath(rfszPersStoragePath);
   }
}

/***************************************************************************
** FUNCTION:  t_U32 spi_tclStaticConfigurationBase::u32GetDeviceHistorySize()
***************************************************************************/
t_U32 spi_tclStaticConfigurationBase::u32GetDeviceHistorySize() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::u32GetDeviceHistorySize"));
   t_U32 u32DeviceHistorySize = 0;
   if(NULL != m_poConnSettings )
   {
      u32DeviceHistorySize = m_poConnSettings->u32GetDeviceHistorySize();
   }
   return u32DeviceHistorySize;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bIsXMLValidationEnabled()
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bIsXMLValidationEnabled() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bIsXMLValidationEnabled"));
   t_Bool bIsXMLValidation = false;
   if(NULL != m_poConnSettings )
   {
      bIsXMLValidation = m_poConnSettings->bIsXMLValidationEnabled();
   }
   return bIsXMLValidation;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vGetMySPINIapAppInfo()
***************************************************************************/
t_Void spi_tclStaticConfigurationBase::vGetMySPINIapAppInfo(trMySPINIapAppInfo& rMySPINIapAppInfo) 
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vGetMySPINIapAppInfo"));
   if(NULL != m_poConnSettings )
   {
      m_poConnSettings->vGetMySPINIapAppInfo(rMySPINIapAppInfo);
   }
}
/***************************************************************************
** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vGetMySPINAoapAppInfo()
*****************************************************************************/
t_Void spi_tclStaticConfigurationBase::vGetMySPINAoapAppInfo(trMySPINAoapAppInfo& rfrMySPINAoapAppInfo) 
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vGetMySPINAoapAppInfo"));
   if(NULL != m_poConnSettings )
   {
      m_poConnSettings->vGetMySPINAoapAppInfo(rfrMySPINAoapAppInfo);
   }
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetCdbGPSSeviceEnabled()
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bGetCdbGPSSeviceEnabled() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bGetCdbGPSSeviceEnabled"));
   t_Bool bCdbGPSSeviceEnabled = false;
   if(NULL != m_poDataServiceSettings )
   {
      bCdbGPSSeviceEnabled = m_poDataServiceSettings->bGetCdbGPSSeviceEnabled();
   }
   return bCdbGPSSeviceEnabled;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetCdbLocationSeviceEnabled()
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bGetCdbLocationSeviceEnabled() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bGetCdbLocationSeviceEnabled"));
   t_Bool bCdbLocationSevice = false;
   if(NULL != m_poDataServiceSettings )
   {
      bCdbLocationSevice = m_poDataServiceSettings->bGetCdbLocationSeviceEnabled();
   }
   return bCdbLocationSevice;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetCdbLocationSeviceEnabled()
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bGetCdbPrivateSeviceEnabled() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bGetCdbPrivateSeviceEnabled"));
   t_Bool bCdbPrivateSevice = false;
   if(NULL != m_poDataServiceSettings )
   {
      bCdbPrivateSevice = m_poDataServiceSettings->bGetCdbPrivateSeviceEnabled();
   }
   return bCdbPrivateSevice;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetCdbMediaMetadataSinkEnabled()
 ***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bGetCdbMediaMetadataSinkEnabled() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bGetCdbMediaMetadataSinkEnabled"));
   t_Bool bMediaMetadataSinkEnabled = false;
   if(NULL != m_poDataServiceSettings )
   {
      bMediaMetadataSinkEnabled = m_poDataServiceSettings->bGetCdbMediaMetadataSinkEnabled();
   }
   return bMediaMetadataSinkEnabled;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetCdbNaviMetadataSinkEnabled()
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bGetCdbNaviMetadataSinkEnabled() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bGetCdbNaviMetadataSinkEnabled"));
   t_Bool bNaviMetadataSinkEnabled = false;
   if(NULL != m_poDataServiceSettings )
   {
      bNaviMetadataSinkEnabled = m_poDataServiceSettings->bGetCdbNaviMetadataSinkEnabled();
   }
   return bNaviMetadataSinkEnabled;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetEnvDataSubEnabled()
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bGetEnvDataSubEnabled() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bGetEnvDataSubEnabled"));
   t_Bool bEnvDataSubEnabled = false;
   if(NULL != m_poDataServiceSettings )
   {
      bEnvDataSubEnabled = m_poDataServiceSettings->bGetEnvDataSubEnabled();
   }
   return bEnvDataSubEnabled;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetGearStatusEnabled()
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bGetGearStatusEnabled() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bGetGearStatusEnabled"));
   t_Bool bGearStatusEnabled = false;
   if(NULL != m_poDataServiceSettings )
   {
      bGearStatusEnabled = m_poDataServiceSettings->bGetGearStatusEnabled();
   }
   return bGearStatusEnabled;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetAccelerometerDataEnabled()
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bGetAccelerometerDataEnabled() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bGetAccelerometerDataEnabled"));
   t_Bool bAccelerometerData = false;
   if(NULL != m_poDataServiceSettings )
   {
      bAccelerometerData = m_poDataServiceSettings->bGetAccelerometerDataEnabled();
   }
   return bAccelerometerData;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetGyroDataEnabled()
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bGetGyroDataEnabled() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bGetGyroDataEnabled"));
   t_Bool bGyroDataEnabled = false;
   if(NULL != m_poDataServiceSettings )
   {
      bGyroDataEnabled = m_poDataServiceSettings->bGetGyroDataEnabled();
   }
   return bGyroDataEnabled;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetParkBrakeDataEnabled()
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bGetParkBrakeDataEnabled() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bGetParkBrakeDataEnabled"));
   t_Bool bParkBrakeData = false;
   if(NULL != m_poDataServiceSettings )
   {
      bParkBrakeData = m_poDataServiceSettings->bGetParkBrakeDataEnabled();
   }
   return bParkBrakeData;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bIsBochNavigation()
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bIsBochNavigation() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bIsBochNavigation"));
   t_Bool bBochNavi = false;
   if(NULL != m_poDataServiceSettings )
   {
      bBochNavi = m_poDataServiceSettings->bIsBochNavigation();
   }
   return bBochNavi;
}

/***************************************************************************
** FUNCTION:  t_U32 spi_tclStaticConfigurationBase::u32GetSpeedTimerInterval()
***************************************************************************/
t_U32 spi_tclStaticConfigurationBase::u32GetSpeedTimerInterval() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::u32GetSpeedTimerInterval"));
   t_U32 u32SpeedTimerInterval = 0;
   if(NULL != m_poDataServiceSettings )
   {
      u32SpeedTimerInterval = m_poDataServiceSettings->u32GetSpeedTimerInterval();
   }
   return u32SpeedTimerInterval;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetLocDataEnabled()
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bGetLocDataEnabled() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bGetLocDataEnabled"));
   t_Bool bLocDataEnabled = false;
   if(NULL != m_poDataServiceSettings )
   {
     bLocDataEnabled = m_poDataServiceSettings->bGetLocDataEnabled();
   }
   return bLocDataEnabled;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetMLNotiSettingVal()
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bGetMLNotiSettingVal() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bGetMLNotiSettingVal"));
   t_Bool bMLNotiSettingVal = false;
   if(NULL != m_poDefaultSettings )
   {
      bMLNotiSettingVal = m_poDefaultSettings->bGetMLNotiSettingVal();
   }
   return bMLNotiSettingVal;
}

/***************************************************************************
** FUNCTION:  tenEnabledInfo spi_tclStaticConfigurationBase::enGetMLLinkEnableSetVal()
***************************************************************************/
tenEnabledInfo spi_tclStaticConfigurationBase::enGetMLLinkEnableSetVal() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::enGetMLLinkEnableSetVal"));
   tenEnabledInfo enMLLinkEnableSetVal = e8USAGE_UNKNOWN;
   if(NULL != m_poDefaultSettings )
   {
      enMLLinkEnableSetVal = m_poDefaultSettings->enGetMLLinkEnableSetVal();
   }
   return enMLLinkEnableSetVal;
}

/***************************************************************************
** FUNCTION:  tenEnabledInfo spi_tclStaticConfigurationBase::enGetDipoEnableSetVal()
***************************************************************************/
tenEnabledInfo spi_tclStaticConfigurationBase::enGetDipoEnableSetVal() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::enGetDipoEnableSetVal"));
   tenEnabledInfo enDipoEnableSetVal = e8USAGE_UNKNOWN;
   if(NULL != m_poDefaultSettings )
   {
      enDipoEnableSetVal = m_poDefaultSettings->enGetDipoEnableSetVal();
   }
   return enDipoEnableSetVal;
}

/***************************************************************************
** FUNCTION:  tenEnabledInfo spi_tclStaticConfigurationBase::enGetAAPEnableSetVal()
***************************************************************************/
tenEnabledInfo spi_tclStaticConfigurationBase::enGetAAPEnableSetVal() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::enGetAAPEnableSetVal"));
   tenEnabledInfo enAAPEnableSetVal = e8USAGE_UNKNOWN;
   if(NULL != m_poDefaultSettings )
   {
      enAAPEnableSetVal = m_poDefaultSettings->enGetAAPEnableSetVal();
   }
   return enAAPEnableSetVal;
}

/***************************************************************************
** FUNCTION: tenEnabledInfo spi_tclStaticConfigurationBase::enGetCarlifeEnableSetVal()
***************************************************************************/
tenEnabledInfo spi_tclStaticConfigurationBase::enGetCarlifeEnableSetVal() const
{
   ETG_TRACE_USR2(("spi_tclStaticConfigurationBase::enGetCarlifeEnableSetVal entered"));
   tenEnabledInfo enCarlifeEnableSetVal = e8USAGE_UNKNOWN;
   if(NULL != m_poDefaultSettings )
   {
      enCarlifeEnableSetVal = m_poDefaultSettings->enGetCarlifeEnableSetVal();
   }
   return enCarlifeEnableSetVal;
}

/***************************************************************************
** FUNCTION: tenEnabledInfo spi_tclStaticConfigurationBase::enGetMySPINEnableSetVal()
***************************************************************************/
tenEnabledInfo spi_tclStaticConfigurationBase::enGetMySPINEnableSetVal() const
{
   ETG_TRACE_USR2(("spi_tclStaticConfigurationBase::enGetMySPINEnableSetVal entered"));
   tenEnabledInfo enMySPINEnableSetVal = e8USAGE_UNKNOWN;
   if(NULL != m_poDefaultSettings )
   {
      enMySPINEnableSetVal = m_poDefaultSettings->enGetMySPINEnableSetVal();
   }
   return enMySPINEnableSetVal;
}

/***************************************************************************
** FUNCTION: tenDriveSideInfo spi_tclStaticConfigurationBase::enGetSteeringWheelPos()
***************************************************************************/
tenDriveSideInfo spi_tclStaticConfigurationBase::enGetSteeringWheelPos() const
{
   ETG_TRACE_USR2(("spi_tclStaticConfigurationBase::enGetSteeringWheelPos entered"));
   tenDriveSideInfo enSteeringWheelPos = e8UNKNOWN_DRIVE_SIDE;
   if(NULL != m_poDefaultSettings )
   {
      enSteeringWheelPos = m_poDefaultSettings->enGetSteeringWheelPos();
   }
   return enSteeringWheelPos;
}

/***************************************************************************
** FUNCTION: t_Bool spi_tclStaticConfigurationBase::enGetAAPEnableSetVal()
***************************************************************************/
tenDeviceSelectionMode spi_tclStaticConfigurationBase::enGetSelectMode() const
{
   ETG_TRACE_USR2(("spi_tclStaticConfigurationBase::enGetSelectMode entered"));
   tenDeviceSelectionMode enSelectMode = e16DEVICESEL_UNKNOWN;
   if(NULL != m_poDefaultSettings )
   {
      enSelectMode = m_poDefaultSettings->enGetSelectMode();
   }
   return enSelectMode;
}

/***************************************************************************
** FUNCTION:  tenDeviceCategory spi_tclStaticConfigurationBase::enGetAndroidDeviceTechnologyPreference()
***************************************************************************/
tenDeviceCategory spi_tclStaticConfigurationBase::enGetAndroidDeviceTechnologyPreference() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::enGetAndroidDeviceTechnologyPreference"));
   tenDeviceCategory enAndroidDeviceTechPref = e8DEV_TYPE_UNKNOWN;
   if(NULL != m_poDefaultSettings )
   {
      enAndroidDeviceTechPref = m_poDefaultSettings->enGetAndroidDeviceTechnologyPreference();
   }
   return enAndroidDeviceTechPref;
}

/***************************************************************************
** FUNCTION:  tenDeviceCategory spi_tclStaticConfigurationBase::enGetAppleDeviceTechnologyPreference()
***************************************************************************/
tenDeviceCategory spi_tclStaticConfigurationBase::enGetAppleDeviceTechnologyPreference() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::enGetAppleDeviceTechnologyPreference"));
   tenDeviceCategory enAppleDeviceTechPref = e8DEV_TYPE_UNKNOWN;
   if(NULL != m_poDefaultSettings )
   {
      enAppleDeviceTechPref = m_poDefaultSettings->enGetAppleDeviceTechnologyPreference();
   }
   return enAppleDeviceTechPref;
}

/***************************************************************************
** FUNCTION:  t_U32 spi_tclStaticConfigurationBase::u32GetStartUpTimeInterval()
***************************************************************************/
t_U32 spi_tclStaticConfigurationBase::u32GetStartUpTimeInterval() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::u32GetStartUpTimeInterval"));
   t_U32 u32StartUpTimeInterval = 0;
   if(NULL != m_poResourceMngrSettings )
   {
      u32StartUpTimeInterval = m_poResourceMngrSettings->u32GetStartUpTimeInterval();
   }
   return u32StartUpTimeInterval;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetLastModeSupport()
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bGetLastModeSupport(tenDeviceCategory enDevCat) const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bGetLastModeSupport"));
   t_Bool bLastModeSupport = false;
   if(NULL != m_poResourceMngrSettings )
   {
      bLastModeSupport = m_poResourceMngrSettings->bGetLastModeSupport(enDevCat);
   }
   return bLastModeSupport;
}

/***************************************************************************
** FUNCTION:  tenLastModeSupport spi_tclStaticConfigurationBase::enGetLastModeSupport()
***************************************************************************/
tenLastModeSupport spi_tclStaticConfigurationBase::enGetLastModeSupport() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::enGetLastModeSupport"));
   tenLastModeSupport enLastModeSupport = e8LASTMODE_DISABLED;
   if(NULL != m_poResourceMngrSettings )
   {
      enLastModeSupport = m_poResourceMngrSettings->enGetLastModeSupport();
   }
   return enLastModeSupport;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vGetScreenOffset()
***************************************************************************/
t_Void spi_tclStaticConfigurationBase::vGetScreenOffset(trScreenOffset &rfoScreenOffset) const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vGetScreenOffset"));
   if(NULL != m_poVideoSettings )
   {
      m_poVideoSettings->vGetScreenOffset(rfoScreenOffset);
   }
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bIsServerSideScalingRequired()
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bIsServerSideScalingRequired()
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bIsServerSideScalingRequired"));
   t_Bool bIsScalingRequired = false;
   if(NULL != m_poVideoSettings )
   {
      bIsScalingRequired = m_poVideoSettings->bIsServerSideScalingRequired();
   }
   return bIsScalingRequired;
}

/***************************************************************************
** FUNCTION:  t_U8 spi_tclStaticConfigurationBase::u8EnableTouchInputEvents()
***************************************************************************/
t_U8 spi_tclStaticConfigurationBase::u8EnableTouchInputEvents()
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::u8EnableTouchInputEvents"));
   t_U8 u8TouchInputEvents = 0;
   if(NULL != m_poVideoSettings )
   {
      u8TouchInputEvents = m_poVideoSettings->u8EnableTouchInputEvents();
   }
   return u8TouchInputEvents;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vSetScreenSize()
***************************************************************************/
t_Void spi_tclStaticConfigurationBase::vSetScreenSize(const trScreenAttributes& corfrScreenAttributes)
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vSetScreenSize"));
   if(NULL != m_poVideoSettings )
   {
      m_poVideoSettings->vSetScreenSize(corfrScreenAttributes);
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vSetContAttestFlag()
***************************************************************************/
t_Void spi_tclStaticConfigurationBase::vSetContAttestFlag(t_U8 u8ContAttestFlag)
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vSetContAttestFlag"));
   if(NULL != m_poVideoSettings )
   {
      m_poVideoSettings->vSetContAttestFlag(u8ContAttestFlag);
   }
}

/***************************************************************************
** FUNCTION:  t_String spi_tclStaticConfigurationBase::szGetVideoCodecType()
***************************************************************************/
t_String spi_tclStaticConfigurationBase::szGetVideoCodecType() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::szGetVideoCodecType"));
   t_String szVideoCodecType;
   if(NULL != m_poVideoSettings )
   {
      szVideoCodecType = m_poVideoSettings->szGetVideoCodecType();
   }
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase:szGetVideoCodecType()-%s ",
      szVideoCodecType.c_str()));
   return szVideoCodecType.c_str();
}

/***************************************************************************
** FUNCTION:  t_U8 spi_tclStaticConfigurationBase::u8GetMaxUnackedFrames()
***************************************************************************/
t_U8 spi_tclStaticConfigurationBase::u8GetMaxUnackedFrames() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::u8GetMaxUnackedFrames"));
   t_U8 u8MaxUnackedFrames = 0;
   if(NULL != m_poVideoSettings )
   {
      u8MaxUnackedFrames = m_poVideoSettings->u8GetMaxUnackedFrames();
   }
   return u8MaxUnackedFrames;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetAutoStartProjection()
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bGetAutoStartProjection() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bGetAutoStartProjection"));
   t_Bool bAutoStartProjection = false;
   if(NULL != m_poVideoSettings )
   {
      bAutoStartProjection = m_poVideoSettings->bGetAutoStartProjection();
   }
   return bAutoStartProjection;
}

/***************************************************************************
** FUNCTION:  t_U8 spi_tclStaticConfigurationBase::u8GetFramesPerSec()
***************************************************************************/
t_U8 spi_tclStaticConfigurationBase::u8GetFramesPerSec() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::u8GetFramesPerSec"));
   t_U8 u8FramesPerSec = 0;
   if(NULL != m_poVideoSettings )
   {
      u8FramesPerSec = m_poVideoSettings->u8GetFramesPerSec();
   }
   return u8FramesPerSec;
}

/***************************************************************************
** FUNCTION:  t_U8 spi_tclStaticConfigurationBase::u8GetContAttestSignFlag()
***************************************************************************/
t_U8 spi_tclStaticConfigurationBase::u8GetContAttestSignFlag() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::u8GetContAttestSignFlag"));
   t_U8 u8ContAttestSignFlag = 0;
   if(NULL != m_poVideoSettings )
   {
      u8ContAttestSignFlag = m_poVideoSettings->u8GetContAttestSignFlag();
   }
   return u8ContAttestSignFlag;
}

/***************************************************************************
** FUNCTION:  tenPixelFormat spi_tclStaticConfigurationBase::enGetPixelFormat()
***************************************************************************/
tenPixelFormat spi_tclStaticConfigurationBase::enGetPixelFormat() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::enGetPixelFormat"));
   tenPixelFormat enPixelFormat = e8_PIXELFORMAT_UNKNOWN;
   if(NULL != m_poVideoSettings )
   {
      enPixelFormat = m_poVideoSettings->enGetPixelFormat();
   }
   return enPixelFormat;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vGetBdclVideoSinkConfig()
***************************************************************************/
t_Void spi_tclStaticConfigurationBase::vGetBdclVideoSinkConfig(std::map< t_String, t_String >& mapVideoConfig) const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vGetBdclVideoSinkConfig"));
   if(NULL != m_poVideoSettings )
   {
      m_poVideoSettings->vGetBdclVideoSinkConfig(mapVideoConfig);
   }
}

/***************************************************************************
** FUNCTION: t_Void spi_tclStaticConfigurationBase::vSetDeviceSelectionMode()
***************************************************************************/
t_Void spi_tclStaticConfigurationBase::vSetDeviceSelectionMode(tenDeviceSelectionMode enSelectionMode, tenDeviceCategory enTechnologyType)
{
   ETG_TRACE_USR2(("spi_tclStaticConfigurationBase::vSetDeviceSelectionMode entered"));
   if(NULL != m_poConnSettings )
   {
      m_poConnSettings->vSetDeviceSelectionMode(enSelectionMode,enTechnologyType);
   }
}

/***************************************************************************
** FUNCTION: tenGeoCoordinateSystemType spi_tclStaticConfigurationBase::vGetBdclVideoSinkConfig()
***************************************************************************/
tenGeoCoordinateSystemType spi_tclStaticConfigurationBase::enGetGeoCoordinateSystem()
{
   ETG_TRACE_USR2(("spi_tclStaticConfigurationBase::enGetGeoCoordinateSystem entered"));
   return e8GEO_COORDINATE_SYSTEM_UNKNOWN;
}

/***************************************************************************
 *** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vGetSpiFeatureSupport(...)
***************************************************************************/
t_Void spi_tclStaticConfigurationBase::vGetSpiFeatureSupport(trSpiFeatureSupport& rfrSpiFeatureSupport)
{
   t_U8 u8EolMLEnableSetting = 0;
   t_U8 u8EolDipoEnableSetting = 0;
   t_U8 u8EolAAPEnableSetting = 0;
   t_U8 u8EolCarlifeEnableSetting = 0;
   t_U8 u8EolOncarEnableSetting = 0;

   //! Set supported SPI features to default value i.e false
   rfrSpiFeatureSupport.vSetMirrorLinkSupport(false);
   rfrSpiFeatureSupport.vSetDipoSupport(false);
   rfrSpiFeatureSupport.vSetAndroidAutoSupport(false);
   rfrSpiFeatureSupport.vSetmySPINSupport(false);
   rfrSpiFeatureSupport.vSetCarlifeSupport(false);
   rfrSpiFeatureSupport.vSetOnCarSupport(false);
   rfrSpiFeatureSupport.vSetCPWSupport(false);

   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vGetSpiFeatureSupport: MIRRORLINK supported = %d ",
         ETG_ENUM(BOOL, u8EolMLEnableSetting)));
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vGetSpiFeatureSupport: DIPO supported = %d ",
         ETG_ENUM(BOOL, u8EolDipoEnableSetting)));
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vGetSpiFeatureSupport: ANDROID AUTO supported = %d ",
         ETG_ENUM(BOOL, u8EolAAPEnableSetting)));
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vGetSpiFeatureSupport: MYSPIN supported = FALSE "));
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vGetSpiFeatureSupport: BAIDU CARLIFE supported = %d ",
         ETG_ENUM(BOOL, u8EolCarlifeEnableSetting)));
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vGetSpiFeatureSupport: ONCAR supported = %d ",
         ETG_ENUM(BOOL, u8EolOncarEnableSetting)));
}

/***************************************************************************
 ** FUNCTION: t_U8 spi_tclStaticConfigurationBase::u8GetAAPRestrictionInfo(t_Bool bDriveRestrictions)
 ***************************************************************************/
t_U8 spi_tclStaticConfigurationBase::u8GetAAPRestrictionInfo(t_Bool bDriveRestrictions)
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::u8GetAAPRestrictionInfo Is request sent for drive restrictions:%d ",
            ETG_ENUM(BOOL, bDriveRestrictions)));

   t_U8 u8RestrictionsInfo = (true == bDriveRestrictions)
                                                           ? m_u8AAPDriveModeRestrictionInfo
                                                           : m_u8AAPParkModeRestrictionInfo;
   ETG_TRACE_USR2(("spi_tclStaticConfigurationBase::u8GetAAPRestrictionInfo : AAP Restrictions:%d ", u8RestrictionsInfo));
   return u8RestrictionsInfo;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vSetAAPFeatureRestrictions(...)
 ***************************************************************************/
t_Void spi_tclStaticConfigurationBase::vSetAAPFeatureRestrictions(const t_U8 cou8ParkModeRestrictionInfo, const t_U8 cou8DriveModeRestrictionInfo)
{
   ETG_TRACE_USR1(("spi_tclStatspi_tclStaticConfigurationBaseicConfiguration::vSetAAPFeatureRestrictions() Entered"));

   m_u8AAPParkModeRestrictionInfo = (cou8ParkModeRestrictionInfo & (0x1F));
   m_u8AAPDriveModeRestrictionInfo = (cou8DriveModeRestrictionInfo & (0x1F));

   ETG_TRACE_USR1(("[PARAM]::spi_tclStaticConfigurationBase() After Masking : AAP Park Mode Restriction = %d, Drive Mode Restriction = %d ",
            m_u8AAPParkModeRestrictionInfo, m_u8AAPDriveModeRestrictionInfo));
}

/***************************************************************************
** FUNCTION   :t_String spi_tclStaticConfigurationBase::szGetSoftwareBuild()
***************************************************************************/
t_String spi_tclStaticConfigurationBase::szGetSoftwareBuild()
{
   ETG_TRACE_USR1(("[FUNC]spi_tclStaticConfigurationBase::szGetSoftwareBuild entered"));
   t_String szSoftwareBuild = "multi";
   ETG_TRACE_USR1((" Software Build is %s",szSoftwareBuild.c_str()));
   return szSoftwareBuild;
}

/***************************************************************************
** FUNCTION   :t_Void spi_tclStaticConfigurationBase::vGetAAPCertificatesConfigData()
***************************************************************************/
t_Void spi_tclStaticConfigurationBase::vGetAAPCertificatesConfigData(trAAPCertConfigData& rfrCertsConfig)
{
    t_U8 u8ArraySize = (sizeof(aAAAutoCertsConfig)/sizeof(trAAutoCertConfigData));
    t_Bool bCertificatePathSet=false;
    StringHandler oStr;
    t_String szAAPCertificateModel;
    vGetAAPCertificatesModel(szAAPCertificateModel);
    for(t_U8 u8Index=0; u8Index<u8ArraySize; u8Index++)
    {
        //fetching the certificate path and key-id specific to the Vehicle Model
        if(szAAPCertificateModel == aAAAutoCertsConfig[u8Index].szCertificateModel)
        {
            rfrCertsConfig.szKeyId = aAAAutoCertsConfig[u8Index].szKeyId;
            rfrCertsConfig.szCertificateFolderPath = aAAAutoCertsConfig[u8Index].szCertificateFolderPath;

            ETG_TRACE_USR1((" Certificate Model %s",aAAAutoCertsConfig[u8Index].szCertificateModel.c_str()));
            ETG_TRACE_USR1((" Certificate Name is %s",aAAAutoCertsConfig[u8Index].szCertificateName.c_str()));

            bCertificatePathSet=true;
            break;
        }
     }

    //If the KDS value is not in range/Not found
    if(!bCertificatePathSet)
    {
        rfrCertsConfig.szKeyId = sczKeyId;
        rfrCertsConfig.szCertificateFolderPath =szCertificateFolderPath;
    }

    ETG_TRACE_USR1((" Key Id is %s",rfrCertsConfig.szKeyId.c_str()));
    ETG_TRACE_USR1((" Certificate Folder Path is %s",rfrCertsConfig.szCertificateFolderPath.c_str()));

}

/***************************************************************************
** FUNCTION   :t_Void spi_tclStaticConfigurationBase::vGetAAPCertificatesModel()
***************************************************************************/
t_Void spi_tclStaticConfigurationBase::vGetAAPCertificatesModel(t_String &rfszAAPCertificateModel)
{
    //"AAAA" is the default value to be used for certificate path and key-id
   rfszAAPCertificateModel = "AAAA";
}

/***************************************************************************
** FUNCTION:  tenDriveSideInfo spi_tclStaticConfigurationBase::enGetDriveSideInfofromKDS()
***************************************************************************/
tenDriveSideInfo spi_tclStaticConfigurationBase::enGetDriveSideInfofromKDS()
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::enGetDriveSideInfofromKDS"));
   return e8UNKNOWN_DRIVE_SIDE;
}

/***************************************************************************
** FUNCTION   :t_Void spi_tclStaticConfigurationBase::vCheckUSBPortEnabled()
***************************************************************************/
t_Void spi_tclStaticConfigurationBase::vCheckUSBPortEnabled(t_Bool &bUSBPort1,t_Bool &bUSBPort2,t_Bool &bUSBPort3)
{
   ETG_TRACE_USR1(("[FUNC]spi_tclStaticConfigurationBase::vCheckUSBPortEnabled entered"));
   bUSBPort1 = true;
   bUSBPort2 = true;
   bUSBPort3 = true;
}

/***************************************************************************
** FUNCTION:  t_String spi_tclStaticConfiguration::vSetNightModeInfo()
***************************************************************************/
t_Void spi_tclStaticConfigurationBase::vSetNightModeInfo(const tenVehicleConfiguration enNightModeInfo)
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vSetNightModeInfo "));
	m_oLock.s16Lock();
	m_enNightModeInfo = enNightModeInfo;
	m_oLock.vUnlock();

}

/***************************************************************************
** FUNCTION:  t_String spi_tclStaticConfiguration::vGetNightModeInfo()
***************************************************************************/
t_Void spi_tclStaticConfigurationBase::vGetNightModeInfo(tenVehicleConfiguration &rfoenNightModeInfo)
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vGetNightModeInfo "));
	m_oLock.s16Lock();
	rfoenNightModeInfo = m_enNightModeInfo;
	m_oLock.vUnlock();

}

/***************************************************************************
** FUNCTION:  t_String spi_tclStaticConfigurationBase::vSetDriveModeInfo()
***************************************************************************/
t_Void spi_tclStaticConfigurationBase::vSetDriveModeInfo(const tenVehicleConfiguration enDriveModeInfo)
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vSetDriveModeInfo"));
	//The data will access from different threads. Hence locked before use.
	m_oLock.s16Lock();
	m_enDriveModeInfo = enDriveModeInfo;
	m_oLock.vUnlock();
}

/***************************************************************************
** FUNCTION:  t_String spi_tclStaticConfigurationBase::vGetDriveModeInfo()
***************************************************************************/
t_Void spi_tclStaticConfigurationBase::vGetDriveModeInfo(tenVehicleConfiguration &rfoenDriveModeInfo)
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vGetDriveModeInfo"));
	m_oLock.s16Lock();
	rfoenDriveModeInfo =  m_enDriveModeInfo;
	m_oLock.vUnlock();
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vIntializeSPISettings
 ***************************************************************************/
t_Void spi_tclStaticConfigurationBase::vIntializeSPISettings()
{
   ETG_TRACE_USR1((" spi_tclStaticConfigurationBase::vIntializeSPISettings() entered"));
   if(NULL != m_poDiscovererSettings)
   {
      m_poDiscovererSettings->vIntializeSPISettings();
   }
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bIsDeviceBlacklisted()
 ***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bIsDeviceBlacklisted(t_S32 s32ProductID, t_S32 s32VendorID,
         tenDeviceCategory enDeviceCtegory)
{
   ETG_TRACE_USR1((" spi_tclStaticConfigurationBase::bIsDeviceBlacklisted() entered"));
   t_Bool bRetVal = false;
   if(NULL != m_poDiscovererSettings)
   {
      bRetVal = m_poDiscovererSettings->bIsDeviceBlacklisted(s32ProductID,s32VendorID,enDeviceCtegory);
   }
   return bRetVal;
}


/***************************************************************************
 ** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vIsDeviceWhitelisted()
 ***************************************************************************/
t_Void spi_tclStaticConfigurationBase::vIsDeviceWhitelisted(t_S32 s32ProductID, t_S32 s32VendorID, tenDeviceCategory &rfenDevCat)
{
   ETG_TRACE_USR1((" spi_tclStaticConfigurationBase::vIsDeviceWhitelisted() entered"));
   if(NULL != m_poDiscovererSettings)
   {
      m_poDiscovererSettings->vIsDeviceWhitelisted(s32ProductID,s32VendorID,rfenDevCat);
   }
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bIsDeviceWhitelistingEnabled
 ***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bIsDeviceWhitelistingEnabled()
{
   ETG_TRACE_USR1((" spi_tclStaticConfigurationBase::bIsDeviceWhitelistingEnabled() entered"));
   t_Bool bRetVal = false;
   if(NULL != m_poDiscovererSettings)
   {
      bRetVal = m_poDiscovererSettings->bIsDeviceWhitelistingEnabled();
   }
   return bRetVal;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bIsDeviceWhitelistingEnabled
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bIsWhitelistEnabledCat(tenDeviceCategory enChosenDeviceCategory)
{
    ETG_TRACE_USR1((" spi_tclStaticConfigurationBase::bIsWhitelistEnabledCat() entered"));
    t_Bool bRetVal = false;
    if(NULL != m_poDiscovererSettings)
    {
    	bRetVal = m_poDiscovererSettings->bIsWhitelistEnabledCat(enChosenDeviceCategory);
    }
    return bRetVal;
}

/***************************************************************************
 ** FUNCTION: t_String spi_tclStaticConfigurationBase::szGetChannelID()
 ***************************************************************************/
t_String spi_tclStaticConfigurationBase::szGetChannelID() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::szGetChannelID() entered "));
   return "";
}

/***************************************************************************
 ** FUNCTION: t_Bool spi_tclStaticConfigurationBase::bSetUpAPonOOBT()
 ***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bSetUpAPonOOBT()
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::bSetUpAPonOOBT() entered "));
   return false;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclStaticConfigurationBase::vCarplayDipoConfigurations()
 ***************************************************************************/
t_Void  spi_tclStaticConfigurationBase::vSelectCarplayConfiguration()
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vCarplayDipoConfigurations() entered "));
   //requires the project specific kds values for changing the dipo configurations appropriately
   spi::io::FileHandler oFileDes(NULL,spi::io::SPI_EN_RDONLY);
   if(0 == oFileDes.s32Access(sczCarplayConfigPath.c_str()))
   {
      ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vCarplayDipoConfigurations() dipo.cfg already exists"));
   }
   else
   {
      t_Bool bRenameResult = oFileDes.bFRename(sczCarplayUSBPath.c_str(), sczCarplayConfigPath.c_str());
      if(true == bRenameResult)
      {
         ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vSelectCarplayConfiguration() dipo.cfg renamed successfully from %s", sczCarplayUSBPath.c_str()));
      }
   }
}

/***************************************************************************
 ** FUNCTION: tenDAPAttestationInfo spi_tclStaticConfigurationBase::enGetDAPAttestationInfo()
 ***************************************************************************/
tenDAPAttestationInfo  spi_tclStaticConfigurationBase::enGetDAPAttestationInfo() const
{
   tenDAPAttestationInfo enDAPAttestationInfo = e8_DAP_ATTESTATION_NOT_REQ;
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::enGetDAPAttestationInfo() left with enDAPAttestationInfo = %d ",
        ETG_ENUM(DAP_ATTEST_INFO,enDAPAttestationInfo)));
   return enDAPAttestationInfo;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclStaticConfigurationBase::vGetEAPAppInfo()
 ***************************************************************************/
t_Void  spi_tclStaticConfigurationBase::vGetEAPAppInfo(trEAPAppInfo& rfrEAPAppInfo)
{
    ETG_TRACE_USR1((" spi_tclStaticConfigurationBase::vGetEAPAppInfo"));
    SPI_INTENTIONALLY_UNUSED(rfrEAPAppInfo);
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetMLNotificationSetting()
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bGetMLNotificationSetting()
{
   t_Bool bMLNotifOnOff = false; //default setting

   //TODO - Read ML Notification setting from calibration?

   ETG_TRACE_USR2(("spi_tclStaticConfigurationBase::bGetMLNotificationSetting() left with MLNotification = %d ",
         ETG_ENUM(BOOL, bMLNotifOnOff)));
   return bMLNotifOnOff;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclStaticConfigurationBase::szGetVehicleModelNumber()
 ***************************************************************************/
t_Void  spi_tclStaticConfigurationBase::szGetVehicleModelNumber(t_String& szVehicleModelNumber) const
{
    ETG_TRACE_USR1((" spi_tclStaticConfigurationBase::szGetVehicleModelNumber"));
    SPI_INTENTIONALLY_UNUSED(szVehicleModelNumber);
}

/*************************************************************************
 ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bXmlReadNode(xmlNode *poNode)
*************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bXmlReadNode(xmlNodePtr poNode)
{
   SPI_INTENTIONALLY_UNUSED(poNode);
    ETG_TRACE_USR1((" spi_tclStaticConfigurationBase::bXmlReadNodeentered"));
   t_Bool bRetVal = false;
   return bRetVal;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vSetSelectionModeStoredInDatapool
 ***************************************************************************/
t_Void spi_tclStaticConfigurationBase::vSetSelectionModeStoredInDatapool(tenDeviceSelectionMode enStoredSelectionMode)
{
   m_enStoredSelectionMode = enStoredSelectionMode;
}

/***************************************************************************
 ** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vStoreDevicePreferencefromDatapool
 ***************************************************************************/
t_Void spi_tclStaticConfigurationBase::vStoreDevicePreferencefromDatapool(tenDeviceCategory enAndroidDevicePreferencefromDatapool, tenDeviceCategory enAppleDevicePreferencefromDatapool)
{
   m_enAndroidDevicePreference = enAndroidDevicePreferencefromDatapool;
   m_enAppleDevicePreference = enAppleDevicePreferencefromDatapool;
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vDisplayAudioSettings()
***************************************************************************/
t_Void spi_tclStaticConfigurationBase::vDisplayAudioSettings()
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vDisplayAudioSettings"));
   if(NULL != m_poAudioSettings )
   {
      m_poAudioSettings->vDisplayAudioSettings();
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vDisplayVideoSettings()
***************************************************************************/
t_Void spi_tclStaticConfigurationBase::vDisplayVideoSettings()
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vDisplayVideoSettings"));
   if(NULL != m_poVideoSettings )
   {
	   m_poVideoSettings->vDisplayVideoSettings();
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vDisplayAppSettings()
***************************************************************************/
t_Void spi_tclStaticConfigurationBase::vDisplayAppSettings()
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vDisplayAppSettings"));
   if(NULL != m_poAppSettings )
   {
	   m_poAppSettings->vDisplayAppSettings();
   }
}

/***************************************************************************
** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vSetVehicleManufacturerInformationtoKDS()
***************************************************************************/
t_Void spi_tclStaticConfigurationBase::vSetVehicleManufacturerInformationtoKDS(t_Bool bResult, trVehicleManufacturerInformation rVehicleManufacturerInfoAttr)
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vSetVehicleManufacturerInformationtoKDS"));
   SPI_INTENTIONALLY_UNUSED(bResult);
   SPI_INTENTIONALLY_UNUSED(rVehicleManufacturerInfoAttr);
}

/***************************************************************************
** FUNCTION:  tenEnabledInfo spi_tclStaticConfigurationBase::enGetOnCarEnableSetVal()
***************************************************************************/
tenEnabledInfo spi_tclStaticConfigurationBase::enGetOnCarEnableSetVal() const
{
   tenEnabledInfo enOnCarEnableSetVal = e8USAGE_UNKNOWN;
   if(NULL != m_poDefaultSettings )
   {
      enOnCarEnableSetVal = m_poDefaultSettings->enGetOnCarEnableSetVal();
   }
   ETG_TRACE_USR2(("[DESC]: spi_tclStaticConfigurationBase :: OnCar enable settings default value %u",
            ETG_ENUM(tenEnabledInfo, enOnCarEnableSetVal)));
   return enOnCarEnableSetVal;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetCarlifeIOSFeatureSupport
 ***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bGetCarlifeIOSFeatureSupport()
{
   t_Bool bIsCarlifeIOSEnabled = false;

   if(NULL != m_poConnSettings)
   {
      bIsCarlifeIOSEnabled = m_poConnSettings->bGetCarlifeIOSFeatureSupport();
   }
   return bIsCarlifeIOSEnabled;
}

/***************************************************************************
 ** FUNCTION:  tenATBoxSupportedState spi_tclStaticConfigurationBase::vGetATBoxSupportedState()
***************************************************************************/
tenATBoxSupportedState spi_tclStaticConfigurationBase::vGetATBoxSupportedState() const
{
   ETG_TRACE_USR1(("spi_tclStaticConfigurationBase::vGetATBoxSupportedState entered"));
   tenATBoxSupportedState enATBoxSupportedState = ATBOX_UNKNOWN;
   return enATBoxSupportedState;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bIsDeviceWhitelisted
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bIsDeviceWhitelisted(t_U32 u32ProductID, t_U32 u32VendorID, tenDeviceCategory enReqDevCat)
{
   ETG_TRACE_USR1((" spi_tclStaticConfigurationBase::bIsDeviceWhitelisted() entered"));
   t_Bool IsDeviceWhitelisted = false;
   if(NULL != m_poDiscovererSettings)
   {
      IsDeviceWhitelisted = m_poDiscovererSettings->bIsDeviceWhitelisted(u32ProductID,u32VendorID,enReqDevCat);
   }
   return IsDeviceWhitelisted;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bIsDeviceWhitelisted
***************************************************************************/
t_Bool spi_tclStaticConfigurationBase::bGetCarlifeGPSDataSupportflag() const
{
   ETG_TRACE_USR1((" spi_tclStaticConfigurationBase::bGetCarlifeGPSDataSupportflag() entered"));
   t_Bool bSupportflag = false;
   if(NULL != m_poDataServiceSettings)
   {
      bSupportflag = m_poDataServiceSettings->bGetCarlifeGPSDataSupportflag();
   }
   return bSupportflag;
}

/***************************************************************************
** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::vGetDefaultKeys
***************************************************************************/
t_Void spi_tclStaticConfigurationBase::vGetDefaultKeys(std::map<tenDeviceCategory,std::set<tenKeyCode>>& rfrmapTechSuppDefkeys)
{
   rfrmapTechSuppDefkeys[e8DEV_TYPE_ANDROIDAUTO] =
   {
       e32MULTIMEDIA_NEXT,e32MULTIMEDIA_PREVIOUS,
       e32DEV_SEARCH,e32DEV_MEDIA,e32DEV_MENU,
       e32DEV_NAV,e32APP_KEYCODE_TELEPHONY,
       e32MULTIMEDIA_PLAY,e32MULTIMEDIA_PAUSE,e32ROTARY_KNOB
   };
   rfrmapTechSuppDefkeys[e8DEV_TYPE_DIPO] =
   {
       e32MULTIMEDIA_NEXT,e32MULTIMEDIA_PREVIOUS,
       e32MULTIMEDIA_PLAY,e32MULTIMEDIA_PAUSE,
       e32DEV_PHONE_FLASH,e32ROTARY_KNOB,
       e32DEV_MENU,e32TWODKNOB0_SHIFT_UP
   };
}
