/***********************************************************************/
/*!
 * \file  spi_tclStaticConfigurationBaseIntfIntfIntf.h.h
 * \brief  Interface Class to spi_tclStaticConfigurationBaseIntfIntf
 *************************************************************************
 \verbatim


 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:
 AUTHOR:         kmo6kor
 COPYRIGHT:      &copy; 2017 Robert Bosch Car Multimedia GmbH
 HISTORY:
 Date        | Author                | Modification
 07.07.2018    kmo6kor			    Initial Version
 \endverbatim
 *************************************************************************/

#include "SPITypes.h"
#include "Lock.h"

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/
/******************************************************************************
 | defines and macros (scope: global)
 |----------------------------------------------------------------------------*/
#ifndef spi_tclStaticConfigurationBaseIntf_
#define spi_tclStaticConfigurationBaseIntf_

class spi_tclStaticConfigurationBaseIntf /*public shl::xml::tclXmlReadable*/
{

   public:
      /***************************************************************************
       *********************************PUBLIC*************************************
       ***************************************************************************/

      /***************************************************************************
       ** FUNCTION:spi_tclStaticConfigurationBaseIntf()
       ***************************************************************************/
      /*!
       * \fn      spi_tclStaticConfigurationBaseIntf()
       * \brief   Constructor
       **************************************************************************/
      spi_tclStaticConfigurationBaseIntf()
      {
      }
      ;

      /***************************************************************************
       ** FUNCTION:~spi_tclStaticConfigurationBaseIntf
       ***************************************************************************/
      /*!
       * \fn      ~spi_tclStaticConfigurationBaseIntf
       * \brief   Destructor
       **************************************************************************/
      virtual ~spi_tclStaticConfigurationBaseIntf()
      {
      }
      ;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBaseIntf::vReadSPISettings()
       ***************************************************************************/
      /*!
       * \fn     t_Void vReadSPISettings()
       * \brief  To read the SPI settings from XML
       * \retval t_Void
       **************************************************************************/
      virtual t_Void vReadSPISettings() = 0;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBaseIntf::bNonCertAppsReq()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bNonCertAppsReq()
       * \brief  To get whether the Non certified Applications are required
       *          when the application certification is enabled for the
       *          particular version of the device.
       * \retval t_Bool
       **************************************************************************/
      virtual t_Bool bGetNonCertAppsReq() const = 0;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBaseIntf::bGetAAPNavigationEnabledInfo()
       ***************************************************************************/
      /*!
       * \fn     bGetAAPNavigationEnabledInfo(t_Bool bEnable)
       * \brief  method to get the navigation feature enable settings for AAP.
       * \param  bEnable       : [IN] TRUE-allows to display navigation updates
       *                         for AAP
       *                         FALSE - doesn't allow
       * \retval t_Bool
       **************************************************************************/
      virtual t_Bool bGetAAPNavigationEnabledInfo() const = 0;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBaseIntf::bGetAAPMediaPlaybackEnabledInfo()
       ***************************************************************************/
      /*!
       * \fn     bGetAAPMediaPlaybackEnabledInfo(t_Bool bEnable)
       * \brief  method to get the media playback feature enable settings for AAP.
       * \param  bEnable       : [IN] TRUE-allows to display media playback updates
       *                         for AAP
       *                         FALSE - doesn't allow
       * \retval t_Bool
       **************************************************************************/
      virtual t_Bool bGetAAPMediaPlaybackEnabledInfo() const = 0;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBaseIntf::bGetAAPNotificationUpdatesEnabledInfo()
       ***************************************************************************/
      /*!
       * \fn     bGetAAPNotificationUpdatesEnabledInfo(t_Bool bEnable)
       * \brief  method to get notification update enable settings for AAP.
       * \param  bEnable       : [IN] TRUE-allows to display notification updates
       *                         for AAP
       *                         FALSE - doesn't allow
       * \retval t_Bool
       **************************************************************************/
      virtual t_Bool bGetAAPNotificationUpdatesEnabledInfo() const = 0;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBaseIntf::bGetTBTUpdatesEnabledInfo()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetTBTUpdatesEnabledInfo()
       * \brief  Provides information on whether TBT updates are enabled or disabled.
       * \retval t_Bool TRUE- Enabled FALSE-Disabled
       **************************************************************************/
      virtual t_Bool bGetTBTUpdatesEnabledInfo() const = 0;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBaseIntf::bGetAssisstantGuideUpdatesEnabledInfo()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetAssisstantGuideUpdatesEnabledInfo()
       * \brief  Provides information on whether Assisstant Guide updates are enabled
       *         or disabled.
       * \retval t_Bool TRUE- Enabled FALSE-Disabled
       **************************************************************************/
      virtual t_Bool bGetAssisstantGuideUpdatesEnabledInfo() const = 0;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBaseIntf::vSetRegion(.)
       ***************************************************************************/
      /*!
       * \fn     t_Void vSetRegion(tenRegion enRegion)
       * \brief  Interface to set the region for application certification.
       *         It gives the info of which region CCC Guidelines should be followed
       *         for the Application Certification Filtering
       * \param  [IN] enRegion : Region enumeration
       * \sa
       **************************************************************************/
      virtual t_Void vSetRegion(tenRegion enRegion) = 0;

      /***************************************************************************
       ** FUNCTION:  t_String spi_tclStaticConfigurationBaseIntf::szGetRegion()
       ***************************************************************************/
      /*!
       * \fn     t_String szGetRegion()
       * \brief  To Get the which regions CCC guidelines should be followed for the
       *          Application certification.
       * \retval t_String
       **************************************************************************/
      virtual t_String szGetRegion() const = 0;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBaseIntf::bGetNonCertAppAllowedInfo()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetNonCertAppAllowedInfo()
       * \brief  Provides information on whether non certified ML apps are allowed
       *         in all modes(drive/park)
       * \retval t_Bool TRUE- Allowed FALSE-Not allowed
       **************************************************************************/
      virtual t_Bool bGetNonCertAppAllowedInfo() const = 0;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBaseIntf::bEnableCTSTest()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bEnableCTSTest()
       * \brief  To get whether the CTS test is enabled.
       * \retval t_Bool  TRUE- Enabled FALSE-Disabled
       **************************************************************************/
      virtual t_Bool bEnableCTSTest() = 0;

      /***************************************************************************
       ** FUNCTION:  t_U8 spi_tclStaticConfigurationBaseIntf::u8GetSourceNumber(tenAudioDir)
       ***************************************************************************/
      /*!
       * \fn      u8GetSourceNumber( tenAudioDir enAudDir)
       * \brief   Function to get the Source Number for Audio Link & selected device
       *          category.
       * \param   [enAudDir]: Audio Link Direction
       * \retval  t_U8 value: Source Number
       **************************************************************************/
      virtual t_U8 u8GetSourceNumber(tenAudioDir enAudDir) const =0;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBaseIntf::vGetRTPPayloadType
       (t_U32& rfu32RTPPayload)
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetRTPPayloadType(t_U32& u32RTPPayload)
       * \brief   Method to get the RTPPayload type
       * \param   [rfu32RTPPayload]: Reference to the RTP Payload type
       * \retval  NONE
       **************************************************************************/
      virtual t_Void vGetRTPPayloadType(t_U32& rfu32RTPPayload) const =0;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBaseIntf::vGetAudioIPL
       (tU32& rfu32AudioIPL)
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetAudioIPL(tU32& rfu32AudioIPL)
       * \brief   Method to get the Audio IPL
       * \param   [rfu32AudioIPL]: Reference to the Audio IPL
       * \retval  t_Bool value: TRUE if Setting is Valid, FALSE Otherwise
       **************************************************************************/
      virtual t_Void vGetAudioIPL(t_U32& rfu32AudioIPL) const =0;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBaseIntf::bGetAudioBlockingTriggerBasedOnCat()
       ***************************************************************************/
      /*!
       * \fn      t_Bool bGetAudBlockTriggerBasedOnCat()
       * \brief   Method to get the Audio Blocking enable/disable trigger based on
       *          Application category or not. This feature works only in conjunction with the
       *          Dynamic Audio feature
       * \retval  t_Bool
       **************************************************************************/
      virtual t_Bool bGetAudBlockTriggerBasedOnCat() const = 0;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBaseIntf::vSetAudBlockTriggerBasedOnCat()
       ***************************************************************************/
      /*!
       * \fn      vSetAudBlockTriggerBasedOnCat()
       * \brief   Method to Set the Audio Blocking enable/disable trigger based on
       *          Application category or not. This feature works only in conjunction with the
       *          Dynamic Audio feature
       * \retval  t_Void
       **************************************************************************/
      virtual t_Void vSetAudBlockTriggerBasedOnCat(t_Bool bIsML11OrAboveDevice) = 0;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBaseIntf::bGetAudBlockTriggerBasedOnGlobaleMute()
       ***************************************************************************/
      /*!
       * \fn      t_Bool bGetAudBlockTriggerBasedOnGlobaleMute()
       * \brief   Method to get the Audio Blocking enable/disable trigger based on
       *          Global mute is enabled or not
       * \retval  t_Bool
       **************************************************************************/
      virtual t_Bool bGetAudBlockTriggerBasedOnGlobaleMute() const =0;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBaseIntf::vGetAudioPipeConfig()
       ***************************************************************************/
      /*!
       * \fn     t_Void vGetAudioPipeConfig()
       * \brief  provides audio configuration for the specified audio steam
       * \param  [OUT]rfrmapAudioPipeConfig : returns audio configuration as string
       **************************************************************************/
      virtual t_Void vGetAudioPipeConfig(tmapAudioPipeConfig &rfmapAudioPipeConfig) =0;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBaseIntf::bGetMLDynAudioSupport()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetMLDynAudioSupport() const
       * \brief  To Get whether ML Audio is supported or not.
       * \retval  t_Bool
       **************************************************************************/
      virtual t_Bool bGetMLDynAudioSupport() const =0;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBaseIntf::vSetMLDynAudioSupport()
       ***************************************************************************/
      /*!
       * \fn     vSetMLDynAudioSupport() const
       * \brief  To Set whether ML Audio is supported or not.
       *         (For ML 1.0 = Not Supported, For Ml 1.1 onwards = supported)
       * \retval  t_Void
       **************************************************************************/
      virtual t_Void vSetMLDynAudioSupport(t_Bool bIsML11OrAboveDevice) =0;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBaseIntf::bGetAAPMediaStreamRecEnabled()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetAAPMediaStreamRecEnabled() const
       * \brief  To Get whether AAP Media audio stream recording is enabled or not.
       * \retval  t_Bool
       **************************************************************************/
      virtual t_Bool bGetAAPMediaStreamRecEnabled() const =0;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBaseIntf::bGetAAPGuidanceStreamRecEnabled()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetAAPGuidanceStreamRecEnabled() const
       * \brief  To Get whether AAP Guidance audio stream recording is enabled or not.
       * \retval  t_Bool
       **************************************************************************/
      virtual t_Bool bGetAAPGuidanceStreamRecEnabled() const =0;

      /***************************************************************************
       ** FUNCTION: t_String spi_tclStaticConfigurationBaseIntf::szGetAudSinkMaxUnackedFrames()
       ***************************************************************************/
      /*!
       * \fn     szGetAudSinkMaxUnackedFrames() const
       * \brief  To Get "max-unacked-frames" value for Audio Sink configuration
       * \retval  t_String
       **************************************************************************/
      virtual t_String szGetAudSinkMaxUnackedFrames() const =0;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBaseIntf::bGetRTPInSupport()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetRTPInSupport() const
       * \brief  To Get whether RTPIn support is enabled or not.
       * \retval  t_Bool
       **************************************************************************/
      virtual t_Bool bGetRTPInSupport() const =0;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBaseIntf::bGetMySIPNAdvAudioSupport()
       ***************************************************************************/
      /*!
       * \fn     bGetMySIPNAdvAudioSupport() const
       * \brief  To Get support for mySPIN Advanced Audio
       * \retval  t_Bool
       **************************************************************************/
      virtual t_Bool bGetMySIPNAdvAudioSupport() const =0;

      /***************************************************************************
       ** FUNCTION: t_String spi_tclStaticConfigurationBaseIntf::szGetDisableRealTimePriorityAudio()
       ***************************************************************************/
      /*!
       * \fn     szGetDisableRealTimePriorityAudio() const
       * \brief  To Get the value of disabling real time priority audio for Audio Sink configuration
       * \retval  t_String
       **************************************************************************/
      virtual t_String szGetDisableRealTimePriorityAudio() const =0;

      /***************************************************************************
       ** FUNCTION: t_String spi_tclStaticConfigurationBaseIntf::szGetAudThreadsRealTimePriority()
       ***************************************************************************/
      /*!
       * \fn     szGetAudThreadsRealTimePriority() const
       * \brief  To Get Audio threads Real time priority value for Audio Sink configuration
       * \retval  t_String
       **************************************************************************/
      virtual t_String szGetAudThreadsRealTimePriority() const =0;

      /***************************************************************************
       ** FUNCTION: t_String spi_tclStaticConfigurationBaseIntf::szGetAudSinkThresholdMs()
       ***************************************************************************/
      /*!
       * \fn     szGetAudSinkThresholdMs() const
       * \brief  To Get Audio Sink Threshold configuration
       * \retval  t_String
       **************************************************************************/
      virtual t_String szGetAudSinkThresholdMs() const =0;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBaseIntf::vGetBdclMediaSinkConfig()
       ***************************************************************************/
      /*!
       * \fn     vGetBdclMediaSinkConfig() const
       * \brief  To Get BDCL Media Audio Sink configuration
       * \retval  t_Void
       **************************************************************************/
      virtual t_Void vGetBdclMediaSinkConfig(std::map<t_String, t_String>& mapAudioConfig) const =0;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBaseIntf::vGetBdclTTSSinkConfig()
       ***************************************************************************/
      /*!
       * \fn     vGetBdclTTSSinkConfig() const
       * \brief  To Get BDCL TTS Audio Sink configuration
       * \retval  t_Void
       **************************************************************************/
      virtual t_Void vGetBdclTTSSinkConfig(std::map<t_String, t_String>& mapAudioConfig) const=0;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBaseIntf::vGetBdclVRSinkConfig()
       ***************************************************************************/
      /*!
       * \fn     vGetBdclVRSinkConfig() const
       * \brief  To Get BDCL VR Audio Sink configuration
       * \retval  t_Void
       **************************************************************************/
      virtual t_Void vGetBdclVRSinkConfig(std::map<t_String, t_String>& mapAudioConfig) const =0;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBaseIntf::vGetBdclVRSourceConfig()
       ***************************************************************************/
      /*!
       * \fn     vGetBdclVRSourceConfig() const
       * \brief  To Get BDCL VR Audio Source configuration
       * \retval  t_Void
       **************************************************************************/
      virtual t_Void vGetBdclVRSourceConfig(std::map<t_String, t_String>& mapAudioConfig) const =0;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBaseIntf::bGetAudioMuteEnabled()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetAudioMuteEnabled() const
       * \brief  To Get whether audio mute is enabled (during last mode audio)
       * \retval  t_Bool
       **************************************************************************/
      virtual t_Bool bGetAudioMuteEnabled() const =0;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBaseIntf::bGetVolumeLockEnabled()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetVolumeLockEnabled() const
       * \brief  To Get whether Volume lock is enabled (during last mode audio)
       * \retval  t_Bool
       **************************************************************************/
      virtual t_Bool bGetVolumeLockEnabled() const =0;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBaseIntf::bGetMLDynAudioApplicability()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetMLDynAudioApplicability() const
       * \brief  To Get whether ML Dnamic Audio is applicable or not from policy.
       * \retval  t_Bool
       **************************************************************************/
      virtual t_Bool bGetMLDynAudioApplicability() const =0;

      /***************************************************************************
       ** FUNCTION:  tenDeviceSelectionMode spi_tclStaticConfigurationBaseIntf::enGetDeviceSelectionMode
       ***************************************************************************/
      /*!
       * \fn     tenDeviceSelectionMode enGetDeviceSelectionMode()
       * \brief  returns Device Selection Mode
       **************************************************************************/
      virtual tenDeviceSelectionMode enGetDeviceSelectionMode(tenDeviceCategory enTechnologyType) const =0;

      /***************************************************************************
       ** FUNCTION:  tenDeviceCategory spi_tclStaticConfigurationBaseIntf::enGetTechnologyPreference
       ***************************************************************************/
      /*!
       * \fn     tenDeviceCategory enGetTechnologyPreference()
       * \brief  returns Device type preference
       **************************************************************************/
      virtual tenDeviceCategory enGetTechnologyPreference(tenDeviceType enDeviceType) const=0;

      /***************************************************************************
       ** FUNCTION:  tenEnabledInfo spi_tclStaticConfigurationBaseIntf::enGetCertificateType
       ***************************************************************************/
      /*!
       * \fn     tenEnabledInfo enGetCertificateType()
       * \brief  returns Certificate type
       **************************************************************************/
      virtual tenCertificateType enGetCertificateType() const=0;

      /***************************************************************************
       ** FUNCTION:  tenSelModePriority spi_tclStaticConfigurationBaseIntf::enGetSelectionModePriority
       ***************************************************************************/
      /*!
       * \fn     tenSelModePriority enGetSelectionModePriority()
       * \brief  returns Priority of selection mode
       **************************************************************************/
      virtual tenSelModePriority enGetSelectionModePriority() const =0;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBaseIntf::vGetPersistentStoragePath
       ***************************************************************************/
      /*!
       * \fn     t_Void vGetPersistentStoragePath()
       * \brief  returns path for persistent Storage
       **************************************************************************/
      virtual t_Void vGetPersistentStoragePath(t_String &rfszPersStoragePath) =0;

      /***************************************************************************
       ** FUNCTION:  t_U32 spi_tclStaticConfigurationBaseIntf::u32GetDeviceHistorySize
       ***************************************************************************/
      /*!
       * \fn     t_U32 u32GetDeviceHistorySize()
       * \brief  returns the Maximum size of Device History Database
       **************************************************************************/
      virtual t_U32 u32GetDeviceHistorySize() const =0;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBaseIntf::bIsXMLValidationEnabled
       ***************************************************************************/
      /*!
       * \fn     t_Bool bIsXMLValidationEnabled()
       * \brief  Method to check whether the XML validation is enabled or not
       * \retval t_Bool
       **************************************************************************/
      virtual t_Bool bIsXMLValidationEnabled() const = 0;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBaseIntf::vGetHeadUnitInfo
       ***************************************************************************/
      /*!
       * \fn      vGetHeadUnitInfo()
       * \brief  Method to retrieve head unit information
       * \retval t_Void
       **************************************************************************/
      virtual t_Void vGetMySPINIapAppInfo(trMySPINIapAppInfo& rMySPINIapAppInfo) = 0;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBaseIntf::vGetMySPINAoapAppInfo
       ***************************************************************************/
      /*!
       * \fn     t_Void vGetMySPINAoapAppInfo(trMySPINAoapAppInfo& rfrMySPINAoapAppInfo)
       * \brief  Method to retrieve head unit information
       * \param  rfrMySPINAoapAppInfo:[IN]Aoap application information
       * \retval t_Void
       **************************************************************************/
      virtual t_Void vGetMySPINAoapAppInfo(trMySPINAoapAppInfo& rfrMySPINAoapAppInfo) = 0;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBaseIntf::bGetCdbGPSSeviceEnabled()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetCdbGPSSeviceEnabled()
       * \brief  Interface to read whether GPS CDB service is enabled
       * \retval t_Bool
       **************************************************************************/
      virtual t_Bool bGetCdbGPSSeviceEnabled() const = 0;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBaseIntf::bGetCdbLocationSeviceEnabled()
       ***************************************************************************/
      /*!
       * \fn     bGetCdbLocationSeviceEnabled()
       * \brief  Interface to read whether Location CDB service is enabled
       * \retval t_Bool
       **************************************************************************/
      virtual t_Bool bGetCdbLocationSeviceEnabled() const = 0;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBaseIntf::bGetCdbPrivateSeviceEnabled()
       ***************************************************************************/
      /*!
       * \fn     bGetCdbPrivateSeviceEnabled()
       * \brief  Interface to read whether Private CDB service is enabled
       * \retval t_Bool
       **************************************************************************/
      virtual t_Bool bGetCdbPrivateSeviceEnabled() const = 0;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBaseIntf::bGetCdbMediaMetadataSinkEnabled()
       ***************************************************************************/
      /*!
       * \fn     bGetCdbMediaMetadataSinkEnabled()
       * \brief  Interface to read whether CDB Sink Media Metadata sink is enabled
       * \retval t_Bool
       **************************************************************************/
      virtual t_Bool bGetCdbMediaMetadataSinkEnabled() const = 0;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBaseIntf::bGetCdbMediaMetadataSeviceEnabled()
       ***************************************************************************/
      /*!
       * \fn     bGetCdbNaviMetadataSinkEnabled()
       * \brief  Interface to read whether CDB Navigation Metadata sink is enabled
       * \retval t_Bool
       **************************************************************************/
      virtual t_Bool bGetCdbNaviMetadataSinkEnabled() const = 0;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBaseIntf::bGetEnvDataSubEnabled()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetEnvDataSubEnabled() const
       * \brief  Interface to get whether to subscribe for Environment data or not
       * \retval t_Bool
       **************************************************************************/
      virtual t_Bool bGetEnvDataSubEnabled() const = 0;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBaseIntf::bGetGearStatusEnabled()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetGearStatusEnabled() const
       * \brief  Interface to get whether to subscribe for Gear Status data or not
       * \retval t_Bool
       **************************************************************************/
      virtual t_Bool bGetGearStatusEnabled() const = 0;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBaseIntf::bGetAccelerometerDataEnabled()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetAccelerometerDataDisable() const
       * \brief  Interface to get whether to subscribe for Accelerometer data or not
       * \retval t_Bool
       **************************************************************************/
      virtual t_Bool bGetAccelerometerDataEnabled() const = 0;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBaseIntf::bGetGyroDataEnabled()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetGyroDataEnabled() const
       * \brief  Interface to get whether to subscribe for Gyroscope data or not
       * \retval t_Bool
       **************************************************************************/
      virtual t_Bool bGetGyroDataEnabled() const = 0;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBaseIntf::bGetParkBrakeDataEnabled()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetParkBrakeDataEnabled() const
       * \brief  Interface to get whether to subscribe for park brake data or not
       * \retval t_Bool
       **************************************************************************/
      virtual t_Bool bGetParkBrakeDataEnabled() const = 0;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBaseIntf::bIsBochNavigation()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bIsBochNavigation() const
       * \brief  Interface to get the type of navigation used.
       * \retval t_Bool
       **************************************************************************/
      virtual t_Bool bIsBochNavigation() const = 0;

      /***************************************************************************
       ** FUNCTION:  t_U32 spi_tclStaticConfigurationBaseIntf:: u32GetSpeedTimerInterval()
       ***************************************************************************/
      /*!
       * \fn     u32GetSpeedTimerInterval() const
       * \brief  Interface to get the Speed timer interval in ms.
       * \retval t_U32
       **************************************************************************/
      virtual t_U32 u32GetSpeedTimerInterval() const = 0;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBaseIntf::bGetLocDataEnabled()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetLocDataEnabled() const
       * \brief  Interface to get location data enabled setting
       * \retval t_Bool
       **************************************************************************/
      virtual t_Bool bGetLocDataEnabled() const = 0;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBaseIntf::bGetMLNotiSettingVal()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetMLNotiSettingVal()
       * \brief  To get the default ML Notification Setting Value
       * \retval t_Bool
       **************************************************************************/
      virtual t_Bool bGetMLNotiSettingVal() const = 0;

      /***************************************************************************
       ** FUNCTION:  tenEnabledInfo spi_tclStaticConfigurationBaseIntf::enGetMLLinkEnableSetVal()
       ***************************************************************************/
      /*!
       * \fn     tenEnabledInfo enGetMLLinkEnableSetVal()
       * \brief  To get the default ML Link Enable Setting Value
       * \retval tenEnabledInfo
       **************************************************************************/
      virtual tenEnabledInfo enGetMLLinkEnableSetVal() const = 0;

      /***************************************************************************
       ** FUNCTION:  tenEnabledInfo spi_tclStaticConfigurationBaseIntf::enGetDipoEnableSetVal()
       ***************************************************************************/
      /*!
       * \fn     tenEnabledInfo enGetDipoEnableSetVal()
       * \brief  To get the default Dipo Enable Setting Value
       * \retval tenEnabledInfo
       **************************************************************************/
      virtual tenEnabledInfo enGetDipoEnableSetVal() const = 0;

      /***************************************************************************
       ** FUNCTION:  tenEnabledInfo spi_tclStaticConfigurationBaseIntf::enGetAAPEnableSetVal()
       ***************************************************************************/
      /*!
       * \fn     tenEnabledInfo enGetAAPEnableSetVal()
       * \brief  To get the default AAP Enable Setting Value
       * \retval tenEnabledInfo
       **************************************************************************/
      virtual tenEnabledInfo enGetAAPEnableSetVal() const = 0;

      /***************************************************************************
       ** FUNCTION:  tenEnabledInfo spi_tclStaticConfigurationBaseIntf::enGetCarlifeEnableSetVal()
       ***************************************************************************/
      /*!
       * \fn     tenEnabledInfo enGetCarlifeEnableSetVal()
       * \brief  To get the default Carlife Enable Setting Value
       * \retval tenEnabledInfo
       **************************************************************************/
      virtual tenEnabledInfo enGetCarlifeEnableSetVal() const = 0;

      /***************************************************************************
       ** FUNCTION:  tenEnabledInfo spi_tclStaticConfigurationBaseIntf::enGetMySPINEnableSetVal()
       ***************************************************************************/
      /*!
       * \fn     tenEnabledInfo enGetMySPINEnableSetVal()
       * \brief  To get the default Myspin enable Setting Value
       * \retval tenEnabledInfo
       **************************************************************************/
      virtual tenEnabledInfo enGetMySPINEnableSetVal() const = 0;

      /***************************************************************************
       ** FUNCTION:  tenDriveSideInfo  spi_tclStaticConfigurationBaseIntf::enGetSteeringWheelPos()
       ***************************************************************************/
      /*!
       * \fn     tenDriveSideInfo  enGetSteeringWheelPos()
       * \brief  To get the default Steering Wheel Position
       * \retval tenDriveSideInfo
       **************************************************************************/
      virtual tenDriveSideInfo enGetSteeringWheelPos() const = 0;

      /***************************************************************************
       ** FUNCTION:  tenDeviceSelectionMode spi_tclStaticConfigurationBaseIntf::enGetSelectMode()
       ***************************************************************************/
      /*!
       * \fn     tenDriveSideInfo  enGetSelectMode()
       * \brief  To get the default Selection Mode
       * \retval tenDeviceSelectionMode
       **************************************************************************/
      virtual tenDeviceSelectionMode enGetSelectMode() const = 0;

      /***************************************************************************
       ** FUNCTION:  tenDeviceCategory spi_tclStaticConfigurationBaseIntf::enGetAndroidDeviceTechnologyPreference()
       ***************************************************************************/
      /*!
       * \fn     tenDeviceCategory  enGetAndroidDeviceTechnologyPreference()
       * \brief  To get the default Technology Preference for Android Devices
       * \retval tenDeviceCategory
       **************************************************************************/
      virtual tenDeviceCategory enGetAndroidDeviceTechnologyPreference() const = 0;

      /***************************************************************************
       ** FUNCTION:  tenDeviceCategory spi_tclStaticConfigurationBaseIntf::enGetAppleDeviceTechnologyPreference()
       ***************************************************************************/
      /*!
       * \fn     tenDeviceCategory  enGetAppleDeviceTechnologyPreference()
       * \brief  To get the default Technology Preference for Apple Devices
       * \retval tenDeviceCategory
       **************************************************************************/
      virtual tenDeviceCategory enGetAppleDeviceTechnologyPreference() const = 0;

      /***************************************************************************
       ** FUNCTION:  t_U32 spi_tclStaticConfigurationBaseIntf::u32GetStartUpTimeInterval()
       ***************************************************************************/
      /*!
       * \fn      u32GetStartUpTimeInterval()
       * \brief   Function to get the CarPlay session startup time interval
       * \param   NONE
       * \retVal  t_U32 : Time interval in milli seconds.
       * \sa
       **************************************************************************/
      virtual t_U32 u32GetStartUpTimeInterval() const = 0;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBaseIntf::bGetLastModeSupport()
       ***************************************************************************/
      /*!
       * \fn     bGetLastModeSupport() const
       * \brief  To Get last mode support configuration for a device category
       * \param  enDevCat [IN]: Device category
       * \retval  t_Bool
       **************************************************************************/
      virtual t_Bool bGetLastModeSupport(tenDeviceCategory enDevCat) const = 0;

      /***************************************************************************
       ** FUNCTION: tenLastModeSupport spi_tclStaticConfigurationBaseIntf::enGetLastModeSupport()
       ***************************************************************************/
      /*!
       * \fn     enGetLastModeSupport() const
       * \brief  To Get last mode support configuration
       * \retval  tenLastModeSupport
       **************************************************************************/
      virtual tenLastModeSupport enGetLastModeSupport() const = 0;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBaseIntf::vGetScreenOffset(t_U32& u32Scre...
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetScreenOffset(t_U32& u32Screen_X_Offset,
       *          t_U32& u32Screen_Y_Offset)
       * \brief   To Get the Screen Offsets
       * \param   u32Screen_X_Offset : [OUT] X Offset
       * \param   u32Screen_Y_Offset : [OUT] Y Offset
       * \retval  t_Void
       * \sa
       **************************************************************************/
      virtual t_Void vGetScreenOffset(trScreenOffset &rfoScreenOffset) const = 0;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBaseIntf::bIsServerSideScalingRequired()
       ***************************************************************************/
      /*!
       * \fn      t_Bool bIsServerSideScalingRequired()
       * \brief   To Check whether the server side scaling required
       * \retval  t_Bool : false if the scaling is not required, else true
       **************************************************************************/
      virtual t_Bool bIsServerSideScalingRequired() = 0;

      /***************************************************************************
       ** FUNCTION:  t_U8 spi_tclStaticConfigurationBaseIntf::u8EnableTouchInputEvents()()
       ***************************************************************************/
      /*!
       * \fn      short t_U32 u8EnableTouchInputEvents()()
       * \brief   To Check whether the touch input handling is required
       * \retval  t_U8 0x0 -   if the touch input handling is not done using wayland
       *                       0x1 - if the Touch Input Handling is done via wayland
       **************************************************************************/
      virtual t_U8 u8EnableTouchInputEvents() = 0;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBaseIntf::vSetScreenSize()
       ***************************************************************************/
      /*!
       * \fn     t_Void vSetScreenSize(const trScreenAttributes& corfrScreenAttributes)
       * \brief  Interface to set the screen size of Head Unit.
       * \param  corScreenAttributes : [IN] Screen Setting attributes.
       * \retval t_Void
       **************************************************************************/
      virtual t_Void vSetScreenSize(const trScreenAttributes& corfrScreenAttributes) = 0;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBaseIntf::vSetContAttestFlag()
       ***************************************************************************/
      /*!
       * \fn      t_Void vSetContAttestFlag(t_U8 u8ContAttestFlag)
       * \brief   To enable/disable content attestation.Enabling or disableing should be done
       *          before the device is selected for the session
       * \pram    u8ContAttestFlag  : [IN] TRUE - Enable attestation
       *                                  FALSE - disable attestation
       * \retval  t_Void
       **************************************************************************/
      virtual t_Void vSetContAttestFlag(t_U8 u8ContAttestFlag) = 0;

      /***************************************************************************
       ** FUNCTION:  t_String spi_tclStaticConfigurationBaseIntf::szGetVideoCodecType()
       ***************************************************************************/
      /*!
       * \fn      t_String szGetVideoCodecType()
       * \brief   To get the video codec type
       *          before the device is selected for the session
       * \retval  t_String
       **************************************************************************/
      virtual t_String szGetVideoCodecType() const = 0;

      /***************************************************************************
       ** FUNCTION:  t_U8 spi_tclStaticConfigurationBaseIntf::u8GetMaxUnackedFrames()
       ***************************************************************************/
      /*!
       * \fn      t_U8 u8GetMaxUnackedFrames()
       * \brief   To get the max number of frames that can be recieved with out ack
       * \retval  t_U8
       **************************************************************************/
      virtual t_U8 u8GetMaxUnackedFrames() const = 0;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBaseIntf::bGetAutoStartProjection()
       ***************************************************************************/
      /*!
       * \fn      t_Bool bGetAutoStartProjection()
       * \brief   To get whether the auto projection start is enabled or not
       * \retval  t_Bool
       **************************************************************************/
      virtual t_Bool bGetAutoStartProjection() const = 0;

      /***************************************************************************
       ** FUNCTION:  t_U8 spi_tclStaticConfigurationBaseIntf::u8GetFramesPerSec()
       ***************************************************************************/
      /*!
       * \fn      t_U8 u8GetFramesPerSec()
       * \brief   To get the frame per pixel rate
       * \retval  t_U8
       **************************************************************************/
      virtual t_U8 u8GetFramesPerSec() const = 0;

      /***************************************************************************
       ** FUNCTION:  t_U8 spi_tclStaticConfigurationBaseIntf::u8GetContAttestSignFlag()
       ***************************************************************************/
      /*!
       * \fn      t_U8 u8GetContAttestSignFlag()
       * \brief   To get the Content attestation info signed flag
       * \retval  t_U8
       **************************************************************************/
      virtual t_U8 u8GetContAttestSignFlag() const= 0;

      /***************************************************************************
       ** FUNCTION:  tenPixelFormat spi_tclStaticConfigurationBaseIntf::enGetPixelFormat()
       ***************************************************************************/
      /*!
       * \fn      tenPixelFormat enGetPixelFormat()
       * \brief   To get the pixel format for display
       * \retval  tenPixelFormat
       **************************************************************************/
      virtual tenPixelFormat enGetPixelFormat() const = 0;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBaseIntf::vGetBdclVideoSinkConfig()
       ***************************************************************************/
      /*!
       * \fn     vGetBdclVideoSinkConfig() const
       * \brief  To Get BDCL VR Video Sink configuration
       * \retval  t_Void
       **************************************************************************/
      virtual t_Void vGetBdclVideoSinkConfig(std::map<t_String, t_String>& mapVideoConfig) const= 0;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBaseIntf::vSetDeviceSelectionMode()
       ***************************************************************************/
      /*!
       * \fn      t_Void vSetDeviceSelectionMode(
       * \brief   Method to set the device selection mode to automatic/manual. Changes
       *          will take into effect on successive connection
       * \param   enSelectionMode : Device selection mode @see tenDeviceSelectionMode
       * \retval  t_Void
       **************************************************************************/
      virtual t_Void vSetDeviceSelectionMode(tenDeviceSelectionMode enSelectionMode,
               tenDeviceCategory enTechnologyType) = 0;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBaseIntf::vIntializeSPISettings
       ***************************************************************************/
      /*!
       * \fn     t_Void vIntializeSPISettings()
       * \brief  Reads settings from xml file and store it internally
       **************************************************************************/
      virtual t_Void vIntializeSPISettings() = 0;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBaseIntf::bIsDeviceBlacklisted
       ***************************************************************************/
      /*!
       * \fn     t_Void bIsDeviceBlacklisted()
       * \brief  to check whether  the device is blacklisted.
       * \retval  t_Bool
       **************************************************************************/
      virtual t_Bool bIsDeviceBlacklisted(t_S32 s32ProductID, t_S32 s32VendorID, tenDeviceCategory enDeviceCtegory =
               e8DEV_TYPE_DEFAULT) = 0;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBaseIntf::vIsDeviceWhitelisted(...
       ***************************************************************************/
      /*!
       * \fn     t_Void vIsDeviceWhitelisted()
       * \brief  to check whether the connected device is whitelisted or not and return category
       **************************************************************************/
      virtual t_Void vIsDeviceWhitelisted(t_S32 s32ProductID, t_S32 s32VendorID, tenDeviceCategory &rfenDevCat)= 0;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBaseIntf::bIsDeviceWhitelistingEnabled(...
       ***************************************************************************/
      /*!
       * \fn     t_Bool bIsDeviceWhitelistingEnabled()
       * \brief  to check whether the whitelisting is enabled for a project or not
       **************************************************************************/
      virtual t_Bool bIsDeviceWhitelistingEnabled()= 0;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBaseIntf::bIsWhitelistEnabledCat(...
       ***************************************************************************/
      /*!
       * \fn     t_Bool bIsWhitelistEnabledCat()
       * \brief  to set the category for which whitelisting is enabled
       **************************************************************************/
      virtual t_Bool bIsWhitelistEnabledCat(tenDeviceCategory enChosenDeviceCategory) = 0;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBaseIntf::vGetDefaultProjUsageSettings
       ***************************************************************************/
      /*!
       * \fn     t_Bool vGetDefaultProjUsageSettings()
       * \brief  Method to retrieve default setting for projection usage
       * \param [OUT] : returns the current value of device projection enable
       * \enSPIType  : indicates the type of SPI technology. e8DEV_TYPE_UNKNOWN indicates default value for any SPI technology
       * \retval t_Void
       **************************************************************************/
      virtual t_Void vGetDefaultProjUsageSettings(tenEnabledInfo &enEnabledInfo, tenDeviceCategory enSPIType) = 0;

      /***************************************************************************
       ** FUNCTION: tenGeoCoordinateSystemType spi_tclStaticConfigurationBaseIntf::enGetGeoCoordinateSystem()
       ***************************************************************************/
      /*!
       * \fn      t_Void enGetGeoCoordinateSystem()
       * \brief   Method to read the supported coordinate system
       * \retval  tenGeoCoordinateSystemType : To indicate type of Geo Coordinate System
       **************************************************************************/
      virtual tenGeoCoordinateSystemType enGetGeoCoordinateSystem() = 0;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBaseIntf::vGetSpiFeatureSupport(...)
       ***************************************************************************/
      /*!
       * \fn      vGetSpiFeatureSupport(trSpiFeatureSupport& rfrSpiFeatureSupport)
       * \brief   Method to read supported SPI features info.
       * \param   rfrSpiFeatureSupport: [OUT] Provides supported SPI features
       * \retval  NONE
       **************************************************************************/
      virtual t_Void vGetSpiFeatureSupport(trSpiFeatureSupport& rfrSpiFeatureSupport) = 0;

      /***************************************************************************
       ** FUNCTION: t_U8 spi_tclStaticConfigurationBaseIntf::u8GetAAPRestrictionInfo(t_Bool bDriveRestrictions)
       ***************************************************************************/
      /*!
       * \fn      t_U8 u8GetAAPRestrictionInfo(t_Bool bDriveRestrictions)
       * \brief   Method to get restrictions enabled in park/drive mode for Carplay
       * \param   bDriveRestrictions: [IN] drive restrictions
       * \retval  NONE
       **************************************************************************/
      virtual t_U8 u8GetAAPRestrictionInfo(t_Bool bDriveRestrictions)= 0;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBaseIntf::vSetAAPFeatureRestrictions(const t_U8 cou8ParkModeRestrictionInfo,const t_U8 cou8DriveModeRestrictionInfo)
       ***************************************************************************/
      /*!
       * \fn      t_Void vSetAAPFeatureRestrictions(const t_U8 cou8ParkModeRestrictionInfo,const t_U8 cou8DriveModeRestrictionInfo)
       * \brief   Method to set Vehicle Park/Drive Mode Restriction
       * \param   NONE
       * \retval  t_Void
       **************************************************************************/

      virtual t_Void vSetAAPFeatureRestrictions(const t_U8 cou8ParkModeRestrictionInfo,
               const t_U8 cou8DriveModeRestrictionInfo)= 0;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBaseIntf::vGetVehicleInfoDataAAP(
       trVehicleBrandInfo& rfrVehicleBrandInfo)
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetVehicleInfoDataAAP(trVehicleBrandInfo& rfrVehicleBrandInfo)
       * \brief   Method to read the EOL value and get the Brand configuration data
       from the look up table
       * \param   rfrVideoConfig: [IN]Vehicle Brand Data
       * \retval  NONE
       **************************************************************************/
      virtual t_Void vGetVehicleInfoDataAAP(trVehicleInfo& rfrVehicleInfo) = 0;

      /***************************************************************************
       ** FUNCTION:  t_String spi_tclStaticConfigurationBaseIntf::szGetModelYear()
       ***************************************************************************/
      /*!
       * \fn      t_String szGetModelYear()
       * \brief   Method to get the Model year of the vehicle
       * \param   NONE
       * \retval  t_String
       **************************************************************************/
      virtual t_String szGetModelYear() = 0;

      /***************************************************************************
       ** FUNCTION: t_String spi_tclStaticConfigurationBaseIntf::szGetSoftwareBuild()
       ***************************************************************************/
      /*!
       * \fn      t_String szGetSoftwareBuild(t_String &rfrSoftwareBuild)
       * \brief   Method to get Software Build version
       * \param   NONE
       * \retval  NONE
       **************************************************************************/
      virtual t_String szGetSoftwareBuild()= 0;

      /***************************************************************************
       ** FUNCTION:  t_String spi_tclStaticConfigurationBaseIntf::szGetHardwareVersion()
       ***************************************************************************/
      /*!
       * \fn      t_String szGetHardwareVersion()
       * \brief   Method to get the accessory hardware version
       * \param   NONE
       * \retval  t_String
       **************************************************************************/
      virtual t_String szGetHardwareVersion() = 0;

      /***************************************************************************
       ** FUNCTION:  t_String spi_tclStaticConfigurationBaseIntf::szGetHeadUnitModelName()
       ***************************************************************************/
      /*!
       * \fn      t_String szGetHeadUnitModelName()
       * \brief   Method to get the headunit model name.
       * \param   NONE
       * \retval  t_String
       **************************************************************************/
      virtual t_String szGetHeadUnitModelName() = 0;

      /***************************************************************************
       ** FUNCTION:  t_String spi_tclStaticConfigurationBaseIntf::szGetHeadUnitManufacturerName()
       ***************************************************************************/
      /*!
       * \fn      t_String szGetHeadUnitManufacturerName()
       * \brief   Method to get the Headunit manufacturer name.
       * \param   NONE
       * \retval  t_String
       **************************************************************************/
      virtual t_String szGetHeadUnitManufacturerName() = 0;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBaseIntf::vGetAAPCertificatesConfigData()
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetCertsConfigData(trAAAutoCertConfigData& rfrCertsConfig)
       * \brief   Method to read the Certificate Configuration file
       * \param   rfrCertsConfig: [IN] Certificate Config Data
       * \retval  NONE
       **************************************************************************/
      virtual t_Void vGetAAPCertificatesConfigData(trAAPCertConfigData& rfrCertsConfig)= 0;

      /***************************************************************************
       ** FUNCTION:  tenDriveSideInfo spi_tclStaticConfigurationBaseIntf::enGetDriveSideInfofromKDS(...)
       ***************************************************************************/
      /*!
       * \fn      enGetDriveSideInfofromKDS()
       * \brief   Method to read Drive side info from KDS
       * \param   NONE
       * \retval  Drive Side enum
       **************************************************************************/
      virtual tenDriveSideInfo enGetDriveSideInfofromKDS() = 0;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBaseIntf::vCheckUSBPortEnabled()
       ***************************************************************************/
      /*!
       * \fn      t_Void vCheckUSBPortEnabled()
       * \brief  Method to Check if USB Port Enabled
       * \param [OUT] : none
       * \retval t_Void
       **************************************************************************/
      virtual t_Void vCheckUSBPortEnabled(t_Bool &bUSBPort1, t_Bool &bUSBPort2, t_Bool &bUSBPort3)= 0;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBaseIntf::vSetDriveModeInfo()
       ***************************************************************************/
      /*!
       * \fn      t_Void vSetDriveModeInfo()
       * \brief   Method set the drive mode information
       * \param   enDriveModeInfo : [IN] Drive mode info.
       * \retval  t_Void
       **************************************************************************/
      virtual t_Void vSetDriveModeInfo(const tenVehicleConfiguration enDriveModeInfo) = 0;

      /***************************************************************************
       ** FUNCTION:  t_String spi_tclStaticConfigurationBaseIntf::szGetClientManufacturerName()
       ***************************************************************************/
      /*!
       * \fn      t_String szGetClientManufacturerName()
       * \brief   Method to get the client manufacturer name.
       * \param   NONE
       * \retval  t_String
       **************************************************************************/
      virtual t_String szGetClientManufacturerName() const = 0;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBaseIntf::vSetNightModeInfo()
       ***************************************************************************/
      /*!
       * \fn      t_Void vSetNightModeInfo()
       * \brief   Method set the night  mode information
       * \param   enNightModeInfo : [IN] Night mode info
       * \retval  t_Void
       **************************************************************************/
      virtual t_Void vSetNightModeInfo(const tenVehicleConfiguration enNightModeInfo) = 0;

      /***************************************************************************
       ** FUNCTION:   t_String spi_tclStaticConfigurationBaseIntf::szGetChannelID()
       ***************************************************************************/
      /*!
       * \fn     t_String szGetChannelID()
       * \brief   Method to get channel ID
       * \retval  szChannelID : Unique Channel ID for Baidu Carlife technology.
       **************************************************************************/
      virtual t_String szGetChannelID() const = 0;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBaseIntf::bGetRotaryCtrlSupport()
       ***************************************************************************/
      /*!
       * \fn      t_Bool bGetRotaryCtrlSupport()
       * \brief   Method to know if Rotary Controller is supported
       * \param   NONE
       * \retval  t_String
       **************************************************************************/
      virtual t_Bool bGetRotaryCtrlSupport() = 0;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBaseIntf::vGetOemIconData(
       trVehicleBrandInfo& rfrVehicleBrandInfo)
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetOemIconData(trVehicleBrandInfo& rfrVehicleBrandInfo)
       * \brief   Method to read the EOL value and get the Brand configuration data
       from the look up table
       * \param   rfrVideoConfig: [IN]Vehicle Brand Data
       * \retval  NONE
       **************************************************************************/
      virtual t_Void vGetOemIconData(trVehicleBrandInfo& rfrVehicleBrandInfo) = 0;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBaseIntf::vGetDiPOExtendedFeaturesSupport()
       ***************************************************************************/
      /*!
       * \fn      vGetDiPOExtendedFeaturesSupport()
       * \brief   Method to get DiPO Extended Features Support
       * \retval  t_Void
       **************************************************************************/
      virtual t_Void vGetDiPOExtendedFeaturesSupport(trExtendedFeaturesSupport& rfrExtendedFeaturesSupport) = 0;


      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBaseIntf::vGetDriveModeInfo()
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetDriveModeInfo()
       * \brief   Method set the drive mode information
       * \param   enDriveModeInfo : [OUT] Drive mode info.
       * \retval  t_Void
       **************************************************************************/
      virtual t_Void vGetDriveModeInfo(tenVehicleConfiguration &rfoenDriveModeInfo) = 0;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBaseIntf::vGetNightModeInfo()
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetNightModeInfo()
       * \brief   Method get the night  mode information
       * \param   rfoenNightModeInfo : [OUT] Night mode info
       * \retval  t_Void
       **************************************************************************/
      virtual t_Void vGetNightModeInfo(tenVehicleConfiguration &rfoenNightModeInfo) = 0;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBaseIntf::vGetDisplayInputParam()
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetDisplayInputParam()
       * \brief   Method get the display input configuration.Input methods enabled.
       * \param   rfu8DisplayInput : [OUT] Display input parameter for DiPO
       * \retval  t_Void
       **************************************************************************/
      virtual t_Void vGetDisplayInputParam(t_U8 &rfu8DisplayInput) = 0;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBaseIntf::bGetNativeNavigationEnabled()
       ***************************************************************************/
      /*!
       * \fn      t_Bool bGetNativeNavigationEnabled()
       * \brief  Method to retrieve setting for native navigation
       * \param [OUT] : none
       * \retval t_Bool
       **************************************************************************/
      virtual t_Bool bGetNativeNavigationEnabled() = 0 ;

      /***************************************************************************
       ** FUNCTION:  virtual t_Bool spi_tclStaticConfigurationBaseIntf::bSetUpAPonOOBT(const t_String coszVehicleModelName)
       ***************************************************************************/
      /*!
       * \fn      virtual t_Bool bSetUpAPonOOBT(const t_String coszVehicleModelName)
       * \brief   Method to Read the OOBT Enabled.
       * \param   NONE
       * \retval  t_Bool
       **************************************************************************/
      virtual t_Bool bSetUpAPonOOBT()= 0;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBaseIntf::vGetAAPCertificatesModel()
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetAAPCertificatesModel(t_String &rfszAAPCertificateModel)
       * \brief   Method to read the Model of the car
       * \param   rfszAAPCertificateModel:       [IN] Device Category
       * \retval  NONE
       **************************************************************************/
      virtual t_Void vGetAAPCertificatesModel(t_String &rfszAAPCertificateModel)= 0;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBaseIntf::vCarplayDipoConfigurations()
       ***************************************************************************/
      /*!
       * \fn      t_Void vCarplayDipoConfigurations()
       * \brief   Method to change the carplay dipo configurations
       * \param   NONE
       * \retval  t_Void
       **************************************************************************/
      virtual t_Void vSelectCarplayConfiguration()= 0;

      /***************************************************************************
       ** FUNCTION: tenDAPAttestationInfo spi_tclStaticConfigurationBaseIntf::enGetDAPAttestationInfo(...)
       ***************************************************************************/
      /*!
       * \fn      enGetDAPAttestationInfo()
       * \brief   Method to get DAP attestation information
       * \retval  tenDAPAttestationInfo
       **************************************************************************/
      virtual tenDAPAttestationInfo enGetDAPAttestationInfo() const = 0;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBaseIntf::vGetEAPAppInfo(
       trEAPAppInfo& rfrEAPAppInfo)
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetEAPAppInfo(trEAPAppInfo& rfrEAPAppInfo)
       * \brief   Method to get EAP App info
       * \param   rfrEAPAppInfo: [IN]EAP App information
       * \retval  NONE
       **************************************************************************/
      virtual t_Void vGetEAPAppInfo(trEAPAppInfo& rfrEAPAppInfo)= 0;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBaseIntf::szGetVehicleModelNumber()
       ***************************************************************************/
      /*!
       * \fn      szGetVehicleModelNumber()
       * \brief   Method to get Vehicle Model Number information
       * \retval  t_Void
       **************************************************************************/
      virtual t_Void szGetVehicleModelNumber(t_String& szVehicleModelNumber) const = 0;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBaseIntf::bIsLocDataEnabled()
       ***************************************************************************/
      /*!
       * \fn      t_Bool bIsLocDataEnabled()
       * \brief  Method to check if Location data is enabled.
       * This method was added to check whether location data is enabled or not.
       * This function enable sensor data irrespective of Native/External Navigation.
       * For PSA,SUZUKI,GM ->this is set to true always
       * For AIVI ->This will be true only for Navi variants.
       * \param [OUT] : none.
       * \retval t_Bool
       **************************************************************************/
      virtual t_Bool bIsLocDataEnabled() = 0;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBaseIntf::vGetTouchScreenType
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetTouchScreenType()
       * \brief  Method to retrieve default setting of Touch Screen Type
       * \param [OUT] : returns the value of Touch screen
       * \retval t_Void
       **************************************************************************/
      virtual t_Void vGetTouchScreenType(tenAAPTouchScreenType &rfenScreenType)= 0;

      /***************************************************************************
       ** FUNCTION:  t_String spi_tclStaticConfigurationBaseIntf::szGetClientID()
       ***************************************************************************/
      /*!
       * \fn      t_String szGetClientID()
       * \brief   Method to get the client ID.
       * \param   NONE
       * \retval  t_String
       **************************************************************************/
      virtual t_String szGetClientID() const = 0;

      /***************************************************************************
       ** FUNCTION:  t_String spi_tclStaticConfigurationBaseIntf::szGetClientFriendlyName()
       ***************************************************************************/
      /*!
       * \fn      t_String szGetClientFriendlyName()
       * \brief   Method to get the client friendly name.
       * \param   NONE
       * \retval  t_String
       **************************************************************************/
      virtual t_String szGetClientFriendlyName() const = 0;

      /***************************************************************************
       ** FUNCTION:  tenRegion spi_tclStaticConfigurationBaseIntf::enGetRegion()
       ***************************************************************************/
      /*!
       * \fn      tenRegion enGetRegion()
       * \brief   Method to get the current vehicle sale region
       * \param   NONE
       * \retval  tenRegion
       **************************************************************************/
      virtual tenRegion enGetRegion() const = 0;

      /***************************************************************************
       ** FUNCTION:  tenATBoxSupportedState spi_tclStaticConfigurationBaseIntf::vGetATBoxSupportedState()
       ***************************************************************************/
      /*!
       * \fn      tenATBoxSupportedState vGetATBoxSupportedState()
       * \brief   Method to read supported AT Box features info.
       * \param   NONE
       * \retval  tenATBoxSupportedState
       **************************************************************************/
      virtual tenATBoxSupportedState vGetATBoxSupportedState() const = 0;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBaseIntf::bGetMLNotificationSetting()
       ***************************************************************************/
      /*!
       * \fn      bGetMLNotificationSetting()
       * \brief   Method to read MLNotification On/Off setting from EOL.
       * \param   NONE
       * \retval  t_Bool: true - if MLNotification is ON, else false.
       **************************************************************************/
      virtual t_Bool bGetMLNotificationSetting() = 0;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBaseIntf::vReadSysConfigs()
       ***************************************************************************/
      /*!
       * \fn      t_Void vReadSysConfigs()
       * \brief   Method to read & store the system configurations from KDS.
       * \param   NONE
       * \retval  t_Void
       **************************************************************************/
      virtual t_Void vReadSysConfigs()= 0;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBaseIntf::vSetSelectionModeStoredInDatapool
       ***************************************************************************/
      /*!
       * \fn     t_Bool vSetSelectionModeStoredInDatapool(tenDeviceSelectionMode enStoredSelectionMode)
       * \brief  Method to get the selection mode stored in datapool
       * \param  enStoredSelectionMode:[IN]Selection mode stored in datapool
       * \retval t_Void
       **************************************************************************/
      virtual t_Void vSetSelectionModeStoredInDatapool(tenDeviceSelectionMode enStoredSelectionMode)= 0;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBaseIntf::vStoreDevicePreferencefromDatapool
       ***************************************************************************/
      /*!
       * \fn     t_Bool vStoreDevicePreferencefromDatapool(tenDeviceCategory enAndroidDevicePreferencefromDatapool..)
       * \brief  Method to store device preference from datapool
       * \param  enAndroidDevicePreferencefromDatapool:[IN]Android device preference
       * \param  enAppleDevicePreferencefromDatapool:[IN]Apple device preference
       * \retval t_Void
       **************************************************************************/
      virtual t_Void vStoreDevicePreferencefromDatapool(tenDeviceCategory enAndroidDevicePreferencefromDatapool,
               tenDeviceCategory enAppleDevicePreferencefromDatapool)= 0;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBaseIntf::vDisplayAudioSettings()
       ***************************************************************************/
      /*!
       * \fn     t_Void vDisplayAudioSettings()
       * \brief  To print the audio settings data
       * \retval  t_Void
       **************************************************************************/
      virtual t_Void vDisplayAudioSettings()= 0;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBaseIntf::vDisplayVideoSettings()
       ***************************************************************************/
      /*!
       * \fn     t_Void vDisplayVideoSettings()
       * \brief  To print the video settings data
       * \retval  t_Void
       **************************************************************************/
      virtual t_Void vDisplayVideoSettings()= 0;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBaseIntf::vDisplayAppSettings()
       ***************************************************************************/
      /*!
       * \fn     t_Void vDisplayAppSettings()
       * \brief  To print the settings
       * \retval t_Void
       **************************************************************************/
      virtual t_Void vDisplayAppSettings()= 0;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBaseIntf::vGetSPIConfigurationData()
       ***************************************************************************/
      /*!
       * \fn      vGetSPIConfigurationData()
       * \brief   Method get the KDS data
       * \param   u8Data : [out]  DataBuffer
       * \retval  t_Void
       **************************************************************************/
      virtual t_Void vGetSPIConfigurationData()= 0;

      /** FUNCTION:  tenEnabledInfo spi_tclStaticConfigurationBaseIntf::enGetOnCarEnableSetVal()
       ***************************************************************************/
      /*!
       * \fn     tenEnabledInfo enGetOnCarEnableSetVal()
       * \brief  To get the default OnCar Enable Setting Value
       * \retval tenEnabledInfo
       **************************************************************************/
      virtual tenEnabledInfo enGetOnCarEnableSetVal() const= 0;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBaseIntf::vSetVehicleManufacturerInformationtoKDS()
       ***************************************************************************/
      /*!
       * \fn     tenEnabledInfo enGetOnCarEnableSetVal()
       * \brief  To set VehicleManufacturerInformation value to static configreader
       * \retval t_Void
       **************************************************************************/
      virtual t_Void vSetVehicleManufacturerInformationtoKDS(t_Bool bResult,
               trVehicleManufacturerInformation rVehicleManufacturerInfoAttr) = 0;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBaseIntf::bGetITCommanderSupport())
       ***************************************************************************/
      /*!
       * \fn      t_Bool bGetITCommanderSupport()
       * \brief   Function to Get ITCommander Support
       * \retval  NONE
       **************************************************************************/
      virtual t_Bool bGetITCommanderSupport() = 0;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBaseIntf::bGetSysVehicleInfo())
       ***************************************************************************/
      /*!
       * \fn      t_Bool bGetSysVehicleInfo()
       * \brief   Function to Get Region, Vendor and Vehicle
       * \retval  t_Bool
       **************************************************************************/
      virtual t_Bool bGetSysVehicleInfo(t_U8 &u8RegionCode, t_U8 &u8VendorCode, t_U16 &u16VehicleCode) = 0;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBaseIntf::bGetCarlifeIOSFeatureSupport
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetCarlifeIOSFeatureSupport()
       * \brief  Method to retrieve Carlife Ios enable info from policy
       * \retval t_Bool
       **************************************************************************/
      virtual t_Bool bGetCarlifeIOSFeatureSupport() = 0;

      /***************************************************************************
       ** FUNCTION: t_U8 spi_tclStaticConfigurationBase::u8ReadOEMBrandFromKDS()
       ***************************************************************************/
      /*!
       * \fn      t_U8 u8ReadOEMBrandFromKDS()
       * \brief   Method to read the KDS value and get the Brand type info
       * \retval  NONE
       **************************************************************************/
      virtual t_U8 u8ReadOEMBrandFromKDS() = 0;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBase::vGetVehicleInfoDataMySPIN()
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetVehicleInfoDataMySPIN()
       * \brief   Method to get the vehicle manufacturer name and variant info
       * \param   rfrVehicleInfo: [IN]Vehicle Info Data
       * \retval  NONE
       **************************************************************************/
      virtual t_Void vGetVehicleInfoDataMySPIN(trVehicleInfo& rfrVehicleInfo) = 0;

      /***************************************************************************
      FUNCTION: t_U16 spi_tclStaticConfigurationBaseIntf::u16ReadVehicleTypeFromKDS())
      ***************************************************************************/
      /*!
      * \fn      t_U16 u16ReadVehicleTypeFromKDS()
      * \brief   Function to Get Vehicle Type
      * \retval  t_U16
      ***************************************************************************/
      virtual t_U16 u16ReadVehicleTypeFromKDS() = 0;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBaseIntf::bIsDeviceWhitelisted
       ***************************************************************************/
      /*!
       * \brief  check if the device is whitelisted based on VID and PID of device
       * \param  [IN] u32ProductID :Product ID
       * \param  [IN] u32VendorID : Vendor ID
       * \param  [IN] enReqDevCat : Device Category
       ***************************************************************************/
      virtual t_Bool bIsDeviceWhitelisted(t_U32 u32ProductID, t_U32 u32VendorID, tenDeviceCategory enReqDevCat) = 0;

      /***************************************************************************
      ** FUNCTION:  t_Bool spi_tclStaticConfigurationBaseIntf::bReadCPWStatus()
      ***************************************************************************/
      /*!
      * \fn      t_Bool bReadCPWStatus()
      * \brief   Method to Read the KDS value of CPW
      * \param   NONE
      * \retval  t_Bool
      **************************************************************************/
      virtual t_Bool bReadCPWStatus() = 0;
    /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetCarlifeGPSDataSupportflag()
    ***************************************************************************/
    /*!
    * \fn     t_Bool bGetCarlifeGPSDataSupportflag() const
    * \brief  Interface to get location data enabled setting for carlfie
    * \retval t_Bool
    **************************************************************************/
    virtual t_Bool bGetCarlifeGPSDataSupportflag() const = 0;

      /***************************************************************************
      ** FUNCTION:  t_Bool spi_tclStaticConfigurationBaseIntf::vGetDefaultKeys
      ***************************************************************************/
      /*!
      * \brief  vGetDefaultKeys from the project
      * \param  [OUT] rfrmapTechSuppDefkeys : Default Supported Keys
      ***************************************************************************/
      virtual t_Void vGetDefaultKeys(std::map<tenDeviceCategory,std::set<tenKeyCode>>& rfrmapTechSuppDefkeys) = 0;

      /***************************************************************************
      ** FUNCTION:  t_Bool spi_tclStaticConfigurationBaseIntf::bGetCarPlayGPSDataSupportflag()
      ***************************************************************************/
      /*!
      * \fn     t_Bool bGetCarPlayGPSDataSupportflag()
      * \brief  Interface to get GPS data enabled setting
      * \retval t_Bool
      **************************************************************************/
      virtual t_Bool bGetCarPlayGPSDataSupportflag() = 0;

      /***************************************************************************
       ****************************END OF PUBLIC *********************************
       ***************************************************************************/
};
#endif //spi_tclStaticConfigurationBaseIntfIntf_
