/***********************************************************************/
/*!
 * \file  spi_tclStaticConfigurationBase.h
 * \brief  Implementation of the Class spi_tclStaticConfigurationBase
 *************************************************************************
 \verbatim


 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:
 AUTHOR:         kmo6kor
 COPYRIGHT:      &copy; 2017 Robert Bosch Car Multimedia GmbH
 HISTORY:
 Date        | Author                | Modification
 07.07.2018    kmo6kor			    Initial Version
 \endverbatim
 *************************************************************************/

#include "SPITypes.h"
#include "Lock.h"
#include "spi_tclCalibrationReaderBase.h"
#include "spi_tclAppSettings.h"
#include "spi_tclConnSettings.h"
#include "spi_tclAudioSettings.h"
#include "spi_tclDataServiceSettings.h"
#include "spi_tclResourceMngrSettings.h"
#include "spi_tclVideoSettings.h"
#include "spi_tclDefaultSettings.h"
#include "spi_tclDiscovererSettings.h"
#include "Xmlable.h"
#include "spi_tclStaticConfigurationBaseIntf.h"

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/
/******************************************************************************
 | defines and macros (scope: global)
 |----------------------------------------------------------------------------*/
#ifndef SPI_TCLSTATICCONFIGURATIONBASE_
#define SPI_TCLSTATICCONFIGURATIONBASE_

class spi_tclStaticConfigurationBase: public spi_tclStaticConfigurationBaseIntf, public shl::xml::tclXmlReadable
{

   public:
      /***************************************************************************
       *********************************PUBLIC*************************************
       ***************************************************************************/

      /***************************************************************************
       ** FUNCTION:spi_tclStaticConfigurationBase(spi_tclCalibrationReaderBase *poCalibrationReader)
       ***************************************************************************/
      /*!
       * \fn      spi_tclStaticConfigurationBase(spi_tclCalibrationReaderBase *poCalibrationReader)
       * \brief   Constructor
       **************************************************************************/
      spi_tclStaticConfigurationBase(spi_tclCalibrationReaderBase *poCalibrationReader);

      /***************************************************************************
       ** FUNCTION:~spi_tclStaticConfigurationBase
       ***************************************************************************/
      /*!
       * \fn      ~spi_tclStaticConfigurationBase
       * \brief   Destructor
       **************************************************************************/
      virtual ~spi_tclStaticConfigurationBase();

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vReadSPISettings()
       ***************************************************************************/
      /*!
       * \fn     t_Void vReadSPISettings()
       * \brief  To read the SPI settings from XML
       * \retval t_Void
       **************************************************************************/
      t_Void vReadSPISettings();

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bNonCertAppsReq()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bNonCertAppsReq()
       * \brief  To get whether the Non certified Applications are required
       *          when the application certification is enabled for the
       *          particular version of the device.
       * \retval t_Bool
       **************************************************************************/
      t_Bool bGetNonCertAppsReq() const;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBase::bGetAAPNavigationEnabledInfo()
       ***************************************************************************/
      /*!
       * \fn     bGetAAPNavigationEnabledInfo(t_Bool bEnable)
       * \brief  method to get the navigation feature enable settings for AAP.
       * \param  bEnable       : [IN] TRUE-allows to display navigation updates
       *                         for AAP
       *                         FALSE - doesn't allow
       * \retval t_Bool
       **************************************************************************/
      t_Bool bGetAAPNavigationEnabledInfo() const;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBase::bGetAAPMediaPlaybackEnabledInfo()
       ***************************************************************************/
      /*!
       * \fn     bGetAAPMediaPlaybackEnabledInfo(t_Bool bEnable)
       * \brief  method to get the media playback feature enable settings for AAP.
       * \param  bEnable       : [IN] TRUE-allows to display media playback updates
       *                         for AAP
       *                         FALSE - doesn't allow
       * \retval t_Bool
       **************************************************************************/
      t_Bool bGetAAPMediaPlaybackEnabledInfo() const;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBase::bGetAAPNotificationUpdatesEnabledInfo()
       ***************************************************************************/
      /*!
       * \fn     bGetAAPNotificationUpdatesEnabledInfo(t_Bool bEnable)
       * \brief  method to get notification update enable settings for AAP.
       * \param  bEnable       : [IN] TRUE-allows to display notification updates
       *                         for AAP
       *                         FALSE - doesn't allow
       * \retval t_Bool
       **************************************************************************/
      t_Bool bGetAAPNotificationUpdatesEnabledInfo() const;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBase::bGetTBTUpdatesEnabledInfo()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetTBTUpdatesEnabledInfo()
       * \brief  Provides information on whether TBT updates are enabled or disabled.
       * \retval t_Bool TRUE- Enabled FALSE-Disabled
       **************************************************************************/
      t_Bool bGetTBTUpdatesEnabledInfo() const;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBase::bGetAssisstantGuideUpdatesEnabledInfo()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetAssisstantGuideUpdatesEnabledInfo()
       * \brief  Provides information on whether Assisstant Guide updates are enabled
       *         or disabled.
       * \retval t_Bool TRUE- Enabled FALSE-Disabled
       **************************************************************************/
      t_Bool bGetAssisstantGuideUpdatesEnabledInfo() const;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBase::vSetRegion(.)
       ***************************************************************************/
      /*!
       * \fn     t_Void vSetRegion(tenRegion enRegion)
       * \brief  Interface to set the region for application certification.
       *         It gives the info of which region CCC Guidelines should be followed
       *         for the Application Certification Filtering
       * \param  [IN] enRegion : Region enumeration
       * \sa
       **************************************************************************/
      t_Void vSetRegion(tenRegion enRegion);

      /***************************************************************************
       ** FUNCTION:  t_String spi_tclStaticConfigurationBase::szGetRegion()
       ***************************************************************************/
      /*!
       * \fn     t_String szGetRegion()
       * \brief  To Get the which regions CCC guidelines should be followed for the
       *          Application certification.
       * \retval t_String
       **************************************************************************/
      t_String szGetRegion() const;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBase::bGetNonCertAppAllowedInfo()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetNonCertAppAllowedInfo()
       * \brief  Provides information on whether non certified ML apps are allowed
       *         in all modes(drive/park)
       * \retval t_Bool TRUE- Allowed FALSE-Not allowed
       **************************************************************************/
      t_Bool bGetNonCertAppAllowedInfo() const;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBase::bEnableCTSTest()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bEnableCTSTest()
       * \brief  To get whether the CTS test is enabled.
       * \retval t_Bool  TRUE- Enabled FALSE-Disabled
       **************************************************************************/
      t_Bool bEnableCTSTest();

      /***************************************************************************
       ** FUNCTION:  t_U8 spi_tclStaticConfigurationBase::u8GetSourceNumber(tenAudioDir)
       ***************************************************************************/
      /*!
       * \fn      u8GetSourceNumber( tenAudioDir enAudDir)
       * \brief   Function to get the Source Number for Audio Link & selected device
       *          category.
       * \param   [enAudDir]: Audio Link Direction
       * \retval  t_U8 value: Source Number
       **************************************************************************/
      t_U8 u8GetSourceNumber(tenAudioDir enAudDir) const;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBase::vGetRTPPayloadType
       (t_U32& rfu32RTPPayload)
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetRTPPayloadType(t_U32& u32RTPPayload)
       * \brief   Method to get the RTPPayload type
       * \param   [rfu32RTPPayload]: Reference to the RTP Payload type
       * \retval  NONE
       **************************************************************************/
      t_Void vGetRTPPayloadType(t_U32& rfu32RTPPayload) const;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBase::vGetAudioIPL
       (tU32& rfu32AudioIPL)
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetAudioIPL(tU32& rfu32AudioIPL)
       * \brief   Method to get the Audio IPL
       * \param   [rfu32AudioIPL]: Reference to the Audio IPL
       * \retval  t_Bool value: TRUE if Setting is Valid, FALSE Otherwise
       **************************************************************************/
      t_Void vGetAudioIPL(t_U32& rfu32AudioIPL) const;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBase::bGetAudioBlockingTriggerBasedOnCat()
       ***************************************************************************/
      /*!
       * \fn      t_Bool bGetAudBlockTriggerBasedOnCat()
       * \brief   Method to get the Audio Blocking enable/disable trigger based on
       *          Application category or not. This feature works only in conjunction with the
       *          Dynamic Audio feature
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bGetAudBlockTriggerBasedOnCat() const;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBase::vSetAudBlockTriggerBasedOnCat()
       ***************************************************************************/
      /*!
       * \fn      vSetAudBlockTriggerBasedOnCat()
       * \brief   Method to Set the Audio Blocking enable/disable trigger based on
       *          Application category or not. This feature works only in conjunction with the
       *          Dynamic Audio feature
       * \retval  t_Void
       **************************************************************************/
      t_Void vSetAudBlockTriggerBasedOnCat(t_Bool bIsML11OrAboveDevice);

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBase::bGetAudBlockTriggerBasedOnGlobaleMute()
       ***************************************************************************/
      /*!
       * \fn      t_Bool bGetAudBlockTriggerBasedOnGlobaleMute()
       * \brief   Method to get the Audio Blocking enable/disable trigger based on
       *          Global mute is enabled or not
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bGetAudBlockTriggerBasedOnGlobaleMute() const;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBase::vGetAudioPipeConfig()
       ***************************************************************************/
      /*!
       * \fn     t_Void vGetAudioPipeConfig()
       * \brief  provides audio configuration for the specified audio steam
       * \param  [OUT]rfrmapAudioPipeConfig : returns audio configuration as string
       **************************************************************************/
      t_Void vGetAudioPipeConfig(tmapAudioPipeConfig &rfmapAudioPipeConfig);

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBase::bGetMLDynAudioSupport()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetMLDynAudioSupport() const
       * \brief  To Get whether ML Audio is supported or not.
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bGetMLDynAudioSupport() const;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBase::vSetMLDynAudioSupport()
       ***************************************************************************/
      /*!
       * \fn     vSetMLDynAudioSupport() const
       * \brief  To Set whether ML Audio is supported or not.
       *         (For ML 1.0 = Not Supported, For Ml 1.1 onwards = supported)
       * \retval  t_Void
       **************************************************************************/
      t_Void vSetMLDynAudioSupport(t_Bool bIsML11OrAboveDevice);

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBase::bGetAAPMediaStreamRecEnabled()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetAAPMediaStreamRecEnabled() const
       * \brief  To Get whether AAP Media audio stream recording is enabled or not.
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bGetAAPMediaStreamRecEnabled() const;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBase::bGetAAPGuidanceStreamRecEnabled()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetAAPGuidanceStreamRecEnabled() const
       * \brief  To Get whether AAP Guidance audio stream recording is enabled or not.
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bGetAAPGuidanceStreamRecEnabled() const;

      /***************************************************************************
       ** FUNCTION: t_String spi_tclStaticConfigurationBase::szGetAudSinkMaxUnackedFrames()
       ***************************************************************************/
      /*!
       * \fn     szGetAudSinkMaxUnackedFrames() const
       * \brief  To Get "max-unacked-frames" value for Audio Sink configuration
       * \retval  t_String
       **************************************************************************/
      t_String szGetAudSinkMaxUnackedFrames() const;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBase::bGetRTPInSupport()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetRTPInSupport() const
       * \brief  To Get whether RTPIn support is enabled or not.
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bGetRTPInSupport() const;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBase::bGetMySIPNAdvAudioSupport()
       ***************************************************************************/
      /*!
       * \fn     bGetMySIPNAdvAudioSupport() const
       * \brief  To Get support for mySPIN Advanced Audio
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bGetMySIPNAdvAudioSupport() const;

      /***************************************************************************
       ** FUNCTION: t_String spi_tclStaticConfigurationBase::szGetDisableRealTimePriorityAudio()
       ***************************************************************************/
      /*!
       * \fn     szGetDisableRealTimePriorityAudio() const
       * \brief  To Get the value of disabling real time priority audio for Audio Sink configuration
       * \retval  t_String
       **************************************************************************/
      t_String szGetDisableRealTimePriorityAudio() const;

      /***************************************************************************
       ** FUNCTION: t_String spi_tclStaticConfigurationBase::szGetAudThreadsRealTimePriority()
       ***************************************************************************/
      /*!
       * \fn     szGetAudThreadsRealTimePriority() const
       * \brief  To Get Audio threads Real time priority value for Audio Sink configuration
       * \retval  t_String
       **************************************************************************/
      t_String szGetAudThreadsRealTimePriority() const;

      /***************************************************************************
       ** FUNCTION: t_String spi_tclStaticConfigurationBase::szGetAudSinkThresholdMs()
       ***************************************************************************/
      /*!
       * \fn     szGetAudSinkThresholdMs() const
       * \brief  To Get Audio Sink Threshold configuration
       * \retval  t_String
       **************************************************************************/
      t_String szGetAudSinkThresholdMs() const;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBase::vGetBdclMediaSinkConfig()
       ***************************************************************************/
      /*!
       * \fn     vGetBdclMediaSinkConfig() const
       * \brief  To Get BDCL Media Audio Sink configuration
       * \retval  t_Void
       **************************************************************************/
      t_Void vGetBdclMediaSinkConfig(std::map<t_String, t_String>& mapAudioConfig) const;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBase::vGetBdclTTSSinkConfig()
       ***************************************************************************/
      /*!
       * \fn     vGetBdclTTSSinkConfig() const
       * \brief  To Get BDCL TTS Audio Sink configuration
       * \retval  t_Void
       **************************************************************************/
      t_Void vGetBdclTTSSinkConfig(std::map<t_String, t_String>& mapAudioConfig) const;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBase::vGetBdclVRSinkConfig()
       ***************************************************************************/
      /*!
       * \fn     vGetBdclVRSinkConfig() const
       * \brief  To Get BDCL VR Audio Sink configuration
       * \retval  t_Void
       **************************************************************************/
      t_Void vGetBdclVRSinkConfig(std::map<t_String, t_String>& mapAudioConfig) const;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBase::vGetBdclVRSourceConfig()
       ***************************************************************************/
      /*!
       * \fn     vGetBdclVRSourceConfig() const
       * \brief  To Get BDCL VR Audio Source configuration
       * \retval  t_Void
       **************************************************************************/
      t_Void vGetBdclVRSourceConfig(std::map<t_String, t_String>& mapAudioConfig) const;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBase::bGetAudioMuteEnabled()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetAudioMuteEnabled() const
       * \brief  To Get whether audio mute is enabled (during last mode audio)
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bGetAudioMuteEnabled() const;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBase::bGetVolumeLockEnabled()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetVolumeLockEnabled() const
       * \brief  To Get whether Volume lock is enabled (during last mode audio)
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bGetVolumeLockEnabled() const;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBase::bGetMLDynAudioApplicability()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetMLDynAudioApplicability() const
       * \brief  To Get whether ML Dnamic Audio is applicable or not from policy.
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bGetMLDynAudioApplicability() const;

      /***************************************************************************
       ** FUNCTION:  tenDeviceSelectionMode spi_tclStaticConfigurationBase::enGetDeviceSelectionMode
       ***************************************************************************/
      /*!
       * \fn     tenDeviceSelectionMode enGetDeviceSelectionMode()
       * \brief  returns Device Selection Mode
       **************************************************************************/
      tenDeviceSelectionMode enGetDeviceSelectionMode(tenDeviceCategory enTechnologyType) const;

      /***************************************************************************
       ** FUNCTION:  tenDeviceCategory spi_tclStaticConfigurationBase::enGetTechnologyPreference
       ***************************************************************************/
      /*!
       * \fn     tenDeviceCategory enGetTechnologyPreference()
       * \brief  returns Device type preference
       **************************************************************************/
      tenDeviceCategory enGetTechnologyPreference(tenDeviceType enDeviceType) const;

      /***************************************************************************
       ** FUNCTION:  tenEnabledInfo spi_tclStaticConfigurationBase::enGetCertificateType
       ***************************************************************************/
      /*!
       * \fn     tenEnabledInfo enGetCertificateType()
       * \brief  returns Certificate type
       **************************************************************************/
      tenCertificateType enGetCertificateType() const;

      /***************************************************************************
       ** FUNCTION:  tenSelModePriority spi_tclStaticConfigurationBase::enGetSelectionModePriority
       ***************************************************************************/
      /*!
       * \fn     tenSelModePriority enGetSelectionModePriority()
       * \brief  returns Priority of selection mode
       **************************************************************************/
      tenSelModePriority enGetSelectionModePriority() const;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vGetPersistentStoragePath
       ***************************************************************************/
      /*!
       * \fn     t_Void vGetPersistentStoragePath()
       * \brief  returns path for persistent Storage
       **************************************************************************/
      t_Void vGetPersistentStoragePath(t_String &rfszPersStoragePath);

      /***************************************************************************
       ** FUNCTION:  t_U32 spi_tclStaticConfigurationBase::u32GetDeviceHistorySize
       ***************************************************************************/
      /*!
       * \fn     t_U32 u32GetDeviceHistorySize()
       * \brief  returns the Maximum size of Device History Database
       **************************************************************************/
      t_U32 u32GetDeviceHistorySize() const;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bIsXMLValidationEnabled
       ***************************************************************************/
      /*!
       * \fn     t_Bool bIsXMLValidationEnabled()
       * \brief  Method to check whether the XML validation is enabled or not
       * \retval t_Bool
       **************************************************************************/
      t_Bool bIsXMLValidationEnabled() const;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vGetHeadUnitInfo
       ***************************************************************************/
      /*!
       * \fn      vGetHeadUnitInfo()
       * \brief  Method to retrieve head unit information
       * \retval t_Void
       **************************************************************************/
      t_Void vGetMySPINIapAppInfo(trMySPINIapAppInfo& rMySPINIapAppInfo);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vGetMySPINAoapAppInfo
       ***************************************************************************/
      /*!
       * \fn     t_Void vGetMySPINAoapAppInfo(trMySPINAoapAppInfo& rfrMySPINAoapAppInfo)
       * \brief  Method to retrieve head unit information
       * \param  rfrMySPINAoapAppInfo:[IN]Aoap application information
       * \retval t_Void
       **************************************************************************/
      t_Void vGetMySPINAoapAppInfo(trMySPINAoapAppInfo& rfrMySPINAoapAppInfo);

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetCdbGPSSeviceEnabled()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetCdbGPSSeviceEnabled()
       * \brief  Interface to read whether GPS CDB service is enabled
       * \retval t_Bool
       **************************************************************************/
      t_Bool bGetCdbGPSSeviceEnabled() const;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetCdbLocationSeviceEnabled()
       ***************************************************************************/
      /*!
       * \fn     bGetCdbLocationSeviceEnabled()
       * \brief  Interface to read whether Location CDB service is enabled
       * \retval t_Bool
       **************************************************************************/
      t_Bool bGetCdbLocationSeviceEnabled() const;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetCdbPrivateSeviceEnabled()
       ***************************************************************************/
      /*!
       * \fn     bGetCdbPrivateSeviceEnabled()
       * \brief  Interface to read whether Private CDB service is enabled
       * \retval t_Bool
       **************************************************************************/
      t_Bool bGetCdbPrivateSeviceEnabled() const;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetCdbMediaMetadataSinkEnabled()
       ***************************************************************************/
      /*!
       * \fn     bGetCdbMediaMetadataSinkEnabled()
       * \brief  Interface to read whether CDB Sink Media Metadata sink is enabled
       * \retval t_Bool
       **************************************************************************/
      t_Bool bGetCdbMediaMetadataSinkEnabled() const;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetCdbMediaMetadataSeviceEnabled()
       ***************************************************************************/
      /*!
       * \fn     bGetCdbNaviMetadataSinkEnabled()
       * \brief  Interface to read whether CDB Navigation Metadata sink is enabled
       * \retval t_Bool
       **************************************************************************/
      t_Bool bGetCdbNaviMetadataSinkEnabled() const;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetEnvDataSubEnabled()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetEnvDataSubEnabled() const
       * \brief  Interface to get whether to subscribe for Environment data or not
       * \retval t_Bool
       **************************************************************************/
      t_Bool bGetEnvDataSubEnabled() const;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetGearStatusEnabled()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetGearStatusEnabled() const
       * \brief  Interface to get whether to subscribe for Gear Status data or not
       * \retval t_Bool
       **************************************************************************/
      t_Bool bGetGearStatusEnabled() const;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetAccelerometerDataEnabled()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetAccelerometerDataDisable() const
       * \brief  Interface to get whether to subscribe for Accelerometer data or not
       * \retval t_Bool
       **************************************************************************/
      t_Bool bGetAccelerometerDataEnabled() const;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetGyroDataEnabled()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetGyroDataEnabled() const
       * \brief  Interface to get whether to subscribe for Gyroscope data or not
       * \retval t_Bool
       **************************************************************************/
      t_Bool bGetGyroDataEnabled() const;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetParkBrakeDataEnabled()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetParkBrakeDataEnabled() const
       * \brief  Interface to get whether to subscribe for park brake data or not
       * \retval t_Bool
       **************************************************************************/
      t_Bool bGetParkBrakeDataEnabled() const;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bIsBochNavigation()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bIsBochNavigation() const
       * \brief  Interface to get the type of navigation used.
       * \retval t_Bool
       **************************************************************************/
      t_Bool bIsBochNavigation() const;

      /***************************************************************************
       ** FUNCTION:  t_U32 spi_tclStaticConfigurationBase:: u32GetSpeedTimerInterval()
       ***************************************************************************/
      /*!
       * \fn     u32GetSpeedTimerInterval() const
       * \brief  Interface to get the Speed timer interval in ms.
       * \retval t_U32
       **************************************************************************/
      t_U32 u32GetSpeedTimerInterval() const;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetLocDataEnabled()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetLocDataEnabled() const
       * \brief  Interface to get location data enabled setting
       * \retval t_Bool
       **************************************************************************/
      t_Bool bGetLocDataEnabled() const;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetMLNotiSettingVal()
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetMLNotiSettingVal()
       * \brief  To get the default ML Notification Setting Value
       * \retval t_Bool
       **************************************************************************/
      t_Bool bGetMLNotiSettingVal() const;

      /***************************************************************************
       ** FUNCTION:  tenEnabledInfo spi_tclStaticConfigurationBase::enGetMLLinkEnableSetVal()
       ***************************************************************************/
      /*!
       * \fn     tenEnabledInfo enGetMLLinkEnableSetVal()
       * \brief  To get the default ML Link Enable Setting Value
       * \retval tenEnabledInfo
       **************************************************************************/
      tenEnabledInfo enGetMLLinkEnableSetVal() const;

      /***************************************************************************
       ** FUNCTION:  tenEnabledInfo spi_tclStaticConfigurationBase::enGetDipoEnableSetVal()
       ***************************************************************************/
      /*!
       * \fn     tenEnabledInfo enGetDipoEnableSetVal()
       * \brief  To get the default Dipo Enable Setting Value
       * \retval tenEnabledInfo
       **************************************************************************/
      tenEnabledInfo enGetDipoEnableSetVal() const;

      /***************************************************************************
       ** FUNCTION:  tenEnabledInfo spi_tclStaticConfigurationBase::enGetAAPEnableSetVal()
       ***************************************************************************/
      /*!
       * \fn     tenEnabledInfo enGetAAPEnableSetVal()
       * \brief  To get the default AAP Enable Setting Value
       * \retval tenEnabledInfo
       **************************************************************************/
      tenEnabledInfo enGetAAPEnableSetVal() const;

      /***************************************************************************
       ** FUNCTION:  tenEnabledInfo spi_tclStaticConfigurationBase::enGetCarlifeEnableSetVal()
       ***************************************************************************/
      /*!
       * \fn     tenEnabledInfo enGetCarlifeEnableSetVal()
       * \brief  To get the default Carlife Enable Setting Value
       * \retval tenEnabledInfo
       **************************************************************************/
      tenEnabledInfo enGetCarlifeEnableSetVal() const;

      /***************************************************************************
       ** FUNCTION:  tenEnabledInfo spi_tclStaticConfigurationBase::enGetMySPINEnableSetVal()
       ***************************************************************************/
      /*!
       * \fn     tenEnabledInfo enGetMySPINEnableSetVal()
       * \brief  To get the default Myspin Enable Setting Value
       * \retval tenEnabledInfo
       **************************************************************************/
      tenEnabledInfo enGetMySPINEnableSetVal() const;

      /***************************************************************************
       ** FUNCTION:  tenDriveSideInfo  spi_tclStaticConfigurationBase::enGetSteeringWheelPos()
       ***************************************************************************/
      /*!
       * \fn     tenDriveSideInfo  enGetSteeringWheelPos()
       * \brief  To get the default Steering Wheel Position
       * \retval tenDriveSideInfo
       **************************************************************************/
      tenDriveSideInfo enGetSteeringWheelPos() const;

      /***************************************************************************
       ** FUNCTION:  tenDeviceSelectionMode spi_tclStaticConfigurationBase::enGetSelectMode()
       ***************************************************************************/
      /*!
       * \fn     tenDriveSideInfo  enGetSelectMode()
       * \brief  To get the default Selection Mode
       * \retval tenDeviceSelectionMode
       **************************************************************************/
      tenDeviceSelectionMode enGetSelectMode() const;

      /***************************************************************************
       ** FUNCTION:  tenDeviceCategory spi_tclStaticConfigurationBase::enGetAndroidDeviceTechnologyPreference()
       ***************************************************************************/
      /*!
       * \fn     tenDeviceCategory  enGetAndroidDeviceTechnologyPreference()
       * \brief  To get the default Technology Preference for Android Devices
       * \retval tenDeviceCategory
       **************************************************************************/
      tenDeviceCategory enGetAndroidDeviceTechnologyPreference() const;

      /***************************************************************************
       ** FUNCTION:  tenDeviceCategory spi_tclStaticConfigurationBase::enGetAppleDeviceTechnologyPreference()
       ***************************************************************************/
      /*!
       * \fn     tenDeviceCategory  enGetAppleDeviceTechnologyPreference()
       * \brief  To get the default Technology Preference for Apple Devices
       * \retval tenDeviceCategory
       **************************************************************************/
      tenDeviceCategory enGetAppleDeviceTechnologyPreference() const;

      /***************************************************************************
       ** FUNCTION:  t_U32 spi_tclStaticConfigurationBase::u32GetStartUpTimeInterval()
       ***************************************************************************/
      /*!
       * \fn      u32GetStartUpTimeInterval()
       * \brief   Function to get the CarPlay session startup time interval
       * \param   NONE
       * \retVal  t_U32 : Time interval in milli seconds.
       * \sa
       **************************************************************************/
      t_U32 u32GetStartUpTimeInterval() const;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBase::bGetLastModeSupport()
       ***************************************************************************/
      /*!
       * \fn     bGetLastModeSupport() const
       * \brief  To Get last mode support configuration for a device category
       * \param  enDevCat [IN]: Device category
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bGetLastModeSupport(tenDeviceCategory enDevCat) const;

      /***************************************************************************
       ** FUNCTION: tenLastModeSupport spi_tclStaticConfigurationBase::enGetLastModeSupport()
       ***************************************************************************/
      /*!
       * \fn     enGetLastModeSupport() const
       * \brief  To Get last mode support configuration
       * \retval  tenLastModeSupport
       **************************************************************************/
      tenLastModeSupport enGetLastModeSupport() const;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vGetScreenOffset(t_U32& u32Scre...
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetScreenOffset(t_U32& u32Screen_X_Offset,
       *          t_U32& u32Screen_Y_Offset)
       * \brief   To Get the Screen Offsets
       * \param   u32Screen_X_Offset : [OUT] X Offset
       * \param   u32Screen_Y_Offset : [OUT] Y Offset
       * \retval  t_Void
       * \sa
       **************************************************************************/
      t_Void vGetScreenOffset(trScreenOffset &rfoScreenOffset) const;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bIsServerSideScalingRequired()
       ***************************************************************************/
      /*!
       * \fn      t_Bool bIsServerSideScalingRequired()
       * \brief   To Check whether the server side scaling required
       * \retval  t_Bool : false if the scaling is not required, else true
       **************************************************************************/
      t_Bool bIsServerSideScalingRequired();

      /***************************************************************************
       ** FUNCTION:  t_U8 spi_tclStaticConfigurationBase::u8EnableTouchInputEvents()()
       ***************************************************************************/
      /*!
       * \fn      short t_U32 u8EnableTouchInputEvents()()
       * \brief   To Check whether the touch input handling is required
       * \retval  t_U8 0x0 -   if the touch input handling is not done using wayland
       *                       0x1 - if the Touch Input Handling is done via wayland
       **************************************************************************/
      t_U8 u8EnableTouchInputEvents();

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBase::vSetScreenSize()
       ***************************************************************************/
      /*!
       * \fn     t_Void vSetScreenSize(const trScreenAttributes& corfrScreenAttributes)
       * \brief  Interface to set the screen size of Head Unit.
       * \param  corScreenAttributes : [IN] Screen Setting attributes.
       * \retval t_Void
       **************************************************************************/
      t_Void vSetScreenSize(const trScreenAttributes& corfrScreenAttributes);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vSetContAttestFlag()
       ***************************************************************************/
      /*!
       * \fn      t_Void vSetContAttestFlag(t_U8 u8ContAttestFlag)
       * \brief   To enable/disable content attestation.Enabling or disableing should be done
       *          before the device is selected for the session
       * \pram    u8ContAttestFlag  : [IN] TRUE - Enable attestation
       *                                  FALSE - disable attestation
       * \retval  t_Void
       **************************************************************************/
      t_Void vSetContAttestFlag(t_U8 u8ContAttestFlag);

      /***************************************************************************
       ** FUNCTION:  t_String spi_tclStaticConfigurationBase::szGetVideoCodecType()
       ***************************************************************************/
      /*!
       * \fn      t_String szGetVideoCodecType()
       * \brief   To get the video codec type
       *          before the device is selected for the session
       * \retval  t_String
       **************************************************************************/
      t_String szGetVideoCodecType() const;

      /***************************************************************************
       ** FUNCTION:  t_U8 spi_tclStaticConfigurationBase::u8GetMaxUnackedFrames()
       ***************************************************************************/
      /*!
       * \fn      t_U8 u8GetMaxUnackedFrames()
       * \brief   To get the max number of frames that can be recieved with out ack
       * \retval  t_U8
       **************************************************************************/
      t_U8 u8GetMaxUnackedFrames() const;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBase::bGetAutoStartProjection()
       ***************************************************************************/
      /*!
       * \fn      t_Bool bGetAutoStartProjection()
       * \brief   To get whether the auto projection start is enabled or not
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bGetAutoStartProjection() const;

      /***************************************************************************
       ** FUNCTION:  t_U8 spi_tclStaticConfigurationBase::u8GetFramesPerSec()
       ***************************************************************************/
      /*!
       * \fn      t_U8 u8GetFramesPerSec()
       * \brief   To get the frame per pixel rate
       * \retval  t_U8
       **************************************************************************/
      t_U8 u8GetFramesPerSec() const;

      /***************************************************************************
       ** FUNCTION:  t_U8 spi_tclStaticConfigurationBase::u8GetContAttestSignFlag()
       ***************************************************************************/
      /*!
       * \fn      t_U8 u8GetContAttestSignFlag()
       * \brief   To get the Content attestation info signed flag
       * \retval  t_U8
       **************************************************************************/
      t_U8 u8GetContAttestSignFlag() const;

      /***************************************************************************
       ** FUNCTION:  tenPixelFormat spi_tclStaticConfigurationBase::enGetPixelFormat()
       ***************************************************************************/
      /*!
       * \fn      tenPixelFormat enGetPixelFormat()
       * \brief   To get the pixel format for display
       * \retval  tenPixelFormat
       **************************************************************************/
      tenPixelFormat enGetPixelFormat() const;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBase::vGetBdclVideoSinkConfig()
       ***************************************************************************/
      /*!
       * \fn     vGetBdclVideoSinkConfig() const
       * \brief  To Get BDCL VR Video Sink configuration
       * \retval  t_Void
       **************************************************************************/
      t_Void vGetBdclVideoSinkConfig(std::map<t_String, t_String>& mapVideoConfig) const;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vSetDeviceSelectionMode()
       ***************************************************************************/
      /*!
       * \fn      t_Void vSetDeviceSelectionMode(
       * \brief   Method to set the device selection mode to automatic/manual. Changes
       *          will take into effect on successive connection
       * \param   enSelectionMode : Device selection mode @see tenDeviceSelectionMode
       * \retval  t_Void
       **************************************************************************/
      t_Void vSetDeviceSelectionMode(tenDeviceSelectionMode enSelectionMode, tenDeviceCategory enTechnologyType);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vIntializeSPISettings
       ***************************************************************************/
      /*!
       * \fn     t_Void vIntializeSPISettings()
       * \brief  Reads settings from xml file and store it internally
       **************************************************************************/
      t_Void vIntializeSPISettings();

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bIsDeviceBlacklisted
       ***************************************************************************/
      /*!
       * \fn     t_Void bIsDeviceBlacklisted()
       * \brief  to check whether  the device is blacklisted.
       * \retval  t_Bool
       **************************************************************************/
      t_Bool bIsDeviceBlacklisted(t_S32 s32ProductID, t_S32 s32VendorID, tenDeviceCategory enDeviceCtegory =
               e8DEV_TYPE_DEFAULT);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vIsDeviceWhitelisted(...
       ***************************************************************************/
      /*!
       * \fn     t_Void vIsDeviceWhitelisted()
       * \brief  to check whether the connected device is whitelisted or not and return category
       **************************************************************************/
      t_Void vIsDeviceWhitelisted(t_S32 s32ProductID, t_S32 s32VendorID, tenDeviceCategory &rfenDevCat);

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bIsDeviceWhitelistingEnabled(...
       ***************************************************************************/
      /*!
       * \fn     t_Bool bIsDeviceWhitelistingEnabled()
       * \brief  to check whether the whitelisting is enabled for a project or not
       **************************************************************************/
      t_Bool bIsDeviceWhitelistingEnabled();

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bIsWhitelistEnabledCat(...
       ***************************************************************************/
      /*!
       * \fn     t_Bool bIsWhitelistEnabledCat()
       * \brief  to set the category for which whitelisting is enabled
       **************************************************************************/
      t_Bool bIsWhitelistEnabledCat(tenDeviceCategory enChosenDeviceCategory);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vGetDefaultProjUsageSettings
       ***************************************************************************/
      /*!
       * \fn     t_Bool vGetDefaultProjUsageSettings()
       * \brief  Method to retrieve default setting for projection usage
       * \param [OUT] : returns the current value of device projection enable
       * \enSPIType  : indicates the type of SPI technology. e8DEV_TYPE_UNKNOWN indicates default value for any SPI technology
       * \retval t_Void
       **************************************************************************/
      virtual t_Void vGetDefaultProjUsageSettings(tenEnabledInfo &enEnabledInfo, tenDeviceCategory /*enSPIType*/)
      {
         enEnabledInfo = e8USAGE_ENABLED;
      }
      ;

      /***************************************************************************
       ** FUNCTION: tenGeoCoordinateSystemType spi_tclStaticConfigurationBase::enGetGeoCoordinateSystem()
       ***************************************************************************/
      /*!
       * \fn      t_Void enGetGeoCoordinateSystem()
       * \brief   Method to read the supported coordinate system
       * \retval  tenGeoCoordinateSystemType : To indicate type of Geo Coordinate System
       **************************************************************************/
      virtual tenGeoCoordinateSystemType enGetGeoCoordinateSystem() override;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vGetSpiFeatureSupport(...)
       ***************************************************************************/
      /*!
       * \fn      vGetSpiFeatureSupport(trSpiFeatureSupport& rfrSpiFeatureSupport)
       * \brief   Method to read supported SPI features info.
       * \param   rfrSpiFeatureSupport: [OUT] Provides supported SPI features
       * \retval  NONE
       **************************************************************************/
      virtual t_Void vGetSpiFeatureSupport(trSpiFeatureSupport& rfrSpiFeatureSupport);

      /***************************************************************************
       ** FUNCTION: t_U8 spi_tclStaticConfigurationBase::u8GetAAPRestrictionInfo(t_Bool bDriveRestrictions)
       ***************************************************************************/
      /*!
       * \fn      t_U8 u8GetAAPRestrictionInfo(t_Bool bDriveRestrictions)
       * \brief   Method to get restrictions enabled in park/drive mode for Carplay
       * \param   bDriveRestrictions: [IN] drive restrictions
       * \retval  NONE
       **************************************************************************/
      virtual t_U8 u8GetAAPRestrictionInfo(t_Bool bDriveRestrictions);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vSetAAPFeatureRestrictions(const t_U8 cou8ParkModeRestrictionInfo,const t_U8 cou8DriveModeRestrictionInfo)
       ***************************************************************************/
      /*!
       * \fn      t_Void vSetAAPFeatureRestrictions(const t_U8 cou8ParkModeRestrictionInfo,const t_U8 cou8DriveModeRestrictionInfo)
       * \brief   Method to set Vehicle Park/Drive Mode Restriction
       * \param   NONE
       * \retval  t_Void
       **************************************************************************/
      virtual t_Void vSetAAPFeatureRestrictions(const t_U8 cou8ParkModeRestrictionInfo,
               const t_U8 cou8DriveModeRestrictionInfo);

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBase::vGetVehicleInfoDataAAP(
       trVehicleBrandInfo& rfrVehicleBrandInfo)
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetVehicleInfoDataAAP(trVehicleBrandInfo& rfrVehicleBrandInfo)
       * \brief   Method to read the EOL value and get the Brand configuration data
       from the look up table
       * \param   rfrVideoConfig: [IN]Vehicle Brand Data
       * \retval  NONE
       **************************************************************************/
      virtual t_Void vGetVehicleInfoDataAAP(trVehicleInfo& /*rfrVehicleInfo*/)
      {
         return;
      }

      /***************************************************************************
       ** FUNCTION:  t_String spi_tclStaticConfigurationBase::szGetModelYear()
       ***************************************************************************/
      /*!
       * \fn      t_String szGetModelYear()
       * \brief   Method to get the Model year of the vehicle
       * \param   NONE
       * \retval  t_String
       **************************************************************************/
      virtual t_String szGetModelYear() = 0;

      /***************************************************************************
       ** FUNCTION: t_String spi_tclStaticConfigurationBase::szGetSoftwareBuild()
       ***************************************************************************/
      /*!
       * \fn      t_String szGetSoftwareBuild(t_String &rfrSoftwareBuild)
       * \brief   Method to get Software Build version
       * \param   NONE
       * \retval  NONE
       **************************************************************************/
      virtual t_String szGetSoftwareBuild();

      /***************************************************************************
       ** FUNCTION:  t_String spi_tclStaticConfigurationBase::szGetHardwareVersion()
       ***************************************************************************/
      /*!
       * \fn      t_String szGetHardwareVersion()
       * \brief   Method to get the accessory hardware version
       * \param   NONE
       * \retval  t_String
       **************************************************************************/
      virtual t_String szGetHardwareVersion() = 0;

      /***************************************************************************
       ** FUNCTION:  t_String spi_tclStaticConfigurationBase::szGetHeadUnitModelName()
       ***************************************************************************/
      /*!
       * \fn      t_String szGetHeadUnitModelName()
       * \brief   Method to get the headunit model name.
       * \param   NONE
       * \retval  t_String
       **************************************************************************/
      virtual t_String szGetHeadUnitModelName() = 0;

      /***************************************************************************
       ** FUNCTION:  t_String spi_tclStaticConfigurationBase::szGetHeadUnitManufacturerName()
       ***************************************************************************/
      /*!
       * \fn      t_String szGetHeadUnitManufacturerName()
       * \brief   Method to get the Headunit manufacturer name.
       * \param   NONE
       * \retval  t_String
       **************************************************************************/
      virtual t_String szGetHeadUnitManufacturerName() = 0;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBase::vGetAAPCertificatesConfigData()
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetCertsConfigData(trAAAutoCertConfigData& rfrCertsConfig)
       * \brief   Method to read the Certificate Configuration file
       * \param   rfrCertsConfig: [IN] Certificate Config Data
       * \retval  NONE
       **************************************************************************/
      virtual t_Void vGetAAPCertificatesConfigData(trAAPCertConfigData& rfrCertsConfig);

      /***************************************************************************
       ** FUNCTION:  tenDriveSideInfo spi_tclStaticConfigurationBase::enGetDriveSideInfofromKDS(...)
       ***************************************************************************/
      /*!
       * \fn      enGetDriveSideInfofromKDS()
       * \brief   Method to read Drive side info from KDS
       * \param   NONE
       * \retval  Drive Side enum
       **************************************************************************/
      virtual tenDriveSideInfo enGetDriveSideInfofromKDS();

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBase::vCheckUSBPortEnabled()
       ***************************************************************************/
      /*!
       * \fn      t_Void vCheckUSBPortEnabled()
       * \brief  Method to Check if USB Port Enabled
       * \param [OUT] : none
       * \retval t_Void
       **************************************************************************/
      virtual t_Void vCheckUSBPortEnabled(t_Bool &bUSBPort1, t_Bool &bUSBPort2, t_Bool &bUSBPort3);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vSetDriveModeInfo()
       ***************************************************************************/
      /*!
       * \fn      t_Void vSetDriveModeInfo()
       * \brief   Method set the drive mode information
       * \param   enDriveModeInfo : [IN] Drive mode info.
       * \retval  t_Void
       **************************************************************************/
      virtual t_Void vSetDriveModeInfo(const tenVehicleConfiguration enDriveModeInfo);

      /***************************************************************************
       ** FUNCTION:  t_String spi_tclStaticConfigurationBase::szGetClientManufacturerName()
       ***************************************************************************/
      /*!
       * \fn      t_String szGetClientManufacturerName()
       * \brief   Method to get the client manufacturer name.
       * \param   NONE
       * \retval  t_String
       **************************************************************************/
      virtual t_String szGetClientManufacturerName() const = 0;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vSetNightModeInfo()
       ***************************************************************************/
      /*!
       * \fn      t_Void vSetNightModeInfo()
       * \brief   Method set the night  mode information
       * \param   enNightModeInfo : [IN] Night mode info
       * \retval  t_Void
       **************************************************************************/
      virtual t_Void vSetNightModeInfo(const tenVehicleConfiguration enNightModeInfo);

      /***************************************************************************
       ** FUNCTION:   t_String spi_tclStaticConfigurationBase::szGetChannelID()
       ***************************************************************************/
      /*!
       * \fn     t_String szGetChannelID()
       * \brief   Method to get channel ID
       * \retval  szChannelID : Unique Channel ID for Baidu Carlife technology.
       **************************************************************************/
      virtual t_String szGetChannelID() const;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetRotaryCtrlSupport()
       ***************************************************************************/
      /*!
       * \fn      t_Bool bGetRotaryCtrlSupport()
       * \brief   Method to know if Rotary Controller is supported
       * \param   NONE
       * \retval  t_String
       **************************************************************************/
      virtual t_Bool bGetRotaryCtrlSupport() = 0;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBase::vGetOemIconData(
       trVehicleBrandInfo& rfrVehicleBrandInfo)
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetOemIconData(trVehicleBrandInfo& rfrVehicleBrandInfo)
       * \brief   Method to read the EOL value and get the Brand configuration data
       from the look up table
       * \param   rfrVideoConfig: [IN]Vehicle Brand Data
       * \retval  NONE
       **************************************************************************/
      virtual t_Void vGetOemIconData(trVehicleBrandInfo& rfrVehicleBrandInfo) = 0;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vGetDiPOExtendedFeaturesSupport()
       ***************************************************************************/
      /*!
       * \fn      vGetDiPOExtendedFeaturesSupport()
       * \brief   Method to get DiPO Extended Features Support
       * \retval  t_Void
       **************************************************************************/
      virtual t_Void vGetDiPOExtendedFeaturesSupport(trExtendedFeaturesSupport& rfrExtendedFeaturesSupport) = 0;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vGetDriveModeInfo()
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetDriveModeInfo()
       * \brief   Method set the drive mode information
       * \param   enDriveModeInfo : [OUT] Drive mode info.
       * \retval  t_Void
       **************************************************************************/
      virtual t_Void vGetDriveModeInfo(tenVehicleConfiguration &rfoenDriveModeInfo);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vGetNightModeInfo()
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetNightModeInfo()
       * \brief   Method get the night  mode information
       * \param   rfoenNightModeInfo : [OUT] Night mode info
       * \retval  t_Void
       **************************************************************************/
      virtual t_Void vGetNightModeInfo(tenVehicleConfiguration &rfoenNightModeInfo);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vGetDisplayInputParam()
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetDisplayInputParam()
       * \brief   Method get the display input configuration.Input methods enabled.
       * \param   rfu8DisplayInput : [OUT] Display input parameter for DiPO
       * \retval  t_Void
       **************************************************************************/
      virtual t_Void vGetDisplayInputParam(t_U8 &rfu8DisplayInput) = 0;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBase::bGetNativeNavigationEnabled()
       ***************************************************************************/
      /*!
       * \fn      t_Bool bGetNativeNavigationEnabled()
       * \brief  Method to retrieve setting for native navigation
       * \param [OUT] : none
       * \retval t_Bool
       **************************************************************************/
      virtual t_Bool bGetNativeNavigationEnabled() {return true;}

      /***************************************************************************
       ** FUNCTION:  virtual t_Bool spi_tclStaticConfigurationBase::bSetUpAPonOOBT(const t_String coszVehicleModelName)
       ***************************************************************************/
      /*!
       * \fn      virtual t_Bool bSetUpAPonOOBT(const t_String coszVehicleModelName)
       * \brief   Method to Read the OOBT Enabled.
       * \param   NONE
       * \retval  t_Bool
       **************************************************************************/
      virtual t_Bool bSetUpAPonOOBT();

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBase::vGetAAPCertificatesModel()
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetAAPCertificatesModel(t_String &rfszAAPCertificateModel)
       * \brief   Method to read the Model of the car
       * \param   rfszAAPCertificateModel:       [IN] Device Category
       * \retval  NONE
       **************************************************************************/
      virtual t_Void vGetAAPCertificatesModel(t_String &rfszAAPCertificateModel);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vCarplayDipoConfigurations()
       ***************************************************************************/
      /*!
       * \fn      t_Void vCarplayDipoConfigurations()
       * \brief   Method to change the carplay dipo configurations
       * \param   NONE
       * \retval  t_Void
       **************************************************************************/
      virtual t_Void vSelectCarplayConfiguration();

      /***************************************************************************
       ** FUNCTION: tenDAPAttestationInfo spi_tclStaticConfigurationBase::enGetDAPAttestationInfo(...)
       ***************************************************************************/
      /*!
       * \fn      enGetDAPAttestationInfo()
       * \brief   Method to get DAP attestation information
       * \retval  tenDAPAttestationInfo
       **************************************************************************/
      virtual tenDAPAttestationInfo enGetDAPAttestationInfo() const;

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBase::vGetEAPAppInfo(
       trEAPAppInfo& rfrEAPAppInfo)
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetEAPAppInfo(trEAPAppInfo& rfrEAPAppInfo)
       * \brief   Method to get EAP App info
       * \param   rfrEAPAppInfo: [IN]EAP App information
       * \retval  NONE
       **************************************************************************/
      virtual t_Void vGetEAPAppInfo(trEAPAppInfo& rfrEAPAppInfo);

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBase::szGetVehicleModelNumber()
       ***************************************************************************/
      /*!
       * \fn      szGetVehicleModelNumber()
       * \brief   Method to get Vehicle Model Number information
       * \retval  t_Void
       **************************************************************************/
      virtual t_Void szGetVehicleModelNumber(t_String& szVehicleModelNumber) const;

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBase::bIsLocDataEnabled()
       ***************************************************************************/
      /*!
       * \fn      t_Bool bIsLocDataEnabled()
       * \brief  Method to check if Location data is enabled.
       * This method was added to check whether location data is enabled or not.
       * This function enable sensor data irrespective of Native/External Navigation.
       * For PSA,SUZUKI,GM ->this is set to true always
       * For AIVI ->This will be true only for Navi variants.
       * \param [OUT] : none.
       * \retval t_Bool
       **************************************************************************/
      virtual t_Bool bIsLocDataEnabled()
      {
         return true;
      }

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBase::vGetTouchScreenType
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetTouchScreenType()
       * \brief  Method to retrieve default setting of Touch Screen Type
       * \param [OUT] : returns the value of Touch screen
       * \retval t_Void
       **************************************************************************/
      virtual t_Void vGetTouchScreenType(tenAAPTouchScreenType &rfenScreenType)
      {
         rfenScreenType = e8_TOUCHSCREEN_RESISTIVE;
      }

      /***************************************************************************
       ** FUNCTION:  t_String spi_tclStaticConfigurationBase::szGetClientID()
       ***************************************************************************/
      /*!
       * \fn      t_String szGetClientID()
       * \brief   Method to get the client ID.
       * \param   NONE
       * \retval  t_String
       **************************************************************************/
      virtual t_String szGetClientID() const = 0;

      /***************************************************************************
       ** FUNCTION:  t_String spi_tclStaticConfigurationBase::szGetClientFriendlyName()
       ***************************************************************************/
      /*!
       * \fn      t_String szGetClientFriendlyName()
       * \brief   Method to get the client friendly name.
       * \param   NONE
       * \retval  t_String
       **************************************************************************/
      virtual t_String szGetClientFriendlyName() const = 0;

      /***************************************************************************
       ** FUNCTION:  tenRegion spi_tclStaticConfigurationBase::enGetRegion()
       ***************************************************************************/
      /*!
       * \fn      tenRegion enGetRegion()
       * \brief   Method to get the current vehicle sale region
       * \param   NONE
       * \retval  tenRegion
       **************************************************************************/
      virtual tenRegion enGetRegion() const = 0;

      /***************************************************************************
       ** FUNCTION:  tenATBoxSupportedState spi_tclStaticConfigurationBase::vGetATBoxSupportedState()
       ***************************************************************************/
      /*!
       * \fn      tenATBoxSupportedState vGetATBoxSupportedState()
       * \brief   Method to read supported AT Box features info.
       * \param   NONE
       * \retval  tenATBoxSupportedState
       **************************************************************************/
      virtual tenATBoxSupportedState vGetATBoxSupportedState() const override;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetMLNotificationSetting()
       ***************************************************************************/
      /*!
       * \fn      bGetMLNotificationSetting()
       * \brief   Method to read MLNotification On/Off setting from EOL.
       * \param   NONE
       * \retval  t_Bool: true - if MLNotification is ON, else false.
       **************************************************************************/
      virtual t_Bool bGetMLNotificationSetting();

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vReadSysConfigs()
       ***************************************************************************/
      /*!
       * \fn      t_Void vReadSysConfigs()
       * \brief   Method to read & store the system configurations from KDS.
       * \param   NONE
       * \retval  t_Void
       **************************************************************************/
      virtual t_Void vReadSysConfigs()
      {
      }
      ;

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::vSetSelectionModeStoredInDatapool
       ***************************************************************************/
      /*!
       * \fn     t_Bool vSetSelectionModeStoredInDatapool(tenDeviceSelectionMode enStoredSelectionMode)
       * \brief  Method to get the selection mode stored in datapool
       * \param  enStoredSelectionMode:[IN]Selection mode stored in datapool
       * \retval t_Void
       **************************************************************************/
      t_Void vSetSelectionModeStoredInDatapool(tenDeviceSelectionMode enStoredSelectionMode);

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::vStoreDevicePreferencefromDatapool
       ***************************************************************************/
      /*!
       * \fn     t_Bool vStoreDevicePreferencefromDatapool(tenDeviceCategory enAndroidDevicePreferencefromDatapool..)
       * \brief  Method to store device preference from datapool
       * \param  enAndroidDevicePreferencefromDatapool:[IN]Android device preference
       * \param  enAppleDevicePreferencefromDatapool:[IN]Apple device preference
       * \retval t_Void
       **************************************************************************/
      t_Void vStoreDevicePreferencefromDatapool(tenDeviceCategory enAndroidDevicePreferencefromDatapool,
               tenDeviceCategory enAppleDevicePreferencefromDatapool);

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBase::vDisplayAudioSettings()
       ***************************************************************************/
      /*!
       * \fn     t_Void vDisplayAudioSettings()
       * \brief  To print the audio settings data
       * \retval  t_Void
       **************************************************************************/
      t_Void vDisplayAudioSettings();

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBase::vDisplayVideoSettings()
       ***************************************************************************/
      /*!
       * \fn     t_Void vDisplayVideoSettings()
       * \brief  To print the video settings data
       * \retval  t_Void
       **************************************************************************/
      t_Void vDisplayVideoSettings();

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vDisplayAppSettings()
       ***************************************************************************/
      /*!
       * \fn     t_Void vDisplayAppSettings()
       * \brief  To print the settings
       * \retval t_Void
       **************************************************************************/
      t_Void vDisplayAppSettings();

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBase::vGetSPIConfigurationData()
       ***************************************************************************/
      /*!
       * \fn      vGetSPIConfigurationData()
       * \brief   Method get the KDS data
       * \param   u8Data : [out]  DataBuffer
       * \retval  t_Void
       **************************************************************************/
      virtual t_Void vGetSPIConfigurationData()
      {
      }

      /** FUNCTION:  tenEnabledInfo spi_tclStaticConfigurationBase::enGetOnCarEnableSetVal()
       ***************************************************************************/
      /*!
       * \fn     tenEnabledInfo enGetOnCarEnableSetVal()
       * \brief  To get the default OnCar Enable Setting Value
       * \retval tenEnabledInfo
       **************************************************************************/
      tenEnabledInfo enGetOnCarEnableSetVal() const;

      /***************************************************************************
       ** FUNCTION:  t_Void spi_tclStaticConfigurationBase::vSetVehicleManufacturerInformationtoKDS()
       ***************************************************************************/
      /*!
       * \fn     tenEnabledInfo enGetOnCarEnableSetVal()
       * \brief  To set VehicleManufacturerInformation value to static configreader
       * \retval t_Void
       **************************************************************************/
      virtual t_Void vSetVehicleManufacturerInformationtoKDS(t_Bool bResult,
               trVehicleManufacturerInformation rVehicleManufacturerInfoAttr);

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBase::bGetITCommanderSupport())
       ***************************************************************************/
      /*!
       * \fn      t_Bool bGetITCommanderSupport()
       * \brief   Function to Get ITCommander Support
       * \retval  NONE
       **************************************************************************/
      virtual t_Bool bGetITCommanderSupport()
      {
         return false;
      }

      /***************************************************************************
       ** FUNCTION: t_Bool spi_tclStaticConfigurationBase::bGetSysVehicleInfo())
       ***************************************************************************/
      /*!
       * \fn      t_Bool bGetSysVehicleInfo()
       * \brief   Function to Get Region, Vendor and Vehicle
       * \retval  t_Bool
       **************************************************************************/
      virtual t_Bool bGetSysVehicleInfo(t_U8 & /*u8RegionCode*/, t_U8 & /*u8VendorCode*/, t_U16 & /*u16VehicleCode*/)
      {
         return false;
      }

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetCarlifeIOSFeatureSupport
       ***************************************************************************/
      /*!
       * \fn     t_Bool bGetCarlifeIOSFeatureSupport()
       * \brief  Method to retrieve Carlife Ios enable info from policy
       * \retval t_Bool
       **************************************************************************/
      t_Bool bGetCarlifeIOSFeatureSupport() override;

      /***************************************************************************
       ** FUNCTION: t_U8 spi_tclStaticConfigurationBase::u8ReadOEMBrandFromKDS()
       ***************************************************************************/
      /*!
       * \fn      t_U8 u8ReadOEMBrandFromKDS()
       * \brief   Method to read the KDS value and get the Brand type info
       * \retval  NONE
       **************************************************************************/
      virtual t_U8 u8ReadOEMBrandFromKDS()
      {
         t_U8 u8OEMBrand = 0xff;
         return u8OEMBrand;
      }

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclStaticConfigurationBase::vGetVehicleInfoDataMySPIN()
       ***************************************************************************/
      /*!
       * \fn      t_Void vGetVehicleInfoDataMySPIN()
       * \brief   Method to get the vehicle manufacturer name and variant info
       * \param   rfrVehicleInfo: [IN]Vehicle Info Data
       * \retval  NONE
       **************************************************************************/
      virtual t_Void vGetVehicleInfoDataMySPIN(trVehicleInfo& rfrVehicleInfo)
      {
      }

      /***************************************************************************
      FUNCTION: t_U16 spi_tclStaticConfigurationBase::u16ReadVehicleTypeFromKDS())
      ***************************************************************************/
      /*!
      * \fn      t_U16 u16ReadVehicleTypeFromKDS()
      * \brief   Function to Get Vehicle Type
      * \retval  t_U16
      ***************************************************************************/
      virtual t_U16 u16ReadVehicleTypeFromKDS()
      {
         t_U16 u16AllianceVehicleType = 0xffff;
         return u16AllianceVehicleType;
      }

      /***************************************************************************
       ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bIsDeviceWhitelisted
       ***************************************************************************/
      /*!
       * \brief  check if the device is whitelisted based on VID and PID of device
       * \param  [IN] u32ProductID :Product ID
       * \param  [IN] u32VendorID : Vendor ID
       * \param  [IN] enReqDevCat : Device Category
       ***************************************************************************/
      t_Bool bIsDeviceWhitelisted(t_U32 u32ProductID, t_U32 u32VendorID, tenDeviceCategory enReqDevCat);

      /***************************************************************************
      ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bReadCPWStatus()
      ***************************************************************************/
      /*!
      * \fn      t_Bool bReadCPWStatus()
      * \brief   Method to Read the KDS value of CPW
      * \param   NONE
      * \retval  t_Bool
      **************************************************************************/
      t_Bool bReadCPWStatus() {return false;}
    /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetCarlifeGPSDataSupportflag()
    ***************************************************************************/
    /*!
    * \fn     t_Bool bGetCarlifeGPSDataSupportflag() const
    * \brief  Interface to get location data enabled setting for carlfie
    * \retval t_Bool
    **************************************************************************/
    t_Bool bGetCarlifeGPSDataSupportflag() const;

      /***************************************************************************
      ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::vGetDefaultKeys
      ***************************************************************************/
      /*!
      * \brief  vGetDefaultKeys from the project
      * \param  [OUT] rfrmapTechSuppDefkeys : Default Supported Keys
      ***************************************************************************/
      virtual t_Void vGetDefaultKeys(std::map<tenDeviceCategory,std::set<tenKeyCode>>& rfrmapTechSuppDefkeys);

      /***************************************************************************
      ** FUNCTION:  t_Bool spi_tclStaticConfigurationBase::bGetCarPlayGPSDataSupportflag()
      ***************************************************************************/
      /*!
      * \fn     t_Bool bGetCarPlayGPSDataSupportflag()
      * \brief  Interface to get GPS data enabled setting
      * \retval t_Bool
      **************************************************************************/
      virtual t_Bool bGetCarPlayGPSDataSupportflag() override
      {
         return true;
      }

      /***************************************************************************
       ****************************END OF PUBLIC *********************************
       ***************************************************************************/
   protected:
      /***************************************************************************
       ****************************END OF PROTECTED *********************************
       ***************************************************************************/

      Lock m_oLock;

      //!calibration Reader pointer
      spi_tclCalibrationReaderBase *m_poCalibrationReader;

      /***************************************************************************
       ****************************END OF PROTECTED *********************************
       ***************************************************************************/

   private:

      /***************************************************************************
       *********************************PRIVATE***********************************
       ***************************************************************************/

      /*************************************************************************
       ** FUNCTION:  virtual spi_tclStaticConfigurationBase::bXmlReadNode(xmlNode *poNode)
       *************************************************************************/
      /*!
       * \fn     virtual bool bXmlReadNode(xmlNode *poNode)
       * \brief  virtual function to read data from a xml node
       * \param  poNode : [IN] pointer to xml node
       * \retval bool : true if success, false otherwise.
       *************************************************************************/
      virtual t_Bool bXmlReadNode(xmlNodePtr poNode);

      /**************************************************************************
       ** FUNCTION:  spi_tclStaticConfigurationBase::spi_tclDynamicConfigurationBase(const...
       **************************************************************************/
      /*!
       * \fn      spi_tclStaticConfigurationBase(const spi_tclStaticConfigurationBase& oStaticConfiguration)
       * \brief   Copy Consturctor, will not be implemented.
       *          Written to avoid coverity error: new in constructor for
       *          spi_tclStaticConfigurationBase which has no Copy Consturctor.
       *          NOTE: This is a technique to disable the Copy Consturctor for this
       *          class. So if an attempt for the copying is made linker complains.
       * \param   [IN] oStaticConfiguration : Property to be set.
       **************************************************************************/
      spi_tclStaticConfigurationBase(const spi_tclStaticConfigurationBase& oStaticConfiguration);

      /**************************************************************************
       ** FUNCTION:  spi_tclStaticConfigurationBase::spi_tclStaticConfigurationBase& operator=(...
       **************************************************************************/
      /*!
       * \fn      spi_tclStaticConfigurationBase& operator=(
       *          const spi_tclStaticConfigurationBase& oStaticConfiguration)
       * \brief   Assingment Operater, will not be implemented.
       *          Written to avoid coverity error:
       *          class 'spi_tclStaticConfigurationBase' which has no assignment operator.
       *          NOTE: This is a technique to disable the assignment operator for this
       *          class. So if an attempt for the assignment is made compiler complains.
       **************************************************************************/
      spi_tclStaticConfigurationBase& operator=(const spi_tclStaticConfigurationBase& oStaticConfiguration);

      /***************************************************************************
       * ! Data members
       ***************************************************************************/

      spi_tclAppSettings* m_poAppSettings;
      spi_tclConnSettings* m_poConnSettings;
      spi_tclAudioSettings* m_poAudioSettings;
      spi_tclDataServiceSettings* m_poDataServiceSettings;
      spi_tclResourceMngrSettings* m_poResourceMngrSettings;
      spi_tclVideoSettings* m_poVideoSettings;
      spi_tclDefaultSettings* m_poDefaultSettings;
      spi_tclDiscovererSettings* m_poDiscovererSettings;

      tenDeviceSelectionMode m_enStoredSelectionMode;

      tenDeviceCategory m_enAndroidDevicePreference;
      tenDeviceCategory m_enAppleDevicePreference;

      //! Drive mode information
      tenVehicleConfiguration m_enDriveModeInfo;

      //! Light information
      tenVehicleConfiguration m_enNightModeInfo;

      t_U8 m_u8AAPParkModeRestrictionInfo;
      t_U8 m_u8AAPDriveModeRestrictionInfo;

      /***************************************************************************
       ****************************END OF PUBLIC *********************************
       ***************************************************************************/

};
#endif //SPI_TCLSTATICCONFIGURATIONBASE_
