/***********************************************************************/
/*!
* \file  spi_tclDynamicConfigurationBase.h
* \brief  Implementation of the Class spi_tclDynamicConfigurationBase
*************************************************************************
\verbatim


PROJECT:        Gen3
SW-COMPONENT:   Smart Phone Integration
DESCRIPTION:    Common  Base Class  SPI Dynamic Config Reader for all Projects
AUTHOR:         kmo6kor
COPYRIGHT:      &copy; 2017 Robert Bosch Car Multimedia GmbH
HISTORY:
Date        | Author                | Modification
             kmo6kor			    Initial Version                                               
\endverbatim
*************************************************************************/

#include "spi_tclDynamicConfigurationReaderBase.h"
#include "Lock.h"

/******************************************************************************
| typedefs (scope: module-local)
|----------------------------------------------------------------------------*/
/******************************************************************************
| defines and macros (scope: global)
|----------------------------------------------------------------------------*/
#ifndef SPI_TCLDYNAMICCONFIGURATIONBASE_
#define SPI_TCLDYNAMICCONFIGURATIONBASE_

class spi_tclDynamicConfigurationBase
{

public:
   /***************************************************************************
   ****************************PUBLIC *********************************
   ***************************************************************************/

   /***************************************************************************
    ** FUNCTION: spi_tclDynamicConfigurationBase::spi_tclDynamicConfigurationBase()
    ***************************************************************************/
   /*!
    * \fn      spi_tclDynamicConfigurationBase()
    * \brief   Constructor
    * \param   [IN] poDynamicConfigurationReader - Pointer to Datapool
    **************************************************************************/
   spi_tclDynamicConfigurationBase(spi_tclDynamicConfigurationReaderBase* poDynamicConfigurationReader);

   /***************************************************************************
    ** FUNCTION: spi_tclDynamicConfigurationBase::~spi_tclDynamicConfigurationBase()
    ***************************************************************************/
   /*!
    * \fn      ~spi_tclDynamicConfigurationBase()
    * \brief   Destructor
    * \param   None
    **************************************************************************/
   virtual ~spi_tclDynamicConfigurationBase();

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclDynamicConfigurationBase::vSetMLNotificationOnOff()
   ***************************************************************************/
   /*!
   * \fn     t_Void vSetMLNotificationOnOff(t_Bool bSetNotificationsOn)
   * \brief  To Set the Notifications to On/Off
   * \param  bSetNotificationsOn : [IN] True-Set Notifications to ON
   *                                    False - Set Notifications to OFF
   * \retval t_Void
   **************************************************************************/
   t_Void vSetMLNotificationOnOff(t_Bool bSetNotificationsOn);

   /***************************************************************************
   ** FUNCTION: t_Bool spi_tclDynamicConfigurationBase::bGetMLNotificationEnabledInfo()
   ***************************************************************************/
   /*!
   * \fn     t_Bool bGetMLNotificationEnabledInfo()
   * \brief  Provides information on whether ML Notifications are enabled or
   *         disabled.
   * \retval t_Bool  TRUE- Enabled FALSE-Disabled
   **************************************************************************/
   t_Bool bGetMLNotificationEnabledInfo() const;

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::vSetDeviceSelectionMode()
    ***************************************************************************/
   /*!
    * \fn      t_Void vSetDeviceSelectionMode(
    * \brief   Method to set the device selection mode to automatic/manual. Changes
    *          will take into effect on successive connection
    * \param   enSelectionMode : Device selection mode @see tenDeviceSelectionMode
    * \retval  t_Void
    **************************************************************************/
   t_Void vSetDeviceSelectionMode(tenDeviceSelectionMode enSelectionMode, tenDeviceCategory enTechnologyType);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::vSetMLNotiSettingVal()
   ***************************************************************************/
   /*!
   * \fn      vSetMLNotiSettingVal(t_Bool bMLNotiSettingVal)
   * \brief   Called to set the default value to notification setting.
   * \param   [IN] bMLNotiSettingVal: default value for notification setting
   * \retval  None
   **************************************************************************/
   t_Void vSetMLNotiSettingVal(t_Bool bMLNotiSettingVal);

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclDynamicConfigurationBase::bSetMLLinkEnableSetVal()
   ***************************************************************************/
   /*!
   * \fn      bSetMLLinkEnableSetVal(tenEnabledInfo enMLLinkEnableSetVal)
   * \brief   Called to set the default value to enable ML usage.
   * \param   [IN] enMLLinkEnableSetVal: default value for enabling ML link setting
   * \retval  None
   **************************************************************************/
   t_Bool bSetMLLinkEnableSetVal(tenEnabledInfo enMLLinkEnableSetVal);

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclDynamicConfigurationBase::bSetDipoEnableSetVal()
   ***************************************************************************/
   /*!
   * \fn      bSetDipoEnableSetVal(tenEnabledInfo enDipoEnableSetVal)
   * \brief   Called to set the default value to enable Dipo usage.
   * \param   [IN] enDipoEnableSetVal: default value for enabling Dipo setting
   * \retval  t_Bool
   **************************************************************************/
   t_Bool bSetDipoEnableSetVal(tenEnabledInfo enDipoEnableSetVal);

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclDynamicConfigurationBase::bSetAAPEnableSetVal()
   ***************************************************************************/
   /*!
   * \fn      bSetAAPEnableSetVal(tenEnabledInfo enAAPEnableSetVal)
   * \brief   Called to set the default value to enable AAP usage.
   * \param   [IN] enAAPEnableSetVal: default value for enabling AAP setting
   * \retval  t_Bool
   **************************************************************************/
   t_Bool bSetAAPEnableSetVal(tenEnabledInfo enAAPEnableSetVal);

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclDynamicConfigurationBase::bSetCarlifeEnableSetVal()
   ***************************************************************************/
   /*!
   * \fn      bSetCarlifeEnableSetVal(tenEnabledInfo enCarlifeEnableSetVal)
   * \brief   Called to set the default value to enable AAP usage.
   * \param   [IN] enCarlifeEnableSetVal: default value for enabling Carlife setting
   * \retval  t_Bool
   **************************************************************************/
   t_Bool bSetCarlifeEnableSetVal(tenEnabledInfo enCarlifeEnableSetVal);

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclDynamicConfigurationBase::bSetMySPINEnableSetVal()
   ***************************************************************************/
   /*!
   * \fn      bSetMySPINEnableSetVal(tenEnabledInfo enmySPINLinkEnable)
   * \brief   Called to set the default value to enable AAP usage.
   * \param   [IN] enmySPINLinkEnable: default value for enabling Myspin setting
   * \retval  t_Bool
   **************************************************************************/
   t_Bool bSetMySPINEnableSetVal(tenEnabledInfo enmySPINLinkEnablel);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::vSetSteeringWheelPos()
   ***************************************************************************/
   /*!
   * \fn      vSetSteeringWheelPos(tenDriveSideInfo enSteeringWheelPos)
   * \brief   Called to set the default value to steering wheel position.
   * \param   [IN] enSteeringWheelPos: default value for steering wheel position
   * \retval  None
   **************************************************************************/
   t_Void vSetSteeringWheelPos(tenDriveSideInfo enSteeringWheelPos);


   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::vSetSelectMode()
   ***************************************************************************/
   /*!
   * \fn      vSetSelectMode(tenDeviceSelectionMode enSelectMode)
   * \brief   Called to set the default value to selection mode.
   * \param   [IN] enSelectMode: default value for selection mode
   * \retval  None
   **************************************************************************/
   t_Void vSetSelectMode(tenDeviceSelectionMode enSelectMode);


   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclDynamicConfigurationBase::bSetAndroidDeviceTechnologyPreference()
   ***************************************************************************/
   /*!
   * \fn      bSetAndroidDeviceTechnologyPreference(tenDeviceCategory enAndroidDeviceTechPrefVal)
   * \brief   Called to set the default value to technology preference.
   * \param   [IN] enAndroidDeviceTechPrefVal: default value -technology preference for Android Device
   * \retval  t_Bool
   **************************************************************************/
   t_Bool bSetAndroidDeviceTechnologyPreference(tenDeviceCategory enAndroidDeviceTechPrefVal);

   /***************************************************************************
   ** FUNCTION:  t_Bool spi_tclDynamicConfigurationBase::bSetAppleDeviceTechnologyPreference()
   ***************************************************************************/
   /*!
   * \fn      bSetAppleDeviceTechnologyPreference(tenDeviceCategory enAppleDeviceTechPrefVal)
   * \brief   Called to set the default value to technology preference.
   * \param   [IN] enAppleDeviceTechPrefVal: default value -technology preference for Apple Device
   * \retval  t_Bool
   **************************************************************************/
   t_Bool bSetAppleDeviceTechnologyPreference(tenDeviceCategory enAppleDeviceTechPrefVal);

   /***************************************************************************
   ** FUNCTION:  t_U8 spi_tclDynamicConfigurationBase::u8ReadVirginStartSetting()
   ***************************************************************************/
   /*!
   * \fn      u8ReadVirginStartSetting( )
   * \brief   Called to read VirginStartSetting.
   * \param   None
   * \retval  t_U8
   **************************************************************************/
   t_U8 u8ReadVirginStartSetting();

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::vWriteVirginStartSetting()
   ***************************************************************************/
   /*!
   * \fn      vWriteVirginStartSetting( )
   * \brief   Called to write VirginStartSetting to datapool.
   * \param   None
   * \retval  t_Void
   **************************************************************************/
   t_Void vWriteVirginStartSetting(t_U8 u8DefaultVgStrt);

   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclDynamicConfigurationBase::bWriteAudioLastMode()
    ***************************************************************************/
   /*!
    * \fn      bWriteAudioLastMode()
    * \brief   Method to write the audio last mode value into Datapool
    * \param   bAudioLastMode: [IN] Display last mode value to be written into datapool
    * \retval  t_Bool
    **************************************************************************/
   t_Bool bWriteAudioLastMode(t_Bool bAudioLastMode);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::bReadSelectionMode
    ***************************************************************************/
   /*!
    * \fn      bReadSelectionMode()
   * \brief   Function to read the SelectionMode
   * \param   [OUT] enSelectionMode : Device selection mode
    * \retval  t_Void
    **************************************************************************/
   t_Void bReadSelectionMode(tenDeviceSelectionMode &enSelectionMode);

   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclDynamicConfigurationBase::bWriteDisplayLastMode()
    ***************************************************************************/
   /*!
    * \fn      bWriteDisplayLastMode()
    * \brief   Method to write the display last mode value into Datapool
    * \param   bDisplayLastMode: [IN] Display last mode value to be written into datapool
    * \retval  t_Bool
    **************************************************************************/
   t_Bool bWriteDisplayLastMode(t_Bool bDisplayLastMode);

   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclDynamicConfigurationBase::bIsDisplayLastModeProjection()
    ***************************************************************************/
   /*!
    * \fn      bIsDisplayLastModeProjection()
    * \brief   Method to read the display last mode from Datapool
    * \retval  t_Bool - True if display last mode is Projection
    **************************************************************************/
   t_Bool bIsDisplayLastModeProjection();

   /***************************************************************************
   ** FUNCTION: t_U8 spi_tclDynamicConfigurationBase::u8GetDipoRestrictionInfo(t_Bool bDriveRestrictions)
   ***************************************************************************/
  /*!
   * \fn      t_U8 u8GetDipoRestrictionInfo(t_Bool bDriveRestrictions)
   * \brief   Method to get restrictions enabled in park/drive mode  for Carplay
   * \param   bDriveRestrictions: [IN] drive restrictions
   * \retval  t_U8
   **************************************************************************/
   virtual t_U8 u8GetDipoRestrictionInfo(t_Bool bDriveRestrictions);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::vSetDipoFeatureRestrictions(const t_U8 cou8ParkModeRestrictionInfo,const t_U8 cou8DriveModeRestrictionInfo)
    ***************************************************************************/
   /*!
    * \fn      t_Void vSetDipoFeatureRestrictions(const t_U8 cou8ParkModeRestrictionInfo,const t_U8 cou8DriveModeRestrictionInfo)
    * \brief   Method to set Vehicle Park/Drive Mode Restriction
    * \param   NONE
    * \retval  t_Void
    **************************************************************************/
   virtual t_Void vSetDipoFeatureRestrictions(const t_U8 cou8ParkModeRestrictionInfo,const t_U8 cou8DriveModeRestrictionInfo);

   /***************************************************************************
    ** FUNCTION:  t_String spi_tclDynamicConfigurationBase::szGetVehicleId()
    ***************************************************************************/
   /*!
    * \fn      szGetVehicleId()
    * \brief   Method to get VehicleId information
    * \retval  t_String
    **************************************************************************/
   virtual t_String szGetVehicleId();

   /***************************************************************************
    ** FUNCTION: t_U16 spi_tclDynamicConfigurationBase::u16GetGeneralRestrictions()
    ***************************************************************************/
   /*!
    * \fn      u16GetGeneralRestrictions()
    * \brief  Method to retrieve setting for general restrictions
    * \param [OUT] : none
    * \retval t_U16
    **************************************************************************/
   virtual t_U16 u16GetGeneralRestrictions(tenDeviceCategory enDeviceCategory);

   /***************************************************************************
   ** FUNCTION:  tenDriveSideInfo spi_tclDynamicConfigurationBase::enGetDriveSideInfo(...)
   ***************************************************************************/
   /*!
   * \fn      enGetDriveSideInfo()
   * \brief   Method to read Drive side info.
   * \param   NONE
   * \retval  Drive Side enum
   **************************************************************************/
   virtual tenDriveSideInfo enGetDriveSideInfo();

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::vSetDriveSideInfo()
   ***************************************************************************/
   /*!
   * \fn      t_Void vSetDriveSideInfo()
   * \brief   Method to set the night mode info
   * \param   enVehicleConfig : [IN] Drive side info.
   * \retval  t_Void
   **************************************************************************/
   virtual t_Void vSetDriveSideInfo(const tenVehicleConfiguration enVehicleConfig);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::vSetDriveSideInfoToDatapool()
   ***************************************************************************/
   /*!
   * \fn      t_Void vSetDriveSideInfoToDatapool()
   * \brief   Method to set the night mode info
   * \param   enVehicleConfig : [IN] Drive side info.
   * \retval  t_Void
   **************************************************************************/
   virtual t_Void vSetDriveSideInfoToDatapool(tenDriveSideInfo enDriveSideInfo);

   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclConfigurationManager::bIsAudioLastModeProjection()
    ***************************************************************************/
   /*!
    * \fn      bIsAudioLastModeProjection()
    * \brief   Method to read the audio last mode from Datapool
    * \retval  t_Bool - True if audio last mode is Projection
    **************************************************************************/
   t_Bool bIsAudioLastModeProjection();

   /***************************************************************************
    ** FUNCTION:  t_String spi_tclDynamicConfigurationBase::vGetSPIVersion()
    ***************************************************************************/
   /*!
    * \fn      szGetSPIVersion()
    * \brief   Method to get SPI Version information in Datapool
    * \retval  t_String
    **************************************************************************/
   t_String szGetSPIVersion() const;

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::vSetVehicleManufacturerInformation(...)
   ***************************************************************************/
   /*!
   * \fn      t_Void vSetVehicleManufacturerInformation(const trVehicleManufacturerInformation& corfrVehicleManufacturerInfoAttr)
   * \brief   Method to set the  Vehicle Manufacturer Information
   * \param   [IN] corfrVehicleManufacturerInfoAttr:Vehicle Manufacturer Information
   * \retval  None
   **************************************************************************/
   virtual t_Void vSetVehicleManufacturerInformation(const trVehicleManufacturerInformation& corfrVehicleManufacturerInfoAttr) = 0;

   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclDynamicConfiguration::bReadAndroidDeviceTechnologyPreference()
    ***************************************************************************/
   /*!
    * \fn      bReadAndroidDeviceTechnologyPreference()
    * \brief   Function to read Android Device the TechnologyPreference
    * \retval  t_Bool
    ***************************************************************************/
   t_Bool bReadAndroidDeviceTechnologyPreference(tenDeviceCategory &enTechnologyPreference);


   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclDynamicConfiguration::bReadAppleDeviceTechnologyPreference()
    ***************************************************************************/
   /*!
    * \fn      bReadAppleDeviceTechnologyPreference()
    * \brief   Function to read Apple Device the TechnologyPreference
    * \retval  t_Bool
    ***************************************************************************/
   t_Bool bReadAppleDeviceTechnologyPreference(tenDeviceCategory &enTechnologyPreference);

   /***************************************************************************
   ** FUNCTION:  tenEnabledInfo spi_tclDynamicConfigurationBase::enReadDipoEnableSettingVal()
   ***************************************************************************/
   /*!
   * \fn      enReadDipoEnableSettingVal
   * \brief   Provides DiPO On/Off setting value stored in Datapool.
   * \param   None
   * \retval  tenEnabledInfo:
   *               e8USAGE_DISABLED - to set value as OFF
   *               e8USAGE_ENABLED - to set value as ON
   ***************************************************************************/
   tenEnabledInfo enReadDipoEnableSettingVal() const;

   /***************************************************************************
   ** FUNCTION:  tenEnabledInfo spi_tclDynamicConfigurationBase::enReadAAPEnableSettingVal()
   ***************************************************************************/
   /*!
   * \fn      enReadAAPEnableSettingVal
   * \brief   Provides AAP On/Off setting value stored in Datapool.
   * \param   None
   * \retval  tenEnabledInfo:
   *               e8USAGE_DISABLED - to set value as OFF
   *               e8USAGE_ENABLED - to set value as ON
   ***************************************************************************/
   tenEnabledInfo enReadAAPEnableSettingVal() const;

   /***************************************************************************
   ** FUNCTION:  tenEnabledInfo spi_tclDynamicConfigurationBase::enReadmySPINEnableSettingVal()
   ***************************************************************************/
   /*!
   * \fn      enReadmySPINEnableSettingVal
   * \brief   Provides mySPIN On/Off setting value stored in Datapool.
   * \param   None
   * \retval  tenEnabledInfo:
   *               e8USAGE_DISABLED - to set value as OFF
   *               e8USAGE_ENABLED - to set value as ON
   ***************************************************************************/
   tenEnabledInfo enReadmySPINEnableSettingVal() const;

   /***************************************************************************
   ** FUNCTION:  tenEnabledInfo spi_tclDynamicConfigurationBase::enReadCarlifeEnableSettingVal()
   ***************************************************************************/
   /*!
   * \fn      enReadCarlifeEnableSettingVal
   * \brief   Provides Carlife On/Off setting value stored in Datapool.
   * \param   None
   * \retval  tenEnabledInfo:
   *               e8USAGE_DISABLED - to set value as OFF
   *               e8USAGE_ENABLED - to set value as ON
   ***************************************************************************/
   tenEnabledInfo enReadCarlifeEnableSettingVal() const;

   /***************************************************************************
   ** FUNCTION:  tenEnabledInfo spi_tclDynamicConfigurationBase::enReadMLEnableSettingVal()
   ***************************************************************************/
   /*!
   * \fn      enReadMLEnableSettingVal
   * \brief   Provides MirrorLink On/Off setting value stored in Datapool.
   * \param   None
   * \retval  tenEnabledInfo:
   *               e8USAGE_DISABLED - to set value as OFF
   *               e8USAGE_ENABLED - to set value as ON
   ***************************************************************************/
   tenEnabledInfo enReadMLEnableSettingVal() const;

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::vClearVehicleId()
    ***************************************************************************/
   /*!
    * \fn      vClearVehicleId()
    * \brief   Method to set VehicleId information
    * \retval  t_Void
    **************************************************************************/
   virtual t_Void vClearVehicleId();

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::vSetScreenConfigs()
   ***************************************************************************/
   /*!
   * \fn      t_Void vSetScreenConfigs(const trDisplayAttributes& corfrDispLayerAttr)
   * \brief   Method to set screen attributes
   * \param   corfrDispLayerAttr : [IN] Display attributes
   * \retval  t_Void
   **************************************************************************/
   virtual t_Void vSetScreenConfigs(const trDisplayAttributes& corfrDispLayerAttr);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::vSetGeneralRestrictions(const t_U8 cou8GeneralRestrictionInfo)
   ***************************************************************************/
   /*!
   * \fn      t_Void vSetGeneralRestrictions(const t_U8 cou8GeneralRestrictionInfo)
   * \brief   Method to set General Restriction
   * \param   NONE
   * \retval  t_Void
   **************************************************************************/
   virtual t_Void vSetGeneralRestrictions(tenDeviceCategory enDeviceCategory,const t_U16 u16GeneralRestrictionInfo);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclDynamicConfigurationBase::vPrintVideoConfig(...)
   ***************************************************************************/
   /*!
   * \fn      t_Void vPrintVideoConfig(trVideoConfigData& rfrVideoConfig)
   * \brief   Method to print video config attributes.
   * \param   NONE
   * \retval  t_Void
   **************************************************************************/
   virtual t_Void vPrintVideoConfig(const trVideoConfigData& corfrVideoConfig);

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclDynamicConfigurationBase::vGetVideoConfigData()
   ***************************************************************************/
   /*!
   * \fn      t_Void vGetVideoConfigData(tenDeviceCategory enDevCat,
   *                 trVideoConfigData& rfrVideoConfig)
   * \brief   Method to read the EOL value and get the video configuration data
              from the look up table
   * \param   enDevCat      : [IN] Provide configurations fro the requested technology
   * \param   rfrVideoConfig: [IN] Video Config Data
   * \retval  NONE
   **************************************************************************/
   t_Void vGetVideoConfigData(tenDeviceCategory enDevCat,
            tvecVideoConfigList& rfvecVideoConfigList);

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclDynamicConfigurationBase::vGetDisplayAttributeData()
   ***************************************************************************/
   /*!
   * \fn      t_Void vGetDisplayAttributeData(trVideoConfigData& rVideoConfigData)
   * \brief   Method to write the display attributes
   * \param   rVideoConfigData      : [IN] The structure to write the
   *                                  display attributes data
   * \retval  NONE
   **************************************************************************/
   t_Void vGetDisplayAttributeData(trVideoConfigData& rVideoConfigData);

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclDynamicConfigurationBase::vGetDisplayLayerAttributeData()
   ***************************************************************************/
   /*!
   * \fn      t_Void vGetDisplayLayerAttributeData(trVideoConfigData& rVideoConfigData)
   * \brief   Method to write the display layer attributes
   * \param   rVideoConfigData      : [IN] The structure to write the
   *                                  display layer attributes data
   * \param   corfrDispLayerAttr    : [IN]  structure consisting of the
                                       display layer attributes.
   * \retval  NONE
   **************************************************************************/
   t_Void vGetDisplayLayerAttributeData(trVideoConfigData& rVideoConfigData,const
                                        trDisplayLayerAttributes& corfrDispLayerAttr);

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclDynamicConfigurationBase::vGetPixelAspectRatio()
   ***************************************************************************/
   /*!
   * \fn      t_Void vGetPixelAspectRatio()
   * \brief   Method to read the pixel aspect ratio
   * \retval  t_Void
   **************************************************************************/
   virtual t_Void vGetPixelAspectRatio(t_String& rfszPixelAspectRatio)
  {
	   SPI_INTENTIONALLY_UNUSED(rfszPixelAspectRatio);
  }

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclDynamicConfigurationBase::vGetDPI()
   ***************************************************************************/
   /*!
   * \fn      t_Void vGetDPI()
   * \brief   Method to read the density pixel aspect ratio
   * \retval  t_Void
   **************************************************************************/
   virtual t_Void vGetDPI(const t_U32& rfu32ScreenHeight,t_U16& rfu16DPI);

   /***************************************************************************
   ** FUNCTION: t_Void spi_tclDynamicConfigurationBase::vGetPrimaryDisplayConfiguration()
   ***************************************************************************/
   /*!
   * \fn      t_String vGetPrimaryDisplayConfiguration(t_String &rfrSoftwareBuild)
   * \brief   Method to get primary display configuration from multiple display configuration
   * \param   rfvecVideoConfigList  : Vector consisting multiple display configurations.
   *          rVideoConfigData [out]: To conssit the primary display configuration.
   * \retval  t_Void
   **************************************************************************/
   t_Void vGetPrimaryDisplayConfiguration(const tvecVideoConfigList& rfvecVideoConfigList,
                                          trVideoConfigData& rVideoConfigData);


   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclDynamicConfigurationBase::bWriteOnCarEnableSetting
    ***************************************************************************/
   /*!
    * \fn      bWriteOnCarEnableSetting(tenEnabledInfo enOnCarLinkEnable)
    * \brief   Sets the OnCar On/Off setting value in datapool.
    *          (True - if enabled, else False)
    * \param   [IN] enOnCarLinkEnable :
    *               e8USAGE_DISABLED - to set value as OFF
    *               e8USAGE_ENABLED - to set value as ON
    * \retval  t_Bool
    ***************************************************************************/
   t_Bool bWriteOnCarEnableSetting(tenEnabledInfo enOnCarLinkEnable);

   /***************************************************************************
    ** FUNCTION:  tenEnabledInfo spi_tclDynamicConfigurationBase::enReadOnCarEnableSetting()
    ***************************************************************************/
   /*!
    * \fn      enReadOnCarEnableSetting
    * \brief   Provides OnCar On/Off setting value stored in Datapool.
    * \param   None
    * \retval  tenEnabledInfo:
    *               e8USAGE_DISABLED - to set value as OFF
    *               e8USAGE_ENABLED - to set value as ON
    ***************************************************************************/
   tenEnabledInfo enReadOnCarEnableSetting() const ;

   /***************************************************************************
    ** FUNCTION:  t_Bool spi_tclDynamicConfigurationBase::bReadVehicleManufacturerInfofromDatapool()
    ***************************************************************************/
   /*!
    * \fn      bReadVehicleManufacturerInfofromDatapool()
    * \brief   Function to Read Vehicle Manufacturer information
    * \retval  t_Bool
    ***************************************************************************/
   virtual t_Bool bReadVehicleManufacturerInfofromDatapool(trVehicleManufacturerInformation& rfrVehicleManufacturerInformation);

   /***************************************************************************
    ** FUNCTION:  t_String spi_tclDynamicConfigurationBase::szGetSoftwareVersion()
    ***************************************************************************/
   /*!
    * \fn      t_String szGetSoftwareVersion()
    * \brief   Method to get the software version
    * \param   NONE
    * \retval  t_String
    **************************************************************************/
   virtual t_String szGetSoftwareVersion() const = 0;

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclConfigurationManager::vGetVehicleBTAddress()
   ***************************************************************************/
   /*!
   * \fn     vGetVehicleBTAddress()
   * \brief  Function to get the vehicle BT address.
   * \param  szVehicleBTAddress:[OUT] Vehicle BT address
   * \retval  None
   **************************************************************************/
   t_Void vGetVehicleBTAddress(t_String &szVehicleBTAddress);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclConfigurationManager::vOnVehicleBTAddress()
   ***************************************************************************/
   /*!
   * \fn     vOnVehicleBTAddress()
   * \brief  Function to set the vehicle BT address.
   * \param  szVehicleBTAddress:[OUT] Vehicle BT address
   * \retval  None
   **************************************************************************/
   t_Void vOnVehicleBTAddress(t_String &szVehicleBTAddress);


   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclConfigurationManager::vRegisterKeyCodes()
   ***************************************************************************/
   /*!
   * \fn     vRegisterKeyCodes()
   * \brief  Function to Register keyinfo to Data Pool
   * \param  corfrmapTechSuppHardkeys:[IN] Hard Keys Supported for all technologies
   * \param  corfrmapTechSuppSoftkeys:[IN] Soft Keys Supported for all technologies
   * \retval  None
   **************************************************************************/
   t_Void vRegisterKeyCodes(const std::map<tenDeviceCategory,std::set<tenKeyCode>>& corfrmapTechSuppHardkeys,
                            const std::map<tenDeviceCategory,std::set<tenKeyCode>>& corfrmapTechSuppSoftkeys);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclConfigurationManager::bGetKeyInfo()
   ***************************************************************************/
   /*!
   * \fn     bGetKeyInfo()
   * \brief  Function to Get the KeyInfo.
   * \param  rfrsetKeyCodes:[OUT] Keys supported
   * \param  enCategory:[IN] For Which Technology
   * \retval  t_Bool
   **************************************************************************/
   t_Bool bGetKeyInfo(std::set<tenKeyCode>& rfrsetKeyCodes,tenDeviceCategory enCategory);

   /***************************************************************************
   ** FUNCTION:  t_Void spi_tclConfigurationManager::vGetSoftKeyInfo()
   ***************************************************************************/
   /*!
   * \fn     vGetSoftKeyInfo()
   *\brief  Function to Get Soft KeyInfo.
   * \param  rfrsetKeyCodes:[OUT] Keys supported
   * \param  enCategory:[IN] For Which Technology
   * \retval  None
   **************************************************************************/
   t_Bool bGetSoftKeyInfo(std::set<tenKeyCode>& rfrsetKeyCodes,tenDeviceCategory enCategory);

   /***************************************************************************
   ****************************END OF PUBLIC *********************************
   ***************************************************************************/
   
protected:
   /***************************************************************************
    *********************************PROTECTED**********************************
    ***************************************************************************/

   //! Lock object to protect usage of Carlife Lib
   Lock  m_oLock;

   //! To store Video configurations of all technologies
   trDisplayAttributes m_rDisplayAttributes;

   //! pointer object to DynamicConfigurationReader base class
   spi_tclDynamicConfigurationReaderBase* m_poDynamicConfigurationReader;


   /***************************************************************************
    ***************************END OF PROTECTED********************************
    **************************************************************************/

private:

   /**************************************************************************
   ** FUNCTION:  spi_tclDynamicConfigurationBase::spi_tclDynamicConfigurationBase(const...
   **************************************************************************/
   /*!
   * \fn      spi_tclDynamicConfigurationBase(const spi_tclDynamicConfigurationBase& oDynamicConfigurationBase)
   * \brief   Copy Consturctor, will not be implemented.
   *          Written to avoid coverity error: new in constructor for
   *          spi_tclDynamicConfigurationBase which has no Copy Consturctor.
   *          NOTE: This is a technique to disable the Copy Consturctor for this
   *          class. So if an attempt for the copying is made linker complains.
   * \param   [IN] oDynamicConfigurationBase : Property to be set.
   **************************************************************************/
   spi_tclDynamicConfigurationBase(const spi_tclDynamicConfigurationBase& oDynamicConfigurationBase);

   /**************************************************************************
   ** FUNCTION:  spi_tclDynamicConfigurationBase::spi_tclDynamicConfigurationBase& operator=(...
   **************************************************************************/
   /*!
   * \fn      spi_tclDynamicConfigurationBase& operator=(
   *          const spi_tclDynamicConfigurationBase& oDynamicConfigurationBase)
   * \brief   Assingment Operater, will not be implemented.
   *          Written to avoid coverity error:
   *          class 'spi_tclDynamicConfigurationBase' which has no assignment operator.
   *          NOTE: This is a technique to disable the assignment operator for this
   *          class. So if an attempt for the assignment is made compiler complains.
   **************************************************************************/
   spi_tclDynamicConfigurationBase& operator=(const spi_tclDynamicConfigurationBase& oDynamicConfigurationBase);

   /***************************************************************************
    ** FUNCTION:  t_Void spi_tclDiPOControlAdapterImpl::vfindVideoDevPath()
    ***************************************************************************/
   /*!
   * \fn      spi_tclDiPOControlAdapterImpl::vfindVideoDevPath()
   * \brief   find the device path used for displaying technology projection screen
   **************************************************************************/
   t_Void vfindVideoDevPath(t_String& rfszVideoDevice);

   /***************************************************************************
   * ! Data members
   ***************************************************************************/

   t_U8 m_u8DipoParkModeRestrictionInfo;

   /***************************************************************************
   *********************************PRIVATE************************************
   ***************************************************************************/


   /***************************************************************************
   ****************************END OF PRIVATE *********************************
   ***************************************************************************/
};
#endif //SPI_TCLDYNAMICCONFIGURATIONBASE_
