/***********************************************************************/
/*!
 * \file  Datapool.h
 * \brief Wrapper class for Datapool usage
 *************************************************************************
 \verbatim

 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Wrapper class for Datapool usage
 AUTHOR:         Ramya Murthy
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date        | Author                | Modification
 13.05.2014  | Ramya Murthy          | Initial Version
 11.09.2014  | Shihabudheen P M      | Modified for BT MAC address writing and reading
 25.06.2015  | Shiva kaumr G         | Dynamic display configuration
 14.03.2016  | Chaitra Srinivasa     | Default settings
 18.07.2017  | Noopur R K            | Added bReadDiPODriveRestrictionInfo,bWriteGeneralRestrictionInfo,
 bReadAAPGeneralRestrictionInfo, bWriteAAPGeneralRestrictionInfo

 \endverbatim
 *************************************************************************/

#ifndef _DATAPOOL_H_
#define _DATAPOOL_H_

/******************************************************************************
 | includes:
 | 1)system- and project- includes
 | 2)needed interfaces from external components
 | 3)internal and external interfaces from this component
 |----------------------------------------------------------------------------*/
#include <set>
#include "BaseTypes.h"
#include "SPITypes.h"
#include "spi_tclDynamicConfigurationReaderBase.h"

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | defines and macros (scope: global)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | variable definition (scope: module-local)
 |----------------------------------------------------------------------------*/

/****************************************************************************/
/*!
 * \class Datapool
 * \brief
 * It provides methods to store & retrieve persistent data in SPI using Datapool.
 ****************************************************************************/
class Datapool: public spi_tclDynamicConfigurationReaderBase
{
   public:

      /***************************************************************************
       *********************************PUBLIC*************************************
       ***************************************************************************/

      /***************************************************************************
       ** FUNCTION:  Datapool::Datapool()
       ***************************************************************************/
      /*!
       * \fn      Datapool()
       * \brief   Default Constructor
       * \sa      ~Datapool()
       ***************************************************************************/
      Datapool();

      /***************************************************************************
       ** FUNCTION: Datapool::~Datapool()
       ***************************************************************************/
      /*!
       * \fn      ~Datapool()
       * \brief   Destructor
       * \sa      Datapool()
       ***************************************************************************/
      virtual ~Datapool();

      /***************************************************************************
       ** FUNCTION:  tenEnabledInfo Datapool::bReadMLEnableSetting()
       ***************************************************************************/
      /*!
       * \fn      bReadMLEnableSetting
       * \brief   Provides MirrorLink On/Off setting value stored in Datapool.
       * \param   None
       * \retval  tenEnabledInfo:
       *               e8USAGE_DISABLED - to set value as OFF
       *               e8USAGE_ENABLED - to set value as ON
       ***************************************************************************/
      tenEnabledInfo bReadMLEnableSetting() const override;

      /***************************************************************************
       ** FUNCTION:  tenEnabledInfo Datapool::bReadDipoEnableSetting()
       ***************************************************************************/
      /*!
       * \fn      bReadDipoEnableSetting
       * \brief   Provides DiPO On/Off setting value stored in Datapool.
       * \param   None
       * \retval  tenEnabledInfo:
       *               e8USAGE_DISABLED - to set value as OFF
       *               e8USAGE_ENABLED - to set value as ON
       ***************************************************************************/
      tenEnabledInfo bReadDipoEnableSetting() const override;

      /***************************************************************************
       ** FUNCTION:  tenEnabledInfo Datapool::bReadAAPEnableSetting()
       ***************************************************************************/
      /*!
       * \fn      bReadAAPEnableSetting
       * \brief   Provides AAP On/Off setting value stored in Datapool.
       * \param   None
       * \retval  tenEnabledInfo:
       *               e8USAGE_DISABLED - to set value as OFF
       *               e8USAGE_ENABLED - to set value as ON
       ***************************************************************************/
      tenEnabledInfo bReadAAPEnableSetting() const override;

      /***************************************************************************
       ** FUNCTION:  tenEnabledInfo Datapool::enReadOnCarEnableSetting()
       ***************************************************************************/
      /*!
       * \fn      enReadOnCarEnableSetting
       * \brief   Provides OnCar On/Off setting value stored in Datapool.
       * \param   None
       * \retval  tenEnabledInfo:
       *               e8USAGE_DISABLED - to set value as OFF
       *               e8USAGE_ENABLED - to set value as ON
       ***************************************************************************/
      tenEnabledInfo enReadOnCarEnableSetting() const override;

      /***************************************************************************
       ** FUNCTION:  tenEnabledInfo Datapool::bReadmySPINEnableSetting()
       ***************************************************************************/
      /*!
       * \fn      bReadmySPINEnableSetting
       * \brief   Provides mySPIN On/Off setting value stored in Datapool.
       * \param   None
       * \retval  tenEnabledInfo:
       *               e8USAGE_DISABLED - to set value as OFF
       *               e8USAGE_ENABLED - to set value as ON
       ***************************************************************************/
      tenEnabledInfo bReadmySPINEnableSetting() const override;

      /***************************************************************************
       ** FUNCTION:  tenEnabledInfo Datapool::bReadCarlifeEnableSetting()
       ***************************************************************************/
      /*!
       * \fn      bReadCarlifeEnableSetting
       * \brief   Provides Carlife On/Off setting value stored in Datapool.
       * \param   None
       * \retval  tenEnabledInfo:
       *               e8USAGE_DISABLED - to set value as OFF
       *               e8USAGE_ENABLED - to set value as ON
       ***************************************************************************/
      tenEnabledInfo bReadCarlifeEnableSetting() const override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bReadMLNotificationSetting()
       ***************************************************************************/
      /*!
       * \fn      bReadMLNotificationSetting
       * \brief   Provides MirrorLink Notification On/Off setting value
       *          stored in Datapool.
       * \param   None
       * \retval  t_Bool: true if enabled
       ***************************************************************************/
      t_Bool bReadMLNotificationSetting() const override;

      /***************************************************************************
       ** FUNCTION:  t_U8 Datapool::u8ReadVirginStartSetting()
       ***************************************************************************/
      /*!
       * \fn      u8ReadVirginStartSetting
       * \brief   Provides Virgin start setting values
       *          stored in Datapool.
       * \param   None
       * \retval  t_U8
       ***************************************************************************/
      t_U8 u8ReadVirginStartSetting() const override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bWriteMLEnableSetting
       ***************************************************************************/
      /*!
       * \fn      bWriteMLNotificationSetting(t_Bool bMirrorLinkEnable)
       * \brief   Sets the MirrorLink On/Off setting value in datapool.
       * \param   [IN] enMirrorLinkEnable :
       *               e8USAGE_DISABLED - to set value as OFF
       *               e8USAGE_ENABLED - to set value as ON
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bWriteMLEnableSetting(tenEnabledInfo enMirrorLinkEnable) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bWriteDipoEnableSetting
       ***************************************************************************/
      /*!
       * \fn      bWriteMLNotificationSetting(t_Bool bDipoEnable)
       * \brief   Sets the DiPO On/Off setting value in datapool.
       *          (True - if enabled, else False)
       * \param   [IN] enDipoLinkEnable :
       *               e8USAGE_DISABLED - to set value as OFF
       *               e8USAGE_ENABLED - to set value as ON
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bWriteDipoEnableSetting(tenEnabledInfo enDipoLinkEnable) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bWriteAAPEnableSetting
       ***************************************************************************/
      /*!
       * \fn      bWriteAAPEnableSetting(t_Bool bAAPEnable)
       * \brief   Sets the DiPO On/Off setting value in datapool.
       *          (True - if enabled, else False)
       * \param   [IN] enAAPLinkEnable :
       *               e8USAGE_DISABLED - to set value as OFF
       *               e8USAGE_ENABLED - to set value as ON
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bWriteAAPEnableSetting(tenEnabledInfo enAAPLinkEnable) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bWriteOnCarEnableSetting
       ***************************************************************************/
      /*!
       * \fn      bWriteOnCarEnableSetting(tenEnabledInfo enOnCarLinkEnable)
       * \brief   Sets the OnCar On/Off setting value in datapool.
       *          (True - if enabled, else False)
       * \param   [IN] enOnCarLinkEnable :
       *               e8USAGE_DISABLED - to set value as OFF
       *               e8USAGE_ENABLED - to set value as ON
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bWriteOnCarEnableSetting(tenEnabledInfo enOnCarLinkEnable) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bWritemySPINEnableSetting
       ***************************************************************************/
      /*!
       * \fn      bWritemySPINNotificationSetting(t_Bool bmySPINEnable)
       * \brief   Sets the mySPIN On/Off setting value in datapool.
       *          (True - if enabled, else False)
       * \param   [IN] enmySPINLinkEnable :
       *               e8USAGE_DISABLED - to set value as OFF
       *               e8USAGE_ENABLED - to set value as ON
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bWritemySPINEnableSetting(tenEnabledInfo enmySPINLinkEnable) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bWriteCarlifeEnableSetting
       ***************************************************************************/
      /*!
       * \fn      bWriteCarlifeEnableSetting(t_Bool bCarlifeEnable)
       * \brief   Sets the Carlife On/Off setting value in datapool.
       *          (True - if enabled, else False)
       * \param   [IN] enCarlifeLinkEnable :
       *               e8USAGE_DISABLED - to set value as OFF
       *               e8USAGE_ENABLED - to set value as ON
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bWriteCarlifeEnableSetting(tenEnabledInfo enCarlifeLinkEnable) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bWriteMLNotificationSetting(t_Bool...)
       ***************************************************************************/
      /*!
       * \fn      bWriteMLNotificationSetting
       * \brief   Sets the MirrorLink Notification On/Off setting value in datapool.
       *          (True - if enabled, else False)
       * \param   [IN] bNotificationEnabled :
       *               True - to set Notfications as enabled
       *               False - to set Notfications as disabled
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bWriteMLNotificationSetting(t_Bool bNotificationEnabled) override;

      /***************************************************************************
       ** FUNCTION:  t_U8 Datapool::u8WriteVirginStartSetting(t_Bool...)
       ***************************************************************************/
      /*!
       * \fn      u8WriteVirginStartSetting
       * \brief   Sets the default value at Virgin Start in datapool.
       * \param   [IN] u8VirginStart : Default value to be set at Virgin Start
       * \retval  t_U8
       ***************************************************************************/
      t_U8 u8WriteVirginStartSetting(t_U8 u8VirginStart) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bWriteBluetoothMacAddress(t_String...)
       ***************************************************************************/
      /*!
       * \fn      bWriteBluetoothMacAddress
       * \brief   Set the accessory bluetooth statck MAC address
       * \param   [IN] szBluetoothMacAddr : Bluetooth MAC address.
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bWriteBluetoothMacAddress(t_String &szBluetoothMacAddr) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bReadBluetoothMacAddress(t_String...)
       ***************************************************************************/
      /*!
       * \fn      bReadBluetoothMacAddress
       * \brief   To read the accessory bluetooth statck MAC address
       * \param   [OUT] szBluetoothMacAddr : Bluetooth MAC address.
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bReadBluetoothMacAddress(t_String &szBluetoothMacAddr) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bWriteScreenAttributes()
       ***************************************************************************/
      /*!
       * \fn      bWriteScreenAttributes(const trDisplayAttributes& corfrDispAttr)
       * \brief   To write screen attributes to data pool
       * \param   [IN] corfrDispAttr : Screen Attributes
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bWriteScreenAttributes(const trDisplayAttributes& corfrDispAttr) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bReadScreenAttributes()
       ***************************************************************************/
      /*!
       * \fn      bReadScreenAttributes(trDisplayAttributes& rfrDispAttr)
       * \brief   To read screen attributes from data pool
       * \param   [IN] rfrDispAttr : Screen Attributes
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bReadScreenAttributes(trDisplayAttributes& rfrDispAttr) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bReadDiPODriveRestrictionInfo()
       ***************************************************************************/
      /*!
       * \fn      bReadDiPODriveRestrictionInfo(t_U8 &cu8DiPORestrictionInfo)
       * \brief   To read DiPO drive restriction info from data pool
       * \param   [OUT] cu8DiPORestrictionInfo : Restriction info.
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bReadDiPODriveRestrictionInfo(t_U8 &cu8DiPORestrictionInfo) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bWriteDiPODriveRestrictionInfo()
       ***************************************************************************/
      /*!
       * \fn      bWriteDiPODriveRestrictionInfo(trDisplayAttributes& rfrDispAttr)
       * \brief   To write DiPO drive restriction info from data pool
       * \param   [IN] cu8DiPORestrictionInfo : Restriction info.
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bWriteDiPODriveRestrictionInfo(t_U8 cu8DiPORestrictionInfo) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bReadAAPGeneralRestrictionInfo()
       ***************************************************************************/
      /*!
       * \fn      bReadAAPGeneralRestrictionInfo(t_U16 u16GeneralRestrictionInfo)
       * \brief   To read AAP general restriction info from data pool
       * \param   [IN] cu8DiPOGeneralRestrictionInfo : AAP General Restriction info.
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bReadAAPGeneralRestrictionInfo(t_U16 &cu8DiPOGeneralRestrictionInfo) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bWriteGeneralRestrictionInfo()
       ***************************************************************************/
      /*!
       * \fn      bWriteGeneralRestrictionInfo(t_U16 u16GeneralRestrictionInfo)
       * \brief   To write AAP general restriction info from data pool
       * \param   [IN] u16GeneralRestrictionInfo : General Restriction info.
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bWriteAAPGeneralRestrictionInfo(t_U16 u16GeneralRestrictionInfo) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bReadDipoGeneralRestrictionInfo()
       ***************************************************************************/
      /*!
       * \fn      bReadDipoGeneralRestrictionInfo(t_U16 &cu8DipoGeneralRestrictionInfo)
       * \brief   To read Dipo general restriction info from data pool
       * \param   [IN] cu8DipoGeneralRestrictionInfo : Dipo General Restriction info.
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bReadDipoGeneralRestrictionInfo(t_U16 &cu8DipoGeneralRestrictionInfo) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bWriteDipoGeneralRestrictionInfo()
       ***************************************************************************/
      /*!
       * \fn      bWriteDipoGeneralRestrictionInfo(t_U16 u16DipoGeneralRestrictionInfo)
       * \brief   To write Dipo general restriction info from data pool
       * \param   [IN] u16DipoGeneralRestrictionInfo : Dipo General Restriction info.
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bWriteDipoGeneralRestrictionInfo(t_U16 u16DipoGeneralRestrictionInfo) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bReadDriveSideInfo()
       ***************************************************************************/
      /*!
       * \fn      bReadDriveSideInfo(tenDriveSideInfo &rfenDriveSideInfo)
       * \brief   To read drive side position information.
       * \param   [OUT] rfenDriveSideInfo : Drive side information.
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bReadDriveSideInfo(tenDriveSideInfo &rfenDriveSideInfo) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bWriteDriveSideInfo()
       ***************************************************************************/
      /*!
       * \fn      bWriteDriveSideInfo(tenDriveSideInfo enDriveSideInfo)
       * \brief   Function to write the drive side information
       * \param   [IN] enDriveSideInfo : Drive side information.
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bWriteDriveSideInfo(const tenDriveSideInfo enDriveSideInfo) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bReadVehicleId()
       ***************************************************************************/
      /*!
       * \fn      bReadVehicleId()
       * \brief   Function to read the Vehicle Id information
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bReadVehicleId(t_String &szVehicleIdentifierentifier) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bWriteVehicleId()
       ***************************************************************************/
      /*!
       * \fn      bWriteVehicleId()
       * \brief   Function to write the Vehicle Id information
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bWriteVehicleId(t_String &szVehicleIdentifierentifier) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bWriteSelectionMode()
       ***************************************************************************/
      /*!
       * \fn      bWriteSelectionMode()
       * \brief   Function to write the SelectionMode
       * \param   [IN] enSelectionMode : Device selection mode
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bWriteSelectionMode(tenDeviceSelectionMode enSelectionMode) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bReadSelectionMode()
       ***************************************************************************/
      /*!
       * \fn      bReadSelectionMode()
       * \brief   Function to read the SelectionMode
       * \param   [OUT] enSelectionMode : Device selection mode
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bReadSelectionMode(tenDeviceSelectionMode &enSelectionMode) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bReadSPIVersion()
       ***************************************************************************/
      /*!
       * \fn      bReadSPIVersion()
       * \brief   Function to read the SPI Version information
       * \param   [OUT] szSPIVersion : SPI Version
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bReadSPIVersion(t_String &szSPIVersion) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bWriteSPIVersion()
       ***************************************************************************/
      /*!
       * \fn      bWriteSPIVersion()
       * \brief   Function to write the SPI Version information
       * \param   [IN] szSPIVersion : SPI Version
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bWriteSPIVersion(t_String szSPIVersion) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bWriteVehicleManufacturerInformation()
       ***************************************************************************/
      /*!
       * \fn      bWriteVehicleManufacturerInformation()
       * \brief   Function to write Vehicle Manufacturer information
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bWriteVehicleManufacturerInformation(
               const trVehicleManufacturerInformation& corfrVehicleManufacturerInformation) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bReadVehicleManufacturerInformation()
       ***************************************************************************/
      /*!
       * \fn      bReadVehicleManufacturerInformation()
       * \brief   Function to Read Vehicle Manufacturer information
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bReadVehicleManufacturerInformation(trVehicleManufacturerInformation& rfrVehicleManufacturerInformation)
               override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bWriteDisplayLastMode()
       ***************************************************************************/
      /*!
       * \fn      bWriteDisplayLastMode()
       * \brief   Function to write display last mode before ignition cycle
       * \param   corfbDispLastMode: [IN] display last mode before ignition off
       *                                  true: Projection, false: Native
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bWriteDisplayLastMode(const t_Bool& corfbDispLastMode) override;

      /***************************************************************************
       ** FUNCTION:  t_Void Datapool::vReadDisplayLastMode()
       ***************************************************************************/
      /*!
       * \fn      vReadDisplayLastMode(t_Bool& rbDispLastMode)
       * \brief   Function to read display last mode before ignition cycle
       * \retval  t_Void
       ***************************************************************************/
      t_Void vReadDisplayLastMode(t_Bool& rbDispLastMode) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bWriteAudioLastMode()
       ***************************************************************************/
      /*!
       * \fn      bWriteAudioLastMode()
       * \brief   Function to write audio last mode before ignition cycle
       * \param   corfbAudioLastMode: [IN] audio last mode before ignition off
       *                                   true: Projection, false: Native
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bWriteAudioLastMode(const t_Bool& corfbAudioLastMode) override;

      /***************************************************************************
       ** FUNCTION:  t_Void Datapool::vReadAudioLastMode()
       ***************************************************************************/
      /*!
       * \fn      vReadAudioLastMode(t_Bool& rbAudioLastMode)
       * \brief   Function to read audio last mode before ignition cycle
       * \retval  t_Void
       ***************************************************************************/
      t_Void vReadAudioLastMode(t_Bool& rbAudioLastMode) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bWriteAndroidDeviceTechnologyPreference()
       ***************************************************************************/
      /*!
       * \fn      bWriteAndroidDeviceTechnologyPreference()
       * \brief   Function to write the Android Device TechnologyPreference
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bWriteAndroidDeviceTechnologyPreference(const tenDeviceCategory coenTechnologyPreference) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bReadAndroidDeviceTechnologyPreference()
       ***************************************************************************/
      /*!
       * \fn      bReadAndroidDeviceTechnologyPreference()
       * \brief   Function to read Android Device the TechnologyPreference
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bReadAndroidDeviceTechnologyPreference(tenDeviceCategory &enTechnologyPreference) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bWriteAppleDeviceTechnologyPreference()
       ***************************************************************************/
      /*!
       * \fn      bWriteAppleDeviceTechnologyPreference()
       * \brief   Function to write the Apple Device TechnologyPreference
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bWriteAppleDeviceTechnologyPreference(const tenDeviceCategory coenTechnologyPreference) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bReadAppleDeviceTechnologyPreference()
       ***************************************************************************/
      /*!
       * \fn      bReadAppleDeviceTechnologyPreference()
       * \brief   Function to read Apple Device the TechnologyPreference
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bReadAppleDeviceTechnologyPreference(tenDeviceCategory &enTechnologyPreference) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::szGetSoftwareVersion()
       ***************************************************************************/
      /*!
       * \fn      szGetSoftwareVersion()
       * \brief   Function to get software version
       * \retval  t_String
       ***************************************************************************/
      t_String szGetSoftwareVersion() override;


      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bWriteHardKeyInfo()
       ***************************************************************************/
      /*!
       * \fn     bWriteHardKeyInfo()
       * \brief  Function to Write hard Key Info into data pool
       * \param  corfrsetSuppHardKeys:[IN] Keys Supported
       * \param  enDeviceCategory:[IN] Device Category
       * \retval t_Bool
       ***************************************************************************/

      t_Bool bWriteHardKeyInfo(const std::set<tenKeyCode>& corfrsetSuppHardKeys,
                               tenDeviceCategory enDeviceCategory) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bReadKeyCodeInfo()
       ***************************************************************************/
      /*!
       * \fn      bReadKeyCodeInfo()
       * \brief   Function to Read KeyCodeInfo from data pool
       * \param   rfrsetKeyCodes  :[OUT] Keys Supported
       * \param   enDeviceCategory:[IN] Device Category
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bReadKeyCodeInfo(std::set<tenKeyCode>& rfrsetKeyCodes,tenDeviceCategory enCategory) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bWriteSoftKeyInfo()
       ***************************************************************************/
      /*!
       * \fn      bWriteSoftKeyInfo()
       * \brief   Function to Write soft Key Info into data pool
       * \param   corfrsetSuppSoftKeys:[IN] Keys Supported
       * \param   enDeviceCategory:[IN] Device Category
       * \retval  t_Bool
       ***************************************************************************/

      t_Bool bWriteSoftKeyInfo(const std::set<tenKeyCode>& corfrsetSuppSoftKeys,
                               tenDeviceCategory enDeviceCategory) override;

      /***************************************************************************
       ** FUNCTION:  t_Bool Datapool::bReadSoftKeyInfo()
       ***************************************************************************/
      /*!
       * \fn      bReadSoftKeyInfo()
       * \brief   Function to Read SoftKeyInfo from data pool
       * \param   rfrsetKeyCodes  :[OUT] Keys Supported
       * \param   enDeviceCategory:[IN] Device Category
       * \retval  t_Bool
       ***************************************************************************/
      t_Bool bReadSoftKeyInfo(std::set<tenKeyCode>& rfrsetKeyCodes,
                              tenDeviceCategory enCategory) override;
      /***************************************************************************
      ****************************END OF PUBLIC***********************************
      ***************************************************************************/

   private:
     /***************************************************************************
     *********************************PRIVATE************************************
     ***************************************************************************/

     //! Map Containing supported Hard key information for technologies
     std::map<tenDeviceCategory,std::set<tenKeyCode>> m_mapTechSuppHardkeys;

     //! Map Containing supported Soft key information for technologies
     std::map<tenDeviceCategory,std::set<tenKeyCode>> m_mapTechSuppSoftkeys;

     /***************************************************************************
     ** FUNCTION:  t_Bool Datapool::vStringCopy()
     ***************************************************************************/
     /*!
      * \fn      vStringCopy()
      * \brief   Function to copy the string to char array
      * \retval  t_Bool
      ***************************************************************************/
      virtual t_Void vStringCopy(t_Char* cDestination, const t_String szSource);

     /***************************************************************************
      ** FUNCTION:  t_Bool Datapool::bReadAAKeyCodeInfo()
      ***************************************************************************/
     /*!
      * \fn      bReadAAKeyCodeInfo()
      * \brief   Function to Read Android Auto KeyCodeInfo from data pool
      * \retval  t_Bool
      ***************************************************************************/
      t_Bool bReadAAKeyCodeInfo(std::set<tenKeyCode>& rfrsetAASuppKeys);

     /***************************************************************************
      ** FUNCTION:  t_Bool Datapool::bReadDiPoKeyCodeInfo()
      ***************************************************************************/
     /*!
      * \fn      bReadDiPoKeyCodeInfo()
      * \brief   Function to Read DiPo KeyCodeInfo from data pool
      * \retval  t_Bool
      ***************************************************************************/
      t_Bool bReadDiPoKeyCodeInfo(std::set<tenKeyCode>& rfrsetDiPoSuppKeys);

     /***************************************************************************
      ** FUNCTION:  t_Bool Datapool::bReadDiPoSoftKeyInfo()
      ***************************************************************************/
     /*!
      * \fn      bReadDiPoSoftKeyInfo()
      * \brief   Function to Read DiPo Soft KeyInfo from data pool
      * \retval  t_Bool
      ***************************************************************************/
      t_Bool bReadDiPoSoftKeyInfo(std::set<tenKeyCode>& rfrsetDiPoSuppKeys);

     /***************************************************************************
      ** FUNCTION:  t_Bool Datapool::bWriteAAHardKeyInfo()
      ***************************************************************************/
     /*!
      * \fn      bWriteAAHardKeyInfo()
      * \brief   Function to Write AA Hard Key Info to data pool
      * \retval  t_Bool
      ***************************************************************************/
      t_Bool bWriteAAHardKeyInfo(const std::set<tenKeyCode>& corfrsetAASuppHardKeys);

     /***************************************************************************
      ** FUNCTION:  t_Bool Datapool::bWriteDiPoHardKeyInfo()
      ***************************************************************************/
     /*!
      * \fn      bWriteDiPoHardKeyInfo()
      * \brief   Function to Write DiPo Hard KeyCodeInfo to data pool
      * \retval  t_Bool
      ***************************************************************************/
      t_Bool bWriteDiPoHardKeyInfo(const std::set<tenKeyCode>& corfrsetDiPoSuppHardKeys);

     /***************************************************************************
      ** FUNCTION:  t_Bool Datapool::bWriteDiPoSofKeyInfo()
      ***************************************************************************/
     /*!
      * \fn      bWriteDiPoSoftKeyInfo()
      * \brief   Function to Write DiPo Soft Key CodeInfo to data pool
      * \retval  t_Bool
      ***************************************************************************/
      t_Bool bWriteDiPoSoftKeyInfo(const std::set<tenKeyCode>& corfrsetDiPoSuppSoftKeys);

     /***************************************************************************
      ****************************END OF PRIVATE *********************************
      ***************************************************************************/
};

#endif  // _DATAPOOL_H_

///////////////////////////////////////////////////////////////////////////////
// <EOF>
