/*!
 *******************************************************************************
 * \file             spi_tclMediator.h
 * \brief            Mediator to distribute messages among SPI components
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Mediator to distribute messages among SPI components
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 17.01.2014 |  Pruthvi Thej Nagaraju       | Initial Version
 27.05.2015 |  Tejaswini H B(RBEI/ECP2)     | Added Lint comments to suppress C++11 Errors
 \endverbatim
 ******************************************************************************/

#ifndef SPI_TCLMEDIATOR_H_
#define SPI_TCLMEDIATOR_H_

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "spi_tclMediatorIntf.h"

/******************************************************************************
 | defines and macros and constants(scope: module-local)
 |----------------------------------------------------------------------------*/

/******************************************************************************
 | typedefs (scope: module-local)
 |----------------------------------------------------------------------------*/

/*!
 * \class spi_tclMediator
 * \brief Mediator to distribute messages among SPI components
 */
class spi_tclMediator: public GenericSingleton<spi_tclMediator>, public spi_tclMediatorIntf
{
   public:

      /***************************************************************************
       *********************************PUBLIC*************************************
       ***************************************************************************/

      /***************************************************************************
       ** FUNCTION:  spi_tclMediator::~spi_tclMediator
       ***************************************************************************/
      /*!
       * \fn     ~spi_tclMediator()
       * \brief  Destructor
       **************************************************************************/
      ~spi_tclMediator();

      /***************************************************************************
       ** FUNCTION:  spi_tclMediator::vRegisterCallbacks(const trConnMngrCallback &rfrConnMngrCb)
       ***************************************************************************/
      /*!
       * \fn     vRegisterCallbacks(const trConnMngrCallback &rfrConnMngrCb)
       * \brief  Register for connection manager callbacks
       * \param  rfrConnMngrCb structure holding connection manager callbacks
       **************************************************************************/
      t_Void vRegisterCallbacks(const trConnMngrCallback &rfrConnMngrCb);

      /***************************************************************************
       ** FUNCTION:  spi_tclMediator::vRegisterCallbacks(const trSelectDeviceCallbacks &rSelDevCb)
       ***************************************************************************/
      /*!
       * \fn     vRegisterCallbacks(const trSelectDeviceCallbacks &rSelDevCb)
       * \brief  Register for connection management callbacks
       * \param  rSelDevCb structure holding select device callbacks
       **************************************************************************/
      t_Void vRegisterCallbacks(const trSelectDeviceCallbacks &rSelDevCb);

      /***************************************************************************
       ** FUNCTION:  spi_tclMediator::vRegisterCallbacks(const trSelectDeviceCallbacks &rSelDevCb)
       ***************************************************************************/
      /*!
       * \fn     vRegisterCallbacks(const trDevDiagnosisCallback &rDevDiagnosisCb)
       * \brief  Register for diagnosis callbacks
       * \param  rDevDiagnosisCb structure holding device connection callbacks
       **************************************************************************/
      t_Void vRegisterCallbacks(const trDevDiagnosisCallback &rDevDiagnosisCb);

      /***************************************************************************
       ** FUNCTION:  spi_tclMediator::vRegisterCallbacks(const trAppLauncherCallbacks &corfrAppLauncherCb)
       ***************************************************************************/
      /*!
       * \fn     vRegisterCallbacks(const trAppLauncherCallbacks &corfrAppLauncherCb)
       * \brief  Register for Launch App callbacks
       * \param  corfrAppLauncherCb: [IN] structure holding AppLauncher callbacks
       **************************************************************************/
      t_Void vRegisterCallbacks(const trAppLauncherCallbacks& corfrAppLauncherCb);

      /***************************************************************************
       ** FUNCTION:  spi_tclMediator::vRegisterCallbacks(const trAAPSensorCallback& corfrDayNightModeCb)
       ***************************************************************************/
      /*!
       * \fn     vRegisterCallbacks(const trAAPSensorCallback& corfrDayNightModeCb)
       * \brief  Register for DayNight callbacks
       * \param  corfrDayNightModeCb: [IN] structure holding DayNight callbacks
       **************************************************************************/
      t_Void vRegisterCallbacks(const trAAPSensorCallback& corfrDayNightModeCb);

      /***************************************************************************
       ** FUNCTION:  spi_tclMediator::vRegisterCallbacks(const trOnCarSensorCallback& corfrDayNightModeCbOnCar)
       ***************************************************************************/
      /*!
       * \fn     vRegisterCallbacks(const trOnCarSensorCallback& corfrDayNightModeCbOnCar)
       * \brief  Register for DayNight callbacks
       * \param  corfrDayNightModeCbOnCar: [IN] structure holding DayNight callbacks
       **************************************************************************/
      t_Void vRegisterCallbacks(const trOnCarSensorCallback& corfrDayNightModeCbOnCar);

      /***************************************************************************
       ** FUNCTION:  spi_tclMediator::vRegisterCallbacks(const trWiFiSetupHndlrCallbacks& corfrDayNightModeCb)
       ***************************************************************************/
      /*!
       * \fn     vRegisterCallbacks(const trWiFiSetupHndlrCallbacks& corfrWiFiSetupHndlrCb)
       * \brief  Register for WiFiSetupHndlrCallbacks callbacks
       * \param  corfrWiFiSetupHndlrCb: [IN] structure holding WiFiSetupHndlr callbacks
       **************************************************************************/
      t_Void vRegisterCallbacks(const trWiFiSetupHndlrCallbacks& corfrWiFiSetupHndlrCb);

      /***************************************************************************
       ** FUNCTION:  spi_tclMediator::vRegisterCallbacks(const trNotifyDevAppStateCallbacks &rAppStateCb)
       ***************************************************************************/
      /*!
       * \fn     vRegisterCallbacks(const trNotifyDevAppStateCallbacks &rAppStateCb)
       * \brief  Register for App State Callbacks
       * \param  corfrAppStateCb structure holding AppState callbacks
       **************************************************************************/
       t_Void vRegisterCallbacks(const trNotifyDevAppStateCallbacks &corfrAppStateCb);

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclMediator::vPostSelectDeviceResult
       ***************************************************************************/
      /*!
       * \fn     vPostSelectDeviceResult
       * \brief  Called by SPI Component to post result for select device request
       * \param  enCompID : SPI component posting select device response
       * \param  enErrorCode   : Result of select device request
       * \sa     spi_tclCmdInterface::vSelectDevice
       **************************************************************************/
       t_Void vPostSelectDeviceRes(tenCompID enCompID, tenErrorCode enErrorCode);

       /***************************************************************************
        ** FUNCTION:  spi_tclMediator::vPostDeviceConnection
        ***************************************************************************/
       /*!
        * \fn     vPostDeviceConnection()
        * \brief  Called by SPI component when a device is Connected
        * \param u32DeviceHandle: Device Handle of the device Connected
        **************************************************************************/
       t_Void vPostDeviceConnection(const t_U32 u32DeviceHandle);

      /***************************************************************************
       ** FUNCTION:  spi_tclMediator::vPostDeviceDisconnection
       ***************************************************************************/
      /*!
       * \fn     vPostDeviceDisconnection()
       * \brief  Called by SPI component when a device is disconnected
       * \param u32DeviceHandle: Device Handle of the device disconnected
       **************************************************************************/
      t_Void vPostDeviceDisconnection(const t_U32 u32DeviceHandle);

      /***************************************************************************
       ** FUNCTION:  spi_tclMediator::vPostDeviceDisconnectionStatus
       ***************************************************************************/
      /*!
       * \fn     vPostDeviceDisconnectionStatus()
       * \brief  Called by SPI component to inform Diagnosis when a device is disconnected
       * \param u32DeviceHandle: Device Handle of the device disconnected
       **************************************************************************/
      t_Void vPostDeviceDisconnectionStatus(const t_U32 u32DeviceHandle);

      /***************************************************************************
       ** FUNCTION:  spi_tclMediator::vPostDeviceConnection
       ***************************************************************************/
      /*!
       * \fn     vPostDeviceConnection()
       * \brief  Called by SPI component to inform Diagnosis when a device is Connected
       * \param u32ProductID: ProductId of the device
       * \param u32VendorID: VendorId of the device
       * \param u32DeviceHandle: Device Handle of the device Connected
       **************************************************************************/
      t_Void vPostDeviceConnectionStatus(const t_U32 u32ProductID, const t_U32 u32VendorID, const t_U32 u32DeviceHandle);

      /***************************************************************************
       ** FUNCTION:  spi_tclMediator::vPostAutoDeviceSelection
       ***************************************************************************/
      /*!
       * \fn     vPostAutoDeviceSelection()
       * \brief  Called by SPI component when a device is selected automatically
       *         by applying selection strategy
       * \param u32DeviceHandle: Device Handle
	   * \param enConnReq : SELECT/DESELECT
	   * \param enSelectReason : Reason for Trigger
	   * \param enDevCategory : Device Category
       **************************************************************************/
      t_Void vPostAutoDeviceSelection(const t_U32 cou32DeviceHandle,
               tenDeviceConnectionReq enConnReq,
			   tenSelectReason enSelectReason = e8_REASON_UNKNOWN,
			   tenDeviceCategory enDevCategory = e8DEV_TYPE_UNKNOWN);

      /***************************************************************************
       ** FUNCTION:  spi_tclMediator::vPostSetUserDeselect
       ***************************************************************************/
      /*!
       * \fn     vPostSetUserDeselect()
       * \brief  Called by SPI component when a device is deselected as a result of
       *         user action
       * \param u32DeviceHandle: Device Handle
       * \param bUserDeselect : User Deselect Falg
       **************************************************************************/
      t_Void vPostSetUserDeselect(const t_U32 cou32DeviceHandle, t_Bool bUserDeselect);


      /***************************************************************************
       ** FUNCTION:  spi_tclMediator::vPostDayNightMode
       ***************************************************************************/
      /*!
       * \fn     vPostDayNightMode()
       * \brief  Called by SPI component to set Day/Night
       * \param enVehicleConfig: contains day/night
       **************************************************************************/
      t_Void vPostDayNightMode(tenVehicleConfiguration enVehicleConfig);

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclMediator::vPostLaunchAppResult
       ***************************************************************************/
      /*!
       * \fn     vPostLaunchAppResult(
       *              cons tenCompID coenCompID
       *              const t_U32 cou32DevId,
       *              const t_U32 cou32AppId,
       *              const tenDiPOAppType coenDiPOAppType,
       *              tenErrorCode enErrorCode,
       *              const trUserContext& rfrCUsrCntxt)
       * \brief  To send the Terminate applications response to the registered 
       *          classes
       * \param  coenCompID       : Uniquely identifies the target Device.
       * \param  cou32DevId       : Unique Device Id
       * \param  cou32AppId       : Application Id
       * \param  coenDiPOAppType  : DiPo App Type
       * \param  enErrorCode      : Error code
       * \param  rfrCUsrCntxt     : User Context
       **************************************************************************/
       t_Void vPostLaunchAppResult(const tenCompID coenCompID,
          const t_U32 cou32DevId,
          const t_U32 cou32AppId,
          const tenDiPOAppType coenDiPOAppType,
          tenErrorCode enErrorCode,
          const trUserContext& rfrCUsrCntxt);

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclMediator::vPostTerminateAppResult
       ***************************************************************************/
      /*!
       * \fn     vPostTerminateAppResult(
       *              cons tenCompID coenCompID
       *              const t_U32 cou32DevId,
       *              const t_U32 cou32AppId,
       *              tenErrorCode enErrorCode,
       *              const trUserContext& rfrCUsrCntxt)
       * \brief  To send the Terminate applications response to the registered 
       *          classes
       * \param  coenCompID       : Uniquely identifies the target Device.
       * \param  cou32DevId       : Unique Device Id
       * \param  cou32AppId       : Application Id
       * \param  enErrorCode      : Error code
       * \param  rfrCUsrCntxt     : User Context
       **************************************************************************/
       t_Void vPostTerminateAppResult(const tenCompID coenCompID,
          const t_U32 cou32DevId,
          const t_U32 cou32AppId,
          tenErrorCode enErrorCode,
          const trUserContext& rfrCUsrCntxt);

       /***************************************************************************
        ** FUNCTION:  spi_tclMediator::vApplySelectionStrategy
        ***************************************************************************/
       /*!
        * \fn     vApplySelectionStrategy()
        * \brief  Called to trigger device selection automatically
        **************************************************************************/
       t_Void vApplySelectionStrategy(tenDeviceCategory enDeviceCategory = e8DEV_TYPE_UNKNOWN);

       /***************************************************************************
        ** FUNCTION:  spi_tclMediator::vPostBTDeviceInfo
        ***************************************************************************/
       /*!
        * \fn     vPostBTDeviceInfo()
        * \brief  Called by SPI component when BT device info of active projection
        *         device becomes available
        * \param  u32DeviceHandle: Unique handle of projection device
        * \param  szBTDeviceName: BT Device name of projection device
        * \param  szBTAddress: BT Address of projection device
        **************************************************************************/
       t_Void vPostBTDeviceInfo(t_U32 u32DeviceHandle, t_String szBTDeviceName,
             t_String szBTAddress);

       /***************************************************************************
        ** FUNCTION:  spi_tclMediator::vPostInitiateWirelessDiscoveryResult
        ***************************************************************************/
       /*!
        * \fn     vPostInitiateWirelessDiscoveryResult()
        * \brief  Called by SPI component when result of InitiateWirelessDiscovery
        *         operation is available
        * \param  bResult: true or false
        **************************************************************************/
       t_Void vPostInitiateWirelessDiscoveryResult(t_Bool bResult);

       /***************************************************************************
        ** FUNCTION:  spi_tclMediator::vPostBTLimitationModeResult
        ***************************************************************************/
       /*!
        * \fn     vPostBTLimitationModeResult()
        * \brief  Called by SPI component when BT Lim mode info is available
        * \param  orfrBTLimitationModeInfo: BT Lim mode info
        **************************************************************************/
       t_Void vPostBTLimitationModeResult(const trBTLimitationModeInfo &corfrBTLimitationModeInfo);

       /***************************************************************************
        ** FUNCTION:  t_Void spi_tclMediator::vPostWiFiCredentials
        ***************************************************************************/
       /*!
        * \fn     vPostWiFiCredentials(t_String szBTMACAddress)
        * \brief  Trigger to provide Wi-Fi Credentials to clients after the required BTLimitationMode is set.
        * \param  szBTMACAddress: Unique ID of the device
		* \retval t_Void
        **************************************************************************/	   
	   t_Void vPostWiFiCredentials(t_String szBTMACAddress);

	   /***************************************************************************
	    ** FUNCTION:  spi_tclMediator::vPostSetBTLimitationModeResult
	    ***************************************************************************/
	   /*!
	    * \fn     vPostSetBTLimitationModeResult()
	    * \brief  Called by SPI component when BT Lim mode info is available
	    * \param  bSetBTLimitationModeResult: Set BT Lim mode Result info
	    **************************************************************************/
	   t_Void vPostSetBTLimitationModeResult(t_Bool bSetBTLimitationModeResult);

       /***************************************************************************
        ** FUNCTION:  t_Void spi_tclMediator::vPostSelectionProgressState
        ***************************************************************************/
       /*!
        * \fn     vPostSelectionProgressState(const t_U32 cou32DeviceHandle, tenDeviceSelectionProgressState enDeviceSelectionProgressState)
        * \brief  Updates device selection progress state.
        * \param  cou32DeviceHandle: [IN] Device ID
        * \param  enDeviceSelectionProgressState : [IN] Selection Progress state
		* \retval t_Void
        **************************************************************************/
	   t_Void vPostSelectionProgressState(const t_U32 cou32DeviceHandle, tenDeviceSelectionProgressState enDeviceSelectionProgressState);
	   
	   /***************************************************************************
        ** FUNCTION:  spi_tclMediator::vPostDeviceChanged
        ***************************************************************************/
       /*!
        * \fn     vPostDeviceChanged()
        * \brief  Called by SPI component when a device is Connected
        * \param u32DeviceHandle: Device Handle of the device Connected
        **************************************************************************/
       t_Void vPostDeviceChanged(const t_U32 u32DeviceHandle);

      /***************************************************************************
       ** FUNCTION:  spi_tclMediator::vPostDevAppState
       ***************************************************************************/
      /*!
       * \fn     vPostAppStateSpeechStatus()
       * \brief  Called by SPI component to send speech app status
       * \param enSpeechAppState: Speech  states
       * \param  enPhoneAppState: phone states
       * \param enNavAppState: Navigation states
       **************************************************************************/
      t_Void vPostDevAppState(tenSpeechAppState enSpeechAppState,
                                 tenPhoneAppState enPhoneAppState,
                                 tenNavAppState enNavAppState);
								 
   
	   /***************************************************************************
	    ** FUNCTION:  spi_tclMediatorIntf::vOnHUWiFiAPStateChange
	    ***************************************************************************/
	   /*!
	    * \fn     vOnHUWiFiAPStateChange(t_Bool bIsSetToOn)
       * \brief  Called when WiFi state changes.
	   * \param  enSelectReason : Reason for Trigger
	   * \retval t_Void
       **************************************************************************/ 
	   t_Void vOnHUWiFiAPStateChange(t_Bool bIsSetToOn);

	   /***************************************************************************
	     ** FUNCTION:  spi_tclMediator::vRegisterCallbacks(const trNotifyRsrcMngrCbs
	   ***************************************************************************/
	   /*!
	   * \fn     vRegisterCallbacks(const trNotifyRsrcMngrCbs &rNotifyRsrcMngrCbs)
	   * \brief  Register for ResourceMngr related callbacks
	   * \param  rNotifyRsrcMngrCbs structure holding callbacks
	   **************************************************************************/
	   t_Void vRegisterCallbacks(const trNotifyRsrcMngrCbs &rNotifyRsrcMngrCbs) override;
       
      /***************************************************************************
       ** FUNCTION:  spi_tclMediatorIntf::vRegisterCallbacks(const trDeviceDisconnectCb& corfrDeviceDisconnectCb)
       ***************************************************************************/
      /*!
       * \fn     vRegisterCallbacks(const trDeviceDisconnectCb& corfrDeviceDisconnectCb)
       * \brief  Register for Device disconnect callback
       * \param  corfrDeviceDisconnectCb: [IN] structure holding Device disconnect callbacks
       **************************************************************************/
      t_Void vRegisterCallbacks(const trDeviceDisconnectCb& corfrDeviceDisconnectCb) override;

	   /***************************************************************************
	   ** FUNCTION:  t_Void spi_tclMediator::vOnDisableBluetoothTrigger(
	   ***************************************************************************/
	   /*!
	   * \fn      t_Void vOnDisableBluetoothTrigger(t_String szBTAddress)
	   * \brief   Method to notify that DisableBT request is received
	   * \param   [IN] szBTAddress: Unique Device ID
	   * \retval  t_Void
	   **************************************************************************/
	   t_Void vOnDisableBluetoothTrigger(t_String szBTAddress) override;

      /***************************************************************************
      ** FUNCTION:  t_Void spi_tclMediator::vOnEvaluateInitalModes()
      ***************************************************************************/
      /*!
      * \fn      t_Void vOnEvaluateInitalModes
      * \brief   Method to check and set inital modes before session is started
      * \param   [IN] corfszBTAddress: Unique Device ID
      * \retval  t_Void
      **************************************************************************/
      t_Void vOnEvaluateInitalModes(const t_String& corfszBTAddress) override;

      /***************************************************************************
       ** FUNCTION:  spi_tclMediator::vOnDeviceConnectstoWirelessAP
       ***************************************************************************/
      /*!
       * \fn     vOnDeviceConnectstoWirelessAP(const t_U32 cou32DeviceHandle, t_Bool &rfrbIsPreselectionValid)
       * \brief  Called when iphone Device connects to Carplay wireless AP
       * \brief  Returns boolean value, True : if the pre-selection request is valid for the device selection request once the device is connects to carplaywirless AP
       * \param  cou32DeviceHandle           : [IN] Device handle
       * \param  rfrbIsPreSelectionValid     : [OUT] boolean value
       **************************************************************************/
      t_Void vOnDeviceConnectstoWirelessAP(const t_U32 cou32DeviceHandle, t_Bool &rfrbIsPreselectionValid);

      /***************************************************************************
       ** FUNCTION: t_Void spi_tclMediator::vPostInitiateWirelessDiscovery()
       ***************************************************************************/
      /*!
       * \fn      vPostInitiateWirelessDiscovery(t_String szBTMACAddress,tenDeviceCategory enDevCategory)
      * \param   szBTMACAddress : [IN] BTMACAddress
      * \param   enDevCategory : Device Category
       * \brief   vPostInitiateWirelessDiscovery Method. Invoked internally when the device appears over carplay wireless AP.
       **************************************************************************/
      t_Void vPostInitiateWirelessDiscovery(const trUserContext& rfcorUsrCntxt,
										t_String szBTMACAddress,
                                        tenDeviceCategory enDevCategory);

      /***************************************************************************
       ** FUNCTION:  spi_tclMediator::vGetDeviceSelectorCurrentSelectionState
       ***************************************************************************/
      /*!
       * \fn     vGetDeviceSelectorCurrentSelectionState(trDeviceSelectionStateDetails& rfrDeviceSelectionDetails)
       * \brief  Triggered When there is InitiateWirelesssDiscovery MethodStart from HMI
       * \param  rfrDeviceSelectionDetails   : [OUT] Device Selection Details
       **************************************************************************/
      t_Void vGetDeviceSelectorCurrentSelectionState(trDeviceSelectionStateDetails& rfrDeviceSelectionDetails);

      /***************************************************************************
      ** FUNCTION: t_Void spi_tclMediator::vReportSessionErrorStatus()
      ***************************************************************************/
      /*!
       * \fn     vReportSessionErrorStatus
       * \param   t_U32 : cou32DevId
       * \param   enDevCategory : Device Category
       * \param   tenSessionStatus : Session status
       * \brief   vReportSessionErrorStatus reports session error status to resource manager
       **************************************************************************/
      t_Void vReportSessionErrorStatus(t_U32 cou32DevId,tenDeviceCategory coenDevCat,tenSessionStatus coenSessionStatus);

      /***************************************************************************
       ** FUNCTION:  spi_tclMediatorIntf::vOnRetrailSetDeviceSelectionState
       ***************************************************************************/
      /*!
       * \fn     vOnRetrailSetDeviceSelectionState()
       * \brief  Called by SPI component when a device is selected automatically
       *         by applying retrail strategy
       * \param u32DeviceHandle: Device Handle
       **************************************************************************/
       t_Void vOnRetrailSetDeviceSelectionState(const t_U32 cou32DeviceHandle);

      //! Generic singleton class is made friend so that the constructor is accessible
      friend class GenericSingleton<spi_tclMediator> ;
   private:

      /***************************************************************************
       *********************************PRIVATE***********************************
       ***************************************************************************/

      /***************************************************************************
       ** FUNCTION:  spi_tclMediator::spi_tclMediator
       ***************************************************************************/
      /*!
       * \fn     spi_tclMediator()
       * \brief  Default Constructor
       **************************************************************************/
      spi_tclMediator();

      //! Structure to hold select device callbacks
      std::vector<trSelectDeviceCallbacks> m_vecrSelDevCbs;

      //! AppLauncher callbacks
      trAppLauncherCallbacks m_rAppLauncherCb;

      //! DayNightMode callback
      trAAPSensorCallback m_rDayNightCb;

      //! DayNightMode callback OnCar
      trOnCarSensorCallback m_rDayNightCbOnCar;

      //! Connection manager callbacks
      trConnMngrCallback m_rConnMngrCb;

      //!WiFiSetupHndlr callbacks
      trWiFiSetupHndlrCallbacks m_rWiFiSetupHndlrCallbacks;

      //! Structure to hold App State callbacks
      std::vector<trNotifyDevAppStateCallbacks> m_vecrAppStateCbs;

      trNotifyRsrcMngrCbs m_rNotifyRsrcMngrCbs;

      //! Structure to hold device diagnosis callbacks
      std::vector<trDevDiagnosisCallback> m_vecrDevDiagnosisCbs;

      trDeviceDisconnectCb m_rDeviceDisconnectCb;

};

#endif /* SPI_TCLMEDIATOR_H_ */
