/*!
 *******************************************************************************
 * \file             spi_tclFactory.cpp
 * \brief            Object Factory class responsible for creation of SPI Objects
 *******************************************************************************
 \verbatim
 PROJECT:        Gen3
 SW-COMPONENT:   Smart Phone Integration
 DESCRIPTION:    Object Factory class responsible for creation of SPI Objects
 COPYRIGHT:      &copy; RBEI

 HISTORY:
 Date       |  Author                      | Modifications
 17.01.2014 |  Pruthvi Thej Nagaraju       | Initial Version
 13.02.2014 |  Shihabudheen P M            | Added 1.poGetMainAppInstance().
                                                  2.vInitialize()
 06.04.2014 |  Ramya Murthy                | Initialisation sequence implementation
 10.06.2014 |  Ramya Murthy                | Audio policy redesign implementation.
 31.07.2014 |  Ramya Murthy                | SPI feature configuration via LoadSettings()
 05.06.1990| Tejaswini HB                  | Added Lint comments to suppress C++11 errors
 \endverbatim
 ******************************************************************************/

/******************************************************************************
 | includes:
 |----------------------------------------------------------------------------*/
#include "spi_tclRespInterface.h"
#include "spi_tclFactory.h"
#include "spi_tclMediator.h"
#include "spi_tclDeviceSwitcher.h"
#include "spi_tclConnSettings.h"
#include "spi_tclDeviceSelector.h"
#include "spi_tclDeviceSwitcher.h"
#include "spi_tclDiscoveryDataIntf.h"
#include "spi_tclUSBDiscoverer.h"
#include "spi_tclUSBDiscovererImpl.h"
#include "spi_tclUSBReset.h"
#include "spi_tclDeviceIDUtility.h"
#include "spi_tclConfigurationManager.h"
#include "spi_tclDynamicConfiguration.h"
#include "spi_tclStaticConfiguration.h"
#include "spi_tclDefSetLoader.h"
#include "spi_tclWiFiSetupHndlr.h"
#include "spi_tclAppLauncher.h"
#include "Datapool.h"
#include "spi_tclCalibrationReaderBase.h"
#include "KdsHandler.h"


#include "spi_tclStaticConfiguration.h"


#include "spi_tclCarplaySwitch.h"
#include "spi_tclDiPoConnection.h"
#include "spi_tclAppleDiscoverer.h"
#ifdef VARIANT_S_FTR_ENABLE_SPI_ANDROIDAUTO
#include "spi_tclAautoConnection.h"
#include "spi_tclAAPSwitch.h"
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_MYSPIN
#include "spi_tclMySPINConnMngr.h"
#include "spi_tclmySPINSwitch.h"
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_BAIDU_CARLIFE
#include "spi_tclBDCLConnection.h"
#include "spi_tclBDCLSwitch.h"
#endif
#ifdef VARIANT_S_FTR_ENABLE_SPI_ONCAR
#include "spi_tclOnCarConnMngr.h"
#include "spi_tclOnCarSwitch.h"
#endif
#include <dlfcn.h>

//! Includes for Trace files
#include "Trace.h"
#ifdef TARGET_BUILD
#ifdef VARIANT_S_FTR_ENABLE_TRC_GEN
#define ETG_DEFAULT_TRACE_CLASS TR_CLASS_SMARTPHONEINT_TCLSERVICE
#include "trcGenProj/Header/spi_tclFactory.cpp.trc.h"
#endif
#endif


#define LOAD_SETTINGS(COMPONENT_PTR) \
   if (NULL != COMPONENT_PTR)                         \
   {                                                  \
      COMPONENT_PTR->vLoadSettings();  \
   }

#define SAVE_SETTINGS(COMPONENT_PTR)   \
   if (NULL != COMPONENT_PTR)          \
   {                                   \
      COMPONENT_PTR->vSaveSettings();  \
   }

#define LOAD_SETTINGS_COMPLETED(COMPONENT_PTR) \
   if (NULL != COMPONENT_PTR)                         \
   {                                                  \
      COMPONENT_PTR->vLoadSettingsCompleted();  \
   }

//lint -save -e1055 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1013 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e1401 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e19 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e10 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e55 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e58 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e48 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e808 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e63 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e40 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e64 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e746 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e515 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e516 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported
//lint -save -e601 PQM_authorized_multi_492_to_494   Reason: C++11 not fully supported  
#ifdef VARIANT_S_FTR_ENABLE_SPI_MIRRORLINK
std::map<tenDiscovererType,GetDiscoverer_t*> Discoverer;
std::map<tenDeviceCategory,GetConnection_t*> Connection;
std::map<tenDeviceCategory,GetDeviceSwitch_t*> DeviceSwitch;
#endif

/***************************************************************************
 *********************************PUBLIC*************************************
 ***************************************************************************/

/***************************************************************************
 ** FUNCTION:  spi_tclFactory::~spi_tclFactory
 ***************************************************************************/
spi_tclFactory::~spi_tclFactory()
{
   ETG_TRACE_USR1(("spi_tclFactory::~spi_tclFactory() entered\n"));

   m_poMainApp       = NULL;
   m_poRespInterface = NULL;
   m_poConnMngr      = NULL;
   m_poVideo         = NULL;
   m_poAppMngr       = NULL;
   m_poAudio         = NULL;
   m_poInputHandler  = NULL;
   m_poBluetooth     = NULL;
   m_poDataService   = NULL;
   m_poRsrcMngr      = NULL;
   m_poLegalDisclaimer = NULL;
   m_poTechnologySelector =NULL;
   m_poDeviceListHandler = NULL;
   m_poWiFi          = NULL;
   m_poDeviceDiscoverer = NULL;
   m_poDiagnosis = NULL;

   m_poSQLite3db   = NULL;
#ifdef VARIANT_S_FTR_ENABLE_SPI_MIRRORLINK
   m_poDynMLLibrary     = NULL;
#endif
   m_poConfigurationManager = NULL;
   m_poLaunchApp = NULL;
   m_poDefSet = NULL;
   m_poWifiSetupHndlr = NULL;
   m_poUSBDiscoverer = NULL;
   m_poDeviceSwitcher = NULL;
   m_poDevHistory = NULL;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclFactory::bInitialize()
 ***************************************************************************/
t_Bool spi_tclFactory::bInitialize()
{
   ETG_TRACE_USR1(("spi_tclFactory::bInitialize() entered"));

   std::map<tenDeviceCategory, spi_tclConnection*> mappoConnHandlers;
   std::map<tenDeviceCategory, spi_tclDeviceSwitchBase*> mappoDeviceSwitchers;
   std::map<tenDiscovererType, spi_tclDiscovererBase*> mapoDiscoverers;
   spi_tclDiscoveryDataIntf *poDiscovererDataIntf = new spi_tclDiscoveryDataIntf();

   spi_tclCalibrationReaderBase* poCalibrationReader = new KdsHandler();

   spi_tclStaticConfigurationBaseIntf* poStaticConfiguration = NULL;
   if( NULL != poCalibrationReader)
   {
      ETG_TRACE_USR2(("spi_tclFactory::bInitialize() creating spi_tclStaticConfigurationBase"));
      poStaticConfiguration = new spi_tclStaticConfiguration(poCalibrationReader);
   }

   spi_tclDynamicConfigurationReaderBase* poDatapoolReader = new Datapool();
   spi_tclDynamicConfigurationBase* poDynamicConfiguration = NULL;

   if(NULL != poDatapoolReader)
   {
      ETG_TRACE_USR2(("spi_tclFactory::bInitialize() creating spi_tclDynamicConfigurationReaderBase"));
      poDynamicConfiguration = new spi_tclDynamicConfiguration(poDatapoolReader);
   }

   if((NULL != poStaticConfiguration) && (NULL!= poDynamicConfiguration))
   {
      ETG_TRACE_USR2(("spi_tclFactory::bInitialize() creating spi_tclConfigurationManager"));
      m_poConfigurationManager = new spi_tclConfigurationManager(poStaticConfiguration,poDynamicConfiguration);
   }

#ifdef VARIANT_S_FTR_ENABLE_SPI_MIRRORLINK
   m_poDynMLLibrary = NULL;
   trSpiFeatureSupport rSpiFeatureSupp;
   if ((NULL != m_poConfigurationManager))
   {
      m_poConfigurationManager->vGetSpiFeatureSupport(rSpiFeatureSupp);
   }

   if(rSpiFeatureSupp.bMirrorLinkSupported())
   {
      // Load the mirrorlink dynamic library
      /* * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
       * RTLD_NOW flag resolves all undefined symbols in the library before dlopen() returns.
       * If this cannot be done, an error is returned.
       * dlopen takes around 900ms to load the dynamic library,libmirrorlinkspi_so.so
       * It takes around 100ms if any required library (RealVNC or ADIT) is not present for the dlopen to return.
       * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * */
      ETG_TRACE_USR1(("start dlopen for /opt/bosch/spi/plugin/libmirrorlinkspi_so.so"));
      m_poDynMLLibrary = dlopen("/opt/bosch/spi/plugin/libmirrorlinkspi_so.so", RTLD_NOW); //m_poDynMLLibrary can be stored in any/existing Datastructure which can be accessible anywhere and can be used when required
      if(NULL == m_poDynMLLibrary)
      {
         ETG_TRACE_ERR(("\n Couldn't load the dynamic library. Error=%s", dlerror()));
      }
      else
      {
         // reset/clear error code. Invoking dlerror() second time, immediately following a prior invocation,
         // will result in NULL being returned.
         dlerror();
         ETG_TRACE_USR1(("spi_tclFactory::bInitialize() dlopen success and before dlsym\n"));
         // dlsym returns the address where that symbol is loaded into memory(load the symbols)
         InitializeMLDynamicLibrary_t* Initialize_MLDynamicLibrary = (InitializeMLDynamicLibrary_t*) dlsym(m_poDynMLLibrary, "InitializeMLDynamicLibrary");//symbol defined in dynMLLibrary,spi_tclMLFactory.cpp
         const char* dlsym_error = dlerror();
         if (dlsym_error)
         {
            ETG_TRACE_ERR(("\n Cannot load symbol InitializeMLDynamicLibrary. Error=%s", dlsym_error));
         }
         else
         {
            ETG_TRACE_USR1(("spi_tclFactory::bInitialize() dlsym success and before Initialize_MLDynamicLibrary\n"));
            Initialize_MLDynamicLibrary();
            ETG_TRACE_USR1(("spi_tclFactory::bInitialize() before creating object\n"));
            if((NULL != Discoverer[e8_DISCOVERER_TYPE_ML]))
            {
   //! create mirrorlink discoverer
               mapoDiscoverers[e8_DISCOVERER_TYPE_ML] = Discoverer[e8_DISCOVERER_TYPE_ML]();
   SPI_NORMAL_ASSERT(NULL == mapoDiscoverers[e8_DISCOVERER_TYPE_ML]);
            }
            if((NULL != Connection[e8DEV_TYPE_MIRRORLINK]))
            {
   //! Create Mirrorlink connection
               mappoConnHandlers[e8DEV_TYPE_MIRRORLINK] = Connection[e8DEV_TYPE_MIRRORLINK]();
   SPI_NORMAL_ASSERT(NULL == mappoConnHandlers[e8DEV_TYPE_MIRRORLINK]);
            }
            if((NULL != DeviceSwitch[e8DEV_TYPE_MIRRORLINK]))
            {
               mappoDeviceSwitchers[e8DEV_TYPE_MIRRORLINK] = DeviceSwitch[e8DEV_TYPE_MIRRORLINK](poDiscovererDataIntf);
   SPI_NORMAL_ASSERT(NULL == mappoDeviceSwitchers[e8DEV_TYPE_MIRRORLINK]);
               ETG_TRACE_USR1((" spi_tclFactory::bInitialize obj success mappoDeviceSwitchers[e8DEV_TYPE_MIRRORLINK] = %p", mappoDeviceSwitchers[e8DEV_TYPE_MIRRORLINK]));
            }
         }
      }
   }
   ETG_TRACE_USR1(("spi_tclFactory::bInitialize() ML object creation END\n"));
#endif


   //! Create DiPo connection
   mappoConnHandlers[e8DEV_TYPE_DIPO] = new spi_tclDiPoConnection;
   SPI_NORMAL_ASSERT(NULL == mappoConnHandlers[e8DEV_TYPE_DIPO]);
   mappoDeviceSwitchers[e8DEV_TYPE_DIPO] = new spi_tclCarplaySwitch;
   SPI_NORMAL_ASSERT(NULL == mappoDeviceSwitchers[e8DEV_TYPE_DIPO]);
   mapoDiscoverers[e8_DISCOVERER_TYPE_APPLE] = new spi_tclAppleDiscoverer();
   SPI_NORMAL_ASSERT(NULL == mapoDiscoverers[e8_DISCOVERER_TYPE_APPLE]);

   //! Create AAP connection
#ifdef VARIANT_S_FTR_ENABLE_SPI_ANDROIDAUTO
   mappoConnHandlers[e8DEV_TYPE_ANDROIDAUTO] = new spi_tclAautoConnection;
   SPI_NORMAL_ASSERT(NULL == mappoConnHandlers[e8DEV_TYPE_ANDROIDAUTO]);
   mappoDeviceSwitchers[e8DEV_TYPE_ANDROIDAUTO] = new spi_tclAAPSwitch;
   SPI_NORMAL_ASSERT(NULL == mappoDeviceSwitchers[e8DEV_TYPE_ANDROIDAUTO]);
#endif

   //! Create MySPIN connection
#ifdef VARIANT_S_FTR_ENABLE_SPI_MYSPIN
   mappoConnHandlers[e8DEV_TYPE_MYSPIN] = new (std::nothrow)spi_tclMySPINConnMngr();
   SPI_NORMAL_ASSERT(NULL == mappoConnHandlers[e8DEV_TYPE_MYSPIN]);
   mappoDeviceSwitchers[e8DEV_TYPE_MYSPIN] = new spi_tclmySPINSwitch;
  SPI_NORMAL_ASSERT(NULL == mappoDeviceSwitchers[e8DEV_TYPE_MYSPIN]);
#endif

  //! Create BDCL connection
#ifdef VARIANT_S_FTR_ENABLE_SPI_BAIDU_CARLIFE
  mappoConnHandlers[e8DEV_TYPE_CARLIFE] = new spi_tclBDCLConnection;
  SPI_NORMAL_ASSERT(NULL == mappoConnHandlers[e8DEV_TYPE_CARLIFE]);
  mappoDeviceSwitchers[e8DEV_TYPE_CARLIFE] = new spi_tclBDCLSwitch;
  SPI_NORMAL_ASSERT(NULL == mappoDeviceSwitchers[e8DEV_TYPE_CARLIFE]);
#endif

  //! Create ONCAR connection
#ifdef VARIANT_S_FTR_ENABLE_SPI_ONCAR
  mappoConnHandlers[e8DEV_TYPE_ONCAR] = new spi_tclOnCarConnMngr();
  SPI_NORMAL_ASSERT(NULL == mappoConnHandlers[e8DEV_TYPE_ONCAR]);

  mappoDeviceSwitchers[e8DEV_TYPE_ONCAR] = new spi_tclOnCarSwitch;
  SPI_NORMAL_ASSERT(NULL == mappoDeviceSwitchers[e8DEV_TYPE_ONCAR]);
#endif

   //! Create component managers
   if ((NULL != m_poRespInterface) && (NULL != m_poMainApp))
   {
      spi_tclMediatorIntf* poMediator = spi_tclMediator::getInstance();
      //! Create Device List handler
      SPI_NORMAL_ASSERT(NULL== m_poConfigurationManager);

      m_poSQLite3db   = new SPIDatabaseSQLite3();
      SPI_NORMAL_ASSERT(NULL == m_poSQLite3db);

      m_poDevHistory = new spi_tclDevHistory(m_poConfigurationManager, m_poSQLite3db);
      SPI_NORMAL_ASSERT(NULL == m_poDevHistory);

      m_poDeviceListHandler = new spi_tclDeviceListHandler(m_poConfigurationManager,m_poDevHistory);
      SPI_NORMAL_ASSERT(NULL== m_poDeviceListHandler);
      spi_tclDeviceIDUtility *poDeviceIDUtil = spi_tclDeviceIDUtility::getInstance(m_poDeviceListHandler);
      SPI_INTENTIONALLY_UNUSED(poDeviceIDUtil);

      spi_tclUSBResetIntf *poUSBReset = new spi_tclUSBReset(poDiscovererDataIntf);
      m_poDeviceSwitcher = new spi_tclDeviceSwitcher(poDiscovererDataIntf, mappoDeviceSwitchers, poUSBReset);

      m_poUSBDiscoverer = new spi_tclUSBDiscovererImpl();
      mapoDiscoverers[e8_DISCOVERER_TYPE_USB] = new spi_tclUSBDiscoverer(m_poUSBDiscoverer, poDiscovererDataIntf);

      m_poDeviceDiscoverer= new spi_tclDeviceDiscoverer(m_poRespInterface, m_poDeviceListHandler,
               mapoDiscoverers,m_poDeviceSwitcher,poDiscovererDataIntf, m_poConfigurationManager, poMediator);
      SPI_NORMAL_ASSERT(NULL== m_poDeviceDiscoverer);

      std::map<tenCompID, spi_tclSelectionIntf*> mapComponents;

      mapComponents[e32COMPID_DISCOVERER] = m_poDeviceDiscoverer;
      m_poConnMngr = new spi_tclConnMngr(m_poRespInterface, m_poDeviceListHandler, mappoConnHandlers,
                poMediator, poDiscovererDataIntf);
      mapComponents[e32COMPID_CONNECTIONMANAGER] = m_poConnMngr;
      SPI_NORMAL_ASSERT(NULL== m_poConnMngr);

      m_poAppMngr = new spi_tclAppMngr(m_poRespInterface);
      SPI_NORMAL_ASSERT(NULL == m_poAppMngr);
      mapComponents[e32COMPID_APPMANAGER] = m_poAppMngr;

      m_poVideo = new spi_tclVideo(m_poMainApp, m_poRespInterface);
      SPI_NORMAL_ASSERT(NULL == m_poVideo);
      mapComponents[e32COMPID_VIDEO] = m_poVideo;

      m_poAudio = new spi_tclAudio(m_poRespInterface);
      SPI_NORMAL_ASSERT(NULL == m_poAudio);
      mapComponents[e32COMPID_AUDIO] = m_poAudio;

      m_poInputHandler = new spi_tclInputHandler(m_poRespInterface);
      SPI_NORMAL_ASSERT(NULL == m_poInputHandler);
      mapComponents[e32COMPID_INPUTHANDLER] = m_poInputHandler;

      m_poBluetooth = new spi_tclBluetooth(m_poRespInterface);
      SPI_NORMAL_ASSERT(NULL == m_poBluetooth);
      mapComponents[e32COMPID_BLUETOOTH] = m_poBluetooth;

      m_poDataService = new spi_tclDataService(m_poMainApp,m_poRespInterface);
      SPI_NORMAL_ASSERT(NULL == m_poDataService);
      mapComponents[e32COMPID_DATASERVICE] = m_poDataService;


      m_poRsrcMngr = new spi_tclResourceMngr(m_poRespInterface);
      SPI_NORMAL_ASSERT(NULL == m_poRsrcMngr);
      mapComponents[e32COMPID_RESOURCEMNGR] = m_poRsrcMngr;

      m_poLegalDisclaimer = new spi_tclLegalDisclaimer(m_poDeviceListHandler, m_poRespInterface, poMediator);
      SPI_NORMAL_ASSERT(NULL == m_poLegalDisclaimer);
      mapComponents[e32COMPID_LEGALDISCLAIMER] = m_poLegalDisclaimer;

      m_poTechnologySelector = new spi_tclTechnologySelector(m_poDeviceListHandler, m_poRespInterface, poMediator);
      SPI_NORMAL_ASSERT(NULL == m_poTechnologySelector);
      mapComponents[e32COMPID_TECHNOLOGYSELECTOR] = m_poTechnologySelector;

       m_poWiFi = new spi_tclWiFi(m_poRespInterface);
      SPI_NORMAL_ASSERT(NULL == m_poWiFi);
       mapComponents[e32COMPID_WIFI] = m_poWiFi;

       m_poDeviceSelector = new spi_tclDeviceSelector(m_poRespInterface, m_poDeviceListHandler, poMediator,
               m_poConfigurationManager, mapComponents);
      SPI_NORMAL_ASSERT(NULL == m_poDeviceSelector);

      m_poDiagnosis = new spi_tclDiagnosis(m_poRespInterface, poMediator,m_poConfigurationManager);
      SPI_NORMAL_ASSERT(NULL == m_poDiagnosis);


      m_poLaunchApp = new spi_tclAppLauncher(m_poRespInterface);
      SPI_NORMAL_ASSERT(NULL == m_poLaunchApp);

      m_poDefSet = new spi_tclDefSetLoader();
      SPI_NORMAL_ASSERT(NULL == m_poDefSet);

      m_poWifiSetupHndlr = new spi_tclWiFiSetupHndlr(m_poRespInterface);
      SPI_NORMAL_ASSERT(NULL == m_poWifiSetupHndlr);

   }

   //! Initialize component managers
   t_Bool bInit = ((NULL != m_poConnMngr)
         && (NULL != m_poAppMngr) && (NULL != m_poVideo)
         && (NULL != m_poAudio) && (NULL != m_poInputHandler)
         && (NULL != m_poBluetooth) && (NULL != m_poDataService)
         && (NULL != m_poRsrcMngr) && (NULL != m_poDeviceSelector)
         && (NULL != m_poWiFi) && (NULL != m_poConfigurationManager)
         && (NULL != m_poDefSet) && (NULL != m_poWifiSetupHndlr)
         && (NULL != m_poDiagnosis) && (NULL != m_poDevHistory)

         /* To be initialised first since MLVideo registers a DLT App ID*/
         && (m_poVideo->bInitialize())
         /* To be initialised before MLDataService and MLAppMgr for CDB and Notifications */
         && (m_poDeviceDiscoverer->bInitialize())
         && (m_poConnMngr->bInitialize())
         && (m_poAppMngr->bInitialize())
         && (m_poAudio->bInitialize())
         && (m_poInputHandler->bInitialize())
         && (m_poBluetooth->bInitialize())
         && (m_poDataService->bInitialize())
         && (m_poRsrcMngr->bInitialize())
         && (m_poDeviceSelector->bInitialize())
         && (m_poWiFi->bInitialize())
         && (m_poDiagnosis->bInitialize())
         && (m_poWiFi->bInitialize())
         && (m_poSQLite3db->bInitialize())
         && (m_poConfigurationManager->bInitialize())
         && (m_poDefSet->bInitialize())
         && (m_poWifiSetupHndlr->bInitialize())
         && (m_poDevHistory->bInitialize()));

   //!After Bt client recievs BT MAC Address we have to write it immediately to Datapool.
   if(NULL != m_poBluetooth)
   {
      m_poBluetooth->vSetBluetoothSettingsInstance(m_poConfigurationManager);
   }

   //@Note: Currently SPI registers its own app ID "SPIN" in main app class.
   //But ML also registers an app ID "MLNK" via mlink_dlt_initialize(), which overwrites "SPIN".
   //Hence it should be ensured AA & BDCL DLT contexts are registered only after the app ID is registered.

   return bInit;
}

/***************************************************************************
 ** FUNCTION:  t_Bool spi_tclFactory::bUnInitialize()
 ***************************************************************************/
t_Bool spi_tclFactory::bUnInitialize()
{
   ETG_TRACE_USR1(("spi_tclFactory::bUnInitialize() entered"));

   //! Uninitialize component managers
   t_Bool bUnInit = true; // return is not handled, hence it is always true.

   if(NULL != m_poDataService)
   {
      m_poDataService->bUnInitialize();
   }
   if(NULL != m_poBluetooth)
   {
      m_poBluetooth->bUnInitialize();
   }
   if(NULL != m_poInputHandler)
   {
      m_poInputHandler->bUnInitialize();
   }
   if(NULL != m_poAudio)
   {
      m_poAudio->bUnInitialize();
   }
   if(NULL != m_poVideo)
   {
      m_poVideo->bUnInitialize();
   }
   if(NULL != m_poAppMngr)
   {
      m_poAppMngr->bUnInitialize();
   }
   if(NULL != m_poConnMngr)
   {
      m_poConnMngr->bUnInitialize();
   }
   if(NULL != m_poRsrcMngr)
   {
      m_poRsrcMngr->bUnInitialize();
   }
   if(NULL != m_poDeviceSelector)
   {
      m_poDeviceSelector->vUnInitialize();
   }
   if(NULL != m_poWiFi)
   {
      m_poWiFi->bUnInitialize();
   }
   if(NULL != m_poDeviceDiscoverer)
   {
      m_poDeviceDiscoverer->bUnInitialize();
   }
   if(NULL != m_poSQLite3db)
   {
      m_poSQLite3db->bUnInitialize();
   }
#ifdef VARIANT_S_FTR_ENABLE_SPI_MIRRORLINK
   if(NULL != m_poDynMLLibrary)
   {
      dlclose(m_poDynMLLibrary);
   }
#endif
   if(NULL != m_poConfigurationManager)
   {
      m_poConfigurationManager->vUninitialize();
   }

   if(NULL != m_poDefSet)
   {
      m_poDefSet->bUnInitialize();
   }

   if(NULL != m_poWifiSetupHndlr)
   {
      m_poWifiSetupHndlr->bUnInitialize();
   }
   if(NULL != m_poDiagnosis)
   {
      m_poDiagnosis->vUnInitialize();
   }
   if(NULL != m_poDevHistory)
   {
      m_poDevHistory->bUnInitialize();
   }
   //! Release resources
   RELEASE_MEM(m_poLegalDisclaimer);
   RELEASE_MEM(m_poTechnologySelector);
   RELEASE_MEM(m_poRsrcMngr);
   RELEASE_MEM(m_poDataService);
   RELEASE_MEM(m_poBluetooth);
   RELEASE_MEM(m_poInputHandler);
   RELEASE_MEM(m_poAudio);
   RELEASE_MEM(m_poVideo);
   RELEASE_MEM(m_poAppMngr);
   RELEASE_MEM(m_poConnMngr);
   RELEASE_MEM(m_poDeviceListHandler);
   RELEASE_MEM(m_poDeviceSelector);
   RELEASE_MEM(m_poWiFi);
   RELEASE_MEM(m_poDeviceDiscoverer);
   RELEASE_MEM(m_poDiagnosis);

   RELEASE_MEM(m_poSQLite3db);
   RELEASE_MEM(m_poDefSet);
   RELEASE_MEM(m_poLaunchApp);
   RELEASE_MEM(m_poWifiSetupHndlr);
   RELEASE_MEM(m_poDeviceSwitcher);
   RELEASE_MEM(m_poConfigurationManager);
   RELEASE_MEM(m_poDevHistory);
   ETG_TRACE_USR1(("spi_tclFactory::bUnInitialize() returned"));

   return bUnInit;
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclFactory::vLoadSettings()
 ***************************************************************************/
t_Void spi_tclFactory::vLoadSettings()
{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));

   if(NULL != m_poConfigurationManager)
   {
       m_poConfigurationManager->vReadSPIPolicySettings();
       if (NULL != m_rCmdInterfaceCallback.fvSetCommonsettingInstance)
       {
          (m_rCmdInterfaceCallback.fvSetCommonsettingInstance)(m_poConfigurationManager);
       }

       if(NULL != m_poConnMngr)
       {
          m_poConnMngr->vSetConnsettingsInstance(m_poConfigurationManager);
       }
       if(NULL != m_poUSBDiscoverer)
       {
          m_poUSBDiscoverer->vSetDiscoverersettingsInstance(m_poConfigurationManager);
       }
       if(NULL != m_poDeviceSwitcher)
       {
          m_poDeviceSwitcher->vSetDiscoverersettingsInstance(m_poConfigurationManager);
       }
       if(NULL != m_poAppMngr)
       {
          m_poAppMngr->vSetAppSettingsInstance(m_poConfigurationManager);
       }
       if(NULL != m_poVideo)
       {
          m_poVideo->vSetVideoSettingsInstance(m_poConfigurationManager);
       }
       if(NULL != m_poInputHandler)
       {
           m_poInputHandler->vSetInputHandlerSettingsInstance(m_poConfigurationManager);
       }
       if(NULL != m_poAudio)
       {
          m_poAudio->vSetAudioSettingsInstance(m_poConfigurationManager);
       }
       if(NULL != m_poDataService)
       {
          m_poDataService->vSetDataServiceSettingsInstance(m_poConfigurationManager);
       }
       if(NULL != m_poRsrcMngr)
       {
          m_poRsrcMngr->vSetResourceMngrSettingsInstance(m_poConfigurationManager);
       }
       if(NULL != m_poWiFi)
       {
          m_poWiFi->vSetWiFiSettingsInstance(m_poConfigurationManager);
       }
       if(NULL != m_poLaunchApp)
       {
          m_poLaunchApp->vSetAudioSettingsInstance(m_poConfigurationManager);
       }
       if(NULL != m_poDefSet)
       {
          m_poDefSet->vSetDefaultSettingsInstance(m_poConfigurationManager);
       }
   }
   LOAD_SETTINGS(m_poDevHistory);
   LOAD_SETTINGS(m_poConnMngr);
   LOAD_SETTINGS(m_poDeviceDiscoverer);
   LOAD_SETTINGS(m_poAppMngr);
   LOAD_SETTINGS(m_poVideo);
   LOAD_SETTINGS(m_poAudio);
   LOAD_SETTINGS(m_poInputHandler);
   LOAD_SETTINGS(m_poBluetooth);
   LOAD_SETTINGS(m_poDataService);
   LOAD_SETTINGS(m_poRsrcMngr);
   LOAD_SETTINGS(m_poDeviceSelector);
   LOAD_SETTINGS(m_poWiFi);
   LOAD_SETTINGS(m_poWifiSetupHndlr);
   LOAD_SETTINGS(m_poDefSet);
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclFactory::vLoadSettingsCompleted()
 ***************************************************************************/
t_Void spi_tclFactory::vLoadSettingsCompleted()
{
    ETG_TRACE_USR1(("spi_tclFactory::vLoadSettingsCompleted entered"));
    LOAD_SETTINGS_COMPLETED(m_poConnMngr);
    LOAD_SETTINGS_COMPLETED(m_poDeviceDiscoverer);
    LOAD_SETTINGS_COMPLETED(m_poAppMngr);
    LOAD_SETTINGS_COMPLETED(m_poVideo);
    LOAD_SETTINGS_COMPLETED(m_poAudio);
    LOAD_SETTINGS_COMPLETED(m_poInputHandler);
    LOAD_SETTINGS_COMPLETED(m_poBluetooth);
    LOAD_SETTINGS_COMPLETED(m_poDataService);
    LOAD_SETTINGS_COMPLETED(m_poRsrcMngr);
    LOAD_SETTINGS_COMPLETED(m_poWiFi);
    LOAD_SETTINGS_COMPLETED(m_poWifiSetupHndlr);

    if(NULL != m_poDeviceDiscoverer)
    {
       m_poDeviceDiscoverer->vStartDeviceDetection();
    }
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclFactory::vSaveSettings()
 ***************************************************************************/
t_Void spi_tclFactory::vSaveSettings()
{
   ETG_TRACE_USR1((" %s entered ", __PRETTY_FUNCTION__));

   SAVE_SETTINGS(m_poConnMngr);
   SAVE_SETTINGS(m_poAppMngr);
   SAVE_SETTINGS(m_poVideo);
   SAVE_SETTINGS(m_poAudio);
   SAVE_SETTINGS(m_poInputHandler);
   SAVE_SETTINGS(m_poBluetooth);
   SAVE_SETTINGS(m_poDataService);
   SAVE_SETTINGS(m_poRsrcMngr);
   SAVE_SETTINGS(m_poDeviceSelector);
   SAVE_SETTINGS(m_poWiFi);
   SAVE_SETTINGS(m_poDeviceDiscoverer);
   SAVE_SETTINGS(m_poDefSet);
   SAVE_SETTINGS(m_poWifiSetupHndlr);
   SAVE_SETTINGS(m_poDevHistory);
}

/***************************************************************************
 ** FUNCTION: t_Void spi_tclFactory::vRestoreSettings()
 ***************************************************************************/
t_Void spi_tclFactory::vRestoreSettings()
{
   ETG_TRACE_USR1((" %s entered", __PRETTY_FUNCTION__));
   //load the default values on call to restore settings
   if (NULL != m_poDefSet)
   {
      m_poDefSet->vRestoreSettings();
   }

   if(NULL != m_poWifiSetupHndlr)
   {
      m_poWifiSetupHndlr->vRestoreSettings();
   }
}

/***************************************************************************
 ** FUNCTION:  spi_tclConnMngr* spi_tclFactory::poGetConnMngrInstance
 ***************************************************************************/
spi_tclConnMngr* spi_tclFactory::poGetConnMngrInstance()
{
   return m_poConnMngr;
}

/***************************************************************************
 ** FUNCTION:  spi_tclVideo* spi_tclFactory::poGetVideoInstance
 ***************************************************************************/
spi_tclVideo* spi_tclFactory::poGetVideoInstance()
{
   return m_poVideo;
}

/***************************************************************************
 ** FUNCTION:  spi_tclMLAppMngr* spi_tclFactory::poGetAppManagerInstance
 ***************************************************************************/
spi_tclAppMngr* spi_tclFactory::poGetAppManagerInstance()
{
   return m_poAppMngr;
}

/***************************************************************************
 ** FUNCTION:  spi_tclAudio* spi_tclFactory::poGetAudioInstance
 ***************************************************************************/
spi_tclAudio* spi_tclFactory::poGetAudioInstance()
{
   return m_poAudio;
}

/***************************************************************************
 ** FUNCTION:  spi_tclInputHandler* spi_tclFactory::poGetInputHandlerInstance
 ***************************************************************************/
spi_tclInputHandler* spi_tclFactory::poGetInputHandlerInstance()
{
   return m_poInputHandler;
}

/***************************************************************************
** FUNCTION:  spi_tclBluetooth* spi_tclFactory::poGetBluetoothInstance
***************************************************************************/
spi_tclBluetooth* spi_tclFactory::poGetBluetoothInstance()
{
   return m_poBluetooth;
}

/***************************************************************************
** FUNCTION:  spi_tclDataService* spi_tclFactory::poGetDataServiceInstance
***************************************************************************/
spi_tclDataService* spi_tclFactory::poGetDataServiceInstance()
{
   return m_poDataService;
}

/***************************************************************************
** FUNCTION:  ahl_tclBaseOneThreadApp* spi_tclFactory::poGetMainAppInstance
***************************************************************************/
ahl_tclBaseOneThreadApp* spi_tclFactory::poGetMainAppInstance()
{
   return m_poMainApp;
}

/***************************************************************************
 ** FUNCTION:  spi_tclFactory::poGetRsrcMngrInstance()
 ***************************************************************************/
spi_tclResourceMngr* spi_tclFactory::poGetRsrcMngrInstance()
{
   return m_poRsrcMngr;
}

/***************************************************************************
 ** FUNCTION:  spi_tclLegalDisclaimer* spi_tclFactory::poGetLegalDiscInstance
 ***************************************************************************/
spi_tclLegalDisclaimer* spi_tclFactory::poGetLegalDiscInstance()
{
   return m_poLegalDisclaimer;
}

/***************************************************************************
 ** FUNCTION:  spi_tclTechnologySelector* spi_tclFactory::poGetTechnologySelectorInstance
 ***************************************************************************/
spi_tclTechnologySelector* spi_tclFactory::poGetTechnologySelectorInstance()
{
   return m_poTechnologySelector;
}

/***************************************************************************
 ** FUNCTION:  spi_tclDeviceSelector* spi_tclFactory::poGetDeviceSelector
 ***************************************************************************/
spi_tclDeviceSelector* spi_tclFactory::poGetDeviceSelector()
{
   return m_poDeviceSelector;
}
/***************************************************************************
 ** FUNCTION:  spi_tclDeviceDiscoverer* spi_tclFactory::poGetDeviceDiscoverer
 ***************************************************************************/
spi_tclDeviceDiscoverer* spi_tclFactory::poGetDeviceDiscoverer()
{
   return m_poDeviceDiscoverer;
}

/***************************************************************************
** FUNCTION:  spi_tclWiFi* spi_tclFactory::poGetWiFiInstance
***************************************************************************/
spi_tclWiFi* spi_tclFactory::poGetWiFiInstance()
{
   return m_poWiFi;
}

/***************************************************************************
 ** FUNCTION:  spi_tclAppLauncher* spi_tclFactory::poGetAppLauncherInstance
 ***************************************************************************/
spi_tclAppLauncher* spi_tclFactory::poGetAppLauncherInstance()
{
   return m_poLaunchApp;
}

/***************************************************************************
 ** FUNCTION:  spi_tclWiFiSetupHndlr* spi_tclFactory::poGetWiFiSetupHndlrInstance
 ***************************************************************************/
spi_tclWiFiSetupHndlr* spi_tclFactory::poGetWiFiSetupHndlrInstance()
{
   return m_poWifiSetupHndlr;
}

/***************************************************************************
 ** FUNCTION: spi_tclFactory::vRegisterCommonSettingCallbacks
 ***************************************************************************/
t_Void spi_tclFactory::vRegisterCommonSettingCallbacks(trCmdInterfaceCallback &rfrCmdInterfaceCallback)
{
    m_rCmdInterfaceCallback = rfrCmdInterfaceCallback;
}

/***************************************************************************
 *********************************PRIVATE***********************************
 ***************************************************************************/
spi_tclFactory::spi_tclFactory() :
   m_poRespInterface(NULL), m_poConnMngr(NULL), m_poVideo(NULL),
   m_poAppMngr(NULL), m_poAudio(NULL), m_poInputHandler(NULL),
   m_poBluetooth(NULL), m_poDataService(NULL), m_poMainApp(NULL), m_poRsrcMngr(NULL),
   m_poLegalDisclaimer(NULL),m_poTechnologySelector(NULL), m_poDeviceListHandler(NULL),
   m_poDeviceSelector(NULL),m_poWiFi(NULL), m_poDeviceDiscoverer(NULL),m_poSQLite3db(NULL),
   m_poConfigurationManager(NULL), m_poLaunchApp(NULL),m_poDefSet(NULL),m_poWifiSetupHndlr(NULL),
   m_poUSBDiscoverer(NULL), m_poDeviceSwitcher(NULL),  m_poDiagnosis(NULL)
{
   ETG_TRACE_USR1(("spi_tclFactory::spi_tclFactory() entered"));
}

/***************************************************************************
 ** FUNCTION:  spi_tclFactory::spi_tclFactory
 ***************************************************************************/
spi_tclFactory::spi_tclFactory(spi_tclRespInterface* poRespInterface,
         ahl_tclBaseOneThreadApp* poMainAppl) :
   m_poRespInterface(poRespInterface), m_poConnMngr(NULL), m_poVideo(NULL),
   m_poAppMngr(NULL), m_poAudio(NULL), m_poInputHandler(NULL),
   m_poBluetooth(NULL), m_poDataService(NULL), m_poMainApp(poMainAppl),  m_poRsrcMngr(NULL),
   m_poLegalDisclaimer(NULL),m_poTechnologySelector(NULL), m_poDeviceListHandler(NULL),
   m_poDeviceSelector(NULL),m_poWiFi(NULL), m_poDeviceDiscoverer(NULL), m_poSQLite3db(NULL),
   m_poConfigurationManager(NULL), m_poLaunchApp(NULL),m_poDefSet(NULL),m_poWifiSetupHndlr(NULL),
   m_poUSBDiscoverer(NULL),m_poDeviceSwitcher(NULL), m_poDiagnosis(NULL)
{
   ETG_TRACE_USR1(("spi_tclFactory::spi_tclFactory(..) entered"));
}

//lint �restore
